/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This file contains the generic Device Driver Framework APIs
 *
 ******************************************************************************/

#include <stdarg.h>
#include <string.h>
#include <stdio.h>

#include "standard.h"

#include "osal.h"
#include "osal_core.h"
#include "os_intf.h"

#include "osal_devices.h"
#include "_osal_devices.h"

/*****************************************************************************
*
*       DEV_n32RegNode
*
*   This API registers a device node with the device driver framework.
*   Registration does not actually initiate any communication or access with
*   the physical device itself, but rather initializes the device driver node
*   with it's name, interface functions and any physical device properties
*   which may be required.
*
*   Inputs:
*       pcDevName - An ASCII null terminated string which identifies this
*           device. A copy of the provided name is retained in the device node.
*       psInterface - A const pointer of type DEV_INTERFACE_STRUCT to this
*           device's interface (the driver implementation). The Framework will
*           internally map the device name to this interface. This pointer must
*           be a static constant which is globally accessible.
*       pvProperties  - A const pointer to this device's properties (if any).
*           The Framework will internally map the device name with this
*           property pointer. The property pointer will typically point
*           to a device driver specific object (structure or value) which
*           specifies a specific physical device (e.g. Bus Address, Device
*           Id, etc.). A copy of the data is maintained by the Framework.
*           This value may be NULL if no properties are required.
*       tPropertySize - The size (in bytes) of the property data. This value
*           may be zero if no properties are required.
*
*   Outputs:
*       DEV_OK on success. Otherwise a value < 0 is returned on error.
*       See section 4.11 for possible error codes.
*
*****************************************************************************/
N32 DEV_n32RegNode(
    const char *pcDevName,
    const OSAL_DEV_INTERFACE_STRUCT *psInterface,
    const void *pvProperties,
    size_t tPropertySize
        )
{
#if OSAL_DEVICES == 1
    DEV_NODE_STRUCT *psNode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check that a valid device name has been provided and
    // has some length but is not too long.
    if(pcDevName == NULL)
    {
        // Error! Null pointer
        return DEV_INVALID_INPUT;
    }
    else
    {
        // Check name length
        size_t tLen = strlen(pcDevName);

        // Length to small?
        if(tLen == 0)
        {
            return DEV_NAME_TOO_SHORT;
        }

        // Length too big?
        if(tLen > OSAL_DEV_MAX_NAME_LENGTH)
        {
            return DEV_NAME_TOO_LONG;
        }
    }

    // Check if an interface was provided, if not what's the point?
    if(psInterface == NULL)
    {
        return DEV_NO_INTERFACE;
    }

    // Find an available node...
    psNode = DEV_psFindNode(pcDevName);

    // This will return a node entry that either matches the name provided
    // or one which is empty (it can accept the new node)
    // Find out which it is...
    if(psNode == NULL)
    {
        // Error! No node can be found
        return DEV_MAX_DRIVERS_EXCEEDED;
    }

    // A node was found, but is it available?
    if(strlen(psNode->acDevName) > 0)
    {
        // Error! A node already exists with this name
        return DEV_NAME_DUPLICATE;
    }

    // Construct a unique name for the 'node mutex'
    snprintf(&acName[0], sizeof(acName), OSAL_NAME_PREFIX"Node:%s", pcDevName);

    // Allocate a mutex to control access to shared structures via nodes
    eReturnCode = OSAL.eSemCreate(
        &psNode->sShared.hMutex, acName, 0, 1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error! Mutex failure
        return DEV_MUTEX_FAILURE;
    }

    // Allocate memory for NODE attributes(properties)
    // that are provided by the caller.
    if(tPropertySize > 0)
    {
        // Allocate memory to contain shared properties
        psNode->sShared.pvProperties = OSALC_pvMemoryAllocate(
            OSAL_NAME_PREFIX"Attribute",
            tPropertySize, FALSE
                );

        if(psNode->sShared.pvProperties != NULL)
        {
            // Copy attributes locally
            OSAL.bMemCpy(psNode->sShared.pvProperties,
                pvProperties, tPropertySize);
        }
        else
        {
            // Error! Cannot allocate memory for properties
            OSAL.eSemDelete(psNode->sShared.hMutex);
            return DEV_OUT_OF_MEMORY;
        }
    }

    // Copy node info into entry and initialize as needed
    strncpy(&psNode->acDevName[0], pcDevName, sizeof(psNode->acDevName));

    psNode->sInterface = *psInterface;
    psNode->sShared.un32OpenCount = 0;
    psNode->sShared.pvPublicData = NULL; // filled in by _bInit() later

    // Release node mutex
    OSAL.eSemGive(psNode->sShared.hMutex);

    return DEV_OK;
#else
    return DEV_ERROR;
#endif
}

/*****************************************************************************
*
*       DEV_bUnRegNode
*
*   Inputs:
*       This API un-registers or removes a device node which has been
*       registered previously. In order for the un-registration to succeed
*       all open connections to the device must be closed at which point the
*       node will be released and all resources acquired will be free.
*
*       pcDevName - An ASCII null terminated string which identifies this
*           device to be unregistered.
*
*   Outputs:
*       BOOLEAN - TRUE if successful, FALSE otherwise.
*
*****************************************************************************/
BOOLEAN DEV_bUnRegNode(
    const char *pcDevName
        )
{
#if OSAL_DEVICES == 1
    BOOLEAN bRetVal = FALSE;
    DEV_NODE_STRUCT *psNode;

    // Check that a valid device name has been provided
    if(pcDevName != NULL)
    {
        // Locate the node the name the caller provided references
        psNode = DEV_psFindNode(pcDevName);
        if(psNode != NULL)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Take control of node shared data
            eReturnCode = OSAL.eSemTake(
                psNode->sShared.hMutex,
                OSAL_OBJ_TIMEOUT_INFINITE );
            if (eReturnCode == OSAL_SUCCESS)
            {
                // Anyone have this device open?
                if(psNode->sShared.un32OpenCount == 0)
                {
                    // Free any attribute memory
                    if(psNode->sShared.pvProperties != NULL)
                    {
                        OSALC_vMemoryFree(psNode->sShared.pvProperties);
                        psNode->sShared.pvProperties = NULL;
                    }

                    // Destroy node mutex
                    OSAL.eSemDelete(psNode->sShared.hMutex);

                    // Invalidate this entry
                    *psNode = gsNodeInit;

                    // All is well
                    bRetVal = TRUE;
                }
                else
                {
                    // Release node mutex
                    OSAL.eSemGive(psNode->sShared.hMutex);
                }
            }
        }
    }
    return bRetVal;
#else
    return FALSE;
#endif
}

/*****************************************************************************
*
*       DEV_hOpen
*
*   This API opens a unique connection or instance of the device driver
*   specified by pcDevName. If this is the first call to open the device
*   driver will also call the device's initialization method (_bInit())
*   if it exists.
*
*   The API returns a unique handle to this instance which may be used for
*   other API calls. The const void pointer; pvArg supplied is device driver
*   specific and must be defined by the driver itself. Typically the value
*   to pass in is a pointer to a structure containing some type of open mode
*   or condition information and is defined in the <driver_name>.h file. This
*   structure (if required) must be documented in a document for this driver.
*   If multiple instances of a device are opened each instance's access to the
*   device will be protected by mutual exclusion.
*
*   Inputs:
*       pcDevName - An ASCII null terminated string which represents the
*           device to open a connection to. If the name does not exists,
*           this API will fail.
*       pvArg - A device driver specific value to open this instance of
*           the device driver with. It typically represents some
*           characteristics, mode or configuration specific to the device.
*
*   Outputs:
*       A handle which represents this connection on success; otherwise
*       DEV_INVALID_HANDLE on failure.
*
*****************************************************************************/
DEV_HDL DEV_hOpen(
    const char *pcDevName,
    const void *pvArg
        )
{
    DEV_NODE_STRUCT *psNode;
    DEV_STRUCT *psDevice = (DEV_STRUCT *)DEV_INVALID_HDL;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check if device node has been registered
    psNode = DEV_psFindNode(pcDevName);
    if(psNode == NULL)
    {
        // Error! Unable to find node.
        return DEV_INVALID_HDL;
    }

    if(psNode->sShared.hMutex == OSAL_INVALID_OBJECT_HDL)
    {
        // Error! Node is found but not registered
        return DEV_INVALID_HDL;
    }

    // Take control of node shared data
    eReturnCode = OSAL.eSemTake(
        psNode->sShared.hMutex,
        OSAL_OBJ_TIMEOUT_INFINITE );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error! Unable to obtain node mutex.
        return DEV_INVALID_HDL;
    }

    // Perform open...
    do
    {
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Node found must be valid, otherwise simply the next available
        // node will be returned.
        if(strlen(&psNode->acDevName[0]) == 0)
        {
            // Invalid node name
            break;
        }

        // Construct an appropriate name for memory to be allocated & created
        snprintf( acName, OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                  OSAL_NAME_PREFIX"%s:%d",
                  pcDevName, psNode->sShared.un32OpenCount);

        // Allocate memory for this instance
        psDevice =
            (DEV_STRUCT *)OSALC_pvMemoryAllocate(
                acName, sizeof(DEV_STRUCT), FALSE);
        if(psDevice == NULL)
        {
            // Memory allocation failed
            break;
        }

        // Populate device instance with a reference to the node it belongs to
        psDevice->psNode = psNode;

        // Populate this instance with device data and initialize as needed
        psDevice->sInfo.pcDevName = &psNode->acDevName[0];
        psDevice->sInfo.un32Instance = psNode->sShared.un32OpenCount;
        psDevice->sInfo.un32Position = 0;
        psDevice->sInfo.pvProperties = psNode->sShared.pvProperties;
        psDevice->sInfo.pvPublicData = NULL; // filled in by _bInit() later
        psDevice->sInfo.pvPrivateData = NULL; // filled in by _n32Open() later

        // Init? Do only on first open
        if(psNode->sShared.un32OpenCount == 0)
        {
            // If no _bInit() method defined...ok
            if(psNode->sInterface.bInit != NULL)
            {
                // Call _bInit() method
                BOOLEAN bInit =
                    psNode->sInterface.bInit(
                        pcDevName, psNode->sShared.pvProperties,
                        &psNode->sShared.pvPublicData);
                if(bInit == FALSE)
                {
                    // Error! Initialization failed
                    OSALC_vMemoryFree(psDevice);
                    psDevice = (DEV_STRUCT *)DEV_INVALID_HDL;
                    break;
                }
            }
        }

        // Record any public data set by the _bInit() method. This
        // is as a result of the bInit() being called or from a previous
        // call to bInit() from a previous open.
        psDevice->sInfo.pvPublicData = psNode->sShared.pvPublicData;

        // Call driver's open method (if it exists)
        if(psNode->sInterface.n32Open != NULL)
        {
            // Open it
            N32 n32Result =
                psNode->sInterface.n32Open(&psDevice->sInfo, pvArg);
            if(n32Result != DEV_OK)
            {
                // Open failed. Exit?
                if(psNode->sShared.un32OpenCount == 0) // Last one
                {
                    // Call exit() method if it exists
                    if(psNode->sInterface.vExit != NULL)
                    {
                        psNode->sInterface.vExit(
                            psNode->sShared.pvProperties,
                            psNode->sShared.pvPublicData
                                );
                    }
                }

                // Clean-up
                OSALC_vMemoryFree(psDevice);
                psDevice = (DEV_STRUCT *)DEV_INVALID_HDL;
                break;
            }
            else // Open!
            {
                // Increment open count on successful open
                psNode->sShared.un32OpenCount++;
                break;
            }
        }

    } while (FALSE);

    // Release node mutex
    OSAL.eSemGive(psNode->sShared.hMutex);

    return (DEV_HDL)psDevice;
}

/*****************************************************************************
*
*       DEV_n32Close
*
*   This API closes a unique connection or instance of the device driver
*   the provided handle (hDevice) represents. If this call closes the last
*   instance of the driver the driver's exit method (_vExit()) will be called
*   if it exists. Once the close operation completes the provided handle is
*   no longer valid.
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*
*   Outputs:
*       DEV_OK on success. Otherwise a value < 0 is returned on error.
*       See section 4.11 for possible error codes.
*
*
*****************************************************************************/
N32 DEV_n32Close(
    DEV_HDL hDevice
        )
{
    DEV_STRUCT *psDevice = (DEV_STRUCT *)hDevice;
    DEV_NODE_STRUCT *psNode;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Check if a valid device handle was provided
    if(hDevice == DEV_INVALID_HDL)
    {
        // Error! No handle
        return DEV_INVALID_INPUT;
    }

    // Call close() method if it exists...if not, ok
    if(psDevice->psNode->sInterface.n32Close != NULL)
    {
        N32 n32Result =
            psDevice->psNode->sInterface.n32Close(
                &psDevice->sInfo );
        if(n32Result != DEV_OK)
        {
            // Error! Cannot close...bail!
            return DEV_ERROR;
        }
    }

    // Grab node ptr
    psNode = psDevice->psNode;

    // Take control of node shared data
    eReturnCode = OSAL.eSemTake(
        psNode->sShared.hMutex,
        OSAL_OBJ_TIMEOUT_INFINITE );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error! Unable to obtain node mutex.
        return DEV_ERROR;
    }

    // Perform device exit/close...
    do
    {
        // Decrement open count
        if(psNode->sShared.un32OpenCount > 0)
        {
            // One less open instance...
            psNode->sShared.un32OpenCount--;

            // Exit?
            if(psNode->sShared.un32OpenCount == 0) // Last one
            {
                // Call exit() method if it exists
                if(psNode->sInterface.vExit != NULL)
                {
                    psNode->sInterface.vExit(
                        psNode->sShared.pvProperties,
                        psNode->sShared.pvPublicData
                            );
                }
            }
        }

        // Free memory for this instance
        if(psDevice != NULL)
        {
            OSALC_vMemoryFree(psDevice);
            psDevice = (DEV_STRUCT *)NULL;
        }

    } while (FALSE);

    // Release node mutex
    OSAL.eSemGive(psNode->sShared.hMutex);

    // All is well
    return DEV_OK;
}

/*****************************************************************************
*
*       DEV_tRead
*
*   This API requests a byte stream read from a device represented by the
*   provided handle hDevice. A device read is byte-stream oriented, and will
*   cause the current position (or accumulated bytes read) to increase as
*   each read is performed. The current position within the device (or index)
*   is represents both the current read and write position. This position may
*   be indexed or a seek performed using the DEV_n32Seek() API. The amount of
*   data to read is given by the size of the object to read (tSize) and the
*   number of objects to read (tNumObjs). The data read from the device is
*   placed at the address provided by pcDst.
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*       pcDst - A pointer which defines where the data read from the
*           device is to be stored.
*       tSize - The size of each object to read from the device.
*       tNumObjs - The number of objects of size tSize to read from the device.
*
*   Outputs:
*       The number of objects successfully read from the device and placed
*       in pcDst. If an error occurs or no more data is available a zero is
*       returned. It should not be assumed that every read will read all the
*       data requested in a single call. The caller must check the return
*       value to determine if all the requested data was read or if a partial
*       read was performed. If a partial read occurs the caller may wish to
*       call the API again to complete the operation.
*
*****************************************************************************/
size_t DEV_tRead(
    char *pcDst,
    size_t tSize,
    size_t tNumObjs,
    DEV_HDL hDevice
        )
{
    size_t tResult = 0;
    DEV_STRUCT *psDevice = (DEV_STRUCT *)hDevice;

    // Check user inputs...
    if( (hDevice != DEV_INVALID_HDL) &&
           (pcDst != NULL) &&
               (tSize != 0) &&
                   (tNumObjs != 0) )
    {
        // User input's are good. Call device's _tRead() method if it exists
        if(psDevice->psNode->sInterface.tRead != NULL)
        {
            // We have the mutex, now call _tRead() method
            tResult =
                psDevice->psNode->sInterface.tRead (
                    &psDevice->sInfo,
                    pcDst,
                    tSize,
                    tNumObjs
                        );

            // If a non-error result, increment read position
            // according to the number of objects read and the size
            // of each object.
            if(tResult > 0)
            {
                psDevice->sInfo.un32Position += (tSize * tResult);
            }
        }
    }

    return tResult;
}

/*****************************************************************************
*
*       DEV_tWrite
*
*   This API requests a byte stream write to a device represented by the
*   provided handle hDevice. A device write is byte-stream oriented, and
*   will cause the current position (or accumulated bytes written) to
*   increase as each write is performed. The current position within the
*   device (or index) is represents both the current read and write position.
*   This position may be indexed or a seek performed using the DEV_n32Seek()
*   API. The amount of data to write is given by the size of the object to
*   write (tSize) and the number of objects to write (tNumObjs). The data
*   written to the device is retrieved from the address provided by pcSrc.
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*       pcDst - A pointer which defines where the data written to the device
*           is to come from.
*       tSize - The size of each object to write to the device.
*       tNumObjs - The number of objects of size tSize to write to the device.
*
*   Outputs:
*       The number of objects successfully written the device from pcSrc.
*       If an error occurs or no more data can be written a zero is returned.
*       It should not be assumed that every write will write all the data
*       requested in a single call. The caller must check the return value
*       to determine if all the requested data was written or if a partial
*       write was performed. If a partial write occurs the caller may wish
*       to call the API again to complete the operation.
*
*
*****************************************************************************/
size_t DEV_tWrite(
    const char *pcSrc,
    size_t tSize,
    size_t tNumObjs,
    DEV_HDL hDevice
        )
{
    size_t tResult = 0;
    DEV_STRUCT *psDevice = (DEV_STRUCT *)hDevice;

    // Check user inputs...
    if( (hDevice != DEV_INVALID_HDL) &&
           (pcSrc != NULL) &&
               (tSize != 0) &&
                   (tNumObjs != 0) )
    {
        // User input's are good. Call device's _tWrite() method if it exists
        if(psDevice->psNode->sInterface.tWrite != NULL)
        {
            // We have the mutex, now call _tWrite() method
            tResult =
                psDevice->psNode->sInterface.tWrite (
                    &psDevice->sInfo,
                    pcSrc,
                    tSize,
                    tNumObjs
                        );

            // If a non-error result, increment write position
            // according to the number of objects written and the size
            // of each object.
            if(tResult > 0)
            {
                psDevice->sInfo.un32Position += (tSize * tResult);
            }
        }
    }

    return tResult;
}

/*****************************************************************************
*
*       DEV_n32Ioctl
*
*   This API requests a device (specified by the handle hDevice) to perform
*   a device driver specific command or control operation to be performed
*   on it. The specific commands, n32Cmd provided are defined by the device
*   driver and should be referenced in a device driver document along with
*   any required parameters. This API takes a variable number of command
*   arguments which are also device specific.
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*       n32Cmd - A command number which represents some device specific
*           operation.
*       ... - A variable number of arguments which will be passed to the device
*           driver ioctl implementation for processing.
*
*   Outputs:
*       DEV_OK on success. Otherwise a value < 0 is returned on error.
*       See section 4.11 for possible error codes. If the command is not
*       supported by the device it is issued to the return value will be
*       DEV_UNSUPPORTED_CMD.
*
*****************************************************************************/
N32 DEV_n32Ioctl(
    DEV_HDL hDevice,
    N32 n32Cmd,
    ...
        )
{
    N32 n32Result = DEV_OK;
    va_list tList; // variable arguments list

    if(hDevice == DEV_INVALID_HDL)
    {
        // Error! Invalid handle
        return DEV_INVALID_INPUT;
    }

    // grab variable arguments (pop)
    va_start(tList, n32Cmd);

    n32Result = DEV_n32vIoctl(hDevice, n32Cmd, &tList);

    // restore stack (push)
    va_end(tList);

    return n32Result;
}

/*****************************************************************************
*
*       DEV_n32vIoctl
*
*   This helper function allows the support ioctl calls with a variable
*   argument list pointer rather than the variable arguments themselves.
*   Otherwise, it is identical to DEV_n32Ioctl(). Separation of this
*   functionality is useful when mapping this DEV_ function to a more
*   standard ioctl() function.
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*       n32Cmd - A command number which represents some device specific
*           operation.
*       tList - A variable number of arguments which will be passed to the device
*           driver ioctl implementation for processing.
*
*   Outputs:
*       DEV_OK on success. Otherwise a value < 0 is returned on error.
*       See section 4.11 for possible error codes. If the command is not
*       supported by the device it is issued to the return value will be
*       DEV_UNSUPPORTED_CMD.
*
*****************************************************************************/
N32 DEV_n32vIoctl(
    DEV_HDL hDevice,
    N32 n32Cmd,
    va_list *ptList
        )
{
    N32 n32Result = DEV_OK;
    DEV_STRUCT *psDevice = (DEV_STRUCT *)hDevice;

    if(hDevice == DEV_INVALID_HDL)
    {
        // Error! Invalid handle
        return DEV_INVALID_INPUT;
    }

    // Handle any common group ioctl's first, then go driver specific
    switch(n32Cmd)
    {
        // Handle seek
        case DEV_IOCTL_SEEK_POSITION:
        {
        	N32 n32Offset;
        	N8 n8Mode;

            // Pull arguments from the list
            n32Offset = (N32)va_arg( *ptList, int );
            n8Mode = (N8)va_arg( *ptList, int );

            // Calculate new position based on seek mode
            switch(n8Mode)
            {
                // Set new position relative to the first position
                case SEEK_SET:
                    psDevice->sInfo.un32Position = n32Offset;
                break;

                // Set new position relative to current position
                case SEEK_CUR:
                    psDevice->sInfo.un32Position += n32Offset;
                break;

                // Set net position relative to the last position
                case SEEK_END:
                    // Unsupported, don't have a concept of the end?
                    // Allow driver itself to handle it...if it can
                    if(psDevice->psNode->sInterface.n32Ioctl != NULL)
                    {
                        n32Result =
                            psDevice->psNode->sInterface.n32Ioctl (
                                &psDevice->sInfo,
                                n32Cmd,
                                ptList
                                    );
                    }
                    else
                    {
                        // Error! Bad input mode
                        n32Result = DEV_INVALID_INPUT;
                    }
                break;

                default:
                    // Error! Bad input mode
                    n32Result = DEV_INVALID_INPUT;
                break;
            }
        }
        break;

        // Any unhandled ioctl's get passed to the driver
        // if an ioctl function was defined
        default:

            // Check ioctl method exists, if so, call it.
            if(psDevice->psNode->sInterface.n32Ioctl != NULL)
            {
                n32Result =
                    psDevice->psNode->sInterface.n32Ioctl (
                        &psDevice->sInfo,
                        n32Cmd,
                        ptList
                            );
            }
            else
            {
                n32Result = DEV_UNSUPPORTED_CMD;
            }

        break;
    }

    return n32Result;
}

/*****************************************************************************
*
*       DEV_n32Seek
*
*   This API allows the caller to seek into (adjust the position of) the
*   current read and write position of a device referenced by the hDevice
*   handle where the next read or write operation will occur. This API is
*   typically useful for media type devices which have some concept of a
*   buffer or array where data may be read from or written to. The current
*   read and write position may be adjusted by a byte-oriented n32Offset
*   and relative to the provided n8Mode.
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*       n32Offset - The number of bytes to adjust the current read and write
*           position relative to the provided n8Mode type.
*       n8Mode - Describes how to apply the provided n32Offset. May be one for
*           the following values:
*           SEEK_SET - Sets the new position n32Offset bytes relative to
*           the start position of the device. (i.e. new position = current
*           position + start)
*           SEEK_CUR - Sets the new position n32Offset bytes relative
*           to the current position (i.e. new position = current position
*           + n32Offset).
*           SEEK_END - Sets the new position n32Offset bytes relative to
*           the end position of the device (i.e. new position =
*           end position + n32Offset).
*
*   Outputs:
*       DEV_OK on success. Otherwise a value < 0 is returned on error.
*       See section 4.11 for possible error codes.
*
*****************************************************************************/
N32 DEV_n32Seek(
    DEV_HDL hDevice,
    N32 n32Offset,
    N8 n8Mode
        )
{
    N32 n32Result;

    n32Result =
        DEV_n32Ioctl (
            hDevice,
            DEV_IOCTL_SEEK_POSITION,
            n32Offset,
            n8Mode
                );

    return n32Result;
}

/*****************************************************************************
*
*       DEV_n32Error
*
*   This API allows the caller to query the error status of a device
*   referenced by the hDevice handle
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*
*   Outputs:
*       A non-zero number indicates an error has been detected.
*
*****************************************************************************/
N32 DEV_n32Error(DEV_HDL hDevice)
{
    N32 n32Result;

    // this API is implemented via device ioctl
    n32Result =
        DEV_n32Ioctl (
            hDevice,
            DEV_IOCTL_GET_ERROR
                );

    return n32Result;
}

/*****************************************************************************
*
*       DEV_n32Eof
*
*   This API allows the caller to query the eof status of a device
*   referenced by the hDevice handle
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*
*   Outputs:
*       A non-zero number indicates end-of-file has been reached.
*
*****************************************************************************/
N32 DEV_n32Eof(DEV_HDL hDevice)
{
    N32 n32Result;

    // this API is implemented via device ioctl
    n32Result =
        DEV_n32Ioctl (
            hDevice,
            DEV_IOCTL_GET_EOF
                );

    return n32Result;
}

/*****************************************************************************
*
*       DEV_vClearerror
*
*   This API allows the caller to clear the error status and eof status
*   indicators of a device referenced by the hDevice handle
*
*   Inputs:
*       hDevice - A handle to a connection or instance of a device driver.
*
*   Outputs:
*       None
*
*****************************************************************************/
void DEV_vClearerror(DEV_HDL hDevice)
{
    // this API is implemented via device ioctl
    DEV_n32Ioctl (
        hDevice,
        DEV_IOCTL_CLEAR_ERROR
            );

    return;
}

/*****************************************************************************
*
*       DEV_psFindNode
*
*       This function provides the ability to look-up a node given the
*       name provided. If a node is found it is returned. If a node
*       is not found, the first available entry is returned. If no
*       entry is available NULL is returned.
*
*   Inputs:
*       pcDevName - An ASCII null terminated string which represents the
*           device node to locate.
*
*   Outputs:
*       A pointer to the found node
*
*****************************************************************************/
static DEV_NODE_STRUCT *DEV_psFindNode (const char *pcDevName)
{
    DEV_NODE_STRUCT *psNode = (DEV_NODE_STRUCT *)NULL;
    UN8 un8Index = 0;

    // look-up name in static array
    for(un8Index = 0; un8Index < DEV_MAX_DEVICE_NODES; un8Index++)
    {
        // Check if this one matches
        if(strcmp(pcDevName, &gasDevices[un8Index].acDevName[0]) == 0)
        {
            // Match!
            psNode = &gasDevices[un8Index];
            break;
        }

        // Check if we don't already have an available one
        if(psNode == NULL)
        {
            // Is this entry available?
            if(strlen(&gasDevices[un8Index].acDevName[0]) == 0)
            {
                // Record first available node entry
                psNode = &gasDevices[un8Index];
            }
        }
    }

    return psNode;
}
