/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/*                           Proprietary & Confidential                       */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*       This module will contain all the Debug Handler
*       functionality.
*
*******************************************************************************/
#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "standard.h"
#include "com.h" // Requires generic COM (serial port) interface
#include "osal.h"
#include "osal_debug.h"
#include "osal_task.h"
#include "osal_core.h"
#include "osal_version.h"
#include "os_intf.h"

#include "osal_debughandler.h"
#include "_osal_debughandler.h"

#if OSAL_DEBUG == 1

/*****************************************************************************
*                                                                            *
*                           PUBLIC FUNCTIONS                                 *
*                                                                            *
*****************************************************************************/

/*****************************************************************************
*
*      DEBUG_eTaskInstall
*
*      This function is used to install the task into the system.
*
*      Inputs:
*               un32Priority - The unique Task Priority assigned to this task
*
*      Outputs:
*                   DEBUG_CODE_OK if installed successfully
*                   DEBUG_CODE_ERROR if not installed successfully
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM DEBUG_eTaskInstall( OSAL_TASK_PRIORITY_ENUM ePriority )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

#if OSAL_DEBUG == 1
    // Initialize all task memory to zero
    OSAL.bMemSet( (void *)&gsDebugTaskControl, 0,
                 sizeof(DEBUG_TASK_CONTROL_STRUCT) );

    // Create an install semaphore.
    eReturnCode = OSAL.eSemCreate(
        &gsDebugTaskControl.hInstallSemaphore,
        "Debug Install", 0, 1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Create the task.
        eReturnCode = OSAL.eTaskCreate(
                &gsDebugTaskControl.hTask,
                DEBUG_TASK_NAME,
                DEBUG_n32Task,
                (void*)&gsDebugTaskControl,
                DEBUG_TASK_STACK_SIZE,
                ePriority,
                DEBUG_TASK_OPTIONS);

        if(eReturnCode == OSAL_SUCCESS)
        {
            // Wait for task to install
            eReturnCode = OSAL.eSemTake(
                gsDebugTaskControl.hInstallSemaphore,
                OSAL_OBJ_TIMEOUT_INFINITE
                    );
        }

        // Destroy the semaphore (no longer needed)
        if(gsDebugTaskControl.hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemDelete(gsDebugTaskControl.hInstallSemaphore);
            gsDebugTaskControl.hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }
#endif

    return eReturnCode;
}

/*****************************************************************************
*
*       DEBUG_vTaskUninstall
*
*       This function is used to remove the Task from the system.
*
*       Inputs:
*               None
*
*       Outputs:
*               None
*
*       Notes:
*               This function runs in the context of the caller.
*
*****************************************************************************/
void DEBUG_vTaskUninstall( void )
{
#if OSAL_DEBUG == 1
    // Inform the task to delete (uninstall) itself
    OSAL.eTaskDelete(gsDebugTaskControl.hTask);
#endif
    return;
}


/*****************************************************************************
*
*       DEBUG_eTaskChangePriority
*
*       This function is used change the priority of the debug task.
*
*       Inputs:
*               eNewPriority - The new task priority assigned to this task
*
*       Outputs:
*               None
*
*       Notes:
*                   DEBUG_CODE_OK if priority changed successfully
*                   DEBUG_CODE_ERROR if not changed successfully
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM DEBUG_eTaskChangePriority( OSAL_TASK_PRIORITY_ENUM eNewPriority )
{
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

#if OSAL_DEBUG == 1
    // Change the priority of the task
    eReturnCode = OSAL.eTaskChangePriority(gsDebugTaskControl.hTask, eNewPriority);
#endif

    return eReturnCode;
}


#endif

/****************************************************************************
*
*       DEBUG_eRegisterDebughandler
*
*       This function is used to register the debug handler for the
*       associated Subsystem ID
*
*       Inputs:
*            pCallBackFunc - the callback function to be used by this handler
*            pvArg - Argument to be passed to pCallBackFunc
*
*       Outputs:
*               OSAL_ERROR = register unsuccessful
*               OSAL_SUCCESS = register successful
*
****************************************************************************/
OSAL_RETURN_CODE_ENUM DEBUG_eRegisterDebughandler(
    const char *pacName,
    OSAL_DEBUG_HANDLER hCallBackFunc,
    const void *pvArg
    )
{
#if OSAL_DEBUG == 0

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

#else

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    DEBUG_HANDLER_STRUCT *psDebugHndlr;

    // create a new Debug Handler Structure (node for LL)
    psDebugHndlr = DEBUG_psCreateNode(pacName);
    if (psDebugHndlr != NULL)
    {
        OSAL_OBJECT_HDL hTask;

        // Get this task's handle
        hTask = OSAL.hTaskGetHandle();

        // Lock Debug Control Structure
        eReturnCode = OSAL.eSemTake(
            gsDebugTaskControl.sShared.hDebugMutex,
            OSAL_OBJ_TIMEOUT_INFINITE);
        if(eReturnCode == OSAL_SUCCESS)
        {
            OSAL_TASK_INFO_STRUCT sTaskInfo;

            // Insert/Update handler in list
            eReturnCode = OSAL.eLinkedListAdd(
                gsDebugTaskControl.sShared.hDebugHandlerList,
                &psDebugHndlr->hMyEntry,
                psDebugHndlr);
            if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
            {
                OSAL_LINKED_LIST_ENTRY hEntry;

                // It's already there. Just modify it

                // Point to the existing node entry
                hEntry = psDebugHndlr->hMyEntry;

                // Destroy the one we already created
                DEBUG_vDestroyNode(psDebugHndlr);

                // Extract pointer of existing node
                psDebugHndlr = OSAL.pvLinkedListThis(hEntry);

                if(pacName != NULL)
                {
                    // Update name
                    DEBUG_bUpdateHandlerName(psDebugHndlr, pacName);
                }
            }
            else if(eReturnCode != OSAL_SUCCESS)
            {
                // Something is wrong
                // Return allocated memory
                DEBUG_vDestroyNode(psDebugHndlr);

                // Unlock debug control structure
                OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);

                return eReturnCode;
            }

            // If successful, update task debug info provided
            // but leave enabled flag alone.

            // Extract that task's info and modify it.
            eReturnCode = OSAL_eTaskGetInfo(hTask, &sTaskInfo);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Modify and set new task debug info
                sTaskInfo.sDebug.pacName = psDebugHndlr->pacName;
                sTaskInfo.sDebug.pvFxnCallback = hCallBackFunc;
                sTaskInfo.sDebug.pvArg = pvArg;
                OSALT_vTaskSetDebug(hTask, &sTaskInfo.sDebug);
            }

            // Unlock debug control structure
            OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);
        }
    }

#endif /* OSAL_DEBUG == 0 */

    return eReturnCode;
}
/****************************************************************************
*
*       DEBUG_bUnRegisterDebughandler
*
*       This function is used to unregister the debug handler.
*       NOTE: This function must be called from within the context of the task
*       that registered it
*
*       Inputs:
*               None
*
*       Outputs:
*
*             BOOLEAN bError
*
*               0 = Couldn't register
*               1 = Okay
****************************************************************************/
OSAL_RETURN_CODE_ENUM DEBUG_eUnRegisterDebughandler( void )
{

#if OSAL_DEBUG == 0

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;

#else

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_OBJECT_NOT_FOUND;
    DEBUG_HANDLER_STRUCT *psDebughandler;

    // Lock Debug Control Structure
    eReturnCode = OSAL.eSemTake(
        gsDebugTaskControl.sShared.hDebugMutex,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        eReturnCode = OSAL_OBJECT_NOT_FOUND;

        // Find it!
        psDebughandler = DEBUG_psFindDebugHandler();
        if (psDebughandler != NULL)
        {
            //  is this the handler selected?
            if ( gsDebugTaskControl.sShared.psSelectedHandler ==
                    psDebughandler )
            {
                // yes, switch to Normal mode before deleting
                // make the pointer to the selected debug handler NULL
                gsDebugTaskControl.sShared.psSelectedHandler =
                                (DEBUG_HANDLER_STRUCT *)NULL;
            }

            // remove it from the list
            eReturnCode = OSAL.eLinkedListRemove(psDebughandler->hMyEntry);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // free up the memory
                DEBUG_vDestroyNode(psDebughandler);
            }
        }

        // Unlock debug control structure
        OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);
    }

#endif /* OSAL_DEBUG == 0 */

    return eReturnCode;
}

/*****************************************************************************
*
*       DEBUG_vExitingDebughandler
*
*       This function is called when exiting Single Task Debug Mode
*       Note: This call can only be made within the DEBUG context.
*
*       Inputs:
*               None
*
*       Outputs:
*
*               None
*****************************************************************************/
void DEBUG_vExitingDebughandler(void)
{
#if OSAL_DEBUG == 1
    OSAL_OBJECT_HDL hTask;

    // Find who the calling tasks is, by getting its handle
    hTask = OSAL.hTaskGetHandle();
    if (hTask != OSAL_INVALID_OBJECT_HDL)
    {
        // This can only be done in the DEBUG task context
        if (hTask == gsDebugTaskControl.hTask)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Lock Debug Control Structure
            eReturnCode = OSAL.eSemTake(
                gsDebugTaskControl.sShared.hDebugMutex,
                OSAL_OBJ_TIMEOUT_INFINITE);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // switch mode to Normal
                // make the pointer to the selected debug handler NULL
                gsDebugTaskControl.sShared.psSelectedHandler =
                                (DEBUG_HANDLER_STRUCT *)NULL;

                // Unlock debug control structure
                OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);
            }
        }
    }
#endif /* OSAL_DEBUG == 1 */

    return;
}

/*****************************************************************************
*
*       DEBUG_vDisableOutputThisTask
*
*       This function is used to enable debug output for a task.
*
*       Inputs:
*               NONE
*
*       Outputs:
*
*               NONE
*****************************************************************************/
void DEBUG_vDisableOutputThisTask( void )
{
#if OSAL_DEBUG == 1

    DEBUG_vCtrlOutputThisTask(FALSE);

#endif /* OSAL_DEBUG == 1 */

    return;
}

/*****************************************************************************
*
*       DEBUG_vEnableOutputThisTask
*
*       This function is used to enable debug output for a task.
*
*       Inputs:
*               NONE
*
*       Outputs:
*
*               NONE
*****************************************************************************/
void DEBUG_vEnableOutputThisTask ( void )
{
#if OSAL_DEBUG == 1

    DEBUG_vCtrlOutputThisTask(TRUE);

#endif /* OSAL_DEBUG == 1 */

    return;
}

/*****************************************************************************
*
*       DEBUG_vCtrlOutputThisTask
*
*       This function is used to enable/disable debug output for a task.
*
*       Inputs:
*               BOOLEAN
*
*       Outputs:
*
*               NONE
*****************************************************************************/
void DEBUG_vCtrlOutputThisTask( BOOLEAN bEnable )
{
#if OSAL_DEBUG == 1

    OSAL_OBJECT_HDL hTask;
    OSAL_TASK_INFO_STRUCT sTaskInfo;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Find who the calling task is, by getting its handle
    hTask = OSAL.hTaskGetHandle();

    // Lock Debug Control Structure
    eReturnCode = OSAL.eSemTake(
        gsDebugTaskControl.sShared.hDebugMutex,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // DEBUG task context? Otherwise must be the context of the task to control
        if (hTask == gsDebugTaskControl.hTask)
        {
            // If so do not allow changes to it. Instead change
            // the task which is currently active. We can do this because
            // if we are in the DEBUG task we have to be doing so by already
            // having the debug mutex obtained.
            hTask = gsDebugTaskControl.sShared.psSelectedHandler->hTask;
        }

        // Extract that task's info and modify it.
        eReturnCode = OSAL_eTaskGetInfo(hTask, &sTaskInfo);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Modify and set new task debug info
            sTaskInfo.sDebug.bOutputEnabled = bEnable;
            OSALT_vTaskSetDebug(hTask, &sTaskInfo.sDebug);
        }

        // Unlock debug control structure
        OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);
    }
#endif /* OSAL_DEBUG == 1 */

    return;
}

/*****************************************************************************
*
*       DEBUG_bUpdateName
*
*       This is the function used to update a debug handler's
*       name.  This function may only be called while in the
*       context of the debug handler which is being renamed.
*       Note: This call can only be made within the DEBUG context.
*
*       Inputs:
*               const char *pacName - the name to set
*
*       Outputs:
*               BOOLEAN     bUpdated
*                               - TRUE if successful
*                               - otherwise FALSE
*
*****************************************************************************/
BOOLEAN DEBUG_bUpdateName( const char *pacName )
{
#if OSAL_DEBUG == 0
    BOOLEAN bUpdated = TRUE;

#else
    BOOLEAN bUpdated = FALSE;

    if(pacName != NULL)
    {
        OSAL_OBJECT_HDL hTask;

        // Find who the calling tasks is, by getting its handle
        hTask = OSAL.hTaskGetHandle();
        if (hTask != OSAL_INVALID_OBJECT_HDL)
        {
            // This can only be done in the DEBUG task context
            if (hTask == gsDebugTaskControl.hTask)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;

                // Lock Debug Control Structure
                eReturnCode = OSAL.eSemTake(
                    gsDebugTaskControl.sShared.hDebugMutex,
                    OSAL_OBJ_TIMEOUT_INFINITE);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // This has to be called from the debug context
                    DEBUG_HANDLER_STRUCT *psDebugHandler =
                        gsDebugTaskControl.sShared.psSelectedHandler;
                    if(psDebugHandler != NULL)
                    {
                        OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
                        OSAL_TASK_INFO_STRUCT sTaskInfo;

                        // Yep
                        bUpdated = DEBUG_bUpdateHandlerName(
                                psDebugHandler,
                                pacName
                            );
                        if(bUpdated == TRUE)
                        {
                            // Extract that task's info and modify it.
                            eReturnCode = OSAL_eTaskGetInfo(
                                psDebugHandler->hTask, &sTaskInfo);
                            if(eReturnCode == OSAL_SUCCESS)
                            {
                                // Modify with new name
                                sTaskInfo.sDebug.pacName =
                                    psDebugHandler->pacName;

                                OSALT_vTaskSetDebug(
                                    psDebugHandler->hTask, &sTaskInfo.sDebug);
                            }
                        }
                    }

                    // Unlock debug control structure
                    OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);
                }
            }
        }
    }
#endif

    return bUpdated;
}

/*****************************************************************************
*
*       DEBUG_pacSetCommandLineText
*
*       This function is used write text to the buffer used for command line
*       processing
*       Note: This call can only be made within the DEBUG context.
*
*       Inputs:
*               pacText - the text to write
*
*       Outputs:
*
*               the pointer to the beginning of the command
*               line buffer (NULL if unsuccessful)
*****************************************************************************/
char *DEBUG_pacSetCommandLineText( char *pacText )
{
#if OSAL_DEBUG == 0
    return pacText;

#else
    if (pacText != NULL)
    {
        OSAL_OBJECT_HDL hTask;

        // Find who the calling tasks is, by getting its handle
        hTask = OSAL.hTaskGetHandle();
        if (hTask != OSAL_INVALID_OBJECT_HDL)
        {
            // This can only be done in the DEBUG task context
            if (hTask == gsDebugTaskControl.hTask)
            {
                // This has to be called from the debug context
                strncpy(&gsDebugTaskControl.sInputMsg.acCommand[0],
                        pacText,
                        sizeof(gsDebugTaskControl.sInputMsg.acCommand));

                return &gsDebugTaskControl.sInputMsg.acCommand[0];
            }
        }
    }

    return NULL;
#endif
}

#if OSAL_DEBUG == 1

/*****************************************************************************
*                                                                            *
*                       FRIEND FUNCTIONS                                     *
*                                                                            *
*****************************************************************************/

/*****************************************************************************
*
*   DEBUG_vDebugHandler
*
*   This function is a debug handler that a function that has registered
*   itself with the debug handler can use to handle command prompt entries
*   in MENU_SINGLE_TASK_CONTROL mode.
*
*   Inputs:
*      *pcCmdLine - Command line entry string before CR/LF
*       *pvArg - A pointer to a registered argument when installing this
*          debug handler.
*
*   Outputs:
*       None.
*
*****************************************************************************/
void DEBUG_vDebugHandler( char *pcCmdLine, const void *pvArg )
{
    char *acTokens = " \n";

    // Make sure NULL pointer is not passed
    if (pcCmdLine != NULL)
    {
        // Tokenize the command line characters
        pcCmdLine = strtok( pcCmdLine, acTokens );

        if (pcCmdLine != NULL)
        {
            if (strcmp(pcCmdLine, "e") == 0)
            {
               OSAL.vControlOutputThisTask(TRUE);
               printf("Task output enabled\n");
            }
            else if (strcmp(pcCmdLine, "d") == 0)
            {
               printf("Task output disabled\n");
               OSAL.vControlOutputThisTask(FALSE);
            }
            else if ((strcmp(pcCmdLine, "help") == 0) ||
                    (strcmp(pcCmdLine, "HELP") == 0))
            {
                printf("\nAvailable commands:\n");
                printf("\te                     -\tenable output\n");
                printf("\td                     -\tdisable output\n");
                printf("\thelp                  -\tdisplay this message\n");
                printf("\texit                  -\texit this menu\n");
            }
            else if (strcmp(pcCmdLine, "exit") == 0)
            {
                OSAL.vDebugExitHandler();
            }
            else
            {
                printf("\nInvalid command.  Type 'help' for a list of commands.\n");
            }
        }
    }

    return;
}

/*****************************************************************************
*                                                                            *
*                       PRIVATE FUNCTIONS                                    *
*                                                                            *
*****************************************************************************/

/*****************************************************************************
*
*       DEBUG_n32Task
*
*       This is the Debug Task.
*
*       Inputs:
*               void    *pvArg // Requred by the OS
*
*       Outputs:
*               N32         // Required by OS
*
*****************************************************************************/
static N32 DEBUG_n32Task( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    int iPrintableChar, iControlChar;
    BOOLEAN bInitialized;
    int iChar;

    // Initialize this task!
    bInitialized = DEBUG_bTaskInitialize();
    if(bInitialized == FALSE)
    {
        // Unallocate task resources
        DEBUG_vTaskUninitialize();

        return !OSAL_TASK_REPORT_NO_ERROR;
    }

    // print the banner before we post the semaphore
    // if we try it after, we could be preempted during
    // the banner printing
    DEBUG_vPrintBanner( );

    // Allow installer waiting for the semaphore to run
    eReturnCode = OSAL.eSemGive(gsDebugTaskControl.hInstallSemaphore);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Unallocate task resources
        DEBUG_vTaskUninitialize();

        return !OSAL_TASK_REPORT_NO_ERROR;
    }

    for(;;)
    {
        // Point to the response processor structure
        DEBUG_TASK_INPUT_MSG_PROCESSING_STRUCT *psInputMsg =
            &gsDebugTaskControl.sInputMsg;

        // Check if we need to allocate a input msg structure
        if(psInputMsg->pacBufferStartPtr == NULL)
        {
            do
            {
                // Allocate a response buffer
                eReturnCode = OSAL.eMessageAllocate(
                    gsDebugTaskControl.hTaskInputQueue,
                    (void**)&psInputMsg->pacBufferStartPtr,
                    OSAL_QUEUE_FLAG_NONBLOCK );

                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Initialize buffer pointers
                    psInputMsg->pacBufferPtr =
                        psInputMsg->pacBufferStartPtr;
                    psInputMsg->pacBufferEndPtr =
                        psInputMsg->pacBufferStartPtr
                        + DEBUG_INPUT_MSG_BUFFER_SIZE;
                }
                else if(eReturnCode == OSAL_MESSAGE_NOT_AVAILABLE)
                {
                    // We have no more messages to allocate
                    // So we just pull one and trash it
                    BOOLEAN bOk = DEBUG_bThrowAwayCommand();
                    if(bOk == TRUE)
                    {
                        // Ok, now try again
                        continue;
                    }
                }
                else
                {
                    printf("\nDEBUG: Error with Input Msg "
                           "allocation! %s\n\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }

                break;

            } while ( TRUE );
        }

        // Wait for a character
        iChar = getchar();
        if(iChar == EOF)
        {
            // Something is wrong, or task is being shutdown
            printf("Error! Task cannot read from comm port (%d)\n",
                   iChar );
            OSAL.eTaskDelay(1); // Allow other tasks to run
            break;
        }

        // Get got a character, report
        OSAL.eTaskReport(OSAL_TASK_REPORT_NO_ERROR);

        // Check if valid data was received
        // Characters we care about must be either
        // printable or control characters only
        iPrintableChar = isprint(iChar);
        iControlChar = iscntrl(iChar);

        if ((iPrintableChar != 0) || (iControlChar != 0))
        {
            if (iChar == DEBUG_BACKSPACE)
            {
                if (psInputMsg->pacBufferPtr !=
                    psInputMsg->pacBufferStartPtr)
                {
                    psInputMsg->pacBufferPtr--;

                    // print a blank space and a backspace to erase
                    // the removed character from the screen
                    printf(" \b");
                }
                else
                {
                    // buffer's empty. We don't want to let them erase
                    // the prompt so we'll reprint the prompt.  This
                    // operation will be the same, regardless of which
                    // mode the debug system is in.
                    putchar(DEBUG_PROMPT);
                }
            }
            else if (iChar == DEBUG_TAB)
            {
                char *pacHistoricalCommand;
                UN32 un32Size;

                // Grab something from the queue
                eReturnCode = OSAL.eQueueGet(
                    gsDebugTaskControl.hTaskInputQueue,
                    (void**)&pacHistoricalCommand,
                    &un32Size,
                    OSAL_OBJ_TIMEOUT_NONE);
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // We got one! Free current message
                    // and use this one instead
                    eReturnCode = OSAL.eMessageFree(
                        psInputMsg->pacBufferStartPtr );
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        psInputMsg->pacBufferStartPtr =
                            pacHistoricalCommand;

                        // Initialize buffer pointers
                        psInputMsg->pacBufferPtr =
                            psInputMsg->pacBufferStartPtr;
                        psInputMsg->pacBufferEndPtr =
                            psInputMsg->pacBufferStartPtr
                            + DEBUG_INPUT_MSG_BUFFER_SIZE;

                        // Print a new prompt
                        DEBUG_vPrompt();

                        // Print historical command
                        do
                        {
                            putchar(*psInputMsg->pacBufferPtr);
                            psInputMsg->pacBufferPtr++;

                        } while(psInputMsg->pacBufferPtr <
                                (psInputMsg->pacBufferStartPtr +
                                 un32Size) );
                    }
                }
                else
                {
                    char *pacEndOfCmd = psInputMsg->pacBufferPtr;

                    // Sorry! No history
                    DEBUG_vPrompt();

                    // Reprint current command (if any)
                    psInputMsg->pacBufferPtr =
                        psInputMsg->pacBufferStartPtr;

                    do
                    {
                        putchar(*psInputMsg->pacBufferPtr);
                        psInputMsg->pacBufferPtr++;

                    } while(psInputMsg->pacBufferPtr <
                                pacEndOfCmd );
                }
            }
            else if(iChar == DEBUG_LINE_FEED)
            {
                // Just drop this character, we dont need it
            }
            else
            {
                // If it's the first character and it is
                // not printable do not save it
                if( !((psInputMsg->pacBufferPtr ==
                       psInputMsg->pacBufferStartPtr)
                    && !iPrintableChar) )
                {
                    // Record this character if there is room
                    if(psInputMsg->pacBufferPtr <=
                       psInputMsg->pacBufferEndPtr)
                    {
                        *psInputMsg->pacBufferPtr++ = iChar;
                    }
                    else
                    {
                        // my buffer is full.
                        // sorry, I can't save this character

                        // go back one, print a blank space and go
                        // back again to erase the last character
                        // from the screen
                        printf("\b \b");
                    }
                }
            }

            // Check for end of input message terminator
            if(iChar == DEBUG_CARRIAGE_RETURN)
            {
                // Compute the command size just entered (number of
                // characters) this will be the number of input
                // characters minus the <lf> (1 chars)
                // If there are not at least 1 or more chars (> 1)
                // then the command size is zero. Don't historically
                // remember it then.
                UN32 un32MsgSize =
                    ( ( psInputMsg->pacBufferPtr -
                     psInputMsg->pacBufferStartPtr ) > 1 ) ?
                            ( psInputMsg->pacBufferPtr -
                                psInputMsg->pacBufferStartPtr - 1 ) : 0;
                if(un32MsgSize == 0)
                {
                    // Just reset input message ptrs, do not
                    // queue a nothing message

                    // Initialize buffer pointers
                    psInputMsg->pacBufferPtr =
                        psInputMsg->pacBufferStartPtr;
                    psInputMsg->pacBufferEndPtr =
                        psInputMsg->pacBufferStartPtr
                        + DEBUG_INPUT_MSG_BUFFER_SIZE;
                }
                else
                {
                    // <cr>
                    putchar('\n');

                    // Before we make a copy of the string to send over,
                    // Set the command string acCommand to all '/0'
                    // Which is effectively a buffer flush
                    OSAL.bMemSet( psInputMsg->acCommand, '\0',
                        DEBUG_TASK_INPUT_MSG_LENGTH );

                    // Since string tokenizers actually modify the
                    // original contents of a string, we need to make
                    // a copy of the one we are going to send off
                    // for processing. This way we can tokenize, edit
                    // alter all we like but the original command
                    // will remain in tact (for historical purposes)/
                    OSAL.bMemCpy(psInputMsg->acCommand,
                        psInputMsg->pacBufferStartPtr, un32MsgSize);

                    // Queue command line before processing it. Message
                    // is put at the end of the queue.
                    eReturnCode = OSAL.eQueuePut(
                        psInputMsg->pacBufferStartPtr,
                        un32MsgSize);
                    if(eReturnCode == OSAL_SUCCESS)
                    {
                        // Lock Debug Control Structure
                        eReturnCode = OSAL.eSemTake(
                            gsDebugTaskControl.sShared.hDebugMutex,
                            OSAL_OBJ_TIMEOUT_INFINITE);
                        if(eReturnCode == OSAL_SUCCESS)
                        {
                            // Process command based on mode
                            if(gsDebugTaskControl.sShared.psSelectedHandler ==
                                            NULL)
                            {
                                // Normal mode...
                                DEBUG_vProcessInputMsgNormalMode(
                                    psInputMsg->acCommand);

                                // Unlock debug control structure
                                OSAL.eSemGive(
                                    gsDebugTaskControl.sShared.hDebugMutex);
                            }
                            else
                            {
                                OSAL_OBJECT_HDL hTask =
                                    gsDebugTaskControl.sShared.psSelectedHandler->hTask;

                                // Unlock debug control structure
                                OSAL.eSemGive(
                                    gsDebugTaskControl.sShared.hDebugMutex);

                                // Single task mode...
                                DEBUG_vProcessInputMsgSingleTaskMode(
                                    psInputMsg->acCommand, hTask);
                            }

                        }

                        // Command is queued, go get another
                        psInputMsg->pacBufferStartPtr =
                            (char *)NULL;
                    }
                    else
                    {
                        // Something wacked happened
                        // Free this command buffer
                        eReturnCode = OSAL.eMessageFree(
                            psInputMsg->pacBufferStartPtr);
                        if(eReturnCode == OSAL_SUCCESS)
                        {
                            psInputMsg->pacBufferStartPtr =
                                (char *)NULL;
                        }
                    }
                } // (un32MsgSize != 0)

                // Print prompt
                DEBUG_vPrompt();

            } // end if(iChar == DEBUG_CARRIAGE_RETURN

        } //  if( iPrintableChar || bControlChar )

    } // forever loop

    return OSAL_TASK_REPORT_NO_ERROR;
}

/*****************************************************************************
*
*       DEBUG_vPrompt
*
*****************************************************************************/
static void DEBUG_vPrompt( void )
{
    const char *pacPrompt = "";
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Lock Debug Control Structure
    eReturnCode = OSAL.eSemTake(
        gsDebugTaskControl.sShared.hDebugMutex,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        if(gsDebugTaskControl.sShared.psSelectedHandler != NULL)
        {
            // Set prompt to selected handler's name
            pacPrompt = gsDebugTaskControl.sShared.psSelectedHandler->pacName;
        }

        // Unlock debug control structure
        OSAL.eSemGive(gsDebugTaskControl.sShared.hDebugMutex);
    }

    // print prompt
    printf("\n%s%c", pacPrompt, DEBUG_PROMPT);

    return;
}

/*****************************************************************************
*
*       DEBUG_bTaskInitialize
*
*       This is the function used to Initialize the CPY Task
*
*       Inputs:
*               None
*
*       Outputs:
*               BOOLEAN     bSuccess
*                               - TRUE if successful
*                               - otherwise FALSE       *
*****************************************************************************/
static BOOLEAN DEBUG_bTaskInitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN16 un16ReportingInterval = 0;
    DEBUG_HANDLER_STRUCT *psDebugHndlr;
    OSAL_TASK_DEBUG_STRUCT sDebug;

    // Initialize response processing structure
    gsDebugTaskControl.sInputMsg.pacBufferStartPtr = (char *)NULL;
    gsDebugTaskControl.sInputMsg.pacBufferPtr = (char *)NULL;
    gsDebugTaskControl.sInputMsg.pacBufferEndPtr = (char *)NULL;

    // Create a control semaphore.
    eReturnCode = OSAL.eSemCreate(
        &gsDebugTaskControl.sShared.hDebugMutex,
        "Debug Ctrl", 0, 1, OSAL_SEM_OPTION_NONE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Create a linked list that is circular and only allows unique items
    eReturnCode = OSAL.eLinkedListCreate(
        &gsDebugTaskControl.sShared.hDebugHandlerList,
         "Debug Handler",
         DEBUG_n16Compare,
         OSAL_LL_OPTION_UNIQUE
             );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Create response queue
    eReturnCode = OSAL.eQueueCreate(
        &gsDebugTaskControl.hTaskInputQueue,
        OSAL_NAME_PREFIX"Debug Resp",
        DEBUG_TASK_INPUT_MSG_BLOCKS,
        DEBUG_TASK_INPUT_MSG_LENGTH,
        OSAL_QUEUE_OPTION_VARIABLE_SIZE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Install the OSAL debug handler

    // Create a Debug Handler Structure (node for LL)
    psDebugHndlr = DEBUG_psCreateNode("Debug Console Menu");
    if (psDebugHndlr == NULL)
    {
        return FALSE;
    }

    // Debug Handler is always enabled
    sDebug.pvFxnCallback = OSALD_vDebugMenuHandler;
    sDebug.pvArg = NULL;
    sDebug.pacName = psDebugHndlr->pacName;
    sDebug.bOutputEnabled = TRUE;
    OSALT_vTaskSetDebug(psDebugHndlr->hTask, &sDebug);

    // Insert new handler into list
    eReturnCode = OSAL.eLinkedListAdd(
        gsDebugTaskControl.sShared.hDebugHandlerList,
        &psDebugHndlr->hMyEntry,
        psDebugHndlr);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Something is wrong
        // Return allocated memory
        DEBUG_vDestroyNode(psDebugHndlr);
        return FALSE;
    }

    // Make DEBUG control available
    eReturnCode = OSAL.eSemGive(
        gsDebugTaskControl.sShared.hDebugMutex);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Register Task with OSAL
    eReturnCode = OSAL.eTaskRegister(
                   un16ReportingInterval,
                   DEBUG_vSleepHandler,
                   DEBUG_vWakeupHandler,
                   DEBUG_vShutdownHandler,
                   NULL );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // All is well
    return TRUE;
}

/*****************************************************************************
*
*       DEBUG_vTaskUninitialize
*
*       This function will un-initialize any components that
*       have been created at start up.
*
*       Inputs:
*               None
*
*       Outputs:
*               None
*
*****************************************************************************/
static void DEBUG_vTaskUninitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Unregister Task with OSAL
    OSAL.eTaskUnregister();

    // Obtain shared data resource
    eReturnCode = OSAL.eSemTake(
        gsDebugTaskControl.sShared.hDebugMutex,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        // That's all folks!
        return;
    }

    // Delete all elements in the linked list and then finally
    // delete the linked list itself.
    if(gsDebugTaskControl.sShared.hDebugHandlerList != OSAL_INVALID_OBJECT_HDL)
    {
        // Deallocate and destroy elements in the linked list
        eReturnCode = OSAL.eLinkedListRemoveAll(
            gsDebugTaskControl.sShared.hDebugHandlerList,
            DEBUG_vDestroyNode);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Destroy the linked list itself
            eReturnCode = OSAL.eLinkedListDelete(
                gsDebugTaskControl.sShared.hDebugHandlerList);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gsDebugTaskControl.sShared.hDebugHandlerList =
                    OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Done with shared data
    // Delete the semaphore (no longer needed) - force post if necessary
    if(gsDebugTaskControl.sShared.hDebugMutex != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsDebugTaskControl.sShared.hDebugMutex);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsDebugTaskControl.sShared.hDebugMutex = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Delete the command queue
    if(gsDebugTaskControl.hTaskInputQueue != OSAL_INVALID_OBJECT_HDL)
    {
        // Delete queue
        eReturnCode = OSAL.eQueueDelete(gsDebugTaskControl.hTaskInputQueue);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsDebugTaskControl.hTaskInputQueue = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Delete the install semaphore (if it exists) - force post
    if(gsDebugTaskControl.hInstallSemaphore != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(gsDebugTaskControl.hInstallSemaphore);
        if(eReturnCode == OSAL_SUCCESS)
        {
            gsDebugTaskControl.hInstallSemaphore = OSAL_INVALID_OBJECT_HDL;
        }
    }


    return;
}

/*******************************************************************************
*
*   DEBUG_bSelectRegisteredTaskHandler
*
*******************************************************************************/
static BOOLEAN DEBUG_bSelectRegisteredTaskHandler ( void *pvData, void *pvArg )
{
    DEBUG_EXECUTE_HANDLER_STRUCT *psExecuteHandler =
        (DEBUG_EXECUTE_HANDLER_STRUCT *)pvArg;
    DEBUG_HANDLER_STRUCT *psThisDebugHandler =
        (DEBUG_HANDLER_STRUCT *)pvData;
    BOOLEAN bContinue = FALSE;

    // Check if arg and data are valid
    if ((psExecuteHandler != NULL) && (psThisDebugHandler != NULL))
    {
        if (psExecuteHandler->bIsIndex == TRUE)
        {
            psExecuteHandler->uCriteria.sIndex.un32Index++;

            // Is this the one?
            if (psExecuteHandler->uCriteria.sIndex.un32EntryNumber !=
                    psExecuteHandler->uCriteria.sIndex.un32Index)
            {
                // Continue iteration
                bContinue = TRUE;
            }
        }
        else
        {
            int iRes;
            iRes = strcmp(psThisDebugHandler->pacName,
                          psExecuteHandler->uCriteria.sName.pacName);
            if (iRes != 0)
            {
                bContinue = TRUE;
            }
        }

        if (bContinue == FALSE)
        {
            // Record the handler we found
            psExecuteHandler->psDebugHandler = psThisDebugHandler;
        }
    }

    // Stop the iteration (error)
    return bContinue;
}

/*****************************************************************************
*
*       DEBUG_vProcessInputMsgNormalMode
*
*       This function is used to process in DEBUG_MODE_NORMAL a msg received
*       from the Debug Comm port.
*
*       This function is always called with the hDebugMutex locked.
*
*****************************************************************************/
static void DEBUG_vProcessInputMsgNormalMode( char *pacCmd )
{
    char    *pcArg; // used for parsing

    pcArg = strtok( pacCmd, "\n" );

    if((pcArg == NULL) || (!strcmp(pcArg,"help")))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN8 un8Count = 0;

        // 1. print out the banner
        DEBUG_vPrintBanner( );

        // 2. Print out the registered tasks
        //    (they can be selected for single task debugging)
        // Iterate the list dumping each element individually

        eReturnCode = OSAL.eLinkedListIterate(
            gsDebugTaskControl.sShared.hDebugHandlerList,
            DEBUG_bPrintRegisteredTask, &un8Count);
        if(eReturnCode == OSAL_NO_OBJECTS)
        {
            printf("No registered tasks!\n");
        }
        else
        {
            // 3. print out the commands
            printf(" d\t\tDisable output for all registered tasks.\n");
            printf(" e\t\tEnable output for all registered tasks.\n");
        }

        printf(" help\t\tReprint this menu.\n");
    }
    else if (!strcmp(pcArg,"d"))
    {
        // user entered 'd'
        // this means they want to disable output for all registered tasks
        OSAL.eLinkedListIterate(gsDebugTaskControl.sShared.hDebugHandlerList,
            DEBUG_bCtrlTaskOutput, (void *)(size_t)FALSE);
    }
    else if (!strcmp(pcArg,"e"))
    {
        // user entered 'e'
        // this means they want to enable output for all registered tasks
        OSAL.eLinkedListIterate(gsDebugTaskControl.sShared.hDebugHandlerList,
            DEBUG_bCtrlTaskOutput, (void *)(size_t)TRUE);
    }
    else
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN32 un32EntryNumber;
        DEBUG_EXECUTE_HANDLER_STRUCT sExecuteHandler;

        // Maybe they entered a number or task name????
        // Valid task entries are one or higher

        sExecuteHandler.psDebugHandler = NULL;

        // Convert entry to an integer
        un32EntryNumber = (UN32)strtol(pcArg, NULL, 10);
        if (un32EntryNumber > 0)
        {
            // Initialize execution struct for index based searching
            sExecuteHandler.bIsIndex = TRUE;
            sExecuteHandler.uCriteria.sIndex.un32EntryNumber = un32EntryNumber;
            sExecuteHandler.uCriteria.sIndex.un32Index = 0;
        }
        else
        {
            sExecuteHandler.bIsIndex = FALSE;
            sExecuteHandler.uCriteria.sName.pacName = pcArg;
        }

        // now we need to see if the number entered makes sense
        // (i.e. is there a node at that position in the LL)
        eReturnCode = OSAL.eLinkedListIterate(
            gsDebugTaskControl.sShared.hDebugHandlerList,
            DEBUG_bSelectRegisteredTaskHandler, &sExecuteHandler);

        // If we found one with this entry number it we will be
        // equal to the one we were searching for
        if((eReturnCode == OSAL_SUCCESS) &&
            (sExecuteHandler.psDebugHandler != NULL))
        {
            // print out the selected debug handler's menu
            // The debug handler's are responsible for their own menus
            // so all we have to do is send the "help" command to the
            // debug handler selected and it'll print out its menu

            OSAL_RETURN_CODE_ENUM eReturnCode;
            OSAL_TASK_INFO_STRUCT sTaskInfo;

            // switch to SINGLE TASK MODE
            // this node is the selected debug handler
            gsDebugTaskControl.sShared.psSelectedHandler =
                sExecuteHandler.psDebugHandler;

            // Extract that task's debug info and use it
            eReturnCode = OSAL_eTaskGetInfo(
                sExecuteHandler.psDebugHandler->hTask,
                &sTaskInfo);
            if(eReturnCode == OSAL_SUCCESS)
            {
                if (sTaskInfo.sDebug.pvFxnCallback !=
                                (OSAL_DEBUG_HANDLER)NULL)
                {
                    char *pacCmdLine;
                    pacCmdLine = DEBUG_pacSetCommandLineText( "help\n" );

                    // Set this task as the current handler
                    gsDebugTaskControl.sShared.psSelectedHandler =
                        sExecuteHandler.psDebugHandler;

                    // Execute handler callback with "help"
                    sTaskInfo.sDebug.pvFxnCallback(pacCmdLine,
                        sTaskInfo.sDebug.pvArg);
                }
                else
                {
                    printf("This task does not have a callback handler!\n");

                    // switch mode to Normal
                    // make the pointer to the selected debug handler NULL
                    gsDebugTaskControl.sShared.psSelectedHandler =
                                    (DEBUG_HANDLER_STRUCT *)NULL;
                }
            }

            return;
        }

        // The number entered is greater than the number of nodes
        // so it is an invalid entry
        printf("That entry was invalid. "
               "Type 'help' for a list of tasks and commands.\n");
    }

    return;
}

/*****************************************************************************
*
*       DEBUG_vProcessInputMsgSingleTaskMode
*
*       This function is used to process in DEBUG_MODE_SINGLE_TASK a msg
*       received from the Debug Comm port.
*
*****************************************************************************/
static void DEBUG_vProcessInputMsgSingleTaskMode(
    char *pacCmd,
    OSAL_OBJECT_HDL hTask
    )
{
    char *pacInput;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_TASK_INFO_STRUCT sTaskInfo;

    // Extract that task's info
    eReturnCode = OSAL_eTaskGetInfo(hTask, &sTaskInfo);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Obtain current task debug information
        OSAL_DEBUG_HANDLER pvFxnCallback = sTaskInfo.sDebug.pvFxnCallback;
        const void *pvArg = sTaskInfo.sDebug.pvArg;

        // copy & parse message
        pacInput = strtok( pacCmd, "\n" );

        // If callback is valid, call it.
        if(pvFxnCallback != (OSAL_DEBUG_HANDLER)NULL)
        {
            // call the debug handler's callback, passing in the input message
            // and the debug handler's argument
            pvFxnCallback(pacInput, pvArg);
        }
    }

    return;
}

/*******************************************************************************
*
*   DEBUG_bPrintRegisteredTask
*
*******************************************************************************/
static BOOLEAN DEBUG_bPrintRegisteredTask ( void *pvData, void *pvArg )
{
    UN8 *pun8Count = (UN8 *)pvArg;
    DEBUG_HANDLER_STRUCT *psThisDebugHandler =
        (DEBUG_HANDLER_STRUCT *)pvData;

    // Check if arg is valid
    if((pun8Count != NULL) && (psThisDebugHandler != NULL))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_TASK_INFO_STRUCT sTaskInfo;

        eReturnCode = OSAL_eTaskGetInfo(psThisDebugHandler->hTask, &sTaskInfo);
        // Check if data is valid
        if(eReturnCode == OSAL_SUCCESS)
        {
           char cOutput = 'D';

           // un8Count keeps track of the node number we're on
           (*pun8Count)++;

           // check if the output flag is Enabled for this node
           if(sTaskInfo.sDebug.bOutputEnabled == TRUE)
           {
              cOutput = 'E';
           }

           // print the task description
           printf("%2d\t%c\t%s\n", *pun8Count, cOutput,
               sTaskInfo.sDebug.pacName);
        }
    }

    // Keep iterating
    return TRUE;
}

#if OSAL_REMOTE_CONSOLE == 1
/*******************************************************************************
*
*   DEBUG_bFileRegisteredTask
*
*******************************************************************************/
static BOOLEAN DEBUG_bFileRegisteredTask ( void *pvData, void *pvArg )
{
    DEBUG_HANDLER_STRUCT *psThisDebugHandler =
        (DEBUG_HANDLER_STRUCT *)pvData;
    DEBUG_HANDLER_ITERATOR_STRUCT *psIterator =
        (DEBUG_HANDLER_ITERATOR_STRUCT *)pvArg;

    // Check if arg is valid
    if((pvArg != NULL) && (pvData != NULL))
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_TASK_INFO_STRUCT sTaskInfo;

        eReturnCode = OSAL_eTaskGetInfo(psThisDebugHandler->hTask, &sTaskInfo);
        // Check if data is valid
        if(eReturnCode == OSAL_SUCCESS)
        {
           // Counter keeps track of the node number we're on
           psIterator->un8Counter++;
           // print the task description
           fprintf(psIterator->psFile, "%d %s\n",
               psIterator->un8Counter,
               sTaskInfo.sDebug.pacName);
        }
    }

    // Keep iterating
    return TRUE;
}
#endif

/*****************************************************************************
*
*       DEBUG_vPrintBanner
*
*       This function is used to print out the Banner
*       Note: for now it's just a dummy banner. In the future we might do one
*       of the following
*       1.  Move to a seperate file that is updated regularly
*       2.  Build the list by requesting version data from other places
*
*       Inputs:
*               NONE
*
*       Outputs:
*
*               NONE
*****************************************************************************/
static void DEBUG_vPrintBanner( void )
{
    UN8 un8MajorVersion, un8MinorVersion, un8SubVersion;

    // Get OSAL version
    OSAL.pacGetVersion(&un8MajorVersion, &un8MinorVersion, &un8SubVersion);

    printf ("\n");
    printf ("/////////////////////////////////////////////////////////\n");
    printf ("//           Sirius-XM Satellite Radio                 //\n");
    printf ("//                                                     //\n");
    printf ("//             OSAL Version %02d.%02d.%02d"
            "                   //\n",
            un8MajorVersion, un8MinorVersion, un8SubVersion);
    printf ("//      OSAL Build Date: %11.11s %8.8s          //\n",
            __DATE__, __TIME__);
    printf ("//                                                     //\n");
    printf ("/////////////////////////////////////////////////////////\n");
}

/****************************************************************************
*
*       DEBUG_psCreateNode
*
*       This function is used to create a debug handler node (instance)
*
*       Inputs:
*             OSAL_DEBUG_HANDLER vDebugCallbackHandler
*
*       Outputs:
*
*             pointer to the newly created Debug Handler DEBUG_HANDLER_STRUCT
*
****************************************************************************/
static DEBUG_HANDLER_STRUCT *DEBUG_psCreateNode( const char *pacName )
{
    DEBUG_HANDLER_STRUCT *psDebughandler = (DEBUG_HANDLER_STRUCT *)NULL;
    OSAL_OBJECT_HDL hTask;

    // debug handler's are distinguished by registering task's handle
    hTask = OSAL.hTaskGetHandle();
    if (hTask != OSAL_INVALID_OBJECT_HDL)
    {
        psDebughandler =
            (DEBUG_HANDLER_STRUCT *)OSALC_pvMemoryAllocate(
            OSAL_NAME_PREFIX"Debug Node",
            sizeof(DEBUG_HANDLER_STRUCT), FALSE);
        if(psDebughandler != NULL)
        {
            BOOLEAN bUpdated;

            // Initialize parameters
            psDebughandler->hTask = hTask;
            psDebughandler->hMyEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
            psDebughandler->pacName = NULL;

            if(pacName == NULL)
            {
                pacName = "Unknown";
            }

            // Update (set) name
            bUpdated = DEBUG_bUpdateHandlerName(
                psDebughandler,
                pacName
                    );
            if(bUpdated == FALSE)
            {
                // Free previously allocated node;
                DEBUG_vDestroyNode(psDebughandler);
                psDebughandler = (DEBUG_HANDLER_STRUCT *)NULL;
            }
        }
    }
    else
    {
         // couldn't get handle of calling task.
         // either something is wrong with OS or the calling function was
         // called from outside a task

         // anyway there's nothing we can do about it.
    }

    return psDebughandler;

}

/*****************************************************************************
*
*       DEBUG_vDestroyNode
*
*       This is the function used to remove all allocated memories and destroy
*       the linked list node
*
*       Inputs:
*               void *pvData
*
*       Outputs:
*
*               None
*****************************************************************************/
static void DEBUG_vDestroyNode(void *pvData)
{
    DEBUG_HANDLER_STRUCT *psDebughandler = (DEBUG_HANDLER_STRUCT *)pvData;

    // Invalidate entry handle
    psDebughandler->hMyEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

    // Free node name memory
    if(psDebughandler->pacName != NULL)
    {
        OSALC_vMemoryFree((void *)psDebughandler->pacName);
        psDebughandler->pacName = NULL;
    }

    // Free node data
    OSALC_vMemoryFree(psDebughandler);
}

/*****************************************************************************
*
*       DEBUG_n16Compare
*
*       This function is used to compare the handles of 2 debug handlers.
*       This is used when searching the LL for a specific node.
*
*
*       Inputs:
*               *pvDH1 - pointer to one of the DEBUG HANDLER being compared
*               *pvDH2 - pointer to the other DEBUG HANDLER being compared
*
*       Outputs:
*
*               n16Result - The difference between the task priority passed in
*                           and the priority of the struct currently being
*                           searched
*****************************************************************************/
static N16 DEBUG_n16Compare( void *pvDH1, void *pvDH2)
{
    N16 n16Result = N16_MIN;
    const DEBUG_HANDLER_STRUCT
        *psDebughandler1 = (const DEBUG_HANDLER_STRUCT *)pvDH1,
        *psDebughandler2 = (const DEBUG_HANDLER_STRUCT *)pvDH2;

    // Check pointers
    if((psDebughandler1 != NULL) && (psDebughandler2 != NULL))
    {
        // Condition result
        if (psDebughandler1->hTask == psDebughandler2->hTask)
        {
            n16Result = 0;
        }
        else
        {
            n16Result = -1;
        }
    }

    return n16Result;
}

/*******************************************************************************
*
*   DEBUG_bCtrlTaskOutput
*
*******************************************************************************/
static BOOLEAN DEBUG_bCtrlTaskOutput ( void *pvData, void *pvArg )
{
    BOOLEAN bEnable = (BOOLEAN)(size_t)pvArg;
    DEBUG_HANDLER_STRUCT *psThisDebugHandler =
        (DEBUG_HANDLER_STRUCT *)pvData;

    // Check if data is valid
    if(psThisDebugHandler != NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_TASK_INFO_STRUCT sTaskInfo;

        // Extract that task's info and modify it.
        eReturnCode = OSAL_eTaskGetInfo(psThisDebugHandler->hTask, &sTaskInfo);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Modify and set new task debug info
            sTaskInfo.sDebug.bOutputEnabled = bEnable;
            OSALT_vTaskSetDebug(psThisDebugHandler->hTask, &sTaskInfo.sDebug);

            // Keep iterating
            return TRUE;
        }
    }

    // Stop iterating
    return FALSE;
}

/*****************************************************************************
*
*       DEBUG_psFindDebugHandler
*
*       This function is used to find a debug handler.
*
*       Inputs:
*               NONE
*
*       Outputs:
*
*               DEBUG_HANDLER_STRUCT *
*****************************************************************************/
static DEBUG_HANDLER_STRUCT *DEBUG_psFindDebugHandler( void )
{
    DEBUG_HANDLER_STRUCT *psDebughandler =
        (DEBUG_HANDLER_STRUCT *)NULL;
    OSAL_OBJECT_HDL hTask;

    // find calling task's handle
    hTask = OSAL.hTaskGetHandle();
    if (hTask != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_LINKED_LIST_ENTRY hFoundEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY; // search from start(top) of list;
        DEBUG_HANDLER_STRUCT sDebugHandler; // this is temp object for search

        // init task handle in our temp object
        sDebugHandler.hTask = hTask;

        // we got the calling task's handle, now
        // search LL to get associated Debug Handler Node
        eReturnCode = OSAL.eLinkedListSearch(
            gsDebugTaskControl.sShared.hDebugHandlerList,
            &hFoundEntry, &sDebugHandler);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Found it!
            psDebughandler =
                (DEBUG_HANDLER_STRUCT *)OSAL.pvLinkedListThis(hFoundEntry);
        }
    }

    return psDebughandler;
}

/*****************************************************************************
*
*       DEBUG_bThrowAwayCommand
*
*****************************************************************************/
static BOOLEAN DEBUG_bThrowAwayCommand ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char *pacThrowAwayCommand;
    UN32 un32Size;

    // We just pull one and trash it
    // Grab something from the queue
    eReturnCode = OSAL.eQueueGet(
        gsDebugTaskControl.hTaskInputQueue,
        (void**)&pacThrowAwayCommand,
        &un32Size,
        OSAL_OBJ_TIMEOUT_INFINITE);
    if(eReturnCode == OSAL_SUCCESS)
    {
        // Free this command buffer
        eReturnCode = OSAL.eMessageFree(pacThrowAwayCommand);
        if(eReturnCode == OSAL_SUCCESS)
        {
            return TRUE;
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*       DEBUG_vSleepHandler
*
*****************************************************************************/
static void DEBUG_vSleepHandler( void *pvArg )
{
    // Nothing to do

    return;
}

/*****************************************************************************
*
*       DEBUG_vWakeupHandler
*
*****************************************************************************/
static void DEBUG_vWakeupHandler( void *pvArg )
{
    // Nothing to do

    return;
}

/*****************************************************************************
*
*       DEBUG_vShutdownHandler
*
*****************************************************************************/
static void DEBUG_vShutdownHandler( void *pvArg )
{
    // Nothing to do

    return;
}

/*****************************************************************************
*
*       DEBUG_bUpdateHandlerName
*
*       This is the function used to update a debug handler's
*       name.
*
*****************************************************************************/
static BOOLEAN DEBUG_bUpdateHandlerName(
    DEBUG_HANDLER_STRUCT *psDebugHandler,
    const char *pacName
        )
{
    BOOLEAN bUpdated = FALSE;
    size_t tPrevNameLen = 0;
    size_t tNewNameLen;

    // Ensure this is a valid pointer
    if (psDebugHandler->pacName != NULL)
    {
        // Calculate the length of the
        // previous name (minus the NULL character)
        tPrevNameLen = strlen(psDebugHandler->pacName);
    }

    // Calculate the length of the new
    // name (minus the NULL character)
    tNewNameLen = strlen(pacName);

    // Only perform allocation if the new name
    // is longer than the previous name
    if (tNewNameLen > tPrevNameLen)
    {
        char *pacNewName;

        // Allocate space for the new name
        pacNewName = (char *)OSALC_pvMemoryAllocate(
            DEBUG_NODE_NAME_STRING,
            tNewNameLen + 1, FALSE);

        // Were we able to allocate space
        // for the new name?
        if (pacNewName != NULL)
        {
            // Copy in name
            strncpy(pacNewName, pacName, tNewNameLen + 1);

            // get rid of the old stuff (if we have anything)
            if(psDebugHandler->pacName != NULL)
            {
                OSALC_vMemoryFree(psDebugHandler->pacName);
            }

            // update with the new stuff
            psDebugHandler->pacName = pacNewName;

            // Flag this as a success
            bUpdated = TRUE;
        }
    }
    else
    {
        // Was there any name here?
        if (psDebugHandler->pacName != NULL)
        {
            // No allocation needed
            // Copy in name
            strncpy(psDebugHandler->pacName, pacName, tPrevNameLen + 1);
        }
        else
        {
            // If there was no any name, and a new name's length is not 
            // bigger (meaning it is also 0), then we're really trying
            // to assign "nothing" to "nothing". 
            // Just say: "It is ok, we did it!"
        }

        bUpdated = TRUE;
    }

    return bUpdated;
}

#endif /* OSAL_DEBUG == 1 */
