/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the CRC implementation for OSAL
 *
 ******************************************************************************/
#include <standard.h>
#include <string.h>

#include "osal_version.h"

#include "osal.h"
#include "os_intf.h"

#include "_osal_crc.h"
#include "osal_crc.h"

/*****************************************************************************
*
*   OSAL_n16CompareCRC
*
*   A LL comparator that is used by the core to match a pre-existing CRC
*   object to a request for a new one as provided in an OSAL_CRC_INFO_STRUCT.
*
*****************************************************************************/
N16 OSAL_n16CompareCRC (
    void *pvObj,
    void *pvInfo
        )
{
#if OSAL_CRC == 1
    OSAL_OBJECT_STRUCT *psObj =
        (OSAL_OBJECT_STRUCT *)pvObj;
    OSAL_CRC_INFO_STRUCT *psCRC =
        (OSAL_CRC_INFO_STRUCT *)pvInfo;

    if (psObj->puInfo->sCRC.eCRCType == psCRC->eCRCType)
    {
        return 0;
    }
#endif
    return 1;
}

/*****************************************************************************
*
*   OSAL_eGetCRC
*
*   This is the function used to get access to a supported CRC
*   implementation.  This function will "install" the requested CRC
*   if necessary, otherwise if the requested CRC is already present in
*   OSAL this function will increment the reference count of that CRC.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eGetCRC (
    OSAL_OBJECT_HDL *phCRC,
    OSAL_CRC_TYPE_ENUM eCRCType
        )
{
#if OSAL_CRC == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hCRC;
    OSAL_CRC_INFO_STRUCT sCRC;
    OSAL_CRC_INFO_STRUCT *psCRC;

    // Check input
    if(phCRC == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phCRC = OSAL_INVALID_OBJECT_HDL;

    // We're looking for this CRC type,
    // so prep our info pointer
    // with a temporary structure
    sCRC.eCRCType = eCRCType;
    psCRC = &sCRC;

    // Validate input first
    if (eCRCType >= OSAL_CRC_TYPE_MAX)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Construct a unique name for the CRC requested
    snprintf( &acName[0], sizeof(acName),
              OSAL_NAME_PREFIX"%s",
              pacEnumText(eCRCType));

    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_CRC,
        0,
        &acName[0],
        &hCRC,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psCRC
            );
    // If the object was obtained and the CRC hasn't been
    // initialized yet
    if((eReturnCode == OSAL_SUCCESS) && (psCRC->pvCRCData == NULL))
    {
        // Initialize it
        eReturnCode = eInitCRCObject( &acName[0], psCRC, eCRCType );
        if (eReturnCode != OSAL_SUCCESS)
        {
            // Remove and destroy the object
            OSALC_eRemoveObject(hCRC);
            return eReturnCode;
        }

        strncpy(&psCRC->acName[0],
            (const char *)&acName[0], sizeof(psCRC->acName));

    }

    if (psObj != NULL)
    {
        // Extract the info pointer for the object
        psCRC = &psObj->puInfo->sCRC;

        // Increment the reference count
        psCRC->tReferenceCount++;

        // Populate the object
        *phCRC = hCRC;

        return OSAL_SUCCESS;
    }

    return OSAL_ERROR;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   OSAL_eReleaseCRC
*
*   This function is called in order to release access to a supported
*   CRC implementation.  This function will "uninstall" the requested CRC
*   if this is the last referece to it.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eReleaseCRC (
    OSAL_OBJECT_HDL hCRC
        )
{
#if OSAL_CRC == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_CRC_INFO_STRUCT *psCRC;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hCRC, OSAL_OBJECT_TYPE_CRC);
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Extract the info pointer for the object
    psCRC = &psObj->puInfo->sCRC;

    // Decrement the usage count and determine if this
    // CRC object may be removed from the system
    if (psCRC->tReferenceCount > 0)
    {
        psCRC->tReferenceCount--;
    }

    if (psCRC->tReferenceCount == 0)
    {
        // Uninitialize the object
        vUnInitCRCObject( psCRC );

        // Remove and destroy the object
        eReturnCode = OSALC_eRemoveObject(hCRC);
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*****************************************************************************
*
*   OSAL_bInitializeCRC
*
*   This function is used to initialize a new CRC calculation.  The actual
*   calculation is dependent on which type of CRC is in use; thus, the CRC
*   interface is used to in order compute the correct initial value.
*
*****************************************************************************/
BOOLEAN OSAL_bInitializeCRC (
    OSAL_OBJECT_HDL hCRC,
    OSAL_CRC_RESULT *ptInitialValue
        )
{
#if OSAL_CRC == 1
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_CRC_INFO_STRUCT *psCRC;
    OSAL_CRC_IMPL_STRUCT *psImpl;
    BOOLEAN bSuccess = FALSE;


    if (ptInitialValue == NULL)
    {
        return FALSE;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hCRC, OSAL_OBJECT_TYPE_CRC);
    if(psObj == NULL)
    {
        return FALSE;
    }

    // Extract the info pointer for the object
    psCRC = &psObj->puInfo->sCRC;

    // Extract the CRC Implementation
    psImpl = (OSAL_CRC_IMPL_STRUCT *)psCRC->pvCRCData;

    if (psImpl != (OSAL_CRC_IMPL_STRUCT *)NULL)
    {
        // Call the appropriate initializer
        bSuccess =
            psImpl->tInitialize( ptInitialValue );
    }

    return bSuccess;
#else
    return FALSE;
#endif
}

/*****************************************************************************
*
*   OSAL_tComputeCRC
*
*   This function is used to compute the requested CRC type based upon the
*   input data.  The actual CRC calculation is dependent on which type of
*   CRC is in use; thus, the CRC interface is used to in order compute the
*   CRC.
*
*****************************************************************************/
OSAL_CRC_RESULT OSAL_tComputeCRC (
    OSAL_OBJECT_HDL hCRC,
    OSAL_CRC_RESULT tInputCRC,
    OSAL_BUFFER_HDL hBuffer,
    size_t tByteOffset,
    size_t tNumBytesToProcess,
    size_t *ptBytesProcessed
        )
{
#if OSAL_CRC == 1
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_BUFFER_ITERATOR_ARG_STRUCT sArg;

    // Verify inputs
    if ( (hBuffer == OSAL_INVALID_BUFFER_HDL)
        || (tNumBytesToProcess == 0)
        || (ptBytesProcessed == NULL)
       )
    {
        return tInputCRC;
    }

    // Initialize the number of bytes
    // we've processed
    *ptBytesProcessed = 0;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(hCRC, OSAL_OBJECT_TYPE_CRC);
    if(psObj == NULL)
    {
        return (OSAL_CRC_RESULT)(0);
    }

    // Set init CRC value
    sArg.tResultCRC = tInputCRC;

    // Extract the info pointer for the object
    sArg.psCRC = &psObj->puInfo->sCRC;

    // Extract the CRC Implementation
    sArg.psImpl = (OSAL_CRC_IMPL_STRUCT *)sArg.psCRC->pvCRCData;

    if (sArg.psImpl != (OSAL_CRC_IMPL_STRUCT *)NULL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Prepare argument
        sArg.tByteOffset = tByteOffset;
        sArg.tNumBytesToProcess = tNumBytesToProcess;
        sArg.tBytesProcessed = 0;

        eReturnCode =
            OSAL.eBufferBlocksIterate(hBuffer, eBufferIteratorCallback,
                    (void*)&sArg);
        if (eReturnCode == OSAL_SUCCESS)
        {
            *ptBytesProcessed = sArg.tBytesProcessed;
        }
    }

    return sArg.tResultCRC;
#else
    // Verify inputs
    if ( (hBuffer == OSAL_INVALID_BUFFER_HDL)
        || (tNumBytesToProcess == 0)
        || (ptBytesProcessed == NULL)
        )
    {
        return tInputCRC;
    }

    // Clear the number of bytes we've processed.
    // This indicates the error to caller
    *ptBytesProcessed = 0;

    return (OSAL_CRC_RESULT)(0);
#endif
}

/*******************************************************************************
*
*   OSAL_eCRCList
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eCRCList ( void )
{
#if (OSAL_CRC == 1) && (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hCRCList;
    UN32 un32CRCs = 0, un32CRCNumber = 0;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of CRCs and list each entry

    // Grab the CRC linked list
    hCRCList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_CRC);
    if(hCRCList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of CRCs in the list
    eReturnCode = OSAL.eLinkedListItems(hCRCList, &un32CRCs);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_CRC, &sStatistics);

    // Check that one or more elements exist in the list
    if(un32CRCs > 0)
    {
        // Print title
        printf("=======================\n");
        printf("| OSAL CRC List       |\n");
        printf("=======================\n");
    }

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hCRCList,
        OSALS_bPrintCRC, &un32CRCNumber);
    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print summary
        printf("%u CRCs(s) currently allocated by the system.\n",
                    un32CRCs);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
               sStatistics.un32Current,
               sStatistics.un32Minimum,
               sStatistics.un32Maximum );

    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

#if (OSAL_CRC == 1) && (OSAL_DEBUG == 1)
/*******************************************************************************
*
*   OSALS_bPrintCRC
*
*******************************************************************************/
BOOLEAN OSALS_bPrintCRC ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    UN32 *pun32CRCNumber = (UN32 *)pvArg;
    const char *pacCreatorObjName = "Unknown";

    // Check if argument is valid
    if(pun32CRCNumber == NULL)
    {
        return FALSE;
    }

    // Prepare CRC number
    (*pun32CRCNumber)++;

    // Check if object is valid
    if(psObj != NULL)
    {
        // Point to this object's info
        OSAL_CRC_INFO_STRUCT sCRC;

        // Copy object info I have into object info provided to
        // OS-specific API. That API can then modify it with what
        // it knows, or do nothing.
        sCRC = psObj->puInfo->sCRC;

        // Extract the creator task's name
        psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
        if(psCreatorObj != NULL)
        {
            // Extract name of creator
#if OSAL_OBJECT_TRACKING == 1
            pacCreatorObjName = psCreatorObj->pacName;
#else
            pacCreatorObjName = "Unknown";
#endif
        }

        printf("CRC #%u\n", *pun32CRCNumber);
        printf("Name = '%s' (hdl = 0x%p)\n",
#if OSAL_OBJECT_TRACKING == 1
               psObj->pacName,
#else
               "Unknown",
#endif
               psObj
                   );
        printf("\tCreator = %s\n", pacCreatorObjName);
        printf("\tCRC '%s' (%u) ReferenceCount = %u\n",
               sCRC.acName, sCRC.eCRCType, sCRC.tReferenceCount);
        puts("");
    }

    // Keep iterating
    return TRUE;
}

#endif /* (OSAL_DEBUG == 1) */

/*****************************************************************************
                             PRIVATE FUNCTIONS
*****************************************************************************/
/* None of these are needed if CRC is not supported */
#if (OSAL_CRC == 1)

/*****************************************************************************
*
*   eInitCRCObject
*
*   Initializes a CRC object which has been created already via
*   OSALC_hCreateObject.  This function initializes the reference count,
*   plugs in the CRC impl requested, and initializes the CRC impl's
*   lookup table if necessary.
*
*****************************************************************************/
static OSAL_RETURN_CODE_ENUM eInitCRCObject (
    const char *pacName,
    OSAL_CRC_INFO_STRUCT *psCRC,
    OSAL_CRC_TYPE_ENUM eType
        )
{
    OSAL_CRC_IMPL_STRUCT *psImpl =
        (OSAL_CRC_IMPL_STRUCT *)NULL;
    size_t tIndex = (size_t)eType;

    if (eType >= OSAL_CRC_TYPE_MAX)
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Track the type of CRC requested
    psCRC->eCRCType = eType;

    // Get the CRC implementation requested
    // (the OSAL_CRC_TYPE_ENUM type is defined
    // using specific array indicies)
    psImpl = (OSAL_CRC_IMPL_STRUCT *)&gsCRCs[tIndex];

    if (psImpl == (OSAL_CRC_IMPL_STRUCT *)NULL)
    {
        // Should never happen!
        return OSAL_ERROR;
    }

    // Initialize the reference count
    psCRC->tReferenceCount = 0;

    // Only generate a lookup table if necessary
    if (   ( psImpl->tLookupTableSize > 0 )
        && ( psImpl->tGenerateTable != NULL )
       )
    {
        BOOLEAN bOk;
        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

        // Construct a unique name for the CRC lookup table
        snprintf( &acName[0], sizeof(acName),
                  "%s:Tbl",
                  &pacName[0] );

        // Allocate memory for the lookup table
        psCRC->pun8LookupTable = (UN8 *)
            OSALC_pvMemoryAllocate(
                &acName[0],
                psImpl->tLookupTableSize,
                TRUE );

        if (psCRC->pun8LookupTable == (UN8 *)NULL)
        {
            vUnInitCRCObject( psCRC );

            return OSAL_ERROR_OUT_OF_MEMORY;
        }

        // Generate the lookup table
        bOk = psImpl->tGenerateTable (
                &psCRC->pun8LookupTable[0] );
        if (bOk == FALSE)
        {
            vUnInitCRCObject( psCRC );

            return OSAL_ERROR;
        }

        // Plug in the correct CRC implementation
        psCRC->pvCRCData = (void *)psImpl;
    }

    return OSAL_SUCCESS;
}

/*****************************************************************************
*
*   vUnInitCRCObject
*
*   Uninitializes a CRC object by freeing the CRC Impl's lookup table
*   if necessary and clearing the Impl pointer.
*
*****************************************************************************/
static void vUnInitCRCObject (
    OSAL_CRC_INFO_STRUCT *psCRC
        )
{
    if (psCRC != NULL)
    {
        // If the table is present, free it & clear it
        if (psCRC->pun8LookupTable != (UN8 *)NULL)
        {
            OSALC_vMemoryFree( (void *)psCRC->pun8LookupTable );
            psCRC->pun8LookupTable = NULL;
        }

        psCRC->pvCRCData = NULL;
    }

    return;
}

/*****************************************************************************
*
*   pacEnumText
*
*   This is a local function which simply maps an enumerated type to
*   a textual representation for formatting the enumerated type.
*
*****************************************************************************/
static const char *pacEnumText(
    OSAL_CRC_TYPE_ENUM eCRCType
        )
{
    const char *pacReturnString = NULL;

    switch(eCRCType)
    {
        case OSAL_CRC_TYPE_CCITT_CRC16:
        {
            pacReturnString = CCITT_CRC16_NAME;
        }
        break;

        case OSAL_CRC_TYPE_NAVTEQ_CRC16:
        {
            pacReturnString = NAVTEQ_CRC16_NAME;
        }
        break;

        case OSAL_CRC_TYPE_ISO3309_CRC32:
        {
            pacReturnString = ISO3309_CRC32_NAME;
        }
        break;

        default:
        {
            pacReturnString = "unknown";
        }
        break;

    }

    return pacReturnString;
}

/*****************************************************************************
*
*   eBufferIteratorCallback
*
*****************************************************************************/
static OSAL_BUFFER_ITERATOR_RESULT_ENUM eBufferIteratorCallback(
    void *pvData,
    size_t tDataSize,
    void *pvArg
        )
{
    OSAL_BUFFER_ITERATOR_ARG_STRUCT *psArg =
        (OSAL_BUFFER_ITERATOR_ARG_STRUCT*)pvArg;
    OSAL_BUFFER_ITERATOR_RESULT_ENUM eReturnCode =
        OSAL_BUFFER_ITERATOR_RESULT_KEEP;

    if (psArg->tByteOffset >= tDataSize)
    {
        psArg->tByteOffset -= tDataSize;
    }
    else
    {
        UN8 *pun8Data = (UN8*)pvData;
        
        // Adjust the buffer based on the provided size
        tDataSize -= psArg->tByteOffset;
        pun8Data += psArg->tByteOffset;

        // Since now the offset is no longer needed
        psArg->tByteOffset = 0;

        // Adjust the data size to do not process more than requested
        if (tDataSize > psArg->tNumBytesToProcess)
        {
            tDataSize = psArg->tNumBytesToProcess;
            eReturnCode = OSAL_BUFFER_ITERATOR_RESULT_STOP;
        }

        if (tDataSize > 0)
        {
            // Compute the requested CRC
            psArg->tResultCRC =
                psArg->psImpl->tCompute(
                psArg->tResultCRC,
                pun8Data,
                tDataSize,
                &psArg->psCRC->pun8LookupTable[0]
                    );

            // Update iteration arg
            psArg->tBytesProcessed += tDataSize;
            psArg->tNumBytesToProcess -= tDataSize;
        }
    }

    return eReturnCode;
}

/*****************************************************************************
*
*   bGenerateCCITT_CRC16LookupTable
*
*   This function calculates the CCITT-CRC16 lookup table, which is
*   utilized to quickly calculate the CRC of incoming packets.
*
*****************************************************************************/
static BOOLEAN bGenerateCCITT_CRC16LookupTable (
    UN8 *pun8LookupTable
        )
{
    CCITT_CRC16 tResult, *ptLookupTable;
    size_t tLookupTableSize = CCITT_CRC16_NUM_LOOKUP_ENTRIES;
    UN16 un16ByteIndex = 0;
    UN8 un8BitIndex = 0;

    // Cast the table to the appropriate type
    ptLookupTable = (CCITT_CRC16 *)pun8LookupTable;

    // Generate CRC table using big brained maths...
    for (un16ByteIndex = 0; un16ByteIndex < tLookupTableSize; un16ByteIndex++)
    {
        tResult = (CCITT_CRC16) un16ByteIndex << 8;
        for (un8BitIndex = 0; un8BitIndex < 8; un8BitIndex++)
        {
            if (tResult & CCITT_CRC16_COMPUTE_MASK)
            {
                tResult = CCITT_CRC16_COMPUTE_VALUE ^ (tResult << 1);
            }
            else
            {
                tResult = tResult << 1;
            }
        }
        // Store the result in the lookup table
        ptLookupTable[un16ByteIndex] = tResult;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bInitializeCCITT_CRC16
*
*   This function initializes a new CCITT-CRC16 calculation.
*
*****************************************************************************/
static BOOLEAN bInitializeCCITT_CRC16 (
    OSAL_CRC_RESULT *ptInitialValue
        )
{
    // Just copy out the initial value
    *ptInitialValue = (OSAL_CRC_RESULT)
        CCITT_CRC16_INITIAL_VALUE;

    return TRUE;
}

/*****************************************************************************
*
*   tComputeCCITT_CRC16
*
*   This function calculates the CRC of a given buffer accoring to the
*   CCITT-CRC16 specification.
*
*****************************************************************************/
static OSAL_CRC_RESULT tComputeCCITT_CRC16 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        )
{
    size_t tIndex;
    CCITT_CRC16 tCRC = (CCITT_CRC16)tCurrentCRC;
    CCITT_CRC16 *ptLookupTable =
        (CCITT_CRC16 *)pun8LookupTable;

    // Iterate over the buffer provided
    for (tIndex = 0; tIndex < tBufferSize; tIndex++)
    {
        // Compute the CRC for each byte
        tCRC = (tCRC<<8)
                ^ ptLookupTable[
                      pun8Buffer[tIndex] ^(UN8)(tCRC>>8)];
    }

    return (OSAL_CRC_RESULT)tCRC;
}

/*****************************************************************************
*
*       bInitializeNAVTEQ_CRC16
*
*       This function initializes a new NAVTEQ-CRC16 calculation.
*
*****************************************************************************/
static BOOLEAN bInitializeNAVTEQ_CRC16(
    OSAL_CRC_RESULT *ptInitialValue
        )
{
    // Just copy out the initial value
    *ptInitialValue = (OSAL_CRC_RESULT)
        NAVTEQ_CRC16_INITIAL_VALUE;

    return TRUE;
}

/*****************************************************************************
*
*       tComputeNAVTEQ_CRC16
*
*       This function calculates the CRC of a given buffer according to the
*       NAVTEQ-CRC16 specification.
*
*****************************************************************************/
static OSAL_CRC_RESULT tComputeNAVTEQ_CRC16 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        )
{
    size_t tIndex;
    NAVTEQ_CRC16 tCRC = (NAVTEQ_CRC16)tCurrentCRC;
    NAVTEQ_CRC16 *ptLookupTable =
        (NAVTEQ_CRC16 *)pun8LookupTable;

    // Iterate over the buffer provided
    for (tIndex = 0; tIndex < tBufferSize; tIndex++)
    {
        // Compute the CRC for each byte
        tCRC = (NAVTEQ_CRC16)((tCRC>>8)
                ^ ptLookupTable[(NAVTEQ_CRC16)
                      (pun8Buffer[tIndex] ^ (tCRC & 0xFF)) & 0xFF ]);
    }

    return (OSAL_CRC_RESULT)tCRC;
}

/*****************************************************************************
*
*       bInitializeXMAPPID_CRC16
*
*       This function initializes a new XMAppID-CRC16 calculation.
*
*****************************************************************************/
static BOOLEAN bInitializeXMAPPID_CRC16(
    OSAL_CRC_RESULT *ptInitialValue
        )
{
    // Just copy out the initial value
    *ptInitialValue = (OSAL_CRC_RESULT)
        XMAPPID_CRC16_INITIAL_VALUE;

    return TRUE;
}

/*****************************************************************************
*
*       tComputeXMAPPID_CRC16
*
*       This function calculates the CRC of a given buffer according to
*       Appendix A of AN-10_ Parsing_Application_Data_r1.7_2007-10-20.pdf
*
*****************************************************************************/
static OSAL_CRC_RESULT tComputeXMAPPID_CRC16 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        )
{
    UN8 un8Data;
    size_t tIndex;
    XMAPPID_CRC16 tCRC = (XMAPPID_CRC16)tCurrentCRC;
    XMAPPID_CRC16 *ptLookupTable =
        (XMAPPID_CRC16 *)pun8LookupTable;

    // Iterate over the buffer provided
    for (tIndex = 0; tIndex < tBufferSize; tIndex++)
    {
        // Compute the CRC for each byte
        un8Data = pun8Buffer[tIndex] ^ (tCRC >> (XMAPPID_CRC16_WIDTH - 8));
        tCRC = ptLookupTable[un8Data] ^ (tCRC << 8);
    }

    /* the final remainder is the CRC */
    //tCRC = (tCRC ^ XMAPPID_CRC16_FINAL_XOR_VALUE);

    return (OSAL_CRC_RESULT)tCRC;

}

/*****************************************************************************
*
*   bGenerateISO3309_CRC32LookupTable
*
*   This function calculates the ISO-3309 CRC32 lookup table, which is
*   utilized to quickly calculate the CRC of incoming packets.
*
*****************************************************************************/
static BOOLEAN bGenerateISO3309_CRC32LookupTable (
    UN8 *pun8LookupTable
        )
{
    ISO3309_CRC32 tResult, *ptLookupTable;
    size_t tLookupTableSize = ISO3309_CRC32_NUM_LOOKUP_ENTRIES;
    UN16 un16ByteIndex = 0;
    UN8 un8BitIndex = 0;

    // Cast the table to the appropriate type
    ptLookupTable = (ISO3309_CRC32 *)pun8LookupTable;

    // Generate CRC table using big brained maths...
    for (un16ByteIndex = 0; un16ByteIndex < tLookupTableSize; un16ByteIndex++)
    {
        tResult = (ISO3309_CRC32) un16ByteIndex;
        for (un8BitIndex = 0; un8BitIndex < 8; un8BitIndex++)
        {
            if (tResult & ISO3309_CRC32_COMPUTE_MASK)
            {
                tResult = ISO3309_CRC32_COMPUTE_VALUE ^ (tResult >> 1);
            }
            else
            {
                tResult = tResult >> 1;
            }
        }
        // Store the result in the lookup table
        ptLookupTable[un16ByteIndex] = tResult;
    }

    return TRUE;
}

/*****************************************************************************
*
*   bInitializeISO3309_CRC32
*
*   This function initializes a new ISO-3309 CRC32 calculation.
*
*****************************************************************************/
static BOOLEAN bInitializeISO3309_CRC32(
    OSAL_CRC_RESULT *ptInitialValue
        )
{
    // Just copy out the initial value
    *ptInitialValue = (OSAL_CRC_RESULT)
        ISO3309_CRC32_INITIAL_VALUE;

    return TRUE;
}

/*****************************************************************************
*
*   tComputeISO3309_CRC32
*
*   This function calculates the CRC of a given buffer according to
*   Appendix D http://www.w3.org/TR/PNG/
*
*****************************************************************************/
static OSAL_CRC_RESULT tComputeISO3309_CRC32 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        )
{
    size_t tIndex;
    ISO3309_CRC32 tCRC = (ISO3309_CRC32)tCurrentCRC;
    ISO3309_CRC32 *ptLookupTable =
        (ISO3309_CRC32 *)pun8LookupTable;

    // Iterate over the buffer provided
    for (tIndex = 0; tIndex < tBufferSize; tIndex++)
    {
        // Compute the CRC for each byte
        tCRC = ptLookupTable[(tCRC ^ pun8Buffer[tIndex]) & 0xFF] ^ (tCRC >> 8);
    }

    return (OSAL_CRC_RESULT)tCRC;
}

#endif /* OSAL_CRC == 1 */
