/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*	GLOBAL HEADER
*******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef OSAL_CORE_H_
#define OSAL_CORE_H_

  /**************/
 /** INCLUDES **/
/**************/
#include <stdlib.h>
#include <stddef.h>

#include "standard.h"

#include "osal.h"

#include "osal_queue.h"
#include "osal_ll.h"

#if OSAL_CRC == 1
#include "osal_crc.h"
#endif /* OSAL_CRC == 1 */

#if OSAL_BUFFER == 1
#include "osal_buf.h"
#endif /* OSAL_BUFFER == 1 */

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// OSAL Object Types
typedef enum osal_object_type_enum
{
    // These are the object types which are tracked in linked lists
    // by the OSAL management functions.
    OSAL_OBJECT_TYPE_FIRST,

    OSAL_OBJECT_TYPE_TASK = OSAL_OBJECT_TYPE_FIRST,
    OSAL_OBJECT_TYPE_QUEUE,
    OSAL_OBJECT_TYPE_SEM,
    OSAL_OBJECT_TYPE_TIMER,
    OSAL_OBJECT_TYPE_MEMORY,
    OSAL_OBJECT_TYPE_LINKED_LIST,

#if OSAL_INTERRUPT == 1
    OSAL_OBJECT_TYPE_INTERRUPT,
#endif /* OSAL_INTERRUPT == 1 */

#if OSAL_LOG == 1
    OSAL_OBJECT_TYPE_LOG,
#endif /* OSAL_LOG == 1 */

#if OSAL_BUFFER == 1
    OSAL_OBJECT_TYPE_BLOCK_POOL,
#endif /* OSAL_BUFFER == 1 */

#if OSAL_CRC == 1
    OSAL_OBJECT_TYPE_CRC,
#endif /* OSAL_CRC == 1 */

    // This is the size or number of types supported
    OSAL_OBJECT_TYPE_SIZE,

    // This is the last type supported
    OSAL_OBJECT_TYPE_LAST = OSAL_OBJECT_TYPE_SIZE - 1,

    // This is used whenever an object type is unknown
    OSAL_OBJECT_TYPE_UNKNOWN = OSAL_OBJECT_TYPE_SIZE,

} OSAL_OBJECT_TYPE_ENUM;

/* General Statistics Structure */
typedef struct osal_statistics_struct
{
    UN32 un32Current;
    UN32 un32Maximum;
    UN32 un32Minimum;

} OSAL_STATISTICS_STRUCT;

// OSAL Core Statistics Utility Update Operation types
typedef enum osalc_update_operation_enum
{
    OSALC_UPDATE_OPERATION_ADD,
    OSALC_UPDATE_OPERATION_SUB,
    OSALC_UPDATE_OPERATION_ABSOLUTE

} OSALC_UPDATE_OPERATION_ENUM;

// OSAL Object Information based on object type.
typedef union osal_object_info_union
{
    OSAL_TASK_INFO_STRUCT sTask;
    OSAL_QUEUE_PRIVATE_INFO_STRUCT sQueue;
    OSAL_SEM_INFO_STRUCT sSem;
    OSAL_TIMER_INFO_STRUCT sTimer;
    OSAL_MEMORY_INFO_STRUCT sMemory;
    OSAL_LL_INFO_STRUCT sLL;

#if OSAL_INTERRUPT == 1
    OSAL_INT_INFO_STRUCT sInt;
#endif /* OSAL_INTERRUPT == 1 */

#if OSAL_LOG == 1
    OSAL_LOG_INFO_STRUCT sLog;
#endif /* OSAL_LOG == 1 */

#if OSAL_BUFFER == 1
    OSAL_BLOCK_POOL_PRIVATE_INFO_STRUCT sBlockPool;
#endif /* OSAL_BUFFER == 1 */

#if OSAL_CRC == 1
    OSAL_CRC_INFO_STRUCT sCRC;
#endif /* OSAL_CRC == 1 */

} OSAL_OBJECT_INFO_UNION;

typedef struct osal_object_struct
{
    // This objects' entry pointer
    OSAL_LINKED_LIST_ENTRY hEntry;

    // Task creator and task name info
    OSAL_OBJECT_HDL hCreatorObj;

    // OS-Specific Object Handle (OS port layer)
    OS_OBJECT_HDL hOS;

    // This object's type
    OSAL_OBJECT_TYPE_ENUM eType;

    // This object's magic number
    UN32 un32MagicNumber;

    // This object's specific info
    OSAL_OBJECT_INFO_UNION *puInfo;

    // This object's user data (if any)
    void *pvUserMemory;

#if OSAL_OBJECT_TRACKING == 1
    // This object's name
    const char *pacName;
#endif

} OSAL_OBJECT_STRUCT;

// Memory Usage Statistics

typedef enum osal_memory_type_enum
{
    OSAL_MEMORY_TYPE_OS,
    OSAL_MEMORY_TYPE_OSAL

} OSAL_MEMORY_TYPE_ENUM;

typedef struct osal_memory_struct
{
    OSAL_STATISTICS_STRUCT sBlocks;
    OSAL_STATISTICS_STRUCT sBytes;

} OSAL_MEMORY_STRUCT;

typedef struct osal_memory_statistics_struct
{
    OSAL_MEMORY_STRUCT sActual;
    OSAL_MEMORY_STRUCT sUser;

} OSAL_MEMORY_STATISTICS_STRUCT;

typedef struct osal_core_struct
{
    // OSAL Core Structure mutex
    OSAL_OBJECT_HDL hCoreMutex; // Mutex handle

    // OSAL Start Handlers
    OSAL_START_HANDLER_STRUCT sStartHandlers;

    // Array of Object lists
    OSAL_OBJECT_HDL ahObjectList[OSAL_OBJECT_TYPE_SIZE];

#if OSAL_OBJECT_TRACKING == 1
    // Various object statistics
    OSAL_STATISTICS_STRUCT asStatistics[OSAL_OBJECT_TYPE_SIZE];
#endif

    // Memory Allocation Statistics (OSAL)
    OSAL_MEMORY_STATISTICS_STRUCT sMemory;

#if OSAL_FILE_SYSTEM == 1
    // Temporary Directory Path
    char *pacTempPath;
#endif

} OSALC_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

BOOLEAN OSALC_bBootstrap( const void *pvArg );
BOOLEAN OSALC_bInitialize( void );
void OSALC_vUninitialize( void );
BOOLEAN OSALC_bLockCore( void );
void OSALC_vUnlockCore( void );
void OSALC_vInitializeStatistics( OSAL_STATISTICS_STRUCT *psStatistics );

void OSALC_vUpdateStatistics( OSALC_UPDATE_OPERATION_ENUM eUpdateOperation,
                              UN32 un32Current,
                              OSAL_STATISTICS_STRUCT *psStatistics );

#if OSAL_OBJECT_TRACKING == 1
void OSALC_vGetObjectStatistics(
                                 OSAL_OBJECT_TYPE_ENUM eType,
                                 OSAL_STATISTICS_STRUCT *psStatistics );
#endif

void OSALC_vUpdateMemoryAllocation(
    OSAL_MEMORY_TYPE_ENUM eType,
    OSALC_UPDATE_OPERATION_ENUM eOperationType,
    size_t tSize
        );

BOOLEAN OSALC_bMemoryUsage(
    UN32 *pun32CurrentAllocatedBlocks,
    UN32 *pun32CurrentActualAllocatedBlocks,
    UN32 *pun32CurrentUserBytes,
    UN32 *pun32CurrentActualBytes,
    UN32 *pun32MaxAllocatedBlocks,
    UN32 *pun32MaxActualAllocatedBlocks,
    UN32 *pun32MaxUserBytes,
    UN32 *pun32MaxActualBytes,
    UN32 *pun32TotalSystemBytes
        );

OSAL_OBJECT_HDL OSALC_hGetObjectList( OSAL_OBJECT_TYPE_ENUM eType );

OSAL_RETURN_CODE_ENUM OSALC_eCreateObject(
    OSAL_OBJECT_TYPE_ENUM eType,
    size_t tAdditionalBytes,
    const char *pacName,
    OSAL_OBJECT_HDL *phObj,
    OSAL_OBJECT_STRUCT **ppsObj,
    OSAL_OBJECT_INFO_UNION **ppsInfo
        );

OSAL_RETURN_CODE_ENUM OSALC_eRemoveObject( OSAL_OBJECT_HDL hObj );

OSAL_OBJECT_STRUCT *OSALC_psGetObjectFromHandle( OSAL_OBJECT_HDL hObj,
                                                 OSAL_OBJECT_TYPE_ENUM eType );
OSAL_OBJECT_STRUCT *OSALC_psFindObjectFromName( OSAL_OBJECT_TYPE_ENUM eType,
                                                const char *pacName );
const char *OSALC_pacGetPriorityName (
    OSAL_TASK_PRIORITY_ENUM ePriority
        );

#if OSAL_FILE_SYSTEM == 1
void OSALC_vUpdateFileSystemTempPath(
    char *pacTempDirPath
        );
char *OSALC_pacGetFileSystemTempPath( void );
#endif

void *OSALC_pvMemoryAllocate (
      const char *pacName,
      size_t tSize,
      BOOLEAN bZeroInitialize
          );

void OSALC_vMemoryFree (
      void *pvMemoryAddress
          );

  /*****************************/
 /** GLOBAL INLINE FUNCTIONS **/
/*****************************/

#endif	// OSAL_CORE_H_
