/******************************************************************************/
/*                     Copyright (c) Sirius XM Radio Inc.                     */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio Inc.          */
/*                         Proprietary & Confidential                         */
/******************************************************************************/
/*******************************************************************************
*
*  DESCRIPTION
*
*       This module will contain all the OSAL core functions.
*
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "osal_time.h"
#include "osal_timer.h"
#include "osal_debughandler.h"
#include "osal_monitor.h"
#include "osal_general.h"
#include "osal_mem.h"

#include "os_intf.h"

#include "osal_core.h"
#include "_osal_core.h"

/******************************************************************************
*
*      OSALC_bBootstrap
*
*	   This is the main bootstrap function which starts the system, It runs
*      within the context of the first task (bootstrap task) which is
*      responsible for starting the OSAL.
*
******************************************************************************/
BOOLEAN OSALC_bBootstrap( const void *pvArg )
{
    BOOLEAN bRetval;
#if OSAL_DEBUG == 1
    OSAL_RETURN_CODE_ENUM eResult;
#endif
    const OSAL_START_HANDLER_STRUCT *psStartHandlers =
     (const OSAL_START_HANDLER_STRUCT *)pvArg;

    do
    {
        // Initialize OSAL Core
        bRetval = OSALC_bInitialize();
        if(bRetval == FALSE)
        {
            // OSAL Core Initialization failed
            break;
        }

        // Qualify/Condition all input start handlers
        OSALC_vQualifyStartHandlers(psStartHandlers);

        // Register any drivers required by the application
        bRetval = gsCore.sStartHandlers.bRegisterDrivers(
            gsCore.sStartHandlers.pvRegisterDriversHandlerArg);
        if(bRetval == FALSE)
        {
            // Driver registration failed.
            break;
        }

        ////////////////////////////////////////////////////////////////////
        /* Install System Services Here - e.g. file systems, stacks, etc. */

        /////////////////////
        // Initialize Time //
        /////////////////////
        bRetval = OSALT_bTimeInitialize();
        if(bRetval == FALSE)
        {
            // Time init failed
            break;
        }

        /////////////////////////
        // Install File System //
        /////////////////////////

#if OSAL_FILE_SYSTEM == 1
        bRetval = OSAL.bFileSystemStart();
        if(bRetval == FALSE)
        {
            // File System start failed
            break;
        }
#endif /* OSAL_FILE_SYSTEM == 1 */

        ////////////////////////////////////////////////////////////////////

        // Install the Task Monitor Handler Task
        bRetval = OSALM_bTaskInstall(
            OSAL_TASK_PRIORITY_LOWEST,
            &gsCore.sStartHandlers.sMonitorHandlers
                );
        if(bRetval == FALSE)
        {
            // Monitor Task installation failed
            break;
        }

#if OSAL_DEBUG == 1
        // Install the Debug Handler Task
        eResult = DEBUG_eTaskInstall(OSAL_TASK_PRIORITY_LOW);
        if(eResult != OSAL_SUCCESS)
        {
            // Debug Handler installation failed!
            break;
        }
#endif

        // Install the Timer task
        // This task registers with both the Task Monitor and Debug Handler
        bRetval = OSALT_bTaskInstall(OSAL_TASK_PRIORITY_HIGHEST);
        if(bRetval == FALSE)
        {
            // Timer Task installation failed
            break;
        }

        // Call user supplied 'start' function now.
        bRetval = gsCore.sStartHandlers.bStartHandler(
            gsCore.sStartHandlers.pvStartHandlerArg);
        if(bRetval == FALSE)
        {
            // User 'bootstrap' failed
            break;
        }

        // All is well!
        return TRUE;

    } while(FALSE);

    // Delete timer task (if installed)
    OSALT_vTaskUninstall();

#if OSAL_DEBUG == 1
    // Delete debug task (if installed)
    DEBUG_vTaskUninstall();
#endif

    // Delete task monitor (if installed)
    OSALM_vTaskUninstall();

#if OSAL_FILE_SYSTEM == 1
    // Uninstall file-system
    OSAL.bFileSystemStop();
#endif /* OSAL_FILE_SYSTEM == 1 */

    // Uninitialize time
    OSALT_vTimeUninitialize();

    // Unregister BSP drivers
    gsCore.sStartHandlers.vUnregisterDrivers(
        gsCore.sStartHandlers.pvRegisterDriversHandlerArg);

    // Delete any OSAL core resources which were created successfully
    OSALC_vUninitialize();

    return FALSE;
}

/*****************************************************************************
*
*      OSALC_bInitialize
*
*      This function is responsible for initializing the OSAL and all the
*      resources it requires. It must be called before any OSAL services
*      are used.
*
*****************************************************************************/
BOOLEAN OSALC_bInitialize( void )
{
    UN8 un8Type;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Initialize core structure
    OSAL_bMemSet(&gsCore,  0, sizeof( gsCore));

    // Initialize start handlers
    gsCore.sStartHandlers = gsDefaultStartHandlers;

    // Initialize statistics for memory allocation
    OSALC_vInitializeStatistics(&gsCore.sMemory.sActual.sBlocks);
    OSALC_vInitializeStatistics(&gsCore.sMemory.sActual.sBytes);
    OSALC_vInitializeStatistics(&gsCore.sMemory.sUser.sBlocks);
    OSALC_vInitializeStatistics(&gsCore.sMemory.sUser.sBytes);

    // Initialize Object Lists and their Statistics

    // This is one of the very first things the function must do
    // It must create the object lists it needs to track all OSAL
    // services and resources requested by the application.
    // Note that the object lists are created "without protection"
    // This is to avoid a "chicken-n-egg" scenario. After all if
    // an object list for a semaphore requires a semaphore itself
    // this is impossible unless first the semaphore list exists.
    // So since the list at initialization does not need protected
    // (multitasking has not started) then there is no need for the
    // protection. We will apply protection once all the lists have
    // been created.
    for(un8Type = OSAL_OBJECT_TYPE_FIRST;
            un8Type <= OSAL_OBJECT_TYPE_LAST; un8Type++)
    {
        if(gasObjects[un8Type].bManage == TRUE)
        {
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

#if OSAL_OBJECT_TRACKING == 1
            // Initialize statistics for this object
            OSALC_vInitializeStatistics(&gsCore.asStatistics[un8Type]);
#endif
            // Construct an appropriate name for a list
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     "%s%u", OSAL_NAME_PREFIX"ObjLL#", un8Type);

            // Create an object list
            eReturnCode = OSAL.eLinkedListCreate(&gsCore.ahObjectList[un8Type],
                acName, gasObjects[un8Type].n16Compare,
                gasObjects[un8Type].un32Options);
            if(eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }
        else
        {
            // We don't use this list
            gsCore.ahObjectList[un8Type] = OSAL_INVALID_OBJECT_HDL;
            eReturnCode = OSAL_SUCCESS;
        }
    }

    // If an object list was not successfully created, call the uninitialize
    // function and return FALSE
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Now that the object lists have been created, I can now create some
    // objects. The first one I need to create is a semaphore which protects
    // access to the OSAL Core Structure elements. We will also use this
    // semaphore as a mechanism for providing thread/task safe execution.
    eReturnCode = OSAL.eSemCreate(&gsCore.hCoreMutex,
        OSAL_NAME_PREFIX"Core Mutex", 1, 1, OSAL_SEM_OPTION_MUTEX);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    // Protect Object Lists
    //
    // Now that all the required object lists have been created we can
    // no protect them in the next step as we promised
    for(un8Type = OSAL_OBJECT_TYPE_FIRST;
            un8Type <= OSAL_OBJECT_TYPE_LAST; un8Type++)
    {
        if(gsCore.ahObjectList[un8Type] != OSAL_INVALID_OBJECT_HDL)
        {
            // Protect the object list
            eReturnCode = OSAL.eLinkedListProtect(gsCore.ahObjectList[un8Type]);
            if(eReturnCode != OSAL_SUCCESS)
            {
                break;
            }
        }
    }

    // If an object list was not successfully protected, call the uninitialize
    // function and return FALSE
    if(eReturnCode != OSAL_SUCCESS)
    {
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*       OSALC_vUninitialize
*
*****************************************************************************/
void OSALC_vUninitialize( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8Type;

    // Check if core data semaphore has been allocated
    if(gsCore.hCoreMutex != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_OBJECT_HDL hMutexToDelete = gsCore.hCoreMutex;
        gsCore.hCoreMutex = OSAL_INVALID_OBJECT_HDL;

        // Delete core data semaphore
        eReturnCode = OSAL.eSemDelete(hMutexToDelete);
        if(eReturnCode != OSAL_SUCCESS)
        {
            gsCore.hCoreMutex = hMutexToDelete;
        }
    }

    // Uninitialize Object Lists
    for(un8Type = OSAL_OBJECT_TYPE_FIRST;
            un8Type <= OSAL_OBJECT_TYPE_LAST; un8Type++)
    {
        if(gsCore.ahObjectList[un8Type] != OSAL_INVALID_OBJECT_HDL)
        {
            // Delete an object list
            eReturnCode = OSAL.eLinkedListDelete(gsCore.ahObjectList[un8Type]);
            if(eReturnCode == OSAL_SUCCESS)
            {
                gsCore.ahObjectList[un8Type] = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    return;
}

/*****************************************************************************
*
*       OSALC_bLockCore
*
*****************************************************************************/
BOOLEAN OSALC_bLockCore( void )
{
    BOOLEAN bOk = FALSE;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    eReturnCode = OSAL.eSemTake(gsCore.hCoreMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if((eReturnCode == OSAL_SUCCESS) ||
                    (eReturnCode == OSAL_ERROR_INVALID_HANDLE))
    {
        bOk = TRUE;
    }

    return bOk;
}

/*****************************************************************************
*
*       OSALC_vUnlockCore
*
*****************************************************************************/
void OSALC_vUnlockCore( void )
{
    OSAL.eSemGive(gsCore.hCoreMutex);
    return;
}

/*****************************************************************************
*
*       OSALC_vInitializeStatistics
*
*****************************************************************************/
void OSALC_vInitializeStatistics( OSAL_STATISTICS_STRUCT *psStatistics )
{
    // Check input pointer
    if(psStatistics != NULL)
    {
        psStatistics->un32Current = 0;
        psStatistics->un32Minimum = UN32_MAX;
        psStatistics->un32Maximum = UN32_MIN;
    }

    return;
}

/*****************************************************************************
*
*       OSALC_vUpdateStatistics
*
*****************************************************************************/
void OSALC_vUpdateStatistics( OSALC_UPDATE_OPERATION_ENUM eUpdateOperation,
                              UN32 un32Current,
                              OSAL_STATISTICS_STRUCT *psStatistics )
{
    // Check input pointer
    if(psStatistics != NULL)
    {
        // Record current value
        switch(eUpdateOperation)
        {
            case OSALC_UPDATE_OPERATION_ADD:
                psStatistics->un32Current += un32Current;
            break;

            case OSALC_UPDATE_OPERATION_SUB:
                psStatistics->un32Current -= un32Current;
            break;

            case OSALC_UPDATE_OPERATION_ABSOLUTE:
                psStatistics->un32Current = un32Current;
            break;

            default:
                // Do nothing
            break;
        }

        // Record maximum value (if exceeded)
        if(psStatistics->un32Current > psStatistics->un32Maximum)
        {
            psStatistics->un32Maximum = psStatistics->un32Current;
        }

        // Record minimum value (if receded)
        if(psStatistics->un32Current < psStatistics->un32Minimum)
        {
            psStatistics->un32Minimum = psStatistics->un32Current;
        }
    }

    return;
}

#if OSAL_OBJECT_TRACKING == 1

/*****************************************************************************
*
*       OSALC_vGetObjectStatistics
*
*****************************************************************************/
void OSALC_vGetObjectStatistics(
                                 OSAL_OBJECT_TYPE_ENUM eType,
                                 OSAL_STATISTICS_STRUCT *psStatistics )
{
    // Check provided pointer
    if(psStatistics != NULL)
    {
        // Verify specified type is one we are managing
        if(eType <= OSAL_OBJECT_TYPE_LAST)
        {
            BOOLEAN bLocked;

            bLocked = OSALC_bLockCore();
            if(bLocked == TRUE)
            {
                // Copy in statistics for the specified object
                *psStatistics = gsCore.asStatistics[eType];

                OSALC_vUnlockCore();
            }
        }
    }

    return;
}

#endif

/*****************************************************************************
*
*       OSALC_vUpdateMemoryAllocation
*
*****************************************************************************/
void OSALC_vUpdateMemoryAllocation(
    OSAL_MEMORY_TYPE_ENUM eType,
    OSALC_UPDATE_OPERATION_ENUM eOperationType,
    size_t tSize
        )
{
    OSAL_MEMORY_STRUCT *psMemory = NULL;

    // OS or OSAL?
    if(eType == OSAL_MEMORY_TYPE_OS)
    {
        psMemory = &gsCore.sMemory.sActual;
    }
    else if(eType == OSAL_MEMORY_TYPE_OSAL)
    {
        psMemory = &gsCore.sMemory.sUser;
    }

    if(psMemory != NULL)
    {
        BOOLEAN bLocked;

        // Lock core data for access
        bLocked = OSALC_bLockCore();
        if(bLocked == TRUE)
        {
            OSALC_vUpdateStatistics(eOperationType, 1,
                &psMemory->sBlocks );
            OSALC_vUpdateStatistics(eOperationType, tSize,
                &psMemory->sBytes );

            // Unlock core data
            OSALC_vUnlockCore();
        }
    }

    return;
}

#if OSAL_OBJECT_TRACKING == 1
/*****************************************************************************
*
*       OSALC_n16CompareObjectNames
*
*		This is the function used to compare two objects by name. Specifically
*       this function is used to override the normal object linked list
*       compare function with one that compares the name of the object
*       instead. Since those objects are not sorted by name, we must
*       start from the beginning and move forward only.
*
*       Inputs:
*               None
*
*       Outputs:
*               0   - Objects have the same value (equal, error)
*               < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
static N16 OSALC_n16CompareObjectNames( void *pvObj1, void *pvObj2)
{
    N16 n16Result = -1;
    OSAL_OBJECT_STRUCT *psObj1 = (OSAL_OBJECT_STRUCT *)pvObj1;
    OSAL_OBJECT_STRUCT *psObj2 = (OSAL_OBJECT_STRUCT *)pvObj2;

    // Check pointers
    if((psObj1 != NULL) && (psObj2 != NULL))
    {
        const char *pacObj1Name, *pacObj2Name;

        // Extract object names
        pacObj1Name = psObj1->pacName;
        pacObj2Name = psObj2->pacName;

        // Check if it is a valid type (by way of comparing if
        // an object name pointer was defined).
        if( (pacObj1Name != NULL) && (pacObj2Name != NULL) )
        {
            int iStrCmpResult;

            // Compare object names
            iStrCmpResult = strcmp(pacObj1Name, pacObj2Name);

            if (iStrCmpResult < 0)
            {
                n16Result = -1;
            }
            else if (iStrCmpResult > 0)
            {
                n16Result = 1;
            }
            else
            {
                n16Result = 0;
            }
        }
    }

    return n16Result;
}

/*****************************************************************************
*
*       OSALC_n16CompareObjectToName
*
*       This is the function used to compare two objects by name. Specifically
*       this function is used to override the normal object linked list
*       compare function with one that compares the name of the object
*       instead. Since those objects are not sorted by name, we must
*       start from the beginning and move forward only.
*
*       Inputs:
*               None
*
*       Outputs:
*               0   - Objects have the same value (equal, error)
*               < 0 - Object1 is less than (before) Object2
*
*****************************************************************************/
static N16 OSALC_n16CompareObjectToName( void *pvObj1, void *pvName)
{
    N16 n16Result = -1;
    OSAL_OBJECT_STRUCT *psObj1 = (OSAL_OBJECT_STRUCT *)pvObj1;
    const char *pacObj2Name = (const char *)pvName;

    // Check pointers
    if((psObj1 != NULL) && (pacObj2Name != NULL))
    {
        const char *pacObj1Name;

        // Extract object name
        pacObj1Name = psObj1->pacName;

        // Check if it is a valid type (by way of comparing if
        // an object name pointer was defined).
        if( (pacObj1Name != NULL) && (pacObj2Name != NULL) )
        {
            int iStrCmpResult;

            // Compare object names
            iStrCmpResult = strcmp(pacObj1Name, pacObj2Name);

            if (iStrCmpResult < 0)
            {
                n16Result = -1;
            }
            else if (iStrCmpResult > 0)
            {
                n16Result = 1;
            }
            else
            {
                n16Result = 0;
            }
        }
    }

    return n16Result;
}
#endif

/*****************************************************************************
*
*       OSALC_hGetObjectList
*
*       This function is used to get an object list ptr based on the type
*       provided.
*
*****************************************************************************/
OSAL_OBJECT_HDL OSALC_hGetObjectList( OSAL_OBJECT_TYPE_ENUM eType )
{
    OSAL_OBJECT_HDL hObj = OSAL_INVALID_OBJECT_HDL;

    // Check that type provided is within range
    if(eType <= OSAL_OBJECT_TYPE_LAST)
    {
        // Range checks out ok, now go get the array of object lists
        // Map to requested object list handle
        hObj = gsCore.ahObjectList[eType];
    }

    return hObj;
}

/*****************************************************************************
*
*   OSALC_eCreateObject
*
*   This is the function used to create a new object of type provided
*   by the input parameter eType. All required allocation and initialization
*   of the new object is performed here. Additionally a number of bytes
*   may be added to the end of the object if required. This is particularly
*   useful when creating objects for requested memory blocks. The additional
*   bytes requested may be used by the object creator as needed.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSALC_eCreateObject(
    OSAL_OBJECT_TYPE_ENUM eType,
    size_t tAdditionalBytes,
    const char *pacName,
    OSAL_OBJECT_HDL *phObj,
    OSAL_OBJECT_STRUCT **ppsObj,
    OSAL_OBJECT_INFO_UNION **ppuInfo
        )
{

    size_t tSize = 0, tInfoSize;
#if OSAL_OBJECT_TRACKING == 1
    size_t tNameSize;
#endif
    OSAL_OBJECT_INFO_UNION *puInfo;
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
    void *pvObject;

    // Verify we have in fact been provided a valid pointer to
    // a name for this object.
    if( (pacName == NULL) ||
        (phObj == NULL) ||
        (ppsObj == NULL)
            )
    {
        // Error!
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Check that type provided is within range
    if(eType >= OSAL_OBJECT_TYPE_UNKNOWN)
    {
        // Error!
        return OSAL_ERROR_INVALID_INPUT;
    }

    // Range checks out ok.

    // Verify and use provided pointers where provided. Otherwise
    // use local pointers for simpler implementation.
    if(ppuInfo == NULL) // Not provided?
    {
        // Use local
        ppuInfo = &puInfo;
    }

    // Initialize returns
    *phObj = OSAL_INVALID_OBJECT_HDL;
    *ppsObj = (OSAL_OBJECT_STRUCT *)NULL;

    // If objects of this type can be shared, try
    // to find a pre-existing instance of this
    // object which matches what was requested
    if (gasObjects[eType].bCanShare == TRUE)
    {
        OSAL_LINKED_LIST_ENTRY hEntry =
            OSAL_INVALID_LINKED_LIST_ENTRY;
        OSAL_RETURN_CODE_ENUM eSearchCode;

        // Search for this object first --
        // we're looking for what was provided
        // in the contents of ppsInfo
        eSearchCode = OSAL.eLinkedListSearch(
            gsCore.ahObjectList[eType],
            &hEntry,
            *ppuInfo);
        if (eSearchCode == OSAL_SUCCESS)
        {
            // A previous instance of this object
            // has been found.  Provide it to the caller
            OSAL_OBJECT_STRUCT *psFoundObj;

            // Extract the object at this entry
            psFoundObj = (OSAL_OBJECT_STRUCT *)OSAL.pvLinkedListThis(hEntry);
            *ppsObj = psFoundObj;

            // Extract the pointer to this object's handle and return to caller.
            *phObj = (OSAL_OBJECT_HDL)psFoundObj;

            // Extract the pointer to this object's info and return to caller.
            *ppuInfo = psFoundObj->puInfo;

            return OSAL_SUCCESS;
        }
    }

    // Initialize returns
    *ppuInfo = (OSAL_OBJECT_INFO_UNION *)NULL;

    // Provide space for the object itself (and header)
    tSize += sizeof(OSAL_OBJECT_STRUCT);

    // Now based on the object being created, we must add additional
    // bytes required for the object's info structure.
    tInfoSize = gasObjects[eType].tSize;
    tSize += tInfoSize;

    // Provide space for any additional bytes requested (user space)
    tSize += tAdditionalBytes;

#if OSAL_OBJECT_TRACKING == 1
    // Compute space required for provided name and null terminator
    tNameSize = strlen(pacName);
    tSize += tNameSize + 1;
#endif

    // If this is a managed object (will belong to a list)
    // make sure we request enough space for the Linked List Entry
    // as well. This is because we will use pre-allocated elements
    if(gasObjects[eType].bManage == TRUE)
    {
        // Call Linked List-Specific API to allocate memory required by the
        // object itself and header.

        pvObject = OSAL.pvLinkedListMemoryAllocate(
            OSAL_NAME_PREFIX"Object", tSize,
            gasObjects[eType].bZeroInitialize);
    }
    else
    {
        // Call OS-Specific API to allocate memory required by the object
        // itself, headers, and the name of the object.

        pvObject = OSALC_pvMemoryAllocate(
            OSAL_NAME_PREFIX"Object", tSize,
            gasObjects[eType].bZeroInitialize);
    }

    if(pvObject != NULL)
    {
#if OSAL_OBJECT_TRACKING == 1
        char *pacObjName = NULL;
        char *pacObjNameEnd;
#endif
        // Establish the start of this object. Move past the pre-allocated
        // linked list element structure if one is required. This is where
        // the actual object begins.
        *ppsObj = (OSAL_OBJECT_STRUCT *)pvObject;

        // Initialize the object linked list entry handle
        (*ppsObj)->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;

        // Record creating object handle
        (*ppsObj)->hCreatorObj = OSAL.hTaskGetHandle();

        // Initialize this object's type
        (*ppsObj)->eType = eType;

        // Initialize the OS-Specific Handle
        (*ppsObj)->hOS = OS_INVALID_OBJECT_HDL;

        // Compute the magic number for this object
        (*ppsObj)->un32MagicNumber = OSALC_un32ComputeMagicNumber(*ppsObj);

        // Assign object info into object memory (right after osal object)
        (*ppsObj)->puInfo = (OSAL_OBJECT_INFO_UNION *)(*ppsObj + 1);

        // Zero initialize object info, to be populated by caller
        OS.bMemSet((*ppsObj)->puInfo, 0, tInfoSize);

        // Assign pointer to any user memory (right after info). Otherwise
        // leave as NULL if no user bytes requested.
        if(tAdditionalBytes > 0)
        {
            // Assign pointer.
            (*ppsObj)->pvUserMemory = ((char *)(*ppsObj)->puInfo + tInfoSize);

#if OSAL_OBJECT_TRACKING == 1
            // Prepare location of object name (right after user memory)
            pacObjName = ((char *)((*ppsObj)->pvUserMemory)) + tAdditionalBytes;
#endif
        }
        else
        {
            // Assign user memory pointer
            (*ppsObj)->pvUserMemory = NULL;

#if OSAL_OBJECT_TRACKING == 1
            // Prepare location of object name (right after user memory)
            pacObjName = ((char *)(*ppsObj)->puInfo + tInfoSize);
#endif
        }

#if OSAL_OBJECT_TRACKING == 1
        // Copy object name into object memory (right after user memory)
        // and assign reference pointer to it.
        strncpy(pacObjName, pacName, tNameSize);
        (*ppsObj)->pacName = pacObjName;

        // Null terminate the string
        pacObjNameEnd = (pacObjName + tNameSize);
        *pacObjNameEnd = '\0'; // Null terminate string
#endif

        // Verify specified type is one we are managing. If so, set statistics
        // and find object list this object belongs to..
        if(eType <= OSAL_OBJECT_TYPE_LAST)
        {
            OSAL_OBJECT_HDL hObjectList;

            // Grab the object's linked list handle
            hObjectList = gsCore.ahObjectList[eType];
            if(hObjectList != OSAL_INVALID_OBJECT_HDL)
            {
                // If the object being created is a memory block
                // then we do not care about duplicate names. We always
                // just add them. Otherwise linked list options
                // will not allow duplicate objects to be created.

                // Add the object to the object list as long as an object
                // with this name does not already exist (if not a MEMORY
                // object).
                eReturnCode = OSAL.eLinkedListAdd(
                    hObjectList,
                    &(*ppsObj)->hEntry,
                    *ppsObj
                        );
                if(eReturnCode == OSAL_SUCCESS)
                {
                    // Success!
#if OSAL_OBJECT_TRACKING == 1
                    BOOLEAN bLocked;

                    // Lock core data for access
                    bLocked = OSALC_bLockCore();
                    if(bLocked == TRUE)
                    {
                        // Increment statistics
                        OSALC_vUpdateStatistics(
                            OSALC_UPDATE_OPERATION_ADD,
                            1,
                            &gsCore.asStatistics[eType]
                                );
                        // Unlock core data
                        OSALC_vUnlockCore();
                    }
#endif
                }
                else
                {
                    // Error! Could not add this entry
                    if(eReturnCode == OSAL_ERROR_LIST_ITEM_NOT_UNIQUE)
                    {
                        // Object with the same name already exists in list
                        // Need to clear (*ppsObj)->hEntry pointer, since it
                        // points to existing live object, but not our new one
                        (*ppsObj)->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
                    }

                    // Now remove and destroy the object
                    OSALC_eRemoveObject((OSAL_OBJECT_HDL)*ppsObj);
                    *ppsObj = NULL;
                    return eReturnCode;
                }
            }
        }

        // Extract the pointer to this object's handle and return to caller.
        *phObj = (OSAL_OBJECT_HDL)*ppsObj;

        // Extract the pointer to this object's info and return to caller.
        *ppuInfo = (*ppsObj)->puInfo;

        // The info(and some other) sections are left to the caller to
        // initialize, but since the allocation selected zero initializes,
        // you can at least know its been init'd to zero.
        eReturnCode = OSAL_SUCCESS;
    }

    return  eReturnCode;
}

/*****************************************************************************
*
*       OSALC_eRemoveObject
*
*       This is the function used to remove an object which was previously
*       created.
*
*****************************************************************************/
OSAL_RETURN_CODE_ENUM OSALC_eRemoveObject( OSAL_OBJECT_HDL hObj )
{
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Extract the object structure from the handle provided and
    // verify if the handle is valid. Any object handle could be provided
    // so we just specify the type as UNKNOWN.
    psObj = OSALC_psGetObjectFromHandle(hObj, OSAL_OBJECT_TYPE_UNKNOWN);
    if(psObj == NULL)
    {
        // Object is not valid
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Check object header info and see if it is ready to remove
    if(psObj->hOS != OS_INVALID_OBJECT_HDL)
    {
        // Object associated OS-layer handle has not been freed.
        return OSAL_ERROR_CANNOT_DELETE_OBJECT;
    }

    // Remove object from the list being monitored, that is
    // if it is being monitored.
    if(psObj->hEntry != OSAL_INVALID_LINKED_LIST_ENTRY)
    {
        eReturnCode = OSAL.eLinkedListRemove(psObj->hEntry);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate this entry
            psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
        }
        else
        {
            // Error!
            return eReturnCode;
        }
    }

    // Update this object type's statistics
    // Verify specified type is one we are managing
    if(gasObjects[psObj->eType].bManage == TRUE)
    {
#if OSAL_OBJECT_TRACKING == 1
        BOOLEAN bLocked;

        // Lock core data for access
        bLocked = OSALC_bLockCore();
        if(bLocked == TRUE)
        {
            // Increment statistics
            OSALC_vUpdateStatistics(
                OSALC_UPDATE_OPERATION_SUB,
                1,
                &gsCore.asStatistics[psObj->eType]
                    );
            // Unlock core data
            OSALC_vUnlockCore();
        }
#endif
    }

    // Now destroy the object itself
    OSALC_vDestroyObject(hObj);

    return OSAL_SUCCESS;
}

/*****************************************************************************
*
*       OSALC_vDestroyObject
*
*		This is the function used to destroy an object which was previously
*       created.
*
*****************************************************************************/
static void OSALC_vDestroyObject( OSAL_OBJECT_HDL hObj )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)hObj;
    OSAL_OBJECT_TYPE_ENUM eType;

    // Now nullify this object
    psObj->hEntry = OSAL_INVALID_LINKED_LIST_ENTRY;
    psObj->hCreatorObj = OSAL_INVALID_OBJECT_HDL;
    psObj->hOS = OS_INVALID_OBJECT_HDL;
    psObj->un32MagicNumber = 0;
#if OSAL_OBJECT_TRACKING == 1
    psObj->pacName = NULL;
#endif
    psObj->puInfo = NULL;
    psObj->pvUserMemory = NULL;
    eType = psObj->eType; // Remember type locally
    psObj->eType = OSAL_OBJECT_TYPE_UNKNOWN;

    // Update this object type's statistics
    // Verify specified type is one we are managing
    if(gasObjects[eType].bManage == TRUE)
    {
        // Free object memory
        OSAL.vLinkedListMemoryFree(psObj);
    }
    else
    {
        // Free object memory
        OSALC_vMemoryFree(psObj);
    }

    // Object is now destroyed
    return ;
}

/*****************************************************************************
*
*       OSALC_psGetObjectFromHandle
*
*		This is the function used to check that a valid object handle is
*       provided and that the handle points to a valid object. All required
*       checks are made and if the object is in fact valid a pointer to the
*       object structure is returned.
*
*****************************************************************************/
OSAL_OBJECT_STRUCT *OSALC_psGetObjectFromHandle( OSAL_OBJECT_HDL hObj,
                                                 OSAL_OBJECT_TYPE_ENUM eType )
{
    OSAL_OBJECT_STRUCT *psObj = NULL;
    UN32 un32MagicNumber;

    // Verify a valid handle was provided
    if(hObj == OSAL_INVALID_OBJECT_HDL)
    {
        // This is an invalid handle...don't do this to me!
        return NULL;
    }

    // Extract handle to pointer
    psObj = (OSAL_OBJECT_STRUCT *)hObj;

    // If the object type if not known, simply make it equal to the
    // embedded type.
    if(eType != OSAL_OBJECT_TYPE_UNKNOWN)
    {
        // Compare types...do they match?
        if(eType != psObj->eType)
        {
            // Nope! Fail
            return NULL;
        }
    }

    // Compute magic number
    un32MagicNumber = OSALC_un32ComputeMagicNumber(psObj);

    // Verify magic number
    if(psObj->un32MagicNumber != un32MagicNumber)
    {
        // Magic number does not match!
        return NULL;
    }

    // All is well!
    return psObj;
}

/*****************************************************************************
*
*       OSALC_un32ComputeMagicNumber
*
*       This function is used to generate a unique magic number which
*       identifies and verifies the validity of an object. If an object
*       does not maintain the proper magic number then some memory corruption
*       or other shenanigans have occurred.

*****************************************************************************/
static UN32 OSALC_un32ComputeMagicNumber(const OSAL_OBJECT_STRUCT *psObj)
{
    UN32 un32MagicNumber = 0;

    // Verify header is valid
    if(psObj != NULL)
    {
        // Magic number's uniquely identify the validity of an object
        un32MagicNumber = ((UN32)psObj->eType + (UN32)psObj)
            ^ 0x55AA55AA;
    }

    return un32MagicNumber;
}

/*****************************************************************************
*
*       OSALC_psFindObjectFromName
*
*		This is the function used to find an obj in an object list whith
*       the same name as that provided by the input parameter pacName.
*
*       Note: This function is NOT reentrant!
*
*****************************************************************************/
OSAL_OBJECT_STRUCT *OSALC_psFindObjectFromName( OSAL_OBJECT_TYPE_ENUM eType,
                                                const char *pacName )
{
#if OSAL_OBJECT_TRACKING == 1

    OSAL_OBJECT_HDL hList = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj = NULL;
    OSAL_LINKED_LIST_ENTRY hEntry =
        OSAL_INVALID_LINKED_LIST_ENTRY; // search from start(top) of list;

    // Check provided name is valid
    if(pacName == NULL)
    {
        return NULL;
    }

    // Check provided name is of correct length
    if(strlen(pacName) > OSAL_MAX_OBJECT_NAME_LENGTH)
    {
        return NULL;
    }

    // Get object list pointer for this type to find
    hList = OSALC_hGetObjectList(eType);

    if(hList != OSAL_INVALID_OBJECT_HDL)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // checking if compare function really compares by name
        if (gasObjects[eType].n16Compare == OSALC_n16CompareObjectNames)
        {
            OSAL_OBJECT_STRUCT sObj;

            sObj.pacName = pacName;
            eReturnCode = OSAL.eLinkedListSearch(hList, &hEntry, &sObj);
        }
        else
        {
            // Use unsorted search
            eReturnCode = OSAL.eLinkedListLinearSearch(
                hList, &hEntry, OSALC_n16CompareObjectToName, (void *)pacName);
        }

        if(eReturnCode != OSAL_SUCCESS)
        {
            return NULL;
        }
        else
        {
            // Extract the object pointer to the one we found
            psObj = (OSAL_OBJECT_STRUCT *)OSAL.pvLinkedListThis(hEntry);
        }
    }

    return psObj;
#else
    return NULL;
#endif
}

/*******************************************************************************
*
*   OSALC_pacGetPriorityName
*
*******************************************************************************/
const char *OSALC_pacGetPriorityName (
    OSAL_TASK_PRIORITY_ENUM ePriority
        )
{
    const char *pacPriorityName = "Unknown";
    UN8 un8Index = 0;

    // Translate enum to a text map
    do
    {
        if(ePriority == gsPriorityMap[un8Index].ePriority)
        {
            break;
        }

    } while(gsPriorityMap[++un8Index].ePriority !=
            OSAL_TASK_PRIORITY_INVALID);

    // Set the current text
    pacPriorityName =
        gsPriorityMap[un8Index].pacText;

    return pacPriorityName;
}

#if OSAL_FILE_SYSTEM == 1
/*****************************************************************************
*
*       OSALC_vUpdateFileSystemTempPath
*
*****************************************************************************/
void OSALC_vUpdateFileSystemTempPath(
    char *pacTempDirPath
        )
{
    char *pacOldTempDirPath = NULL;
    BOOLEAN bLocked;

    // Lock core data for access
    bLocked = OSALC_bLockCore();
    if(bLocked == TRUE)
    {
        // save the old path
        pacOldTempDirPath = gsCore.pacTempPath;

        // Update path (NULL is a valid value when stopping the file system)
        gsCore.pacTempPath = pacTempDirPath;

        // Unlock core data
        OSALC_vUnlockCore();

        // did we have an old path, if so we need to free the memory holding
        // the string?
        if (pacOldTempDirPath != NULL)
        {
            // yes, we must free the memory
            OSALC_vMemoryFree(pacOldTempDirPath);
        }
    }

    return;
}

/*****************************************************************************
*
*       OSALC_pacGetFileSystemTempPath

*   NOTE: The core should be locked prior to calling this function and
*         unlocked when the pointer is done being used.
*
*****************************************************************************/
char *OSALC_pacGetFileSystemTempPath( void )
{
    char *pacTempDirPath = NULL;

    // get the temp path
    pacTempDirPath = gsCore.pacTempPath;

    return pacTempDirPath;
}
#endif

/*****************************************************************************
*
*       OSALC_vQualifyStartHandlers
*
*****************************************************************************/
static void OSALC_vQualifyStartHandlers(
    const OSAL_START_HANDLER_STRUCT *psStartHandlers )
{
    // Check if we were even provided a valid structure, if not we'll make do.
    if(psStartHandlers == NULL)
    {
        // Use internal handlers (defaults)
        psStartHandlers = &gsCore.sStartHandlers;
        return;
    }

    // If a valid handler is provided, replace the core default. Otherwise
    // leave the default in place.
    if (psStartHandlers->bRegisterDrivers != NULL)
    {
        gsCore.sStartHandlers.bRegisterDrivers
                        = psStartHandlers->bRegisterDrivers;
        gsCore.sStartHandlers.pvRegisterDriversHandlerArg
                        = psStartHandlers->pvRegisterDriversHandlerArg;
    }
    if (psStartHandlers->vUnregisterDrivers != NULL)
    {
        gsCore.sStartHandlers.vUnregisterDrivers
                        = psStartHandlers->vUnregisterDrivers;
        gsCore.sStartHandlers.pvRegisterDriversHandlerArg
                        = psStartHandlers->pvRegisterDriversHandlerArg;
    }
    if (psStartHandlers->bStartHandler != NULL)
    {
        gsCore.sStartHandlers.bStartHandler = psStartHandlers->bStartHandler;
        gsCore.sStartHandlers.pvStartHandlerArg
                        = psStartHandlers->pvStartHandlerArg;
    }
    if (psStartHandlers->sMonitorHandlers.vMonitorErrorHandler != NULL)
    {
        gsCore.sStartHandlers.sMonitorHandlers.vMonitorErrorHandler
                        = psStartHandlers->sMonitorHandlers.vMonitorErrorHandler;
        gsCore.sStartHandlers.sMonitorHandlers.pvMonitorErrorHandlerArg
                        = psStartHandlers->sMonitorHandlers.pvMonitorErrorHandlerArg;
    }
    if (psStartHandlers->sMonitorHandlers.vHeartBeatHandler != NULL)
    {
        gsCore.sStartHandlers.sMonitorHandlers.vHeartBeatHandler
                        = psStartHandlers->sMonitorHandlers.vHeartBeatHandler;
        gsCore.sStartHandlers.sMonitorHandlers.pvHeartBeatHandlerArg
                        = psStartHandlers->sMonitorHandlers.pvHeartBeatHandlerArg;
    }
    if (psStartHandlers->sMonitorHandlers.vWatchdogStartHandler != NULL)
    {
        gsCore.sStartHandlers.sMonitorHandlers.vWatchdogStartHandler
                        = psStartHandlers->sMonitorHandlers.vWatchdogStartHandler;
        gsCore.sStartHandlers.sMonitorHandlers.pvWatchdogHandlerArg
                        = psStartHandlers->sMonitorHandlers.pvWatchdogHandlerArg;
    }
    if (psStartHandlers->sMonitorHandlers.vWatchdogStopHandler != NULL)
    {
        gsCore.sStartHandlers.sMonitorHandlers.vWatchdogStopHandler
                        = psStartHandlers->sMonitorHandlers.vWatchdogStopHandler;
        gsCore.sStartHandlers.sMonitorHandlers.pvWatchdogHandlerArg
                        = psStartHandlers->sMonitorHandlers.pvWatchdogHandlerArg;
    }
    if (psStartHandlers->sMonitorHandlers.vWatchdogKickHandler != NULL)
    {
        gsCore.sStartHandlers.sMonitorHandlers.vWatchdogKickHandler
                        = psStartHandlers->sMonitorHandlers.vWatchdogKickHandler;
        gsCore.sStartHandlers.sMonitorHandlers.pvWatchdogHandlerArg
                        = psStartHandlers->sMonitorHandlers.pvWatchdogHandlerArg;
    }

    return;
}

/*****************************************************************************
*
*       OSALC_bRegisterDrivers
*
*****************************************************************************/
static BOOLEAN OSALC_bRegisterDrivers(void *pvArg)
{
    puts("OSALC_bRegisterDrivers()\n");
    return TRUE;
}

/*****************************************************************************
*
*       OSALC_vUnregisterDrivers
*
*****************************************************************************/
static void OSALC_vUnregisterDrivers(void *pvArg)
{
    puts("OSALC_vUnregisterDrivers()\n");
    return;
}


/*****************************************************************************
*
*       OSALC_bStartHandler
*
*****************************************************************************/
static BOOLEAN OSALC_bStartHandler(const void *pvArg)
{
    puts("OSALC_bStartHandler()\n");
    return TRUE;
}


/*****************************************************************************
*
*       OSALC_vMonitorErrorHandler
*
*****************************************************************************/
static void OSALC_vMonitorErrorHandler(
    OSAL_RETURN_CODE_ENUM eErrorCode,
    const OSAL_TASK_INFO_STRUCT *psTaskInfo,
    void *pvArg
        )
{
    printf("OSALC_vMonitorErrorHandler() - %s:%s(%d)\n",
#if OSAL_OBJECT_TRACKING == 1
        psTaskInfo->sDebug.pacName,
#else
        "Task",
#endif
        OSAL_pacGetReturnCodeName(eErrorCode),
        eErrorCode
            );
    return;
}


/*****************************************************************************
*
*       OSALC_vHeartBeatHandler
*
*****************************************************************************/
static void OSALC_vHeartBeatHandler(void *pvArg)
{
    puts("OSALC_vHeartBeatHandler()\n");
    return;
}


/*****************************************************************************
*
*       OSALC_vWatchdogStartHandler
*
*****************************************************************************/
static void OSALC_vWatchdogStartHandler(void *pvArg)
{
    puts("OSALC_vWatchdogStartHandler()\n");
    return;
}


/*****************************************************************************
*
*       OSALC_vWatchdogStopHandler
*
*****************************************************************************/
static void OSALC_vWatchdogStopHandler(void *pvArg)
{
    puts("OSALC_vWatchdogStopHandler()\n");
    return;
}


/*****************************************************************************
*
*       OSALC_vWatchdogKickHandler
*
*****************************************************************************/
static void OSALC_vWatchdogKickHandler(void *pvArg)
{
    puts("OSALC_vWatchdogKickHandler()\n");
    return;
}

/*****************************************************************************
*
*       OSALC_pvMemoryAllocate
*
*****************************************************************************/
void *OSALC_pvMemoryAllocate (
      const char *pacName,
      size_t tSize,
      BOOLEAN bZeroInitialize
          )
{
    void *pvObj;
    size_t tActualSizeAllocated = 0;

    pvObj = OS.pvMemoryAllocate(
        pacName, tSize, bZeroInitialize,
        &tActualSizeAllocated
            );

    // Update memory allocation statistics (OS, Add)
    OSALC_vUpdateMemoryAllocation(
        OSAL_MEMORY_TYPE_OS, OSALC_UPDATE_OPERATION_ADD, tActualSizeAllocated );

    return pvObj;
}

/*****************************************************************************
*
*       OSALC_vMemoryFree
*
*****************************************************************************/
void OSALC_vMemoryFree (
      void *pvMemoryAddress
          )
{
    size_t tNumBytesFreed = 0;

    // Free node data
    OS.vMemoryFree(pvMemoryAddress, &tNumBytesFreed);

    // Update memory allocation statistics (OS, Subtract)
    OSALC_vUpdateMemoryAllocation(
        OSAL_MEMORY_TYPE_OS, OSALC_UPDATE_OPERATION_SUB, tNumBytesFreed );

    return;
}

/*******************************************************************************
*
*   OSALC_bMemoryUsage
*
*******************************************************************************/
BOOLEAN OSALC_bMemoryUsage(
    UN32 *pun32CurrentAllocatedBlocks,
    UN32 *pun32CurrentActualAllocatedBlocks,
    UN32 *pun32CurrentUserBytes,
    UN32 *pun32CurrentActualBytes,
    UN32 *pun32MaxAllocatedBlocks,
    UN32 *pun32MaxActualAllocatedBlocks,
    UN32 *pun32MaxUserBytes,
    UN32 *pun32MaxActualBytes,
    UN32 *pun32TotalSystemBytes
        )
{
    BOOLEAN bLocked, bRetval = FALSE;

    bLocked = OSALC_bLockCore();
    if(bLocked == TRUE)
    {
        // Copy in statistics for the ones they want
        if(pun32CurrentAllocatedBlocks != NULL)
        {
            *pun32CurrentAllocatedBlocks =
                gsCore.sMemory.sUser.sBlocks.un32Current;
        }

        if(pun32CurrentActualAllocatedBlocks != NULL)
        {
            *pun32CurrentActualAllocatedBlocks =
                gsCore.sMemory.sActual.sBlocks.un32Current;
        }

        if(pun32CurrentUserBytes != NULL)
        {
            *pun32CurrentUserBytes =
                gsCore.sMemory.sUser.sBytes.un32Current;
        }

        if(pun32CurrentActualBytes != NULL)
        {
            *pun32CurrentActualBytes =
                gsCore.sMemory.sActual.sBytes.un32Current;
        }

        if( pun32MaxAllocatedBlocks != NULL )
        {
            *pun32MaxAllocatedBlocks =
                gsCore.sMemory.sUser.sBlocks.un32Maximum;
        }

        if( pun32MaxActualAllocatedBlocks != NULL )
        {
            *pun32MaxActualAllocatedBlocks =
                gsCore.sMemory.sActual.sBlocks.un32Maximum;
        }

        if ( pun32MaxUserBytes != NULL )
        {
            *pun32MaxUserBytes =
                gsCore.sMemory.sUser.sBytes.un32Maximum;
        }

        if ( pun32MaxActualBytes != NULL )
        {
            *pun32MaxActualBytes =
                gsCore.sMemory.sActual.sBytes.un32Maximum;
        }

        OSALC_vUnlockCore();

        // Call OS-Specific API
        if(pun32TotalSystemBytes != NULL)
        {
            *pun32TotalSystemBytes = OS.un32TotalSystemBytes();
        }

        bRetval = TRUE;
    }

    return bRetval;
}
