/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the generic device driver implementation for
 *  a Satellite Radio Hardware (SRH) collection.
 *
 ******************************************************************************/

#include <standard.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>

#include "osal.h"

// Sirius SRH Driver Includes
#include "srh.h"
#include "_srh.h"

#include "sms_fcsxm.h"
/*****************************************************************************
*
*       _bInit
*
*   Performs any device initialization for the physical device this driver
*   is written for. This method is called only when the first driver instance
*   is opened.
*
*   Inputs:
*       pcDevName - A pointer to an ASCII null terminated string which
*           identifies this device. A copy of the provided name is retained
*           in the device node and this is a pointer to that name.
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       ppvPublicData - A pointer to a pointer which represents any
*           persistent object which contains device-global or public
*           data required by all driver instances.
*
*   Outputs:
*       TRUE on success. Otherwise FALSE should be returned on error.
*
*****************************************************************************/
static BOOLEAN _bInit (
                        const char *pcDevName,
                        void const *pvProperties,
                        void **ppvPublicData
                        )
{
    UN8 un8SRMIndex, un8NumSRMs;
    SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;
    SRH_PUBLIC_STRUCT *psPublicData;

    // Extract device properties from the Device Driver Framework
    SRH_DEVICE_STRUCT *psProperties =
        (SRH_DEVICE_STRUCT *)pvProperties;

    // Verify that the SRM and DECODER names are unique
    if (_bVerifyUniqueNames(psProperties) == FALSE)
    {
        // Error!
        printf("SRH Error! Names are not unique.\n");
        return FALSE;
    }

    // Detremine the number of SRM's supported
    un8NumSRMs = psProperties->un8NumberOfSRMs;

    // Allocate a structure to hold the public control structure
    // required by the driver.
    // Set the public data pointer for the Framework to maintain
    *ppvPublicData =
        OSAL.pvMemoryAllocate(
            pcDevName,
            sizeof(SRH_PUBLIC_STRUCT) + (sizeof(OSAL_OBJECT_HDL) * un8NumSRMs),
            TRUE );
    psPublicData = (SRH_PUBLIC_STRUCT *)*ppvPublicData;
    if(psPublicData == NULL)
    {
        // Error!
        printf("SRH Error! Cannot allocate memory.\n");
        return FALSE;
    }

    // Copy the SRH's physical interface information.
    psPublicData->psPhy = psProperties;

    for (un8SRMIndex = 0;
         un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
         un8SRMIndex++)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        // Index into this SRM's GPIO mutex
        OSAL_OBJECT_HDL *phMutex =
            &((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8SRMIndex];

        *phMutex = OSAL_INVALID_OBJECT_HDL;
        psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

        // Initialize Module I/O. Note exclusive access is guaranteed
        // at this point courtesy of the OSAL device driver framework.

// Bosch ID#11: Disable the SMS GPIO control, because Bosch GPIO control is enabled 
#ifdef FC_SXM_SMS_GPIO_CTRL
        // Put SRM in Reset (RESET_M)
        if(psCurSRM->sReset.pvEnable != SRH_NULL_GPIO)
        {
            psCurSRM->sReset.pvEnable();
        }

        // Disable SRM Power (SHDN)
        if(psCurSRM->sPower.pvDisable != SRH_NULL_GPIO)
        {
            psCurSRM->sPower.pvDisable();
        }

        // Disable SRM Antenna Power (V_ANT)
        if(psCurSRM->sAntenna.pvDisable != SRH_NULL_GPIO)
        {
            psCurSRM->sAntenna.pvDisable();
        }
#endif
        // Initialize Mutex for SRM I/O
        eReturnCode = OSAL.eSemCreate(
            phMutex, psCurSRM->pacSRMName, 1, 1, OSAL_SEM_OPTION_NONE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error! Mutex failure
            return FALSE;
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*       _vExit
*
*   Performs any device un-initialization or clean-up procedures required
*   when the device is no longer needed. This method is called only when
*   all open connections to the driver have been closed.
*
*   Inputs:
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       pvPublicData - - A pointer to a persistent object which contains
*           device-global or public data required by all driver instances.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void _vExit (
                        void const *pvProperties,
                        void *pvPublicData
                        )
{
    UN8 un8SRMIndex;
    SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

    // Extract public data from the Device Driver Framework
    SRH_PUBLIC_STRUCT *psPublicData =
        (SRH_PUBLIC_STRUCT *) pvPublicData;

    if (psPublicData != NULL)
    {
        for (un8SRMIndex = 0;
             un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
             un8SRMIndex++)
        {
            OSAL_RETURN_CODE_ENUM eReturnCode;

            // Index into this SRM's GPIO mutex
            OSAL_OBJECT_HDL *phMutex =
                &((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8SRMIndex];

            psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

            // Initialize Module I/O. Note exclusive access is guaranteed
            // at this point courtesy of the OSAL device driver framework.

// Bosch ID#11: Disable the SMS GPIO control, because Bosch GPIO control is enabled 
#ifdef FC_SXM_SMS_GPIO_CTRL
            // Put SRM in Reset (RESET_M)
            if(psCurSRM->sReset.pvEnable != SRH_NULL_GPIO)
            {
                psCurSRM->sReset.pvEnable();
            }

            // Disable SRM Power (SHDN)
            if(psCurSRM->sPower.pvDisable != SRH_NULL_GPIO)
            {
                psCurSRM->sPower.pvDisable();
            }

            // Disable SRM Antenna Power (V_ANT)
            if(psCurSRM->sAntenna.pvDisable != SRH_NULL_GPIO)
            {
                psCurSRM->sAntenna.pvDisable();
            }
#endif
            // Destroy the mutex
            eReturnCode = OSAL.eSemDelete(*phMutex);
            if (eReturnCode == OSAL_SUCCESS)
            {
                *phMutex = OSAL_INVALID_OBJECT_HDL;
            }
        }
    }

    // Just free any public data allocated by _bInit()
    OSAL.vMemoryFree(pvPublicData);

    return;
}

/*****************************************************************************
*
*       _n32Open
*
*   Opens a connection (instance) to the specified device by allocating any
*   resources or state information required for this instance.
*
*   This function currently supports three different types of connections:
*       ssp         - The driver is being used to communicate to the SSP port
*                     of an SRM.
*       sdtp        - The driver is being used to communicate to the SDTP port
*                     of an SRM.
*       sxi         - The driver is being used to communicate to the SXi port
*                     of an SRM.
*       discovery   - The driver isn't opening any physical ports or files.
*                     Instead it is being used just to discover what is
*                     defined in the SRH driver and to use IOCTLs.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver.
*       pvArg - A pointer to some device driver specific data which represents
*           a mode or configuration this instance is to be open with.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Open (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        const void *pvArg
                        )
{
    // Extract public data from the Device Driver Framework
    SRH_PUBLIC_STRUCT *psPublicData =
        (SRH_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    SRH_PRIVATE_STRUCT *psPrivateData;

    // Check input
    if(pvArg == NULL)
    {
        // Error!
        return DEV_ERROR;
    }

    // Why are we doing this?  Well, changes to the SRH
    // driver can slip by application developers fairly
    // easily, and we can use this field to detect
    // at least if one field which can be validated
    // is actually valid
    if (VERIFY_DEVICE_GROUP(psPublicData->psPhy->tDeviceGroup) == FALSE)
    {
        // Error!
        return DEV_ERROR;
    }

    // Allocate some private data structure which really just represents
    // our driver's instance state and control information.
    psPrivateData =
        (SRH_PRIVATE_STRUCT *)OSAL.pvMemoryAllocate(
            psDeviceInfo->pcDevName,
            sizeof(SRH_PRIVATE_STRUCT),
            TRUE
                );
    if(psPrivateData == NULL)
    {
        // Error!
        return DEV_ERROR;
    }

    // Populate private data...

    // If we are opening the device driver in "discovery" mode
    // we don't actually have a physical connection to a file (port).
    // Therefore, psFile is NULL.
    if ( strcmp((const char*)pvArg, "discovery") == 0)
    {
        psPrivateData->psFile = NULL;
    }
    else
    {
        // Parse input device parameters
        psPrivateData->psFile = _psParseArg(psPublicData->psPhy,
                                            (const char*)pvArg);
        if(psPrivateData->psFile == NULL)
        {
            // Error! Open on driver is unsuccessful
            printf("Error! Cannot parse provided arguments.\n");
            OSAL.vMemoryFree(psPrivateData);
            return DEV_ERROR;
        }
    }

    // Inform the Framework of this private data
    psDeviceInfo->pvPrivateData = (void *)psPrivateData;

    return DEV_OK;
}

/*****************************************************************************
*
*       _n32Close
*
*   Closes a connection (instance) to the specified device and releases any
*   resources obtained when the device was opened.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members
*           can be accessed to retrieve information about the device itself
*           and the driver instance. See Section 4.10.2 for more detail on
*           this structures members.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is returned
*       on error. See section 4.11 for possible error codes. If anything other
*       than DEV_OK is returned the instance handle will not be released and
*       will remain open.
*
*****************************************************************************/
static N32 _n32Close (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo
                        )
{
    // Extract private data from the Device Driver Framework
    SRH_PRIVATE_STRUCT *psPrivateData =
        (SRH_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Close any open file handle this instance represents
    if(psPrivateData != NULL && psPrivateData->psFile != NULL)
    {
        // Close it
        fclose(psPrivateData->psFile);
    }

    // Now, finally free any private data allocated by _n32Open()
    OSAL.vMemoryFree(psDeviceInfo->pvPrivateData);

    return DEV_OK;
}

/*****************************************************************************
*
*       _tRead
*
*   Allows byte wide (stream oriented) reads from the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcDst - The destination pointer where device data is to be read.
*       tSize - The size of each object to read.
*       tNumObjs - The number of objects to read.
*
*   Outputs:
*       The number of objects successfully read from the device and placed
*       in pcDst. If an error occurs or no data was read a zero is returned.
*       It should not be assumed that every read request must always be carried
*       out to completion in a single call. The caller is required to check the
*       return value to determine if all the requested data was read or if a
*       partial read was performed. If a partial read occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tRead (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        char *pcDst,
                        size_t tSize,
                        size_t tNumObjs
                        )
{
    size_t tNumRead = 0;

    // Extract private data from the Device Driver Framework
    SRH_PRIVATE_STRUCT *psPrivateData =
        (SRH_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Perform device 'read' if instance has a port
    if(psPrivateData->psFile != NULL)
    {
        tNumRead = fread(pcDst, tSize, tNumObjs, psPrivateData->psFile);
    }

    return tNumRead;
}

/*****************************************************************************
*
*       _tWrite
*
*   Allows byte wide (stream oriented) writes to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcSrc - The source pointer where device data is to be written from.
*       tSize - The size of each object to write.
*       tNumObjs - The number of objects to write.
*
*   Outputs:
*       The number of objects successfully written to the device from pcSrc.
*       If an error occurs or no data was written a zero is returned. It should
*       not be assumed that every write request must always be carried out to
*       completion in a single call. The caller is required to check the return
*       value to determine if all the requested data was written or if a
*       partial write was performed. If a partial write occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tWrite (
                         const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                         const char *pcSrc,
                         size_t tSize,
                         size_t tNumObjs
                         )
{
    size_t tNumWritten = 0;

    // Extract private data from the Device Driver Framework
    SRH_PRIVATE_STRUCT *psPrivateData =
        (SRH_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Perform device 'write' if instance has a port
    if(psPrivateData->psFile != NULL)
    {
        tNumWritten = fwrite(pcSrc, tSize, tNumObjs, psPrivateData->psFile);
    }

    return tNumWritten;
}

/*****************************************************************************
*
*       _n32Ioctl
*
*   Allows driver specific command and control to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       n32Cmd - A number assigned to a particular device command. Typically
*           this is defined in the device driver's public header file
*           (i.e. <device_driver_name>.h)
*       tList - A pointer to a variable argument list which contains zero
*           or more command specific arguments. The number and type of each
*           argument must be defined in a device driver document for this
*           particular device driver.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. Any unsupported ioctl commands should return
*       DEV_UNSUPPORTED_COMMAND. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Ioctl (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        N32 n32Cmd,
                        va_list *ptList
                        )
{
    N32 n32Err = DEV_OK;

    // Extract public data from the Device Driver Framework
    SRH_PUBLIC_STRUCT *psPublicData =
        (SRH_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    // Extract private data from the Device Driver Framework
    SRH_PRIVATE_STRUCT *psPrivateData =
        (SRH_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Handle command
    switch(n32Cmd)
    {
        case SRH_IOCTL_ANTENNA_POWER_ENABLE:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if(psSRM != NULL)
            {
                // Enable SRM Antenna Power
                if(psSRM->sAntenna.pvEnable != SRH_NULL_GPIO)
                {
                    psSRM->sAntenna.pvEnable();
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_ANTENNA_POWER_DISABLE:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if(psSRM != NULL)
            {
                // Disable SRM Antenna Power
                if(psSRM->sAntenna.pvDisable != SRH_NULL_GPIO)
                {
                    psSRM->sAntenna.pvDisable();
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_POWER_ON:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            UN8 un8Index;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, &un8Index);

            if(psSRM != NULL)
            {
                // Enable SRM Power
                if(psSRM->sPower.pvEnable != SRH_NULL_GPIO)
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;
                    // Index into this SRM's GPIO mutex
                    OSAL_OBJECT_HDL hMutex =
                        ((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8Index];

                    // Take control of gpio mutex
                    eReturnCode = OSAL.eSemTake(
                        hMutex, OSAL_OBJ_TIMEOUT_INFINITE );
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        psSRM->sPower.pvEnable();
                        OSAL.eSemGive(hMutex);
                    }
                    else
                    {
                        n32Err = DEV_MUTEX_FAILURE;
                    }
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_POWER_OFF:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            UN8 un8Index;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, &un8Index);

            if(psSRM != NULL)
            {
                // Disable SRM Power
                if(psSRM->sPower.pvDisable != SRH_NULL_GPIO)
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;
                    // Index into this SRM's GPIO mutex
                    OSAL_OBJECT_HDL hMutex =
                        ((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8Index];

                    // Take control of gpio mutex
                    eReturnCode = OSAL.eSemTake(
                        hMutex, OSAL_OBJ_TIMEOUT_INFINITE );
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        psSRM->sPower.pvDisable();
                        OSAL.eSemGive(hMutex);
                    }
                    else
                    {
                        n32Err = DEV_MUTEX_FAILURE;
                    }
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_RESET_ENABLE:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            UN8 un8Index;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, &un8Index);

            if(psSRM != NULL)
            {
                // Enable SRM Reset
                if(psSRM->sReset.pvEnable != SRH_NULL_GPIO)
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;
                    // Index into this SRM's GPIO mutex
                    OSAL_OBJECT_HDL hMutex =
                        ((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8Index];

                    // Take control of gpio mutex
                    eReturnCode = OSAL.eSemTake(
                        hMutex, OSAL_OBJ_TIMEOUT_INFINITE );
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        psSRM->sReset.pvEnable();
                        OSAL.eSemGive(hMutex);
                    }
                    else
                    {
                        n32Err = DEV_MUTEX_FAILURE;
                    }
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_RESET_DISABLE:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            UN8 un8Index;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, &un8Index);

            if(psSRM != NULL)
            {
                // Disable SRM Reset
                if(psSRM->sReset.pvDisable != SRH_NULL_GPIO)
                {
                    OSAL_RETURN_CODE_ENUM eReturnCode;
                    // Index into this SRM's GPIO mutex
                    OSAL_OBJECT_HDL hMutex =
                        ((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8Index];

                    // Take control of gpio mutex
                    eReturnCode = OSAL.eSemTake(
                        hMutex, OSAL_OBJ_TIMEOUT_INFINITE );
                    if (eReturnCode == OSAL_SUCCESS)
                    {
                        psSRM->sReset.pvDisable();
                        OSAL.eSemGive(hMutex);
                    }
                    else
                    {
                        n32Err = DEV_MUTEX_FAILURE;
                    }
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_RESET:
        case SRH_IOCTL_APP_RESET:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            UN8 un8Index;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, &un8Index);

            if(psSRM != NULL)
            {
                OSAL_RETURN_CODE_ENUM eReturnCode;
                // Index into this SRM's GPIO mutex
                OSAL_OBJECT_HDL hMutex =
                    ((OSAL_OBJECT_HDL *)(psPublicData + 1))[un8Index];

                if ((n32Cmd == SRH_IOCTL_APP_RESET) &&
                    (psSRM->sReset.pvReset == SRH_NULL_GPIO))
                {
                    // This is invalid configuration. Cannot execute
                    n32Err = DEV_UNSUPPORTED_CMD;
                    break;
                }

                // Take control of gpio mutex
                eReturnCode = OSAL.eSemTake(
                    hMutex, OSAL_OBJ_TIMEOUT_INFINITE );
                if (eReturnCode == OSAL_SUCCESS)
                {
                    if (n32Cmd == SRH_IOCTL_APP_RESET)
                    {
                        psSRM->sReset.pvReset();
                    }
                    else
                    {
                        // Reset SRM (RESET_M)
                        if(psSRM->sReset.pvEnable != SRH_NULL_GPIO)
                        {
                            psSRM->sReset.pvEnable();
                        }

                        // According to SX-9840-0017 Section 7.6 we need to
                        // pull RESET_M low for a minimum of 50 msec.
                        OSAL.eTaskDelay(100); // note: changed to 100 msec

                        // Release SRM (RESET_M)
                        if(psSRM->sReset.pvDisable != SRH_NULL_GPIO)
                        {
                            psSRM->sReset.pvDisable();
                        }
                    }

                    OSAL.eSemGive(hMutex);
                }
                else
                {
                    n32Err = DEV_MUTEX_FAILURE;
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_GET_APP_RESET_CONTROL:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            BOOLEAN *pbAppResetControl;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);
            pbAppResetControl = va_arg( *ptList, BOOLEAN*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, 
                psPublicData->psPhy, (UN8*)NULL);

            if(psSRM != NULL && pbAppResetControl != NULL)
            {
                // Process command
                *pbAppResetControl = psSRM->bAppResetControl;
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_GET_NUM_ANTENNAS:
        {
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            UN8 *pun8NumAntennas;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);
            pun8NumAntennas = va_arg( *ptList, UN8 *);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if(psSRM != NULL && pun8NumAntennas != NULL)
            {
                // Process command
                *pun8NumAntennas = psSRM->un8NumberOfAntennas;
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_SRM_EXISTS:
        {
            // Determine if a SRM (identified by string) exists
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if (psSRM == NULL)
            {
                // Sorry! No SRM found by that name
                n32Err = DEV_ERROR;
            }
        }
        break;

        case SRH_IOCTL_DESCRIBE_SRMS:
        {
            SRH_DEVICE_SRM_TUPLE_STRUCT **ppsMap, *psMap;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
            UN8 un8SRMIndex;
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            UN8 *pun8NumSRMs;
            SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

            // Extract input
            ppsMap = va_arg( *ptList, SRH_DEVICE_SRM_TUPLE_STRUCT**);
            pun8NumSRMs = va_arg( *ptList, UN8*);

            if ( (ppsMap == NULL) || (pun8NumSRMs == NULL) )
            {
                n32Err = DEV_INVALID_INPUT;
                break;
            }

            // Construct an appropriate name for memory to be allocated
            snprintf( &acName[0], sizeof(acName),
                      "SRM_DESC" );

            psMap = (SRH_DEVICE_SRM_TUPLE_STRUCT *)
                OSAL.pvMemoryAllocate(
                            &acName[0] ,
                            psPublicData->psPhy->un8NumberOfSRMs *
                                sizeof(SRH_DEVICE_SRM_TUPLE_STRUCT),
                            TRUE );

            if (psMap == NULL)
            {
                n32Err = DEV_OUT_OF_MEMORY;
                break;
            }

            // Scan through the list of SRMs
            for ( un8SRMIndex = 0;
                  un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
                  un8SRMIndex++)
            {
                psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

                //Copy SRM name into the Tuple
                psMap[un8SRMIndex].pacSRMName = psCurSRM->pacSRMName;

                // Initialize the SRM's capabilities to 0 -- this will
                // be populated as we inspect this SRM's decoders
                psMap[un8SRMIndex].tSRMCapabilities = 0;

                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules;
                     un8ModuleIndex++ )
                {
                    for ( un32DecoderIndex = 0;
                      un32DecoderIndex <
                          psCurSRM->pasModuleMap[un8ModuleIndex].
                              un32NumDecoders;
                      un32DecoderIndex++ )
                    {
                        psMap[un8SRMIndex].tSRMCapabilities |=
                            psCurSRM->pasModuleMap[un8ModuleIndex].
                                pasDecoders[un32DecoderIndex].tDecoderCapabilities;
                    }
                }


            }

            *pun8NumSRMs = psPublicData->psPhy->un8NumberOfSRMs;
            *ppsMap = psMap;
            n32Err = DEV_OK;
        }
        break;

        case SRH_IOCTL_MODULE_EXISTS:
        {
            // Determine if a Module (identified by string) exists
            UN8 un8SRMIndex;
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            MODULE_ID *ptModuleId;
            BOOLEAN *pbPrimary;
            char *pacDecoderName;
            BOOLEAN bFound = FALSE;
            SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

            // Extract input
            pacDecoderName = va_arg( *ptList, char*);
            ptModuleId = va_arg( *ptList, MODULE_ID*);
            pbPrimary = va_arg( *ptList, BOOLEAN*);

            // Scan through the list of SRMs
            for ( un8SRMIndex = 0;
                  un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs && bFound == FALSE;
                  un8SRMIndex++)
            {
                psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

                // Scan through the list of modules
                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules && bFound == FALSE;
                     un8ModuleIndex++
                         )
                {

                    // Scan through the list of decoders supported and determine
                    // which entry in the array matches this decoder string.
                    for( un32DecoderIndex = 0;
                         un32DecoderIndex < psCurSRM->pasModuleMap[un8ModuleIndex].un32NumDecoders;
                         un32DecoderIndex++
                             )
                    {
                        if(strcmp(pacDecoderName,psCurSRM->
                                  pasModuleMap[un8ModuleIndex].
                                      pasDecoders[un32DecoderIndex].
                                          pacDecoderName)==0 )
                        {
                            // Yep! It's here.

                            // copy module id
                            *ptModuleId =
                                psCurSRM->pasModuleMap[un8ModuleIndex].tModuleId;

                            // copy Primary flag
                            *pbPrimary =
                                psCurSRM->pasModuleMap[un8ModuleIndex].bPrimary;

                            bFound = TRUE;
                            break;
                        }
                    }
                }
            }

            // Check if one was found
            if(bFound == FALSE)
            {
                // Sorry! Not supported.
                n32Err = DEV_ERROR;
            }
        }
        break;

        case SRH_IOCTL_MODULE_CONFIGURATION:
        {
            UN8 un8ModuleIndex;
            MODULE_ID tModuleId;
            SXI_MODULE_CFG_STRUCT *psSXiModuleCfg;
            SRH_DEVICE_MODULE_MAP_STRUCT *psMap;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            char *pacSRMName;

            // Extract input
            pacSRMName = va_arg( *ptList, char *);
            tModuleId = (MODULE_ID)va_arg( *ptList, int);
            psSXiModuleCfg = va_arg(*ptList, SXI_MODULE_CFG_STRUCT*);

            // Set default return value
            n32Err = DEV_INVALID_INPUT;

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if ( (psSRM == NULL) || (psSXiModuleCfg == NULL))
            {
                break;
            }

            for( un8ModuleIndex = 0;
                un8ModuleIndex < psSRM->un8NumberOfModules;
                un8ModuleIndex++ )
            {
                psMap = &psSRM->pasModuleMap[un8ModuleIndex];
                if (psMap->tModuleId == tModuleId)
                {
                    *psSXiModuleCfg = psMap->sSXiConfig;
                    n32Err = DEV_OK;
                    break;
                }
            }
        }
        break;

        case SRH_IOCTL_PROTOCOL_IDENTIFIER:
        {
            // Return the protocol id
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            PROTOCOL_IDENTIFIER_ENUM *peProtId;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);
            peProtId = va_arg( *ptList, PROTOCOL_IDENTIFIER_ENUM *);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if(psSRM != NULL && peProtId != NULL)
            {
                // Process command
                *peProtId = psSRM->eProtocolId;
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case SRH_IOCTL_DECODER_EXISTS:
        {
            // Determine if a Decoder (identified by string) exists
            UN8 un8SRMIndex;
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            DECODER_ID *ptDecoderId;
            char *pacDecoderName;
            BOOLEAN bFound = FALSE;
            SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

            // Extract input
            pacDecoderName = va_arg( *ptList, char*);
            ptDecoderId = va_arg( *ptList, DECODER_ID*);

            // Scan through the list of SRMs
            for ( un8SRMIndex = 0;
                  un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
                  un8SRMIndex++)
            {
                psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

                // Scan through the list of modules
                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules;
                     un8ModuleIndex++
                         )
                {
                    // Scan through the list of decoders supported and determine
                    // which entry in the array matches this decoder string.
                    for( un32DecoderIndex = 0;
                         un32DecoderIndex < psCurSRM->
                             pasModuleMap[un8ModuleIndex].un32NumDecoders;
                         un32DecoderIndex++
                             )
                    {
                        if(strcmp(pacDecoderName,psCurSRM->pasModuleMap[un8ModuleIndex].
                                      pasDecoders[un32DecoderIndex].pacDecoderName)==0 )
                        {
                            // Yep! It's here.

                            // copy decoder id
                            if(ptDecoderId != NULL)
                            {
                                *ptDecoderId = psCurSRM->pasModuleMap[un8ModuleIndex].
                                        pasDecoders[un32DecoderIndex].tDecoderId;
                            }

                            bFound = TRUE;
                            break;
                        }
                    }
                    if (bFound == TRUE)
                    {
                        break;
                    }

                }
                // Check if one was found
                if(bFound == FALSE)
                {
                    // Sorry! Not supported.
                    n32Err = DEV_ERROR;
                }
            }
        }
        break;

        case SRH_IOCTL_DECODER_CAPABILITIES:
        {
            // Report the capabilities of the decoder described
            // by the user's inputs
            UN8 un8SRMIndex;
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            char *pacDecoderName;
            SRH_DEVICE_CAPABILITIES_MASK *ptDecoderCapabilities;
            BOOLEAN bFound = FALSE;
            SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

            // Extract input
            pacDecoderName = va_arg( *ptList, char*);
            ptDecoderCapabilities =
                va_arg( *ptList, SRH_DEVICE_CAPABILITIES_MASK*);

            // Scan through the list of SRMs
            for ( un8SRMIndex = 0;
                  un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
                  un8SRMIndex++)
            {
                psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

                // Scan through the list of modules
                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules;
                     un8ModuleIndex++
                         )
                {
                    // Scan through the list of decoders supported and determine
                    // which entry in the array matches this decoder string.
                    for( un32DecoderIndex = 0;
                         un32DecoderIndex < psCurSRM->pasModuleMap[un8ModuleIndex].un32NumDecoders;
                         un32DecoderIndex++
                             )
                    {
                        if(strcmp(pacDecoderName,psCurSRM->pasModuleMap[un8ModuleIndex].
                                      pasDecoders[un32DecoderIndex].pacDecoderName)==0 )
                        {
                            // Yep! It's here.

                            // copy decoder capabilities
                            *ptDecoderCapabilities =
                               psCurSRM->pasModuleMap[un8ModuleIndex].
                                pasDecoders[un32DecoderIndex].tDecoderCapabilities;

                            bFound = TRUE;
                            break;
                        }
                    }
                }
            }

            if (bFound == FALSE)
            {
                // Didn't locate the requested decoder
                n32Err = DEV_ERROR;
            }
        }
        break;

        case SRH_IOCTL_DESCRIBE_DECODERS:
        {
            // Get the module map
            UN8 un8SRMIndex;
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            MODULE_ID tModuleId;
            SRH_DEVICE_DECODER_TUPLE_STRUCT **pvMap, *pvMapTmp;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
            UN32 *pun32NumDecoders;
            SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

            // Extract input
            tModuleId = (MODULE_ID)va_arg( *ptList, int);
            pvMap = va_arg( *ptList, SRH_DEVICE_DECODER_TUPLE_STRUCT**);
            pun32NumDecoders = va_arg( *ptList, UN32*);

            n32Err = DEV_INVALID_INPUT;

            // Construct an appropriate name for memory to be allocated
            snprintf(&acName[0], sizeof(acName),
                "MODULEMAP:%u", tModuleId);

            // Scan through the list of SRMs
            for ( un8SRMIndex = 0;
                  un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
                  un8SRMIndex++)
            {
                psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules;
                     un8ModuleIndex++
                         )
                {
                    if ( psCurSRM->pasModuleMap[un8ModuleIndex].tModuleId == tModuleId)
                    {
                        size_t tSize;

                        // Allocate a sturcture to hold the public control structure
                        // required by the driver.
                        // Set the public data pointer for the Framework to maintain
                        *pun32NumDecoders =
                            psCurSRM->pasModuleMap[un8ModuleIndex].un32NumDecoders;

                        tSize = sizeof(SRH_DEVICE_DECODER_TUPLE_STRUCT) *
                                       (*pun32NumDecoders);

                        pvMapTmp =
                            (SRH_DEVICE_DECODER_TUPLE_STRUCT*)OSAL.pvMemoryAllocate(
                                acName,
                                tSize,
                                TRUE );
                        if(pvMapTmp == NULL)
                        {
                            return DEV_INVALID_INPUT;
                        }


                        for( un32DecoderIndex = 0;
                             un32DecoderIndex < *pun32NumDecoders;
                             un32DecoderIndex++
                                )
                        {
                            // Copy the data into the tuple struct

                            // Decoder Name
                            pvMapTmp[un32DecoderIndex].pacDecoderName =
                                psCurSRM->pasModuleMap[un8ModuleIndex].
                                    pasDecoders[un32DecoderIndex].pacDecoderName;

                            // Decoder Path Id
                            pvMapTmp[un32DecoderIndex].tDecoderPathId =
                                psCurSRM->pasModuleMap[un8ModuleIndex].
                                    pasDecoders[un32DecoderIndex].tDecoderId;

                            // Decoder Capabilities
                            pvMapTmp[un32DecoderIndex].tDecoderCapabilities =
                                psCurSRM->pasModuleMap[un8ModuleIndex].
                                    pasDecoders[un32DecoderIndex].tDecoderCapabilities;
                        }
                        *pvMap = pvMapTmp;
                        n32Err = DEV_OK;
                        break;

                    }
                }
            }
        }
        break;

        case SRH_IOCTL_DESCRIBE_MODULES:
        {
            SRH_DEVICE_MODULE_TUPLE_STRUCT **ppsMap, *psMap;
            char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            UN32 *pun32NumModules;
            char *pacSRMName;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;

            // Extract input
            pacSRMName = va_arg( *ptList, char*);
            ppsMap = va_arg( *ptList, SRH_DEVICE_MODULE_TUPLE_STRUCT**);
            pun32NumModules = va_arg( *ptList, UN32*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if ( (psSRM == NULL) || (ppsMap == NULL) ||
                            (pun32NumModules == NULL) )
            {
                n32Err = DEV_INVALID_INPUT;
                break;
            }

            // Construct an appropriate name for memory to be allocated
            snprintf( &acName[0], sizeof(acName),
                      "MODULE_DESC" );

            psMap = (SRH_DEVICE_MODULE_TUPLE_STRUCT *)
                OSAL.pvMemoryAllocate(
                            &acName[0] ,
                            psSRM->un8NumberOfModules *
                                sizeof(SRH_DEVICE_MODULE_TUPLE_STRUCT),
                            TRUE );

            if (psMap == NULL)
            {
                n32Err = DEV_OUT_OF_MEMORY;
                break;
            }

            for( un8ModuleIndex = 0;
                 un8ModuleIndex < psSRM->un8NumberOfModules;
                 un8ModuleIndex++ )
            {
                // Populate module ID and the primary flag
                psMap[un8ModuleIndex].tModuleId =
                    psSRM->pasModuleMap[un8ModuleIndex].tModuleId;
                psMap[un8ModuleIndex].bPrimary =
                    psSRM->pasModuleMap[un8ModuleIndex].bPrimary;

                // Initialize the module's capabilities to 0 -- this will
                // be populated as we inspect this module's decoder map
                psMap[un8ModuleIndex].tModuleCapabilities = 0;

                for ( un32DecoderIndex = 0;
                      un32DecoderIndex <
                          psSRM->pasModuleMap[un8ModuleIndex].
                              un32NumDecoders;
                      un32DecoderIndex++ )
                {
                    // Put together the capabilities of all the decoders
                    // which belong to this module in order to compute the
                    // module's capabilities
                    psMap[un8ModuleIndex].tModuleCapabilities |=
                        psSRM->pasModuleMap[un8ModuleIndex].
                            pasDecoders[un32DecoderIndex].tDecoderCapabilities;
                }
            }

            *pun32NumModules = psSRM->un8NumberOfModules;
            *ppsMap = psMap;
            n32Err = DEV_OK;
        }
        break;

        case SRH_IOCTL_WRITE_DRIVER_CONFIGURATION:
        {
            UN8 un8SRMIndex;
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;

            FILE *psFileToWrite;

            // Extract input
            psFileToWrite = va_arg( *ptList, FILE*);

            // Verify input
            if (psFileToWrite == NULL)
            {
                n32Err = DEV_INVALID_INPUT;
                break;
            }

            // Write out SRH info
            fprintf(psFileToWrite, "[SRH]\n");
            fprintf(psFileToWrite, "\tNumSRMs: %d\n",
                psPublicData->psPhy->un8NumberOfSRMs);

            // Scan through the list of SRMs
            for ( un8SRMIndex = 0;
                  un8SRMIndex < psPublicData->psPhy->un8NumberOfSRMs;
                  un8SRMIndex++)
            {
                psCurSRM = &psPublicData->psPhy->pasSRMMap[un8SRMIndex];

                fprintf(psFileToWrite, "\t[SRM]\n");
                fprintf(psFileToWrite, "\t\tName: %s\n",
                    psCurSRM->pacSRMName);
                fprintf(psFileToWrite, "\t\tNumAntennas: %d\n",
                    psCurSRM->un8NumberOfAntennas);

                fprintf(psFileToWrite, "\t\tProtocol: ");

                if (psCurSRM->eProtocolId == PROTOCOL_IDENTIFIER_SSP)
                {
                    fprintf(psFileToWrite, "SSP");
                }
                else if (psCurSRM->eProtocolId == PROTOCOL_IDENTIFIER_SSP_V2)
                {
                    fprintf(psFileToWrite, "SSPv2");
                }
                else if (psCurSRM->eProtocolId == PROTOCOL_IDENTIFIER_SXI)
                {
                    fprintf(psFileToWrite, "SXI");
                }
                else
                {
                    fprintf(psFileToWrite, "%d", psCurSRM->eProtocolId);
                }
                fprintf(psFileToWrite, "\n");

                fprintf(psFileToWrite, "\t\tPort: %s\n",
                    psCurSRM->sPort.pcDeviceName);
                fprintf(psFileToWrite, "\t\tPortSettings: %s\n",
                    psCurSRM->sPort.pcMode);


                fprintf(psFileToWrite, "\t\tNumModules: %d\n",
                    psCurSRM->un8NumberOfModules);

                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules;
                     un8ModuleIndex++
                         )
                {
                    fprintf(psFileToWrite, "\t\t[MODULE]\n");
                    fprintf(psFileToWrite, "\t\t\tID: %d\n",
                        psCurSRM->pasModuleMap[un8ModuleIndex].tModuleId);
                    fprintf(psFileToWrite, "\t\t\tNumDecoders: %d\n",
                        psCurSRM->pasModuleMap[un8ModuleIndex].un32NumDecoders);

                    for( un32DecoderIndex = 0;
                         un32DecoderIndex < psCurSRM->pasModuleMap[un8ModuleIndex].un32NumDecoders;
                         un32DecoderIndex++
                            )
                    {
                        fprintf(psFileToWrite, "\t\t\t[DECODER]\n");
                        fprintf(psFileToWrite, "\t\t\t\tName: %s\n",
                                psCurSRM->pasModuleMap[un8ModuleIndex].
                                    pasDecoders[un32DecoderIndex].pacDecoderName);
                        fprintf(psFileToWrite, "\t\t\t\tID: %d\n",
                                psCurSRM->pasModuleMap[un8ModuleIndex].
                                    pasDecoders[un32DecoderIndex].tDecoderId);
                        fprintf(psFileToWrite, "\t\t\t\tCapabilities: ");

                        if (psCurSRM->pasModuleMap[un8ModuleIndex].
                                pasDecoders[un32DecoderIndex].tDecoderCapabilities &
                                SRH_DEVICE_CAPABILITY_AUDIO )
                        {
                            fprintf(psFileToWrite, "Audio ");
                        }

                        if (psCurSRM->pasModuleMap[un8ModuleIndex].
                                pasDecoders[un32DecoderIndex].tDecoderCapabilities &
                                SRH_DEVICE_CAPABILITY_DATA )
                        {
                            fprintf(psFileToWrite, "Data ");
                        }

                        if (psCurSRM->pasModuleMap[un8ModuleIndex].
                                pasDecoders[un32DecoderIndex].tDecoderCapabilities &
                                SRH_DEVICE_CAPABILITY_VIDEO )
                        {
                            fprintf(psFileToWrite, "Video ");
                        }

                        fprintf(psFileToWrite, "\n");

                        if (psCurSRM->pasModuleMap[un8ModuleIndex].
                                pasDecoders[un32DecoderIndex].psSDTP != NULL)
                        {
                            fprintf(psFileToWrite, "\t\t\t\tSDTPPort: %s\n",
                                    &psCurSRM->pasModuleMap[un8ModuleIndex].
                                        pasDecoders[un32DecoderIndex].psSDTP->sPort.pcDeviceName[0]);
                            fprintf(psFileToWrite, "\t\t\t\tSDTPPortSettings: %s\n",
                                    &psCurSRM->pasModuleMap[un8ModuleIndex].
                                        pasDecoders[un32DecoderIndex].psSDTP->sPort.pcMode[0]);
                            fprintf(psFileToWrite, "\t\t\t\tSDTPPortNumber: %u\n",
                                    psCurSRM->pasModuleMap[un8ModuleIndex].
                                        pasDecoders[un32DecoderIndex].psSDTP->un8PortId);
                        }

                        fprintf(psFileToWrite, "\t\t\t[END DECODER]\n");
                    }

                    fprintf(psFileToWrite, "\t\t[END MODULE]\n");
                }

                fprintf(psFileToWrite, "\t[END SRM]\n");
            }

            n32Err = fprintf(psFileToWrite, "[END SRH]\n");

            if (n32Err > 0)
            {
                n32Err = DEV_OK;
            }
            else
            {
                n32Err = DEV_ERROR;
            }
        }
        break;

        case SRH_IOCTL_DESCRIBE_PORT:
        {
            // Report the configuration of the sdtp port described
            // by the user's inputs
            UN8 un8ModuleIndex;
            UN32 un32DecoderIndex;
            MODULE_ID tModuleId;
            DECODER_ID tDecoderId;
            SRH_DEVICE_PORT_TUPLE_STRUCT *psTuple;
            SRH_DEVICE_SRM_MAP_STRUCT *psSRM;
            BOOLEAN bFound = FALSE;
            BOOLEAN bIsCommandPort;
            char *pacSRMName;

            // Extract input
            bIsCommandPort = (BOOLEAN)va_arg( *ptList, int);
            pacSRMName = va_arg( *ptList, char *);
            tModuleId = (MODULE_ID)va_arg( *ptList, int);
            tDecoderId = (DECODER_ID)va_arg( *ptList, int);
            psTuple =
                va_arg( *ptList, SRH_DEVICE_PORT_TUPLE_STRUCT*);

            // See if this SRM exists
            psSRM = _psFindSRMInSRH(pacSRMName, psPublicData->psPhy, NULL);

            if (psSRM == NULL)
            {
                n32Err = DEV_INVALID_INPUT;
                break;
            }
            else
            {
                BOOLEAN bOk;

                if (bIsCommandPort == TRUE)
                {
                    bFound = TRUE;

                    // Process the port mode text and convert
                    bOk = _bProcessPortMode(
                        &psSRM->sPort.pcMode[0],
                        psTuple );

                    if (bOk == FALSE)
                    {
                        n32Err = DEV_ERROR;
                    }
                }
                else
                {

                    SRH_DEVICE_DECODER_MAP_STRUCT *psDecoder;

                    // Scan through the list of modules
                    for( un8ModuleIndex = 0;
                         un8ModuleIndex < psSRM->un8NumberOfModules;
                         un8ModuleIndex++
                             )
                    {
                        // Find the module id specified
                        if ( psSRM->pasModuleMap[un8ModuleIndex].tModuleId == tModuleId )
                        {
                            // Scan through the list of decoders in this module and determine
                            // which entry in the array matches this decoder id.
                            for( un32DecoderIndex = 0;
                                un32DecoderIndex < psSRM->pasModuleMap[un8ModuleIndex].un32NumDecoders;
                                un32DecoderIndex++
                                )
                            {
                                // If the decoder ids match, we found our decoder
                                if (tDecoderId == psSRM->pasModuleMap[un8ModuleIndex].
                                          pasDecoders[un32DecoderIndex].tDecoderId )
                                {
                                    if (psSRM->pasModuleMap[un8ModuleIndex].
                                          pasDecoders[un32DecoderIndex].psSDTP != NULL)
                                    {
                                        psDecoder =
                                            &psSRM->pasModuleMap[un8ModuleIndex].
                                              pasDecoders[un32DecoderIndex];

                                        // Copy the port id
                                        psTuple->un8PortId = psDecoder->psSDTP->un8PortId;

                                        // Process the port mode text and convert
                                        bOk = _bProcessPortMode(
                                            &psDecoder->psSDTP->sPort.pcMode[0],
                                            psTuple );

                                        if (bOk == FALSE)
                                        {
                                            n32Err = DEV_ERROR;
                                        }
                                    }
                                    bFound = TRUE;
                                    break;
                                }
                            }
                            break;
                        }

                    }
                }
            }

            if (bFound == FALSE)
            {
                // Didn't locate the requested port
                n32Err = DEV_ERROR;
            }
        }
        break;

        case SRH_IOCTL_DEVICE_GROUP:
        {
            DEVICE_GROUP *ptDeviceGroup;

            ptDeviceGroup = va_arg( *ptList, DEVICE_GROUP*);

            // Check the provisioned device group
            // is valid
            if (VERIFY_DEVICE_GROUP(psPublicData->psPhy->tDeviceGroup) == TRUE)
            {
                *ptDeviceGroup = psPublicData->psPhy->tDeviceGroup;
                n32Err = DEV_OK;
            }
            else
            {
                // Error!
                n32Err = DEV_ERROR;
            }
        }
        break;

        default:

            // Not an SRH ioctl, might be a request for an underlying driver
            // since the SRH driver is a 'stacked' driver.
            n32Err =
                psPrivateData->psFile->psIntf->iIoctl(
                    (FILE *)psPrivateData->psFile->pvHdl, n32Cmd, ptList);

        break;
    }

    return n32Err;
}

/*****************************************************************************
*
*       _psParseArg
*
*   This function parses the input SRM port string.
*
*   Inputs:
*       pcArg - A pointer to the port string provided by the
*           caller (application). Ex: "ssp"
*
*   Outputs:
*       pointer to the file handler for the port to be opened.
*
*****************************************************************************/
static FILE *_psParseArg (
    SRH_DEVICE_STRUCT *psPhy,
    const char *pcArg
    )
{
    FILE *psFile = (FILE *)NULL;

    // Verify a valid argument was passed.
    if(pcArg == NULL)
    {
        return (FILE*)NULL;
    }

    // Open appropriate port...
    if (strstr(pcArg, "ssp") != 0 ||
        strstr(pcArg, "sdtp") != 0  ||
        strstr(pcArg, "sxi") != 0)
    {
        UN8 un8SRMIndex;
        UN8 un8ModuleIndex;
        UN32 un32DecoderIndex;
        BOOLEAN bFound = FALSE;
        char *pacHardwareName;
        SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM;
        SRH_DEVICE_MODULE_MAP_STRUCT *psCurModule;

        // Now extract the hardware names from the input
        pacHardwareName = strchr( pcArg, ':' );
        pacHardwareName += 1;

        // Scan through the hardware maps and determine
        // which hardware is identified by this name.
        for (un8SRMIndex = 0;
             un8SRMIndex < psPhy->un8NumberOfSRMs && bFound == FALSE;
             un8SRMIndex++ )
        {
            psCurSRM = &psPhy->pasSRMMap[un8SRMIndex];

            if (strstr(pcArg, "ssp") != 0 ||
                strstr(pcArg, "sxi") != 0 )
            {
                if(strcmp(pacHardwareName, psCurSRM->pacSRMName)==0 )
                {
                    // Open command & control port
                    // Note that the command & control port is
                    // opened for a specified SRM name
                    psFile =
                        fopen( psCurSRM->sPort.pcDeviceName, psCurSRM->sPort.pcMode );
                    if(psFile == NULL)
                    {
                        printf("Error! Unable to open device '%s' with mode '%s'.\n",
                            psCurSRM->sPort.pcDeviceName, psCurSRM->sPort.pcMode);
                    }

                    bFound = TRUE;
                    break;
                }
            }
            else if (strstr(pcArg, "sdtp") != 0)
            {
                // Don't make the ssp port search iterate modules & decoders
                for( un8ModuleIndex = 0;
                     un8ModuleIndex < psCurSRM->un8NumberOfModules && bFound == FALSE;
                     un8ModuleIndex++ )
                {
                    psCurModule = &psCurSRM->pasModuleMap[un8ModuleIndex];

                    for ( un32DecoderIndex = 0;
                          un32DecoderIndex < psCurModule->un32NumDecoders && bFound == FALSE;
                          un32DecoderIndex++ )
                    {
                        if(strcmp(pacHardwareName,
                                  psCurModule->pasDecoders[un32DecoderIndex].pacDecoderName)==0 )
                        {
                            // This is the decoder requested
                            // Open SDTP Port
                            psFile = fopen (
                                &psCurModule->pasDecoders[un32DecoderIndex].psSDTP->sPort.pcDeviceName[0],
                                &psCurModule->pasDecoders[un32DecoderIndex].psSDTP->sPort.pcMode[0]
                                );
                            if(psFile == NULL)
                            {
                                printf("Error! Unable to open device '%s' with mode '%s'.\n",
                                    psCurSRM->sPort.pcDeviceName, psCurSRM->sPort.pcMode);
                            }

                            bFound = TRUE;
                            break;

                        } // Compare names
                    } // Decoder iteration
                } // Module iteration
            } // SDTP port open
            else
            {
                // Error!
                printf("Error! Cannot handle argument '%s'.\n", pcArg);
                break;
            }

        } // SRM iteration
    }

    return psFile;
}


/*****************************************************************************
*
*       _bVerifyUniqueNames
*
*   This function loops through the SRH struct to verify that all the SRM and
*   decoder names in that SRM are valid
*
*   Inputs:
*       psPhy - A pointer to the SRH device struct that is being initilized
*
*   Outputs:
*       TRUE if the names of all the hardware names are unique
*       FALSE if the hardware names are not unique
*
*****************************************************************************/
static BOOLEAN _bVerifyUniqueNames (
    SRH_DEVICE_STRUCT *psPhy
    )
{
    UN8 un8SRMIndex, un8SRMVerifyIndex;
    UN8 un8ModuleIndex, un8ModuleVerifyIndex;
    UN32 un32DecoderIndex, un32DecoderVerifyIndex;
    const char *pacCurSRHName, *pacCurDecoderName;
    BOOLEAN bNamesAreUnique = TRUE;
    SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM, *psCurVerifySRM;
    SRH_DEVICE_MODULE_MAP_STRUCT *psCurModule, *psCurVerifyModule;
    SRH_DEVICE_DECODER_MAP_STRUCT *psCurDecoder, *psCurVerifyDecoder;

    for (un8SRMIndex = 0;
         un8SRMIndex < psPhy->un8NumberOfSRMs && bNamesAreUnique == TRUE;
         un8SRMIndex++)
    {
        psCurSRM = &psPhy->pasSRMMap[un8SRMIndex];
        pacCurSRHName = psCurSRM->pacSRMName;

        for (un8SRMVerifyIndex = un8SRMIndex+1;
             un8SRMVerifyIndex< psPhy->un8NumberOfSRMs && bNamesAreUnique == TRUE;
             un8SRMVerifyIndex++)
        {
            if (strcmp(pacCurSRHName, psPhy->pasSRMMap[un8SRMVerifyIndex].pacSRMName)==0)
            {
                printf("A non-unique SRM name was found %s\n",
                    psPhy->pasSRMMap[un8SRMVerifyIndex].pacSRMName);
                bNamesAreUnique = FALSE;
            }
        }

        // Check Decoder Names
        for (un8ModuleIndex = 0;
             un8ModuleIndex < psCurSRM->un8NumberOfModules && bNamesAreUnique == TRUE;
             un8ModuleIndex++)
        {
            psCurModule = &psCurSRM->pasModuleMap[un8ModuleIndex];

            for (un32DecoderIndex = 0;
                 un32DecoderIndex < psCurModule->un32NumDecoders
                    && bNamesAreUnique == TRUE;
                 un32DecoderIndex++)
            {
                psCurDecoder = &psCurModule->pasDecoders[un32DecoderIndex];
                pacCurDecoderName = psCurDecoder->pacDecoderName;

                // Check the name of all the decoders in this module
                for (un32DecoderVerifyIndex = un32DecoderIndex+1;
                     un32DecoderVerifyIndex< psCurModule->un32NumDecoders && bNamesAreUnique == TRUE;
                     un32DecoderVerifyIndex++)
                {
                    psCurVerifyDecoder =
                        &psCurModule->pasDecoders[un32DecoderVerifyIndex];

                    if (strcmp(pacCurDecoderName, psCurVerifyDecoder->pacDecoderName)==0)
                    {
                        printf("A non-unique Decoder name was found %s\n",
                            psCurVerifyDecoder->pacDecoderName);
                        bNamesAreUnique = FALSE;
                    }
                }

                // Check the name of all the decoders in all other modules of this SRM
                for (un8ModuleVerifyIndex = un8ModuleIndex+1;
                     un8ModuleVerifyIndex < psCurSRM->un8NumberOfModules && bNamesAreUnique == TRUE;
                     un8ModuleVerifyIndex++)
                {
                    psCurVerifyModule = &psCurSRM->pasModuleMap[un8ModuleVerifyIndex];

                    for (un32DecoderVerifyIndex = 0;
                         un32DecoderVerifyIndex < psCurVerifyModule->un32NumDecoders
                            && bNamesAreUnique == TRUE;
                         un32DecoderVerifyIndex++)
                    {
                        psCurVerifyDecoder =
                            &psCurVerifyModule->pasDecoders[un32DecoderVerifyIndex];

                        if (strcmp(pacCurDecoderName, psCurVerifyDecoder->pacDecoderName)==0)
                        {
                            printf("A non-unique Decoder name was found %s\n",
                                psCurVerifyDecoder->pacDecoderName);
                            bNamesAreUnique = FALSE;
                        }
                    }

                }

                // Check the name of all the decoders in all other SRMs
                for (un8SRMVerifyIndex = un8SRMIndex+1;
                     un8SRMVerifyIndex< psPhy->un8NumberOfSRMs && bNamesAreUnique == TRUE;
                     un8SRMVerifyIndex++)
                {
                    psCurVerifySRM = &psPhy->pasSRMMap[un8SRMVerifyIndex];

                    for (un8ModuleVerifyIndex = 0;
                         un8ModuleVerifyIndex < psCurVerifySRM->un8NumberOfModules && bNamesAreUnique == TRUE;
                         un8ModuleVerifyIndex++)
                    {
                        psCurVerifyModule = &psCurVerifySRM->pasModuleMap[un8ModuleVerifyIndex];

                        for (un32DecoderVerifyIndex = 0;
                             un32DecoderVerifyIndex < psCurVerifyModule->un32NumDecoders && bNamesAreUnique == TRUE;
                             un32DecoderVerifyIndex++)
                        {
                            psCurVerifyDecoder =
                                &psCurVerifyModule->pasDecoders[un32DecoderVerifyIndex];

                            if (strcmp(pacCurDecoderName, psCurVerifyDecoder->pacDecoderName)==0)
                            {
                                bNamesAreUnique = FALSE;
                            }
                        }

                    }

                }
            }
        }
    }

    return bNamesAreUnique;
}

/*****************************************************************************
*
*       _psFindSRMInSRH
*
*   This function loops through the SRH struct to find a SRM by name.
*
*   Inputs:
*       pacSRMName - The name fo the SRM to find
*       psPhy - A pointer to the SRH device struct that is being initilized
*
*   Outputs:
*       A pointer to the SRM struct.  Will be NULL if not found.
*
*****************************************************************************/
static SRH_DEVICE_SRM_MAP_STRUCT *_psFindSRMInSRH (
    char *pacSRMName,
    SRH_DEVICE_STRUCT *psPhy,
    UN8 *pun8Index
    )
{
    UN8 un8SRMIndex;
    SRH_DEVICE_SRM_MAP_STRUCT *psCurSRM = NULL;

    for (un8SRMIndex = 0; un8SRMIndex < psPhy->un8NumberOfSRMs; un8SRMIndex++)
    {
        if ( strcmp(pacSRMName,
                    psPhy->pasSRMMap[un8SRMIndex].pacSRMName) ==0)
        {
            psCurSRM =  &psPhy->pasSRMMap[un8SRMIndex];
            if(pun8Index != NULL)
            {
                *pun8Index = un8SRMIndex;
            }
            break;
        }
    }

    return psCurSRM;
}

/*****************************************************************************
*
*       _bProcessPortMode
*
*   This function populates an SRH_DEVICE_SDTP_PORT_TUPLE_STRUCT based
*   upon the pcMode attribute of the SRH_DEVICE_PORT_STRUCT.
*
*   Inputs:
*       pacPortMode - The port mode (format: baud,parity,data bits, stop bits
*       psTuple - A pointer to the port tuplestruct that is being populated
*
*   Outputs:
*       BOOLEAN TRUE if okay, FALSE on error.
*
*****************************************************************************/
static BOOLEAN _bProcessPortMode (
    const char *pacPortMode,
    SRH_DEVICE_PORT_TUPLE_STRUCT *psTuple
    )
{
    char cParity;
    UN16 un16DataBits, un16StopBits;
    size_t tStrLen;

    // Get the length of the port mode provided
    tStrLen = strlen( pacPortMode );

    // Verify length of the port mode
    if ( tStrLen == 0)
    {
        return FALSE;
    }

    // Begin parsing argument (e.g. 9600,N,8,1)
    sscanf(pacPortMode, "%7u,%c,%1hu,%1hu",
           &psTuple->un32BaudRate, &cParity, &un16DataBits, &un16StopBits);

    // Normalize input of parity (which is a char) to lower case
    cParity = (char)tolower(cParity);

    // Copy in others
    psTuple->un8DataBits = (UN8)un16DataBits;
    psTuple->un8StopBits = (UN8)un16StopBits;

    // Translate parity
    switch (cParity)
    {
        case 'n':
        {
            psTuple->eParity = SRH_DEVICE_PORT_PARITY_NONE;
        }
        break;

        case 'e':
        {
            psTuple->eParity = SRH_DEVICE_PORT_PARITY_EVEN;
        }
        break;

        case 'o':
        {
            psTuple->eParity = SRH_DEVICE_PORT_PARITY_ODD;
        }
        break;

        default:
        {
            psTuple->eParity = SRH_DEVICE_PORT_PARITY_UNKNOWN;
        }
        break;
    }

    return TRUE;
}
