/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the device driver implementation for
 *  a SiriusXM Receiver Module (SRM) X65.
 *
 ******************************************************************************/

#include <stdio.h>

#include "standard.h"
#include "osal.h"

#include "_x65_driver.h"

/*****************************************************************************
*
*   SRH_vRegisterDriver
*
*   This function registers the Satellite Radio Hardware (SRH) X65.
*
*   Inputs:
*       pacUARTConfig - A c-style string that specifies the UART
*           parity, stop, start, and speed settings. If this parameter
*           is NULL, we will use the defaults from gasSRMMap.
*
*   Outputs:
*       None.
*
*****************************************************************************/
void SRH_vRegisterDriver( char *pacUARTConfig )
{
    ///////////////////////////////////
    // SRH Device Driver Registration
    ///////////////////////////////////

    SRH_DEVICE_STRUCT sSRHDevice;

    // Initialize structure
    OSAL.bMemSet(&sSRHDevice, 0, sizeof(sSRHDevice));

    // Our device group
    sSRHDevice.tDeviceGroup = X65_DEVICE_GROUP;

    // SRM configuration;  The array of SRMs provided MUST BE
    // a static array!
    sSRHDevice.un8NumberOfSRMs = X65_NUM_OF_SRMS;
    sSRHDevice.pasSRMMap = &gasSRMMap[0];

    // If application requests manual RESET control, it has to provide 
    // own reset function.
    if( (sSRHDevice.pasSRMMap->bAppResetControl == TRUE) && 
        (sSRHDevice.pasSRMMap->sReset.pvReset == NULL) )
    {
        OSAL.iFPrintf(stderr, 
            "***********************************************************\n"
            "* FATAL ERROR: Application HW control over GPIO lines is  *\n"
            "*              requested. Please assign sReset.pvReset    *\n"
            "*              function as well                           *\n"
            "***********************************************************\n"
            "\n");
    }

    // If the application passed in specific UART configuration
    // settings, set them here. Otherwise, we default to
    // "115200,N,8,1"
    if ( pacUARTConfig != NULL)
    {
        sSRHDevice.pasSRMMap->sPort.pcMode = pacUARTConfig;
    }

    // Register Device...
    OSAL.n32RegNode( "srh:",
                    &GsSRHInterface, // Use generic SRH interface (srh.h)
                    &sSRHDevice,
                    sizeof(SRH_DEVICE_STRUCT));

    return;
}

/*****************************************************************************
*
*   SRH_vUnRegisterDriver
*
*   This function unregisters the Satellite Radio Hardware (SRH) Cygnus
*   on the DM355EVM platform.
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*****************************************************************************/
void SRH_vUnRegisterDriver( void )
{
    // Unregister SRM device
    OSAL.bUnRegNode("srh:");

    return;
}
