/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the device driver implementation for
 *  the Memory Buffer vitrual device.
 *
 ******************************************************************************/
#include <string.h>
#include "standard.h"
#include "osal.h"

// We support some com IOCTLs
#include "com.h"

// Mem Buffer Driver Includes
#include "mem_buffer.h"
#include "_mem_buffer.h"

/*****************************************************************************
*
*       _bInit
*
*   Performs any device initialization for the virtual device this driver
*   is written for. This method is called only when the first driver instance
*   is opened.
*
*   Inputs:
*       pcDevName - A pointer to an ASCII null terminated string which
*           identifies this device. A copy of the provided name is retained
*           in the device node and this is a pointer to that name.
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       ppvPublicData - A pointer to a pointer which represents any
*           persistent object which contains device-global or public
*           data required by all driver instances.
*
*   Outputs:
*       TRUE on success. Otherwise FALSE should be returned on error.
*
*****************************************************************************/
static BOOLEAN _bInit (
                        const char *pcDevName,
                        void const *pvProperties,
                        void **ppvPublicData
                        )
{
    MEM_BUFFER_PUBLIC_STRUCT *psPublicData;
    MEM_BUFFER_PROPERTIES_STRUCT *psPropData =
        (MEM_BUFFER_PROPERTIES_STRUCT *)pvProperties;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Make sure that we have a property structure
    if (psPropData == NULL)
    {
        // Error!
        printf("MEM_BUFFER Error! No properties defined!.\n");
        return FALSE;
    }

    // Make sure we have approriate values in our property structure
    if ( (psPropData->un16NumBlocks == 0) ||
         (psPropData->un16BlockSize == 0) )
    {
        // Error!
        printf("MEM_BUFFER Error! invalid properties.\n");
        return FALSE;
    }

    // Allocate a sturcture to hold the public control structure
    // required by the driver.
    // Set the public data pointer for the Framework to maintain
    psPublicData =
    *ppvPublicData =
        OSAL.pvMemoryAllocate(
            pcDevName,
            sizeof(MEM_BUFFER_PUBLIC_STRUCT),
            TRUE );
    if(psPublicData == NULL)
    {
        // Error!
        printf("MEM_BUFFER Error! Cannot allocate memory.\n");
        return FALSE;
    }

    // Create the memory block pool and the buffer
    // Construct a unique name for the block pool
    snprintf( &acName[0], sizeof(acName), "%s:BlockPool", pcDevName);

    // Create an rx frame block pool
    eReturnCode =
        OSAL.eBlockPoolCreate (
            &psPublicData->hBlockPool,
            &acName[0],
            psPropData->un16BlockSize,
            psPropData->un16NumBlocks,
            OSAL_BLOCK_POOL_OPTION_NONE
            );

    if (eReturnCode != OSAL_SUCCESS)
    {
        // Error!
        printf("MEM_BUFFER Error! Cannot create block pool.\n");
        return FALSE;
    }

    // Create a buffer to read/write from.  Enable blocking
    psPublicData->hBuffer =
        OSAL.hBufferAllocate(
            psPublicData->hBlockPool,
            TRUE, // Enable read blocking
            TRUE, // Enable write blocking
            OSAL_BUFFER_ALLOCATE_OPTION_NONE );

    if (psPublicData->hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        // Error!
        printf("MEM_BUFFER Error! Cannot create buffer.\n");
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*       _vExit
*
*   Performs any device un-initialization or clean-up procedures required
*   when the device is no longer needed. This method is called only when
*   all open connections to the driver have been closed.
*
*   Inputs:
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       pvPublicData - - A pointer to a persistent object which contains
*           device-global or public data required by all driver instances.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void _vExit (
                        void const *pvProperties,
                        void *pvPublicData
                        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    MEM_BUFFER_PUBLIC_STRUCT *psPublicData =
        (MEM_BUFFER_PUBLIC_STRUCT *)pvPublicData;

    // Free our buffer
    eReturnCode = OSAL.eBufferFree(psPublicData->hBuffer);

    if (eReturnCode == OSAL_SUCCESS)
    {
        psPublicData->hBuffer = OSAL_INVALID_BUFFER_HDL;

        // Free the block pool
        eReturnCode = OSAL.eBlockPoolDelete(psPublicData->hBlockPool);

        if (eReturnCode == OSAL_SUCCESS)
        {
            psPublicData->hBlockPool = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Free any public data allocated by _bInit()
    OSAL.vMemoryFree(pvPublicData);

    return;
}

/*****************************************************************************
*
*       _n32Open
*
*   Opens a connection (instance) to the specified device by allocating any
*   resources or state information required for this instance.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver.
*       pvArg - A pointer to some device driver specific data which represents
*           a mode or configuration this instance is to be open with.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Open (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        const void *pvArg
                        )
{
	N32 n32RetVal = DEV_ERROR;
    BOOLEAN bValid;
    MEM_BUFFER_PRIVATE_STRUCT *psPrivateData = NULL;

    // Extract public data from the Device Driver Framework
    MEM_BUFFER_PUBLIC_STRUCT *psPublicData =
        (MEM_BUFFER_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    // Check input
    if(pvArg == NULL)
    {
        // Error!
        return DEV_ERROR;
    }

    do
    {
        // We can only allow two connections, one read and one write
        if(psDeviceInfo->un32Instance > 2)
        {
            // Bail, only two instances allowed
            break;
        }

        // Allocate private data structure
        psPrivateData = (MEM_BUFFER_PRIVATE_STRUCT *)
            OSAL.pvMemoryAllocate(
                psDeviceInfo->pcDevName,
                sizeof(MEM_BUFFER_PRIVATE_STRUCT),
                TRUE);

        if (psPrivateData == NULL)
        {
            break;
        }

        // Parse the arguments
        bValid = _bParseArg((char *)pvArg, &psPrivateData->eMode);

        if (bValid == FALSE)
        {
            // Error parsing the argument
            n32RetVal = DEV_INVALID_INPUT;
            break;
        }

        if (psPrivateData->eMode == MEM_BUFFER_MODE_READ &&
            psPublicData->psReadConnection == OSAL_INVALID_OBJECT_HDL)
        {
            psPublicData->psReadConnection = psPrivateData;
        }
        else if (psPrivateData->eMode == MEM_BUFFER_MODE_WRITE &&
                 psPublicData->psWriteConnection == OSAL_INVALID_OBJECT_HDL)
        {
            psPublicData->psWriteConnection = psPrivateData;
        }
        else
        {
            // Only allow two connections, one read and one write
            break;
        }

        // Inform the Framework of this private data
        // since we are all good at this point
        psDeviceInfo->pvPrivateData = (void *)psPrivateData;

        return DEV_OK;

    } while (FALSE);

    // We hit some sort of error, free memory and get out of here
    OSAL.vMemoryFree(psPrivateData);

    return n32RetVal;
}

/*****************************************************************************
*
*       _n32Close
*
*   Closes a connection (instance) to the specified device and releases any
*   resources obtained when the device was opened.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members
*           can be accessed to retrieve information about the device itself
*           and the driver instance. See Section 4.10.2 for more detail on
*           this structures members.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is returned
*       on error. See section 4.11 for possible error codes. If anything other
*       than DEV_OK is returned the instance handle will not be released and
*       will remain open.
*
*****************************************************************************/
static N32 _n32Close (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo
                        )
{
    // Extract public and private data from the Device Driver Framework
    MEM_BUFFER_PUBLIC_STRUCT *psPublicData =
        (MEM_BUFFER_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    MEM_BUFFER_PRIVATE_STRUCT *psPrivateData =
        (MEM_BUFFER_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Mark the connection as invalid in the public data
    if (psPrivateData->eMode == MEM_BUFFER_MODE_READ)
    {
        psPublicData->psReadConnection = OSAL_INVALID_OBJECT_HDL;
    }
    else if (psPrivateData->eMode == MEM_BUFFER_MODE_WRITE)
    {
        psPublicData->psWriteConnection = OSAL_INVALID_OBJECT_HDL;
    }
    else
    {
        return DEV_ERROR;
    }

    // Free our private data
    OSAL.vMemoryFree(psPrivateData);

    return DEV_OK;
}

/*****************************************************************************
*
*       _tRead
*
*   Allows byte wide (stream oriented) reads from the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcDst - The destination pointer where device data is to be read.
*       tSize - The size of each object to read.
*       tNumObjs - The number of objects to read.
*
*   Outputs:
*       The number of objects successfully read from the device and placed
*       in pcDst. If an error occurs or no data was read a zero is returned.
*       It should not be assumed that every read request must always be carried
*       out to completion in a single call. The caller is required to check the
*       return value to determine if all the requested data was read or if a
*       partial read was performed. If a partial read occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tRead (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        char *pcDst,
                        size_t tSize,
                        size_t tNumObjs
                        )
{
    size_t tBytesRead;
	size_t tLength = tSize * tNumObjs;

    // Extract public and private data from the Device Driver Framework
    MEM_BUFFER_PUBLIC_STRUCT *psPublicData =
        (MEM_BUFFER_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    MEM_BUFFER_PRIVATE_STRUCT *psPrivateData =
        (MEM_BUFFER_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Perform read from the head of the buffer
    tBytesRead = OSAL.tBufferReadHead(psPublicData->hBuffer,
        (void*)pcDst, tLength, psPrivateData->n32Timeout);

    // Calculate number of objects read
    tNumObjs = tBytesRead / tSize;

    return tNumObjs;
}

/*****************************************************************************
*
*       _tWrite
*
*   Allows byte wide (stream oriented) writes to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcSrc - The source pointer where device data is to be written from.
*       tSize - The size of each object to write.
*       tNumObjs - The number of objects to write.
*
*   Outputs:
*       The number of objects successfully written to the device from pcSrc.
*       If an error occurs or no data was written a zero is returned. It should
*       not be assumed that every write request must always be carried out to
*       completion in a single call. The caller is required to check the return
*       value to determine if all the requested data was written or if a
*       partial write was performed. If a partial write occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tWrite (
                         const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                         const char *pcSrc,
                         size_t tSize,
                         size_t tNumObjs
                         )
{
    size_t tBytesWritten;
    size_t tLength = tSize * tNumObjs;

    // Extract public and private data from the Device Driver Framework
    MEM_BUFFER_PUBLIC_STRUCT *psPublicData =
        (MEM_BUFFER_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    MEM_BUFFER_PRIVATE_STRUCT *psPrivateData =
        (MEM_BUFFER_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Perform write to the tail of the buffer
    tBytesWritten = OSAL.tBufferWriteTail(psPublicData->hBuffer,
        (void*)pcSrc, tLength, psPrivateData->n32Timeout);

    tNumObjs = tBytesWritten / tSize;

    // Return number of objects written
    return tNumObjs;
}


/*****************************************************************************
*
*       _n32Ioctl
*
*   Allows driver specific command and control to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       n32Cmd - A number assigned to a particular device command. Typically
*           this is defined in the device driver's public header file
*           (i.e. <device_driver_name>.h)
*       tList - A pointer to a variable argument list which contains zero
*           or more command specific arguments. The number and type of each
*           argument must be defined in a device driver document for this
*           particular device driver.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. Any unsupported ioctl commands should return
*       DEV_UNSUPPORTED_COMMAND. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Ioctl (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        N32 n32Cmd,
                        va_list *ptList
                        )
{
    N32 n32Err = DEV_OK;

    // Extract  private data from the Device Driver Framework
    MEM_BUFFER_PRIVATE_STRUCT *psPrivateData =
        (MEM_BUFFER_PRIVATE_STRUCT *)psDeviceInfo->pvPrivateData;

    // Handle command
    switch(n32Cmd)
    {
        case COM_IOCTL_SET_RTO:
        {
            if (psPrivateData->eMode == MEM_BUFFER_MODE_READ)
            {
                // Extract input
                N32 n32TimeoutMsec = va_arg( *ptList, N32 );

                // Set the timeout value
                psPrivateData->n32Timeout = n32TimeoutMsec;
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case COM_IOCTL_SET_WTO:
        {
            if (psPrivateData->eMode == MEM_BUFFER_MODE_WRITE)
            {
                // Extract input
                N32 n32TimeoutMsec = va_arg( *ptList, N32 );

                // Set the timeout value
                psPrivateData->n32Timeout = n32TimeoutMsec;
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case DEV_IOCTL_GET_ERROR:
        {
            // return the error indicator
            n32Err =  psPrivateData->n32Error;
        }
        break;

        case DEV_IOCTL_CLEAR_ERROR:
        {
            // clear any error
            psPrivateData->n32Error = 0;
        }
        break;

        default:
            // Return error, unhandled request
            n32Err = DEV_UNSUPPORTED_CMD;
        break;
    }

    return n32Err;
}

/*****************************************************************************
*
*       _bParseArg
*
*   This function parses the input line-parameters string.
*
*   Inputs:
*       pcArg - A pointer to the line-parameters string provided by the
*           caller (application). Ex: "w" or "r"
*       peMode - A pointer to the mode that has been parsed from the pcArg string
*
*   Outputs:
*       A value of TRUE is returned if the input parameters are valid, otherwise
*       a value of FALSE is returned on error.
*
*****************************************************************************/
static BOOLEAN _bParseArg (
    const char *pcArg,
    MEM_BUFFER_MODE_ENUM *peMode
    )
{

    // Verify a valid argument was passed
    if(pcArg == NULL)
    {
        return FALSE;
    }

    // Parse the mode
    if (strcmp(pcArg, "w") == 0)
    {
        *peMode = MEM_BUFFER_MODE_WRITE;
    }
    else if (strcmp(pcArg, "r") == 0)
    {
        *peMode = MEM_BUFFER_MODE_READ;
    }
    else
    {
        // unknown mode, return false
        *peMode = MEM_BUFFER_MODE_INVALID;
        return FALSE;
    }

    // We had valud input and could parse a known mode, return SUCCESS
    return TRUE;
}


