/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Object:SQL implementation for the
 *  Sirius Module Services (SMS)
 *
 ******************************************************************************/
#include <string.h>
#include <stdarg.h>

#include "standard.h"
#include "osal.h"

#include "sms_api.h"

#include "sql_interface_obj.h"
#include "_sql_interface_obj.h"

#include "sqlite3.h"


/*****************************************************************************
                             PUBLIC FUNCTIONS
*****************************************************************************/

/*****************************************************************************
*
*   bInitializeSqlite
*
*****************************************************************************/
static BOOLEAN bInitializeSqlite ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    int iSqliteRet;

    // Has the control struct been created yet?
    if (NULL != gpsCntrlStruct)
    {
        // Error! This object must be initialized only once!
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: SQL Interface is already initialized.\n");
        return FALSE;
    }

    // Try to initialize control object
    do
    {
        gpsCntrlStruct = (SQL_INTERFACE_CTRL_STRUCT *)
            OSAL.pvMemoryAllocate(
                SQL_CONTROL_OBJECT_NAME,
                sizeof(SQL_INTERFACE_CTRL_STRUCT),
                TRUE );
        if (NULL == gpsCntrlStruct)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": ERROR: Unable to allocate SQL Interface Control Object.\n");
            break;
        }

        gpsCntrlStruct->un16NumConnections = 0;
        gpsCntrlStruct->hSem = OSAL_INVALID_OBJECT_HDL;

        // Create Control Semaphore (initially unlocked)
        eReturnCode = OSAL.eSemCreate(
            &(gpsCntrlStruct->hSem),
            SQL_CONTROL_SEMAPHORE_NAME,
            1, 1, OSAL_SEM_OPTION_NONE);
        if (OSAL_SUCCESS != eReturnCode)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": ERROR: Unable to create SQL Interface Control Semaphore.\n");
            break;
        }

#if ENABLE_SQL_LOGGING == 1
        iSqliteRet = sqlite3_config(SQLITE_CONFIG_LOG, vLogSQL, NULL);
        if (SQLITE_OK != iSqliteRet)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": ERROR: Could not enable SQLite logging service.\n");
        }
#endif

        // Initialize SQLite library
        iSqliteRet = sqlite3_initialize();
        if (SQLITE_OK != iSqliteRet)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": ERROR: Unable to initialize SQLite.\n");
            break;
        }

        // Everything's OK
        return TRUE;

    } while (FALSE);

    // Error. Rollback.
    if (NULL != gpsCntrlStruct)
    {
        if (OSAL_INVALID_OBJECT_HDL != gpsCntrlStruct->hSem)
        {
            eReturnCode = OSAL.eSemDelete(gpsCntrlStruct->hSem);
            if (OSAL_SUCCESS != eReturnCode)
            {
                printf(SQL_INTERFACE_OBJECT_NAME
                    ": ERROR: Unable to delete SQL Interface Control Semaphore.\n");
            }
            gpsCntrlStruct->hSem = OSAL_INVALID_OBJECT_HDL;
        }

        OSAL.vMemoryFree(gpsCntrlStruct);
        gpsCntrlStruct = NULL;
    }

    return FALSE;
}

/*****************************************************************************
*
*   vUninitializeSqlite
*
*****************************************************************************/
static void vUninitializeSqlite ( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (NULL == gpsCntrlStruct)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: SQL Interface is not initialized.\n");
        return;
    }

    // Check if the object is busy
    eReturnCode = OSAL.eSemTake(gpsCntrlStruct->hSem, OSAL_OBJ_TIMEOUT_NONE);
    if (OSAL_SUCCESS != eReturnCode)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: Unable to uninitialize SQL Interface."
            " SQL Object is busy. \n");
        return;
    }

    if (0 != gpsCntrlStruct->un16NumConnections)
    {
        // Just output error message
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: Not all SQL connections are closed.\n");
    }

#ifdef SQLITE_OMIT_AUTOINIT
    sqlite3_shutdown();
#endif

    eReturnCode = OSAL.eSemDelete(gpsCntrlStruct->hSem);
    if (OSAL_SUCCESS != eReturnCode)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: Unable to delete SQL Control Semaphore.\n");
    }
    gpsCntrlStruct->hSem = OSAL_INVALID_OBJECT_HDL;

    OSAL.vMemoryFree(gpsCntrlStruct);
    gpsCntrlStruct = NULL;

    return;
}

/*****************************************************************************
*
*   hConnectSqlite
*
*       This API is used to establish a connection with an sqlite database. If
*       susccessful, the handle returned will be used to interact with the
*       database (queries, updates etc)
*
*   Inputs:
*       pacDatabaseFile - name of database file (including path) to connect to
*       bReadOnlyDB - flag to indicate if the database file is read-only
*           (TRUE = read-only  FALSE = read/write)
*       peErrorCode - an enumerated error code which will indicate the type of
*           error which caused the connection not successfully be established.
*
*   Outputs:
*       Valid SQL_INTERFACE_OBJECT handle on success;
*       SQL_INTERFACE_INVALID_OBJECT on failure
*****************************************************************************/
SQL_INTERFACE_OBJECT hConnectSqlite (
    const char *pacDatabaseFile,
    SQL_INTERFACE_OPTIONS_MASK tOptions,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    DATASERVICE_ERROR_CODE_ENUM eLocalDSErrorCode;
    SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)NULL;

    if (peErrorCode == NULL)
    {
        // If they didn't provide an error code pointer
        // just use our local one.
        peErrorCode = &eLocalDSErrorCode;
    }

    // Initialize error code
    *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;

    if (tOptions == SQL_INTERFACE_OPTIONS_ALL)
    {
        return SQL_INTERFACE_INVALID_OBJECT;
    }

    if( (tOptions & ~SQL_INTERFACE_OPTIONS_ALL) != SQL_INTERFACE_OPTIONS_NONE )
    {
        return SQL_INTERFACE_INVALID_OBJECT;
    }

    // has the control struct been created yet?
    if (NULL == gpsCntrlStruct)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: SQL Interface is not initialized.\n");
        return SQL_INTERFACE_INVALID_OBJECT;
    }

    // Lock the control struct
    eReturnCode = OSAL.eSemTake(gpsCntrlStruct->hSem, OSAL_OBJ_TIMEOUT_INFINITE);
    if (eReturnCode == OSAL_SUCCESS)
    {
        // now the cntrl struct is locked

        char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
        static UN32 un32InitInstance = 0;
        BOOLEAN bDatabaseOk = FALSE, bPerformFSCheck = TRUE;
        UN8 un8Attributes = 0;
        int flags = SQLITE_OPEN_READONLY;
        int iStrCmpResult;

        // do the following
        do
        {
            // From now we assume that there is no error
            *peErrorCode = DATASERVICE_ERROR_CODE_NONE;

            // 0. see if this is a file on the file system
            //    or a special sqlite database type
            if (pacDatabaseFile[0] == '\0')
            {
                // An empty string indicates that the caller wants
                // a database with a temp file name
                bPerformFSCheck = FALSE;
                flags = SQLITE_OPEN_CREATE | SQLITE_OPEN_READWRITE;
            }
            else
            {
                iStrCmpResult = strcmp(pacDatabaseFile, SQL_MEMORY_DATABASE_NAME);
                if (iStrCmpResult == 0)
                {
                    // The :memory: name indicates that the caller wants to
                    // open a memory-based database
                    bPerformFSCheck = FALSE;
                    flags = SQLITE_OPEN_CREATE | SQLITE_OPEN_READWRITE;
                }
            }

            // Check file presence / attributes if we need to do a file system check
            if (bPerformFSCheck == TRUE)
            {
                BOOLEAN bFileFound;

                bFileFound = OSAL.bFileSystemGetFileAttributes( pacDatabaseFile, &un8Attributes);

                if (bFileFound == TRUE)
                {
                    // If we are going to write to this file, verify attributes allow it
                    if ((tOptions & SQL_INTERFACE_OPTIONS_READONLY) != SQL_INTERFACE_OPTIONS_READONLY)
                    {
                        if ((un8Attributes & OSAL_FILE_ATTR_WRITE) != OSAL_FILE_ATTR_WRITE )
                        {
                            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_READONLY;
                            break;
                        }

                        flags = SQLITE_OPEN_READWRITE;
                    }
                }
                else
                {
                    // File not found, so set the error code to indicate this
                    *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND;

                    // If we aren't supposed to create the file, exit now
                    if ((tOptions & SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND) !=
                        SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND)
                    {
                        break;
                    }

                    flags = SQLITE_OPEN_CREATE | SQLITE_OPEN_READWRITE;
                }

            }

            // Construct a name
            snprintf( &acName[0], sizeof(acName),
                     SQL_INTERFACE_OBJECT_NAME":%u",
                     un32InitInstance++ );

            // Construct an object for this connection
            psObj = (SQL_OBJECT_STRUCT *)OSAL.pvMemoryAllocate(
                &acName[0],
                sizeof(SQL_OBJECT_STRUCT),
                TRUE);
            if(psObj == NULL)
            {
                // Error!
                printf(SQL_INTERFACE_OBJECT_NAME": Unable to create SQL interface object.\n");
                *peErrorCode = DATASERVICE_ERROR_CODE_GENERAL;
                break;
            }

            // Initialize the buffer so that it is of size 0+SQL_BUFFER_INCREASE_MARGIN
            bExpandBuffer(psObj, 0, FALSE);

            // Initialize the database handle
            psObj->psDB = (sqlite3 *)NULL;

            // 5. open the db
            psObj->iLastSQLErrorCode =
                sqlite3_open_v2(
                    pacDatabaseFile,       // Name of database file
                    &psObj->psDB,          // Pointer to handle
                    flags,                 // Flags configured earlier
                    (const char*)0 );      // VFS to use (NULL is default)

            // 6. check integrity if we aren't skipping it
            if (psObj->iLastSQLErrorCode == SQLITE_OK)
            {
#if OSAL_DEBUG == 1
                // If we are a debug build, enable the extended result codes before
                // we do any further work.  I guess we could always enable these, I'm not sure
                // what the performance hit will be if they were always enabled.  They are just
                // not useful outside of debugging.
                psObj->iLastSQLErrorCode = sqlite3_extended_result_codes(psObj->psDB, TRUE);

                if (psObj->iLastSQLErrorCode != SQLITE_OK)
                {
                    fprintf(stderr, "can't enable extended error codes\n");
                }
#endif
                if ( (tOptions & SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK) !=
                         SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK)
                {
                    bDatabaseOk = bCheckIntegrity((SQL_INTERFACE_OBJECT)psObj);
                    if (bDatabaseOk == FALSE)
                    {
                        *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_CORRUPT;
                        printf(SQL_INTERFACE_OBJECT_NAME": DB corrupt %s\n", pacDatabaseFile);
                    }
                }
                else
                {
                    // Skipping the DB check, mark it as ok
                    bDatabaseOk = TRUE;
                }
            }
            else
            {
                printf(SQL_INTERFACE_OBJECT_NAME
                    ": Unable to open DB: sqlite3_open_v2() returned %u\n",
                    psObj->iLastSQLErrorCode);
                 *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
            }

            // If the connection failed or the database is corrupt,
            // we must fail this call too
            if ((psObj->iLastSQLErrorCode != SQLITE_OK) || (bDatabaseOk == FALSE))
            {
                // Close the database connection and
                // destroy / clear our object
                vDestroyObject(psObj);
                psObj = (SQL_OBJECT_STRUCT*)SQL_INTERFACE_INVALID_OBJECT;

                break;
            }

            // We now have one more open connection
            gpsCntrlStruct->un16NumConnections++;

            if (*peErrorCode != DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND)
            {
                // Report that there are no errors unless we ended up
                // creating a new database.
                *peErrorCode = DATASERVICE_ERROR_CODE_NONE;
            }

        } while (FALSE);

        // unlock the cntrl struct
        OSAL.eSemGive(gpsCntrlStruct->hSem);
    }
    else
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: Unable to lock SQL Interface Control object.\n");
    }

    return (SQL_INTERFACE_OBJECT)psObj;
}

/*****************************************************************************
*
*   vDisconnectSqlite
*
*       This API is used to close an existing connection with a database, and
*       free any internal resouces created.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database being
*           disconnected from
*
*   Outputs:
*       None
*****************************************************************************/
static void vDisconnectSqlite (
    SQL_INTERFACE_OBJECT hSQL
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    if (NULL == gpsCntrlStruct)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: SQL Interface is not initialized.\n");
        return;
    }

    // Lock the control struct
    eReturnCode = OSAL.eSemTake(gpsCntrlStruct->hSem, OSAL_OBJ_TIMEOUT_INFINITE);
    if (eReturnCode == OSAL_SUCCESS)
    {
        SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

        // Destroy this connection object
        vDestroyObject(psObj);

        // Can we decrement the number of open
        // connections?
        if (gpsCntrlStruct->un16NumConnections > 0)
        {
            // Yes, reduce by one
            gpsCntrlStruct->un16NumConnections--;
        }
        else
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": ERROR: Unexpected SQL disconnection request.\n");
        }

        // unlock the cntrl struct
        OSAL.eSemGive(gpsCntrlStruct->hSem);
    }
    else
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": ERROR: Unable to lock SQL Interface Control object.\n");
    }

    return;
}

/*****************************************************************************
*
*   bCheckIntegrity
*
*****************************************************************************/
static BOOLEAN bCheckIntegrity (
    SQL_INTERFACE_OBJECT hSQL
        )
{
    BOOLEAN bDatabaseOk = FALSE;

    if (hSQL != SQL_INTERFACE_INVALID_OBJECT)
    {
        SQL_OBJECT_STRUCT *psObj =
            (SQL_OBJECT_STRUCT *)hSQL;
        BOOLEAN bSuccess;

        // Query the DB to check integrity
        bSuccess = bQuery (
            (SQL_INTERFACE_OBJECT)psObj,
            SQL_INTEGRITY_CHECK,
            bIntegrityQueryResult,
            (void *)&bDatabaseOk
                );
        if (bSuccess == FALSE)
        {
            bDatabaseOk = FALSE;
        }
    }

    return bDatabaseOk;
}

/*****************************************************************************
*
*   bCopyMainDatabase
*
*****************************************************************************/
static BOOLEAN bCopyMainDatabase (
    SQL_INTERFACE_OBJECT hDest,
    SQL_INTERFACE_OBJECT hSrc
        )
{
    BOOLEAN bResult = FALSE;
    SQL_OBJECT_STRUCT *psDestObj = (SQL_OBJECT_STRUCT *)hDest;
    SQL_OBJECT_STRUCT *psSrcObj = (SQL_OBJECT_STRUCT *)hSrc;
    sqlite3_backup *psBackup;
    UN32 un32SrcPageSize;

    if ((hDest == NULL) || (hSrc == NULL))
    {
        return bResult;
    }

    // Get the page size of the source DB
    un32SrcPageSize = un32DBPageSize(hSrc);
    if (un32SrcPageSize == 0)
    {
        return FALSE;
    }

    // Tell the dst DB to use the same page size,
    // but don't worry about the result because
    // we're just trying to be diligent about this
    // but it might not matter/work and we'll
    // only know how this goes by actually performing
    // the copy down below
    bSetDBPageSize(hDest, un32SrcPageSize);

    // Initialize the copy of the main databases.
    psBackup = sqlite3_backup_init(
        psDestObj->psDB,
        "main",
        psSrcObj->psDB,
        "main" );

    if (psBackup)
    {
        // -1 indicates all pages
        psDestObj->iLastSQLErrorCode = sqlite3_backup_step(psBackup, -1);

        if (psDestObj->iLastSQLErrorCode == SQLITE_DONE)
        {
            bResult = TRUE;
        }
        else
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Unable to copy DB: sqlite3_backup_step() returned %u\n",
                psDestObj->iLastSQLErrorCode);
        }

        psDestObj->iLastSQLErrorCode = sqlite3_backup_finish(psBackup);

        if (psDestObj->iLastSQLErrorCode != SQLITE_OK)
        {
            bResult = FALSE;
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Unable to copy DB: sqlite3_backup_finish() returned %u\n",
                psDestObj->iLastSQLErrorCode);
        }
    }

    return bResult;
}


/*****************************************************************************
*
*   bAttachDBToConnection
*
*       This API is used to attach another database to a pre-existing
*       connection with an sqlite database.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database on which to
*              attach the specified DB
*       pacDatabaseFile - name of database file (including path) to connect to
        pacAttachName - attach name to be used for any following SQL queries
*       peErrorCode - an enumerated error code which will indicate the type of
*           error if the attachment could not be made.
*
*   Outputs:
*       TRUE if the command was successfully executed; FALSE if it was not
*****************************************************************************/
static BOOLEAN bAttachDBToConnection (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacDatabaseFile,
    const char *pacAttachName,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        )
{
    // We have no errors yet ...
    *peErrorCode = DATASERVICE_ERROR_CODE_NONE;

    do
    {
        size_t tLength;
        char *pacErrorMessage, *pacResult, acBuffer[SQL_MAX_ATTACH_CMD_LEN];
        BOOLEAN bFileFound;
        UN8 un8Attributes;

        SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

        // Check to make sure we don't have a NULL reference
        // to our DB
        if ( SQL_INTERFACE_INVALID_OBJECT == hSQL )
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Passed NULL DB to eAttachDBToConnection!\n");
            break;
        }

        // Make sure our attach name isn't too long
        tLength = strlen(pacAttachName);
        if ( tLength > SQL_MAX_ATTACH_NAME_LEN )
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Attach name exceeds %d characters!\n",
                SQL_MAX_ATTACH_NAME_LEN);
            break;
        }

        // Make sure our file name isn't too long
        tLength = strlen(pacDatabaseFile);
        if ( tLength > SQL_MAX_PATH_LEN )
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Database name exceeds %d characters!\n",
                SQL_MAX_PATH_LEN);
            break;
        }

        // Make sure database exists on disk; if we call ATTACH with a
        // missing database, SQLITE will just create it for us, and
        // replacing the reference with a blank DB is NOT what we want!
        bFileFound = OSAL.bFileSystemGetFileAttributes( pacDatabaseFile, &un8Attributes);

        if ( FALSE == bFileFound )
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Can not attach a non-existent DB file: %s\n",
                pacDatabaseFile);
            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND;
            break;
        }

        // Clear out the work buffer
        OSAL.bMemSet(&acBuffer[0], (UN8)0, SQL_MAX_ATTACH_CMD_LEN);

        // As we're dealing with a filename, we have to the SQL interface's special
        // printf function (which contains format characters to escape things like
        // filenames in an SQL-compliant manner.)
        pacResult = sqlite3_snprintf(SQL_MAX_ATTACH_CMD_LEN, &acBuffer[0],
                SQL_ATTACH_DB, pacDatabaseFile, pacAttachName);

        if ( NULL == pacResult )
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                    ": call to sqlite3_snprintf failed!\n");
            break;
        }

        // Rather than defer to bExecuteCommand, we execute the cmd here (in order
        // to capture iLastSQLErrorCode)
        psObj->iLastSQLErrorCode =
                        sqlite3_exec(
                            psObj->psDB,           // Database
                            &acBuffer[0],          // SQL Command
                            NULL,                  // Callback function
                            NULL,                  // Callback argument
                            &pacErrorMessage);     // Error string

        // Verify the command succeeded
        if ( SQLITE_OK == psObj->iLastSQLErrorCode )
        {
            return TRUE;
        }

        // If the command wasn't successful, print & free the error
        // message -- (ensure it's there though)
        if ( NULL != pacErrorMessage )
        {
            printf(SQL_INTERFACE_OBJECT_NAME": Error! %s.\n", pacErrorMessage);
            sqlite3_free( pacErrorMessage );
        }

        if ( SQLITE_CORRUPT == psObj->iLastSQLErrorCode )
        {
            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_CORRUPT;
        }
        else
        {
            *peErrorCode = DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE;
        }

    } while ( FALSE );

    return FALSE;
}

/*****************************************************************************
*
*   vDetachDBFromConnection
*
*       This API is used to detach a secondary database from an existing
*       connection.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database being
*           disconnected from
*       pacAttachName - The SQL name that the database was attached under
*
*   Outputs:
*       None
*****************************************************************************/
static void vDetachDBFromConnection (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacAttachName
        )
{
    char *pacErrorMessage, *pacResult, acBuffer[SQL_MAX_DETACH_CMD_LEN];

    SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

    if (SQL_INTERFACE_INVALID_OBJECT == hSQL)
    {
        printf(SQL_INTERFACE_OBJECT_NAME": Invalid SQL object!\n");
        return;
    }

    // Clear out the work buffer
    OSAL.bMemSet(&acBuffer[0], (UN8)0, SQL_MAX_DETACH_CMD_LEN);

    // Generate the detach command
    pacResult = sqlite3_snprintf(SQL_MAX_DETACH_CMD_LEN, &acBuffer[0],
            SQL_DETACH_DB, pacAttachName);

    // If that failed, just return as there's really nothing else we
    // can do.
    if ( NULL == pacResult )
    {
        printf(SQL_INTERFACE_OBJECT_NAME
                ": call to sqlite3_snprintf failed!\n");
        return;
    }

    // Rather than defer to bExecuteCommand, we execute the cmd here (in order
    // to capture iLastSQLErrorCode)
    psObj->iLastSQLErrorCode =
                    sqlite3_exec(
                        psObj->psDB,           // Database
                        &acBuffer[0],          // SQL Command
                        NULL,                  // Callback function
                        NULL,                  // Callback argument
                        &pacErrorMessage);     // Error string

    // Verify the command succeeded
    if ( SQLITE_OK != psObj->iLastSQLErrorCode )
    {
        // If the command wasn't successful, print & free the error
        // message -- (ensure it's there though)
        if ( NULL != pacErrorMessage )
        {
            printf(SQL_INTERFACE_OBJECT_NAME": Error! %s.\n", pacErrorMessage);
            sqlite3_free( pacErrorMessage );
        }
    }
}

/*****************************************************************************
*
*   bExecuteCommand
*
*       This API is used to execute a SQL command for an exisitng valid
*       database connection.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database on which the
*              command is to be executed
*       pacCommandString - Valid pointer to an array of characters containing
*              the command (SQL syntax) to be executed
*
*   Outputs:
*       TRUE if the command was successfully executed; FALSE if it was not
*****************************************************************************/
static BOOLEAN bExecuteCommand (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacCommandString
        )
{
    // Ensure the object and pointer are valid
    if ((hSQL != NULL) && (pacCommandString != NULL))
    {
        SQL_OBJECT_STRUCT *psObj =
            (SQL_OBJECT_STRUCT *)hSQL;
        char *pacErrorMessage;

        // Execute the command
        psObj->iLastSQLErrorCode =
                        sqlite3_exec(
                            psObj->psDB,           // Database
                            &pacCommandString[0],  // SQL Command
                            NULL,                  // Callback function
                            NULL,                  // Callback argument
                            &pacErrorMessage);     // Error string

        // Verify the command succeeded
        if (psObj->iLastSQLErrorCode == SQLITE_OK)
        {
            return TRUE;
        }

        // If the command wasn't successful, print & free the error
        // message -- (ensure it's there though)
        if (pacErrorMessage != NULL)
        {
            printf(SQL_INTERFACE_OBJECT_NAME": Error! %s.\n", pacErrorMessage);
            sqlite3_free( pacErrorMessage );
        }
    }

    return FALSE;
}

/*****************************************************************************
*
*   bExecutePreparedCommand
*
*       This API is used to prepare and execute a SQL command for an exisitng
*       valid database connection. Preparing refers to binding values to
*       parameters in the query.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database on which the
*           command is to be executed
*       pacCommandString - Valid pointer to an array of characters containing
*           the command (SQL syntax) to be executed
*       tNumColumns - number of columns in the statement which must be bound in
*           the bColumnCallback
*       bColumnCallback - pointer to the column callback in which the binding
*           will be performed based on the column type
*       pvCallbackArg - pointer to a caller defined argument which will be sent
*           to the callback
*
*   Outputs:
*       TRUE if the command was successfully prepared and executed;
*       FALSE if it was not
*****************************************************************************/
static BOOLEAN bExecutePreparedCommand (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacCommandString,
    size_t tNumColumns,
    PREPARED_QUERY_COLUMN_CALLBACK bColumnCallback,
    void *pvCallbackArg
        )
{
    // Validate SMS Object

    // Ensure the object and pointer are valid
    if ((hSQL != NULL) && (pacCommandString != NULL) &&
        (bColumnCallback != NULL) && (tNumColumns > 0))
    {
        do
        {
            SQL_OBJECT_STRUCT *psObj =
                (SQL_OBJECT_STRUCT *)hSQL;
            sqlite3_stmt *psStatement = NULL;
            SQL_COLUMN_INDEX tColIdx;
            BOOLEAN bSuccess;
            SQL_BIND_TYPE_ENUM eCurColType = SQL_BIND_TYPE_UNKNOWN;
            void *pvCurColData = NULL;
            size_t tCurColDataSize = 0;

            // Prepare our statement so we can bind the data type
            // and populate the blob column
            psObj->iLastSQLErrorCode = sqlite3_prepare_v2(
                psObj->psDB,                      // Database
                &pacCommandString[0],             // SQL Command
                -1,                               // String is zero-terminated
                &psStatement,                     // Prepared statement pointer pointer
                NULL );                           // No Tail

            if (psObj->iLastSQLErrorCode != SQLITE_OK)
            {
                printf(SQL_INTERFACE_OBJECT_NAME
                    ": Error! Prepared Command Execution failure (%u)\n",
                    psObj->iLastSQLErrorCode);

                 break;
            }

            for (tColIdx = 0; tColIdx < tNumColumns; tColIdx++)
            {
                bSuccess = bColumnCallback(tColIdx, &eCurColType,
                    &tCurColDataSize, &pvCurColData, pvCallbackArg);

                if (bSuccess == FALSE)
                {
                    break;
                }

                switch (eCurColType)
                {
                    case SQL_BIND_TYPE_UN32:
                    {
                        psObj->iLastSQLErrorCode = sqlite3_bind_int(
                             psStatement,                   // Compiled statement
                             (int)tColIdx+1,                // SQL parameter index
                             (int)(size_t)pvCurColData);    // int value
                    }
                    break;
                    case SQL_BIND_TYPE_N64:
                    {
                        psObj->iLastSQLErrorCode = sqlite3_bind_int64(
                             psStatement,               // Compiled statement
                             (int)tColIdx+1,            // SQL parameter index
                             *((N64*)pvCurColData));    // int value
                    }
                    break;
                    case SQL_BIND_TYPE_C_STRING:
                    {
                        psObj->iLastSQLErrorCode = sqlite3_bind_text(
                            psStatement,                   // Compiled statement
                            (int)tColIdx+1,                // SQL parameter index
                            (const char *)pvCurColData,    // the value
                            -1,                            // auto length
                            SQLITE_TRANSIENT);             // SQLite will copy it
                    }
                    break;

                    case SQL_BIND_TYPE_STRING_OBJECT:
                    {
                        psObj->iLastSQLErrorCode = iBindString(
                             psObj,
                             psStatement,                   // Compiled statement
                             (int)tColIdx+1,                // SQL parameter index
                             (STRING_OBJECT)pvCurColData);
                    }
                    break;

                    case SQL_BIND_TYPE_BLOB:
                    {
                        // Bind the blob type and provide blob data
                        psObj->iLastSQLErrorCode = sqlite3_bind_blob(
                             psStatement,                   // Compiled statement
                             (int)tColIdx+1,                // SQL parameter index
                             pvCurColData,                  // Blob value
                             tCurColDataSize,               // Blob data byte size
                             SQLITE_TRANSIENT);
                    }
                    break;

                    default:
                    break;
                }

                if (psObj->iLastSQLErrorCode != SQLITE_OK)
                {
                    printf(SQL_INTERFACE_OBJECT_NAME
                        ": Error! Bind blob failure (%u)\n",
                        psObj->iLastSQLErrorCode);

                     break;
                }
            }

            //
            // ??? Should we fail out here if a bind failed?  The default behavior
            // of step is to null unbound templates(?,?NNN,...).  Will step complete
            // and result in success even though we failed to bind?  There are
            // incomplete column assignments.
            //

            // Evaluate the statement we have prepared
            psObj->iLastSQLErrorCode = sqlite3_step(psStatement);
            if (psObj->iLastSQLErrorCode != SQLITE_DONE)
            {
                // Informational only.  sqlites_finalize() must always
                // be called after a successful call to
                // sqlite3_prepare_v2() to avoid resource leaks.
                printf(SQL_INTERFACE_OBJECT_NAME
                    ": Error! Statement evaluation failure (%u)\n",
                    psObj->iLastSQLErrorCode);
            }

            // Finalize & free the statement
            psObj->iLastSQLErrorCode = sqlite3_finalize(psStatement);

            if (psObj->iLastSQLErrorCode != SQLITE_OK)
            {
                printf(SQL_INTERFACE_OBJECT_NAME
                    ": Error! Finalize failure (%u)\n",
                    psObj->iLastSQLErrorCode);

                break;
            }

            return TRUE;

        } while (FALSE);
    }

    return FALSE;
}

/*****************************************************************************
*
*   bStartTransaction
*
*       This API is used to start an SQL transaction.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database on which the
*           an SQL transaction is to be started.
*
*   Outputs:
*       TRUE if the transaction was successfully started. FALSE if it was not
*****************************************************************************/
static BOOLEAN bStartTransaction (
    SQL_INTERFACE_OBJECT hSQL
        )
{
    SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

    if (SQL_INTERFACE_INVALID_OBJECT == hSQL)
    {
        return FALSE;
    }

    if (SQL_PREPARED_STATEMENT_INVALID_HANDLE == psObj->hStartTransactionStmt)
    {
        // create both transaction statements
        // in order to be sure that we will be able to end transaction

        psObj->hStartTransactionStmt = hCreatePreparedStatement(
            hSQL, SQL_START_TRANSACTION);

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE ==
            psObj->hStartTransactionStmt)
        {
            return FALSE;
        }

        psObj->hEndTransactionStmt = hCreatePreparedStatement(
            hSQL, SQL_END_TRANSACTION);

        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE ==
            psObj->hEndTransactionStmt)
        {
            vDestroyPreparedStatement(hSQL, psObj->hStartTransactionStmt);
            psObj->hStartTransactionStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;

            return FALSE;
        }
    }

    return bExecutePreparedStatement(
        hSQL, psObj->hStartTransactionStmt, NULL, NULL, 0);
}

/*****************************************************************************
*
*   bEndTransaction
*
*       This API is used to end an SQL transaction.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database on which the
*           an SQL transaction is to be ended.
*
*   Outputs:
*       TRUE if the transaction was successfully ended. FALSE if it was not
*****************************************************************************/
static BOOLEAN bEndTransaction (
    SQL_INTERFACE_OBJECT hSQL
        )
{
    SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

    return bExecutePreparedStatement(
        hSQL, psObj->hEndTransactionStmt, NULL, NULL, 0);
}

/*****************************************************************************
*
*   bQuery
*
*       This API is used to perform an SQL query on an exisitng
*       valid database connection.
*
*   Inputs:
*       hSQL - Valid SQL_INTERFACE_OBJECT handle of a database on which the
*           query is to be performed
*       pacQueryString - Valid pointer to an array of characters containing
*           the query (SQL syntax) to be executed
*       tbRowResultHandler - pointer to the row handler function which will
*           process the data retreived from the query
*       pvArg - pointer to a caller defined argument which will be passed into
*           tbRowResultHandler
*
*   Outputs:
*       TRUE if the command was successfully prepared and executed;
*       FALSE if it was not
*****************************************************************************/
static BOOLEAN bQuery (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacQueryString,
    SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
    void *pvArg
        )
{
    SQL_OBJECT_STRUCT *psObj =
        (SQL_OBJECT_STRUCT *)hSQL;
    sqlite3_stmt *psStatement = NULL;

    do
    {
        // Ensure the object and pointer are valid
        if ((hSQL == NULL) ||
            (pacQueryString == NULL) ||
            (tbRowResultHandler == NULL))
        {
            break;
        }

        // Compile the string into a statement
        psObj->iLastSQLErrorCode =
            sqlite3_prepare(
                psObj->psDB,
                pacQueryString,
                -1, // NULL-terminated string
                &psStatement,
                NULL // Won't have a tail if things go well
                    );

        if (psObj->iLastSQLErrorCode != SQLITE_OK)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Error! SQL error (%u) while preparing query: %s\n",
                psObj->iLastSQLErrorCode,
                sqlite3_errmsg(psObj->psDB));

            break;
        }

        // Execute statement
        psObj->iLastSQLErrorCode = iProcessQuery(
            psStatement, tbRowResultHandler, pvArg);

        if ((psObj->iLastSQLErrorCode != SQLITE_DONE) &&
            (psObj->iLastSQLErrorCode != SQLITE_ROW))
        {
            printf(SQL_INTERFACE_OBJECT_NAME": Error! Query error %u: %s\n",
                psObj->iLastSQLErrorCode, sqlite3_errmsg(psObj->psDB));
        }

        // Finalize & free the statement
        psObj->iLastSQLErrorCode = sqlite3_finalize(psStatement);

        if (psObj->iLastSQLErrorCode != SQLITE_OK)
        {
            printf(SQL_INTERFACE_OBJECT_NAME": Error! Finalize failure (%u)\n",
                psObj->iLastSQLErrorCode);

             break;
        }

        return TRUE;

    } while (FALSE);

    return FALSE;
}

/*****************************************************************************
*
*   hCreatePreparedStatement
*
*****************************************************************************/
static SQL_PREPARED_STATEMENT_HANDLE hCreatePreparedStatement (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacStatementString
        )
{
    sqlite3_stmt *psStatement = SQL_PREPARED_STATEMENT_INVALID_HANDLE;
    SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

    // Range check.
    if( (hSQL == NULL) || (pacStatementString == NULL) )
    {
        return NULL;
    }

    // Create the statement.
    psObj->iLastSQLErrorCode = sqlite3_prepare_v2(
        psObj->psDB,                      // Database
        &pacStatementString[0],         // SQL statement
        -1,                             // String is zero-terminated
        &psStatement,                   // Prepared statement pointer pointer
        NULL );                         // No Tail

    if (psObj->iLastSQLErrorCode != SQLITE_OK)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": Error! PpsCreatePreparedStatement failure (%u)\n",
            psObj->iLastSQLErrorCode);
        psStatement = SQL_PREPARED_STATEMENT_INVALID_HANDLE;
    }
    return psStatement;
}

/*****************************************************************************
*
*   vDestroyPreparedStatement
*
*****************************************************************************/
static void vDestroyPreparedStatement (
    SQL_INTERFACE_OBJECT hSQL,
    SQL_PREPARED_STATEMENT_HANDLE hStatement
        )
{
    if ((NULL != hSQL) &&
        (SQL_PREPARED_STATEMENT_INVALID_HANDLE != hStatement))
    {
        SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

        psObj->iLastSQLErrorCode = sqlite3_finalize( hStatement );
        if (psObj->iLastSQLErrorCode != SQLITE_OK)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Error! bDestroyPreparedStatement failure (%u)\n",
                psObj->iLastSQLErrorCode);
        }
    }

    return;
}

/*****************************************************************************
*
*   bExecutePreparedStatement
*
*****************************************************************************/
static BOOLEAN bExecutePreparedStatement (
    SQL_INTERFACE_OBJECT hSQL,
    SQL_PREPARED_STATEMENT_HANDLE hStatement,
    SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
    void *pvArg,
    size_t tNumParams,
    ...
        )
{
    BOOLEAN bOk = TRUE;
    SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

    if ((SQL_INTERFACE_INVALID_OBJECT == hSQL) ||
        (SQL_PREPARED_STATEMENT_INVALID_HANDLE == hStatement))
    {
        return FALSE;
    }

    // Bind statement parameters
    if (tNumParams > 0)
    {
        va_list vlParams;
        SQL_BIND_PARAMETER_STRUCT const *psParams;

        va_start(vlParams, tNumParams);
        psParams = va_arg(vlParams, SQL_BIND_PARAMETER_STRUCT const *);
        va_end(vlParams);

        psObj->iLastSQLErrorCode = iBindParameters(
            psObj, (sqlite3_stmt*)hStatement, psParams, tNumParams);

        if (SQLITE_OK != psObj->iLastSQLErrorCode)
        {
            return FALSE;
        }
    }

    // Execute
    psObj->iLastSQLErrorCode = iProcessQuery(
        (sqlite3_stmt*)hStatement, tbRowResultHandler, pvArg);

    if ((psObj->iLastSQLErrorCode != SQLITE_DONE) &&
        (psObj->iLastSQLErrorCode != SQLITE_ROW))
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": Error! bExecutePreparedStatement (%u)\n",
            psObj->iLastSQLErrorCode);
        bOk = FALSE;
    }

    // Reset
    psObj->iLastSQLErrorCode = sqlite3_reset((sqlite3_stmt*)hStatement);
    if (psObj->iLastSQLErrorCode != SQLITE_OK)
    {
        printf(SQL_INTERFACE_OBJECT_NAME
            ": Error! bExecutePreparedStatement reset statement failure (%u)\n",
            psObj->iLastSQLErrorCode);
        bOk = FALSE;
    }

    return bOk;
}

/*****************************************************************************
*
*   bGetLastInsertRowID
*
*****************************************************************************/
static BOOLEAN bGetLastInsertRowID (
	SQL_INTERFACE_OBJECT hSQL,
	UN32* pun32RowID
		)
{
	SQL_OBJECT_STRUCT *psObj = (SQL_OBJECT_STRUCT *)hSQL;

	// Range check.
	if( (hSQL == NULL) || (pun32RowID == NULL) )
	{
		return FALSE;
	}

	*pun32RowID = (UN32)sqlite3_last_insert_rowid(psObj->psDB);

	return TRUE;
}

/*****************************************************************************
                             FRIEND FUNCTIONS
*****************************************************************************/

/*****************************************************************************
                             PRIVATE FUNCTIONS*
*****************************************************************************/

/*****************************************************************************
*
*   iBindParameters
*
*****************************************************************************/
static int iBindParameters (
    SQL_OBJECT_STRUCT *psObj,
    sqlite3_stmt *psStatement,
    const SQL_BIND_PARAMETER_STRUCT *psParams,
    size_t tNumParams
        )
{
    size_t tIndex;
    int iSqlErrorCode = SQLITE_OK;

    // bind provided arguments
    for (tIndex = 0; tIndex < tNumParams; tIndex++)
    {
        switch (psParams[tIndex].eType)
        {
            case SQL_BIND_TYPE_UN32:
            {
                iSqlErrorCode = sqlite3_bind_int(
                        psStatement, tIndex + 1,
                        (int)psParams[tIndex].pvData);
            }
            break;

            case SQL_BIND_TYPE_N64:
            {
                iSqlErrorCode = sqlite3_bind_int64(
                        psStatement, tIndex + 1,
                        *(const sqlite3_int64 *)psParams[tIndex].pvData);
            }
            break;

            case SQL_BIND_TYPE_C_STRING:
            {
                iSqlErrorCode = sqlite3_bind_text(
                    psStatement, tIndex + 1,
                    (const char *)psParams[tIndex].pvData, -1,
                    SQLITE_TRANSIENT);
            }
            break;

            case SQL_BIND_TYPE_BLOB:
            {
                iSqlErrorCode = sqlite3_bind_blob(
                    psStatement, tIndex + 1,
                    psParams[tIndex].pvData,
                    psParams[tIndex].tSize,
                    SQLITE_TRANSIENT);
            }
            break;

            case SQL_BIND_TYPE_STRING_OBJECT:
            {
                iSqlErrorCode = iBindString(psObj,
                    psStatement, tIndex + 1,
                    (STRING_OBJECT)psParams[tIndex].pvData);
            }
            break;

            default:
            {
                iSqlErrorCode = SQLITE_ERROR;
            }
            break;
        }

        if (SQLITE_OK != iSqlErrorCode)
        {
            printf(SQL_INTERFACE_OBJECT_NAME
                ": Error! SQL parameter %d cannot be bound (%d)\n",
                tIndex + 1, iSqlErrorCode);
            break;
        }
    }

    return iSqlErrorCode;
}

/*****************************************************************************
*
*   iBindString
*
*****************************************************************************/
static int iBindString (
    SQL_OBJECT_STRUCT *psObj,
    sqlite3_stmt *psStatement,
    int iColIndex,
    STRING_OBJECT hString
        )
{
    int iSqlErrorCode = SQLITE_ERROR;

    do
    {
        size_t tLength = 0;

        if (hString != STRING_INVALID_OBJECT)
        {
            tLength = STRING.tSize(hString);

            // make sure the buffer is the right size
            if (psObj->tBufferSize <= tLength)
            {
                BOOLEAN bOk;

                // We need to expand the buffer
                bOk = bExpandBuffer( psObj, tLength, FALSE );
                if (bOk == FALSE)
                {
                    // Can't expand the buffer!
                    break;
                }
            }

            // Copy the string to our buffer
            STRING.tCopyToCStr(hString,
                &psObj->pacBuffer[0],
                psObj->tBufferSize);
        }

        // Bind the buffer
        iSqlErrorCode = sqlite3_bind_text(
                psStatement,
                iColIndex,
                (const char *)&psObj->pacBuffer[0],
                (int)tLength,       // we know the length
                SQLITE_TRANSIENT);

    } while (FALSE);

    return iSqlErrorCode;
}

/*****************************************************************************
*
*   bExpandBuffer
*
*   Expand the scratch buffer to the requested size.  If bCopy == TRUE, then
*   copy the contents of the old buffer to the new one
*
*****************************************************************************/
static BOOLEAN bExpandBuffer (
    SQL_OBJECT_STRUCT *psObj,
    size_t tNewSizeRequired,
    BOOLEAN bCopy
        )
{
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    char *pacNewBuffer;

    // Add some margin to the requested size
    tNewSizeRequired += SQL_BUFFER_INCREASE_MARGIN;

    // Generate a name for this buffer
    snprintf(&acName[0], sizeof(acName),
        SQL_INTERFACE_OBJECT_NAME": Buffer (%u)",
        tNewSizeRequired);

    // Now, allocate the new space we want
    pacNewBuffer = (char *) OSAL.pvMemoryAllocate(
                    &acName[0],
                    tNewSizeRequired,
                    TRUE);

    if (pacNewBuffer != NULL)
    {
        // Do we have a previous valid buffer?
        if (psObj->pacBuffer != NULL)
        {
            // Were we asked to copy its contents
            // into the new buffer?
            if (bCopy == TRUE)
            {
                // Copy the entire contents of the old
                // buffer into the new buffer
                OSAL.bMemCpy(
                    &pacNewBuffer[0], &psObj->pacBuffer[0],
                    psObj->tBufferSize);
            }

            // Destroy the old buffer
            OSAL.vMemoryFree(psObj->pacBuffer);
        }

        // Use the new buffer now
        psObj->pacBuffer = pacNewBuffer;
        psObj->tBufferSize = tNewSizeRequired;

        return TRUE;
    }

    return FALSE;
}

/*****************************************************************************
*
*   bIntegrityQueryResult
*
*****************************************************************************/
static BOOLEAN bIntegrityQueryResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        )
{
    SQL_QUERY_COLUMN_STRUCT *psCurrentCol;
    BOOLEAN *pbReturn = (BOOLEAN *)pvArg;

    if (n32NumberOfColumns == 0 )
    {

        *pbReturn = FALSE;
    }
    else
    {
        N32 n32Index;

        for (n32Index = 0; n32Index <n32NumberOfColumns; n32Index++)
        {
            psCurrentCol = &psColumns[n32Index];
            switch (psCurrentCol->eType)
            {
                case SQL_COLUMN_TYPE_C_STRING:
                {
                    if (strcmp((const char *)psCurrentCol->uData.sCString.pcData,
                               SQL_INTEGRITY_VERIFICATION) == 0)
                    {
                        *pbReturn = TRUE;
                    }
                    else
                    {
                        *pbReturn = FALSE;
                        return FALSE;
                    }
                }
                break;

                case SQL_COLUMN_TYPE_INTEGER:
                case SQL_COLUMN_TYPE_BLOB:
                case SQL_COLUMN_TYPE_UNKNOWN:
                default:
                break;
            }
        }
    }

    return TRUE;
}

/*****************************************************************************
*
*   bSetDBPageSize
*
*****************************************************************************/
static BOOLEAN bSetDBPageSize (
    SQL_INTERFACE_OBJECT hSQL,
    UN32 un32PageSize
        )
{
    char acPageSizeCmd[SQLITE_SET_PAGESIZE_LEN];
    BOOLEAN bOk;

    // Format the SQL Command
    snprintf(
        &acPageSizeCmd[0],
        SQLITE_SET_PAGESIZE_LEN,
        SQLITE_SET_PAGESIZE,
        un32PageSize);

    // Set the page size now
    bOk =  SQL_INTERFACE.bExecuteCommand(
        hSQL, &acPageSizeCmd[0] );

    return bOk;
}

/*****************************************************************************
*
*   un32DBPageSize
*
*****************************************************************************/
static UN32 un32DBPageSize (
    SQL_INTERFACE_OBJECT hSQL
        )
{
    BOOLEAN bSuccess;
    SQL_PRAGMA_RESULT_STRUCT sResult;
    UN32 un32PageSize = 0;

    // Initialize the result structure
    sResult.bSuccess = FALSE;
    sResult.un32PragmaValue = 0;

    // Get the page size
    bSuccess = SQL_INTERFACE.bQuery(
        hSQL, SQLITE_GET_PAGESIZE,
        (SQL_QUERY_RESULT_HANDLER)bProcessQueryPragma,
        (void *)&sResult);
    if ((bSuccess == TRUE) &&
        (sResult.bSuccess == TRUE))
    {
        un32PageSize = sResult.un32PragmaValue;
    }

    return un32PageSize;
}

/*****************************************************************************
*
*   bProcessQueryPragma
*
*****************************************************************************/
static BOOLEAN bProcessQueryPragma (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SQL_PRAGMA_RESULT_STRUCT *psResult
        )
{
    if (psResult == NULL)
    {
        return FALSE;
    }

    if ((n32NumberOfColumns == 1 ) &&
        (psColumn[0].eType == SQL_COLUMN_TYPE_INTEGER))
    {
        psResult->un32PragmaValue = psColumn[0].uData.sUN32.un32Data;
        psResult->bSuccess = TRUE;
    }

    // Stop iterating
   return FALSE;
}

/*****************************************************************************
*
*   iProcessQuery
*
*****************************************************************************/
static int iProcessQuery (
    sqlite3_stmt * psStatement,
    SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
    void *pvArg
        )
{
    int iSqlErrorCode;

    iSqlErrorCode = sqlite3_step(psStatement);

    if ((NULL != tbRowResultHandler) && (SQLITE_ROW == iSqlErrorCode))
    {
        UN32 un32NumColumns;

        un32NumColumns = (UN32)sqlite3_column_count(psStatement);
        if (un32NumColumns > 0)
        {
            UN32 un32ColIdx;

            // Create our memory to store each row
            SQL_QUERY_COLUMN_STRUCT *psRowData =
                (SQL_QUERY_COLUMN_STRUCT *) OSAL.pvMemoryAllocate(
                    SQL_INTERFACE_OBJECT_NAME":Query2RowData",
                    sizeof(SQL_QUERY_COLUMN_STRUCT) * un32NumColumns,
                    TRUE);

            if (NULL != psRowData)
            {
                do
                {
                    BOOLEAN bKeepIterating;

                    // Loop through each column
                    for (un32ColIdx = 0; un32ColIdx < un32NumColumns; un32ColIdx++)
                    {
                        int iColType;

                        iColType = sqlite3_column_type(psStatement, un32ColIdx);

                        // Fetch the name if we haven't gotten it yet
                        if (psRowData[un32ColIdx].pcColName == NULL)
                        {

                            psRowData[un32ColIdx].pcColName =
                                sqlite3_column_name(psStatement, un32ColIdx);
                        }

                        switch (iColType)
                        {
                            case SQLITE_INTEGER:
                            {
                                psRowData[un32ColIdx].eType = SQL_COLUMN_TYPE_INTEGER;
                                // Get the native sqlite format first.
                                psRowData[un32ColIdx].n64NativeInteger =
                                    sqlite3_column_int64(psStatement, un32ColIdx);
                                // Cast to 32 bits for compatibility with existing
                                // code and convenience. n64 native format support
                                // was added after the union structure.
                                psRowData[un32ColIdx].uData.sUN32.un32Data =
                                    (UN32)psRowData[un32ColIdx].n64NativeInteger;
                            }
                            break;

                            case SQLITE_TEXT:
                            {
                                psRowData[un32ColIdx].eType = SQL_COLUMN_TYPE_C_STRING;
                                psRowData[un32ColIdx].uData.sCString.pcData =
                                    sqlite3_column_text(psStatement, un32ColIdx);
                            }
                            break;

                            case SQLITE_BLOB:
                            {
                                psRowData[un32ColIdx].eType = SQL_COLUMN_TYPE_BLOB;
                                psRowData[un32ColIdx].uData.sBLOB.pvData =
                                    sqlite3_column_blob(psStatement, un32ColIdx);
                                psRowData[un32ColIdx].uData.sBLOB.tSize =
                                    (size_t)sqlite3_column_bytes(psStatement, un32ColIdx);

                            }
                            break;

                            case SQLITE_FLOAT:
                            case SQLITE_NULL:
                            default:
                            {
                                // Unhandled or unknown
                                psRowData[un32ColIdx].eType = SQL_COLUMN_TYPE_UNKNOWN;
                            }
                            break;
                        }
                    }

                    // Execute the caller's result handler function for this row
                    bKeepIterating = tbRowResultHandler(
                            psRowData, un32NumColumns, pvArg);

                    if (bKeepIterating == FALSE)
                    {
                        // Stop iterating the data rows now
                        break;
                    }

                    // Get our next row
                    iSqlErrorCode = sqlite3_step(psStatement);

                } while (SQLITE_ROW == iSqlErrorCode);

                // Free our row data
                OSAL.vMemoryFree(psRowData);
            }
        }
    }

    return iSqlErrorCode;
}

/*****************************************************************************
*
*   vDestroyObject
*
*****************************************************************************/
static void vDestroyObject (
    SQL_OBJECT_STRUCT *psObj
        )
{
    if (psObj != NULL)
    {
        // free connection's buffer if it exists
        if (psObj->pacBuffer != NULL)
        {
            OSAL.vMemoryFree(psObj->pacBuffer);
            psObj->pacBuffer = NULL;
        }

        // Destroy internal prepared statements
        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE !=
            psObj->hStartTransactionStmt)
        {
            sqlite3_finalize((sqlite3_stmt *)psObj->hStartTransactionStmt);
            psObj->hStartTransactionStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;
        }
        if (SQL_PREPARED_STATEMENT_INVALID_HANDLE !=
            psObj->hEndTransactionStmt)
        {
            sqlite3_finalize((sqlite3_stmt *)psObj->hEndTransactionStmt);
            psObj->hEndTransactionStmt =
                SQL_PREPARED_STATEMENT_INVALID_HANDLE;
        }

        // Close the database connection if necessary
        if (psObj->psDB != NULL)
        {
            sqlite3_close(psObj->psDB);
            psObj->psDB = NULL;
        }

        // Last step
        OSAL.vMemoryFree(psObj);
    }

    return;
}

#if ENABLE_SQL_LOGGING == 1
/*****************************************************************************
*
*   vLogSQL
*
*   Function called by the SQLite library if the logging service is enabled
*   and a message needs to be logged.
*****************************************************************************/
static void vLogSQL(
    void *pvArg,
    int iCode,
    const char *pcMsg
        )
{
    fprintf(stderr, "[SQL:%d]%s\n", iCode, pcMsg);
    return;
}
#endif
