/******************************************************************************/
/*                   Copyright (c) Sirius-XM Satellite Radio, Inc.            */
/*                            All Rights Reserved                             */
/*        Licensed Materials - Property of Sirius-XM Satellite Radio, Inc.    */
/******************************************************************************/

//
// C Standard Library Include Files
//

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

//
// Standard SMS Macro Definitions
//

#include "standard.h"

//
// OSAL's header
//

#include "osal.h"

//
// Device drivers
//

#include "srh.h"
#include "posix_serial.h"
#include "sio_shim.h"

//
// SMS Includes
//

#include "sms_api.h"

//
// Local constants; these could go in a separate "configuration" header file, but
// we leave them here to keep this example simple.
//


#ifdef ANDROID

// This is for Android build
#define DEFAULT_SERIAL_DEVICE  "/dev/ttyO0"
#define DEFAULT_SMS_PATH_BASE  "/data/siriusxm"
#define SMS_TEMP_DIR_PATH      "/data/tmp"

#elif BEAGLEBONE_BLACK

// This is for BeagleBone Black(BBB) Linux build
#define DEFAULT_SERIAL_DEVICE  "/dev/ttyUSB0"
#define DEFAULT_SMS_PATH_BASE  "sirius-xm"
#define SMS_TEMP_DIR_PATH      "ramdisk"

#else

// This is for PC Linux build
#define DEFAULT_SERIAL_DEVICE  "/dev/ttyS0"
#define DEFAULT_SMS_PATH_BASE  "/sirius-xm"
#define SMS_TEMP_DIR_PATH      "/tmp"

#endif

//
// Our coding standard states that all preprocessor directives must go in
// column zero, but that would render the following block illegible. We'll
// break the rules this once just for pragmatism.
//

#ifdef ANDROID
    #if (SMS_DEVICE_EMULATION==1)
        // Mimosa doesn't work on high speed
        #define SMS_UART_PARAMS ("115200,N,8,1")
    #else
        #ifdef RECORD_ADV_IR_ENABLE
            #define SMS_UART_PARAMS ("1846154,N,8,1")
        #else
            #define SMS_UART_PARAMS ("115200,N,8,1")
        #endif
    #endif
#elif BEAGLEBONE_BLACK
    #define SMS_UART_PARAMS ("230400,N,8,1")
#else
    #define SMS_UART_PARAMS ("115200,N,8,1")
#endif

#define DEFAULT_SMS_CFG_PATH                DEFAULT_SMS_PATH_BASE"/sms_hb"
#define DEFAULT_CLI_RESOURCES_DIRECTORY     DEFAULT_SMS_PATH_BASE"/cli/"

//
// Global variables; these are not defined as static, as they'll be
// accessed by sms_app.c
//

char *gpacTempDirPath                = SMS_TEMP_DIR_PATH;
char *gpacDefaultSmsCfgPath          = DEFAULT_SMS_CFG_PATH;
char *gpacDefaultCliResourcesDir     = DEFAULT_CLI_RESOURCES_DIRECTORY;
FILE *psGPIO = NULL;

extern char *gpacSerialDevice;
extern char *gpacRadioName;

//
// Local prototypes; these are not defined as static, as they'll be
// accessed by sms_app.c
//

BOOLEAN      bRegisterDrivers( void *pvArg );
void         vUnRegisterDrivers( void *pvArg );
BOOLEAN      bPlatformSpecificInit( void *pvArg );

/*****************************************************************************
*
*      bRegisterDrivers
*
*****************************************************************************/
BOOLEAN bRegisterDrivers( void *pvArg )
{
    N32 n32RetVal;

    /////////////////////////////
    // Register Serial Port(s)...
    /////////////////////////////

    // Register native posix serial port...
    n32RetVal =
        OSAL.n32RegNode(
            gpacSerialDevice,
            &GsPosixSerialInterface,
            gpacSerialDevice,
            strlen(gpacSerialDevice) + 1);
    if(n32RetVal != DEV_OK)
    {
        printf("Error! Cannot Register '%s'\n", gpacSerialDevice);
        return FALSE;
    }
    else
    {
        printf("Registered '%s' via '%s'\n",
        		gpacSerialDevice, gpacSerialDevice);
    }

    // Register posix serial port via a shim i/o driver.
    // This driver allows us to use the serial port's modem
    // signals as independently controlled gpio.
    n32RetVal =
        OSAL.n32RegNode(
            gpacRadioName,
            &GsSioInterface,
            gpacSerialDevice,
            strlen(gpacSerialDevice) + 1);

    if( n32RetVal != DEV_OK )
    {
        printf("Error! Cannot Register '%s'\n", gpacRadioName);
        return FALSE;
    }

    /////////////////////////////
    // Register SRH Driver(s)
    /////////////////////////////
    SRH_vRegisterDriver( SMS_UART_PARAMS );

    return TRUE;
}

/*****************************************************************************
*
* vUnRegisterDrivers
*
* Simply unregisters our drivers and nodes as part of shutdown.
*
*****************************************************************************/
void vUnRegisterDrivers( void *pvArg )
{
    // Unregister SRH Drivers
    SRH_vUnRegisterDriver();

    OSAL.bUnRegNode(gpacRadioName);
    OSAL.bUnRegNode(gpacSerialDevice);

    return;
}

/*****************************************************************************
*
* bPlatformSpecificInit
*
* Platform-specific setup code (can simply return TRUE if there's
* nothing to do for a specific platform.)
*
*****************************************************************************/
BOOLEAN bPlatformSpecificInit( void *pvArg )
{
    // Open Serial-I/O shim for SXM-Module GPIO. We need to assign
    // our global file handle so that our SRH HW implementation
    // can use the modem signals to drive power and reset of the
    // SXM-module.

    psGPIO = fopen( gpacRadioName, SIO_SHIM_IO_MODE );
    if(psGPIO == NULL)
    {
        printf("Error! Cannot open device %s for SXM-Module GPIO.\n",
        		gpacRadioName);
        return FALSE;
    }
    else
    {
        printf("Opened device %s for SXM-Module GPIO.\n", gpacRadioName);
    }

    return TRUE;
}
