/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the generic device driver implementation for
 *  a device that uses the Posix Serial Interface (character driver).
 *
 ******************************************************************************/

#include <ctype.h>

#include <pthread.h>

#ifdef ANDROID
   #include <linux/serial_core.h>
#elif defined __QNX__
#else
   #include <linux/serial.h>
#endif

/* For "ioctl()" */
#include <sys/ioctl.h>

/* For "open()" */
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* For Errors */
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

/* For terminal configuration */
#include <termios.h>
#include <unistd.h>

#include "standard.h"
#include "osal.h"

// Device Driver Framework Common Serial Communications Port Header
#include "com.h"

// Posix Serial Driver Includes
#include "posix_serial.h"
#include "_posix_serial.h"

#ifdef ioctl
#undef ioctl // Force usage of native ioctl()
#endif

// Do not support tracing
#define TRACE_START()
#define TRACE_END()

/*****************************************************************************
*
*       _bInit
*
*   Performs any device initialization for the physical device this driver
*   is written for. This method is called only when the first driver instance
*   is opened.
*
*   Inputs:
*       pcDevName - A pointer to an ASCII null terminated string which
*           identifies this device. A copy of the provided name is retained
*           in the device node and this is a pointer to that name.
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       ppvPublicData - A pointer to a pointer which represents any
*           persistent object which contains device-global or public
*           data required by all driver instances.
*
*   Outputs:
*       TRUE on success. Otherwise FALSE should be returned on error.
*
*****************************************************************************/
static BOOLEAN _bInit (
                        const char *pcDevName,
                        void const *pvProperties,
                        void **ppvPublicData
                        )
{
    POSIX_SERIAL_PUBLIC_STRUCT *psPublicData;

    TRACE_START();

    // Must provide a valid 'properties' pointer
    if(pvProperties == NULL)
    {
        TRACE_END();
        return FALSE;
    }

    // Allocate a structure to hold the public control structure
    // required by the driver.
    // Set the public data pointer for the Framework to maintain
    psPublicData =
    *ppvPublicData =
        OSAL.pvMemoryAllocate(
            pcDevName,
            sizeof(POSIX_SERIAL_PUBLIC_STRUCT),
            TRUE );
    if(psPublicData == NULL)
    {
        // Error!
        printf("POSIX_SERIAL Error! Cannot allocate memory.\n");
        TRACE_END();
        return FALSE;
    }

    // Populate local device name (from properties)
    psPublicData->pacDeviceName = (const char *)pvProperties;

    // Initialize File Descriptor
    psPublicData->fd = -1;

    // Initialize Error field
    psPublicData->n32Error = 0;

    // Initialize Eof field
    psPublicData->n32Eof = 0;

    // Initialize IOCTL Statistics
    OSAL.bMemSet(&psPublicData->sStats, 0, sizeof(psPublicData->sStats));

    TRACE_END();
    return TRUE;
}

/*****************************************************************************
*
*       _vExit
*
*   Performs any device un-initialization or clean-up procedures required
*   when the device is no longer needed. This method is called only when
*   all open connections to the driver have been closed.
*
*   Inputs:
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       pvPublicData - - A pointer to a persistent object which contains
*           device-global or public data required by all driver instances.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void _vExit (
                        void const *pvProperties,
                        void *pvPublicData
                        )
{
    TRACE_START();
    // Just free any public data allocated by _bInit()
    OSAL.vMemoryFree(pvPublicData);

    TRACE_END();
    return;
}

/*****************************************************************************
*
*       _n32Open
*
*   Opens a connection (instance) to the specified device by allocating any
*   resources or state information required for this instance.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver.
*       pvArg - A pointer to some device driver specific data which represents
*           a mode or configuration this instance is to be open with.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Open (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        const void *pvArg
                        )
{
    N32 n32RetVal = DEV_ERROR;
    BOOLEAN bValid;
    struct termios termios_p;
    int iErr;
    // Extract public data from the Device Driver Framework
    POSIX_SERIAL_PUBLIC_STRUCT *psPublicData =
        (POSIX_SERIAL_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    TRACE_START();

    // Check input
    if(pvArg == NULL)
    {
        // Error!
        TRACE_END();
        return DEV_ERROR;
    }

    // We can only allow one connection per USART
    if(psDeviceInfo->un32Instance > 1)
    {
        // Bail, only one instance allowed
        TRACE_END();
        return DEV_ERROR;
    }

    // Inform the Framework there is no private data, there is no
    // private data since multiple connections to this driver cannot exist
    psDeviceInfo->pvPrivateData = (void *)NULL;

    // Open port using the OS Device Manager
    psPublicData->fd = open(
            psPublicData->pacDeviceName, O_RDWR);
    if(psPublicData->fd < 0)
    {
        // Cannot get file device descriptor
        TRACE_END();
        return DEV_ERROR;
    }

    // Get attributes
    iErr = tcgetattr( psPublicData->fd, &termios_p);
    if(iErr == 0)
    {
        // Init our terminal settings
        bValid = _bInitTermSettings(&termios_p);
        if(bValid == FALSE)
        {
            // Something is off
            TRACE_END();
            return DEV_ERROR;
        }

        // Next we need to parse the provided open argument
        bValid =
            _bParseArg(
                (const char*)pvArg,
                &termios_p);
        if(bValid == FALSE)
        {
            // Invalid input args
            TRACE_END();
            return DEV_ERROR;
        }

        // Set attributes
        iErr = tcsetattr( psPublicData->fd, TCSAFLUSH, &termios_p);
        if(iErr == 0)
        {
            n32RetVal = DEV_OK;
        }
        else
        {
            // Error
            close(psPublicData->fd);
            psPublicData->fd = -1;
            TRACE_END();
            return DEV_ERROR;
        }
    }
    else
    {
        // Error
        close(psPublicData->fd);
        psPublicData->fd = -1;
        TRACE_END();
        return DEV_ERROR;
    }

    // Activate line signals
    _vSetSignals(psPublicData, COM_SIGNAL_RTS | COM_SIGNAL_DTR);

    TRACE_END();
    return n32RetVal;
}

/*****************************************************************************
*
*       _n32Close
*
*   Closes a connection (instance) to the specified device and releases any
*   resources obtained when the device was opened.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members
*           can be accessed to retrieve information about the device itself
*           and the driver instance. See Section 4.10.2 for more detail on
*           this structures members.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is returned
*       on error. See section 4.11 for possible error codes. If anything other
*       than DEV_OK is returned the instance handle will not be released and
*       will remain open.
*
*****************************************************************************/
static N32 _n32Close (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo
                        )
{
    int iErr;
    N32 n32RetVal = DEV_ERROR;

    // Extract public data from the Device Driver Framework
    POSIX_SERIAL_PUBLIC_STRUCT *psPublicData =
        (POSIX_SERIAL_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    TRACE_START();
    // Close open file descriptor
    iErr = close(psPublicData->fd);
    if(iErr == 0)
    {
        psPublicData->fd = -1;
        n32RetVal = DEV_OK;
    }

    TRACE_END();
    return n32RetVal;
}

/*****************************************************************************
*
*       _tRead
*
*   Allows byte wide (stream oriented) reads from the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcDst - The destination pointer where device data is to be read.
*       tSize - The size of each object to read.
*       tNumObjs - The number of objects to read.
*
*   Outputs:
*       The number of objects successfully read from the device and placed
*       in pcDst. If an error occurs or no data was read a zero is returned.
*       It should not be assumed that every read request must always be carried
*       out to completion in a single call. The caller is required to check the
*       return value to determine if all the requested data was read or if a
*       partial read was performed. If a partial read occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tRead (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        char *pcDst,
                        size_t tSize,
                        size_t tNumObjs
                        )
{
    ssize_t tBytesRead;
    size_t tLength = tSize * tNumObjs;

    // Extract public data from the Device Driver Framework
    POSIX_SERIAL_PUBLIC_STRUCT *psPublicData =
        (POSIX_SERIAL_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    TRACE_START();

    // Perform byte-wise read from the device
    tBytesRead = read(psPublicData->fd, pcDst, tLength);

    // Clearing EOF flag. It will be set later if needed
    psPublicData->n32Eof = FALSE;

    if (tBytesRead < 0)
    {
        N32 iLocalErrno = (N32)errno;

        // an error occurred
        tNumObjs = 0;

        // record error state
        // See ioctl() implementation below for RTO. If EINTR occurs it
        // it because we have released the reader due to change in RTO.
        if( EINTR != iLocalErrno )
        {
            psPublicData->n32Error = iLocalErrno;
        }
    }
    else
    {
       // If tBytesRead is zero but we requested some, it means EOF condition
       if((tBytesRead == 0) && (tLength > 0))
       {
          psPublicData->n32Eof = TRUE;
       }
       tNumObjs = (size_t)tBytesRead;
    }

    // Increment statistics
    psPublicData->sStats.un32RxBytes +=  tNumObjs;

    // Calculate number of objects read
    tNumObjs /= tSize;

    TRACE_END();
    return tNumObjs;
}

/*****************************************************************************
*
*       _tWrite
*
*   Allows byte wide (stream oriented) writes to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcSrc - The source pointer where device data is to be written from.
*       tSize - The size of each object to write.
*       tNumObjs - The number of objects to write.
*
*   Outputs:
*       The number of objects successfully written to the device from pcSrc.
*       If an error occurs or no data was written a zero is returned. It should
*       not be assumed that every write request must always be carried out to
*       completion in a single call. The caller is required to check the return
*       value to determine if all the requested data was written or if a
*       partial write was performed. If a partial write occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tWrite (
                         const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                         const char *pcSrc,
                         size_t tSize,
                         size_t tNumObjs
                         )
{
    ssize_t tBytesWritten;
    size_t tLength = tSize * tNumObjs, tNumWritten;

    // Extract public data from the Device Driver Framework
    POSIX_SERIAL_PUBLIC_STRUCT *psPublicData =
        (POSIX_SERIAL_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    TRACE_START();

    // Write what I can to the device...
    tBytesWritten = write(psPublicData->fd, pcSrc, tLength);

    if (tBytesWritten < 0)
    {
        // an error occured

        tNumWritten = 0;

        // record error state
        psPublicData->n32Error = tBytesWritten;
    }
    else
    {
        tNumWritten = (size_t) tBytesWritten;
    }

    // Increment statistics
    psPublicData->sStats.un32TxBytes +=  tNumWritten;

    // Return number of objects written
    // Note: tSize is guaranteed not to be equal to zero by the framework.
    TRACE_END();
    return tNumWritten / tSize;
}


/*****************************************************************************
*
*       _n32Ioctl
*
*   Allows driver specific command and control to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       n32Cmd - A number assigned to a particular device command. Typically
*           this is defined in the device driver's public header file
*           (i.e. <device_driver_name>.h)
*       tList - A pointer to a variable argument list which contains zero
*           or more command specific arguments. The number and type of each
*           argument must be defined in a device driver document for this
*           particular device driver.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. Any unsupported ioctl commands should return
*       DEV_UNSUPPORTED_COMMAND. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Ioctl (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        N32 n32Cmd,
                        va_list *ptList
                        )
{
    N32 n32Err = DEV_OK;

    // Extract public data from the Device Driver Framework
    POSIX_SERIAL_PUBLIC_STRUCT *psPublicData =
        (POSIX_SERIAL_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    TRACE_START();

    // Handle command
    switch(n32Cmd)
    {
        case COM_IOCTL_SET_SIGNALS:
        {
            // Extract input
            UN8 un8Signals = (UN8)va_arg( *ptList, int );

            // Process command
            _vSetSignals(psPublicData, un8Signals);
        }
        break;

        case COM_IOCTL_GET_SIGNALS:
        {
            // Extract input
            UN8 *pun8Signals = va_arg( *ptList, UN8 *);
            if(pun8Signals != NULL)
            {
                // Process command
                *pun8Signals = _un8GetSignals(psPublicData);
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case COM_IOCTL_CLR_SIGNALS:
        {
            // Extract input
            UN8 un8Signals = (UN8)va_arg( *ptList, int );

            // Process command
            _vClrSignals(psPublicData, un8Signals);
        }
        break;

        case COM_IOCTL_SET_RTO:
        {
            // Extract input
            N32 n32TimeoutMsec = va_arg( *ptList, N32 );
            int iErr;
            struct termios termios_p;

            // Get attributes
            iErr = tcgetattr( psPublicData->fd, &termios_p);
            if(iErr != 0)
            {
                n32Err = DEV_INVALID_INPUT;
                break;
            }

            // If timeout provided is 'infinite' then implement an
            // indefinite timeout based only on received bytes
            if(n32TimeoutMsec == OSAL_OBJ_TIMEOUT_INFINITE)
            {
                termios_p.c_cc[VMIN] =  1; /* At least one character */
                termios_p.c_cc[VTIME] =  0; /* no timeout */
            }
            // If timeout provided is 'none' then implement a
            // completely non-blocking read.
            else if(n32TimeoutMsec == OSAL_OBJ_TIMEOUT_NONE)
            {
                termios_p.c_cc[VMIN] =  0; /* No minimum */
                termios_p.c_cc[VTIME] =  0; /* no timeout */
            }
            // Some timeout component has been provided. Implement a
            // blocking read with timeout
            else
            {
                termios_p.c_cc[VMIN] =  0; /* No minimum */
                termios_p.c_cc[VTIME] =  n32TimeoutMsec / 100; /* 1/10 seconds */
            }

            // Set attributes
            iErr = tcsetattr( psPublicData->fd, TCSANOW, &termios_p);
            if(iErr != 0)
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case COM_IOCTL_SET_WTO:
        {
            // Extract input
            // N32 n32Timeout = va_arg( *ptList, N32 );
            n32Err = DEV_UNSUPPORTED_CMD;
        }
        break;

        case COM_IOCTL_GET_STATISTICS:
        {
            // Extract input
            COM_IOCTL_STATISTICS_STRUCT *psStats =
                va_arg( *ptList, COM_IOCTL_STATISTICS_STRUCT * );

            if(psStats != NULL)
            {
               *psStats = psPublicData->sStats;
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case COM_IOCTL_RESET_STATISTICS:
        {
            // Reset IOCTL Statistics

            OSAL.bMemSet(&psPublicData->sStats,
                         0, sizeof(psPublicData->sStats));
        }
        break;

        case COM_IOCTL_DUMP_STATISTICS:
        {
            // Dump IOCTL Statistics

            // Extract input
            FILE *psFile = va_arg( *ptList, FILE * );

            if(psFile != NULL)
            {
               fprintf(psFile, "\n\nDriver (%s) Statistics\n",
                    psPublicData->pacDeviceName);
               fprintf(psFile, "\tTx Bytes: %u\n",
                    psPublicData->sStats.un32TxBytes);
               fprintf(psFile, "\tRx Bytes: %u\n",
                    psPublicData->sStats.un32RxBytes);
               fprintf(psFile, "\tOverrun Errors: %u\n",
                    psPublicData->sStats.un32OverrunErrors);
               fprintf(psFile, "\tFraming Errors: %u\n",
                    psPublicData->sStats.un32FramingErrors);
               fprintf(psFile, "\tParity Errors: %u\n",
                    psPublicData->sStats.un32ParityErrors);
               fprintf(psFile, "\tBreak Errors: %u\n",
                    psPublicData->sStats.un32BreakErrors);
               fprintf(psFile, "\tGeneral Errors: %u\n",
                    psPublicData->sStats.un32Errors);
               fprintf(psFile, "\n\n");
            }
            else
            {
               n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case COM_IOCTL_SET_SIGNAL_CALLBACK:
        {
            // Extract input
            COM_SIGNAL_CALLBACK_HDLR pvSignalCallback =
                va_arg( *ptList, COM_SIGNAL_CALLBACK_HDLR );
            void *pvSignalCallbackArg =
                va_arg( *ptList, void * );

            // Set signal callback
            psPublicData->pvSignalCallback = pvSignalCallback;
            psPublicData->pvSignalCallbackArg = pvSignalCallbackArg;

            n32Err = DEV_UNSUPPORTED_CMD;
        }
        break;

        case COM_IOCTL_CHANGE_SETTINGS:
        {
            // Extract input
            char *pcSettings =
                va_arg( *ptList, char * );

            if (pcSettings != NULL)
            {
                struct termios termios_p;
                BOOLEAN bValid;
                int iErr;

                // Get attributes
                iErr = tcgetattr( psPublicData->fd, &termios_p);
                if(iErr == 0)
                {
                    // Next we need to parse the provided argument
                    bValid =
                        _bParseArg(
                            (const char*)pcSettings,
                            &termios_p);

                    if(bValid == TRUE)
                    {
                        // Set attributes
                        iErr = tcsetattr( psPublicData->fd,
                            TCSAFLUSH, &termios_p);
                        if(iErr != 0)
                        {
                            n32Err = DEV_ERROR;
                        }
                    }
                    else
                    {
                        n32Err = DEV_ERROR;
                    }
                }
                else
                {
                    n32Err = DEV_ERROR;
                }
            }
            else
            {
                n32Err = DEV_INVALID_INPUT;
            }
        }
        break;

        case DEV_IOCTL_GET_ERROR:
        {
            // return the error indicator
            n32Err =  psPublicData->n32Error;
        }
        break;

        case DEV_IOCTL_GET_EOF:
        {
#ifdef __QNX__
            // get and save the latest eof indication
            psPublicData->n32Eof = (N32)eof(psPublicData->fd);
#endif

            // return the eof indication
            n32Err = psPublicData->n32Eof;
        }
        break;

        case DEV_IOCTL_CLEAR_ERROR:
        {
            // clear any error
            psPublicData->n32Error = 0;

            // clear eof
            psPublicData->n32Eof = 0;
        }
        break;

        default:
            // Return error, unhandled request
            n32Err = DEV_UNSUPPORTED_CMD;
        break;
    }

    TRACE_END();
    return n32Err;
}

/*****************************************************************************
*
*       _bInitTermSettings
*
*   This function initilizes the termios structure
*
*   Inputs:
*       termios_p - A pointer to the termios structure to intilize for this port
*   Outputs:
*       A value of TRUE is returned if the input parameters are valid, otherwise
*       a value of FALSE is returned on error.
*
*****************************************************************************/
static BOOLEAN _bInitTermSettings (
    struct termios *termios_p
        )
{
#ifdef __QNX__
    int iErr = 0;
#endif

    TRACE_START();
    // Verify a valid argument was passed
    if(termios_p == NULL)
    {
        TRACE_END();
        return FALSE;
    }

    // The cfmakeraw() function sets the terminal attributes as follows:
    // termios_p->c_iflag &= ~(IGNBRK|BRKINT|PARMRK|ISTRIP|INLCR|
    //  IGNCR|ICRNL|IXON);
    // termios_p->c_oflag &= ~OPOST;
    // termios_p->c_lflag &= ~(ECHO|ECHONL|ICANON|ISIG|IEXTEN);
    // termios_p->c_cflag &= ~(CSIZE|PARENB);
    // termios_p->c_cflag |= CS8;
#ifdef __QNX__
    iErr = cfmakeraw(termios_p);
    if(iErr != 0)
    {
        return FALSE;
    }
    // no HW flow control
    termios_p->c_cflag &= (~IHFLOW);
    termios_p->c_cflag &= (~OHFLOW);
#else
    cfmakeraw(termios_p);
#endif

    // Input Modes
    // INPCK - Enable input parity check.
    termios_p->c_iflag |= (INPCK);

    // Output Modes
    // OPOST - Perform output processing.
    termios_p->c_oflag = termios_p->c_oflag;

    // Control Modes
    // CREAD - Enable receiver
    termios_p->c_cflag |= CREAD;

    // Local Modes
    termios_p->c_lflag = termios_p->c_lflag;

    // Control Characters
    termios_p->c_cc[VMIN]  =  1; /* At least one character */
    termios_p->c_cc[VTIME] =  0; /* No time component */

    cfsetispeed( termios_p, B9600 );
    cfsetospeed( termios_p, B9600 );

    TRACE_END();
    return TRUE;
}

/*****************************************************************************
*
*       _bParseArg
*
*   This function parses the input line-parameters string.
*
*   Inputs:
*       pcArg - A pointer to the line-parameters string provided by the
*           caller (application). Ex: "115200,N,8,1"
*       pun32Mode - A pointer to a UN32 which will contain the appropriate mode
*           register settings to apply when opening the port.
*       pun32BaudRate - A pointer to a UN32 which will contain the baud rate
*           in bps parsed from the input string.
*
*   Outputs:
*       A value of TRUE is returned if the input parameters are valid, otherwise
*       a value of FALSE is returned on error.
*
*****************************************************************************/
static BOOLEAN _bParseArg (
    const char *pcArg,
    struct termios *termios_p
    )
{
    char cParity;
    UN16 un16DataBits, un16StopBits;
    UN32 un32BaudRate;

    TRACE_START();
    // Verify a valid argument was passed, if so we can parse the input.
    if(pcArg == NULL)
    {
        TRACE_END();
        return FALSE;
    }

    // Begin parsing argument (e.g. 9600,N,8,1)
    sscanf(pcArg, "%u,%c,%hu,%hu",
           &un32BaudRate, &cParity, &un16DataBits, &un16StopBits);

    // Normalize input of parity (which is a char) to lower case
    cParity = tolower(cParity);

    // Process arguments...

    // Baud Rate
    switch(un32BaudRate)
    {
       case 57600:
          un32BaudRate = B57600;
          break;

       case 115200:
          un32BaudRate = B115200;
          break;

       case 230400:

#ifdef __QNX__
           // this doesn't have a definition in termios.h,
           // but it works nonetheless
          un32BaudRate = 230400L;
#else
          un32BaudRate = B230400;
#endif
       break;
       // Bosch ID #27 New baud rate is added to support BOSCH platform
       case 460800:
          un32BaudRate = B460800;
          break;
		  
       case 921600:
          un32BaudRate = B921600;
          break;   

       case 9600:
       case 19200:
       case 38400:

       default:
          fprintf(stderr, "Baud rate %d not supported by this serial port or the radio protocol\n", un32BaudRate);
          return FALSE;
    }

    // Set the baud rate
    cfsetispeed( termios_p, un32BaudRate );
    cfsetospeed( termios_p, un32BaudRate );

    // Parity: N, E, O, M, S
    termios_p->c_cflag &= ~(PARENB | PARODD);
#ifdef __QNX__
    termios_p->c_cflag &= ~(PARSTK);
#endif
    switch(cParity)
    {
        case 'n':
        break;

        case 'm':
            termios_p->c_cflag |= (PARENB | PARODD);
#ifdef __QNX__
            termios_p->c_cflag |= (PARSTK);
#endif
        break;

        case 's':
         termios_p->c_cflag |= (PARENB);
#ifdef __QNX__
         termios_p->c_cflag |= (PARSTK);
#endif
        break;

        case 'o':
            termios_p->c_cflag |= (PARENB | PARODD);
        break;

        case 'e':
            termios_p->c_cflag |= (PARENB) ;
        break;

        default:
            // Invalid parity
            TRACE_END();
            return FALSE;
    }

    // Data bits
    termios_p->c_cflag &= ~(CSIZE); // clear all bits
    switch(un16DataBits)
    {
        case 5:
            termios_p->c_cflag |= CS5;
        break;

        case 6:
            termios_p->c_cflag |= CS6;
        break;

        case 7:
            termios_p->c_cflag |= CS7;
        break;

        case 8:
            termios_p->c_cflag |= CS8;
        break;

        default:
            // Invalid data bits
            TRACE_END();
            return FALSE;
    }

    // Stop bits
    switch(un16StopBits)
    {
        case 1:
            termios_p->c_cflag &= ~CSTOPB;
        break;

        case 2:
            termios_p->c_cflag |= CSTOPB;
        break;

        case 15:
        default:
            // Invalid stop bits
            TRACE_END();
            return FALSE;
    }

    TRACE_END();
    return TRUE;
}

/*****************************************************************************
*
*       _vSetSignals
*
*       Allows the caller to set RTS and DTR on the provided port
*
*       Inputs:
*               psPublicData - A pointer to the USART public data
*               un8Signals - A bitmask of signals to set (1)
*
*       Outputs:
*               None
*
*****************************************************************************/
static void _vSetSignals(POSIX_SERIAL_PUBLIC_STRUCT *psPublicData, UN8 un8Signals)
{
    int iSignals = 0;
    TRACE_START();

    if(!ioctl(psPublicData->fd, TIOCMGET, &iSignals))
    {
        // It is only possible to set/clr RTS and DTR
        if(un8Signals & COM_SIGNAL_RTS)
        {
            iSignals |= TIOCM_RTS;
        }
        if(un8Signals & COM_SIGNAL_DTR)
        {
            iSignals |= TIOCM_DTR;
        }

        ioctl(psPublicData->fd, TIOCMSET, &iSignals);
    }

    TRACE_END();
    return;
}

/*****************************************************************************
*
*       _un8GetSignals
*
*       Allows the caller to get the IO signals on a the provided port
*
*       Inputs:
*               psPublicData - A pointer to the USART public data
*
*       Outputs:
*               un8Signals - A bitmask of current signals
*
*****************************************************************************/
static UN8 _un8GetSignals(POSIX_SERIAL_PUBLIC_STRUCT *psPublicData)
{
   int iSignals = 0, iErr;
   UN8 un8Signals = 0;

   TRACE_START();

   iErr = ioctl(psPublicData->fd, TIOCMGET, &iSignals);
   if(iErr == 0)
   {
      if(iSignals & TIOCM_RNG)
      {
         un8Signals |= COM_SIGNAL_RI;
      }
      if(iSignals & TIOCM_DSR)
      {
         un8Signals |= COM_SIGNAL_DSR;
      }
      if(iSignals & TIOCM_CAR)
      {
         un8Signals |= COM_SIGNAL_DCD;
      }
      if(iSignals & TIOCM_CTS)
      {
         un8Signals |= COM_SIGNAL_CTS;
      }
      if(iSignals & TIOCM_RTS)
      {
         un8Signals |= COM_SIGNAL_RTS;
      }
      if(iSignals & TIOCM_DTR)
      {
         un8Signals |= COM_SIGNAL_DTR;
      }
   }

   TRACE_END();
   return un8Signals;
}

/*****************************************************************************
*
*       _vClrSignals
*
*       Allows the caller to clear RTS and DTR on the provided port
*
*       Inputs:
*               psPublicData - A pointer to the USART public data
*               un8Signals - A bitmask of signals to clear (0)
*
*       Outputs:
*               None
*
*****************************************************************************/
static void _vClrSignals(POSIX_SERIAL_PUBLIC_STRUCT *psPublicData, UN8 un8Signals)
{
    int iSignals = 0;
    TRACE_START();

    // It is only possible to set/clr RTS and DTR
    if(!ioctl(psPublicData->fd, TIOCMGET, &iSignals))
    {
        // It is only possible to set/clr RTS and DTR
        if(un8Signals & COM_SIGNAL_RTS)
        {
            iSignals &= ~(TIOCM_RTS);
        }
        if(un8Signals & COM_SIGNAL_DTR)
        {
            iSignals &= ~(TIOCM_DTR);
        }

        ioctl(psPublicData->fd, TIOCMSET, &iSignals);
    }

    TRACE_END();
    return;
}

