/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_wsalerts_list.cpp
* @brief       Implementation of handling Weather and Security Alert information to HMI
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/
#include "fc_sxm_common.h"
#include "fc_sxm_tcl_wsalerts_list.h"
#include "fc_sxm_service_sxm_wsalerts.h"
#include "fc_sxm_tcl_wsalerts_app.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_tcl_wsalerts_utils.h"
#include "fc_sxm_sms_util.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_WSALERTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_wsalerts_list.cpp.trc.h"
#endif

/*=============================================================================
=======                     FORWARD DECLARATIONS                        =======
=============================================================================*/
set<fc_sxm_trWsAlertsListEntry*,fc_sxm_tclDataListTyped<fc_sxm_trWsAlertsList,
         fc_sxm_trWsAlertsListConfig>::trCompareListEntry>
fc_sxm_trWsAlertsList::_setCombinedSortedEntries;

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*********************************************************************
 *
 *FUNCTION:    vPrintSpecific
 *
 *DESCRIPTION:  Print configuration
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsListConfig::vPrintSpecific() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenFilter=%d",
             ETG_CENUM(SMSAPI_OUTPUT_OPTION_ENUM, rFilterCfg.enWsAlertsMsgObjectType)));
}

/*********************************************************************
 *
 *FUNCTION:    bCompareWsAlertPriority
 *
 *DESCRIPTION:  Compare function for weather Alert priority
 *PARAMETER:    None
 *
 *RETURNVALUE: TRUE if left item is greater than right item
 *
 ********************************************************************/
tBool fc_sxm_trWsAlertsList::bCompareWsAlertPriority(fc_sxm_trWsAlertsListEntry const *prLeft,
         fc_sxm_trWsAlertsListEntry const *prRight) const {

   tU16 sLeftPriority = prLeft->u16MsgPriority;
   tU16 sRightPriority = prRight->u16MsgPriority;

   if (sLeftPriority < sRightPriority)
   {
      return false;
   }else if (sLeftPriority > sRightPriority)
         {
             return true;
         }
         else
             {
                 return bCompareListEntryId(prLeft, prRight);
             }
}
/*********************************************************************
 *
 *FUNCTION:    bCompareItem
 *
 *DESCRIPTION:  Compare function
 *PARAMETER:    None
 *
 *RETURNVALUE: TRUE if left is greater than right
 *
 ********************************************************************/
bool fc_sxm_trWsAlertsList::bCompareItem(fc_sxm_trWsAlertsListEntry const *prLeft,
         fc_sxm_trWsAlertsListEntry const *prRight) const
{
      //based on sorted method compare
      switch (_rListCfg.rSortCfg.enSortMethod) {

         case fc_sxm_enWsAlertListSortMethod_NONE:
            {
                return TRUE; //in case of no sorted method
            }
         case fc_sxm_enWsAlertListSortMethod_DISTANCE:
            {
                return bCompareSmsDistance(prLeft, prRight);
            }
         case fc_sxm_enWsAlertListSortMethod_PRIORITY:
           {
                return bCompareWsAlertPriority(prLeft, prRight);
           }
         default:
         return bCompareListEntryId(prLeft, prRight); //default comparison
   }
   return TRUE;
}

/*********************************************************************
 *
 *FUNCTION:    bFilterPassed
 *
 *DESCRIPTION:  Filter function
 *PARAMETER:    None
 *
 *RETURNVALUE: TRUE if item found in the list
 *
 ********************************************************************/
bool fc_sxm_trWsAlertsList::bFilterPassed(fc_sxm_trWsAlertsListEntry const &rItem) const {
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::bFilterPassed"));
   (tVoid) rItem; /*To Suppress lint warnings*/
   return TRUE;
}

/*********************************************************************
 *
 *FUNCTION:    bIsAlertEnabled
 *
 *DESCRIPTION:  Filter function
 *PARAMETER:    None
 *
 *RETURNVALUE: TRUE if item found in the list
 *
 ********************************************************************/
tBool fc_sxm_trWsAlertsList::bIsAlertEnabled(fc_sxm_trWsAlertsListEntry const &roEntry) const
{
   tBool bRetVal = FALSE;
   const map<string,fc_sxm_trWSAlertConfig> mapAlertConfig = fc_sxm_tclWsAlertsApp::instance()->rGetAlertMapEntries();
   // To iterate through alert types
   SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, roEntry.vecAlertTypes)
   {
      const map<string,fc_sxm_trWSAlertConfig>::const_iterator cIt = mapAlertConfig.find(vecAlertTypeIter->oAlertExtRefCode);
      //Check if the AlertTypeId is Enabled.
      if ((cIt != mapAlertConfig.end())
               && (cIt->second.bIsEnabledInHMI == TRUE))
      {
         bRetVal = TRUE;
         break;
      }
   }
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::bIsAlertEnabled bRetVal:%d",bRetVal));
   return bRetVal;
}

/*********************************************************************
 *
 *FUNCTION:    bIsAlertSupported
 *
 *DESCRIPTION:  Filter function
 *PARAMETER:    roEntry
 *
 *RETURNVALUE: TRUE if item found in the list
 *
 ********************************************************************/
tBool fc_sxm_trWsAlertsList::bIsAlertSupported(fc_sxm_trWsAlertsListEntry const &roEntry) const
{
   tBool bRetVal = FALSE;
   const map<string,fc_sxm_trWSAlertConfig> mapAlertConfig = fc_sxm_tclWsAlertsApp::instance()->rGetAlertMapEntries();

   // To iterate through alert types
   SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, roEntry.vecAlertTypes)
   {
      const map<string,fc_sxm_trWSAlertConfig>::const_iterator cIt = mapAlertConfig.find(vecAlertTypeIter->oAlertExtRefCode);
      //Check if the AlertTypeId is Enabled.
      if ((cIt != mapAlertConfig.end())
               && (cIt->second.bIsEnabledInMap == TRUE))
      {
         bRetVal = TRUE;
         break;
      }
   }
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::bIsAlertEnabled bRetVal:%d",bRetVal));
   return bRetVal;
}

/*********************************************************************
 *
 *FUNCTION:     f32GetDistanceInfo
 *
 *DESCRIPTION:  Find the distance between the Device position and each Alert location point.
 *PARAMETER:    vecShapeGeoPoints,rDevLocation
 *
 *RETURNVALUE: f32Distance
 *
 ********************************************************************/
tF32 fc_sxm_trWsAlertsList::f32GetDistanceInfo(const vector<midw_ext_fi_tcl_PositionWGS84>& vecShapeGeoPoints,
         const midw_ext_fi_tcl_PositionWGS84& rDevLocation) const
{
   tF32 f32Distance = 0.0F;
   
   // Check if the vehicle Position is inside the provided polygon
   tBool bIsInside = fc_sxm_tclWsAlertsUtils::bIsPointInsidePolygon(vecShapeGeoPoints, rDevLocation);

   if (bIsInside)
   {
      // if the position is inside the polygon, device is inside the alert location
      ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::f32GetDistanceInfo Vehicle position is inside the polygon"));
   }
   else
   {
      // Calculate the distance from alert location for nearby and near destination to the device position
      f32Distance = fc_sxm_tclWsAlertsUtils::s32GetDistanceToPolygon(vecShapeGeoPoints, rDevLocation);
      ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::f32GetDistanceInfo f32Distance = %f", f32Distance));
   }
   return f32Distance;
}


/*********************************************************************
 *
 *FUNCTION:     vCopyAlertInfoToFi
 *
 *DESCRIPTION:  Fill WSAlert Info
 *PARAMETER:    roEntry, oFiEntry
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vCopyAlertInfoToFi(fc_sxm_trWsAlertsListEntry const &roEntry,
         midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const
{
   midw_ext_fi_tcl_PositionWGS84 rShapeGeopoint, rDevLocation;
   vector<midw_ext_fi_tcl_PositionWGS84> vecShapeGeoPoints;
   tS32 s32Lat, s32Lon;
   tU8  u8ReqType= fc_sxm_tclWsAlertsApp::instance()->u8GetListReqType();

   // Get Alert MSG ID
   oFiEntry.MessageID = roEntry.u32AlertMSGID;
   // Get Alert msg
   fc_sxm_vString2Fi(roEntry.oMsgText.c_str(), oFiEntry.Text);
   // Get Alert Start time
   oFiEntry.StartTime.Time = static_cast<tS32>(roEntry.oStartTime);
   // Get Alert End time
   oFiEntry.EndTime.Time = static_cast<tS32>(roEntry.oEndTime);
   // Get Alert Priority
   oFiEntry.Priority = roEntry.u16MsgPriority;
   // IsMarinzone : FALSE if alert region is a State or Province
   oFiEntry.bIsMarineZone = roEntry.bIsMarineZone;

   //Language: English - 0 , Spanish - 1 and French - 2
   oFiEntry.Language.enType = static_cast<midw_ext_fi_tcl_e8_Language::tenType>(roEntry.enMessageLang);
   oFiEntry.Distance = 0.0F;

   //Trace function for debugging
   vTraceFiEntry(oFiEntry);

   tF32 fhLat = fc_sxm_fFixed2Float(LOCATION.hLat (_hLocation));
   tF32 fhLon = fc_sxm_fFixed2Float(LOCATION.hLon (_hLocation));

   rDevLocation.Latitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32DegreeFloatToSms(fhLat);
   rDevLocation.Longitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32DegreeFloatToSms(fhLon);

   set<tF32> setWsAlertDistance;

   // Iterate through Alert Locations and find the distance between nearest alert location to the current position.
   SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertLocation>, vecAlertLocIter, roEntry.vecWSAlertLocations)
   {
     midw_ext_fi_tcl_WSAlertLocation rAlertLocation;
     fc_sxm_vString2Fi(vecAlertLocIter->oDescription.c_str(), rAlertLocation.Description);

     ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vCopyAlertInfoToFi LocName :%s",
              vecAlertLocIter->oDescription.c_str()));

     SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertGeoPoint>, vecAlertShapeIter, vecAlertLocIter->vecShapeGeoPoints)
     {
        // check if it is for Alert map then send data in WGS84 format
        if (u8ReqType == (tU32)fc_sxm_tenListType_AnotherLocation)
        {
           s32Lat = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(vecAlertShapeIter->fLat);
           s32Lon = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(vecAlertShapeIter->fLon);
        }
        else
        {
           s32Lat = fc_sxm_tclHelpers::fc_sxm_s32DegreeFloatToSms(vecAlertShapeIter->fLat);
           s32Lon = fc_sxm_tclHelpers::fc_sxm_s32DegreeFloatToSms(vecAlertShapeIter->fLon);
        }
        rShapeGeopoint.Latitude.s32Value = s32Lat;
        rShapeGeopoint.Longitude.s32Value = s32Lon;

        rAlertLocation.ShapeGeoPoints.push_back(rShapeGeopoint);
        vecShapeGeoPoints.push_back(rShapeGeopoint);
     }
     oFiEntry.AlertLocations.push_back(rAlertLocation);
     // Other than map data, distance should be sent to HMI
     if (u8ReqType != (tU32) fc_sxm_tenListType_AnotherLocation)
     {
        tF32 f32Distance = f32GetDistanceInfo(vecShapeGeoPoints, rDevLocation);
        setWsAlertDistance.insert(f32Distance);
        ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::f32Distance =%f", f32Distance));
     }
   }

   //For Alert Map, distance information is not required
   if (u8ReqType != (tU32) fc_sxm_tenListType_AnotherLocation)
   {
      set<tF32>::const_iterator cIt = setWsAlertDistance.begin();
      if (cIt != setWsAlertDistance.end())
      {
         // Get Distance information for each Alert message
         oFiEntry.Distance = fc_sxm_fKm2Miles(*cIt);
         ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::fiDistance =%f", oFiEntry.Distance));
      }
   }
}


/*********************************************************************
 *
 *FUNCTION:     vFillWSAlertInfo
 *
 *DESCRIPTION:  Fill WSAlert Info
 *PARAMETER:    roEntry, oFiEntry
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vFillWSAlertInfo(fc_sxm_trWsAlertsListEntry const &roEntry,
         midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const
{
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vFillWSAlertInfo Entered"));
   const map<string, fc_sxm_trWSAlertConfig> mapAlertConfig = fc_sxm_tclWsAlertsApp::instance()->rGetAlertMapEntries();

   vCopyAlertInfoToFi(roEntry,oFiEntry);

   //Iterate through to get Alert type ID and Extref codes
   SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, roEntry.vecAlertTypes)
   {
      const map<string, fc_sxm_trWSAlertConfig>::const_iterator cIt = mapAlertConfig.find(vecAlertTypeIter->oAlertExtRefCode);
      //Check if the AlertTypeId is Enabled.
      if ((cIt != mapAlertConfig.end())
               && (cIt->second.bIsEnabledInHMI == TRUE))
      {
         midw_ext_fi_tcl_WSAlertInfo rAlertType;

         rAlertType.AlertTypeID.enType = (midw_ext_fi_tcl_e8_WSAlertTypes::tenType)
                  (vecAlertTypeIter->u32AlertTypeId);
         fc_sxm_vString2Fi(vecAlertTypeIter->oAlertExtRefCode.c_str(), rAlertType.AlertExtRefCode);
         oFiEntry.AlertTypes.push_back(rAlertType);

         ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vFillWSAlertInfo ExtRefCode :%s", vecAlertTypeIter->oAlertExtRefCode.c_str()));
      }
   }
}


/*********************************************************************
 *
 *FUNCTION:     vFillWSAlertInfoMap
 *
 *DESCRIPTION:  Fill WSAlert Info for MAP
 *PARAMETER:    roEntry, oFiEntry
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vFillWSAlertInfoMap(fc_sxm_trWsAlertsListEntry const &roEntry,
         midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const
{
    ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vFillWSAlertInfoMap Entered"));
    const map<string, fc_sxm_trWSAlertConfig> mapAlertConfig = fc_sxm_tclWsAlertsApp::instance()->rGetAlertMapEntries();
    vCopyAlertInfoToFi(roEntry,oFiEntry);

    //Iterate through to get Alert type ID and Extref codes
    SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, roEntry.vecAlertTypes)
    {
       const map<string, fc_sxm_trWSAlertConfig>::const_iterator cIt = mapAlertConfig.find(vecAlertTypeIter->oAlertExtRefCode);
       //Check if the AlertTypeId is Enabled.
       if ((cIt != mapAlertConfig.end())
                && (cIt->second.bIsEnabledInMap == TRUE))
       {
         midw_ext_fi_tcl_WSAlertInfo rAlertType;

         rAlertType.AlertTypeID.enType = (midw_ext_fi_tcl_e8_WSAlertTypes::tenType) (vecAlertTypeIter->u32AlertTypeId);
         fc_sxm_vString2Fi(vecAlertTypeIter->oAlertExtRefCode.c_str(), rAlertType.AlertExtRefCode);
         rAlertType.AlertNameID = cIt->second.u32AlertNameID;
         rAlertType.AlertGroupId = cIt->second.u32AlertGroupId;
         rAlertType.AlertPrio = cIt->second.u32AlertPrio;

         rAlertType.AlertColor.Red = cIt->second.trRGBAcolors.u8Red;
         rAlertType.AlertColor.Green = cIt->second.trRGBAcolors.u8Green;
         rAlertType.AlertColor.Blue = cIt->second.trRGBAcolors.u8Blue;
         rAlertType.AlertColor.Alpha = cIt->second.trRGBAcolors.u8Alpha;

         oFiEntry.AlertTypes.push_back(rAlertType);

         ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vFillWSAlertInfoMap AlertTypeId :%d ExtRefCode :%s",
                  rAlertType.AlertTypeID.enType,vecAlertTypeIter->oAlertExtRefCode.c_str()));
      }
   }
}

/*********************************************************************
 *
 *FUNCTION:    vTraceFiEntry
 *
 *DESCRIPTION:  For debuging purpose
 *PARAMETER:    &oFiEntry
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vTraceFiEntry(midw_ext_fi_tcl_WSAlertMsg const &oFiEntry) const
{
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vTraceFiEntry Details"));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::AlertMSG_ID : %d",oFiEntry.MessageID));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::hAlertMsgText : %s", oFiEntry.Text.szValue));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::priority:%d",oFiEntry.Priority));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::tStartTime : %d",oFiEntry.StartTime.Time));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::tEndTime : %d",oFiEntry.EndTime.Time));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::AlertLocType: bIsMarineLoc : %d",oFiEntry.bIsMarineZone));
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::Language:%d",oFiEntry.Language.enType));
}

/*********************************************************************
 *
 *FUNCTION:    vAddEntry
 *DESCRIPTION: Invoke AddEntry method if Nearby or Near destination list is ready
 *PARAMETER:   &rListEntry
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vAddEntry(const fc_sxm_trWsAlertsListEntry& rListEntry)
{
   ETG_TRACE_USR2(("fc_sxm_trWsAlertsList::vAddEntry u32GetId = %d",u32GetId()));
   if((u32GetId() == static_cast<tU32> (fc_sxm_tenListType_NearBy)) ||
            (u32GetId() == static_cast<tU32>(fc_sxm_tenListType_NearDest)))
   {
      fc_sxm_tclWsAlertsApp::instance()->vAddEntry(rListEntry);
   }
}

/*********************************************************************
 *
 *FUNCTION:     vDeleteEntry
 *DESCRIPTION:  Invoke the DeleteEntry method if Nearby or Near destination list is ready
 *PARAMETER:    &rListEntry
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vDeleteEntry(const fc_sxm_trWsAlertsListEntry& rListEntry)
{
   if((u32GetId() == static_cast<tU32> (fc_sxm_tenListType_NearBy)) ||
            (u32GetId() == static_cast<tU32>(fc_sxm_tenListType_NearDest)))
   {
      fc_sxm_tclWsAlertsApp::instance()->vDeleteEntry(rListEntry);
   }
}

/*********************************************************************
 *
 *FUNCTION:     vEmitCombinedList
 *DESCRIPTION:  Invoke if the request type is for combined list
 *PARAMETER:    &rAdressing
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vEmitCombinedList(fc_sxm_trAdressing const &rAdressing)
{
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::Inside COMBINED LIST"));
   midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgListMethodResult oMRes;

   //consume all the requests till it is last.
   tBool bSendUpdate = fc_sxm_tclWsAlertsApp::instance()->bGetSendUpdate();
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::bSendUpdate :%d",bSendUpdate));

   //copy current active list entries to temp set.
   SXM_FOREACH_CONST(trSortedSetType, iter, rGetSortedEntries())
   {
      _setCombinedSortedEntries.insert(*iter);
   }

   if (bSendUpdate)
   {
      //for last request,prepare method result
      SXM_FOREACH_CONST(trSortedSetType, iterIn, _setCombinedSortedEntries)
      {
         ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::SXM_FOREACH_CONST FOR COMBINED LIST"));
         //check if this alert is disabled
         if (!bIsAlertEnabled(**iterIn))
         {
            //OK,don't sent this alert to HMI
            continue;
         }
         midw_ext_fi_tcl_WSAlertMsg oFiEntry;
         //fill the msg details from our internal list
         vFillWSAlertInfo(**iterIn, oFiEntry);
         //finally copy to method result
         oMRes.WSAlertMsglist.push_back(oFiEntry);
      }
      oMRes.ListId = (tU32) fc_sxm_tenListType_Custom;
      //finally clear temporary set
      _setCombinedSortedEntries.clear();
      fc_sxm_tclWsAlertsApp::instance()->vSetSendUpdate(false);
   }
   else
   {
      // Until the last request, no need to send the request to HMI
      return;
   }
   //finally send method result
   fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*********************************************************************
 *
 *FUNCTION:     vAlertMapListEntries
 *DESCRIPTION:  Invoke if the request type is for alert map list
 *PARAMETER:    &rAdressing, mapAlertConfig
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vEmitMapList(fc_sxm_trAdressing const &rAdressing)
{
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::Inside Alert map  List"));
   midw_ext_sxm_wsalertsfi_tclMsgGetWeatherAlertDataMethodResult oMapMRes;
   //for last request,prepare method result
   SXM_FOREACH_CONST(trSortedSetType, iterIn, rGetSortedEntries())
   {
      ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::SXM_FOREACH_CONST Alert map  List"));
      //check if this alert is supported
      if (!bIsAlertSupported(**iterIn))
      {
         //OK,don't sent this alert to HMI
         continue;
      }
      midw_ext_fi_tcl_WSAlertMsg oFiEntry;
      //fill the msg details from our internal list
      vFillWSAlertInfoMap(**iterIn, oFiEntry);
      //finally copy to method result
      oMapMRes.Alerts.push_back(oFiEntry);
   }

   oMapMRes.UpdateCounter = fc_sxm_tclWsAlertsApp::instance()->u32GetUpdateCounter();
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList:: Method Result UpdateCounter=%d",oMapMRes.UpdateCounter));

   //finally send method result
   fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(rAdressing, oMapMRes);
}

tVoid fc_sxm_trWsAlertsList::vEmitStandardList(fc_sxm_trAdressing const &rAdressing)
{
   midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgListMethodResult oMRes;
   /*Fill all available Weather alerts info from internal list*/
   SXM_FOREACH_CONST(trSortedSetType, iter, rGetSortedEntries())
   {
      ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::SXM_FOREACH_CONST FOR NORMAL LIST"));
      //check if this alert is disabled
      if (!bIsAlertEnabled(**iter))
      {
         //OK,dont sent this alert to HMI
         continue;
      }
      //fill the msg details from our internal list
      midw_ext_fi_tcl_WSAlertMsg oFiEntry;
      vFillWSAlertInfo(**iter, oFiEntry);
      //finally copy to method result
      oMRes.WSAlertMsglist.push_back(oFiEntry);
   }
   oMRes.ListId = u32GetId();

   //finally send method result
   fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(rAdressing, oMRes);
}


/*********************************************************************
 *FUNCTION:     vEmit
 *DESCRIPTION:  Update the FI-entry from the list and send message
 *PARAMETER:    1)Msg address (GetWsAlertsInfoList)
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_trWsAlertsList::vEmit(fc_sxm_trAdressing const &rAdressing)
{
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vEmit START"));

   //check if it is for combined list request
   tU8 u8ReqType = fc_sxm_tclWsAlertsApp::instance()->u8GetListReqType();
   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::u8ReqType :%d", u8ReqType));
   // Copy Alert map entries into local map

   if (u8ReqType == (tU32) fc_sxm_tenListType_Custom)
   {
      vEmitCombinedList(rAdressing);
   }
   // check if it is for Alert map list request
   else if (u8ReqType == (tU32) fc_sxm_tenListType_AnotherLocation)
   {
      vEmitMapList(rAdressing);
   }
   //then it is for normal requests
   else
   {
      vEmitStandardList(rAdressing);
   }

#  ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING

   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vEmit(%d) enListState=%d enListMode=%d numEntries=%u START",
                     oMRes.ListId,
                     ETG_CENUM(fc_sxm_tenListState, enGetListState()),
                     ETG_CENUM(fc_sxm_tenListMode, enGetListMode()),
                     rGetSortedEntries().size()));
#  else
#  endif

        ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::rGetSortedEntries SIZE: %d",rGetSortedEntries().size()));

      /* send the method result*/
   #  ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
        // write the list-operation to the method-result
      vListOperation2Fi(oMRes.Operation);
	#  endif

   ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vEmit END"));
}


#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
/*
  will be called by base-class when data are ready and the list shall be sent to hmi
 */
tVoid fc_sxm_trWsAlertsList::vStartTimer() {
    ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vStartTimer START"));
    fc_sxm_tclWsAlertsApp::instance()->vStartListTimer();
}


/*
  will be called by base-class to check the timer status
 */
tBool fc_sxm_trWsAlertsList::bIsListModeTimerRunning() {
    tBool bRes=fc_sxm_tclWsAlertsApp::instance()->bIsListUpdateTimerRunning();
    ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vIsListModeTimerRunning: %d", bRes));

    return bRes;
}
/*
  will be called by base-class when list is released
 */
tVoid fc_sxm_trWsAlertsList::vStopTimer() {
    ETG_TRACE_USR4(("fc_sxm_trWsAlertsList::vStopTimer STOP"));

    fc_sxm_tclWsAlertsApp::instance()->vStopListTimer();
}
#endif


/*=============================================================================
=======                   END OF FILE                                   =======
=============================================================================*/
