/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_wsalerts_dsrl.cpp
* @brief       Implementation of the Weather and Security Alerts DSRL functionalities.
               It invokes the SMS libraries and copied into the internal list.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/
#include "fc_sxm_tcl_wsalerts_dsrl.h"
#include "fc_sxm_sms_util.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_WSALERTS_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_wsalerts_dsrl.cpp.trc.h"
#endif

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*********************************************************************
 *FUNCTION:     fc_sxm_tclWsAlertsDSRL
 *DESCRIPTION:  Constructor
 *PARAMETER:    enDsrlType
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclWsAlertsDSRL::fc_sxm_tclWsAlertsDSRL(fc_sxm_tenDSRLType enDsrlType):
    fc_sxm_tclConfigDSRL<fc_sxm_trWsAlertsDSRLCfg>(enDsrlType,
                                                     fc_sxm_tclWsAlertsApp::instance(),
                                                     TR_CLASS_FC_SXM_WSALERTS_DSRL)
{
	ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL Constructor"));
}

/*********************************************************************
 *
 *FUNCTION:    vPrintSpecific
 *
 *DESCRIPTION:  Print configuration
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_trWsAlertsDSRLCfg::vPrintSpecific() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t enSortMethod            =%u",
                         ETG_CENUM(fc_sxm_tenWsAlertSortMethod, enSortMethod)
                       ));
}

/*********************************************************************
 *FUNCTION:     ~fc_sxm_tclWsAlertsDSRL
 *DESCRIPTION:  Destructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclWsAlertsDSRL::~fc_sxm_tclWsAlertsDSRL()
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL destructor"));
}

/*********************************************************************
 *FUNCTION:     vOnDSRLUpdate
 *DESCRIPTION:  Called when DSRL is updated             
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsDSRL::vOnDSRLUpdate(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::vOnDSRLUpdate"));
}

/*********************************************************************
 *FUNCTION:    bHandleFilterCallback
 *DESCRIPTION:   DSRL filter callback
 *PARAMETER:    hDSRL, hEntry
 *RETURNVALUE:  tBool
 ********************************************************************/
tBool fc_sxm_tclWsAlertsDSRL::bHandleFilterCallback(DSRL_OBJECT hDSRL,
                                       DSRL_ENTRY_OBJECT hEntry)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::bHandleFilterCallback"));
    /*Todo: Parameters dereferenced to suppress lint warnings*/
    (tVoid) hDSRL ;
    (tVoid) hEntry;

    return TRUE;
}

/*****************************************************************************
*FUNCTION:     bShapePointsIterator
*DESCRIPTION:  To read all the locations from weather_ref.db to fc_sxm_trWsAlertsweatherDB list
*PARAMETER:    hLocation - A Valid LOCATION_OBJECT , pvIterateArg - Pointer to the caller function
*RETURNVALUE:  TRUE/FALSE
*****************************************************************************/
tBool fc_sxm_tclWsAlertsDSRL::cb_ShapePointsIterator(LOCATION_OBJECT hLocation,void* pvArg)
{
   tBool bRetVal = FALSE;
   fc_sxm_trWsAlertLocation *poAlertLocation = static_cast<fc_sxm_trWsAlertLocation*>(pvArg);

   if ((LOCATION_INVALID_OBJECT != hLocation) && (OSAL_NULL != poAlertLocation))
   {
      fc_sxm_trWsAlertGeoPoint oWsAlertShapePt;
      oWsAlertShapePt.fLat = fc_sxm_fFixed2Float(LOCATION.hLat (hLocation));
      oWsAlertShapePt.fLon = fc_sxm_fFixed2Float(LOCATION.hLon (hLocation));

      poAlertLocation->vecShapeGeoPoints.push_back(oWsAlertShapePt);
      bRetVal = TRUE;
   }
   else
   {
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::cb_ShapePointsIterator _hLocation INVALID"));
      bRetVal = FALSE;
   }
   return bRetVal;
}

/*****************************************************************************
*FUNCTION:     cb_WsAlertsIterateLocations
*DESCRIPTION:  To read all the locations from weather_ref.db to fc_sxm_trWsAlertsweatherDB list
*PARAMETER:    hLocation - A Valid LOCATION_OBJECT , pvIterateArg - Pointer to the caller function
*RETURNVALUE:  TRUE/FALSE
*****************************************************************************/
tBool fc_sxm_tclWsAlertsDSRL::cb_WSAlertsIterateLocations(WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation,
         void* pvIterateArg)
{
   SHAPE_OBJECT hShape;
   SMSAPI_RETURN_CODE_ENUM eReturnCode;
   fc_sxm_trWsAlertsListEntry *poListEntry = static_cast<fc_sxm_trWsAlertsListEntry*>(pvIterateArg);

   if (poListEntry != OSAL_NULL)
   {
      if (hWsAlertsLocation != WS_ALERTS_LOCATION_INVALID_OBJECT)
      {
         fc_sxm_trWsAlertLocation oWsAlertLocation;

         // Extract the Alert Location ExtType from Weather Alert Location Object
         fc_sxm_vCopySmsString2Stl(WS_ALERTS_LOCATION.hLocationExtType(hWsAlertsLocation),oWsAlertLocation.oLocationExtType);
         // Extract the Alert Location Description from Weather Alert Location Object
         fc_sxm_vCopySmsString2Stl(WS_ALERTS_LOCATION.hDescription(hWsAlertsLocation),oWsAlertLocation.oDescription);
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::AlertLocDescription : %s", oWsAlertLocation.oDescription.c_str()));
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::AlertLocExtType: %s", oWsAlertLocation.oLocationExtType.c_str()));

         // Extract the Shape points from Weather Alert Location Object
         hShape = WS_ALERTS_LOCATION.hShape(hWsAlertsLocation);

         // IterateShape call back function to get shape points for each alert location
         eReturnCode = SHAPE.eIteratePoints(hShape, cb_ShapePointsIterator,&oWsAlertLocation);

         poListEntry->vecWSAlertLocations.push_back(oWsAlertLocation);

         if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
         {
            return FALSE;
         }
         return TRUE;
      }
   }
   else
   {
	   ETG_TRACE_ERR(("fc_sxm_tclWsAlertsDSRL::poListEntry NULL"));
   }
   return FALSE;
}

/*****************************************************************************
 *FUNCTION:     cb_GetWSAlertsMsgTypes
 *DESCRIPTION:  Get the Alert types
 *PARAMETER:    hMsgType
 *PARAMETER:    prMsg
 *RETURNVALUE:  tBool
 *****************************************************************************/
tBool fc_sxm_tclWsAlertsDSRL:: cb_GetWSAlertTypes(STRING_OBJECT hMsgType,
    void *prMsg)
{
   fc_sxm_trWsAlertsListEntry *poListEntry = static_cast<fc_sxm_trWsAlertsListEntry*>(prMsg);
   if (poListEntry != OSAL_NULL)
   {
      if (hMsgType != STRING_INVALID_OBJECT)
      {
         fc_sxm_trWsAlertType oWsAlertTypes;

         fc_sxm_vCopySmsString2Stl(hMsgType,oWsAlertTypes.oAlertExtRefCode);
         poListEntry->vecAlertTypes.push_back(oWsAlertTypes);

         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::AlertType:%s", oWsAlertTypes.oAlertExtRefCode.c_str()));
         return TRUE;
      }
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclWsAlertsDSRL:: poListEntry NULL"));
   }
   return FALSE;
}

/*****************************************************************************
 *FUNCTION:     vMapWSAlertTypeId
 *DESCRIPTION:  Copy the Alert Type id into the Internal list
 *PARAMETER:    rListEntry
 *RETURNVALUE:  None
 *****************************************************************************/
tVoid fc_sxm_tclWsAlertsDSRL::vMapWSAlertTypeId(fc_sxm_trWsAlertsListEntry &rListEntry) const
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::vMapWSAlertTypeId Entered"));

   const map<string,fc_sxm_trWSAlertConfig> mapAlertConfig(fc_sxm_tclWsAlertsApp::instance()->rGetAlertMapEntries());

   // For Each Alert type and finding Alert ID for each Alert Type
   SXM_FOREACH(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, rListEntry.vecAlertTypes)
   {
      // with Exref code find the corresponding AlertTypeId from the map
      const map<string,fc_sxm_trWSAlertConfig>::const_iterator cIt = mapAlertConfig.find(vecAlertTypeIter->oAlertExtRefCode);

      if(cIt != mapAlertConfig.end())
      {
         tU32 u32AlertTypeId = cIt->second.u32AlertTypeId;
         vecAlertTypeIter->u32AlertTypeId = u32AlertTypeId;

         ETG_TRACE_USR2(("fc_sxm_tclWsAlertsDSRL::vMapWSAlertTypeId  AlertTypeID:%d",
                        vecAlertTypeIter->u32AlertTypeId));
      }
      else
      {
         // AlertType ID not found
      }
      ETG_TRACE_USR2(("fc_sxm_tclWsAlertsDSRL::vMapWSAlertTypeId  ExtRefCode:%s",
               vecAlertTypeIter->oAlertExtRefCode.c_str()));
   }
}

/*********************************************************************
 *FUNCTION:    s16HandleSortCallback
 *DESCRIPTION:  Sort callback
 *PARAMETER:    hDSRL, hEntry1, hEntry2
 *RETURNVALUE:  tS16
 ********************************************************************/
tS16 fc_sxm_tclWsAlertsDSRL::s16HandleSortCallback(DSRL_OBJECT hDSRL,
                                      DSRL_ENTRY_OBJECT hEntry1,
                                      DSRL_ENTRY_OBJECT hEntry2)
{
	ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::s16HandleSortCallback - Return 1"));
   /*Todo: Parameters dereferenced to suppress lint warnings*/
   (tVoid) hEntry1 ;
   (tVoid) hEntry2 ;

   (tVoid) hDSRL ;
	return 1;
}

/*********************************************************************
 *FUNCTION:     vGetWSAlertInfo
 *DESCRIPTION:  Get the WSAlert Message Information from SMS
 *PARAMETER:    WS_ALERT_MSG_OBJECT hWsAlertMsg
 *              fc_sxm_trWsAlertsListEntry rListEntry
 *RETURNVALUE:  tBool
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsDSRL::vGetWSAlertInfo(fc_sxm_trWsAlertsListEntry& rListEntry,WS_ALERT_MSG_OBJECT& hWsAlertMsg) const
{
   // Extract the Alert MSG ID from the SMS
    rListEntry.u32AlertMSGID = WS_ALERT_MSG.tID(hWsAlertMsg);
    rListEntry.u32SxmId = rListEntry.u32AlertMSGID;

    rListEntry.oMsgText.clear();
    //Actual alert message with full information start/end time and extref code
    fc_sxm_vCopySmsString2Stl(WS_ALERT_MSG.hMsgText(hWsAlertMsg),rListEntry.oMsgText);

    // Extract the Alert MSG Start/End Date and Time from the SMS
    rListEntry.oStartTime = WS_ALERT_MSG.tStartTime(hWsAlertMsg);
    rListEntry.oEndTime = WS_ALERT_MSG.tEndTime (hWsAlertMsg);
    // To extract Alert message priority from the hWsAlertMsg
    rListEntry.u16MsgPriority = WS_ALERT_MSG.un16Priority(hWsAlertMsg);

    // To extract Alert message language from the hWsAlertMsg
    // English - 0 , Spanish - 1 and French - 2
    rListEntry.enMessageLang = WS_ALERT_MSG.eMsgLang(hWsAlertMsg);

    //TRUE if alert region is a Marine Zone
    //FALSE if alert region is a State or Province
    rListEntry.bIsMarineZone = WS_ALERT_MSG.bIsMarineZone(hWsAlertMsg);

    ETG_TRACE_USR4(("Display Weather and Security Alert Message List"));
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::MSG_ID : %d",rListEntry.u32AlertMSGID));
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::hAlertMsgText : %s",rListEntry.oMsgText.c_str()));
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::Priority:%d",rListEntry.u16MsgPriority));
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::AlertLocType: bIsMarineLoc : %d",rListEntry.bIsMarineZone));
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::Language :%d",rListEntry.enMessageLang));
}

/*********************************************************************
 *FUNCTION:     bHandleIterateCallback
 *DESCRIPTION:  WsAlerts service DSRL iterate callback
 *PARAMETER:    DSRL_OBJECT hDSRL, DSRL_ENTRY_ID tEntryID,
 *              DSRL_ENTRY_STATUS_ENUM eStatus, DSRL_ENTRY_OBJECT hEntryObject
 *RETURNVALUE:  tBool
 ********************************************************************/
tBool fc_sxm_tclWsAlertsDSRL::bHandleIterateCallback(DSRL_OBJECT hDSRL,
                                                 DSRL_ENTRY_ID tEntryID,
                                                 DSRL_ENTRY_STATUS_ENUM eStatus,
                                                 DSRL_ENTRY_OBJECT hEntryObject)
{
   (tVoid) hDSRL ;
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::bHandleIterateCallback - START, DSRL_ENTRY_ID = %d , eStatus=%d", tEntryID,eStatus));
   fc_sxm_trWsAlertsListEntry rListEntry;
   SMSAPI_RETURN_CODE_ENUM eReturnCode;

   WS_ALERT_MSG_OBJECT hWsAlertMsg = WS_ALERT_MSG_INVALID_OBJECT;

   //Extract the Weather Alert Message handle
   hWsAlertMsg = DSRL_ENTRY.hWSAlertMsg(hEntryObject);

   if(hWsAlertMsg == WS_ALERT_MSG_INVALID_OBJECT)
   {
       return FALSE;
   }
   // Get the WSAlertInfo from the SMS
   vGetWSAlertInfo(rListEntry,hWsAlertMsg);
   
   rListEntry.u32DsrlEntryID=tEntryID;
   
   eReturnCode = WS_ALERT_MSG.eIterateAlertTypes(hWsAlertMsg, cb_GetWSAlertTypes, &rListEntry);

   if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
   {
       return FALSE;
   }

   // Iterate the list of Alert locations
   eReturnCode = WS_ALERT_MSG.eIterateLocations(hWsAlertMsg,cb_WSAlertsIterateLocations, &rListEntry);

   if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
   {
       return FALSE;
   }
   // Copy the AlertTypeID into the internal list
   vMapWSAlertTypeId(rListEntry);

   // Alert Pop-up
   // check for DSRL ENTRY STATUS
   if(eStatus == DSRL_ENTRY_STATUS_NEW || eStatus == DSRL_ENTRY_STATUS_CHANGED)
   {
       rListEntry.enState = fc_sxm_enListEntryState_New;
   }

   /* Finally Update this Weather Alert Entry into our internal list*/
   fc_sxm_trWsAlertsList::vUpdateEntry(eStatus, rListEntry);
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsDSRL::bHandleIterateCallback - END"));
   return TRUE;
}


/*********************************************************************
 *FUNCTION:     vHandleDsrlEntryInvalid
 *DESCRIPTION:  Remove the DSRL Entry
 *PARAMETER:    tEntryID
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsDSRL::vHandleDsrlEntryInvalid(DSRL_ENTRY_ID tEntryID)
{
   fc_sxm_trWsAlertsList::vRemoveEntry(tEntryID);
}

/*=============================================================================
=======                   END OF FILE                                   =======
=============================================================================*/
