/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_tabweather_list.h
* @brief       Declaration of handling Tabweather information to the HMI
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_TABWEATHER_LIST_H_
#define _FC_SXM_TCL_TABWEATHER_LIST_H_

#include "fc_sxm_tcl_data_list.hpp"
#include "fc_sxm_tcl_base_dsrl.h"
#include "fc_sxm_tabweather_types.h"

#define WIND_DIRECTION_INVALID 65535
#define WIND_SPEED_INVALID 255
#define TEMP_INVALID 32767
#define RANGE_INVALID 0
#define CHANCE_INVALID 255

typedef enum
{
    fc_sxm_weather_enAIR_POLLEN_LEVEL_LOW = 0,
    fc_sxm_weather_enAIR_POLLEN_LEVEL_MODERATE,
    fc_sxm_weather_enAIR_POLLEN_LEVEL_HIGH,
    fc_sxm_weather_enAIR_POLLEN_LEVEL_INVALID
} fc_sxm_weather_enPollenLevel;

struct fc_sxm_trTabweatherTypeListEntry {
   WEATHER_FORECAST_TYPE_ENUM enForecastType;
   tU8 u8EventCode;
   string oDescription;
   string oStateName;
   LOC_ID u32TabweatherLocId;
   string oWeatherStation;
   tS16 s16CurrentTemp;
   tS16 s16TempHigh;
   tS16 s16TempLow;
   tU8 u8PrecipitationChance;
   tF32 f32PrecipitationAmountRangeLow;
   tF32 f32PrecipitationAmountRangeHigh;
   tU8 u8WindSpeed;
   tU16 u16WindDirection;
   tU8 u8HumidityRangeLow;
   tU8 u8HumidityRangeHigh;
   tU8 u8UVInfoRangeLow;
   tU8 u8UVInfoRangeHigh;
   fc_sxm_weather_enPollenLevel enPollenLevel;
   AIR_QUALITY_ENUM enAirQuality;
   FORECAST_CLOUD_COVER_ENUM enCloudCover;
   tU32 u32TimeStamp;

   fc_sxm_trTabweatherTypeListEntry() : enForecastType(WEATHER_FORECAST_TYPE_MAX),
                                        u8EventCode((tU8)WEATHER_EVENT_DATA_NA),
                                        oDescription(""),
                                        u32TabweatherLocId((tU32)LOC_INVALID_ID),
                                        s16CurrentTemp((tS16)TEMP_INVALID),
                                        s16TempHigh((tS16)TEMP_INVALID),
                                        s16TempLow((tS16)TEMP_INVALID),
                                        u8PrecipitationChance((tU8)CHANCE_INVALID),
                                        f32PrecipitationAmountRangeLow((tF32)RANGE_INVALID),
                                        f32PrecipitationAmountRangeHigh((tF32)RANGE_INVALID),
                                        u8WindSpeed((tU8)WIND_SPEED_INVALID),
                                        u16WindDirection((tU16)WIND_DIRECTION_INVALID),
                                        u8HumidityRangeLow((tU8)RANGE_INVALID),
                                        u8HumidityRangeHigh((tU8)RANGE_INVALID),
                                        u8UVInfoRangeLow((tU8)RANGE_INVALID),
                                        u8UVInfoRangeHigh((tU8)RANGE_INVALID),
                                        enPollenLevel(fc_sxm_weather_enAIR_POLLEN_LEVEL_INVALID),
                                        enAirQuality(AIR_QUALITY_NO_DATA),
                                        enCloudCover(FORECAST_CLOUD_COVER_NO_DATA),
                                        u32TimeStamp((tU32)0)
                                        {  }

  tVoid fc_sxm_trTabweatherDefaultListEntry(tU16 un16ForecastType)
   {
      enForecastType = (WEATHER_FORECAST_TYPE_ENUM)un16ForecastType;
      oDescription.clear();
      s16CurrentTemp = TEMP_INVALID;
      s16TempHigh = TEMP_INVALID;
      s16TempLow = TEMP_INVALID;
      u8EventCode = (tU8)WEATHER_EVENT_DATA_NA;
      u8WindSpeed = WIND_SPEED_INVALID;
      u16WindDirection = WIND_DIRECTION_INVALID;
      enPollenLevel = fc_sxm_weather_enAIR_POLLEN_LEVEL_INVALID;
      enAirQuality = AIR_QUALITY_NO_DATA;
      u8HumidityRangeLow = RANGE_INVALID;
      u8HumidityRangeHigh = RANGE_INVALID;
      u8PrecipitationChance = CHANCE_INVALID;
      f32PrecipitationAmountRangeHigh = RANGE_INVALID;
      f32PrecipitationAmountRangeLow = RANGE_INVALID;
      u8UVInfoRangeHigh = RANGE_INVALID;
      u8UVInfoRangeLow = RANGE_INVALID;
      enCloudCover = FORECAST_CLOUD_COVER_NO_DATA;
      u32TimeStamp = 0U;
   };
};

// structure with data of a list-entry for HMI
struct fc_sxm_trTabweatherListEntry:public fc_sxm_trBaseListEntry {
public:
    map<WEATHER_FORECAST_TYPE_ENUM, fc_sxm_trTabweatherTypeListEntry> listTabweatherTypes;
	// todo: add needed members
};

/*
  sort-methods supported
*/
typedef enum
{
   fc_sxm_enTabweatherListSortMethod_NONE = 0

} fc_sxm_tenTabweatherListSortMethod;

/* sort-configuration consisting of sort-method and additional parameters */
struct fc_sxm_trTabweatherListSortCfg {
    fc_sxm_tenTabweatherListSortMethod enSortMethod;
    /* Check if configuration has changed.
       to check if sorting has chanded, configuration has to provice operator!=
       attention:
       operator is used in the way that it only returns TRUE if
       the changes are relevant for the current configuration.
       e.g. if sorting by distance is not enabled, changing of the 
       location has no effect.
    */

    tBool operator!=(fc_sxm_trTabweatherListSortCfg const &b) const {
    	(void)b;// solve prio2 lint info 713
        return FALSE;
    }

};

/*
  sort-configuration consisting of filter-method and additional parameters
  e.g. for which brand shall we filter
*/
struct fc_sxm_trTabweatherListFilterCfg {
	WEATHER_MSG_OBJECT_TYPE_ENUM enWeatherMsgObjectType;

	fc_sxm_trTabweatherListFilterCfg() :
	   enWeatherMsgObjectType(WEATHER_MSG_TYPE_INVALID)
	{}

    /* to check if filter has chanded, configuration has to provice operator!= 
       see sort-configuration 
    */
    tBool operator!=(fc_sxm_trTabweatherListFilterCfg const &b) const {
        SXM_RET_1_IF_NEQ(enWeatherMsgObjectType);
        return FALSE;
    }

};

/*
  struct containing all our configuration.
  Must be dericed from fc_sxm_trBaseListConfig
*/
struct fc_sxm_trTabweatherListConfig:public fc_sxm_trBaseListConfig {
public:
    typedef fc_sxm_trTabweatherListEntry trListEntry;
    // typedef to make type of list-entries visible for templates
    // post-sort and post-filter to be applied after dsrl-iterate is done
    fc_sxm_trTabweatherListSortCfg rSortCfg;
    fc_sxm_trTabweatherListFilterCfg rFilterCfg;
    // see sort-config
    tBool operator!=(fc_sxm_trTabweatherListConfig const &b) const {
        SXM_RET_1_IF_NEQ(rSortCfg);
        SXM_RET_1_IF_NEQ(rFilterCfg);
        return FALSE;
    }
    /*
      overwrite method from base-class if the location is used in this configuration.
      While not sorting or filtering for distance, the location is not used.
      Only if location is used, on change of location actions will be taken by the framework
    */
    virtual tBool bUsesLocation() const {
        return rSortCfg.enSortMethod==fc_sxm_enTabweatherListSortMethod_NONE;
    }

    virtual tVoid vPrintSpecific() const;
    ~fc_sxm_trTabweatherListConfig(){};

};


class midw_ext_fi_tcl_WeatherInfo;
/*
  The Tabweather-list, derive from template fc_sxm_tclDataListTyped.
  Template-parameters are our list-class and our configuration.
*/
class fc_sxm_trTabweatherList:public fc_sxm_tclDataListTyped<fc_sxm_trTabweatherList, fc_sxm_trTabweatherListConfig> {
 public:
    /*
      constructor:
      Simply pass parameters to base-class
    */
    fc_sxm_trTabweatherList(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp):
        fc_sxm_tclDataListTyped<fc_sxm_trTabweatherList, fc_sxm_trTabweatherListConfig>(u32ListId, enListType, poDataApp, TR_CLASS_FC_SXM_TABWEATHER_APP)           
    {}
        
        /* compare-fn that shall compare according to fc_sxm_trTabweatherListSortCfg
           Attention:
           Strict ordering has to be proviced: *prLeft<*prRight || *prRights<*prLeft
           (not strict weak ordering as for stl-containers)
           This is accomplished by comparing a unique id as fallback in case of *prLeft==*prRight
        */
    virtual bool bCompareItem(fc_sxm_trTabweatherListEntry const *prLeft, fc_sxm_trTabweatherListEntry const *prRight) const;

    // compare-fn that shall filter according to fc_sxm_trTabweatherListFilterCfg
    virtual bool bFilterPassed(fc_sxm_trTabweatherListEntry const &rItem)  const;
    
 protected:
    /*
      will be called by base-class when data are ready and the list shall be sent to hmi
     */
    virtual tVoid vEmit(fc_sxm_trAdressing const &rAdressing);

 private:
    /*
       Utility method to Print all the FI entries being passed to MIDW
     */
    tVoid vPrintFIFields (const midw_ext_fi_tcl_WeatherInfo& oFiEntry) const;
    
};


#endif
/*=============================================================================
=======                   END OF FILE                                   =======
=============================================================================*/
