/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_tabweather_dsrl.cpp
* @brief       Implementation of the Tabweather DSRL functionalities.
               It invokes the SMS libraries and copied into the internal list.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/
/*=============================================================================
 =======                            INCLUDES                             =======
 =============================================================================*/
#include "fc_sxm_tcl_tabweather_app.h"
#include "fc_sxm_tcl_tabweather_dsrl.h"
#include "fc_sxm_tcl_tabweather_list.h"
#include "fc_sxm_sms_util.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_TABWEATHER_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_tabweather_dsrl.cpp.trc.h"
#endif

tVoid fc_sxm_trTabweatherDSRLCfg::vPrintSpecific() const {
	ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t enSortMethod            =%u", ETG_CENUM(fc_sxm_tenTabweatherSortMethod, enSortMethod)));
	ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t enWeatherMsgObjectType  =%u", ETG_CENUM(WEATHER_MSG_OBJECT_TYPE_ENUM, rFilterCfg.enWeatherMsgObjectType)));
}

/*********************************************************************
 *FUNCTION:     fc_sxm_tclTabweatherDSRL
 *DESCRIPTION:  Constructor
 *PARAMETER:    enDsrlType
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclTabweatherDSRL::fc_sxm_tclTabweatherDSRL( fc_sxm_tenDSRLType enDsrlType ) :
		fc_sxm_tclConfigDSRL<fc_sxm_trTabweatherDSRLCfg>(enDsrlType,
				fc_sxm_tclTabweatherApp::instance(),
				TR_CLASS_FC_SXM_TABWEATHER_DSRL) { /* Default CTOR*/ }

/*********************************************************************
 *FUNCTION:     ~fc_sxm_tclTabweatherDSRL
 *DESCRIPTION:  Destructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclTabweatherDSRL::~fc_sxm_tclTabweatherDSRL() {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL destructor"));
}

/*********************************************************************
 *FUNCTION:     vOnDSRLUpdate
 *DESCRIPTION:  Called when DSRL is updated             
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vOnDSRLUpdate(tVoid) {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vOnDSRLUpdate"));
	//Update the property to indicate clients
#if 0    
	fc_sxm_tcl_trTabweatherPropertyNewUpdateTabweatherInfoListStatus TabweatherUpdate;

	midw_ext_fi_tcl_SxmListInfo ListInfo;

	//ListInfo.ListType=midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY;
	//ListInfo.Status=midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_READY ;
	/* Set the Property as update available */
	TabweatherUpdate.oFiMsg.TabweatherListStatus.ListInfo.push_back(ListInfo);
	/* Define Protected Property update */
	fc_sxm_tclTabweatherProperties TabweatherUpdateProperty;
	TabweatherUpdateProperty.oNewUpdateTabweatherInfoListStatus.vSetAndNotify(TabweatherUpdate);
#endif//if 0
}

/*********************************************************************
 *FUNCTION:    bHandleFilterCallback
 *DESCRIPTION:   DSRL filter callback
 *PARAMETER:    hDSRL, hEntry
 *RETURNVALUE:  tBool
 ********************************************************************/
tBool fc_sxm_tclTabweatherDSRL::bHandleFilterCallback(DSRL_OBJECT hDSRL,
		DSRL_ENTRY_OBJECT hEntry) {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::bHandleFilterCallback"));
	/*Todo: Parameters dereferenced to suppress lint warnings*/
	(tVoid) hDSRL;
	WEATHER_MSG_OBJECT hWeatherMsg =
			(WEATHER_MSG_OBJECT) DSRL_ENTRY.hWeatherMsg(hEntry);

	if (hWeatherMsg != WEATHER_MSG_INVALID_OBJECT) {
		if (WEATHER_MSG_TYPE_FORECAST
				== (WEATHER_MSG_OBJECT_TYPE_ENUM) WEATHER_MSG.eType(hWeatherMsg)) {
			/* filter the weather msg type forecast*/
			ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::bHandleFilterCallback - Return TRUE"));
			return TRUE;
		} else {
			ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::bHandleFilterCallback - Return FALSE"));
			return FALSE;
		}
	} else {
		ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::bHandleFilterCallback - WEATHER_MSG_INVALID_OBJECT"));
		return FALSE;
	}
}

/*********************************************************************
 *FUNCTION:    s16HandleSortCallback
 *DESCRIPTION:  Sort callback
 *PARAMETER:    hDSRL, hEntry1, hEntry2
 *RETURNVALUE:  tS16
 ********************************************************************/
tS16 fc_sxm_tclTabweatherDSRL::s16HandleSortCallback(DSRL_OBJECT hDSRL,
		DSRL_ENTRY_OBJECT hEntry1, DSRL_ENTRY_OBJECT hEntry2) {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::s16HandleSortCallback - Return 1"));
	/*Todo: Parameters dereferenced to suppress lint warnings*/
	(tVoid) hEntry1;
	(tVoid) hEntry2;
	(tVoid) hDSRL;
	return 1;
}

/*********************************************************************
 *FUNCTION:     bHandleIterateCallback
 *DESCRIPTION:  TabWeather service DSRL iterate callback
 *PARAMETER:    DSRL_OBJECT hDSRL, DSRL_ENTRY_ID tEntryID,
 *              DSRL_ENTRY_STATUS_ENUM eStatus, DSRL_ENTRY_OBJECT hEntryObject
 *RETURNVALUE:  tBool
 ********************************************************************/
tBool fc_sxm_tclTabweatherDSRL::bHandleIterateCallback(DSRL_OBJECT hDSRL,
		DSRL_ENTRY_ID tEntryID, DSRL_ENTRY_STATUS_ENUM eStatus,
		DSRL_ENTRY_OBJECT hEntryObject) {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::bHandleIterateCallback - Start, DSRL_ENTRY_ID = %d , eStatus=%d", tEntryID,eStatus));

	WEATHER_MSG_OBJECT hWeatherMsg = DSRL_ENTRY.hWeatherMsg(hEntryObject);

	if (hWeatherMsg != WEATHER_MSG_INVALID_OBJECT) {
		fc_sxm_trTabweatherListEntry rListEntry;
		fc_sxm_trTabweatherTypeListEntry rEntry;
		FORECAST_OBJECT hForecast;
		LOCATION_OBJECT hLocation;

		for (tU16 un16ForecastType = (tU16) WEATHER_FORECAST_TYPE_CURRENT;
				un16ForecastType < (tU16) WEATHER_FORECAST_TYPE_MAX;
				un16ForecastType++) {
			/*Forecast object obtained for each forecast type*/
			WEATHER_FORECAST_TYPE_ENUM enWeatherForecastType =
					(WEATHER_FORECAST_TYPE_ENUM) un16ForecastType;
			SMSAPI_RETURN_CODE_ENUM eWeatherResult = WEATHER_MSG.eGetForecast(
					hWeatherMsg, enWeatherForecastType, &hForecast);

			if (eWeatherResult == SMSAPI_RETURN_CODE_SUCCESS) {
				/* Get All Forecast object properties*/
				rEntry.enForecastType = enWeatherForecastType;
				ETG_TRACE_USR4(("====%u====",ETG_CENUM(WEATHER_FORECAST_TYPE_ENUM,un16ForecastType)));
				vTemperatureForecast(hForecast, rEntry);
				vWeatherEventForecast(hForecast, rEntry);
				vWindForecast(hForecast, rEntry);
				vAirForecast(hForecast, rEntry);
				vHumidityForecast(hForecast, rEntry);
				vPrecipitationForecast(hForecast, rEntry);
				vUVInfoForecast(hForecast, rEntry);
				vCloudCoverForecast(hForecast, rEntry);
				vTimeStampForecast(hForecast, rEntry);
			} else {
				ETG_TRACE_USR4(("Data not available for ====%u====",ETG_CENUM (WEATHER_FORECAST_TYPE_ENUM,un16ForecastType)));
				/*Forecast object not available for a forecast type, Assign default value to parameters*/
				rEntry.fc_sxm_trTabweatherDefaultListEntry(enWeatherForecastType);
			}
			/* Obtain Locid and weather station name */
			hLocation = WEATHER_MSG.hLocation(hWeatherMsg);
			vLocationInfo(hLocation, rEntry);
			/*All Parameters copied to the list based on the Forecast type*/
			rListEntry.listTabweatherTypes[enWeatherForecastType] = rEntry;
		}
		/*End of for(un16ForecastType) loop*/
		rListEntry.u32SxmId = rEntry.u32TabweatherLocId;
		rListEntry.hLocation = hLocation;
		rListEntry.u32DsrlEntryID = tEntryID;
		ETG_TRACE_USR4(("***** End Tabweather Info *****"));

		fc_sxm_trTabweatherList::vUpdateEntry(eStatus, rListEntry);
		ETG_TRACE_USR4(("bHandleIterateCallback - vUpdateEntry : SXM-ID:%d",rListEntry.u32SxmId));

		/*Methodresult for AddFavorite, Sent once*/
		if ((TRUE == fc_sxm_tclTabweatherApp::instance()->bGetbSetAsFavFlag())
				&& (fc_sxm_tclTabweatherApp::instance()->GetFavDsrl()) == hDSRL
				&& (fc_sxm_tclTabweatherApp::instance()->GetLocidForFav())
						== rEntry.u32TabweatherLocId) {
			ETG_TRACE_USR4(("bHandleIterateCallback - Fav. DSRL, Send the data to HMI"));
			fc_sxm_tclTabweatherApp::instance()->vSendFavList( rEntry.u32TabweatherLocId, FALSE);
			fc_sxm_tclTabweatherApp::instance()->vSetbSetAsFavFlag(FALSE);
		}

	} else {
		ETG_TRACE_ERR(("fc_sxm_tclTabweatherDSRL::bHandleIterateCallback, Weather msg. invalid object"));
	}

	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::bHandleIterateCallback - End,DSRL ID=%d",u32GetId()));
	return TRUE;
}

/*********************************************************************
 *FUNCTION:     vHandleDsrlEntryInvalid
 *DESCRIPTION:  Remove the DSRL Entry
 *PARAMETER:    tEntryID
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vHandleDsrlEntryInvalid( DSRL_ENTRY_ID tEntryID) {
	tU32 u32DsrlId = u32GetId();
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vHandleDsrlEntryInvalid, Entry ID=%d,dsrl id=%d",tEntryID,u32DsrlId));
	fc_sxm_trTabweatherList::vRemoveEntry(tEntryID);
}

/*********************************************************************
 *FUNCTION:     vWeatherEventForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for WeatherEventForecast
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vWeatherEventForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vWeatherEventForecast Entry"));
	WEATHER_EVENT_OBJECT hWeatherEvent;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eWeatherEvent(hForecast, &hWeatherEvent);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		WEATHER_EVENT_CODE_ENUM peEventCode;
		/*Weather event object*/
		enReturnCode = WEATHER_EVENT.eCode(hWeatherEvent, &peEventCode);
		rEntry.u8EventCode =
				(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
						(tU8) peEventCode :
						(tU8) WEATHER_EVENT_DATA_NA;

		/* Weather description*/
		STRING_OBJECT hEventDescription = WEATHER_EVENT.hDescription(hWeatherEvent);
		if (STRING_INVALID_OBJECT != hEventDescription) {
         fc_sxm_vCopySmsString2Stl(hEventDescription,rEntry.oDescription);
		}
		else {
			rEntry.oDescription = "";
		}
	}
	ETG_TRACE_USR4(("rEntry.oDescription = %s", rEntry.oDescription.c_str()));
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vWeatherEventForecast Exit\tWeather event code=%d",rEntry.u8EventCode));
}

/*********************************************************************
 *FUNCTION:     vTemperatureForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for Temperature
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vTemperatureForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vTemperatureForecast Entry"));
	TEMPERATURE_OBJECT hTemperature;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eTemperature(hForecast, &hTemperature);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		tS16 s16CurrentTemp, s16HighTemp, s16LowTemp;
		/*Temperature object - Current, High and Low*/
		enReturnCode = TEMPERATURE.eCurrent(hTemperature, &s16CurrentTemp);
		rEntry.s16CurrentTemp =
				(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
						s16CurrentTemp : TEMP_INVALID;

		enReturnCode = TEMPERATURE.eHigh(hTemperature, &s16HighTemp);
		rEntry.s16TempHigh =
				(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
						s16HighTemp : TEMP_INVALID;

		enReturnCode = TEMPERATURE.eLow(hTemperature, &s16LowTemp);
		rEntry.s16TempLow =
				(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
						s16LowTemp : TEMP_INVALID;
	}
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vTemperatureForecast Exit\tTemp = %d, \tHigh temp = %d, \tLow Temp = %d",
			rEntry.s16CurrentTemp,rEntry.s16TempHigh,rEntry.s16TempLow));
}

/*********************************************************************
 *FUNCTION:     vPrecipitationForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for Precipitation
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vPrecipitationForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vPrecipitationForecast Entry"));
	PRECIPITATION_OBJECT hPrecipitation;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.ePrecipitation(hForecast, &hPrecipitation);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		/*Precipitation object - Chance and Amount range*/
		tU8 pun8Chance;
		enReturnCode = PRECIPITATION.eChance(hPrecipitation, &pun8Chance);
		rEntry.u8PrecipitationChance =
				(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
						pun8Chance : CHANCE_INVALID;

		/* Create OSAL objects for hRangeLow and hRangeHigh */
		OSAL_FIXED_OBJECT_DATA atPrecpRangeDataLow[OSAL_FIXED_OBJECT_SIZE], atPrecpRangeDataHigh[OSAL_FIXED_OBJECT_SIZE];
		OSAL_FIXED_OBJECT hRangeLow = OSAL_FIXED.hCreateInMemory(0, 0,
				atPrecpRangeDataLow);
		OSAL_FIXED_OBJECT hRangeHigh = OSAL_FIXED.hCreateInMemory(0, 0,
				atPrecpRangeDataHigh);

		enReturnCode = PRECIPITATION.eAmountRange(hPrecipitation, hRangeLow, hRangeHigh);
		if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
			rEntry.f32PrecipitationAmountRangeHigh = fc_sxm_fFixed2Float(hRangeHigh);
			rEntry.f32PrecipitationAmountRangeLow = fc_sxm_fFixed2Float(hRangeLow);
		} else {
			rEntry.f32PrecipitationAmountRangeHigh = RANGE_INVALID;
			rEntry.f32PrecipitationAmountRangeLow = RANGE_INVALID;
		}
	}
	ETG_TRACE_USR4(("Precipitation Chance (in percent) = %d \tPrecip. amount range: %f - %f inches,",
			rEntry.u8PrecipitationChance, rEntry.f32PrecipitationAmountRangeLow, rEntry.f32PrecipitationAmountRangeHigh));
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vPrecipitationForecast Exit"));
}

/*********************************************************************
 *FUNCTION:     vWindForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for Wind
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vWindForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vWindForecast Entry"));
	WIND_OBJECT hWind;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eWind(hForecast, &hWind);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		tU8 pun8WindSpeed;
		tU16 pun16WindDirection;

		/*Wind object - Get Speed*/
		enReturnCode = WIND.eSpeed(hWind, &pun8WindSpeed);
		rEntry.u8WindSpeed =
				(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
						pun8WindSpeed : WIND_SPEED_INVALID;

		/*Wind object - Get Direction*/
		enReturnCode = WIND.eDirection(hWind, &pun16WindDirection);
		rEntry.u16WindDirection =
				((pun8WindSpeed == 0)
						|| (rEntry.u8WindSpeed == WIND_SPEED_INVALID)
						|| (enReturnCode
								!= SMSAPI_RETURN_CODE_SUCCESS)) ?
						WIND_DIRECTION_INVALID : pun16WindDirection;
	}
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vWindForecast Exit\tWind Direction = %u, and Speed = %u\n",rEntry.u16WindDirection, rEntry.u8WindSpeed));
}

/*********************************************************************
 *FUNCTION:     vHumidityForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for Humidity
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vHumidityForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vHumidityForecast Entry"));
	HUMIDITY_OBJECT hHumidity;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eHumidity(hForecast, &hHumidity);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		tU8 pun8HumidityRangeLow, pun8HumidityRangeHigh;
		/*Humidity object - Range*/
		enReturnCode = HUMIDITY.eRange(hHumidity, &pun8HumidityRangeLow, &pun8HumidityRangeHigh);
		if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
			rEntry.u8HumidityRangeLow = pun8HumidityRangeLow;
			rEntry.u8HumidityRangeHigh = pun8HumidityRangeHigh;
		} else {
			rEntry.u8HumidityRangeLow = RANGE_INVALID;
			rEntry.u8HumidityRangeHigh = RANGE_INVALID;
		}

	}
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vHumidityForecast Exit\tHumidity Range = %d - %d\n",rEntry.u8HumidityRangeLow,rEntry.u8HumidityRangeHigh));
}

/*********************************************************************
 *FUNCTION:     vUVInfoForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for UVInfo
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vUVInfoForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vUVInfoForecast Entry"));
	UVINFO_OBJECT hUVInfo;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eUVInfo(hForecast, &hUVInfo);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		tU8 pun8UVIndexRangeLow, pun8UVIndexRangeHigh;
		/*UV Info object - Index range*/
		enReturnCode = UVINFO.eIndexRange(hUVInfo, &pun8UVIndexRangeLow, &pun8UVIndexRangeHigh);
		if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
			rEntry.u8UVInfoRangeHigh = pun8UVIndexRangeHigh;
			rEntry.u8UVInfoRangeLow = pun8UVIndexRangeLow;
		} else {
			rEntry.u8UVInfoRangeHigh = RANGE_INVALID;
			rEntry.u8UVInfoRangeLow = RANGE_INVALID;
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vUVInfoForecast Exit\tUVIndex: %d - %d\n",
			rEntry.u8UVInfoRangeLow, rEntry.u8UVInfoRangeHigh));
}

/*********************************************************************
 *FUNCTION:     vAirForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for Air
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vAirForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vAirForecast Entry"));
	AIR_OBJECT hAir;
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eAir(hForecast, &hAir);
	if (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
		/*Air object - Pollen level and Air quality*/
		AIR_POLLEN_LEVEL_ENUM pePollenLevel;
		enReturnCode = AIR.ePollenLevel(hAir,&pePollenLevel);
		rEntry.enPollenLevel = (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
				(fc_sxm_weather_enPollenLevel)pePollenLevel : fc_sxm_weather_enAIR_POLLEN_LEVEL_INVALID;

		AIR_QUALITY_ENUM peAirQuality;
		enReturnCode = AIR.eQuality(hAir,&peAirQuality);
		rEntry.enAirQuality = (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) ?
				peAirQuality : AIR_QUALITY_NO_DATA;
	}
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vAirForecast Exit\tPollen Level: %d\tAir quality: %d\n",rEntry.enPollenLevel, rEntry.enAirQuality));
}

/*********************************************************************
 *FUNCTION:     vCloudCoverForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for CloudCover
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vCloudCoverForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vCloudCoverForecast Entry"));
	FORECAST_CLOUD_COVER_ENUM eCloudCover;
	/* Retrieve Cloud cover */
	SMSAPI_RETURN_CODE_ENUM enReturnCode = FORECAST.eCloudCover(hForecast, &eCloudCover);
	rEntry.enCloudCover = (enReturnCode == SMSAPI_RETURN_CODE_SUCCESS)?
			eCloudCover : FORECAST_CLOUD_COVER_NO_DATA;
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vCloudCoverForecast Exit\tenCloudCover = %u", rEntry.enCloudCover));
}

/*********************************************************************
 *FUNCTION:     vTimeStampForecast
 *DESCRIPTION:  Extracts WeatherEvent from Forecast object and fills rEntry fields for Timestamp
 *PARAMETER:    1) Forecast object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vTimeStampForecast( FORECAST_OBJECT hForecast, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
	rEntry.u32TimeStamp = FORECAST.un32TimeStamp(hForecast);
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherDSRL::vTimeStampForecast Enter/Exit\tu32TimeStamp = %u", rEntry.u32TimeStamp));
}
/*********************************************************************
 *FUNCTION:     vLocationInfo
 *DESCRIPTION:  Extracts Location Info including Weather Station's name and LOCID
 *PARAMETER:    1) Location object handle; 2) REference to the structure that needs to be filled up
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherDSRL::vLocationInfo( LOCATION_OBJECT hLocation, fc_sxm_trTabweatherTypeListEntry& rEntry) const {
   LOCID_OBJECT hLocId = LOCATION.hLocID(hLocation);
   tChar acDescription[100] = { '\0' };

   rEntry.u32TabweatherLocId = LOCID.tID(hLocId);
   STRING_OBJECT hStateName = LOCATION.hState(hLocation);
   STRING_OBJECT hDescription = LOCATION.hDescription(hLocation);
   fc_sxm_vCopySmsString2Stl(hStateName,rEntry.oStateName);
   STRING.tCopyToCStr(hDescription, &acDescription[0], 100);
   fc_sxm_vCopySmsString2Stl(hDescription,rEntry.oWeatherStation);

   ETG_TRACE_USR4(("\nLOCID: %u", rEntry.u32TabweatherLocId));
   ETG_TRACE_USR4(("\nWeatherStation: %s", rEntry.oWeatherStation.c_str()));
   ETG_TRACE_USR4(("\nStateName: %s", rEntry.oStateName.c_str()));
}


/*=============================================================================
 =======                   END OF FILE                                   =======
 =============================================================================*/
