/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_stocks_list.h
* @brief       Stocks sort and filter list handler declaration.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_STOCKS_LIST_H_
#define _FC_SXM_TCL_STOCKS_LIST_H_

/******************************************
 * Header Files
 *******************************************/
#include "fc_sxm_tcl_data_list.hpp"


/******************************************
 * Enum values
 *******************************************/
typedef enum stock_price_direction_enum
{
	fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_INVALID = 0u,
	fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UP,
	fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UNCHANGED,
	fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_DOWN,
	fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UNKNOWN
} fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM;

/******************************************
 * Structure with data of a list-entry for HMI
 *******************************************/
struct fc_sxm_trStockQuoteListEntry:public fc_sxm_trBaseListEntry {
public:
	string strStockSymbol;
	OSAL_tMSecond msecReportTime;
	tF32 f32MarketPrice;
	tF32 f32Delta;
	mutable fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM enStockDirection;

	// Utility method added for printing values stored in fc_sxm_trStockQuoteListEntry
	tVoid vPrintStockListEntry ( tU8 u8EntriesCount, tU32 msecDuration) const ;

	// Default Constructor
	fc_sxm_trStockQuoteListEntry( DSRL_ENTRY_ID tEntryID = 0 ) : fc_sxm_trBaseListEntry(),
      strStockSymbol(""),
      msecReportTime (0u),
      f32MarketPrice(0.0f),
      f32Delta(0.0f),
      enStockDirection(fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_INVALID)
      {
		// Default constructor
		hLocation = LOCATION_INVALID_OBJECT;
		u32SxmId = tEntryID;
		u32DsrlEntryID = tEntryID;
      }
};

/******************************************
 * Sort-methods supported
 * sort-configuration consisting of sort-method and additional parameters
 * e.g. for which Stocks-type shall the price be sorted.
 *******************************************/
struct fc_sxm_trStocksListSortCfg {
	/* fc_sxm_tenStocksListSortMethod enSortMethod; */
	/* Check if configuration has changed.
	   to check if sorting has chanded, configuration has to provice operator!=
attention:
operator is used in the way that it only returns TRUE if
the changes are relevant for the current configuration.
e.g. if sorting by distance is not enabled, changing of the
location has no effect.
	 */
	bool operator!=(fc_sxm_trStocksListSortCfg const &b) const
	{
		(tVoid)b;
		// Overloaded operator "!="
		return FALSE;
	}

	fc_sxm_tenStocksDSRLSortMethod enSortMethod;
};

/******************************************
 * Sort-configuration consisting of filter-method and additional parameters
 * e.g. for which brand shall we filter
 *******************************************/

struct fc_sxm_trStocksListFilterCfg {
	/* to check if filter has chanded, configuration has to provice operator!=
	   see sort-configuration
	 */
	bool operator!=(fc_sxm_trStocksListFilterCfg const &b) const
	{
		(tVoid)b;
		// Overloaded operator "!="
		return FALSE;
	}

	fc_sxm_tenStocksDSRLFilterConfig enFilterCfg;
};

/******************************************
 * Struct containing all our configuration.
 * Must be dericed from fc_sxm_trBaseListConfig
 *******************************************/

struct fc_sxm_trStocksListConfig:public fc_sxm_trBaseListConfig {
	// typedef to make type of list-entries visible for templates
public:
	typedef fc_sxm_trStockQuoteListEntry trListEntry;
	// post-sort and post-filter to be applied after dsrl-iterate is done
	fc_sxm_trStocksListSortCfg rSortCfg;
	fc_sxm_trStocksListFilterCfg rFilterCfg;
	// see sort-config
	bool operator!=(fc_sxm_trStocksListConfig const &b) const
	{
		(tVoid)b;
		// Overloaded operator "!="
		return FALSE;
	}

	virtual tBool bUsesLocation() const
	{
		/*
		   overwrite method from base-class if the location is used in this configuration.
		   While not sorting or filtering for distance, the location is not used.
		   Only if location is used, on change of location actions will be taken by the framework
		 */
		return FALSE;
	}

};

/***********************************************************************
 * The Stocks-list, derive from template fc_sxm_tclDataListTyped.
 * Template-parameters are our list-class and our configuration.
 ************************************************************************/

class fc_sxm_trStocksList:public fc_sxm_tclDataListTyped<fc_sxm_trStocksList, fc_sxm_trStocksListConfig> {
 public:
		/*
constructor:
Simply pass parameters to base-class
		 */
		fc_sxm_trStocksList(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp):
        fc_sxm_tclDataListTyped<fc_sxm_trStocksList, fc_sxm_trStocksListConfig>(u32ListId, enListType, poDataApp, static_cast <tU16> (TR_CLASS_FC_SXM_STOCKS_APP))
		{}
		/* compare-fn that shall compare according to fc_sxm_trStocksListSortCfg
Attention:
Strict ordering has to be proviced: *prLeft<*prRight || *prRights<*prLeft
(not strict weak ordering as for stl-containers)
This is accomplished by comparing a unique id as fallback in case of *prLeft==*prRight
		 */

		/*
		   Overriding two methods bCompareItem and bFilterPassed and have hard coded the return values.
		 */

		virtual bool bCompareItem(fc_sxm_trStockQuoteListEntry const *prLeft, fc_sxm_trStockQuoteListEntry const *prRight) const;

		bool bCompareStockSymbol(fc_sxm_trStockQuoteListEntry const *prLeft, fc_sxm_trStockQuoteListEntry const *prRight) const;
		bool bCompareFavoriteOrder(fc_sxm_trStockQuoteListEntry const *prLeft, fc_sxm_trStockQuoteListEntry const *prRight) const;

		virtual bool bFilterPassed(fc_sxm_trStockQuoteListEntry const &rItem) const
		{
			// compare-fn that shall filter according to fc_sxm_trStocksListFilterCfg
			//default implementation for stocks
			(tVoid)rItem;
			return TRUE;
		}

		tBool bIsFavoriteStockSymbol (const string &strCompareSymbol) ;

 protected:
		/*
		   will be called by base-class when data are ready and the list shall be sent to hmi
		 */
		virtual tVoid vEmit(fc_sxm_trAdressing const &rAdressing);

 private:
		// Validates an Entry to find out if its corrupted or not
		bool bValidateEntry (fc_sxm_trStockQuoteListEntry const *poStockQuoteListEntry) const;
		// Check if cutoff time is reached for a particular stock
		bool bIsCutOffTimeReached( const OSAL_tMSecond &msecCurTime, const OSAL_tMSecond &msecEntryTime) const ;
};

#endif //_FC_SXM_TCL_STOCKS_LIST_H_

/*=============================================================================
  =======                   END OF FILE                                   =======
  =============================================================================*/

