/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_racing.cpp
* @brief       Sports motor racing class.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sports_app_interface.h"
#include "fc_sxm_tcl_sports_racing.h" 
#ifndef FC_SXM_SPORTS_UTEST
#include "fc_sxm_tcl_sports_utils.h"
#else
#include "sports/fc_sxm_tcl_sports_utils.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_NONTEAMSPORTS
#include "trcGenProj/Header/fc_sxm_tcl_sports_racing.cpp.trc.h"
#endif

#define SXM_SPORTS_SMSE_AUTO_RANK_INDEX 2

fc_sxm_tclMotorSport::fc_sxm_tclMotorSport()
{
	// Default constructor
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport CTOR"));
}
/*
 * Method to Extract Motor Sport Schedule Row Data to Method Result
 */
tBool fc_sxm_tclMotorSport::bExtractScheduleRow(tVoid* pHandle, SXMSportsRow &TableRow,tS32 s32TableParam,tU32 /*u32EpochValue*/, tBool /*bIsTeamBased*/)
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractScheduleRow Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)

	//Create local Object to hold parsed Table Data
	fc_sxm_trMotorRaceScheduleTable trMotorRaceScheduleTableData;

	//Extract the sxm_sports_begin Row Data to our internal object provided
	tBool bAvailable = poUtils->bExtractMotorRaceScheduleRow(TableRow, trMotorRaceScheduleTableData);
	if (bAvailable)
	{
		string strEventString = trMotorRaceScheduleTableData.strTrackName + trMotorRaceScheduleTableData.strRaceName;
		ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractAndFillScheduleRow Unique Event String = %s", strEventString.c_str()));

		// Generate Unique_ID for the input String
		trMotorRaceScheduleTableData.u32EventID = poUtils->u32ComputeHashCode(&strEventString);
		trMotorRaceScheduleTableData.bIsRankAvailable = TableRow.present & (1 << SXM_SPORTS_SMSE_AUTO_RANK_INDEX);
		ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractAndFillScheduleRow Unique Event ID = %u\tbIsRankAvailable =%u", trMotorRaceScheduleTableData.u32EventID, trMotorRaceScheduleTableData.bIsRankAvailable));

		// Data from sxm_sports_begin_nested
		if ((trMotorRaceScheduleTableData.bIsRankAvailable) &&
				(!bExtractNestedScheduleRow(pHandle,trMotorRaceScheduleTableData.u32RankListTableRef,trMotorRaceScheduleTableData)))
		{
			ETG_TRACE_ERR(("Error Parsing Motor Sports Nested Schedule Table Data..."));
			trMotorRaceScheduleTableData.bIsRankAvailable = FALSE;
		}
		// Other Data's
		trMotorRaceScheduleTableData.s32TableParam = s32TableParam;

		//Check for if this Event already available(To Avoid Duplicate entry)
		if(bIsEventIDAvailable(trMotorRaceScheduleTableData.u32EventID))
		{
			//If successfully parsed,  copy to the public vector based on the schedule day
			_setSortedMotorRaceSchedules.insert(trMotorRaceScheduleTableData);
		}
	}
	else
	{
		ETG_TRACE_ERR(("Error Parsing Motor Sports Schedule Table Data..."));
	}

	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractScheduleRow Exited.."));

	//return the status
	return bAvailable;
}

tBool fc_sxm_tclMotorSport::bExtractNestedScheduleRow(ptr pHandle, tU32 u32RankListTableRef,fc_sxm_trMotorRaceScheduleTable &trMotorRaceScheduleTableData) const
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractNestedScheduleRow Entered"));
	ptr ext;
	SXMSportsRow row;
	sxm_result_code enRetCode;
	tBool bAvailable = FALSE;

	if ((enRetCode = (sxm_result_code)sxm_sports_begin_nested(pHandle, SXM_SPORTS_TABLE, u32RankListTableRef, &ext, &row)) == SXM_E_OK)
	{
		//The sxm_sports_extract_row routine returns the data for the Leader/Winner event rank list as a row in the table SXM_SPORTS_TABLE.
		//Each league would be reported in one row.
		if ((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
		{
			//Macro : Traverse  the Event Rank List Bits present and Copy the information to Data
			if (row.present & (1 << (SXM_SPORTS_SMSE_AUTO_RANK_INDEX)))// Render the Ranking Table
			{
				//Get instance for Utils
				fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
				SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)
				bAvailable = poUtils->bExtractMotorRaceScheduleRankListRow(row, trMotorRaceScheduleTableData.trMotorRaceRankList);
			}
		}
		else
			ETG_TRACE_ERR(("Error Extracting  Motor Sports Nested Schedule Table Data..."));
		(tVoid) sxm_sports_end(ext);

	}
	else
	{
		ETG_TRACE_ERR(("sxm_sports_begin_nested failed at fc_sxm_tclNonTeamSports::bExtractAndFillScheduleRow\t enRetCode = %u",enRetCode));
	}

	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractNestedScheduleRow Exit"));
	return bAvailable;
}


/*
 * Method to Extract Golf Schedule Rank List Data to Method Result
 */
tBool fc_sxm_tclMotorSport::bExtractEventRankListRow(SXMSportsRow &TableRow)
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractAndFillScheduleRankListRow Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)

	//Create local Object to hold parsed Table Data
	fc_sxm_trMotorRaceRankList trMotorRaceScheduleRankListData;

	//Extract the SMS Row Data to our internal object provided
	tBool bAvailable = poUtils->bExtractMotorRaceScheduleRankListRow(TableRow,trMotorRaceScheduleRankListData);

	if(bAvailable)
	{
		//If successfully parsed,  copy the data to FI and Send through Method Result to HMI
		_vectorMotorRaceRankList.push_back(trMotorRaceScheduleRankListData);
	}
	else
	{
		ETG_TRACE_ERR(("Error Parsing Motor Race Schedule Table Data..."));
	}

	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::bExtractAndFillScheduleRow Exited.."));

	//return the status
	return bAvailable;
}

/*
 * Method to Clear previous Game List Data
 */
tVoid fc_sxm_tclMotorSport::vClearSchedulesList()
{
	_setSortedMotorRaceSchedules.clear();
}

/*
 * Method to Clear previous Rank List Data
 */
tVoid fc_sxm_tclMotorSport::vClearEventRankList()
{
	_vectorMotorRaceRankList.clear();
}


tU32 fc_sxm_tclMotorSport::u32GetEventListIndex(tU32 u32EventID)const
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::u32GetEventListIndex entry\t_setSortedMotorRaceSchedules size=%u", _setSortedMotorRaceSchedules.size()));

	tU32 u32ListIndex = 0u;

	tBool bfound = FALSE;

	for(set<fc_sxm_trMotorRaceScheduleTable,trCompareMotorRaceSchedules>::const_iterator cit = _setSortedMotorRaceSchedules.begin();
			cit != _setSortedMotorRaceSchedules.end(); ++cit,++u32ListIndex)
	{
		if (cit->u32EventID == u32EventID)
		{
			bfound = TRUE;
			break;
		}
	}

	if ( FALSE == bfound )
	{
		u32ListIndex = DEFAULT_GAME_EVENT_ID;
	}

	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::u32GetEventListIndex exited\tu32ListIndex size=%u", u32ListIndex));

	return u32ListIndex;
}

 /* Method to Fill Schedule Row Data to Method Result
 */
tVoid fc_sxm_tclMotorSport::vCopySchedulesToFI(midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult &oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::vCopySchedulesToFI Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	ETG_TRACE_USR4(("_setSortedMotorRaceSchedules size=%d",_setSortedMotorRaceSchedules.size()));
	tBool bDSTStatus = _poApp->bGetClockDST();
	string strTimeZone = _poApp->pGetTimeZone();

	for(set<fc_sxm_trMotorRaceScheduleTable,trCompareMotorRaceSchedules>::const_iterator cit = _setSortedMotorRaceSchedules.begin();
			cit != _setSortedMotorRaceSchedules.end(); ++cit)
	{
		midw_ext_fi_tcl_EventSchedule oEventScheduleInfo;

		//Copy actual Data recieved from SMS
	
		oEventScheduleInfo.EventState.enType = (midw_ext_fi_tcl_e8_Game_Status::tenType)cit->u32EventState;
		poUtils->fc_sxm_vString2Fi(cit->strRaceName.c_str(), oEventScheduleInfo.Tourney_Race);
		poUtils->fc_sxm_vString2Fi(cit->strTrackName.c_str(), oEventScheduleInfo.Course_Track);
		oEventScheduleInfo.Purse_Laps = cit->u32RaceLaps;

	     //GameStartTime Info
		tChar chDate[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };
		tChar chTime[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };
		
		if(cit->u32GameStartTime)
		{
			poUtils->vGetOSALDate(cit->u32GameStartTime, chDate, strTimeZone.c_str(), bDSTStatus);
			poUtils->vGetOSALTime(cit->u32GameStartTime, chTime, strTimeZone.c_str(), bDSTStatus);
		}
		
		poUtils->fc_sxm_vString2Fi(chDate,oEventScheduleInfo.Game_Time.Date);
	
		poUtils->fc_sxm_vString2Fi(chTime,oEventScheduleInfo.Game_Time.Time);

		// Nested Schedule Info
		poUtils->fc_sxm_vString2Fi(cit->trMotorRaceRankList.strCarNumber.c_str(), oEventScheduleInfo.CarNumber);
		poUtils->fc_sxm_vString2Fi(cit->trMotorRaceRankList.strCarMake.c_str(), oEventScheduleInfo.CarMake);
		oEventScheduleInfo.Yardage_Lap = cit->trMotorRaceRankList.tu32LeaderLaps;

		string strWinner = ("");
		// If the EventState is in Progress or Suspended or Abandon State , Send the Leader Name to HMI
		if((oEventScheduleInfo.EventState.enType == midw_ext_fi_tcl_e8_Game_Status::FI_EN_GAME_STATUS_IN_PROGRESS)
			||(oEventScheduleInfo.EventState.enType == midw_ext_fi_tcl_e8_Game_Status::FI_EN_GAME_STATUS_SUSPENDED)
			||(oEventScheduleInfo.EventState.enType == midw_ext_fi_tcl_e8_Game_Status::FI_EN_GAME_STATUS_ABANDONED))
		{
			ETG_TRACE_USR4(("Entered if condition \tDrivername = %s", cit->trMotorRaceRankList.strDriverName.c_str()));
			strWinner = cit->trMotorRaceRankList.strDriverName.c_str();
			poUtils->fc_sxm_vString2Fi(cit->trMotorRaceRankList.strCarMake.c_str(), oEventScheduleInfo.CarMake);
			poUtils->fc_sxm_vString2Fi(cit->trMotorRaceRankList.strCarNumber.c_str(), oEventScheduleInfo.CarNumber);
		}
		else
		{
			ETG_TRACE_USR4(("Entered else condition \tWinnername = %s", cit->strWinnerName.c_str()));
			strWinner = cit->strWinnerName.c_str();
		}
		poUtils->fc_sxm_vString2Fi(strWinner.c_str(), oEventScheduleInfo.Winner);
		//Channel Info
		midw_ext_fi_tcl_XMChannel oXMChannel;
		if(cit->u32NatChan != 0u)
		{
			oXMChannel.ChannelNumber = cit->u32NatChan;
			oXMChannel.ChannelType.enType = midw_ext_fi_tcl_XMChannelType::FI_EN_NATIONAL;
			oEventScheduleInfo.XMChannelList.push_back(oXMChannel);
		}

		// Other Info
	    oEventScheduleInfo.IsRankListAvl = cit->bIsRankAvailable;
	    oEventScheduleInfo.TableParam = cit->s32TableParam;
	    oEventScheduleInfo.TableRef = cit->u32RankListTableRef;
	    oEventScheduleInfo.Event_ID = cit->u32EventID;
	    //Prepare Method result with actual SMS data
		oMRes.ScheduledEvents.Eventlist.push_back(oEventScheduleInfo);
	}

	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::vCopySchedulesToFI exited %u", oMRes.ScheduledEvents.Eventlist.size()));

}


/*
 * Method to Clear previous Game List Data
 */
tVoid fc_sxm_tclMotorSport::vPrintSchedules(midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult &oMResSchedule) const
{
	//Print Data
	for(tU32 u32index=0;u32index<oMResSchedule.ScheduledEvents.Eventlist.size();u32index++)
	{
		//All details from the passed object are printed on TTFIs
		ETG_TRACE_USR4(("EventState =%d",oMResSchedule.ScheduledEvents.Eventlist[u32index].EventState.enType));
		ETG_TRACE_USR4(("Tourney_Race =%s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Tourney_Race.szValue));
		ETG_TRACE_USR4(("Course_Track=%s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Course_Track.szValue));
		ETG_TRACE_USR4(("Purse_Laps=%d",oMResSchedule.ScheduledEvents.Eventlist[u32index].Purse_Laps));
		ETG_TRACE_USR4(("Winner =%s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Winner.szValue));

		ETG_TRACE_USR4(("Game Date = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Game_Time.Date.szValue));
		ETG_TRACE_USR4(("Game Time = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Game_Time.Time.szValue ));


		ETG_TRACE_USR4(("Yardage_Lap = %u\tIsRankAvailable =%u\tEvent_ID = %u",oMResSchedule.ScheduledEvents.Eventlist[u32index].Yardage_Lap,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].IsRankListAvl,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].Event_ID));
		ETG_TRACE_USR4(("CarMake = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].CarMake.szValue ));

		ETG_TRACE_USR4(("Game TableParam = %d\tGame TableRef = %d",oMResSchedule.ScheduledEvents.Eventlist[u32index].TableParam,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].TableRef ));
		ETG_TRACE_USR4(("CarNumber  = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].CarNumber.szValue));
	}
}

/*
 * Method to Fill Schedule Row Data to Method Result
 */
tVoid fc_sxm_tclMotorSport::vCopyRankListToFI(midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult &oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::vCopyRankListToFI Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	for(vector<fc_sxm_trMotorRaceRankList>::const_iterator cit = _vectorMotorRaceRankList.begin();
			cit != _vectorMotorRaceRankList.end(); ++cit)
	{
		midw_ext_fi_tcl_EventRank oRankInfo;

		//Copy actual Data recieved from SMS
		poUtils->fc_sxm_vString2Fi(cit->strRank.c_str(), oRankInfo.Rank);
		oRankInfo.Laps = cit->tu32LeaderLaps;
		poUtils->fc_sxm_vString2Fi(cit->strDriverName.c_str(), oRankInfo.Name);
		poUtils->fc_sxm_vString2Fi(cit->strCarNumber.c_str(), oRankInfo.Number);
		poUtils->fc_sxm_vString2Fi(cit->strCarMake.c_str(), oRankInfo.Score_Make);

		//Prepare Method result with actual SMS data
		oMRes.EventInfo.RankList.push_back(oRankInfo);
	}

	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::vCopyRankListToFI exited %u", oMRes.EventInfo.RankList.size()));
}

/*
 * Method to Print Golf Schedules
 */
tVoid fc_sxm_tclMotorSport::vPrintRankList(midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult &oMResSchedule) const
{
	//Print Data
	for(tU32 u32index=0;u32index<oMResSchedule.EventInfo.RankList.size();u32index++)
	{
		//All details from the passed object are printed on TTFIs
		ETG_TRACE_USR4((" Make =%s",oMResSchedule.EventInfo.RankList[u32index].Score_Make.szValue));
		ETG_TRACE_USR4((" Name =%s",oMResSchedule.EventInfo.RankList[u32index].Name.szValue));
		ETG_TRACE_USR4((" Rank =%s",oMResSchedule.EventInfo.RankList[u32index].Rank.szValue));
		ETG_TRACE_USR4((" Laps =%d",oMResSchedule.EventInfo.RankList[u32index].Laps));
		ETG_TRACE_USR4((" Number =%s",oMResSchedule.EventInfo.RankList[u32index].Number.szValue));
	}
}

/*
 * Method returns the Rank List Index.
 */

tU8 fc_sxm_tclMotorSport::u8GetRankedListIndex() const
{
	ETG_TRACE_USR4(("fc_sxm_tclMotorSport::u8GetRankedListIndex Entered/Exited"));
	//Rank List Index for MotorSport is 2
	return SXM_SPORTS_SMSE_AUTO_RANK_INDEX;
}
/*
 * Method returns TRUE if the Unique Event ID provided is not part of the events list sent to HMI
 */
tBool fc_sxm_tclMotorSport::bIsEventIDAvailable(tU32 u32EventID) const
{
	tBool bInsertFlag = TRUE;
	for (set<fc_sxm_trMotorRaceScheduleTable,trCompareMotorRaceSchedules>::const_iterator cIt = _setSortedMotorRaceSchedules.begin(); cIt != _setSortedMotorRaceSchedules.end(); ++cIt)
	{
		if ((cIt->u32EventID) == u32EventID)
		{
			// Same game is identified in the existing list
			ETG_TRACE_USR4(("Identified a Duplicate Event ID = %u",u32EventID));
			bInsertFlag = FALSE;
			break;
		}
	}
	return bInsertFlag;
}
