/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_manager.cpp
* @brief       Sports application manager
* 			   responsible for managing different sport objects.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"
#include "fc_sxm_tcl_sports_manager.h"
#include "fc_sxm_tcl_sports_base.h"
#include "fc_sxm_tcl_sports_utils.h"
#include "fc_sxm_tcl_pers_sports_data.h"
#include "fc_sxm_tcl_sports_factory.h"
#include "fc_sxm_service_sxm_sports.h"
#include "fc_sxm_tcl_sxmapp_manager.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_sports_manager.cpp.trc.h"
#endif


class fc_sxm_tclSportsService;
/*
 * Constructor for the class
 */
fc_sxm_tclSportsManager::fc_sxm_tclSportsManager(): _bSportsListStatus(FALSE),
	_poSportsFactory(OSAL_NULL),
	_u8ActiveSportID(0xff),
	_u32ActiveRootLeagueID(0xffffffff)
{
	//Default constructor
	_mapSportsObjects.clear();
}

/*
 * Destructor for the class
 */
fc_sxm_tclSportsManager::~fc_sxm_tclSportsManager()
{
	//TODO: Destructor for the class
	vClearSportsObjects();
	//_poSportsFactory is already destructed in method vDeInitSportsManager()
	_poSportsFactory = OSAL_NULL;
}

/*
 * Method stores the list status of sports service. If
 * _bSportsListStatus = TRUE, it implies that the list of sports and the root affiliate list for every sport is available
 */
tVoid fc_sxm_tclSportsManager::vSetSportsListStatus(tBool bStatus)
{
	//Set status if list of sports and the root affiliate are available or not
	_bSportsListStatus = bStatus;
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSetSportsListStatus, new status = %u", _bSportsListStatus));
}

/*
 * Method gets the instance of Sports Object Factory
 */
tVoid fc_sxm_tclSportsManager::vInitSportsManager()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vInitSportsManager"));
	// Get instance of the sports factory
	_poSportsFactory = fc_sxm_tclSportsFactory::instance();
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsFactory);
}

/*
 * Method destroys the instance of Sports Object Factory
 */
tVoid fc_sxm_tclSportsManager::vDeInitSportsManager()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vDeInitSportsManager"));
	// Delete instance of sports factory
	FC_SXM_MEMORY_CLEAR(_poSportsFactory)
}
/*
 * Method to Route Msg to corresponding Sport
 */
tVoid fc_sxm_tclSportsManager::vRouteMsgToSport(fc_sxm_trMsgSportsSMSeLeagueCallback const *prMsg)
{
	//Find if the sportID to which the league data belongs
	SXM_IF_FIND_MAP(tU8, fc_sxm_tclISportsBase* , iter, _mapSportsObjects, prMsg->SMSeCallbackData.u8SportID)
	{
		//If the sport ID is available, delegate the league to the individual class
		iter->second->vHandleLeagueUpdates(prMsg);
	}
}

/*
 * Method requests Sports Object factory to create one object for every sport
 */
tVoid fc_sxm_tclSportsManager::vCreateSportsObject(fc_sxm_trSportsList& sportConfig)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vCreateSportsObjects() Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsFactory);
	//Map sportsID with the Sports Object class
	fc_sxm_tclISportsBase* poNewSport = _poSportsFactory->getSport(sportConfig.u8SportID,
			sportConfig.strSportName.c_str()) ;
	if (OSAL_NULL != poNewSport)
	{
		try
		{
			// Allow only permissible sport objects to be inserted to sports map
		    _mapSportsObjects[sportConfig.u8SportID] = poNewSport;
		}

		catch(...)
		{
			ETG_TRACE_ERR(("fc_sxm_tclSportsManager::Exception bad_alloc"));
			if (poNewSport != OSAL_NULL)
			{
				OSAL_DELETE poNewSport;
				poNewSport = OSAL_NULL;
			}

		}

	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vCreateSportsObjects() Exit"));
}

/*
 * Method Builds league list for the given sportID. Processing of league details is delegated to individual classes.
 */
tVoid fc_sxm_tclSportsManager::vBuildLeagueList(fc_sxm_trLeagueListData& league)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vBuildLeagueList() Entered"));
	//Find if the sportID to which the league is affiliated to, is already available.
	SXM_IF_FIND_MAP(tU8, fc_sxm_tclISportsBase* , iter, _mapSportsObjects, league.u8SportID)
	{
		//If the sport ID is available, delegate the league to the individual class
		iter->second->vBuildLeagueList(league);
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vBuildLeagueList() Exit"));
}

/*
 * Utillity Method to clear the list of all sports
 */
tVoid fc_sxm_tclSportsManager::vClearSportsObjects(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vClearSportsObjects() Entered\t sportsList size = %u", _mapSportsObjects.size()));
	//All data stored in
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{
		delete (cIT->second);
	}
	_mapSportsObjects.clear();
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vClearSportsObjects() Exit\t sportsList size = %u", _mapSportsObjects.size()));
}

/*
 * Utillity Method to print the list of all sports
 */
tVoid fc_sxm_tclSportsManager::vPrintSportsList() const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vPrintSportsList() Entered"));
	//Print all sportIDs and their respective sportNames one by one in IO console
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{
		cIT->second->vPrintSportsDetails();
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vPrintSportsList() Exit"));
}

/*
 * Utillity Method to print the list of all leagues affiliated to all sports
 */
tVoid fc_sxm_tclSportsManager::vPrintLeagueList() const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vPrintLeagueList() Entered"));
	//Iterate the sport List one by one
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{
		ETG_TRACE_USR4(("*********************************************************"));
		//Print all leagues affiliated to one sport
		cIT->second->vPrintLeagues();
		ETG_TRACE_USR4(("*********************************************************"));
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vPrintLeagueList() Exit"));
}

/*
 * Method sends FI methodResult for GetSportList
 */
tVoid fc_sxm_tclSportsManager::vSendSportsList(fc_sxm_trMsgSportsMStartGetSportList const *prMsg) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendSportsList() Entered"));
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	midw_ext_sxm_sportsfi_tclMsgGetSportListMethodResult oMRes;
	//Loop thru the sports list and fill the corresponding FI entries
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{
		midw_ext_fi_tcl_SportInfo oFiEntry;
		oFiEntry.SPORT_ID = cIT->second->u8GetSportID();
		poUtils->fc_sxm_vString2Fi(cIT->second->sGetSportName(),oFiEntry.SportName);
		oMRes.SportList.SportList.push_back(oFiEntry);
		ETG_TRACE_USR4(("Sport ID = %u\t Sport Name = %s",
				oFiEntry.SPORT_ID, cIT->second->sGetSportName()));
	}

	// Send FI message
	fc_sxm_tclSportsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendSportsList() Exit"));
}

/*
 * Method forwards FI msg for GetRootAffiliateList to the individual sports object to take action
 */
tVoid fc_sxm_tclSportsManager::vSendLeagueList(fc_sxm_trMsgSportsMStartGetRootAffiliateList const *prMsg) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendLeagueList() Entered"));
	midw_ext_sxm_sportsfi_tclMsgGetRootAffiliateListMethodStart const &oMStart=prMsg->oFiMsg;
	//Read the user selected SportID
	tU8 u8SportID = oMStart.SportID;
	ETG_TRACE_USR4(("Requested Sport = %u", u8SportID));

	//Find if the sport ID is part of the list
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, u8SportID)
	{
		//Delegate the action to be taken to individual sport objects
		cIT->second->vSendLeagueInfo(prMsg->rAdressing);
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendLeagueList() Exit"));
}

/*
 * Method forwards FI msg for GetAffiliateDetails to the individual sports object to take action
 */
tVoid fc_sxm_tclSportsManager::vSendAffiliateDetails(fc_sxm_trMsgSportsMStartGetAffiliateDetails const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendAffiliateDetails() Entered...Affiliate ID = %u\t Sport = %u",
			prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.SportID));
	//User selected Root Affiliate ID and Sports ID are contained in prMsg
	vRemoveActiveRequest(prMsg->oFiMsg.SportID, prMsg->oFiMsg.AFFILIATE_ID);
	//Find if the sport ID is part of the list
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, prMsg->oFiMsg.SportID)
	{
		//Let the Sport Object decide how the league request would be
		cIT->second->vGetLeagueTree(prMsg->oFiMsg.AFFILIATE_ID, prMsg->rAdressing);
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendAffiliateDetails() Exit"));
}

/*
 * Method forwards FI msg for GetAllTeams to the individual sports object to take action
 */
tVoid fc_sxm_tclSportsManager::vSendAllTeams(fc_sxm_trMsgSportsMStartGetAllTeams const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendAllTeams() Entered...Affiliate ID = %u\t Sport = %u",
				prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.SportID));
	//User selected Root Affiliate ID and Sports ID are contained in prMsg
	//Find if the sport ID is part of the list
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, prMsg->oFiMsg.SportID)
	{
		//Let the Sport Object decide how the league request would be
		cIT->second->vGetAllTeams(prMsg->oFiMsg.AFFILIATE_ID, prMsg->rAdressing);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendAllTeams() Exit"));
}
/*
 * Method forwards FI msg for GetAllTeams to the individual sports object to take action
 */
tVoid fc_sxm_tclSportsManager::vSendGameList(fc_sxm_trMsgSportsMStartGetGameList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendGameList() Entered...Affiliate ID = %u\t Sport = %u\t GameType =%u",
				prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.SportID,prMsg->oFiMsg.GameType.enType));

	//User selected Root Affiliate ID and Sports ID are contained in prMsg
	//Find if the sport ID is part of the list
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, prMsg->oFiMsg.SportID)
	{
		//cIT->second->vSetTeamID(prMsg->oFiMsg.TEAM_ID);
		if(prMsg->oFiMsg.TEAM_ID == FC_SXM_TEAM_INVALID_ID)
		{
			//Let the Sport Object decide how the league request would be
			cIT->second->vGetSchedules(prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.GameType.enType,prMsg->oFiMsg.Game_ID,prMsg->rAdressing);
		}
		else
		{
			ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendGameList FOR VALID TEAM USE CASE"));
			cIT->second->vIgnoreUpdates(TRUE);
			//Remove the all request
			vRemoveActiveRequest(prMsg->oFiMsg.SportID, prMsg->oFiMsg.AFFILIATE_ID);
			vSendLeagueRequest(prMsg->rAdressing);
			//Get Schedules for a Team
			cIT->second->vGetTeamSchedules(prMsg->oFiMsg.AFFILIATE_ID,prMsg->oFiMsg.TEAM_ID,prMsg->oFiMsg.GameType.enType,prMsg->oFiMsg.Game_ID,prMsg->rAdressing);
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendGameList() Exit"));
}

tU32 fc_sxm_tclSportsManager::u32LoadSportsList(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::u32LoadSportsList() Entered"));

	/* Get persisted list size */
	tU32 u32SportListSize = trSportsUserData::instance()->u32GetSportListSize();
	ETG_TRACE_USR4(("u32SportListSize - %d", u32SportListSize));
	tU32 u32Count=0;
	for(; u32Count<u32SportListSize; ++u32Count) {
		fc_sxm_trSportsList sportConfig; 
		/* Get sport id & sport name from sport list at every index */
		trSportsUserData::instance()->vGetSportList(u32Count, sportConfig.u8SportID, sportConfig.strSportName);
		/* Copy sport id and sport name in sports class */
		vCreateSportsObject(sportConfig);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::u32LoadSportsList() Exit\t u32Count = %u", u32Count));
	return u32Count;
}

tVoid fc_sxm_tclSportsManager::vLoadLeagueList(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vLoadLeagueList() Entered"));

	/* Get persisted list size */
	tU32 u32RootListSize = trSportsUserData::instance()->u32GetRootListSize();
	ETG_TRACE_USR4(("u32RootListSize - %d", u32RootListSize));
	for(tU32 u32Count=0; u32Count<u32RootListSize; ++u32Count) {
		fc_sxm_trLeagueListData league;
		/* Get sport id, season, affiliate ID & affiliate name from root list at every index */
		trSportsUserData::instance()->vGetRootList(u32Count, league.u8SportID, league.bInSeason, league.u32AfID, league.strAfName);
		league.strAflLName = league.strAfName;
		league.strAflSName = league.strAfName;
		/* Copy sport id, season, affiliate ID & affiliate name in sports class */
		vBuildLeagueList(league);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vLoadLeagueList() Exit"));
}

tVoid fc_sxm_tclSportsManager::vStoreSportsList(tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vStoreSportsList() Entered"));
	/* Clear persisted list data */
	trSportsUserData::instance()->vClearSportList(); 
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{	/* Copy each data into persistent list */
		string strSportName = cIT->second->sGetSportName();
		trSportsUserData::instance()->vSetSportList(cIT->second->u8GetSportID(), strSportName);
		ETG_TRACE_USR4(("Sport ID = %u\t Sport Name = %s", cIT->second->u8GetSportID(), cIT->second->sGetSportName()));
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vStoreSportsList() Exit"));
}

tVoid fc_sxm_tclSportsManager::vStoreLeagueList(tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vStoreLeagueList() Entered"));
	/* Clear persisted list data */
	trSportsUserData::instance()->vClearRootList();
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{	/* Store each league info */	
		cIT->second->vStoreLeagueInfo();
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vStoreLeagueList() Exit"));
}

tVoid fc_sxm_tclSportsManager::vCreateFavoriteLeagueList(tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vCreateFavoriteLeagueList() Entered"));

	/* Get persisted favorite list size */
	tU32 u32FavoriteListSize = trSportsUserData::instance()->u32GetFavoriteListSize();
	ETG_TRACE_USR4(("u32FavoriteListSize - %d", u32FavoriteListSize));
	for(tU32 u32Count=0; u32Count<u32FavoriteListSize; ++u32Count) {
		trFavoriteListElement FavoriteElement;
		trSportsUserData::instance()->vGetFavoriteList(u32Count, FavoriteElement);
		/* Search for current favorite sport from map */
		SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, FavoriteElement.u8SportID)
		{
			/* Request to create league list for current sport object */
			cIT->second->vCreateLeagueRequest(FavoriteElement.u32LeagueID);
		}
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vCreateFavoriteLeagueList() Exit"));
}


/*
*Method to forward the action to get the news for the selected affiliate and send those news details to HMI
*/
tVoid fc_sxm_tclSportsManager::vSendNewsDetails(fc_sxm_trMsgSportsMStartGetNewsDetails const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendNewsDetails() Entered"));
	//Read user selected Root Affiliate ID and Sports ID
	tU32 u32AffID = prMsg->oFiMsg.AFFILIATE_ID;
	tU8 u8SportID = prMsg->oFiMsg.SportID;
	//vector<string> vecNewsItem;
	//vector<string>::const_iterator cNewsIter;
	ETG_TRACE_USR4(("Requested Affiliate ID = %u\t Sport = %u", u32AffID, u8SportID));
	//Find if the sport ID is part of the list
	//csportIT cIT = _mapSportsObjects.find(u8SportID);
	//if ( cIT != _mapSportsObjects.end())
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, prMsg->oFiMsg.SportID)
	{
		//Let the Sport Object decide how the league request would be
		cIT->second->vGetNewsItems(u32AffID, prMsg->rAdressing); //17 has to be changed with define
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendNewsDetails() Exit"));
}


/*
 * Method forwards FI msg for GetScheduledEvents to the individual sports object to take action
 */
tVoid fc_sxm_tclSportsManager::vSendScheduledEvents(fc_sxm_trMsgSportsMStartGetScheduledEvents const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendScheduledEvents() Entered...Affiliate ID = %u\t Sport = %u\t GameType =%u",
				prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.SportID,prMsg->oFiMsg.EventType.enType));
	//User selected Root Affiliate ID and Sports ID are contained in prMsg
	//Find if the sport ID is part of the list
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, prMsg->oFiMsg.SportID)
	{
		//Let the Sport Object decide how the league request would be
		cIT->second->vGetSchedules(prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.EventType.enType,prMsg->oFiMsg.Event_ID,prMsg->rAdressing);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendScheduledEvents() Exit"));
}


/*
 * Method forwards FI msg for GeteventRank List to the individual sports object to take action
 */
tVoid fc_sxm_tclSportsManager::vSendEventRankList(fc_sxm_trMsgSportsMStartGetEventRankList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendEventRankList() Entered...Affiliate ID = %u\t Sport = %u\t EventType =%u\t TableParam =%d\tTableRef = %d",
				prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.SportID,prMsg->oFiMsg.EventType.enType,prMsg->oFiMsg.TableParam, prMsg->oFiMsg.TableRef));
	//User selected Root Affiliate ID and Sports ID are contained in prMsg
	//Find if the sport ID is part of the list
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, prMsg->oFiMsg.SportID)
	{
		//Let the Sport Object decide how the league request would be
		cIT->second->vGetEventRankList(prMsg->oFiMsg.AFFILIATE_ID, prMsg->oFiMsg.EventType.enType,prMsg->oFiMsg.TableParam, prMsg->oFiMsg.TableRef, prMsg->rAdressing);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendEventRankList() Exit"));
}

/* Method to set a Team as Favorite */
tBool fc_sxm_tclSportsManager::bSetFavoriteTeam(tU8 u8SportId,tU32 u32LeagueId,tU32 u32TeamId) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::bSetFavoriteTeam() Entered u8SportId=%d,u32LeagueId=%d,,u32TeamId=%d",u8SportId,u32LeagueId,u32TeamId));
	tBool bSuccess = FALSE;
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , mIter, _mapSportsObjects, u8SportId)
	{
		bSuccess = ((*mIter).second)->bSetFavoriteTeam(u32LeagueId,u32TeamId);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::bSetFavoriteTeam() Exit\tbSuccess = %u", bSuccess));
	return bSuccess;
}

/* Method to remove all League Request */
tVoid fc_sxm_tclSportsManager::vRemoveLeagueRequests (tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vRemoveLeagueRequests() Entered "));
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{
		cIT->second->vRemoveLeagueRequests();
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vRemoveLeagueRequests() Exit "));
}
/*
 * Method resets all league data from every sport object
 */
tVoid fc_sxm_tclSportsManager::vClearLeagueData (tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vClearLeagueData() Entered "));
	SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects)
	{
		// reset leagues for every sport object
		cIT->second->vResetLeagues();
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vClearLeagueData() Exit "));
}

/*
 * Return count of sports objects
 */
tU32 fc_sxm_tclSportsManager::u32GetSportsObjectCount(tVoid) const
{
	return (tU32)(_mapSportsObjects.size());
}
tVoid fc_sxm_tclSportsManager::vNotifyLeagueUpdates (tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vNotifyLeagueUpdates() Entered "));

	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, _u8ActiveSportID)
	{
		cIT->second->vNotifyLeagueUpdates();
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vNotifyLeagueUpdates() Exit "));
}
/*
 * Method finds the active request if any and removes it
 */
tVoid fc_sxm_tclSportsManager::vRemoveActiveRequest(tU8 u8SportID, tU32 u32LeagueID)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vRemoveActiveRequest() Entered u8SportID:%u,_u8ActiveSportID:%u, u32LeagueID:%u" , u8SportID,_u8ActiveSportID,u32LeagueID));
	
	if (u8SportID != _u8ActiveSportID)
	{
		ETG_TRACE_USR4(("SportID not equal"));
		//Cancel all requests from _u8ActiveSportID
		SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, _u8ActiveSportID)
		{
			cIT->second->vRemoveLeagueRequests();
		}
	}
	else
	{
		ETG_TRACE_USR4(("SportID equal"));
		SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, _u8ActiveSportID)
		{
			tU32 u32ReqRootLeague = cIT->second->u32GetRootAffiliate(u32LeagueID);
			if (_u32ActiveRootLeagueID != u32ReqRootLeague)
			{
				ETG_TRACE_USR4(("SportID equal ..... Root Affl is different"));
				//Cancel request from _u8ActiveSportID for Rootaffiliate _u32RootAffiliateID
				cIT->second->vRemoveLeagueRequests();
			}
			else
			{
				//Take no action
				ETG_TRACE_USR4(("SportID equal ..... Root Affl equal .... No action taken"));
			}
		}
	}
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, u8SportID)
	{
		_u8ActiveSportID = u8SportID;
		_u32ActiveRootLeagueID = cIT->second->u32GetRootAffiliate(u32LeagueID);
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vRemoveActiveRequest() Exit\t%u", _u32ActiveRootLeagueID));
}

/*
 * Method creates a new league request
 */
tVoid fc_sxm_tclSportsManager::vSendLeagueRequest(fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendLeagueRequest() Entered\tsportID = %u\t_u32ActiveRootLeagueID = %u", _u8ActiveSportID, _u32ActiveRootLeagueID));
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, _u8ActiveSportID)
	{
		cIT->second->vSetCBAddress(_u32ActiveRootLeagueID,rAdressing);
		cIT->second->vCreateLeagueRequest(_u32ActiveRootLeagueID);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendLeagueRequest() Exit"));
}

/*
 * Time change notification is made to the active sports object
 */
tVoid fc_sxm_tclSportsManager::vSendTimeChange() const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::vSendTimeChange() Entered\tsportID = %u", _u8ActiveSportID));
	SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclISportsBase* , cIT, _mapSportsObjects, _u8ActiveSportID)
	{
		cIT->second->vNotifySportDataUpdate(fc_sxm_tclSportsBase::u16GetActiveLeague(),fc_sxm_tclSportsBase::u8GetActiveScheduleType());
	}

}
