/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_factory.cpp
* @brief       Sports factory class.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sports_app.h"
#include "fc_sxm_tcl_sports_base_interface.h"
#include "fc_sxm_tcl_sports_factory.h"
#include "fc_sxm_tcl_sports_team.h"
#include "fc_sxm_tcl_sports_golf.h"
#include "fc_sxm_tcl_sports_racing.h"
#include "fc_sxm_tcl_sports_aggregate.h"
#include "fc_sxm_tcl_sports_common.h"
#include "fc_sxm_tcl_sports_config.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_sports_factory.cpp.trc.h"
#endif

namespace fc_sxm_sports_list_config {
	tU8 u8SupportedSports[] = {
#ifdef FC_SXM_ENABLE_SPORT_AGGREGATE
		(tU8) fc_sxm_tenSportID_Aggregate, //Enable Aggregate
#endif //FC_SXM_ENABLE_SPORT_AGGREGATE

#ifdef FC_SXM_ENABLE_SPORT_FOOTBALL
		(tU8) fc_sxm_tenSportID_Football, //Enable Football
#endif //FC_SXM_ENABLE_SPORT_FOOTBALL

#ifdef FC_SXM_ENABLE_SPORT_BASEBALL
		(tU8) fc_sxm_tenSportID_Baseball, //Enable Baseball
#endif //FC_SXM_ENABLE_SPORT_BASEBALL

#ifdef FC_SXM_ENABLE_SPORT_BASKETBALL
		(tU8) fc_sxm_tenSportID_Basketball, // Enable Basketball
#endif //FC_SXM_ENABLE_SPORT_BASKETBALL

#ifdef FC_SXM_ENABLE_SPORT_ICEHOCKEY
		(tU8) fc_sxm_tenSportID_IceHockey, // Enable IceHockey
#endif //FC_SXM_ENABLE_SPORT_BASKETBALL

#ifdef FC_SXM_ENABLE_SPORT_MOTORSPORT
		(tU8) fc_sxm_tenSportID_MotorSport, // Enable MotorSport
#endif //FC_SXM_ENABLE_SPORT_MOTORSPORT

#ifdef FC_SXM_ENABLE_SPORT_GOLF
		(tU8) fc_sxm_tenSportID_Golf, // Enable Golf
#endif //FC_SXM_ENABLE_SPORT_GOLF

#ifdef FC_SXM_ENABLE_SPORT_SOCCER
		(tU8) fc_sxm_tenSportID_Soccer, // Enable Soccer
#endif //FC_SXM_ENABLE_SPORT_SOCCER

#ifdef FC_SXM_ENABLE_SPORT_TENNIS
		(tU8) fc_sxm_tenSportID_Tennis // Enable Tennis
#endif //FC_SXM_ENABLE_SPORT_TENNIS
	};
} //end of namespace fc_sxm_sports_list_config
/*
 * Private constructor for Sports Factory
 */
fc_sxm_tclSportsFactory::fc_sxm_tclSportsFactory() :
		_setSupportedSports(fc_sxm_sports_list_config::u8SupportedSports,
		fc_sxm_sports_list_config::u8SupportedSports +
		sizeof(fc_sxm_sports_list_config::u8SupportedSports) / sizeof(tU8))
{
	//As the configured sports list is a const member, do a list initialization
	ETG_TRACE_USR4(("Private CTOR for sports factory"));
}

/*
 * Method makes a decision if a given sport can be allowed to be part of the sport list or not
 */
tBool fc_sxm_tclSportsFactory::bIsSportSupported(tU8 u8ID) const
{
	tBool bSupport = FALSE;
	//Allow the sport to be part of the sport list only if its ID is pre-configured
	SXM_IF_FIND_SET_CONST(tU8, cSetIt, _setSupportedSports, u8ID)
	{
		bSupport = TRUE;
	}
	return bSupport;
}

/*
 * Method implements the logic based on which the type of sport object would be created.
 */
fc_sxm_tclSportsFactory::fc_sxm_tenSportType fc_sxm_tclSportsFactory::enGetSportType(tU8 u8ID, tCString strName) const
{
	// TODO: revisit this method once the mapping between the sportID and sportName is finalised.
	ETG_TRACE_USR4(("fc_sxm_tclSportsFactory::enGetSportType() Entered"));
	fc_sxm_tenSportType enType = fc_sxm_enSportType_Other;
	// Currently doing a hardcoding of sport IDs with Sport Names
	switch(u8ID)
	{
		case fc_sxm_tenSportID_MotorSport:
		{
			enType = fc_sxm_enSportType_MotorSport;
			break;
		}
		case fc_sxm_tenSportID_Golf:
		{
			enType = fc_sxm_enSportType_Golf;
			break;
		}
		case fc_sxm_tenSportID_Aggregate:
		case fc_sxm_tenSportID_Tennis:
		{
			enType = fc_sxm_enSportType_Other;
			break;
		}

		default:
		{
			enType = fc_sxm_enSportType_TeamSports;
			break;
		}
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsManager::enGetSportType() Exit\t enType = %u\t sportName = %s", enType, strName));
	return enType;
}

/*
 * Method initialises the sport object created with sportID, sportName and AppObserver
 */
tVoid fc_sxm_tclSportsFactory::vInitSport(fc_sxm_tclISportsBase* poSportObject, tU8 u8ID, tCString strName) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsFactory::vInitSport Entered"));

	if (OSAL_NULL != poSportObject)
	{
		// Do object initilization only when memory allocation for the object is successful
		poSportObject->vInitSport(u8ID, strName);
		poSportObject->vSetObserver(fc_sxm_tclSportsApp::instance());
		ETG_TRACE_USR4(("fc_sxm_tclSportsFactory::getSport Success"));
	}
	else
		ETG_TRACE_ERR(("fc_sxm_tclSportsFactory::getSport Failure for sportID = %u\t sportName = %s", u8ID, strName));

	ETG_TRACE_USR4(("fc_sxm_tclSportsFactory::vInitSport Exit"));
}

/*
 * Method creates a new sport object based on SportID and SportName passed and assigns the newly created object
 * to poSportObject
 */
fc_sxm_tclISportsBase* fc_sxm_tclSportsFactory::poCreateSport( tU8 u8ID, tCString strName) const
{
	//Method returns a new sport object and implements the sports object factory
	fc_sxm_tenSportType enType = enGetSportType(u8ID, strName);
	if(fc_sxm_enSportType_TeamSports == enType)
		return OSAL_NEW fc_sxm_tclTeamSports();
	else if (fc_sxm_enSportType_Golf == enType)
		return OSAL_NEW fc_sxm_tclGolf();
	else if (fc_sxm_enSportType_MotorSport == enType)
		return OSAL_NEW fc_sxm_tclMotorSport();
	else if (fc_sxm_enSportType_Other == enType)
		return OSAL_NEW fc_sxm_tclAggregate();
	else
		return OSAL_NULL;
}

/*
 * Sports Object Factory implementation
 */
fc_sxm_tclISportsBase* fc_sxm_tclSportsFactory::getSport(tU8 u8ID, tCString strName) const
{
	// Return sports object based on the input configurations provided to the Factory
	// Currently the same object is created for all TeamSports such as Football, Baseball, Soccer
	ETG_TRACE_USR4(("fc_sxm_tclSportsFactory::getSport"));
	fc_sxm_tclISportsBase* poSportObject = OSAL_NULL;
	if (bIsSportSupported(u8ID))
	{
		try
		{
			poSportObject = poCreateSport( u8ID, strName);
			vInitSport(poSportObject, u8ID, strName);
		}
		catch(...)
		{
			ETG_TRACE_ERR(("fc_sxm_tclSportsFactory::getSport::exception"));
			if(poSportObject != OSAL_NULL)
			{
				OSAL_DELETE poSportObject;
				poSportObject = OSAL_NULL;
			}

		}

	}
	return poSportObject;
}
