/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_base.cpp
* @brief       Base class implementation for Sports Objects.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sports_app_interface.h"
#include "fc_sxm_tcl_sports_base.h"
#include "fc_sxm_smse.h"

#ifndef FC_SXM_SPORTS_UTEST
#include "fc_sxm_tcl_sports_utils.h"
#include "fc_sxm_tcl_pers_sports_data.h"
#else
#include "sports/fc_sxm_tcl_sports_utils.h"
#include "sports/fc_sxm_tcl_pers_sports_data.h"
#endif // end of FC_SXM_SPORTS_UTEST


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_BASE
#include "trcGenProj/Header/fc_sxm_tcl_sports_base.cpp.trc.h"
#endif

#define SPORTS_MAX_LEAGUEID_SIZE 5		//in bytes

tU32 fc_sxm_tclSportsBase::_u32ActiveLeague = 0u;
fc_sxm_trAdressing fc_sxm_tclSportsBase::_rMsgAdress ;
tU8 fc_sxm_tclSportsBase::_u8ScheduleType = SCHEDULE_INVALID_ID;

/*****************************************************************************
 * FUNCTION    : callback_request                                            *
 * CLASS       :                                							 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Callback Method called by SMSe when new updates are         *
 *               available for particular League						     *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : handle - Valid handle to the League Data			         *
 * PARAMETER   : type - Type of the data updated	                         *
 * PARAMETER   : tableParam - Table Param for Table Data                     *
 * PARAMETER   : userData - Pointer to user data	                         *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/

tVoid cb_vLeagueUpdateCallback(ptr handle, tInt type, tInt tableParam, ptr userData)
{

	ETG_TRACE_USR4(( "callback_request handle=%p,type=%u,tableParam=%u,userData=%p",
			handle,type,tableParam,userData ));

	tS8 s8TableType = SPORTS_TABLE_IX(tableParam);
	ETG_TRACE_USR4(( "Table type = %u", s8TableType));
	// Take action only for known tables and reject all unknown tables
	if ((s8TableType >= FC_SXM_SPORTS_SCHEDULE_SMSE_LAST) && (s8TableType <= FC_SXM_SPORTS_TABLE_SMSE_NEWS))
	{
		ETG_TRACE_USR4(( "Take action for a known table type"));
		//Type cast userData
		fc_sxm_trLeagueRequest* poLeagueRequest = static_cast <fc_sxm_trLeagueRequest *> (userData);
		SXM_ASSERT_RETURN(OSAL_NULL != poLeagueRequest)

		fc_sxm_tclSportsAppIFace* poApp = static_cast <fc_sxm_tclSportsAppIFace *> (poLeagueRequest->poApp);
		SXM_ASSERT_RETURN(OSAL_NULL != poApp)

		//Create New Message to send to our thread
		fc_sxm_trMsgSportsSMSeLeagueCallback leagueCallbackMsg;

		//Prepare the message with all details
		leagueCallbackMsg.SMSeCallbackData.u8SportID = poLeagueRequest->u8SportID;
		leagueCallbackMsg.SMSeCallbackData.u32LeagueID = poLeagueRequest->u32LeagueID;
		leagueCallbackMsg.SMSeCallbackData.poLeagueHandle = handle;
		leagueCallbackMsg.SMSeCallbackData.intUpdateType = type;
		leagueCallbackMsg.SMSeCallbackData.intTableParam = tableParam;
		//Post message to our thread i.e Sports App
		poApp->vPostInternalMsg(leagueCallbackMsg);
	}
}

/*****************************************************************************
 * FUNCTION    : Constructor                                                 *
 * CLASS       : fc_sxm_tclSportsBase                               		 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Automatically Called when Creating object for class type    *
 *               fc_sxm_tclSportsBase									     *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None								                         *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclSportsBase::fc_sxm_tclSportsBase() : _u8SPID(0u),
		_strSPName(""),
		_poApp(OSAL_NULL),
		_u32ActiveTeamID(0xffffffff),
		_u32FocussedGameID(0xffffffff),
		_bIgnoreUpdate(FALSE)

{
    ETG_TRACE_USR4(("fc_sxm_tcl_sports_base constructor"));
    _mapLeague.clear();
    _mapLeagueDetails.clear();
    _vectRequests.clear();
    _mapPropertyUpdate.clear();
}
/*
   desturctor
 */
fc_sxm_tclSportsBase::~fc_sxm_tclSportsBase()
{
	// Clear all user data
    ETG_TRACE_USR4(("fc_sxm_tcl_sports_base desturctor"));
    fc_sxm_tclSportsBase::vResetLeagues();
    _poApp = OSAL_NULL;
    _mapPropertyUpdate.clear();
}

/*
 * Method sets the sportID and sportName for the sportObject
 */
tVoid fc_sxm_tclSportsBase::vInitSport(tU8 u8ID, tCString strName)
{
	//Initialize the sportsObject
	_u8SPID = u8ID;
	_strSPName = strName;
}

/*
 * Method sets app object to be the observer for sport objects
 */
tVoid fc_sxm_tclSportsBase::vSetObserver(fc_sxm_tclSportsAppIFace* poSportObserver)
{
	_poApp = poSportObserver;
}

/*****************************************************************************
 * FUNCTION    : vGetLeagueTree                                        *
 * CLASS       : fc_sxm_tclSportsBase                               		 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Called when HMI enters particular League	/Sub League      *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : 1) u8LeagueID - League ID
 * 			     2) rAdressing - MethodResult needs to be sent to this address*
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclSportsBase::vGetLeagueTree(tU32 u32LeagueID, fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("vGetLeagueTree u32LeagueID = %u",u32LeagueID));
	// The method needs to take care of one of 3 options based on HMI input
	// Option 1) RootAffl selected, but LeagueTree yet to be constructed
	// Option 2) RootAffl selected, LeagueTree is already constructed
	// Option 3) Sub League selected, LeagueTree is expected to be available already

	fc_sxm_tclSportsBase::_u8ScheduleType = SCHEDULE_INVALID_ID;

	tBool bRootAffl = bIsRootAffiliate(u32LeagueID);
	fc_sxm_tenLeagueReqState enState = enGetRequestState(u32LeagueID);

	vSetCBAddress(u32LeagueID, rAdressing);
	if( ( TRUE == bRootAffl ) &&
			((fc_sxm_tenLeagueReqState_Initial == enState)||(fc_sxm_tenLeagueReqState_Closed == enState)))
	{
		//Make SMSe request only if the league tree for the selected league is not constructed yet
		vCreateLeagueRequest(u32LeagueID);
	}
	else if ((fc_sxm_tenLeagueReqState_Requested == enState)||(fc_sxm_tenLeagueReqState_ReIssued == enState))
	{
		//A request is already made but CB for league state UPDATED is yet to be received
		vSendLeagueTreeError(rAdressing);
	}
	else
	{
		//League Tree already available....
		//Just copy the correct parameters in MethodResult and send it back to HMI
		vSendLeagueTree(u32LeagueID);
	}
}

/*****************************************************************************
 * FUNCTION    : vSendLeagueTree                                        *
 * CLASS       : fc_sxm_tclSportsBase                               		 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method sends LeagueTree Details as CCA msg to HMI           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None
 * RETURNVALUE : None                                                        *
 *****************************************************************************/

tVoid fc_sxm_tclSportsBase::vSendLeagueTree(tU32 u32LeagueID) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSendLeagueTree Entered \tLeague ID = %u", u32LeagueID));
	//Method can be invoked from 2 places, 1) vGetLeagueTree() when the requested information is available already or
	// 2) From SMSe request CB where the HMI requested information has just been fetched

	//For invocation from SMSe request CB, do check if the LeagueID's match....
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
	tBool bTakeAction = TRUE;
	if( ( TRUE == bIsRootAffiliate(u32LeagueID) ) &&
			(u32LeagueID != fc_sxm_tclSportsBase::_u32ActiveLeague))
	{
		bTakeAction = FALSE;
	}

	if (TRUE == bTakeAction)
	{
		midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult oMRes;
		//First Fill Aff Common Data
		oMRes.AffiliateDetails.SPORT_ID = _u8SPID;
		oMRes.AffiliateDetails.AFFILIATE_ID = u32LeagueID;

		//Then Fill Aff Fixed List Data
		vFillAffFixedList(u32LeagueID,oMRes);

		//Then Fill Aff Dynamic Data i.e Sub League List
		fc_sxm_trLeagueListData trLeagueListData;
		vGetLeagueData(u32LeagueID,trLeagueListData);
		vFillSubLeagueList(trLeagueListData, oMRes);

		//Finally Send Method Result
		_poApp->vSendFiMsg(fc_sxm_tclSportsBase::_rMsgAdress, oMRes);
	}
	else
	{
		ETG_TRACE_ERR(("Active League is different.... Active League = %u", fc_sxm_tclSportsBase::_u32ActiveLeague));
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSendLeagueTree Exit"));
}
/*****************************************************************************
 * FUNCTION    : vFillFixedList                                        *
 * CLASS       : fc_sxm_tclSportsBase                               		 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION :Method To Fill fixed List of elements into Method Result
 * * ------------------------------------------------------------------------- *
 * PARAMETER   : LeagueID , Method Result
 * RETURNVALUE : None                                                        *
 *****************************************************************************/

tVoid fc_sxm_tclSportsBase::vFillAffFixedList(tU32 u32LeagueID,midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult &oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vFillAffFixedList Entered \tLeague ID = %u", u32LeagueID));

	for(tU8 u8Index=FC_SXM_SPORTS_HMI_AFF_FIXED_LIST_START;u8Index<FC_SXM_SPORTS_HMI_AFF_FIXED_LIST_END;u8Index++)
	{
		midw_ext_fi_tcl_AffiliateDetailInfo affl;

		affl.Category_ID.enType = (midw_ext_fi_tcl_AffiliateCategories::tenType)u8Index;

		affl.isDataAvailable.enType = (midw_ext_fi_tcl_e8CategoryAvailability::tenType)u8IsDataAvailable(u32LeagueID,u8Index);

		oMRes.AffiliateDetails.AffiliateDetails.push_back(affl);
	}

	vPrintAffFixedList(oMRes);

}
/*
 * Method to Check if Data is Available from SMS and return status
 */
tU8 fc_sxm_tclSportsBase::u8IsDataAvailable(tU32 u32LeagueID,tU8 u8DataType) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bIsDataAvailable Entered \tLeague ID = %u\t u8DataType=%u", u32LeagueID,u8DataType));

	//look up table - category type
	tU8 u8ArrCatType[FC_SXM_SPORTS_HMI_AFF_FIXED_LIST_END] = { 0, 1, 1, 0, 2};	//News-0,AllTeam-1,Recent-1,Today-0,Future-2

	tBool bRootAff = bIsRootAffiliate(u32LeagueID);

	tU8 u8DataAvail = FC_SXM_SPORTS_DATA_NOT_APPLICABLE;	//by Default Not Applicable

	switch(u8DataType)
	{
		case midw_ext_fi_tcl_AffiliateCategories::FI_EN_NEWS:
		{
			//News is Applicable Only For Root Affiliates
			if(bRootAff)
			{
				u8DataAvail = (tU8)bIsNewsAvailable(u32LeagueID);
			}
			break;
		}
		case midw_ext_fi_tcl_AffiliateCategories::FI_EN_ALLTEAMS:
		{
			//All Teams to be displayed only if it is available
			if(bRootAff && bIsAllTeamAvailable(u32LeagueID))
			{
				u8DataAvail = (tU8)TRUE;
			}
			break;
		}
		case midw_ext_fi_tcl_AffiliateCategories::FI_EN_RECENT:
		case midw_ext_fi_tcl_AffiliateCategories::FI_EN_TODAY:
		case midw_ext_fi_tcl_AffiliateCategories::FI_EN_FUTURE:
		{
			//Currently For Tennis No Schedules Supported
			if(_u8SPID != (tU8) fc_sxm_tenSportID_Tennis)
			{
				u8DataAvail = (tU8)bIsScheduleAvilable(u32LeagueID,u8ArrCatType[u8DataType]);
			}
			break;
		}
/*		case midw_ext_fi_tcl_AffiliateCategories::FI_EN_TOPTEAMS:
		{
			 TODO : To be Implemented for Top Teams
			if(bRootAff && bIsTeamSport())
			{

			}
			break;
		}*/
		default:
			break;

	}

	return u8DataAvail;
}
tVoid fc_sxm_tclSportsBase::vPrintAffFixedList(midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult &oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vPrintAffFixedList Entered "));

	//Print Data
	for(tU8 u8Index=FC_SXM_SPORTS_HMI_AFF_FIXED_LIST_START;u8Index<=FC_SXM_SPORTS_HMI_AFF_FIXED_LIST_END;u8Index++)
	{
		//All details from the passed object are printed on TTFIs
		ETG_TRACE_USR4((" Category_ID = %u",oMRes.AffiliateDetails.AffiliateDetails[u8Index].Category_ID.enType));
		ETG_TRACE_USR4((" isDataAvailable = %u",oMRes.AffiliateDetails.AffiliateDetails[u8Index].isDataAvailable.enType));
	}
}
/*****************************************************************************
 * FUNCTION    : vCreateLeagueRequest                                        *
 * CLASS       : fc_sxm_tclSportsBase                               		 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Fires a request to SMSe Library to fetch league tree        *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : 1) u8LeagueID - League ID
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclSportsBase::vCreateLeagueRequest(tU32 u32LeagueID)
{
	SXM_FOREACH_CONST(vector<fc_sxm_trLeagueRequest*>, cIT, _vectRequests)
	{
		if (u32LeagueID == (*cIT)->u32LeagueID)
		{
			ETG_TRACE_USR4(("vCreateLeagueRequest --- REQUEST ALREADY EXISTS"));
			return;
		}
	}
	//create new league request user data
	fc_sxm_trLeagueRequest* poLeagueRequest = OSAL_NEW fc_sxm_trLeagueRequest;
	SXM_ASSERT_RETURN(OSAL_NULL != poLeagueRequest)

	poLeagueRequest->u8SportID = _u8SPID;
	poLeagueRequest->u32LeagueID = u32LeagueID;
	poLeagueRequest->poApp = _poApp;
	
	// For solving custodial pointer lint error -START
	fc_sxm_trLeagueRequest* poDummy = poLeagueRequest;
	(tVoid) poDummy;
	//poDummy = poLeagueRequest;
	//For solving custodial pointer lint error-END	

	//Push the LeagueRequests to _vectRequests which can be used to remove Request or
	// to free memory alloted to struct fc_sxm_trLeagueRequest
	_vectRequests.push_back(poLeagueRequest);
	tChar chLeagueID[SPORTS_MAX_LEAGUEID_SIZE] = { '\0' };

	sxm_result_code enRetCode;

	//Prefix # before LeagueID
	OSAL_s32PrintFormat(chLeagueID,"%s%lu","#", (tULong)u32LeagueID);
	ETG_TRACE_USR4(("vCreateLeagueRequest chLeagueID = %s",chLeagueID));

	//Request SMSe to get League Data
	if ( (enRetCode = ( sxm_result_code) sxm_sports_request(chLeagueID, cb_vLeagueUpdateCallback, poLeagueRequest, &poLeagueRequest->leagueHandle) ) == SXM_E_OK)
	{
		ETG_TRACE_USR4(("League Request Fired Succesfully for %s", chLeagueID));
		fc_sxm_tenLeagueReqState enNewReqState = (fc_sxm_tenLeagueReqState_Closed == enGetRequestState(u32LeagueID)) ?
				fc_sxm_tenLeagueReqState_ReIssued : fc_sxm_tenLeagueReqState_Requested;
		//Update the league request status
		vSetLeagueRequestStatus(u32LeagueID, enNewReqState);
		ETG_TRACE_USR4(("League Request State for %u", enNewReqState));
	}
	else
	{
		vSendLeagueTreeError(fc_sxm_tclSportsBase::_rMsgAdress);
		ETG_TRACE_ERR(("sxm_sports_request failed\t ERROR CODE = %u", enRetCode));
	}
	ETG_TRACE_USR4(("vCreateLeagueRequest Exit... Current live requests for Sport(%u) = %u",
			ETG_CENUM(fc_sxm_tenSportID,_u8SPID), _vectRequests.size()));
}

/*****************************************************************************
 * FUNCTION    : vGetLeagueData                                        *
 * CLASS       : fc_sxm_tclSportsBase                               		 *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : function which gets League Data info for a specific League ID*
 * ------------------------------------------------------------------------- *
 * PARAMETER   : 1) u8LeagueID - League ID				                     *
 * 					2) trLeagueListData - League Data to be fetched			 *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclSportsBase::vGetLeagueData(tU32 u32LeagueID,fc_sxm_trLeagueListData& trLeagueListData) const
{
	ETG_TRACE_USR4(("vGetLeagueData u32LeagueID = %u",u32LeagueID));

	//Check if the given LeagueID belongs to any one of the Root Affiliate
	SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, _mapLeague)
	{
		//Check League Family
		SXM_IF_FIND_SET_CONST(tU32, cSetIt, cMapIt->second.setFamily, u32LeagueID)
		{
			//Find League
			if(TRUE == bFindLeague(cMapIt->second, u32LeagueID,trLeagueListData))
			{
				ETG_TRACE_USR4(("vGetLeagueData Found LeagueID = %u",trLeagueListData.u32AfID));

				break;
			}
		}
	}
}

/*
 * Method packs the sub league details into Method Result for GetAffiliateDetails
 */
tVoid fc_sxm_tclSportsBase::vFillSubLeagueList(fc_sxm_trLeagueListData const& affiliate, midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult &oMRes)  const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vFillSubLeagueList entered"));
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	//Iterate thru the list of sub leagues and fill the sub league Name and sub league ID into Method Result
	//Method Fills up data to be sent as Method Result by HMI for method Get Affiliate Details
	SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, affiliate.mapSubLeagues)
	{
		// Pack the league tree info to the method Result format
		midw_ext_fi_tcl_AffiliateDetailInfo affl;

		//Say it is for Sub Leagues
		affl.Category_ID.enType = midw_ext_fi_tcl_AffiliateCategories::FI_EN_SUBLEAGUES;
		//Copy Affiliate ID
		affl.SUB_LEAGUE_INFO.AFFILIATE_ID = cMapIt->second.u32AfID;
		//Copy Aff Name
		poUtils->fc_sxm_vString2Fi(cMapIt->second.strAflSName.c_str(),affl.SUB_LEAGUE_INFO.AffiliateName);
		//Copy Season Status
		affl.SUB_LEAGUE_INFO.InSeason = cMapIt->second.bInSeason;

		//Finally Add to Method Result
		oMRes.AffiliateDetails.AffiliateDetails.push_back(affl);

		//Print
		ETG_TRACE_USR4(("Sub League Name %s",affl.SUB_LEAGUE_INFO.AffiliateName.szValue));
		ETG_TRACE_USR4(("Sub League ID %d",affl.SUB_LEAGUE_INFO.AFFILIATE_ID));
		ETG_TRACE_USR4(("Sub League Season %d",affl.SUB_LEAGUE_INFO.InSeason));

	} //end for loop
}

/*
 * Method traverses the Sub Leagues, identifies the requested league and Fills in Method Result, the details
 * of the requested league
 */
tBool  fc_sxm_tclSportsBase::bFindLeague(fc_sxm_trLeagueListData const& affiliate, tU32 u32LeagueID, fc_sxm_trLeagueListData& trOutLeagueListData)  const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bFindLeague entered"));

	tBool bReturn = FALSE;
	if (u32LeagueID == affiliate.u32AfID)
	{
		ETG_TRACE_USR4(("bFindLeague u32LeagueID = %u",u32LeagueID));

		bReturn = TRUE;

		//Assign found affliliate data
		trOutLeagueListData = affiliate;

		//Fill up Method Result over here
		//vFillSubLeagueList(affiliate, oMRes);
	}
	else
	{
		//Check if the given LeagueID belongs to any one of the Root Affiliate
		SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, affiliate.mapSubLeagues)
		{
			if (u32LeagueID == cMapIt->second.u32AfID)
			{
				bReturn = bFindLeague(cMapIt->second, u32LeagueID, trOutLeagueListData);
			}
			else
			{
				//Check if the requested league belongs to the affiliate family
				SXM_IF_FIND_SET_CONST(tU32, cSetIt, cMapIt->second.setFamily, u32LeagueID)
				{
					bReturn = bFindLeague(cMapIt->second, u32LeagueID, trOutLeagueListData);
				} //end of if loop (cSet != cMapIt->second.setFamily.end())
			}
			if (TRUE == bReturn)
				break;
		} //end of for loop
	}
	return bReturn;
}

/*
 * Method resets all RootAffiliates and their sub-affiliates from the soprts object
 */
tVoid fc_sxm_tclSportsBase::vResetLeagues()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vResetLeagues Entered"));
	// Reset all cached data
	fc_sxm_tclSportsBase::vRemoveLeagueRequests();
	_mapLeague.clear();
	_mapLeagueDetails.clear();
	_vectRequests.clear();
	_u32ActiveTeamID = 0xffffffff;
	_u32FocussedGameID = 0xffffffff;
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vResetLeagues Exit"));
}

/*
 * Method removes all the pending requests made to SMSe for Table updates and also
 * frees the memory alloted to all _vectRequests Entries
 */
tVoid fc_sxm_tclSportsBase::vRemoveLeagueRequests(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vRemoveLeagueRequests Entered"));
	//Method to Cancel all live requests and clear memory
	vector< fc_sxm_trLeagueRequest* >::reverse_iterator rIt = _vectRequests.rbegin();
	for (; rIt != _vectRequests.rend(); ++rIt)
	{
		sxm_result_code enRetCode;
		fc_sxm_trLeagueRequest* leagueRequest = *rIt;
		if ((enRetCode = (sxm_result_code) sxm_sports_remove(leagueRequest->leagueHandle)) == SXM_E_OK)
		{
			vSetLeagueRequestStatus(leagueRequest->u32LeagueID, fc_sxm_tenLeagueReqState_Closed);
			leagueRequest->poApp = OSAL_NULL;
			// The request is not active anymore... Delete memory alloted to the individual structure
			FC_SXM_MEMORY_CLEAR(leagueRequest)
		}
		else
		{
			ETG_TRACE_ERR(("sxm_sports_remove FAILURE\terror code = %u", enRetCode));
		}
	} //end of for loop
	//Clear the list so that it contains no memory
	_vectRequests.clear();
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vRemoveLeagueRequests Exit"));
}

/*
 * Utility method to print the list of all sports along with this pointer address as Trace msg in Console IO
 */
tVoid fc_sxm_tclSportsBase::vPrintSportsDetails() const
{
	//TODO: remove method once HMI flow is in place
	ETG_TRACE_USR4(("vPrintSportsDetails\t Addr = %x\t sportID = %u\t sportName = %s", this, _u8SPID, _strSPName.c_str()));
}

/*
 * Utility method to print the list of all leagues affiliated to the sports object as Trace msg in Console IO
 */
tVoid fc_sxm_tclSportsBase::vPrintLeagues() const
{
	//TODO: remove method once HMI flow is in place
	ETG_TRACE_USR4(("vPrintLeagues"));
	map<tU32, struct fc_sxm_trLeagueListData>::const_iterator citer = _mapLeague.begin();
	ETG_TRACE_USR4(("vPrintSportsDetails\t Addr = %x\t sportID = %u\t sportName = %s", this, _u8SPID, _strSPName.c_str()));
	for ( tU8 u8Count = 0; citer != _mapLeague.end(); ++citer, ++u8Count)
	{
		//Print the values stored in structure fc_sxm_trLeagueListData
		struct fc_sxm_trLeagueListData league = citer->second;
		ETG_TRACE_USR4(("%u) League ---> u8SportID = %u\t u32AfID = %u\t u32GDRef = %u\t _bInSeason = %u",
				u8Count, league.u8SportID, league.u32AfID, league.u32GDRef, league.bInSeason));
		ETG_TRACE_USR4(("League --->  strSportName = %s", league.strSportName.c_str()));
		ETG_TRACE_USR4(("League --->  strAfName = %s", league.strAfName.c_str()));
		ETG_TRACE_USR4(("League --->  _strAflLName = %s", league.strAflLName.c_str()));
		ETG_TRACE_USR4(("League --->  _strAflSName = %s", league.strAflSName.c_str()));
	}
}

/*
 * Add a league to the current league list.
 * If the league is already part of the list, then overwrite it with the new data
 */
tVoid fc_sxm_tclSportsBase::vBuildLeagueList(fc_sxm_trLeagueListData &league)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vBuildLeagueList Entered"));
	//If the given root affiliate id is already present, overwrite it with the new value.
	//If not present, create a new league entry
	_mapLeague[league.u32AfID] = league;
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vBuildLeagueList Exit"));
}

/*
 * Method returns sport ID associated for the sport object
 */
tU8 fc_sxm_tclSportsBase::u8GetSportID() const
{
	// return the sportID associated for the sport object
	return _u8SPID;
}

/*
 * Method returns sport Name associated for the sport object
 */
tCString fc_sxm_tclSportsBase::sGetSportName() const
{
	// return the sportName associated for the sport object
	return _strSPName.c_str();
}

/*
 * Method stores the LeagueData received as CallBack from SMSe. LeagueData will contain
 * the table data of the selected league
 */
tVoid fc_sxm_tclSportsBase::vHandleLeagueUpdates(fc_sxm_trMsgSportsSMSeLeagueCallback const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vHandleLeagueUpdates Entered\tSportID = %u\tLegueID = %u\ttype = %d\ttableParam = %d",
			_u8SPID, prMsg->SMSeCallbackData.u32LeagueID, prMsg->SMSeCallbackData.intUpdateType, prMsg->SMSeCallbackData.intTableParam));

	//Ensure no corruption of data has happened
	SXM_ASSERT_RETURN(OSAL_NULL != prMsg)

	fc_sxm_tenLeagueReqState enState = enGetRequestState(prMsg->SMSeCallbackData.u32LeagueID);
	//This is to avoid queuing
	//First check whether the league still active
	if(SXM_SPORTS_TABLE == prMsg->SMSeCallbackData.intUpdateType &&
			fc_sxm_tenLeagueReqState_Closed == enState)
	{
		//Then just consume this message and return immediately
		return;
	}

	//Check if the league exists
	fc_sxm_trLeagueData leagueData;
	SXM_IF_FIND_MAP(tU32, fc_sxm_trLeagueData, iter, _mapLeagueDetails, prMsg->SMSeCallbackData.u32LeagueID)
	{
		//If found,copy existing data
		leagueData = iter->second;
		ETG_TRACE_USR4(("callback_request: LeagueID bExists!"));
	}

   	leagueData.poLeagueHandle = prMsg->SMSeCallbackData.poLeagueHandle ;

	//Check Which type of League Data updated
    switch(prMsg->SMSeCallbackData.intUpdateType)
    {
    	//Check if League Data is ready to fetch
		case SXM_SPORTS_UPDATED:
		{
			if(fc_sxm_tenLeagueReqState_ReIssued != enState)
			{
				//Update the league request status
				leagueData.u32LeagueStatus = SXM_SPORTS_UPDATED;
				_mapLeagueDetails[prMsg->SMSeCallbackData.u32LeagueID] = leagueData;
				ETG_TRACE_USR3(("_mapLeagueDetails.count() = %u", _mapLeagueDetails.size()));

				vResetLeague(prMsg->SMSeCallbackData.u32LeagueID);
				vBuildLeagueFamily(prMsg->SMSeCallbackData.u32LeagueID);
				vAddTeams(prMsg->SMSeCallbackData.u32LeagueID);
			}
			vSetLeagueRequestStatus(prMsg->SMSeCallbackData.u32LeagueID, fc_sxm_tenLeagueReqState_Updated);
			if(!bGetIgnoreUpdates())
			{
				//To get League Update
				vSendLeagueTree(prMsg->SMSeCallbackData.u32LeagueID);
			}
			else if(_u32ActiveTeamID != FC_SXM_TEAM_INVALID_ID)
			//else
			{
				ETG_TRACE_USR4(("Update callback_request: vGetTeamSchedules"));
				//Set Ignore update as False
				vIgnoreUpdates(FALSE);
				//To get Favorite Team update
				vGetTeamSchedules(fc_sxm_tclSportsBase::_u32ActiveLeague,_u32ActiveTeamID, fc_sxm_tclSportsBase::_u8ScheduleType,_u32FocussedGameID, fc_sxm_tclSportsBase::_rMsgAdress);
			}
			break;
		}
		//Check if League Table is updated
		case SXM_SPORTS_TABLE:
		{
			SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
			ETG_TRACE_USR3(("vHandleLeagueUpdates: Update for Table data Happened League ID : %u",prMsg->SMSeCallbackData.u32LeagueID));
			leagueData.setLeagueTableParam.insert(prMsg->SMSeCallbackData.intTableParam);
			_mapLeagueDetails[prMsg->SMSeCallbackData.u32LeagueID] = leagueData;
			//Get League ID received for a Schedule Table
			tU32 u32TableLeague = SPORTS_TABLE_AF(prMsg->SMSeCallbackData.intTableParam);
			//Get for the day we received from SMSe

			/*fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
		    SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)*/

			tU8 u8ScheduleDay = _poApp->u8GetDayMapIndex(prMsg->SMSeCallbackData.intTableParam);
			vNotifySportDataUpdate(u32TableLeague,u8GetUpdateType(u8ScheduleDay));
			break;
		}
		default:
			break;
    }

    //finally, update map with new league details
    //_mapLeagueDetails[prMsg->SMSeCallbackData.u32LeagueID] = leagueData;
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vHandleLeagueUpdates Exit"));
}

/*
 * Method that sends back the FI msg methodResult for GetRootAffiliateList.
 * For the associated sports object, all root affiliated belonging to that object are sent back
 */
tVoid fc_sxm_tclSportsBase::vSendLeagueInfo(fc_sxm_trAdressing const &rAdressing) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSendLeagueInfo(fc_sxm_trMsgSportsMStartGetRootAffiliateList) Entered for sport = %s", _strSPName.c_str()));
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

	midw_ext_sxm_sportsfi_tclMsgGetRootAffiliateListMethodResult oMRes;
	oMRes.RootAffiliateList.SPORT_ID = _u8SPID;
	tU8 u8Count = 0u;
	//Iterate thru all available leagues one by one and fill up the FIentry
	SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, _mapLeague)
	{
		//Send only the root affiliate list. The sub leagues should be ignored
		if ( 0u != cMapIt->second.u8Level )
			continue;

		midw_ext_fi_tcl_AffiliateInfo oFiEntry;
		oFiEntry.AFFILIATE_ID = cMapIt->second.u32AfID;
		//TODO: check if encoding of char* to tString is still required
		poUtils->fc_sxm_vString2Fi(((cMapIt->second.strAflSName).c_str()),oFiEntry.AffiliateName);
		oFiEntry.InSeason = cMapIt->second.bInSeason;
		oMRes.RootAffiliateList.RootAffiliateList.push_back(oFiEntry);
		++u8Count;

		ETG_TRACE_USR4(("fc_sxm_tclSportsBase\t %u) League ID = %u\tleagueName = %s",
				u8Count, oFiEntry.AFFILIATE_ID, (cMapIt->second.strAflSName).c_str()));
	}
	ETG_TRACE_USR4(("For SportID = %u -----> No of Root Affiliates = %u", _u8SPID, oMRes.RootAffiliateList.RootAffiliateList.size()));

	// Send FI message
	_poApp->vSendFiMsg(rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSendLeagueInfo(fc_sxm_trMsgSportsMStartGetRootAffiliateList) Exit"));
}

tVoid fc_sxm_tclSportsBase::vStoreLeagueInfo(tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vStoreLeagueInfo() Entered"));
	/* Get each sport details */
	SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, _mapLeague)
	{	/* Get data of root affiliate list only */
		if(0u == cMapIt->second.u8Level) {
			/* Copy each data into persistent list */
			trSportsUserData::instance()->vSetRootList(_u8SPID,
													   cMapIt->second.bInSeason, 
													   cMapIt->second.u32AfID, 
													   cMapIt->second.strAfName);
		}
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vStoreLeagueInfo() Exit"));
}

/*
 * Method fills the list of teams available within a league
 */
tVoid fc_sxm_tclSportsBase::vAddTeams(tU32 u32LeagueID)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vAddTeams Entered"));

	mapLeagueIter requestIt = _mapLeagueDetails.find(u32LeagueID);
	map<tU32, fc_sxm_trLeagueListData>::iterator listIt = _mapLeague.find(u32LeagueID);
	if ((requestIt != _mapLeagueDetails.end()) &&
			(listIt != _mapLeague.end()))
	{
		ptr ext = OSAL_NULL;
		sxm_result_code enRetCode;
		SXMSportsRow row, header;

		// Begin extracting the sports tree
		if ((enRetCode = (sxm_result_code) sxm_sports_begin(requestIt->second.poLeagueHandle, SXM_SPORTS_TEAMS, 0, &ext, &header)) == SXM_E_OK)
		{
			fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
			SXM_ASSERT_RETURN( OSAL_NULL!= poUtils )

			// Iterate until all rows are read.
			while((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
			{
				//The sxm_sports_extract_row routine returns the data for leagues from all sports as a row in the table SXM_SPORTS_LEAGUES.
				//Each league would be reported in one row.
				fc_sxm_trTeamInfo newTeam;
				if (TRUE == poUtils->bExtractTeamRow(row, newTeam))
				{
					listIt->second.mapTeams[newTeam.u32TeamID] = newTeam;
				} // end of loop "if (TRUE == poUtils->bExtractTeamRow(row, newTeam))"
			} //end of while loop
			//Free the handle used to read Row contents in sxm_sports_begin call
			enRetCode = (sxm_result_code) sxm_sports_end(ext);
			ETG_TRACE_USR4(("vAddTeams enRetCode = %u", enRetCode));
		} //end if loop sxm_sports_begin
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vAddTeams Exit"));
}

/*
 * Method stores the LeagueID to the corresponding address to which Method result has to be sent
 * As the LeagueRequest works asynchronously, store the active league selection made in HMI and the
 * address to send MethodResult
 */
tVoid fc_sxm_tclSportsBase::vSetCBAddress(tU32 u32LeagueID, fc_sxm_trAdressing const &rAddr) const
{
	//Store the active league and the address to which method result has to be redirected
	fc_sxm_tclSportsBase::_u32ActiveLeague = u32LeagueID;
	fc_sxm_tclSportsBase::_rMsgAdress = rAddr;
}
/*
 * Method returns the if the selected league has teams affiliated to it
 */
tBool fc_sxm_tclSportsBase::bIsAllTeamAvailable(tU32 /*u32LeagueID*/) const
{
	// For non-team sports, there are no team information is possible. Always return FALSE when
	// enquiry is made for team information
	// For team sports the method is over-ridden
	return FALSE;
}

/*
 * Method adds the AllTeams information provided by SMSe to methodResult for GetAllTeams
 */
tVoid fc_sxm_tclSportsBase::vGetAllTeams(tU32 u32LeagueID, fc_sxm_trAdressing const &rAdressing) const
{
	//Warning: The code flow should not have reached here. The user must not have been allowed to select
	//vGetAllTeams for NonTeamSports category

	//For team sports this method is over-ridden
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetAllTeams Entered"));

	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
	midw_ext_sxm_sportsfi_tclMsgGetAllTeamsMethodResult oMRes;
	oMRes.AllTeamList.SPORT_ID = _u8SPID;
	oMRes.AllTeamList.AFFILIATE_ID = u32LeagueID;
	oMRes.AllTeamList.TeamDetails.clear();

	//For Non-team sports.... the team details information must not be available
	//Send FI msg for GetAllTeams
	_poApp->vSendFiMsg(rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetAllTeams Exit"));
}
/*
 * Method to check if news item is available for the selected league
 */
tBool fc_sxm_tclSportsBase::bIsNewsAvailable(tU32 u32LeagueID) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bIsNewsAvailable Entered"));
	tBool bAvailable = FALSE;
	// Check if the League entered from HMI is already part of request made from
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueData, cMapIter, _mapLeagueDetails, u32LeagueID)
	{
		//Iterate thru the list of table params to find if atleast one table contains News data
		SXM_FOREACH_CONST( set<tS32>, cSetIt, cMapIter->second.setLeagueTableParam)
		{
			//If the value returned by macro SPORTS_TABLE_IX for the given table param is 17,
			// then the table contains News data
			if ((FC_SXM_SPORTS_TABLE_SMSE_NEWS == SPORTS_TABLE_IX(*cSetIt)) &&
					( bValidateTable(cMapIter->second.poLeagueHandle, *cSetIt)) )
			{
				bAvailable = TRUE;
				break;
			}
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bIsNewsAvailable Exit\tbAvailable = %u", bAvailable));
	return bAvailable;
}

/*
 * Method to get news details for the selected affiliate/league
 */
tVoid fc_sxm_tclSportsBase::vGetNewsItems(tU32 u32LeagueID, fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetNewsItems Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
	fc_sxm_tclSportsBase::_u8ScheduleType = FC_SXM_SPORTS_NEWS;
	midw_ext_sxm_sportsfi_tclMsgGetNewsDetailsMethodResult oMRes;
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueData, cMapIter, _mapLeagueDetails, u32LeagueID)
	{
		ptr ext = OSAL_NULL;
		SXMSportsRow row, header;
		sxm_result_code enRetCode;

		SXM_FOREACH_CONST( set<tS32>, cSetIt, cMapIter->second.setLeagueTableParam)
		{
			if (FC_SXM_SPORTS_TABLE_SMSE_NEWS == SPORTS_TABLE_IX(*cSetIt))
			{
				ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetNewsItems inside league"));
				//If there are no news for the selected league, SMSe return code would be SXM_E_NOENT
				//If there is an error encountered while extracting row, SMSe return code would be SXM_E_ERROR
				if ((enRetCode = (sxm_result_code) sxm_sports_begin(cMapIter->second.poLeagueHandle, SXM_SPORTS_TABLE, (*cSetIt), &ext, &header)) == SXM_E_OK)
				{
					fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
					SXM_ASSERT_RETURN( OSAL_NULL!= poUtils )
					ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetNewsItems sxm_sports_begin"));

					// Iterate until all rows are read.
					while((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
					{
						//The sxm_sports_extract_row routine returns the news for the selected league as a row in the table SXM_SPORTS_TABLE.
						//Each news item would be reported in one row.
						string strNews;
						if (TRUE == poUtils->bExtractNewsRow(row, strNews))
						{
							midw_ext_fi_tcl_NewsInfo oFiEntry;
							poUtils->fc_sxm_vString2Fi((strNews.c_str()),oFiEntry.NewsInDetail);
							oMRes.News.NewsList.push_back(oFiEntry);
						} // end of if loop
					} //end of while loop
					//Free the handle used to read Row contents in sxm_sports_begin call
					(tVoid) sxm_sports_end(ext);
				}
				else
				{
					ETG_TRACE_ERR(("fc_sxm_tclSportsBase::vGetNewsItems\tErrorCode = %u", enRetCode));
				}
			} // if (u8TIX == 17)
		} // SXM_FOREACH_CONST
	} // SXM_IF_FIND_MAP_CONST

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetNewsItems \t oMRes.News.NewsList.size = %u", oMRes.News.NewsList.size()));
	// Send FI message
	_poApp->vSendFiMsg(rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vGetNewsItems Exit"));
}

/*
 * Method Resets the League Tree for the given Root Affiliate
 */
tVoid fc_sxm_tclSportsBase::vResetLeague(tU32 u32LeagueID)
{
	// Identify the Root Affiliate and reset data of all its child leagues
	SXM_IF_FIND_MAP(tU32, fc_sxm_trLeagueListData, iter, _mapLeague, u32LeagueID)
	{
		iter->second.vResetLeagueTree();
	}
}

/*
 * Method Builds the Family Tree from SMSe CB for the given LeagueID
 */
tVoid fc_sxm_tclSportsBase::vBuildLeagueFamily(tU32 u32LeagueID)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vBuildLeagueFamily Entered"));
	// Get iterator to the League Map and the SMSe pending requests map
	map<tU32, fc_sxm_trLeagueListData>::iterator listDataIter = _mapLeague.find(u32LeagueID);
	map<tU32, fc_sxm_trLeagueData>::const_iterator cRequestIt = _mapLeagueDetails.find(u32LeagueID);

	if ((cRequestIt != _mapLeagueDetails.end())
			&& (cRequestIt->second.u32LeagueStatus == SXM_SPORTS_UPDATED)
			&& (listDataIter != _mapLeague.end()))
	{
		ptr ext = OSAL_NULL;
		sxm_result_code enRetCode;
		SXMSportsRow row, header;

		// Begin extracting the sports tree
		if ((enRetCode = (sxm_result_code) sxm_sports_begin(cRequestIt->second.poLeagueHandle, SXM_SPORTS_TREE, 0, &ext, &header)) == SXM_E_OK)
		{
			fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
			SXM_ASSERT_RETURN( OSAL_NULL!= poUtils )

			// Iterate until all rows are read.
			while((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
			{
				//The sxm_sports_extract_row routine returns the data for leagues from all sports as a row in the table SXM_SPORTS_LEAGUES.
				//Each league would be reported in one row.
				fc_sxm_trLeagueListData league;
				if (TRUE == poUtils->bExtractLeagueRow(row, league))
				{
					if ( 0u == league.u8Level )
					{
						//Handle differently for Root Affiliate
						league.setFamily.insert(league.u32AfID);
						league.enRequestState = fc_sxm_tenLeagueReqState_Updated;
						//league.bTreeAvl = TRUE;
						listDataIter->second = league;
					}
					else
					{
						//Add sub league to the root affiliate
						(tVoid) bAddSubLeagues(listDataIter->second, league);
					}
				} // end of loop "if (TRUE == poUtils->bExtractLeagueRow(row, league))"
			} //end of while loop
			//Free the handle used to read Row contents in sxm_sports_begin call
			enRetCode = (sxm_result_code) sxm_sports_end(ext);
			ETG_TRACE_USR4(("enRetCode = %u", enRetCode));
		} //end if loop sxm_sports_begin
		else
		{
			ETG_TRACE_ERR(("sxm_sports_begin failed at fc_sxm_tclSportsBase::vFillLeagueTree\t enRetCode = %u",
					enRetCode));
		}
	}
	else
	{
		ETG_TRACE_ERR(("Cannot retrieve information for the given league %u", u32LeagueID));
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vBuildLeagueFamily Exit"));
}

/*
 * Method adds sub affiliate to the Parent Affiliate
 */
tBool fc_sxm_tclSportsBase::bAddSubLeagues(fc_sxm_trLeagueListData &parent, fc_sxm_trLeagueListData &child)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bAddSubLeagues parent = %s", parent.strAfName.c_str()));
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bAddSubLeagues child = %s", child.strAfName.c_str()));
	//Traverse thru the entire list to find the appropriate position to fit the child league
	// If the parent and child's level differ by 1, then child is an immediate successor of the given parent league
	tBool bSuccess = FALSE;
	if (1u == (child.u8Level - parent.u8Level))
	{
		size_t length = parent.strAfName.length() ;
		if (child.strAfName.compare(0, length, parent.strAfName) == 0)
		{
			bSuccess = TRUE;
			parent.setFamily.insert(child.u32AfID);
			parent.mapSubLeagues[child.u32AfID] = child;
		}
	}
	else
	{
		SXM_FOREACH_MAP (tU32 , fc_sxm_trLeagueListData, listIt, parent.mapSubLeagues)
		{
			//Call bAddSubLeagues method recursively to build league Family
			size_t length = listIt->second.strAfName.length() ;
			if ((child.strAfName.compare(0, length, listIt->second.strAfName) == 0)
					&& ((bSuccess = bAddSubLeagues( listIt->second, child)) == TRUE ))
			{
				//break if an appropriate position to fit the child league is identified
				parent.setFamily.insert(child.u32AfID);
				break;
			} // end if loop for compare
		} // end for loop
	}
	return bSuccess;
}

/*
 * Method returns TRUE if the given affiliate is a Root Affiliate, returns FALSE if not
 */
tBool fc_sxm_tclSportsBase::bIsRootAffiliate(tU32 u32LeagueID) const
{
	tBool bRootLeague = FALSE;
	//Try to find if the requested affiliate is a root affiliate
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueListData, cIter, _mapLeague, u32LeagueID)
	{
		bRootLeague = TRUE;
	}
	return bRootLeague;
}
/*
 * Method returns for the Root Affiliate ID for the given sub league
 */
tU32 fc_sxm_tclSportsBase::u32GetRootAffiliate(tU32 u32SubLeague) const
{
	//The input parameter u32SubLeague can be an immediate child and multi-level's below the root league
	ETG_TRACE_USR4(("u32GetRootAffiliate u32SubLeague = %u\t_mapLeague count = %u",u32SubLeague, _mapLeague.size()));
	tU32 u32RootAffl = 0u;

	//Check if the given LeagueID belongs to any one of the Root Affiliate
	SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, _mapLeague)
	{
		ETG_TRACE_USR4(("League Current state = %u", cMapIt->second.enRequestState));
		SXM_IF_FIND_SET_CONST(tU32, cSetIt, cMapIt->second.setFamily, u32SubLeague)
		{
			u32RootAffl = cMapIt->first;
			break;
		}
	}
	ETG_TRACE_USR4(("u32GetRootAffiliate u32SubLeague(%u)'s Root Affiliate is (%u)",u32SubLeague, u32RootAffl));
	return u32RootAffl;
}
/*
 * Method to know a league has further subleagues
 */
tBool fc_sxm_tclSportsBase::bHasSubLeagues(tU32 u32League,fc_sxm_trLeagueListData& trLeagueListData) const
{
	ETG_TRACE_USR4(("bHasSubLeagues u32League = %u",u32League));

	tBool bHasSubLeague = FALSE;

	//Get League Data
	vGetLeagueData(u32League,trLeagueListData);

	//Check if this league has Sub Leagues
	if(trLeagueListData.mapSubLeagues.size() > 0)
	{
		//Yes
		bHasSubLeague = TRUE;
	}

	ETG_TRACE_USR4(("bHasSubLeagues u32League(%d) has sub leagues (%u)",u32League , trLeagueListData.mapSubLeagues.size() ));

	return bHasSubLeague;
}
/*
 * Method returns the current LeagueRequest State
 */
fc_sxm_tenLeagueReqState fc_sxm_tclSportsBase::enGetRequestState(tU32 u32LeagueID) const
{
	//Check if the LeagueRequest State is already available from the previous requests made
	fc_sxm_tenLeagueReqState curState = fc_sxm_tenLeagueReqState_Unavailable;
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueListData, cIter, _mapLeague, u32LeagueID)
	{
		//curState = cIter->second.bTreeAvl;
		curState = cIter->second.enRequestState;
	}
	else
	{
		ETG_TRACE_ERR(("Either the given league(%u) does not belong to the current sport(%u) OR its not a root affiliate", u32LeagueID, _u8SPID));
	}
	return curState;
}

/*
 * Method updates League Request State Machine
 */
tVoid fc_sxm_tclSportsBase::vSetLeagueRequestStatus (tU32 u32LeagueID, fc_sxm_tenLeagueReqState enNewState)
{
	//Search the gove Root Affiliate ID and update its state
	SXM_IF_FIND_MAP(tU32, fc_sxm_trLeagueListData, iter, _mapLeague, u32LeagueID)
	{
		iter->second.enRequestState = enNewState;
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSetLeagueRequestStatus Entered/Exit\tu32LeagueID = %u\tenNewState = %u",
			u32LeagueID, enNewState));
}

/*
 * Method sends error message to HMI
 */
tVoid fc_sxm_tclSportsBase::vSendLeagueTreeError(fc_sxm_trAdressing const &rAdressing) const
{
	//Method packs error type into FI msg to HMI
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSendLeagueTreeError Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
	_poApp->vSendErrorMsg (rAdressing, (tInt)midw_ext_fi_tcl_e8SportsErrorTypes::FI_EN_UNAVAILABLE);
	//fc_sxm_tclSportsService::vSendError(rAdressing,
		//						(tInt)midw_ext_fi_tcl_e8SportsErrorTypes::FI_EN_UNAVAILABLE);
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSendLeagueTreeError Exit"));
}

/*
 * Method to get Update Type for the requested Schedule day
 */

tU8 fc_sxm_tclSportsBase::u8GetUpdateType(tU8 u8ScheduleDay) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::u8GetUpdateTypeForScheduleDay Entered"));

	tU8 u8UpdateType = static_cast<tU8> (midw_ext_fi_tcl_SportsUpdateType::FI_EN_UNKNOWN);

	if(u8ScheduleDay == FC_SXM_SPORTS_TABLE_SMSE_NEWS)
	{
		ETG_TRACE_USR4(("Update for News available"));
		u8UpdateType = static_cast<tU8> (midw_ext_fi_tcl_SportsUpdateType::FI_EN_NEWS);
	}
	//Check if the table param is Today game
	else if (u8ScheduleDay == FC_SXM_SPORTS_SCHEDULE_SMSE_TODAY)
	{
		ETG_TRACE_USR4(("Update for Today available"));
		u8UpdateType = static_cast<tU8> (midw_ext_fi_tcl_SportsUpdateType::FI_EN_TODAY);
	}
	//Check if the table param with in Recent game list Range
	else if(u8ScheduleDay < FC_SXM_SPORTS_SCHEDULE_SMSE_TODAY)
	{
		ETG_TRACE_USR4(("Update for Recent available"));
		u8UpdateType = static_cast<tU8> (midw_ext_fi_tcl_SportsUpdateType::FI_EN_RECENT);
	}
	//Check if the table param with in Future game list Range
	else if ((u8ScheduleDay > FC_SXM_SPORTS_SCHEDULE_SMSE_TODAY) && (u8ScheduleDay <= FC_SXM_SPORTS_SCHEDULE_SMSE_NEXT))
	{
		ETG_TRACE_USR4(("Update for Future available"));
		u8UpdateType = static_cast<tU8> (midw_ext_fi_tcl_SportsUpdateType::FI_EN_FUTURE);
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::u8GetUpdateTypeForScheduleDay Exit"));
	return u8UpdateType;

}

/* Method to Notify  Sports Data Update to HMI */
tVoid fc_sxm_tclSportsBase::vNotifySportDataUpdate(tU32 u32UpdateLeague,tU8 u8UpdateType)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifySportDataUpdate Entered\tSportID = %u\tUpdateLeagueID = %u\tUpdate Type = %d",
									_u8SPID, u32UpdateLeague, u8UpdateType));

	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

	ETG_TRACE_USR4(("_u32ActiveLeague = %u", fc_sxm_tclSportsBase::_u32ActiveLeague ));
	tU8 u8UpdatedType = u8GetActiveUpdateType(u8UpdateType);
	tU8 u8UpdateStatus = u8IsDataAvailable(u32UpdateLeague,u8UpdatedType);

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifySportDataUpdate Entered\t DataAvailableFlag = %u",u8UpdateStatus ));

	if (u8UpdateStatus != static_cast<tU8> (midw_ext_fi_tcl_e8CategoryAvailability::FI_EN_NOT_APPLICABLE))
	{
		if (u32UpdateLeague == fc_sxm_tclSportsBase::_u32ActiveLeague)// Either Root Affiliate or Sub Affiliate if both Active and Update Leagues are same
		{
			ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifySportDataUpdate : Update happened at same league level"));
			vHandlePropertyUpdate(u32UpdateLeague,u8UpdateType,u8UpdateStatus);

		}
		else // Update happened for League ID other than active League ID
		{
			ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifySportDataUpdate : Update happened at sub league levels"));
			//Get League Data
			fc_sxm_trLeagueListData trActiveLeagueListData,trUpdateLeagueListData;
			vGetLeagueData(fc_sxm_tclSportsBase::_u32ActiveLeague,trActiveLeagueListData);// Gets the sub league list from the provided sub league ID
			//Find SXM Update League present in the received League List
			if(TRUE == bFindLeague(trActiveLeagueListData, u32UpdateLeague,trUpdateLeagueListData))
			{
				ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifySportDataUpdate : Update happened at sub league levels"));
				vHandlePropertyUpdate(u32UpdateLeague,u8UpdateType,u8UpdateStatus);

			}
		}
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifySportDataUpdate Exit"));
}

tVoid fc_sxm_tclSportsBase::vAccumulateLeagueUpdates(tU8 u8SportID, tU32 u32UpdateLeague ,tU8 u8UpdateType , tU8 u8UpdateStatus)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vAccumulateLeagueUpdates Entered"));

	fc_sxmSportsDataUpdate Updatelist;

	Updatelist.u32LeagueID = u32UpdateLeague;
	Updatelist.u8SportID = u8SportID;
	Updatelist.u8UpdateStatus = u8UpdateStatus;

	_mapPropertyUpdate[u8UpdateType] = Updatelist;

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vAccumulateLeagueUpdates Map Size = %u " ,_mapPropertyUpdate.size()));

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vAccumulateLeagueUpdates EXit\t Sport ID = %u , LeagueID = %u, UpdateType = %u , UpdateStatus = %u",
			_mapPropertyUpdate[u8UpdateType].u8SportID,_mapPropertyUpdate[u8UpdateType].u32LeagueID,u8UpdateType,_mapPropertyUpdate[u8UpdateType].u8UpdateStatus ));
}

tVoid fc_sxm_tclSportsBase::vNotifyLeagueUpdates(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifyLeagueUpdates Entered"));

	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

	for (map<tU8 , fc_sxmSportsDataUpdate>::iterator it =_mapPropertyUpdate.begin(); it!=_mapPropertyUpdate.end(); ++it)
	{
		ETG_TRACE_USR4(("<---fc_sxm_tclSportsBase::vNotifyLeagueUpdates() Send Update"));
		_poApp->vSendSportsDataUpdateStatus(it->second.u8SportID,it->second.u32LeagueID,it->first,it->second.u8UpdateStatus);
	}
	_mapPropertyUpdate.clear();
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vNotifyLeagueUpdates Exit"));
}

tVoid fc_sxm_tclSportsBase::vHandlePropertyUpdate(tU32 u32UpdateLeague , tU8 u8UpdateType ,tU8 u8UpdateStatus)
{
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vHandlePropertyUpdate Entry"));
	if (!_poApp->bIsSportsTimerRunning())
	{
		// Send the first property Update to HMI
		_poApp->vSendSportsDataUpdateStatus(_u8SPID,u32UpdateLeague,u8UpdateType,u8UpdateStatus);
		ETG_TRACE_USR4(("Start the Timer to collect the Property Update"));
		_poApp->vStartSportsTimer(500); /* Time in milliSeconds */
	}
	else /* Collect the property Updates to the MIDW List and finally update the List to HMI after ignoring the duplicate Updates for the same type on Timer Expiry */
	{
		vAccumulateLeagueUpdates(_u8SPID,u32UpdateLeague,u8UpdateType,u8UpdateStatus);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vHandlePropertyUpdate Exit"));
}
/*
 * Method returns the game/event availability for a particular league
 */
tBool fc_sxm_tclSportsBase::bIsScheduleAvilable(tU32 u32LeagueID, tU8 u8GameType) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bIsScheduleAvilable Entered u32LeagueID=%d,u8GameType=%d",u32LeagueID,u8GameType));

	tBool bIsSchavil = FALSE;

	//Check if it is a Root League
	if(bIsRootAffiliate(u32LeagueID))
	{
		ETG_TRACE_USR4(("u32LeagueID=%d is a Root League",u32LeagueID));

		bIsSchavil = bFilterSchedules(u32LeagueID, AFFILIATE_INVALID_ID, FALSE, u8GameType);

	}
	//Sub League Case
	else
	{
		tU32 u32RootLeagueID = u32GetRootAffiliate(u32LeagueID);

		if(u32RootLeagueID != AFFILIATE_INVALID_ID)
		{
			ETG_TRACE_USR4(("u32LeagueID=%d has a Root League=%d", u32LeagueID, u32RootLeagueID));

			fc_sxm_trLeagueListData trLeagueListData;
			//Check if it has further sub leagues
			if(bHasSubLeagues(u32LeagueID,trLeagueListData))
			{
				//Get Sub League ID for the intermediate League
				SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, trLeagueListData.mapSubLeagues)
				{
					bIsSchavil = bFilterSchedules(u32RootLeagueID, cMapIt->second.u32AfID, TRUE, u8GameType);

					if(bIsSchavil == TRUE)
					{
						break;
					}
				}

			}
			else		/* Last Sub League */
			{
				bIsSchavil = bFilterSchedules(u32RootLeagueID, u32LeagueID , TRUE ,u8GameType);
			}
		}
		else
		{
			ETG_TRACE_ERR(("u32LeagueID=%d not belongs to any Root League",u32LeagueID));
		}
	}

	return bIsSchavil;
}
/*
 * Method returns the game/event availability for a particular league
 */
tBool fc_sxm_tclSportsBase::bFilterSchedules(tU32 u32RootLeagueID, tU32 u32SubLeagueID, tBool bForSubLeague,tU8 u8GameType) const
{
	ETG_TRACE_USR1(("fc_sxm_tclSportsBase::bFilterSchedules Entered u32RootLeagueID=%d,u32SubLeagueID=%d,bForSubLeague=%u,u8GameType=%u",
				u32RootLeagueID,u32SubLeagueID,bForSubLeague,u8GameType));
	SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp, FALSE)

	tBool bAvail = FALSE;

	//Check if the league exists
	//SXM_IF_FIND_MAP(tU32, fc_sxm_trLeagueData, cit, _mapLeagueDetails,u32RootLeagueID)
	// to fix NCG3D-53
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueData, cit, _mapLeagueDetails,u32RootLeagueID)
	{
		SXM_FOREACH_CONST( set<tS32>, cSetIt, cit->second.setLeagueTableParam)
		{
			//Check League ID recieved for a Table
			tU32 u32TableLeague = SPORTS_TABLE_AF(*cSetIt);
			ETG_TRACE_USR4(("u32TableLeague = %d",u32TableLeague));

			//Check to filter for Sub League
			if(bForSubLeague && (u32SubLeagueID != u32TableLeague))
			{
				ETG_TRACE_USR2(("Not for u32SubLeagueID=%d",u32SubLeagueID));
				continue;		//Continue to Check next table param
			}

            //Check for the day we recieved from SMSe
			tU8 u8ScheduleDay = _poApp->u8GetDayMapIndex(*cSetIt);
			ETG_TRACE_USR4(("u8ScheduleDay = %d",u8ScheduleDay));

			//Check if HMI request is for the Same Day we recieved from SMS
			if ((bFindSchedule(u8GameType,u8ScheduleDay)) &&
					(bValidateTable(cit->second.poLeagueHandle, *cSetIt)))
			{
				//OK
				bAvail = TRUE;
				//Atleast 1 game/event is available to diplay
				//Then no need to continue further
				break;
			}
		}
	}
	else
	{
		ETG_TRACE_ERR(("Not found a League=%d",u32SubLeagueID));
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bFilterSchedules exited.."));

	//nothing matches return FALSE
	return bAvail;
}
/*
 * Method to Check if a ScheduleDay belongs to gameType(Today/Recent/Future)
 */
tBool fc_sxm_tclSportsBase::bFindSchedule(tU8 u8GameType,tU8 u8ScheduleDay) const
{
	//Check for the gametype
	switch(u8GameType)
	{
		//for Recent Game
		case FC_SXM_SPORTS_SCHEDULE_RECENT:
		{
			//Check if the table param with in Recent game list Range
			if(u8ScheduleDay < FC_SXM_SPORTS_SCHEDULE_SMSE_TODAY)
			{
				ETG_TRACE_USR4(("Schedules for Recent avil"));
				return TRUE;
			}
			break;
		}
		//for Todays Game
		case FC_SXM_SPORTS_SCHEDULE_TODAY:
		{
			//Check if the table param is Today game
			if(u8ScheduleDay == FC_SXM_SPORTS_SCHEDULE_SMSE_TODAY)
			{
				ETG_TRACE_USR4(("Schedules for Today avail"));
				return TRUE;
			}
			break;
		}
		case FC_SXM_SPORTS_SCHEDULE_FUTURE:
		{
			//Check if the table param with in Future game list Range
			if(u8ScheduleDay > FC_SXM_SPORTS_SCHEDULE_SMSE_TODAY &&
					u8ScheduleDay <= FC_SXM_SPORTS_SCHEDULE_SMSE_NEXT)
			{
				ETG_TRACE_USR4(("Schedules for Future avail"));
				return TRUE;
			}
			break;
		}
		default:
			break;
	}

	return FALSE;
}
/*
 * Method returns the game/event availability for a particular league
 */
tVoid fc_sxm_tclSportsBase::vGetSchedules(tU32 u32LeagueID, tU8 u8GameType , tU32 u32GameID ,fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR1(("fc_sxm_tclSportsBase::vGetSchedules Entered u32LeagueID=%d,u8GameType=%u",u32LeagueID, u8GameType));

	//Set Current Active LEague
	fc_sxm_tclSportsBase::_u32ActiveLeague = u32LeagueID;

	//Set Schedule Type for Ordering
	fc_sxm_tclSportsBase::_u8ScheduleType  = u8GameType;

	//Before we get the new Game/Event List From SMS
	//Clear the Old List
	vClearSchedulesList();

	//Check if it is a Root League
	if(bIsRootAffiliate(u32LeagueID))
	{
		ETG_TRACE_USR4(("u32LeagueID=%d is a Root League",u32LeagueID));

		vFilterSchedules(u32LeagueID, AFFILIATE_INVALID_ID, FALSE, u8GameType);

	}
	//Sub League Case
	else
	{
		tU32 u32RootLeagueID = u32GetRootAffiliate(u32LeagueID);

		if(u32RootLeagueID != AFFILIATE_INVALID_ID)
		{
			ETG_TRACE_USR4(("u32LeagueID=%d has a Root League=%d", u32LeagueID, u32RootLeagueID));

			fc_sxm_trLeagueListData trLeagueListData;
			//Check if it has further sub leagues
			if(bHasSubLeagues(u32LeagueID,trLeagueListData))
			{
				//Get Sub League ID for the intermediate League
				SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, trLeagueListData.mapSubLeagues)
				{
					vFilterSchedules(u32RootLeagueID, cMapIt->second.u32AfID, TRUE, u8GameType);
				}
			}
			else		/* Last Sub League */
			{
				vFilterSchedules(u32RootLeagueID, u32LeagueID , TRUE ,u8GameType);
			}
		}
		else
		{
			ETG_TRACE_ERR(("u32LeagueID=%d not belongs to any Root League",u32LeagueID));
		}
	}

	//Finally Send Method Result to HMI
	vEmit(u32GameID,rAdressing);
}
/*
 * Method returns the game/event availability for a particular league
 */
tVoid fc_sxm_tclSportsBase::vFilterSchedules(
		tU32 u32RootLeagueID, tU32 u32SubLeagueID, tBool bForSubLeague,tU8 u8GameType)
{
	ETG_TRACE_USR1(("fc_sxm_tclSportsBase::vFilterSchedules Entered u32RootLeagueID=%d,u32SubLeagueID=%d,bForSubLeague=%u,u8GameType=%u",
			u32RootLeagueID,u32SubLeagueID,bForSubLeague,u8GameType));
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

	//Check if the league exists
	//SXM_IF_FIND_MAP(tU32, fc_sxm_trLeagueData, cit, _mapLeagueDetails,u32RootLeagueID)
	// to fix NCG3D-53
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueData, cit, _mapLeagueDetails,u32RootLeagueID)
	{
		SXM_FOREACH_CONST( set<tS32>, cSetIt, cit->second.setLeagueTableParam)
		{
			//Check League ID recieved for a Table
			tU32 u32TableLeague = SPORTS_TABLE_AF(*cSetIt);

			ETG_TRACE_USR4(("u32TableLeague = %d",u32TableLeague));

			//Check to filter for Sub League
			if(bForSubLeague && (u32SubLeagueID != u32TableLeague))
			{
				ETG_TRACE_USR2(("Not for u32SubLeagueID=%d",u32SubLeagueID));
				continue;		//Continue to Check next table param
			}

			//Check for the day we recieved from SMSe
			tU8 u8ScheduleDay = _poApp->u8GetDayMapIndex(*cSetIt);
			ETG_TRACE_USR4(("u8ScheduleDay = %d",u8ScheduleDay));
			//Check if HMI request is for the Same Day we recieved from SMS
			if(bFindSchedule(u8GameType,u8ScheduleDay))
			{
				vRetrieveSchedules(cit->second.poLeagueHandle,*cSetIt);
			}
		}
	}
	else
	{
		ETG_TRACE_ERR(("Not found a League=%d",u32SubLeagueID));
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vFilterSchedules exited.."));
}

tVoid fc_sxm_tclSportsBase::vRetrieveSchedules(tVoid* pLeagueHandle, tS32 s32TableParam)
{
	ETG_TRACE_USR4(("vRetrieveSchedules pLeagueHandle = %x, s32TableParam = %d",pLeagueHandle,s32TableParam));

	ptr ext = OSAL_NULL;
	sxm_result_code enRetCode;
	SXMSportsRow row,header;

	if ((enRetCode = (sxm_result_code) sxm_sports_begin(pLeagueHandle, SXM_SPORTS_TABLE, s32TableParam , &ext, &header)) == SXM_E_OK)
	{
		tU32 u32EpochValue = header.value[SXM_SPORTS_TABLE_EPOCH] * 86400 + 2*3600; // Epoch in Secs

		// Iterate until all rows are read.
		while((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
		{
			//The sxm_sports_extract_row routine returns the data for leagues from all sports as a row in the table SXM_SPORTS_LEAGUES.
			//Each league would be reported in one row.
			//Extract Schedule Row From SMSe
			(tVoid) bExtractScheduleRow(ext, row, s32TableParam,u32EpochValue, FALSE);
		} //end for loop

		(tVoid) sxm_sports_end(ext);

	} //end if loop
	else
	{
		ETG_TRACE_ERR(("sxm_sports_begin failed at fc_sxm_tclSportsBase::vRetrieveSchedules\t enRetCode = %u",
				enRetCode));
		if (SXM_E_NOENT == enRetCode) // If no_ent error happened in extraction, remove the corresponding table param
		{
			ETG_TRACE_USR4(("Expiry Of Table Param.."));
			vRemoveExpiredTableParam(s32TableParam);
		}
	}
}

tVoid fc_sxm_tclSportsBase::vRemoveExpiredTableParam(tS32 s32TableParam)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vRemoveExpiredTableParam Entered\ts32TableParam = %d", s32TableParam));

	SXM_FOREACH_MAP(tU32, fc_sxm_trLeagueData, requestIt, _mapLeagueDetails)
	{
		SXM_FOREACH( set<tS32>, setTableParamIter, requestIt->second.setLeagueTableParam)
		{
			if(*setTableParamIter == s32TableParam)
			{
				ETG_TRACE_USR4(("Identified Table Param for Removal"));
				requestIt->second.setLeagueTableParam.erase(s32TableParam);
				break;
			}
		}
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vRemoveExpiredTableParam Exit"));
}

/*
 * Method returns true if valid row entries are present for the given Table Param
 */
tBool fc_sxm_tclSportsBase::bValidateTable(tVoid* pLeagueHandle, tS32 s32TableParam) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bValidateTable Entered\ts32TableParam = %d\tTABLE TYPE = %d", s32TableParam, SPORTS_TABLE_IX(s32TableParam)));
	ETG_TRACE_USR4(("SportID = %u\t SportName = %s", _u8SPID, _strSPName.c_str()));
	SXM_ASSERT_RETURN_VAL((OSAL_NULL != pLeagueHandle), FALSE)
	tBool bValid = FALSE;
	ptr ext = OSAL_NULL;
	sxm_result_code enRetCode;
	SXMSportsRow row, header;

	// SXM_SPORTS_TABLE refer to all Schedules type as well as NEWS items
	if ((enRetCode = (sxm_result_code) sxm_sports_begin(pLeagueHandle, SXM_SPORTS_TABLE, s32TableParam , &ext, &header)) == SXM_E_OK)
	{
		ETG_TRACE_USR4(("sxm_sports_begin HEADER is VALID"));
		// Check if the first row contains valid row entries
		if((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
		{
			bValid = TRUE;
		}
		else
		{
			// Error occured while extracting row...
			ETG_TRACE_ERR(("EXTRACT ROW ERROR @fc_sxm_tclSportsBase::bValidateTable \tenRetCode = %u", enRetCode));
		}
		(tVoid) sxm_sports_end(ext);
	}
	else
	{
		ETG_TRACE_ERR(("sxm_sports_begin failed at fc_sxm_tclSportsBase::bValidateTable\t enRetCode = %u",
				enRetCode));
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bValidateTable Exit\tValid Rows present(%u)\tenRetCode(%u)", bValid, enRetCode));
	return bValid;
}

tU8 fc_sxm_tclSportsBase::u8GetActiveUpdateType(tU8 u8UpdateType) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::u8GetActiveUpdateType u8UpdateType = %u", u8UpdateType));
	tU8 u8UpdatedType = 0;
	switch (u8UpdateType)
	{
		case midw_ext_fi_tcl_SportsUpdateType::FI_EN_TODAY:
		{
			u8UpdatedType = static_cast<tU8> (midw_ext_fi_tcl_AffiliateCategories::FI_EN_TODAY);
		}
		break;

		case midw_ext_fi_tcl_SportsUpdateType::FI_EN_RECENT:
		{
			u8UpdatedType = static_cast<tU8> (midw_ext_fi_tcl_AffiliateCategories::FI_EN_RECENT);
		}
		break;

		case midw_ext_fi_tcl_SportsUpdateType::FI_EN_FUTURE:
		{
			u8UpdatedType = static_cast<tU8> (midw_ext_fi_tcl_AffiliateCategories::FI_EN_FUTURE);
		}
		break;

		case midw_ext_fi_tcl_SportsUpdateType::FI_EN_NEWS:
		{
			u8UpdatedType = static_cast<tU8> (midw_ext_fi_tcl_AffiliateCategories::FI_EN_NEWS);
		}
		break;
		default:
			break;
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::u8GetActiveUpdateType u8UpdatedType = %u", u8UpdatedType));
	return u8UpdatedType;
}

//Function to update whether SXM_SPORTS_UPDATED to be handled or not
tVoid fc_sxm_tclSportsBase::vIgnoreUpdates(tBool bIgnore)
{
	_bIgnoreUpdate = bIgnore;
}

//Function to return whether SXM_SPORTS_UPDATED to be handled or not
tBool fc_sxm_tclSportsBase::bGetIgnoreUpdates() const
{
	return _bIgnoreUpdate;
}

