/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_phonetics_base.cpp
* @brief       Base class implementation for Phonetics service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_phonetics_base.h"
#include "fc_sxm_tcl_phonetics_proxy.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PHONETICS_BASE
#include "trcGenProj/Header/fc_sxm_tcl_phonetics_base.cpp.trc.h"
#endif


fc_sxm_tclPhoneticsBase::fc_sxm_tclPhoneticsBase()
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase Default constructor"));
    _poPhoneticsProxy = OSAL_NULL;

}

/****************************************************************************
 *
 *FUNCTION:          fc_sxm_tclPhoneticsBase
 *
 *DESCRIPTION:       Constructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclPhoneticsBase::fc_sxm_tclPhoneticsBase(Ifc_sxm_tclPhoneticsProxy *pIPhoneticsProxy)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase Parameterized constructor"));
    _poPhoneticsProxy = pIPhoneticsProxy;

}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_tclPhoneticsBase
 *
 *DESCRIPTION:       Destructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclPhoneticsBase::~fc_sxm_tclPhoneticsBase()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase destructor"));
    _poPhoneticsProxy = OSAL_NULL;

}
/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vUpdatePhoneticsInfo                                          *
 *                                                                                 *
 *DESCRIPTION:       Properties Update of phonetics data to SDS                    *
 *                                                                                 *
 *PARAMETERS:        PHONETICS_FILE_TYPE_ENUM  &  SMS_LANGUAGE_ENUM                *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tclPhoneticsBase::vConvertPhoneticsInfo(midw_ext_sxm_phoneticsfi_tclMsgPhoneticsUpdateStatus &oFiMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vConvertPhoneticsInfo"));

    // coverting sms types to local types
    tU8 nPhoneType = u8GetPhoneticsFileType(oFiMsg.PhoneticsType);
    tU8 nLangType = u8GetPhoneticsLanguageType(oFiMsg.LanguageType);
    oFiMsg.PhoneticsType = nPhoneType;
    oFiMsg.LanguageType = nLangType;
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vGetPhoneticsData                                             *
 *                                                                                 *
 *DESCRIPTION:       get Phonetics data for the specified file type                *
 *                                                                                 *
 *PARAMETERS:        midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodResult   *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tclPhoneticsBase::vGetPhoneticsData(midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodResult &ofiTxObj)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData"));

    tU8 nPhonetype = (ofiTxObj).PhoneticsType;

    if(nPhonetype & midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_CHANNELS)
    {
        ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData:Channels Phonetics type"));
    }
    if(nPhonetype & midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_CATEGORIES)
    {
        ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData:Categories Phonetics type"));
    }
    if(nPhonetype & midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_MARKETS)
    {
        ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData:Markets Phonetics type"));
    }
    if(nPhonetype & midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_TEAMS)
    {
        ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData:Teams Phonetics type"));
    }
    if(nPhonetype & midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_LEAGUES)
    {
        ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData:Leagues Phonetics type"));
    }
    if(nPhonetype == midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_UNKNOWN)
    {
        ETG_TRACE_ERR(("fc_sxm_tclPhoneticsBase::vGetPhoneticsData:Unknown Phonetics type"));
    }
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          bReadPhoneticEntry                                            *
 *                                                                                 *
 *DESCRIPTION:       Parsing phones strings for the specified phonetic type        *
 *                   and language                                                  *
 *                                                                                 *
 *PARAMETERS:        File Pinter & vector<midw_ext_fi_tclString>                   *
 *                                                                                 *
 *RETURNVALUES:      tBool                                                         *
 *                                                                                 *
 **********************************************************************************/
tBool fc_sxm_tclPhoneticsBase::bReadPhoneticEntry(OSAL_FILE_STRUCT *psPhoneticsFile,
        vector<midw_ext_fi_tclString> &vPhoneticsString)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::bReadPhoneticEntry"));
    tBool bPhoneticsRead = FALSE;

    if((_poPhoneticsProxy != OSAL_NULL) && (psPhoneticsFile != OSAL_NULL))
    {
        char acBuffer[PHONETICS_MAX_NUMCHARS];
        tSize tLen;
        UN8 un8Index = 0;
        // Verify file position / format with a check for
        // the field delimiter
        char cData = (char )_poPhoneticsProxy->iFgetc(psPhoneticsFile);
        if (cData != '|')
        {
            // Nothing to do here, but that's okay
            // since this is probably the end of the file
            return FALSE;
        }

        // Read phonetics
        do
        {
            // Read the next byte
            tLen = _poPhoneticsProxy->tFread(&acBuffer[un8Index], sizeof(char), 1, psPhoneticsFile);
            if(tLen != 1)
            {
                // Error occured
                break;
            }
            if ((bIsDelimeter(acBuffer[un8Index]))
                    || (bIsEndOfLine(acBuffer[un8Index])) )
            {

                // Yes, but is there anything here?
                if (un8Index == 0)
                {
                    // Nope, we have a bad file here
                    break;
                }
            }

            // Save this character for later
            cData = acBuffer[un8Index];

            bPhoneticsRead = bAddPhoneticsIntoList(acBuffer, un8Index, vPhoneticsString);

            if(bIsEndOfLine(cData))
            {
                break;
            }

        } while (un8Index < PHONETICS_MAX_NUMCHARS);
    }
    return bPhoneticsRead;
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          bAddPhoneticsIntoList                                         *
 *                                                                                 *
 *DESCRIPTION:       Pushing Phonemes string into list                             *
 *                                                                                 *
 *                                                                                 *
 *PARAMETERS:        Char Pointer,  un8Index & vector<midw_ext_fi_tclString>       *
 *                                                                                 *
 *RETURNVALUES:      tBool                                                         *
 *                                                                                 *
 **********************************************************************************/
tBool fc_sxm_tclPhoneticsBase::bAddPhoneticsIntoList(char* pcBuffer, UN8 &un8Index,
        vector<midw_ext_fi_tclString> &vPhoneticsString)
{
    // Did we find a delimiter or end of line?
    if ((bIsDelimeter(pcBuffer[un8Index]))
            || (bIsEndOfLine(pcBuffer[un8Index])) )
    {
        // Save this character for later
        char cData = pcBuffer[un8Index];
        // Null terminate this string
        pcBuffer[un8Index] = '\0';

        //Pushing phonemes string into list
        vPhoneticsString.push_back(pcBuffer);
        if (!bIsEndOfLine(cData))
        {
            un8Index = 0;
        }
        return TRUE;
    }
    else
    {
        // Keep collecting characters
        un8Index++;
        return FALSE;
    }
}

tBool fc_sxm_tclPhoneticsBase::bIsDelimeter(const char cChar)
{
    return ((cChar == '|') ?
            TRUE :
            FALSE);
}

tBool fc_sxm_tclPhoneticsBase::bIsEndOfLine(const char cChar)
{
    return ((cChar == '\n') ?
            TRUE :
            FALSE);
}

/* convert SMS file type to local file type */
tU8 fc_sxm_tclPhoneticsBase::u8GetPhoneticsFileType(tU8 eFileType)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::u8GetPhoneticsFileType"));

    //2^n
    return((tU8)(0x0001 << eFileType));
}

/* convert SMS language type to local language type */
tU8 fc_sxm_tclPhoneticsBase::u8GetPhoneticsLanguageType(tU8 eLangType)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsBase::u8GetPhoneticsLanguageType"));

    //2^n
    return((tU8)(0x0001 << eLangType));
}
