/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_phonetics_app.cpp
* @brief       Application handler implementation for Phonetics service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_phonetics_app.h"
#include "fc_sxm_service_sxm_phonetics.h"
#include "fc_sxm_tcl_phonetics_base_if.h"
#include "fc_sxm_tcl_phonetics_proxy.h"
#include "fc_sxm_tcl_phonetics_channels.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PHONETICS_APP
#include "trcGenProj/Header/fc_sxm_tcl_phonetics_app.cpp.trc.h"
#endif


/****************************************************************************
 *                                                                          *
 *FUNCTION:          fc_sxm_tclPhoneticsApp                                 *
 *                                                                          *
 *DESCRIPTION:       Constructor                                            *
 *                                                                          *
 *PARAMETERS:        None                                                   *
 *                                                                          *
 *RETURNVALUES:      None                                                   *
 *                                                                          *
 ****************************************************************************/
fc_sxm_tclPhoneticsApp::fc_sxm_tclPhoneticsApp(): fc_sxm_tclDataApp("sxmPhoneticsApp", _enStaticServiceId,
        CCA_C_U16_SRV_SXM_PHONETICS, "phonetics",
        static_cast <tU16> (TR_CLASS_FC_SXM_PHONETICS_APP)),
        _hPhonetics(PHONETICS_SERVICE_INVALID_OBJECT)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhonetics constructor"));
    _pPhoneticsBase = OSAL_NULL;
    _poPhoneticsProxy = OSAL_NULL;
}

/****************************************************************************
 *                                                                          *
 *FUNCTION:          ~fc_sxm_tclPhoneticsApp                                *
 *                                                                          *
 *DESCRIPTION:       Destructor                                             *
 *                                                                          *
 *PARAMETERS:        None                                                   *
 *                                                                          *
 *RETURNVALUES:      None                                                   *
 *                                                                          *
 ****************************************************************************/
fc_sxm_tclPhoneticsApp::~fc_sxm_tclPhoneticsApp()
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp destructor"));

    _hPhonetics = PHONETICS_SERVICE_INVALID_OBJECT;
    _pPhoneticsBase = OSAL_NULL;
    _poPhoneticsProxy = OSAL_NULL;

}

/****************************************************************************
 *                                                                          *
 *FUNCTION:          vInitialize                                            *
 *                                                                          *
 *DESCRIPTION:       called in cca-context by sxmapp-manager                *
 *                                                                          *
 *PARAMETERS:        None                                                   *
 *                                                                          *
 *RETURNVALUES:      None                                                   *
 *                                                                          *
 ****************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vInitialize"));
    fc_sxm_tclBaseApp::vInitialize();
    _poPhoneticsProxy = OSAL_NEW fc_sxm_tclPhoneticsProxy();

    //TODO : currently support only for Channel type
    if(_poPhoneticsProxy != OSAL_NULL)
    {
        _pPhoneticsBase =  OSAL_NEW fc_sxm_tcl_phonetics_channels(_poPhoneticsProxy);
    }
}

/****************************************************************************
 *                                                                          *
 *FUNCTION:          vDeInitialize                                          *
 *                                                                          *
 *DESCRIPTION:        called in cca-context by sxmapp-manager               *
 *                                                                          *
 *PARAMETERS:        None                                                   *
 *                                                                          *
 *RETURNVALUES:      None                                                   *
 *                                                                          *
 ****************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vDeInitialize"));

    // call vDeInitialize() of base-class to stop the worker thread
    fc_sxm_tclBaseApp::vDeInitialize();
    OSAL_DELETE _pPhoneticsBase;

    //App is owner of proxy,so delete proxy object here
    OSAL_DELETE _poPhoneticsProxy;
}

/*******************************************************************************
 *                                                                             *
 *FUNCTION:          vDispatchMsgFromQ                                         *
 *                                                                             *
 *DESCRIPTION:       this method is called by the thread-function.             *
                     For each message that shall be handled put and entry here.*                                                                          *
 *PARAMETERS:        fc_sxm_tclMessage                                         *
 *                                                                             *
 *RETURNVALUES:      None                                                      *
 *                                                                             *
 ******************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg)
{

    // Get Action Value
    tU32 u32Action=poThreadMsg->u32GetAction();

    switch (u32Action) {

    // dispatch Phonetics Service State mesaage to CCA
    // Also Do this for every method exposed thru FI
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgPhoneticsSMSFileEventCallback, this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgDataMStartGetPhoneticsData, this);
    default:
        ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                u32Action));
        break;
    }

}

static void cb_vPhoneticsFileEventCallback ( PHONETICS_SERVICE_OBJECT /*hPhoneticsService*/, PHONETICS_FILE_TYPE_ENUM eFileType,
        SMS_LANGUAGE_ENUM eLanguage, const char */*pacTTSFilePath*/, const char */*pacRecFilePath*/,
        void *pvFileCallbackArg )
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::cb_vPhoneticsFileEventCallback"));

    fc_sxm_tclPhoneticsApp* poPhonetics = static_cast < fc_sxm_tclPhoneticsApp* > (pvFileCallbackArg);
    if ( OSAL_NULL != poPhonetics )
    {
        //TODO : currently support only for Channel type
        if(eFileType == PHONETICS_FILE_TYPE_CHANNELS)
        {
            //PostMsgNew
            fc_sxm_trMsgPhoneticsSMSFileEventCallback rMsg;
            (rMsg.oFiMsg).PhoneticsType = (tU8) eFileType;
            (rMsg.oFiMsg).LanguageType = (tU8) eLanguage;
            poPhonetics->vPostMsgNew(rMsg);
        }
    }
}

/*******************************************************************************
 *                                                                             *
 *FUNCTION:          vStartSmsService                                          *
 *                                                                             *
 *DESCRIPTION:       Method to start SMS service.                              *
 *                                                                             *
 *PARAMETERS:        None                                                      *
 *                                                                             *
 *RETURNVALUES:      None                                                      *
 *                                                                             *
 ******************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vStartSmsService()
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vStartSmsService"));
    if(_poPhoneticsProxy != OSAL_NULL)
    {
        _hPhonetics = _poPhoneticsProxy->vStartSmsService(
                FC_SXM_SRH_DRIVER_NAME,                //SRH driver name
                FC_SXM_DEFAULT_SMS_PHONETICS_APP_PATH, //Phonetics dic file full path
                DATASERVICE_EVENT_ALL,                 //Caller wished Events
                cb_vDataServiceEventCallback,          //Event Call back function
                (tVoid*)this,                          //Event Call back Argument
                cb_vPhoneticsFileEventCallback,        //File Call back function
                (tVoid*)this,                          //File Call back Argument
                DATASERVICE_OPTION_NONE                //Options to config the service
        );
        //Check for Invalid Object
        if(_hPhonetics == PHONETICS_SERVICE_INVALID_OBJECT)
        {
            ETG_TRACE_ERR(("Failed to start Phonetics Service"));
        }
        else
        {
            ETG_TRACE_USR4(("Phonetics service started successfully"));
        }
    }
}

/*********************************************************************************
 *                                                                               *
 *FUNCTION:          vClearServiceData                                           *
 *                                                                               *
 *DESCRIPTION:       To clear all the stored data during reinitialisation of SMS.*
 *                   This is a overridden method from the data app.              *
 *PARAMETERS:        None                                                        *
 *                                                                               *
 *RETURNVALUES:      None                                                        *
 *                                                                               *
 ********************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vClearServiceData()
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vClearServiceData(%s)",
            szGetName()));
}

/*********************************************************************************
 *                                                                               *
 *FUNCTION:          vProcessBaseAppMsg                                          *
 *                                                                               *
 *DESCRIPTION:       handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd     *
                     (Mandatory)                                                 *
 *                                                                               *
 *PARAMETERS:        None                                                        *
 *                                                                               *
 *RETURNVALUES:      None                                                        *
 *                                                                               *
 ********************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
    // Method to handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) Entered code = %u",
            prMsg->u8MsgCode));

    tVoid (fc_sxm_tclPhoneticsApp::*funcPtrArray[]) (fc_sxm_trMsgCmdAppTtfisCmd const *) const = {
            &fc_sxm_tclPhoneticsApp::vHandleTTFIsCmdGetChannelNames,
            &fc_sxm_tclPhoneticsApp::vHandleTTFIsCmdGetChannelNames

    };

    //Store the values read for each index to the appropriate field in fc_sxm_trPhoneticsList structure
    (this->*funcPtrArray[prMsg->u8MsgCode])(prMsg);
}
/*
 * TTFIs cmd --> Handle C_SXM_PHONETICS_MSTART_GET_PHONETICS_DATA %u() %u() issued from TTFIs
 * where first param is the AffiliateID and the second param is the phoneticsID
 */
tVoid fc_sxm_tclPhoneticsApp::vHandleTTFIsCmdGetChannelNames(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    //Handle for TTFI's cmd GetAffiliateDetails
    midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodStart oMStart;
    oMStart.PhoneticsType = prMsg->au8Data[0];
    oMStart.LanguageType = prMsg->au8Data[1];
    fc_sxm_vSendSelfMessage(oMStart);
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vPrintReportSpecific                                          *
 *                                                                                 *
 *DESCRIPTION:       Method prints the status/values stored for phonetics service. *
 *
 *PARAMETERS:        None                                                          *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vPrintReportSpecific()
{
    // for debugging purpose
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t Phonetics specific data:"));

}

tVoid fc_sxm_tclPhoneticsApp::vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const */*prMsg*/)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::fc_sxm_trMsgDataSmsEvtServiceUp"));

    //TODO : currently support only for Channel type
    tU8 ePhoneType = midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_CHANNELS;

    //all Language combination
    tU8 eLangType = (midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_ENGLISH
            | midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_SPANISH
            | midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_FRENCH);

    fc_sxm_tcl_trPhoneticsPropertyNewUpdatePhoneticsInfoListStatus oProperty;
    midw_ext_sxm_phoneticsfi_tclMsgPhoneticsUpdateStatus &oFiMsg =oProperty.oFiMsg;
    oFiMsg.PhoneticsType = ePhoneType;
    oFiMsg.LanguageType = eLangType;

    ETG_TRACE_USR4(("fc_sxm_trMsgDataSmsEvtServiceUp: Phoneme type: %d Language type: %d",ePhoneType,eLangType));

    /*Send FI Status*/
    fc_sxm_tclPhonetics_properties::instance()->oNewUpdatePhoneticsInfoListStatus.vSet(oProperty);
    fc_sxm_tclPhonetics_properties::instance()->oNewUpdatePhoneticsInfoListStatus.vNotify();
}
/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vProcess                                                      *
 *                                                                                 *
 *DESCRIPTION:       vProcess for Property update by SMS file event callback       *
 *                                                                                 *
 *PARAMETERS:        fc_sxm_trMsgPhoneticsSMSFileEventCallback                     *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vProcess(fc_sxm_trMsgPhoneticsSMSFileEventCallback const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vProcess"
            "(fc_sxm_trMsgPhoneticsSMSFileEventCallback)"));

    fc_sxm_tcl_trPhoneticsPropertyNewUpdatePhoneticsInfoListStatus oProperty;
    midw_ext_sxm_phoneticsfi_tclMsgPhoneticsUpdateStatus &oFiMsg = oProperty.oFiMsg;
    oFiMsg = prMsg->oFiMsg;
    if(_pPhoneticsBase != OSAL_NULL)
    {
        //Convert phonetics type and language
        _pPhoneticsBase->vConvertPhoneticsInfo(oFiMsg);
    }

    /*Send FI Status*/
    fc_sxm_tclPhonetics_properties::instance()->oNewUpdatePhoneticsInfoListStatus.vSet(oProperty);
    fc_sxm_tclPhonetics_properties::instance()->oNewUpdatePhoneticsInfoListStatus.vNotify();
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vProcess                                                      *
 *                                                                                 *
 *DESCRIPTION:       vProcess for method start by SDS for phonetics data           *
 *                                                                                 *
 *PARAMETERS:        fc_sxm_trMsgDataMStartGetPhoneticsData                        *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tclPhoneticsApp::vProcess(fc_sxm_trMsgDataMStartGetPhoneticsData const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclPhoneticsApp::vProcess"
            "(fc_sxm_trMsgDataMStartGetPhoneticsData)"));

    midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodResult ofiTxObj;
    /* Set Phonetics type and Language type in method result */
    (ofiTxObj).PhoneticsType = (prMsg->oFiMsg).PhoneticsType;
    (ofiTxObj).LanguageType = (prMsg->oFiMsg).LanguageType;

    if(_pPhoneticsBase != OSAL_NULL)
    {
        /* select phonetics type */
        _pPhoneticsBase->vGetPhoneticsData(ofiTxObj);

    }
    /* Send the method result to our service*/
    fc_sxm_tclPhoneticsService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj);

}
