/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_parking_manager.cpp
* @brief       class to manage parking list objects.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_parking_manager.h"
#include "fc_sxm_tcl_parking_factory.h"
#include "fc_sxm_tcl_parking_base.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PARKING_APP
#include "trcGenProj/Header/fc_sxm_tcl_parking_manager.cpp.trc.h"
#endif

#define SXM_FOREACH_PARKING_LIST(u8ListID)                           \
    for(u8ListID=midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY;  \
            u8ListID<=midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE; \
            ++u8ListID)

/*
 * Constructor for the class
 */
fc_sxm_tclParkingManager::fc_sxm_tclParkingManager():
    _poParkingFactory(OSAL_NULL),
    _poParkingApp(OSAL_NULL),
    _poSDKProxy(OSAL_NULL)

{
    //Default constructor
    _mapParkingList.clear();
}

/*
 * Destructor for the class
 */
fc_sxm_tclParkingManager::~fc_sxm_tclParkingManager()
{
    //TODO: Destructor for the class
    vClearParkingList();
    //_poParkingFactory is already destructed in method vDeInitParkingManager()
    _poParkingFactory = OSAL_NULL;
    _poParkingApp = OSAL_NULL;
    _poSDKProxy = OSAL_NULL;
}

/*
 * Method gets the instance of Parking Object Factory
 */
tVoid fc_sxm_tclParkingManager::vInitParkingManager(fc_sxm_tclIParkingApp* poApp,fc_sxm_tclIParkingSdkProxy* poSDKProxy)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vInitParkingManager"));
    // Get instance of the parking factory
    _poParkingFactory = fc_sxm_tclParkingFactory::instance();
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingFactory);

    //initialize members
    _poParkingApp = poApp;
    _poSDKProxy = poSDKProxy;
}

/*
 * Method destroys the instance of Parking Object Factory
 */
tVoid fc_sxm_tclParkingManager::vDeInitParkingManager()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vDeInitParkingManager"));
    // Delete instance of parking factory
    FC_SXM_MEMORY_CLEAR(_poParkingFactory)
}
/*
 * Method requests Parking Object factory to create one object for every parking
 */
tVoid fc_sxm_tclParkingManager::vCreateParkingList()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vCreateParkingList() Entered"));

    tU8 u8ListID;
    SXM_FOREACH_PARKING_LIST(u8ListID)
    {

        SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingFactory);
        //Map parkingID with the Parking Object class
        fc_sxm_tclIParkingBase* poParkingList = _poParkingFactory->poGetParkingList(u8ListID) ;
        if (OSAL_NULL != poParkingList)
        {
            poParkingList->vSetApp(_poParkingApp,_poSDKProxy);
            // Allow only permissible parking objects to be inserted to parking map
            _mapParkingList[u8ListID] = poParkingList;
        }
        ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vCreateParkingList() Exit"));
    }
}
/*
 * Method requests Parking Object factory to create one object for every parking
 */
tVoid fc_sxm_tclParkingManager::vRemoveRequests() const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vRemoveRequests() Entered"));

    tU8 u8ListID;
    SXM_FOREACH_PARKING_LIST(u8ListID)
    {
        //Map parkingID with the Parking Object class
        fc_sxm_tclIParkingBase* poParkingList = poGetListByID(u8ListID);
        if (OSAL_NULL != poParkingList)
        {
            poParkingList->vRemoveRequest();
        }
        ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vRemoveRequests() Exit"));
    }
}

/*
 * Method requests Parking Object factory to create one object for every parking
 */
tVoid fc_sxm_tclParkingManager::vSetFilterConfig(const midw_ext_sxm_parkingfi_tclMsgSetFilterConfigMethodStart &oMStart) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vSetFilterConfig() Entered"));

    tU8 u8ListID;
    SXM_FOREACH_PARKING_LIST(u8ListID)
    {
        //Map parkingID with the Parking Object class
        fc_sxm_tclIParkingBase* poParkingList = poGetListByID(u8ListID);
        if (OSAL_NULL != poParkingList)
        {
            poParkingList->vSetFilterConfig(oMStart);
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vSetFilterConfig() Exit"));
}
/*
 * Method requests Parking Object factory to create one object for every parking
 */
tVoid fc_sxm_tclParkingManager::vSetFilterConfig(const midw_ext_sxm_parkingfi_tclMsgSetFilterConfigurationMethodStart &oMStart) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vSetFilterConfiguration() Entered"));

    tU8 u8ListID;
    SXM_FOREACH_PARKING_LIST(u8ListID)
    {
        //Map parkingID with the Parking Object class
        fc_sxm_tclIParkingBase* poParkingList = poGetListByID(u8ListID);
        if (OSAL_NULL != poParkingList)
        {
            poParkingList->vSetFilterConfig(oMStart);
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vSetFilterConfiguration() Exit"));
}
/*
 * Method to return a list pointer for a listID
 */
fc_sxm_tclIParkingBase* fc_sxm_tclParkingManager::poGetListByID(tU8 u8ListID) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::poGetListByID u8ListID=%d",u8ListID));

    SXM_IF_FIND_MAP_CONST(tU8, fc_sxm_tclIParkingBase*, cIT, _mapParkingList, u8ListID)
    {
        return cIT->second;
    }

    return OSAL_NULL;
}
/*
 * Utillity Method to clear the list of all parking
 */
tVoid fc_sxm_tclParkingManager::vClearParkingList(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vClearParkingList() Entered\t parkingList size = %u", _mapParkingList.size()));
    //All data stored in
    SXM_FOREACH_MAP_CONST(tU8, fc_sxm_tclIParkingBase* , cIT, _mapParkingList)
    {
        delete (cIT->second);
    }
    _mapParkingList.clear();

    ETG_TRACE_USR4(("fc_sxm_tclParkingManager::vClearParkingList() Exit\t parkingList size = %u", _mapParkingList.size()));
}
