/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_parking_filterconfig.cpp
* @brief       Class to handle filter configuration from HMI
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_parking_filterconfig.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PARKING_FILTERCONFIG
#include "trcGenProj/Header/fc_sxm_tcl_parking_filterconfig.cpp.trc.h"
#endif

#define SXM_TRUE_IF_EQ(var) \
        ((var == b.var)? TRUE : FALSE) \

#define SXM_TRUE_IF_UNKNOWN(type,var) \
        ((var.enType == type::FI_EN_UNKNOWN)? TRUE : FALSE) \

#define SXM_TRUE_IF_PARKING_STATUS_UNKNOWN(type,var) \
        ((var.enType == type::FI_EN_PARKING_STATUS_UNKNOWN)? TRUE : FALSE) \

/****************************************************************************
 *
 *FUNCTION:          fc_sxm_trParkingFilterConfig
 *
 *DESCRIPTION:       Constructor
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_trParkingFilterConfig::fc_sxm_trParkingFilterConfig()
{
    vClearMemberData();
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig constructor"));
}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_trParkingFilterConfig
 *
 *DESCRIPTION:       destructor
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_trParkingFilterConfig::~fc_sxm_trParkingFilterConfig()
{
    vClearMemberData();
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig destructor"));
}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_trParkingFilterConfig
 *
 *DESCRIPTION:       function to clear internal member data
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_trParkingFilterConfig::vClearMemberData()
{
    _enCash.enType = midw_ext_fi_tcl_e8CashOrCard::FI_EN_UNKNOWN;
    _enCard.enType = midw_ext_fi_tcl_e8CashOrCard::FI_EN_UNKNOWN;
    _enSelfParking.enType = midw_ext_fi_tcl_e8SelfParking::FI_EN_UNKNOWN;
    _enParkingType.enType = midw_ext_fi_tcl_e8ParkingType::FI_EN_UNKNOWN;
    _enOpenStatus.enType = midw_ext_fi_tcl_e8OpenStatus::FI_EN_PARKING_STATUS_UNKNOWN;
    _enSUVCharges.enType = midw_ext_fi_tcl_e8SUVCharges::FI_EN_UNKNOWN;

    _oFilterConfig.FilterConfig.Amenities.Card.clear();
    _oFilterConfig.FilterConfig.Amenities.SelfParking.clear();
    _oFilterConfig.FilterConfig.Amenities.ParkingType.clear();
    _oFilterConfig.FilterConfig.Amenities.SUVCharges.clear();

    _oFilterConfig.FilterConfig.ClosedLocations.clear();
    _u32Height = 0;
}
/****************************************************************************
 *
 *FUNCTION:          bFilterParking
 *
 *DESCRIPTION:       method to check if parking location can be filtered
 *
 *PARAMETERS:        fc_sxm_trParkingFilterConfig - parking location values
 *
 *RETURNVALUES:      bool
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bFilterParkingStation(fc_sxm_trParkingFilterConfig &b)
{
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bFilterParkingStation"));

    //Check if Cash is filtered by user
    if(((!(SXM_TRUE_IF_UNKNOWN(midw_ext_fi_tcl_e8CashOrCard,_enCash))) &&  SXM_TRUE_IF_EQ(_enCash))
            && ((!(SXM_TRUE_IF_UNKNOWN(midw_ext_fi_tcl_e8CashOrCard,_enCard))) &&  SXM_TRUE_IF_EQ(_enCard)) )
    {
        return TRUE;
    }
    //Check if Parking Type is filtered by user
    if((!(SXM_TRUE_IF_UNKNOWN(midw_ext_fi_tcl_e8ParkingType,_enParkingType))) &&  SXM_TRUE_IF_EQ(_enParkingType))
    {
        return TRUE;
    }
    //Check if SelfParking is filtered by user
    if((!(SXM_TRUE_IF_UNKNOWN(midw_ext_fi_tcl_e8SelfParking,_enSelfParking))) &&  SXM_TRUE_IF_EQ(_enSelfParking))
    {
        return TRUE;
    }
    //Check if OpenStatus is filtered by user
    if((!(SXM_TRUE_IF_PARKING_STATUS_UNKNOWN(midw_ext_fi_tcl_e8OpenStatus,_enOpenStatus))) &&  SXM_TRUE_IF_EQ(_enOpenStatus))
    {
        return TRUE;
    }
    //Check if SUV Charges is filtered by user
    if((!(SXM_TRUE_IF_UNKNOWN(midw_ext_fi_tcl_e8SUVCharges,_enSUVCharges))) &&  SXM_TRUE_IF_EQ(_enSUVCharges))
    {
        return TRUE;
    }

    //Check if user has set minimum entrance
    if((!(_u32Height == 0)) && (!(b._u32Height == 0)) && (b._u32Height < _u32Height))
    {
        return TRUE;
    }

    ETG_TRACE_USR4(("bFilterParkingStation - don't filter parking location..."));

    //OK,not matching any of above
    //dont filter particular station
    return FALSE;
}
/****************************************************************************
 *
 *FUNCTION:          bIsParkingStationAllowed
 *
 *DESCRIPTION:       method to check if parking location can be filtered
 *
 *PARAMETERS:        midw_ext_fi_tcl_AmenityInfo - parking location amenety info
 *PARAMETERS:        midw_ext_fi_tcl_e8OpenStatus - Parking Location Open Status
 *PARAMETERS:        tU32 - Parking Location Primary Height
 *
 *RETURNVALUES:      TRUE - Parking Allowed
 *RETURNVALUES:      FALSE - parking Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bIsParkingStationAllowed(midw_ext_fi_tcl_AmenityInfo &oAmenInfo,midw_ext_fi_tcl_e8OpenStatus &enOpenStatus,tU32 u32Height)
{
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bIsParkingStationAllowed"));

    //Check for all Filter Settings applied by User
    return  bCheckOpenStatus(enOpenStatus) &&
            bCheckLocationHeight(u32Height) &&
            bCheckOpen24Hours(oAmenInfo.Open24Hours) &&
            bCheckCash(oAmenInfo.Cash) &&
            bCheckCard(oAmenInfo.Card) &&
            bCheckParkingType(oAmenInfo.ParkingType) &&
            bCheckMetroLocation(oAmenInfo.MetroLocation) &&
            bCheckAirPort(oAmenInfo.AirPort) &&
            bCheckOvernightParking(oAmenInfo.OvernightParking) &&
            bCheckReservations(oAmenInfo.Reservations) &&
            bCheckSelfParking(oAmenInfo.SelfParking) &&
            bCheckSecurity(oAmenInfo.Security) &&
            bCheckTollTag(oAmenInfo.TollTag) &&
            bCheckSUVCharges(oAmenInfo.SUVCharges) &&
            bCheckRV(oAmenInfo.RV);
}
/****************************************************************************
 *
 *FUNCTION:          bCheckOpenStatus
 *
 *DESCRIPTION:       method to check for Open Status
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8OpenStatus
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckOpenStatus(midw_ext_fi_tcl_e8OpenStatus &enOpenStatus)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.ClosedLocations.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.ClosedLocations[tU32Index] == enOpenStatus)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bCheckOpenStatus bIsParkingLocationAllowed=%d",bIsParkingLocationAllowed));

    //Finally return the status
    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckLocationHeight
 *
 *DESCRIPTION:       method to check for Location Height
 *
 *PARAMETERS:        tU32
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckLocationHeight(tU32 u32Height)
{
    tBool bIsParkingLocationAllowed = FALSE;

    //On 2 Cases,parking location is allowed to send to HMI
    //1.If User Filter is OFF i.e. EntranceHeight will be zero(0) from HMI
    //2.If Parking location Primary Height is Higher than User selected Height
    if((_oFilterConfig.FilterConfig.EntranceHeight == 0) || (u32Height > _oFilterConfig.FilterConfig.EntranceHeight))
    {
        bIsParkingLocationAllowed = TRUE;
    }

    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bCheckLocationHeight bIsParkingLocationAllowed=%d",bIsParkingLocationAllowed));

    //finally return status
    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckOpen24Hours
 *
 *DESCRIPTION:       method to check for Open 24 hOurs settings
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8Open24Hours
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckOpen24Hours(midw_ext_fi_tcl_e8Open24Hours& enOpen24Hours)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.Open24Hours.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.Open24Hours[tU32Index] == enOpen24Hours)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckCash
 *
 *DESCRIPTION:       method to check for Cash Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8CashOrCard
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckCash(midw_ext_fi_tcl_e8CashOrCard& enCash)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.Cash.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.Cash[tU32Index] == enCash)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bCheckCash bIsParkingLocationAllowed=%d",bIsParkingLocationAllowed));

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckCard
 *
 *DESCRIPTION:       method to check for Card Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8CashOrCard
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckCard(midw_ext_fi_tcl_e8CashOrCard& enCard)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.Card.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.Card[tU32Index] == enCard)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckParkingType
 *
 *DESCRIPTION:       method to check for Parkingtype Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8ParkingType
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckParkingType(midw_ext_fi_tcl_e8ParkingType& enParkingType)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.ParkingType.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.ParkingType[tU32Index] == enParkingType)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bCheckParkingType bIsParkingLocationAllowed=%d",bIsParkingLocationAllowed));
    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckMetroLocation
 *
 *DESCRIPTION:       method to check for MetroLocation Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8MetroLocation
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckMetroLocation(midw_ext_fi_tcl_e8MetroLocation& enMetroLocation)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.MetroLocation.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.MetroLocation[tU32Index] == enMetroLocation)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckAirPort
 *
 *DESCRIPTION:       method to check for AirPort Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8AirPort
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckAirPort(midw_ext_fi_tcl_e8AirPort& enAirPort)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.AirPort.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.AirPort[tU32Index] == enAirPort)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckOvernightParking
 *
 *DESCRIPTION:       method to check for OvernightParking Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8OvernightParking
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckOvernightParking(midw_ext_fi_tcl_e8OvernightParking& enOvernightParking)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.OvernightParking.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.OvernightParking[tU32Index] == enOvernightParking)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckReservations
 *
 *DESCRIPTION:       method to check for Reservations Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8Reservations
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckReservations(midw_ext_fi_tcl_e8Reservations& enReservations)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.Reservations.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.Reservations[tU32Index] == enReservations)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckSelfParking
 *
 *DESCRIPTION:       method to check for SelfParking Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8SelfParking
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckSelfParking(midw_ext_fi_tcl_e8SelfParking& enSelfParking)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size =(tU32)(_oFilterConfig.FilterConfig.Amenities.SelfParking.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.SelfParking[tU32Index] == enSelfParking)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bCheckSelfParking bIsParkingLocationAllowed=%d",bIsParkingLocationAllowed));
    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckSecurity
 *
 *DESCRIPTION:       method to check for Security Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8Security
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckSecurity(midw_ext_fi_tcl_e8Security& enSecurity)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.Security.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.Security[tU32Index] == enSecurity)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckTollTag
 *
 *DESCRIPTION:       method to check for TollTag Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8TollTag
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckTollTag(midw_ext_fi_tcl_e8TollTag& enTollTag)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.TollTag.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.TollTag[tU32Index] == enTollTag)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckSUVCharges
 *
 *DESCRIPTION:       method to check for SUVCharges Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8SUVCharges
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckSUVCharges(midw_ext_fi_tcl_e8SUVCharges& enSUVCharges)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.SUVCharges.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.SUVCharges[tU32Index] == enSUVCharges)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig bCheckSUVCharges bIsParkingLocationAllowed=%d",bIsParkingLocationAllowed));
    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckRV
 *
 *DESCRIPTION:       method to check for RV Amenety
 *
 *PARAMETERS:        midw_ext_fi_tcl_e8RV
 *
 *RETURNVALUES:      TRUE - Allowed
 *RETURNVALUES:      FALSE - Not allowed
 *
 ******************************************************************************/
tBool fc_sxm_trParkingFilterConfig::bCheckRV(midw_ext_fi_tcl_e8RV& enRV)
{
    tBool bIsParkingLocationAllowed = FALSE;
    tU32 u32Size = (tU32)(_oFilterConfig.FilterConfig.Amenities.RV.size());

    //Check if list is Empty
    if(u32Size == 0)
    {
        //Then User has not Set the filter
        //So Don't filter this location
        bIsParkingLocationAllowed = TRUE;
    }
    else
    {
        //User has Set a Filter
        for(tU32 tU32Index = 0; tU32Index < u32Size; ++tU32Index)
        {
            //Check if User filter is matching to Parking Location
            if(_oFilterConfig.FilterConfig.Amenities.RV[tU32Index] == enRV)
            {
                //If So,Don't Filter this location
                bIsParkingLocationAllowed = TRUE;
            }
        }
    }

    return bIsParkingLocationAllowed;
}
/****************************************************************************
 *
 *FUNCTION:          vSetFilterConfig
 *
 *DESCRIPTION:       method to set filter configuration from HMI
 *
 *PARAMETERS:        defferent filter config
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_trParkingFilterConfig::vSetFilterConfig(midw_ext_fi_tcl_e8CashOrCard enCash,
        midw_ext_fi_tcl_e8CashOrCard enCard,
        midw_ext_fi_tcl_e8SelfParking enSelfParking,
        midw_ext_fi_tcl_e8ParkingType enParkingType,
        midw_ext_fi_tcl_e8OpenStatus enOpenStatus,
        midw_ext_fi_tcl_e8SUVCharges enSUVCharges,
        tU32 u32Height
)
{
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig vSetFilterConfig "
            "Cash=%u,Card=%u,SelfPark=%u,ParkingType=%u,OpenStatus=%u,SUV=%u,Height=%d",
            enCash.enType,enCard.enType,enSelfParking.enType,enParkingType.enType,
            enOpenStatus.enType,enSUVCharges.enType,u32Height));

    _enCash = enCash;
    _enCard = enCard;
    _enSelfParking = enSelfParking;
    _enParkingType = enParkingType;
    _enOpenStatus = enOpenStatus;
    _enSUVCharges = enSUVCharges;
    _u32Height = u32Height;

}
/****************************************************************************
 *
 *FUNCTION:          vSetFilterConfig
 *
 *DESCRIPTION:       method to set filter configuration from HMI
 *
 *PARAMETERS:        defferent filter config
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_trParkingFilterConfig::vSetFilterConfig(const midw_ext_sxm_parkingfi_tclMsgSetFilterConfigurationMethodStart &oMStart)
{
    ETG_TRACE_USR4(("fc_sxm_trParkingFilterConfig vSetFilterConfig "));

    _oFilterConfig = oMStart;

}

