/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_parking_app.cpp
* @brief       Parking application handler
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sms_init.h"
#include "fc_sxm_smse.h"
#include "fc_sxm_tcl_parking_app.h"
#include "fc_sxm_service_sxm_parking.h"
#include "fc_sxm_tcl_parking_sdk_proxy.h"
#include "fc_sxm_tcl_parking_properties.h"
#include "fc_sxm_tcl_parking_fi_msgs.h"
#include "fc_sxm_tcl_parking_manager.h"
#include "fc_sxm_tcl_parking_base.h"
#include "fc_sxm_generic_utils.h"
#include "fc_sxm_tcl_parking_common.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PARKING_APP
#include "trcGenProj/Header/fc_sxm_tcl_parking_app.cpp.trc.h"
#endif

tBool fc_sxm_tclParkingApp::bCompareDistance(midw_ext_fi_tcl_ParkingInfo const& first,midw_ext_fi_tcl_ParkingInfo const& second)
{
    return first.Distance < second.Distance;
}

/****************************************************************************
 *
 *FUNCTION:          fc_sxm_tclParkingApp
 *
 *DESCRIPTION:       Constructor
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingApp::fc_sxm_tclParkingApp():
fc_sxm_tclDataApp("sxmParkingApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_PARKING, "parking", static_cast <tU16> (TR_CLASS_FC_SXM_PARKING_APP)),
_hParking(DATASERVICE_MGR_INVALID_OBJECT),
_poParkingManager(OSAL_NULL),
_poProxy(OSAL_NULL),
_rSDKDestLocation(0),
_rFrozenDevicePos(0)

{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp constructor"));
}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_tclParkingApp
 *
 *DESCRIPTION:       Destructor
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingApp::~fc_sxm_tclParkingApp()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp destructor"));

    _hParking = DATASERVICE_MGR_INVALID_OBJECT;

    _poParkingManager = OSAL_NULL;

    _poProxy = OSAL_NULL;
    _rSDKDestLocation.s32Latitude = 0;
    _rSDKDestLocation.s32Longitude = 0;

    _rFrozenDevicePos.s32Latitude = 0;
    _rFrozenDevicePos.s32Longitude = 0;

}
/****************************************************************************
 *
 *FUNCTION:          vInitialize
 *
 *DESCRIPTION:       called in cca-context by sxmapp-manager
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vInitialize"));

    //Create the parking worker thread
    fc_sxm_tclBaseApp::vInitialize();

    //instantiate proxy class
    _poProxy = fc_sxm_tclParkingSdkProxy::instance();
    _poParkingManager = fc_sxm_tclParkingManager::instance();

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    _poParkingManager->vInitParkingManager(this,_poProxy);

    //Create List required for parking
    vCreateParkingList();
}
/****************************************************************************
 *
 *FUNCTION:          vDeInitialize
 *
 *DESCRIPTION:       called in cca-context by sxmapp-manager
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vDeInitialize"));

    //remove any existing request
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    _poParkingManager->vRemoveRequests();

    //App is owner of proxy,so delete proxy object here
    FC_SXM_MEMORY_CLEAR(_poProxy)
    
	//delete Parking Manager
	OSAL_DELETE _poParkingManager;
	_poParkingManager = OSAL_NULL;

    // call vDeInitialize() of base-class to stop the worker thread
    fc_sxm_tclBaseApp::vDeInitialize();
}
/****************************************************************************
 *
 *FUNCTION:          vCreateParkingList
 *
 *DESCRIPTION:       Create Object for Parking List.
 *
 *PARAMETERS:       fc_sxm_tenParkingListType
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vCreateParkingList()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::CreateParkingList "));

    //Create Parking List
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    _poParkingManager->vCreateParkingList();


}

/****************************************************************************
 *
 *FUNCTION:          vDispatchMsgFromQ
 *
 *DESCRIPTION:       this method is called by the thread-function.
                    For each message that shall be handled put and entry here.
 *PARAMETERS:       fc_sxm_tclMessage
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) {

    // Get Action Value
    (tVoid)poThreadMsg; //to avoid lint warning

    tU32 u32Action=poThreadMsg->u32GetAction();


    switch (u32Action) {

    // dispatch Parking Service State mesaage to CCA
    // Also Do this for every method exposed thru FI
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartGetParkingList,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingSMSeRequestCallback,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingSMSePOICallback,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartSetParkingFilterConfig,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartSetParkingFilterConfiguration,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartSaveParkingID,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartRemoveParkingID,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartGetParkingInfo,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingMStartRestoreFavorites,this);
    SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgParkingPSetListMode,this);

    default:
        ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                u32Action));
        break;
    }

}
/****************************************************************************
 *
 *FUNCTION:          vStartSmsService
 *
 *DESCRIPTION:       Method to start SMSe service
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vStartSmsService()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vStartSmsService"));

    SXM_ASSERT_RETURN(OSAL_NULL!=_poProxy);

    tS32 s32Debuglevel = 0;
    tS32 s32ParkingStatus ;

    //Change DS state to Initial
    fc_sxm_trMsgDataSmsEvtServiceState rInitMsg;
	//Transition to INITIAL state by default
    rInitMsg.eState = DATASERVICE_STATE_INITIAL;
    rInitMsg.eProductState = DATA_PRODUCT_STATE_READY;
    rInitMsg.enErrorCode = DATASERVICE_ERROR_CODE_NONE;
	fc_sxm_tclDataApp::vProcessGenericMsg(&rInitMsg);

    //Method invokes SMSe to start parking service. The DebugLevel for SMS app is currently set to 0
    s32ParkingStatus = _poProxy->s32StartService(cb_vDataParkingServiceCallback,s32Debuglevel);

    //Check if it is success
    if (s32ParkingStatus == (tS32) SXM_E_OK )
    {
        _hParking = DATASERVICE_MGR_OBJECT(SXM_E_STATE);
    }
    //Error in starting Parking Service
    else
    {
        _hParking = DATASERVICE_MGR_INVALID_OBJECT;
    	fc_sxm_trMsgDataSmsEvtServiceState rMsg;

        ETG_TRACE_USR4(("fc_sxm_tclParkingApp::sxm_parking starts failed:%d",s32ParkingStatus));

        //Change DS state to Initial
        rMsg.eState = DATASERVICE_STATE_ERROR;

        //Check if it because of baseline database.
        if (s32ParkingStatus == (tS32) SXM_E_NO_DB)
        {
        	rMsg.enErrorCode = DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND;
        }
        else if(s32ParkingStatus == (tS32) SXM_E_BAD_DB )
        {
            rMsg.enErrorCode = DATASERVICE_ERROR_CODE_DATABASE_CORRUPT;
        }

        //finally call base function to handle new state
    	fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);
    }
}
/****************************************************************************
 *
 *FUNCTION:          vStopSmsService
 *
 *DESCRIPTION:       Method to stop SMSe service
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp:: vStopSmsService()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vStopSmsService"));

    //Clear internal Data
    _bSmsServiceStopping=TRUE;
    vClearServiceData();
    vAllowService(FALSE);

    if(_hParking != DATASERVICE_MGR_INVALID_OBJECT)
    {
		//Check pointers for NULL
		SXM_ASSERT_RETURN(OSAL_NULL!=_poProxy);
		//Method invokes SMSe to stop parking service.
		tS32 s32ParkingStatus =  _poProxy->s32StopService();

		ETG_TRACE_USR4(("fc_sxm_tclParkingApp::sxm_parking stop status:%d",s32ParkingStatus));
    }

		//send the new list status to clients
		vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY);
		vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION);
		vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

		//The updated state is passed back to DataApp
		fc_sxm_trMsgDataSmsEvtServiceState rMsg;
		rMsg.eState = DATASERVICE_STATE_STOPPED;
		rMsg.eProductState = DATA_PRODUCT_STATE_READY;
		rMsg.enErrorCode = DATASERVICE_ERROR_CODE_NONE;

		fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);

}
/****************************************************************************
 *
 *FUNCTION:          vClearServiceData
 *
 *DESCRIPTION:       To clear all the stored data during reinitialisation of SMS.
                    This is a overridden method from the data app.
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vClearServiceData()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vClearServiceData(%s)",
            szGetName()));

    //Clear Favorite entries if any
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);

    fc_sxm_tclIParkingBase* poGetListById = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

    if(poGetListById != OSAL_NULL)
    {
    	poGetListById->vClearPOIList();
    }


}
/****************************************************************************
 *
 *FUNCTION:          vOnFiServiceStateChanged
 *
 *DESCRIPTION:       Application get notified about the new fi-service-state
 *DESCRIPTION:       that shall be sent to clients
 *PARAMETERS:        midw_ext_fi_tcl_SxmDataSrvState
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
/*

*/
tVoid fc_sxm_tclParkingApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState) {
    fc_sxm_tcl_trParkingPropertyServiceDataStatus oProperty;

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vOnFiServiceStateChanged Service State = %d",
                    ETG_CENUM(midw_ext_fi_tcl_e8_SxmDataServiceStatus::tenType, oFiState.Status.enType) ));

    oProperty.oFiMsg.DataServiceState=oFiState;
    fc_sxm_tclParkingProperties::instance()->oParkingServiceDataStatus.vSetAndNotify(oProperty);
}
/****************************************************************************
 *
 *FUNCTION:          vProcessBaseAppMsg
 *
 *DESCRIPTION:       Method will be called whenver Parking Service Ready
 *
 *PARAMETERS:       Msg
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const *prMsg)
{

    (tVoid) prMsg ;

    //we should start request for the current position
    vOnNewDevicePosition(const_cast<fc_sxm_trSDKLocation&>(rGetSDKDeviceLocation()));

    //and destination
    vCreatePOIList();
}
/*
   Method Invoked for every change in clock details
*/
tVoid fc_sxm_tclParkingApp::vSetClockParams(fc_sxm_trMsgDataAppSetClockParams const *prMsg){

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vSetClockParams fc_sxm_trMsgDataAppSetClockParams WeekDay=%d",prMsg->oLocalTime.u8Weekday));

    //Copy to locally
    _oLocalTime = prMsg->oLocalTime;

}
/****************************************************************************
 *
 *FUNCTION:          vProcessBaseAppMsg
 *
 *DESCRIPTION:       handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd (Mandatory)
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
    // Method to handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) Entered code = %u",
            prMsg->u8MsgCode));

    tVoid (fc_sxm_tclParkingApp::*funcPtrArray[]) (fc_sxm_trMsgCmdAppTtfisCmd const *) const = {
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdParkingList,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdParkingList,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdSetParkingFilter,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdSetParkingInfo,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdSaveParkingID,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdRemoveParkingID,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdRestoreFavorites,
            &fc_sxm_tclParkingApp::vHandleTTFIsCmdSetListMode
    };

    //Store the values read for each index to the appropriate field in fc_sxm_trParkingList structure
    (this->*funcPtrArray[prMsg->u8MsgCode])(prMsg);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdParkingList
 *
 *DESCRIPTION:       Method to handle TTFis command for Parking List
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdParkingList(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodStart oMStart;
    oMStart.ListID = prMsg->au8Data[0];
    //Send Self message
    fc_sxm_vSendSelfMessage(oMStart);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdSetParkingFilter
 *
 *DESCRIPTION:       Method to handle TTFis command for Setting Filter Config
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdSetParkingFilter(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vHandleTTFIsCmdSetParkingFilter"));
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgSetFilterConfigurationMethodStart oMStart;

    const tU8 u8InvalidType = 5;

    //Copy Amen Info for Cash
    if(prMsg->au8Data[0] < u8InvalidType)
    {
        midw_ext_fi_tcl_e8CashOrCard rCash;
        rCash.enType = static_cast<midw_ext_fi_tcl_e8CashOrCard::tenType>(prMsg->au8Data[0]);
        oMStart.FilterConfig.Amenities.Cash.push_back(rCash);
    }

    //Copy Amen Info for PArking Type
    if(prMsg->au8Data[1] < u8InvalidType)
    {
        midw_ext_fi_tcl_e8ParkingType rParkingType;
        rParkingType.enType = static_cast<midw_ext_fi_tcl_e8ParkingType::tenType>(prMsg->au8Data[1]);
        oMStart.FilterConfig.Amenities.ParkingType.push_back(rParkingType);
    }

    //Copy Amen Info for Self parking
    if(prMsg->au8Data[2] < u8InvalidType)
    {
        midw_ext_fi_tcl_e8SelfParking rSelfParking;
        rSelfParking.enType = static_cast<midw_ext_fi_tcl_e8SelfParking::tenType>(prMsg->au8Data[2]);
        oMStart.FilterConfig.Amenities.SelfParking.push_back(rSelfParking);
    }

    //Copy Amen Info for SUVCharges
    if(prMsg->au8Data[3] < u8InvalidType)
    {
        midw_ext_fi_tcl_e8SUVCharges rSUVCharges;
        rSUVCharges.enType = static_cast<midw_ext_fi_tcl_e8SUVCharges::tenType>(prMsg->au8Data[3]);
        oMStart.FilterConfig.Amenities.SUVCharges.push_back(rSUVCharges);
    }

    //Copy Amen Info for Open Status
    if(prMsg->au8Data[4] < u8InvalidType)
    {
        midw_ext_fi_tcl_e8OpenStatus rOpenStatus;
        rOpenStatus.enType = static_cast<midw_ext_fi_tcl_e8OpenStatus::tenType>(prMsg->au8Data[4]);
        oMStart.FilterConfig.ClosedLocations.push_back(rOpenStatus);
    }

   oMStart.FilterConfig.EntranceHeight = SXM_GET_U32(&prMsg->au8Data[5]);

   ETG_TRACE_USR4(("oMStart.FilterConfig.EntranceHeight=%d",oMStart.FilterConfig.EntranceHeight));

   //Send Self message
   fc_sxm_vSendSelfMessage(oMStart);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdSetParkingInfo
 *
 *DESCRIPTION:       Method to handle TTFis command for Parking Info
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdSetParkingInfo(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vHandleTTFIsCmdSetParkingInfo"));
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgGetParkingStationInfoMethodStart oMStart;

    oMStart.ListID = SXM_GET_U32(&prMsg->au8Data[0]);
    oMStart.ParkingID =SXM_GET_U32(&prMsg->au8Data[4]);

    //Send Self message
    fc_sxm_vSendSelfMessage(oMStart);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdSaveParkingID
 *
 *DESCRIPTION:       Method to handle TTFis command for Saving parkingID
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdSaveParkingID(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vHandleTTFIsCmdSaveParkingID"));
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgAddFavoriteMethodStart oMStart;

    oMStart.ParkingID =SXM_GET_U32(&prMsg->au8Data[0]);

    //Send Self message
    fc_sxm_vSendSelfMessage(oMStart);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdRemoveParkingID
 *
 *DESCRIPTION:       Method to handle TTFis command for removing parkingID
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdRemoveParkingID(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vHandleTTFIsCmdRemoveParkingID"));
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgRemoveFavoriteMethodStart oMStart;

    oMStart.ParkingID =SXM_GET_U32(&prMsg->au8Data[0]);

    //Send Self message
    fc_sxm_vSendSelfMessage(oMStart);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdRestoreFavorites
 *
 *DESCRIPTION:       Method to handle TTFis command for restoring default favorites
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdRestoreFavorites(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vHandleTTFIsCmdRestoreFavorites"));
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgRestoreFavoritesMethodStart oMStart;

    oMStart.FavoriteList.push_back(SXM_GET_U32(&prMsg->au8Data[0]));
    oMStart.FavoriteList.push_back(SXM_GET_U32(&prMsg->au8Data[4]));
    oMStart.FavoriteList.push_back(SXM_GET_U32(&prMsg->au8Data[8]));

    //Send Self message
    fc_sxm_vSendSelfMessage(oMStart);
}
/****************************************************************************
 *
 *FUNCTION:          vHandleTTFIsCmdSetListMode
 *
 *DESCRIPTION:       Method to handle TTFis command to Set List mode
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vHandleTTFIsCmdSetListMode(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vHandleTTFIsCmdSetListMode"));
    //just prepare internal message
    midw_ext_sxm_parkingfi_tclMsgSxmListModeSet oPSet;

    oPSet.ListMode.ListID =(tU32)prMsg->au8Data[0];
    //int iMode = au8Data[1];
    oPSet.ListMode.Mode.enType = (midw_ext_fi_tcl_e8_SxmListMode::tenType)prMsg->au8Data[1];

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::ttfis-cmd PSetListId ListID=%d",
                    oPSet.ListMode.ListID
                    ));
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::ttfis-cmd PSetListMode ListMode=%d",
                    oPSet.ListMode.Mode.enType
                    ));

    fc_sxm_vSendSelfMessage(oPSet);
}
/****************************************************************************
 *
 *FUNCTION:          vPrintReportSpecific
 *
 *DESCRIPTION:       Method prints the status/values stored for parking service
 *
 *PARAMETERS:       fc_sxm_trMsgCmdAppTtfisCmd
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vPrintReportSpecific()
{
    // for debugging purpose
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t Parking specific data:"));
}
/****************************************************************************
 *
 *FUNCTION:          vPostInternalMsg
 *
 *DESCRIPTION:       Method post msg to internal message queue. The msg would be acted upon from SMSe parking thread context
 *
 *PARAMETERS:       fc_sxm_trMsgParkingSMSeRequestCallback
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vPostInternalMsg(const fc_sxm_trMsgParkingSMSeRequestCallback& rReqCB)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp vPostInternalMsg"));
    vPostMsgNew(rReqCB);
}
/****************************************************************************
 *
 *FUNCTION:          vPostInternalMsg
 *
 *DESCRIPTION:       Method post msg to internal message queue. The msg would be acted upon from SMSe parking(POI) thread context
 *
 *PARAMETERS:       fc_sxm_trMsgParkingSMSePOICallback
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vPostInternalMsg(const fc_sxm_trMsgParkingSMSePOICallback& rReqCB)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp vPostInternalMsg fc_sxm_trMsgParkingSMSePOICallback "));
    vPostMsgNew(rReqCB);
}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process Parking List message
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartGetParkingList
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartGetParkingList const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartGetParkingList) Entered ListID=%d",prMsg->oFiMsg.ListID));

    midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult oMRes;
    oMRes.ListID = prMsg->oFiMsg.ListID;

    //Send the List to HMI
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID((tU8)(oMRes.ListID));

    if(poParkingList != OSAL_NULL)
    {
        tS32 s32ListStatus = poParkingList->s32GetParkingList(oMRes);

        if(s32ListStatus == SXM_E_OK)
        {
            if(prMsg->oFiMsg.ListID == (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION)
            {
                vSortListByDistance(oMRes);
            }
            else if(prMsg->oFiMsg.ListID == (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE)
            {
                vSortListByUserOrder(oMRes);
            }

            //else if freeze mode is set,update the list with new device position
            if(poParkingList->trGetListMode().enType == midw_ext_fi_tcl_e8_SxmListMode::FI_EN_FREEZE)
            {
            	vUpdateDistance(oMRes);
            }

            //for debuggig purpose only
            poParkingList->vPrintParkingList(oMRes);
        }
    }
    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartGetParkingList) Exit"));
}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process Parking Info message
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartGetParkingInfo
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartGetParkingInfo const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartGetParkingInfo) Entered"));

    midw_ext_sxm_parkingfi_tclMsgGetParkingStationInfoMethodResult oMRes;
    oMRes.ListID = prMsg->oFiMsg.ListID;
    midw_ext_fi_tcl_ParkingInfo &ParkingStationInfo = oMRes.ParkingStationInfo;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID((tU8)(oMRes.ListID));

    if(poParkingList != OSAL_NULL)
    {
        //Send the List to HMI
        tS32 s32ListStatus = poParkingList->s32GetParkingInfo(prMsg->oFiMsg.ParkingID,ParkingStationInfo);

        if(s32ListStatus == SXM_E_OK)
        {
            //for debuggig purpose only
            poParkingList->vPrintParkingInfo(ParkingStationInfo);
        }
    }
    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartGetParkingInfo) Exit"));
}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process SetFilterConfig request from HMI
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartSetParkingFilterConfig
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartSetParkingFilterConfig const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartSetParkingFilterConfig) Entered"));

    midw_ext_sxm_parkingfi_tclMsgSetFilterConfigMethodResult oMRes;
    oMRes.Status = TRUE;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
     _poParkingManager->vSetFilterConfig(prMsg->oFiMsg);

    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process SetFilterConfig request from HMI
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartSetParkingFilterConfig
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartSetParkingFilterConfiguration const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartSetParkingFilterConfiguration) Entered"));

    midw_ext_sxm_parkingfi_tclMsgSetFilterConfigurationMethodResult oMRes;
    oMRes.Status = TRUE;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
     _poParkingManager->vSetFilterConfig(prMsg->oFiMsg);

    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process Save Favorite request from HMI
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartSaveParkingID
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartSaveParkingID const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartSaveParkingID) Entered ParkingID=%d",prMsg->oFiMsg.ParkingID));

    midw_ext_sxm_parkingfi_tclMsgAddFavoriteMethodResult oMRes;
    tBool bStatus = FALSE;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

    if(poParkingList != OSAL_NULL)
    {
        //add this to our internal list
        bStatus = poParkingList->bAddFavorite(prMsg->oFiMsg.ParkingID);

        if(bStatus)
        {
            vector<PPOISpec> vecCombinedPPOIs;
            vGetCombinedPPOIList(vecCombinedPPOIs);

            //finally Set POI list as PPOI
            poParkingList->s32SetPPOIList(vecCombinedPPOIs);
        }
    }
    oMRes.Status = bStatus;
    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process remove Favorite request from HMI
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartRemoveParkingID
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartRemoveParkingID const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartRemoveParkingID) Entered ParkingID=%d",prMsg->oFiMsg.ParkingID));

    midw_ext_sxm_parkingfi_tclMsgRemoveFavoriteMethodResult oMRes;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poFavParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);
    tBool bStatus = FALSE;
    if(poFavParkingList != OSAL_NULL)
    {
        //POI entry bit should be cleared
        poFavParkingList->vClearPOIUserFlag(prMsg->oFiMsg.ParkingID);

        vector<PPOISpec> vecCombinedPPOIs;
        vGetCombinedPPOIList(vecCombinedPPOIs);

        //finally Set POI list as PPOI
        poFavParkingList->s32SetPPOIList(vecCombinedPPOIs);

        bStatus = poFavParkingList->bRemoveFavorite(prMsg->oFiMsg.ParkingID);
    }
    oMRes.Status = bStatus;
    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process Restore favorites request form HMI
 *
 *PARAMETERS:       fc_sxm_trMsgParkingMStartRestoreFavorites
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartRestoreFavorites const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingMStartRestoreFavorites) Entered"));

    midw_ext_sxm_parkingfi_tclMsgRestoreFavoritesMethodResult oMRes;
    tBool bStatus = FALSE;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

    if(poParkingList != OSAL_NULL)
    {
        //add this to our internal list
        SXM_FOREACH_CONST(vector<tU32>,itr,prMsg->oFiMsg.FavoriteList)
        {
            if(poParkingList->bAddFavorite(*itr))
            {
                bStatus = TRUE;
            }
        }

        //If we successfully added a entry to favorite list
        if(bStatus)
        {
            //prepare combined list with new favorite entry
            vector<PPOISpec> vecCombinedPPOIs;
            vGetCombinedPPOIList(vecCombinedPPOIs);

            //finally Set POI list to SDK
            poParkingList->s32SetPPOIList(vecCombinedPPOIs);
        }
    }
    oMRes.Status = bStatus;
    //finally send method result to HMI
    fc_sxm_tclParkingService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    //also send property update to HMI that favorite list ready to be fetched
    vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process List Mode request from HMI
 *
 *PARAMETERS:       fc_sxm_trMsgParkingPSetListMode
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingPSetListMode const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingPSetListMode) Entered"));

    fc_sxm_tcl_trParkingPropertySxmListMode oProperty;
    oProperty.oFiMsg.ListMode = prMsg->oFiMsg.ListMode;

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID((tU8)(prMsg->oFiMsg.ListMode.ListID));

    if(poParkingList != OSAL_NULL)
    {
        poParkingList->vSetListMode(prMsg->oFiMsg.ListMode.Mode);
    }
    //finally sending property update to HMI
    fc_sxm_tclParkingProperties::instance()->oSxmListMode.vSetAndNotify(oProperty);
}
/****************************************************************************
 *
 *FUNCTION:          vFreezeDevicePosition
 *
 *DESCRIPTION:       Method to Save Devisce position state
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vFreezeDevicePosition()
{
	_rFrozenDevicePos= rGetSDKDeviceLocation();
}
/****************************************************************************
 *
 *FUNCTION:          vReleaseDevicePosition
 *
 *DESCRIPTION:       Method to Resume setting position updates to SDK
 *
 *PARAMETERS:       None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vReleaseDevicePosition()
{
	vOnNewDevicePosition(const_cast<fc_sxm_trSDKLocation&>(rGetSDKDeviceLocation()));
}
/****************************************************************************
 *
 *FUNCTION:          vOnNewDevicePosition
 *
 *DESCRIPTION:       handle device position from Navi
 *
 *PARAMETERS:       fc_sxm_trSDKLocation
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vOnNewDevicePosition(fc_sxm_trSDKLocation& rLocation)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vOnNewDevicePosition long=%d,lat=%d",
            rLocation.s32Longitude,rLocation.s32Latitude));

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY);

    if(poParkingList != OSAL_NULL)
    {
    	//If particular list is released then only process new position
    	if(poParkingList->trGetListMode().enType == midw_ext_fi_tcl_e8_SxmListMode::FI_EN_RELEASE)
    	{
			//Set Device position recieved from Navi
			poParkingList->vSetPosition(rLocation.s32Longitude,rLocation.s32Latitude);

			//we can only Call Start Request If service is Ready
			//So check if Service is Ready
			if(enGetSmsServiceState() == DATASERVICE_STATE_READY)
			{
				//finally,Start or Modify request as per new position
				poParkingList->vStartRequest();

				vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY);
			}
    	}
    }

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vOnNewDevicePosition Exit"));
}
tVoid fc_sxm_tclParkingApp::vProcessGenericMsg(fc_sxm_trMsgDataAppSetDestLocation const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::fc_sxm_trMsgDataAppSetDestLocation lat=%d,lon=%d",
            prMsg->rLocation.s32Latitude,prMsg->rLocation.s32Longitude));

   /* Check If same Location is set again */
   if((_rSDKDestLocation.s32Latitude==prMsg->rLocation.s32Latitude) &&
            (_rSDKDestLocation.s32Longitude==prMsg->rLocation.s32Longitude))
    {
       ETG_TRACE_USR4(("fc_sxm_tclParkingApp::fc_sxm_trMsgDataAppSetDestLocation Same lat & lon"));
       return;
    }

   _rSDKDestLocation.s32Latitude = prMsg->rLocation.s32Latitude;
   _rSDKDestLocation.s32Longitude = prMsg->rLocation.s32Longitude;

   if(enGetSmsServiceState() == DATASERVICE_STATE_READY)
   {
       //finally Create POI list
       vCreatePOIList();
   }
}
/****************************************************************************
 *
 *FUNCTION:          vCreatePOIList
 *
 *DESCRIPTION:       Method to create Destination POI list
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vCreatePOIList()
{

	ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vCreatePOIList Entered"));

    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION);

    if(poParkingList != OSAL_NULL)
    {
        fc_sxm_trSDKLocation rDestLocation;
        //before we set to SDK,convert the location from SMS type to Nav type
        rDestLocation.s32Latitude = fc_sxm_tclGenericUtils::s32DegreeSmsToNav(_rSDKDestLocation.s32Latitude);
        rDestLocation.s32Longitude = fc_sxm_tclGenericUtils::s32DegreeSmsToNav(_rSDKDestLocation.s32Longitude);

        //Set new destination position recieved from Navi
        poParkingList->vSetPosition(rDestLocation.s32Longitude,rDestLocation.s32Latitude);

        //first clear user flag for old entries
        poParkingList->vClearPPOIUserFlag();

        //Refresh internal POI list based on new destination
        poParkingList->s32RefreshPOIList();

        vector<PPOISpec> vecCombinedPPOIs;
        vGetCombinedPPOIList(vecCombinedPPOIs);

        //finally Set POI list as PPOI
        poParkingList->s32SetPPOIList(vecCombinedPPOIs);

        //Send update to HMI
        vSendUpdate(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION);

    }
}
/****************************************************************************
 *
 *FUNCTION:          vGetCombinedPPOIList
 *
 *DESCRIPTION:       Method to prepare combined POI list from Favorites & Desdtination
 *
 *PARAMETERS:        vector<PPOISpec>&
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vGetCombinedPPOIList(vector<PPOISpec>& vecPPOIs)
{
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poFavParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);
    fc_sxm_tclIParkingBase* poDestParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION);

    //Check for NULL
    if(poFavParkingList == OSAL_NULL || poDestParkingList == OSAL_NULL)
    {
        return;
    }
    //First push Destination entries to combined list
    SXM_FOREACH_MAP_CONST(tU32,trPOIData,DestIter,poDestParkingList->mapGetPOIList())
    {
        PPOISpec rPOISpec;
        rPOISpec.ppoi_id = DestIter->first;
        //we are interested only in LSB
        rPOISpec.type_flags = (byte)(DestIter->second.u16Userflag);

        const map<tU32,trPOIData>& mapFavList = poFavParkingList->mapGetPOIList();
        //Check if any destination entry matched to favorite
        SXM_IF_FIND_MAP_CONST(tU32,trPOIData,FavIter, mapFavList,rPOISpec.ppoi_id)
        {
            //OK,just OR the both user flags with favortie flag.
            rPOISpec.type_flags = (byte)(((DestIter->second.u16Userflag | FavIter->second.u16Userflag)));
            ETG_TRACE_USR4(("PPOI=%d is common - userflag=%d",DestIter->first,DestIter->second.u16Userflag ));
        }
        //then push to combined list
        vecPPOIs.push_back(rPOISpec);
    }
    //Now,only push elements from favorite
    SXM_FOREACH_MAP_CONST(tU32,trPOIData,FavIter,poFavParkingList->mapGetPOIList())
    {
        tBool bNotExists = TRUE;
        //We have already added common entries b/w favorite & destination in above loop
        //so now just which are different b/w two list
        const map<tU32,trPOIData>& mapDestList = poDestParkingList->mapGetPOIList();
        SXM_IF_FIND_MAP_CONST(tU32,trPOIData,DestIter,mapDestList,FavIter->first)
        {
            bNotExists = FALSE;
        }
        //this entry is not common
        if(bNotExists)
        {
            PPOISpec rPOISpec;
            rPOISpec.ppoi_id = FavIter->first;
            rPOISpec.type_flags = (byte)(FavIter->second.u16Userflag);

            //push to combined list
            vecPPOIs.push_back(rPOISpec);
        }
    }
}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process Parking callabck message
 *
 *PARAMETERS:       fc_sxm_trMsgParkingSMSeRequestCallback
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingSMSeRequestCallback const */*prMsg*/)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingSMSeRequestCallback) Entered"));

    /* Define a Property to send to CCA*/
    fc_sxm_tcl_trParkingPropertyDataUpdate oProperty;

    /* Make local reference of the Property */
    midw_ext_sxm_parkingfi_tclMsgParkingInfoListsStatusStatus &oFiMsg=oProperty.oFiMsg;

    //by default assign values to nearby case
    midw_ext_fi_tcl_SxmListInfo oListInfo;
    oListInfo.ListType.enType = midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY;
    oListInfo.ListID = (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY;

    if(enGetSmsServiceState() == DATASERVICE_STATE_READY)
    {
		SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
		fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY);

		SXM_ASSERT_RETURN(OSAL_NULL!=poParkingList);
		if(poParkingList->trGetListMode().enType == midw_ext_fi_tcl_e8_SxmListMode::FI_EN_FREEZE)
		{
   		oListInfo.Status.enType =    midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_CHANGED;
    	}
		else
		{
			oListInfo.Status.enType =    midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_READY;
		}
    }
    else
    {
    	oListInfo.Status.enType =    midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_INVALID;
    }

    ETG_TRACE_USR4(("fc_sxm_trMsgParkingSMSeRequestCallback Status=%d",oListInfo.Status.enType));

    /* Assign other List Info*/
    oFiMsg.InfoList.ListInfo.push_back(oListInfo);

    /* Finally Send the Notification for the Property update*/
    fc_sxm_tclParkingProperties::instance()->oParkingDataUpdateStatus.vSet(oProperty);
    fc_sxm_tclParkingProperties::instance()->oParkingDataUpdateStatus.vNotify();

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingSMSeRequestCallback) Exit"));
}
/****************************************************************************
 *
 *FUNCTION:          vProcess
 *
 *DESCRIPTION:       Method to process Parking POI callabck message
 *
 *PARAMETERS:       fc_sxm_trMsgParkingSMSePOICallback
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingSMSePOICallback const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingSMSePOICallback) Entered"));

    /* Define a Property to send to CCA*/
    fc_sxm_tcl_trParkingPropertyDataUpdate oProperty;

    /* Make local reference of the Property */
    midw_ext_sxm_parkingfi_tclMsgParkingInfoListsStatusStatus &oFiMsg=oProperty.oFiMsg;
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);

    midw_ext_fi_tcl_SxmListInfo oListInfo;
    fc_sxm_tclIParkingBase* poParkingList = OSAL_NULL;

    if(prMsg->u32FilterMask == FC_SXM_PARKING_DEST_FILTER_MASK)
    {
        //by default assign values to nearby case
        oListInfo.ListType.enType = midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION;
        oListInfo.ListID = (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION;

        poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION);

    }
    else if(prMsg->u32FilterMask == FC_SXM_PARKING_FAV_FILTER_MASK)
    {
        //by default assign values to nearby case
        oListInfo.ListType.enType = midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE;
        oListInfo.ListID = (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE;

        poParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

    }
	else
	{
		//Just Ignore and return 
		//for all other Filter Mask from SDK.
		return;
	}

   if(enGetSmsServiceState() == DATASERVICE_STATE_READY)
   {
		SXM_ASSERT_RETURN(OSAL_NULL!=poParkingList);
		if(poParkingList->trGetListMode().enType == midw_ext_fi_tcl_e8_SxmListMode::FI_EN_FREEZE)
		{
   		oListInfo.Status.enType =    midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_CHANGED;
    	}
		else
		{
			oListInfo.Status.enType =    midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_READY;
		}
    }
    else
    {
    	oListInfo.Status.enType =    midw_ext_fi_tcl_e8_SxmListStatus::FI_EN_INVALID;
    }

    ETG_TRACE_USR4(("fc_sxm_trMsgParkingSMSePOICallback Status=%d",oListInfo.Status.enType));

    /* Assign other List Info*/
    oFiMsg.InfoList.ListInfo.push_back(oListInfo);

    /* Finally Send the Notification for the Property update*/
    fc_sxm_tclParkingProperties::instance()->oParkingDataUpdateStatus.vSet(oProperty);
    fc_sxm_tclParkingProperties::instance()->oParkingDataUpdateStatus.vNotify();

    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess(fc_sxm_trMsgParkingSMSeRequestCallback) Exit"));
}
/****************************************************************************
 *
 *FUNCTION:          vSortListByDistance
 *
 *DESCRIPTION:       Method to Sort the List by parking location distance
 *
 *PARAMETERS:       List to be sorted
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vSortListByDistance(midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult& oMRes) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vSortListByDistance"));

    //using std sort: sorting Method Result by distance
    std::sort(oMRes.ParkingInfoList.begin(), oMRes.ParkingInfoList.end(), bCompareDistance);

}
/****************************************************************************
 *
 *FUNCTION:          vSortListByUserOrder
 *
 *DESCRIPTION:       Method to Sort the List as per user save order
 *
 *PARAMETERS:       List to be sorted
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vSortListByUserOrder(midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult& oMRes) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vSortListByUserOrder"));
    SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
    fc_sxm_tclIParkingBase* poFavParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

    if(poFavParkingList != OSAL_NULL)
    {
        //Take Temporary Map which automatically sorts entries by key
        //Key for this map is "order which user have saved favorite"
        map<tU32,midw_ext_fi_tcl_ParkingInfo> mapTempFavList;

        vector<midw_ext_fi_tcl_ParkingInfo>::iterator itrMRes;

        //for each favorite entry from method result
        for(itrMRes = oMRes.ParkingInfoList.begin(); itrMRes != oMRes.ParkingInfoList.end();++itrMRes)
        {
            const map<tU32,trPOIData>& mapFavList = poFavParkingList->mapGetPOIList();        //LINT warning
            //Check with internal favorite list
            SXM_IF_FIND_MAP_CONST(tU32,trPOIData,FavIter,mapFavList,itrMRes->ID)
            {
                //if found,copy to temporary map,key as "sortOrder"
                mapTempFavList.insert(pair<tU32,midw_ext_fi_tcl_ParkingInfo>(FavIter->second.u32Index,*itrMRes));
            }
        }
        //Clear current method result list
        //As we are going to fill this list again with sorted entries
        oMRes.ParkingInfoList.clear();

        //now,copy the entries from Temp to Method result
        SXM_FOREACH_MAP_CONST(tU32,midw_ext_fi_tcl_ParkingInfo,Iter,mapTempFavList)
        {
            oMRes.ParkingInfoList.push_back(Iter->second);
        }

    }

}
/****************************************************************************
 *
 *FUNCTION:          vUpdateDistance
 *
 *DESCRIPTION:       Method to update distance value as per device postion
 *DESCRIPTION:       captured when entered list mode.
 *
 *PARAMETERS:       List of parking locations to be updated
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vUpdateDistance(midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult& oMRes)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vUpdateDistance"));

	vector<midw_ext_fi_tcl_ParkingInfo>::iterator itrMRes;

	//for each favorite entry from method result
	for(itrMRes = oMRes.ParkingInfoList.begin(); itrMRes != oMRes.ParkingInfoList.end();++itrMRes)
	{
		itrMRes->Distance = f32GetCurrentDistance(itrMRes->PrimaryLocation.Latitude.s32Value,itrMRes->PrimaryLocation.Longitude.s32Value);
	}
}

/****************************************************************************
 *
 *FUNCTION:          bIsFavorite
 *
 *DESCRIPTION:       Method to find whther particular ParkingID is favorite or not
 *
 *PARAMETERS:       ListID
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tBool fc_sxm_tclParkingApp::bIsFavorite(tU32 u32ParkingID) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::bIsFavorite u32ParkingID=%d",u32ParkingID));
    SXM_ASSERT_RETURN_VAL(OSAL_NULL!=_poParkingManager,FALSE);
    fc_sxm_tclIParkingBase* poFavParkingList = _poParkingManager->poGetListByID(midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE);

    tBool bStatus = FALSE;
    if(poFavParkingList != OSAL_NULL)
    {
        const map<tU32,trPOIData>& mapFavList = poFavParkingList->mapGetPOIList();  //LINT
        //Check if ParkingID exists from internal favorite list
        SXM_IF_FIND_MAP_CONST(tU32,trPOIData,FavIter,mapFavList,u32ParkingID)
        {
            //OK,found
            bStatus = TRUE;
        }
    }
    return bStatus;

}
/****************************************************************************
 *
 *FUNCTION:          f32GetDistanceFromDestination
 *
 *DESCRIPTION:       Method to find distance b/w two SXMPoints
 *DESCRIPTION:       i.e from destination position to particular parking location
 *
 *PARAMETERS:        Parking Location lat & lon
 *
 *RETURNVALUES:      distance
 *
 ******************************************************************************/

tF32 fc_sxm_tclParkingApp::f32GetDistanceFromDestination(tF32 f32Lat,tF32 f32Lon)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::f32GetDistanceFromDestination f32Lat=%f,f32Lon=%f",f32Lat,f32Lon));

    //we need sdk points to calculate distance
    SXMPoint rSXMDevPoint,rSXMLocPoint;

    //prepare Device location sdk point
    rSXMDevPoint.lat = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(fc_sxm_tclGenericUtils::s32DegreeSmsToNav(_rSDKDestLocation.s32Latitude));
    rSXMDevPoint.lon = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(fc_sxm_tclGenericUtils::s32DegreeSmsToNav(_rSDKDestLocation.s32Longitude));

    ETG_TRACE_USR4(("rSXMDevPoint.lat=%f,rSXMDevPoint.lon=%f",rSXMDevPoint.lat,rSXMDevPoint.lon));

    //prepare Parking Location point
    //from where we would like to find distance
    rSXMLocPoint.lat = f32Lat;
    rSXMLocPoint.lon = f32Lon;

    return (tF32)sxm_point_true_distance(sxm_point_distance(&rSXMDevPoint,&rSXMLocPoint));
}
/****************************************************************************
 *
 *FUNCTION:          f32GetDistanceFromFrozenDevicePos
 *
 *DESCRIPTION:       Method to find distance b/w two SXMPoints
 *DESCRIPTION:       i.e from device position when list is freezed to particular parking location
 *
 *PARAMETERS:        Parking Location lat & lon
 *
 *RETURNVALUES:      distance
 *
 ******************************************************************************/

tF32 fc_sxm_tclParkingApp::f32GetDistanceFromFrozenDevicePos(tF32 f32Lat,tF32 f32Lon)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::f32GetDistanceFromFrozenDevicePos f32Lat=%f,f32Lon=%f",f32Lat,f32Lon));

    //we need sdk points to calculate distance
    SXMPoint rSXMDevPoint,rSXMLocPoint;

    //prepare Device location sdk point
    rSXMDevPoint.lat = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(_rFrozenDevicePos.s32Latitude);
    rSXMDevPoint.lon = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(_rFrozenDevicePos.s32Longitude);

    ETG_TRACE_USR4(("rSXMDevPoint.lat=%f,rSXMDevPoint.lon=%f",rSXMDevPoint.lat,rSXMDevPoint.lon));

    //prepare Parking Location point
    //from where we would like to find distance
    rSXMLocPoint.lat = f32Lat;
    rSXMLocPoint.lon = f32Lon;

    return (tF32)sxm_point_true_distance(sxm_point_distance(&rSXMDevPoint,&rSXMLocPoint));
}
/****************************************************************************
 *
 *FUNCTION:          f32GetCurrentDistance
 *
 *DESCRIPTION:       Method to find distance b/w two SXMPoints
 *DESCRIPTION:       i.e from new Device position to particular parking location
 *
 *PARAMETERS:        Parking Location lat & lon
 *
 *RETURNVALUES:      distance
 *
 ******************************************************************************/

tF32 fc_sxm_tclParkingApp::f32GetCurrentDistance(tS32 s32Lat,tS32 s32Lon)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::f32GetCurrentDistance s32Lat=%d,s32Lon=%d",s32Lat,s32Lon));

    //we need sdk points to calculate distance
    SXMPoint rSXMDevPoint,rSXMLocPoint;

    //prepare Device location sdk point
    rSXMDevPoint.lat = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(rGetSDKDeviceLocation().s32Latitude);
    rSXMDevPoint.lon = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(rGetSDKDeviceLocation().s32Longitude);

    rSXMLocPoint.lat = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(s32Lat);
    rSXMLocPoint.lon = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(s32Lon);

    //ETG_TRACE_USR4(("rSXMDevPoint.lat=%f,rSXMDevPoint.lon=%f",rSXMDevPoint.lat,rSXMDevPoint.lon));
   //ETG_TRACE_USR4(("rSXMLocPoint.lat=%f,rSXMLocPoint.lon=%f",rSXMLocPoint.lat,rSXMLocPoint.lon));

    return (tF32)sxm_point_true_distance(sxm_point_distance(&rSXMDevPoint,&rSXMLocPoint));
}
/****************************************************************************
 *
 *FUNCTION:          vSendUpdate
 *
 *DESCRIPTION:       Method to send property update to HMI
 *
 *PARAMETERS:       ListID
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vSendUpdate(tU32 u32ListID)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vSendUpdate u32ListID=%d",u32ListID));

    switch(u32ListID)
    {
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY:
        {
            //Send update to HMI
            fc_sxm_trMsgParkingSMSeRequestCallback rMsg;
            vProcess(&rMsg);
        }
        break;
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION:
        {
            //Send update to HMI
            fc_sxm_trMsgParkingSMSePOICallback rMsg;
            rMsg.u32FilterMask = FC_SXM_PARKING_DEST_FILTER_MASK;
            vProcess(&rMsg);
        }
        break;
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE:
        {
            //Send update to HMI
            fc_sxm_trMsgParkingSMSePOICallback rMsg;
            rMsg.u32FilterMask = FC_SXM_PARKING_FAV_FILTER_MASK;
            vProcess(&rMsg);
        }
        break;
        default:
        break;
    }

}
/****************************************************************************
 *
 *FUNCTION:          u32GetCapacity
 *
 *DESCRIPTION:       method to return capacity value based on listID
 *
 *PARAMETERS:        ListID
 *
 *RETURNVALUES:      Capacity
 *
 ******************************************************************************/
tU32 fc_sxm_tclParkingApp::u32GetCapacity(tU32 u32ListID) const
{
    switch(u32ListID)
    {
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY:
        {
            return FC_SXM_PARKING_NEARBY_CAPACITY;
        }
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION:
        {
            return FC_SXM_PARKING_NEARDESTINATION_CAPACITY;
        }
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE:
        {
            return FC_SXM_PARKING_FAVORITES_CAPACITY;
        }
        default:
            return 0;
    }
}
/****************************************************************************
 *
 *FUNCTION:          f32GetRadius
 *
 *DESCRIPTION:       method to return radius value based on listID
 *
 *PARAMETERS:        ListID
 *
 *RETURNVALUES:      Radius
 *
 ******************************************************************************/
tF32 fc_sxm_tclParkingApp::f32GetRadius(tU32 u32ListID) const
{
    switch(u32ListID)
    {
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY:
        {
            return (tF32)FC_SXM_SXM_PARKING_NEARBY_RADIUS;
        }
        case (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION:
        {
            return (tF32)FC_SXM_SXM_PARKING_NEARDEST_RADIUS;
        }
       default:
            return 0.0F;
    }
}
/****************************************************************************
 *
 *FUNCTION:          vProcessTimer
 *
 *DESCRIPTION:       method will be called when timer gets expired
 *
 *PARAMETERS:        message
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingApp::vProcessTimer(fc_sxm_trMsgParkingListUpdateTimer const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vProcess fc_sxm_trMsgListUpdateTimer ListID=%d",prMsg->u32ListID));

   //stope timer if running still
   vStopListTimer();

   /* finally send update to HMI */
   vSendUpdate(prMsg->u32ListID);

   SXM_ASSERT_RETURN(OSAL_NULL!=_poParkingManager);
   fc_sxm_tclIParkingBase* poParkingList = _poParkingManager->poGetListByID((tU8)(prMsg->u32ListID));

   SXM_ASSERT_RETURN(OSAL_NULL!=poParkingList);
   if(poParkingList->trGetListMode().enType == midw_ext_fi_tcl_e8_SxmListMode::FI_EN_FREEZE)
	{
	   vStartListTimer(prMsg->u32ListID);
	}
}
/****************************************************************************
 *
 *FUNCTION:          vStartListTimer
 *
 *DESCRIPTION:       Method to start the Listupdate Timer for parking app
 *
 *PARAMETERS:        ListID for which timer to start
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vStartListTimer(tU32 u32ListID)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vStartListTimer u32ListID=%d",u32ListID));

    //prepare Msg to timer
    fc_sxm_trMsgParkingListUpdateTimer rMsg;
    rMsg.u32ListID = u32ListID;

    //Set Msg to timer
    _mListUpdateTimer.vSetMsg(rMsg);

    //Start Timer
    _mListUpdateTimer.vStart(FC_SXM_LIST_DATA_UPDATE_PERIOD);
}
/****************************************************************************
 *
 *FUNCTION:          vStopListTimer
 *
 *DESCRIPTION:       Method to stop the Listupdate Timer for parking app
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingApp::vStopListTimer(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingApp::vStopListTimer()"));

    //Stop List timer
    _mListUpdateTimer.vStop();
}
/****************************************************************************
 *
 *FUNCTION:          vStopListTimer
 *
 *DESCRIPTION:       Method to know if list timer is running/activated
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tBool fc_sxm_tclParkingApp::bIsListUpdateTimerRunning(tVoid) const
{
    return(_mListUpdateTimer.bIsRunning());
}

