/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_theater_list.h
* @brief       Theater List Handler declaration.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_THEATER_LIST_H_
#define _FC_SXM_TCL_THEATER_LIST_H_

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_data_list.hpp"
#include "fc_sxm_tcl_base_dsrl.h"
#include "fc_sxm_movies_fi.h"
#include "fc_sxm_movies_types.h"

struct fc_sxm_trMovieRating
{
   MOVIE_RATING_ENUM enMovieRating;
   string sMovieRatingTesxt;
};
struct fc_sxm_trMovieDetails
{
   MOVIE_ID movie_id;
   string movieName;
   string movieActors;
   string movieSummury;
   UN8 movieLength;
   fc_sxm_trMovieRating movieRating;
   vector<UN32,allocator<UN32> > movieShowTimes;	
 };

/* structure with data of a list-entry for HMI */
struct fc_sxm_trTheatreListEntry:public fc_sxm_trBaseListEntry {
public:
   midw_ext_fi_tcl_TheatreDetails TheaterDetails;
   float distance;
   map<MOVIE_ID, fc_sxm_trMovieDetails> MoviesList;
public:
   tVoid vPrintTheatreAmenities () const;
};


/*
  sort-methods supported
*/
typedef enum
{
   fc_sxm_enTheaterSortMethod_NONE,
   fc_sxm_enTheaterSortMethod_DISTANCE

} fc_sxm_tenTheatreListSortMethod;

/*
  sort-configuration consisting of sort-method and additional parameters
*/
struct fc_sxm_trTheatreListSortCfg {
   fc_sxm_tenMoviesSortMethod enSortMethod;

   fc_sxm_trTheatreListSortCfg() :
      enSortMethod(fc_sxm_enMoviesSortMethod_NONE)
   {}

   /* Check if configuration has changed.
    to check if sorting has chanded, configuration has to provice operator!=
    attention:
    operator is used in the way that it only returns TRUE if
    the changes are relevant for the current configuration.
    e.g. if sorting by distance is not enabled, changing of the 
    location has no effect.
   */
   bool operator!=(fc_sxm_trTheatreListSortCfg const &b) const {
     SXM_RET_1_IF_NEQ(enSortMethod);
     return false;
   }
};

/*
  sort-configuration consisting of filter-method and additional parameters
*/
struct fc_sxm_trTheatreListFilterCfg {

   fc_sxm_tenMoviesFilterMethod enFilterMethod;
   tU32 u32MovieID;
   string SearchName;

   fc_sxm_trTheatreListFilterCfg() :
      enFilterMethod(fc_sxm_enMoviesFilterMethod_NONE),
      u32MovieID(0),
      SearchName("")
   {}

   /* to check if filter has chanded, configuration has to provice operator!= 
    see sort-configuration 
   */
   bool operator!=(fc_sxm_trTheatreListFilterCfg const &b) const {
      SXM_RET_1_IF_NEQ(enFilterMethod);
      SXM_RET_1_IF_NEQ(u32MovieID);
      SXM_RET_1_IF_NEQ(SearchName);

     return false;
   }

};
/*
  sort-configuration consisting of filter-method and additional parameters
*/
struct fc_sxm_trTheatreListCapacityCfg {

   /* to check if filter has chanded, configuration has to provice operator!= 
    see sort-configuration 
   */
   bool operator!=(fc_sxm_trTheatreListCapacityCfg const &b) const {
	   (tVoid)b;
     return true;
   }

};

/*
  struct containing all our configuration.
  Must be dericed from fc_sxm_trBaseListConfig
*/
struct fc_sxm_trTheatreListConfig:public fc_sxm_trBaseListConfig {
public:
   /* typedef to make type of list-entries visible for templates */
   typedef fc_sxm_trTheatreListEntry trListEntry;
   /* post-sort and post-filter to be applied after dsrl-iterate is done */
   fc_sxm_trTheatreListSortCfg rSortCfg;
   fc_sxm_trTheatreListFilterCfg rFilterCfg;
   /* see sort-config */
   bool operator!=(fc_sxm_trTheatreListConfig const &b) const {
      SXM_RET_1_IF_NEQ(rSortCfg);
      SXM_RET_1_IF_NEQ(rFilterCfg);
      return false;
   }
   /*
      overwrite method from base-class if the location is used in this configuration.
      While not sorting or filtering for distance, the location is not used.
      Only if location is used, on change of location actions will be taken by the framework
   */
   virtual tBool bUsesLocation() const {
     return rSortCfg.enSortMethod==fc_sxm_enMoviesSortMethod_DISTANCE;
   }
    virtual tVoid vPrintSpecific() const;

};


/*
  The Theater-list, derive from template fc_sxm_tclDataListTyped.
  Template-parameters are our list-class and our configuration.
*/
class fc_sxm_trTheatreList:public fc_sxm_tclDataListTyped<fc_sxm_trTheatreList, fc_sxm_trTheatreListConfig> {
 public:
   /*
      constructor:
      Simply pass parameters to base-class
   */
    fc_sxm_trTheatreList(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp):
     fc_sxm_tclDataListTyped<fc_sxm_trTheatreList, fc_sxm_trTheatreListConfig>(u32ListId, enListType, poDataApp, TR_CLASS_FC_SXM_MOVIES_APP) 
   {}
   /* compare-fn that shall compare according to fc_sxm_trFuelListSortCfg
     Attention:
     Strict ordering has to be proviced: *prLeft<*prRight || *prRights<*prLeft
     (not strict weak ordering as for stl-containers)
     This is accomplished by comparing a unique id as fallback in case of *prLeft==*prRight
   */
   virtual bool bCompareItem(fc_sxm_trTheatreListEntry const *prLeft, fc_sxm_trTheatreListEntry const *prRight) const;

   /* compare-fn that shall filter according to fc_sxm_trFuelListFilterCfg */
   virtual bool bFilterPassed(fc_sxm_trTheatreListEntry const &rItem) const;
   /*  function that shall filter theater playing particular movie */
   bool bFilterMovie(fc_sxm_trTheatreListEntry const &rEntry,tU32 u32MoviesID) const;

   /*
      Method to Send complete US Movies List to FI
   */
   static tVoid vSendMoviesList(tU32 ListID,fc_sxm_trAdressing const &rAdressing);
   /*
      Method to retrieve complete US Movies from internal list
   */
   static tU32 u32GetUSMoviesList(midw_ext_fi_tcl_MoviesList &oMovieList);
   /*
      Method to update a movie into our internal list
   */
   static tBool bUpdateMovieEntry(fc_sxm_trMovieDetails const &MovieDetails);
   /*
      Method to delete/clear complete US movies list.
   */
   static tVoid vClearMovieList();
   /*
      Method to Get US movies list Count 
   */
   static tU32 u32GetUSMoviesListCount();
   /*
      Method to Search a Movie From US Movies List
   */
   static tVoid vSearchMovie(midw_ext_fi_tclString trSearchMovieName,fc_sxm_trAdressing const &rAdressing);
   /*
      Method to Send Movies Details to FI
   */
   static tVoid vSendUSMovieInfo(tU32 u32ListID,tU32 u32MovieID,fc_sxm_trAdressing const &rAdressing);
   /*
      helper to get the Theater-movies
   */
   tU32 u32GetMoviesList(fc_sxm_trTheatreListEntry const *prEntry,midw_ext_fi_tcl_MoviesList &oMovieList) const;
   /*
      helper to get the Particular Movie Information
   */
   tBool bGetMovieInfo(tU32 u32MovieID,fc_sxm_trTheatreListEntry const *prEntry,midw_ext_fi_tcl_MovieDetails &oMovieDetails) const;

protected:
   /*
      will be called by base-class when data are ready and the list shall be sent to hmi
   */
   virtual tVoid vEmit(fc_sxm_trAdressing const &rAdressing);
#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING 
   virtual tVoid vStartTimer();
   virtual tBool bIsListModeTimerRunning(); 
   virtual tVoid vStopTimer();
#endif

private:
   /*
      Method to find MOVIE_ID exists in MoviesList
   */
   bool bIsMovieFound(midw_ext_fi_tcl_MoviesList const &oMoviesList, MOVIE_ID const n16MovieID) const;
   /*
      helper to get Search a String in another string
   */
   static tBool bSearchFound(string trSearchString,midw_ext_fi_tclString trOriginalString);
   /*
   helper to Retireve Movie details from our internal info.
   */
   static tVoid vRetrieveMovieDetails(fc_sxm_trMovieDetails MovieDetails,midw_ext_fi_tcl_MovieDetails &rMovieDetails);
   /*
      helper to get Convert a String Characters to Uppercase
   */
   static tVoid ConvertToUpper(string &str);
   /*
      Method to Retrieve Movies List containing MovieName
   */
   static tU32 u32SearchMoviesList(midw_ext_fi_tclString trSearchMovieName,
                              midw_ext_fi_tcl_MoviesList &oMovieList);

   /*
      this method will convert Time from int type value to string type.
   */
   static string sGetMovieRunTime(UN8 const un8MovieDuration);
   /*
      helper method to convert a int value to string value
   */
   static string sConvertInt2Str(unsigned int number);
   /*
      this method will return distance b/w two locations.
   */
   tF32 fGetFIDistance(fc_sxm_trTheatreListEntry const *prEntry) const;
   /*
      this method will retrieve Movie Show Times from our internal list to FI
   */
   tVoid vGetMovieShowTimesList(vector<UN32,allocator<UN32> > internalMovieTimes,
                                midw_ext_fi_tcl_MovieTimesList &FIMovieTimesList) const;
   /*
      method to compare 2 theater Names
   */
   bool bCompareTheaterName(fc_sxm_trTheatreListEntry const *prLeft, fc_sxm_trTheatreListEntry const *prRight) const;

};


#endif
