/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_movies_dsrl.cpp
* @brief       Implementation of Movies DSRL for a specified location.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_movies_app.h"
#include "fc_sxm_tcl_movies_dsrl.h"
#include "fc_sxm_tcl_theater_list.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_MOVIES_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_movies_dsrl.cpp.trc.h"
#endif


tVoid fc_sxm_trMoviesDSRLCfg::vPrintSpecific() const {
    // for debuging purpose
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t enSortMethod=%u", ETG_CENUM(fc_sxm_tenMoviesSortMethod, enSortMethod)));
}

/*********************************************************************
 *
 *FUNCTION:     fc_sxm_tclMoviesDSRL
 *
 *DESCRIPTION:  Constructor
 *             
 *PARAMETER:    fc_sxm_tenDSRLType - DSRL Type 
				DATASERVICE_MGR_OBJECT - MOVIES SERVICE OBJECT
 *
 ********************************************************************/
fc_sxm_tclMoviesDSRL::fc_sxm_tclMoviesDSRL(fc_sxm_tenDSRLType enDsrlType):
   fc_sxm_tclConfigDSRL<fc_sxm_trMoviesDSRLCfg>(enDsrlType,
                                                fc_sxm_tclMoviesApp::instance(),
                                                TR_CLASS_FC_SXM_MOVIES_DSRL)
{}
    

/*********************************************************************
 *
 *FUNCTION:     ~fc_sxm_tclMoviesDSRL
 *
 *DESCRIPTION:  Destructor
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclMoviesDSRL::~fc_sxm_tclMoviesDSRL()
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL destructor"));
}

/*********************************************************************
 *
 *FUNCTION:     vOnDSRLUpdate
 *
 *DESCRIPTION:  Called when DSRL is completely Iterated
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclMoviesDSRL::vOnDSRLUpdate(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::vOnDSRLUpdate"));
}

/*********************************************************************
 *
 *FUNCTION:     cb_bDSRLFilter
 *
 *DESCRIPTION:  Movies service DSRL filter callback
 *
 *PARAMETER:    DSRL_OBJECT
				DSRL_ENTRY_OBJECT - THEATER ENTRY
 *
 *RETURNVALUE:  tBool - TRUE for Success
						FALSE for failure
 *
 ********************************************************************/
tBool fc_sxm_tclMoviesDSRL::bHandleFilterCallback(DSRL_OBJECT hDSRL, 
                                                DSRL_ENTRY_OBJECT hEntry)
{
    //   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::bHandleFilterCallback"));

   /* 
      These are added to avoid Lint Warnings
      must be removed if you are using them later
   */
   (tVoid)hDSRL;
   (tVoid)hEntry;

   return TRUE;
}

/*********************************************************************
 *
 *FUNCTION:     cb_n16DSRLSort
 *
 *DESCRIPTION:  Movies service DSRL sort callback
 *
 *PARAMETER:    DSRL_OBJECT
				DSRL_ENTRY_OBJECT1 - THEATER ENTRY1
				DSRL_ENTRY_OBJECT2 - THEATER ENTRY2
 *
 *RETURNVALUE:  tS16 
 ********************************************************************/
tS16 fc_sxm_tclMoviesDSRL::s16HandleSortCallback(DSRL_OBJECT hDSRL,
                                               DSRL_ENTRY_OBJECT hEntry1,
                                               DSRL_ENTRY_OBJECT hEntry2)
{
    //   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::s16HandleSortCallback"));
   
   /* These are added to avoid Lint Warnings
      must be removed if you are using them later
   */
   (tVoid)hDSRL;
   (tVoid)hEntry1;
   (tVoid)hEntry2;

   return 0;
}
/*********************************************************************
 *
 *FUNCTION:     vHandleDsrlEntryInvalid
 *
 *DESCRIPTION:  Removes invalid DSRL entry
 *
 *PARAMETER:    tEntryID - Entry ID
 *
 *RETURNVALUE:  None 
 *
 ********************************************************************/
tVoid fc_sxm_tclMoviesDSRL::vHandleDsrlEntryInvalid(DSRL_ENTRY_ID tEntryID) {
   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::vHandleDsrlEntryInvalid"));
   fc_sxm_trTheatreList::vRemoveEntry(tEntryID); // removes the invalid entry
}
/*********************************************************************
 *
 *FUNCTION:     cb_bDSRLIterate
 *
 *DESCRIPTION:  Movies service DSRL iterate callback 
				Called for every theater entry exist in the given raius
 *
 *PARAMETER:    DSRL_OBJECT
				DSRL_ENTRY_ID
				DSRL_ENTRY_STATUS_ENUM
				DSRL_ENTRY_OBJECT - THEATER ENTRY
 *
 *RETURNVALUE:  tS16 
 *
 ********************************************************************/
tBool fc_sxm_tclMoviesDSRL::bHandleIterateCallback(DSRL_OBJECT hDSRL, 
                                                 DSRL_ENTRY_ID tEntryID,
                                                 DSRL_ENTRY_STATUS_ENUM eStatus,
                                                 DSRL_ENTRY_OBJECT hEntryObject)
{

   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::cb_bDSRLIterate"));

   (tVoid)hDSRL;   /* These are added to avoid Lint Warnings */
   (tVoid)tEntryID;   /*  must be removed if you are using them later */

   THEATER_OBJECT hTheater;
   /* Extract the Theater object */
   hTheater = DSRL_ENTRY.hTheater(hEntryObject);

   /* Check for the Valid Theater object */
   if (hTheater == THEATER_INVALID_OBJECT)
   {
      ETG_TRACE_ERR(("Theater Object not created!"));
      return TRUE;
   }

   /* Create Theater Specific objects */
   LOCID_OBJECT hLocID;
   tBool isFavorite;
   SMSAPI_RETURN_CODE_ENUM eReturnCode;

   /* define Theater Location */
   LOCATION_OBJECT hTheaterLocation;
   hTheaterLocation = THEATER.hLocation(hTheater);

   /* Check for the Valid Theater Location */
   if(LOCATION_INVALID_OBJECT == hTheaterLocation)
   {
      ETG_TRACE_ERR(("Theater Location not created!"));
      return TRUE;
   }

   /* Create Local CCA Theater Info object */
   fc_sxm_trTheatreListEntry rListEntry;

   /* Extract Theater Data from Location */
   fc_sxm_vCopySmsString2Fi(LOCATION.hDescription(hTheaterLocation) , rListEntry.TheaterDetails.TheatreName );
   fc_sxm_vCopySmsString2Fi(LOCATION.hStreetNum(hTheaterLocation) , rListEntry.TheaterDetails.TheatreStreetNum );
   fc_sxm_vCopySmsString2Fi(LOCATION.hStreetName(hTheaterLocation) , rListEntry.TheaterDetails.TheatreStreetName );
   fc_sxm_vCopySmsString2Fi(LOCATION.hCity(hTheaterLocation) , rListEntry.TheaterDetails.TheatreCity );
   fc_sxm_vCopySmsString2Fi(LOCATION.hState(hTheaterLocation) , rListEntry.TheaterDetails.TheatreState );
   fc_sxm_vCopySmsString2Fi(LOCATION.hZipCode(hTheaterLocation) , rListEntry.TheaterDetails.TheatreZip );
   fc_sxm_vCopySmsString2Fi(LOCATION.hPhone(hTheaterLocation),rListEntry.TheaterDetails.TheatrePhone );

   /* assign favorite theatre type*/
   isFavorite = FAVORITES.bIsFavorite(hEntryObject);
   rListEntry.TheaterDetails.isTheaterFavorite = isFavorite;

   /* assign Theater Location & LOCID values*/
   hLocID = LOCATION.hLocID(hTheaterLocation);
   rListEntry.u32SxmId = rListEntry.TheaterDetails.Theater_LOCID = LOCID.tID(hLocID);
   rListEntry.u32DsrlEntryID=tEntryID;
   rListEntry.hLocation = hTheaterLocation;

   /* Fill Theatre Amenities*/
   vGetTheatreAmenities (hTheater, rListEntry.TheaterDetails.TheaterAmenities);

   /* Asign Distance value between target to this theater 
   DISTANCE_OBJECT distobj = DISTANCE_INVALID_OBJECT;
   LOCATION_OBJECT deviceloc = hGetSMSLocation();
   distobj = LOCATION.hDistance(rListEntry.hLocation,deviceloc);
   if(DISTANCE_INVALID_OBJECT != distobj )
   {
      OSAL_FIXED_OBJECT distval = OSAL_FIXED_INVALID_OBJECT;
      distval = DISTANCE.hValue(distobj,DISTANCE_UNIT_TYPE_MILES);
      if(OSAL_FIXED_INVALID_OBJECT != distval)
      {
         rListEntry.distance =  fc_sxm_fFixed2Float(distval);
      }
   }
   */

   #if DEBUG_MOVIES
      ETG_TRACE_USR4(("\n"));
      ETG_TRACE_USR4(("LOC ID: %d", rListEntry.u32SxmId));
      ETG_TRACE_USR4(("Theater Name: %s", rListEntry.TheaterDetails.TheatreName.szValue));
      ETG_TRACE_USR4(("Theater Street Name: %s", rListEntry.TheaterDetails.TheatreStreetName.szValue));
      ETG_TRACE_USR4(("Theater City: %s", rListEntry.TheaterDetails.TheatreCity.szValue));
      ETG_TRACE_USR4(("Theater State: %s", rListEntry.TheaterDetails.TheatreState.szValue));
      ETG_TRACE_USR4(("Theater Zip: %s", rListEntry.TheaterDetails.TheatreZip.szValue));
      ETG_TRACE_USR4(("Theater Phone: %s", rListEntry.TheaterDetails.TheatrePhone.szValue));

      if(rListEntry.TheaterDetails.isTheaterFavorite == TRUE)
      {
         ETG_TRACE_USR4(("Entry is in Favorite List!"));
      }else{
         ETG_TRACE_USR4(("Entry is not in Favorite List!"));
      }
      rListEntry.vPrintTheatreAmenities ();
   #endif

   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::cb_bDSRLIterate eIterateMovies START"));

   /* Query For Movies playing in this Theater*/
   eReturnCode = THEATER.eIterateMovies(hTheater, bIterateTheaterMovies, (void*)&rListEntry);

   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::cb_bDSRLIterate eIterateMovies END"));


   if(eReturnCode == SMSAPI_RETURN_CODE_NO_OBJECTS)
   {
      ETG_TRACE_ERR(("No Movie is playing in this theatre"));
      /* But continue to update this theater entry into our internal list*/
   }
   /* Check return Code for Error*/
   else if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
   {
      ETG_TRACE_ERR(("Error in retrieving Movies"));
      /* But still continue to update this theater entry into our internal list*/
   }

   /* Finally Update this Theater Entry into our internal list*/
   fc_sxm_trTheatreList::vUpdateEntry(eStatus, rListEntry);

   return TRUE;
}

BOOLEAN fc_sxm_tclMoviesDSRL::bIterateTheaterMovies (
   THEATER_OBJECT hTheater,
   MOVIE_OBJECT hMovie,
   void *pvEventCallbackArg
)
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::bIterateTheaterMovies IN"));
   fc_sxm_trMovieDetails moviedetails;

   /* prepare a pointer to point to actual theater list*/
   fc_sxm_trTheatreListEntry *TheaterListEntry = (fc_sxm_trTheatreListEntry *)pvEventCallbackArg;

   if(NULL == TheaterListEntry)
   {
      ETG_TRACE_ERR(("Error in TheaterMovies Argument"));
      return FALSE;
   }

   /* Copy the movie's information */
   MOVIE_ID movieId = MOVIE.tID(hMovie);

   /* Extract Movie details from MOVIE Object*/
   moviedetails.movie_id = movieId;
   fc_sxm_vCopySmsString2Stl(MOVIE.hName(hMovie),moviedetails.movieName);
   fc_sxm_vCopySmsString2Stl(MOVIE.hActors(hMovie),moviedetails.movieActors);
   fc_sxm_vCopySmsString2Stl(MOVIE.hSynopsis(hMovie),moviedetails.movieSummury);

   MOVIE_RATING_ENUM MovieRating = MOVIE.eRating(hMovie);
   moviedetails.movieRating.enMovieRating = MovieRating;
   fc_sxm_vCopySmsString2Stl(MOVIE.hRatingText(hMovie),moviedetails.movieRating.sMovieRatingTesxt);

   moviedetails.movieLength = MOVIE.un8RunTime(hMovie);

   /* And Copy to our internal list*/
   TheaterListEntry->MoviesList[movieId] = moviedetails;

   #if DEBUG_MOVIES

      ETG_TRACE_USR4(("MOVIE ID: %d", movieId));
      ETG_TRACE_USR4(("Movie Name: %s", TheaterListEntry->MoviesList[movieId].movieName.c_str()));
      ETG_TRACE_USR4(("Movie Actors: %s", TheaterListEntry->MoviesList[movieId].movieActors.c_str()));
      ETG_TRACE_USR4(("Movies Summury: %s", TheaterListEntry->MoviesList[movieId].movieSummury.c_str()));
      ETG_TRACE_USR4(("Movie Rating: %d", TheaterListEntry->MoviesList[movieId].movieRating.enMovieRating));
      ETG_TRACE_USR4(("Movie Length: %d", TheaterListEntry->MoviesList[movieId].movieLength));

   #endif

   /* Query for Show Times of this Movie*/
   SMSAPI_RETURN_CODE_ENUM eReturnCode = THEATER.eIterateTimes(hTheater, bIterateMovieShowTimes,
   (void *)TheaterListEntry);

   /* Check return code for No Show Times*/
   if (eReturnCode == SMSAPI_RETURN_CODE_NO_OBJECTS)
   {
      ETG_TRACE_ERR(("No ShowTimes"));
      return FALSE;
   }/* Check return code for Error*/
   else if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
   {
      ETG_TRACE_ERR(("Error in Retrieving Movie Show Times"));
      return FALSE;
   }

   ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::bIterateTheaterMovies OUT"));
   /* return TRUE if every thing is OK*/
   return TRUE;

}

BOOLEAN fc_sxm_tclMoviesDSRL::bIterateMovieShowTimes (
   THEATER_OBJECT /*hTheater*/,
   MOVIE_OBJECT hMovie,
   UN32 un32StartTime,
   void *pvEventCallbackArg
)
{
   // ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::bIterateMovieShowTimes IN"));

   /* prepare a pointer to point to actual theater list*/
   fc_sxm_trTheatreListEntry *TheaterListEntry = (fc_sxm_trTheatreListEntry *)pvEventCallbackArg;

   if(NULL == TheaterListEntry)
   {
      ETG_TRACE_ERR(("Error in MovieShowTimes Argument"));
      return FALSE;
   }

   /* Extract Movie ID from MOVIE Object */
   MOVIE_ID movie_id = MOVIE.tID(hMovie);

   /* Copy the show times to corresponding Movie( i.e movie_id)*/
   TheaterListEntry->MoviesList[movie_id].movieShowTimes.push_back(un32StartTime);

#if DEBUG_MOVIES
      ETG_TRACE_USR4(("MOVIE ID: %d", movie_id));
      ETG_TRACE_USR4(("Movie ShowTimes: %d", TheaterListEntry->MoviesList[movie_id].movieShowTimes.back()));
#endif

   // ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::bIterateMovieShowTimes OUT"));
   /* return TRUE if every thing is OK*/
   return TRUE;
}

/*
 * Method Extracts and fills Theatre Amenities from the theatre object passed
 */
tVoid fc_sxm_tclMoviesDSRL::vGetTheatreAmenities (THEATER_OBJECT hTheater,
		                          vector <midw_ext_fi_tcl_e8_TheatreAmenities>& hAmenityList) const
{
	AMENITY_STRUCT allAmenities[THEATER_NUM_AMENITIES];
	hAmenityList.clear();
	/*Initialize Status*/
	for(tU8 u8Index = 0; u8Index < THEATER_NUM_AMENITIES; u8Index++)
	{
		/*Invalidate all the amenity entries*/
		allAmenities[u8Index].eStatus = AMENITY_STATUS_INVALID;
	}
	SMSAPI_RETURN_CODE_ENUM eReturnCode = THEATER.eAmenities(
			hTheater,
            THEATER_NUM_AMENITIES,
            &allAmenities[0]);

	if ( SMSAPI_RETURN_CODE_SUCCESS == eReturnCode )
	{
		// Fill all available amenities
		for(tU8 u8Index = 0; u8Index < THEATER_NUM_AMENITIES; u8Index++)
		{
			if (AMENITY_STATUS_AVAILABLE == allAmenities[u8Index].eStatus )
			{
				midw_ext_fi_tcl_e8_TheatreAmenities amenity;
				/*Get all available Amenitys*/
				amenity.enType = (midw_ext_fi_tcl_e8_TheatreAmenities::tenType) allAmenities[u8Index].uAmenity.eTheaterAmenity;
				hAmenityList.push_back(amenity);
			}
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclMoviesDSRL::vGetTheatreAmenitiesList OUT\t amenities.count = %u", hAmenityList.size()));
}

/*
 * Utility method to print theatre amenities
 */
tVoid fc_sxm_trTheatreListEntry::vPrintTheatreAmenities () const
{
	ETG_TRACE_USR4(("fc_sxm_trTheatreListEntry::vPrintTheatreAmenities IN\t TheatreAmenities count = %u", TheaterDetails.TheaterAmenities.size()));
	vector <midw_ext_fi_tcl_e8_TheatreAmenities>::const_iterator cIt = TheaterDetails.TheaterAmenities.begin();
	for(; cIt != TheaterDetails.TheaterAmenities.end(); ++cIt)
	{
		ETG_TRACE_USR4(("TheatreAmenity available = %u", cIt->enType));
	}

	ETG_TRACE_USR4(("fc_sxm_trTheatreListEntry::vPrintTheatreAmenities OUT"));
}
