/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_canadian_fuel_list.h
* @brief       Declaration of List class for canadian fuel service
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_CANADIAN_FUEL_LIST_H_
#define _FC_SXM_TCL_CANADIAN_FUEL_LIST_H_

#include "fc_sxm_tcl_data_list.hpp"
#include "fc_sxm_tcl_base_dsrl.h"
#include "fc_sxm_canadian_fuel_types.h"
#include "fc_sxm_canadian_fuel_fi.h"


#define PRINT_SIZE 30
class midw_ext_fi_tcl_CanFuelInfo;

struct fc_sxm_trCanadianFuelTypeListEntry {
    FUEL_TYPE_ENUM enCanadianFuelType;
    tU32 u32Price;
    string oFuelName;
    tU32 u32AgeInDays;
    tBool bAvailable;
};

/* structure with data of a list-entry for HMI*/
struct fc_sxm_trCanadianFuelListEntry:public fc_sxm_trBaseListEntry {
public:
    string oStationName;
    string oBrandName;
    tS32 oLat;
    tS32 oLon;
    tBool bFavorite;
    //For Station Info Only
    string oStreetName;
    string oCity;
    string oState;
    string oZipCode;
    string oPhone;
    string oAmenity;
    tF32  DistanceKm;
    tF32  DistanceMiles;
	tU32  u32NumPrices;
    vector<midw_ext_fi_tcl_e8_CanFuelStationAmenity> ListAmenity;
    multimap<FUEL_TYPE_ENUM, fc_sxm_trCanadianFuelTypeListEntry> listFuelTypes;
    
};


/*
  sort-methods supported
*/
typedef enum
{
   fc_sxm_enCanadianFuelSortMethod_NONE = 1,
   fc_sxm_enCanadianFuelSortMethod_PRICE,
   fc_sxm_enCanadianFuelSortMethod_DISTANCE,
   fc_sxm_enCanadianFuelSortMethod_BRANDNAME,
   fc_sxm_enCanadianFuelSortMethod_FIRST_SAVED_FIRST

} fc_sxm_tenCanadianFuelListSortMethod;

/*
  sort-configuration consisting of sort-method and additional parameters
e.g. for which fuel-type shall the price be sorted.
*/
struct fc_sxm_trCanadianFuelListSortCfg {
    fc_sxm_tenCanadianFuelListSortMethod enSortMethod;
    FUEL_TYPE_ENUM enCanadianFuelType;

    fc_sxm_trCanadianFuelListSortCfg():
        enSortMethod(fc_sxm_enCanadianFuelSortMethod_NONE),
        enCanadianFuelType(FUEL_TYPE_REGULAR)
    {}
    /* Check if configuration has changed.
       to check if sorting has chanded, configuration has to provice operator!=
       attention:
       operator is used in the way that it only returns TRUE if
       the changes are relevant for the current configuration.
       e.g. if sorting by distance is not enabled, changing of the 
       location has no effect.
    */
    bool operator!=(fc_sxm_trCanadianFuelListSortCfg const &b) const {
        SXM_RET_1_IF_NEQ(enSortMethod);
        if (enSortMethod==fc_sxm_enCanadianFuelSortMethod_PRICE) {
            SXM_RET_1_IF_NEQ(enCanadianFuelType);
        }
        return false;
    }


};

/*
  sort-configuration consisting of filter-method and additional parameters
  e.g. for which brand shall we filter
*/
struct fc_sxm_trCanadianFuelListFilterCfg {
    FUEL_TYPE_ENUM enCanadianFuelType;
    string  sBrand;
    string  sFuelName;
    /* to check if filter has chanded, configuration has to provice operator!= 
       see sort-configuration 
    */
    bool operator!=(fc_sxm_trCanadianFuelListFilterCfg const &b) const {
        SXM_RET_1_IF_NEQ(enCanadianFuelType);
        SXM_RET_1_IF_NEQ(sBrand);
        return false;
    }

};

/*
  struct containing all our configuration.
  Must be dericed from fc_sxm_trBaseListConfig
*/
struct fc_sxm_trCanadianFuelListConfig:public fc_sxm_trBaseListConfig {
public:
    // typedef to make type of list-entries visible for templates
    typedef fc_sxm_trCanadianFuelListEntry trListEntry;
    // post-sort and post-filter to be applied after dsrl-iterate is done
    fc_sxm_trCanadianFuelListSortCfg rSortCfg;
    fc_sxm_trCanadianFuelListFilterCfg rFilterCfg;
    // see sort-config
    bool operator!=(fc_sxm_trCanadianFuelListConfig const &b) const {
        SXM_RET_1_IF_NEQ(rSortCfg);
        SXM_RET_1_IF_NEQ(rFilterCfg);
        return false;
    }
    /*
      overwrite method from base-class if the location is used in this configuration.
      While not sorting or filtering for distance, the location is not used.
      Only if location is used, on change of location actions will be taken by the framework
    */
    virtual tBool bUsesLocation() const{
        return rSortCfg.enSortMethod==fc_sxm_enCanadianFuelSortMethod_DISTANCE;
    }

    virtual tVoid vPrintSpecific() const;
};


/*
  The canadian fuel-list, derive from template fc_sxm_tclDataListTyped.
  Template-parameters are our list-class and our configuration.
*/
class fc_sxm_trCanadianFuelList:public fc_sxm_tclDataListTyped<fc_sxm_trCanadianFuelList, fc_sxm_trCanadianFuelListConfig> {
 public:
    /*
      constructor:
      Simply pass parameters to base-class
    */
    fc_sxm_trCanadianFuelList(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp):
        fc_sxm_tclDataListTyped<fc_sxm_trCanadianFuelList, fc_sxm_trCanadianFuelListConfig>(u32ListId, enListType, poDataApp, TR_CLASS_FC_SXM_CANADIAN_FUEL_APP) 
    {}

        /* compare-fn that shall compare according to fc_sxm_trCanadianFuelListSortCfg
           Attention:
           Strict ordering has to be proviced: *prLeft<*prRight || *prRights<*prLeft
           (not strict weak ordering as for stl-containers)
           This is accomplished by comparing a unique id as fallback in case of *prLeft==*prRight
        */
    virtual bool bCompareItem(fc_sxm_trCanadianFuelListEntry const *prLeft, fc_sxm_trCanadianFuelListEntry const *prRight) const;

    // compare-fn that shall filter according to fc_sxm_trFuelListFilterCfg
    virtual bool bFilterPassed(fc_sxm_trCanadianFuelListEntry const &rItem) const;

 protected:
    /*
      will be called by base-class when data are ready and the list shall be sent to hmi
     */
    virtual tVoid vEmit(fc_sxm_trAdressing const &rAdressing);
#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
    virtual tVoid vStartTimer();
    virtual tBool bIsListModeTimerRunning(); 
    virtual tVoid vStopTimer();
#endif


 private:
    /*
      helper to get the fuel-price
    */
    tU32 u32GetFuelPrice(fc_sxm_trCanadianFuelListEntry const *prEntry) const;
    tVoid vFillListEntry(fc_sxm_trCanadianFuelListEntry const &roEntry, midw_ext_fi_tcl_CanFuelInfo &oFiEntry) const;
    tVoid vTraceFiEntry(midw_ext_fi_tcl_CanFuelInfo const &oFiEntry) const;
       /*Get Distace in km or Miles Units*/
    tBool bValidateEntry (fc_sxm_trCanadianFuelListEntry const *poFuelListEntry) const;
    midw_ext_sxm_canadian_fuelfi_tclMsgGetCanadianFuelInfoListMethodResult _oMResFuelInfoList;
    /*
          method to compare 2 Brand Names
    */
    bool bCompareBrandName(fc_sxm_trCanadianFuelListEntry const *prLeft, fc_sxm_trCanadianFuelListEntry const *prRight) const;
    /*
       helper to get Convert a String Characters to Uppercase
    */
    static tVoid ConvertToUpper(string &str);

    virtual tBool bIsListDataValid();
};


#endif //_FC_SXM_TCL_CANADIAN_FUEL_LIST_H_
