/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_canadian_fuel_dsrl.cpp
* @brief       Implementation for canadian fuel DSRL for specific location.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_canadian_fuel_types.h"
#include "fc_sxm_tcl_canadian_fuel_app.h"
#include "fc_sxm_tcl_canadian_fuel_dsrl.h"
#include "fc_sxm_tcl_canadian_fuel_list.h"
#include "fc_sxm_sms_util.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_CANADIAN_FUEL_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_canadian_fuel_dsrl.cpp.trc.h"
#endif

/*Define static variable*/
set<string> fc_sxm_tclCanadianFuelDSRL::_poBrandNameList;
#define FC_SXM_CANADIAN_FUEL_AGE_EXPIRY 4


tVoid fc_sxm_trCanadianFuelDSRLCfg::vPrintSpecific() const {
   /* for debuging purpose*/
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenSortMethod=%u", ETG_CENUM(fc_sxm_tenCanadianFuelDSRLSortMethod, enSortMethod)));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenFuelType  =%u", ETG_CENUM(FUEL_TYPE_ENUM, rFilterCfg.enCanadianFuelType)));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tsBrand      =%s", rFilterCfg.sBrand.c_str()));
}


fc_sxm_tclCanadianFuelDSRL::fc_sxm_tclCanadianFuelDSRL(fc_sxm_tenDSRLType enDsrlType):
    fc_sxm_tclConfigDSRL<fc_sxm_trCanadianFuelDSRLCfg>(enDsrlType,
                                               fc_sxm_tclCanadianFuelApp::instance(),
                                               TR_CLASS_FC_SXM_CANADIAN_FUEL_DSRL)
{
   /*Initialize flag to TRUE For first time Update of brand Name*/
   _u8BrandUpdateStatus=TRUE;
}

/*********************************************************************
 *
 *FUNCTION:     ~fc_sxm_tclCanadianFuelDSRL
 *
 *DESCRIPTION:  Destructor
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclCanadianFuelDSRL::~fc_sxm_tclCanadianFuelDSRL()
{
    ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL destructor"));


}

/*********************************************************************
 *
 *FUNCTION:     vOnDSRLUpdate
 *
 *DESCRIPTION:  Called when DSRL is updated
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclCanadianFuelDSRL::vOnDSRLUpdate(tVoid)
{
    /*Update the property to indicate clients*/
    ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vOnDSRLUpdate %d",enGetDSRLType()));
    fc_sxm_tcl_trCanadianFuelPropertyBrandNameListStatus oBrandProperty;
    midw_ext_sxm_canadian_fuelfi_tclMsgCanadianFuelBrandNameListStatus &oFiMsg=oBrandProperty.oFiMsg;

/*Do not send property for Favorite DSRL*/
    if(fc_sxm_enDSRLType_Favorites != enGetDSRLType())  {
       /*Check For New Band Updated from SMS*/
       if(TRUE==_u8BrandUpdateStatus)
       {
          ETG_TRACE_USR4(("vOnDSRLUpdate: Brand Name Update to FI"));
          /*Clear brand name list from Fi to create new*/
          oFiMsg.BrandName.BrandNameList.clear();
          /*Default Brand Name: All Brands*/
          oFiMsg.BrandName.BrandNameList.push_back("All Brands");
          /*Update Available Brand Name From SMS*/
          SXM_FOREACH(set<string>,iter,fc_sxm_tclCanadianFuelDSRL::_poBrandNameList)
          {
            oFiMsg.BrandName.BrandNameList.push_back((*iter).c_str());
          }
          /*Trace : Brand Name List*/
          for (tU32 i=0; i<oBrandProperty.oFiMsg.BrandName.BrandNameList.size(); i++)
          {
            ETG_TRACE_USR4(("Brand : %s", oBrandProperty.oFiMsg.BrandName.BrandNameList[i].szValue));
          }
          /*Notify to Fi for new brand list*/
          fc_sxm_tclCanadianFuelProperties::instance()->oBrandNameStatus.vSetAndNotify(oBrandProperty);
       }

       /*Update TRUE if any new Brand Found*/
       _u8BrandUpdateStatus=FALSE;
       
       /*Send fuel type Update*/
       fc_sxm_tclCanadianFuelApp::instance()->vSendCanadianFuelTypes();
    }

    ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vOnDSRLUpdate END"));
}

/*********************************************************************
 *
 *FUNCTION:     cb_bDSRLFilter
 *
 *DESCRIPTION:  Fuel service DSRL filter callback
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclCanadianFuelDSRL::bHandleFilterCallback(DSRL_OBJECT hDSRL,
                                                DSRL_ENTRY_OBJECT hEntry)
{
    /* This line is added to avoid LINT warnings. Please
      remove as soon as the variables are used.*/
    (tVoid) hDSRL; 
    (tVoid) hEntry;
    //    ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::cb_bDSRLFilter"));
    #if 0
    fc_sxm_trCanadianFuelDSRLFilterCfg  const &rFilterCfg= rGetCurrentConfig().rFilterCfg;

    /* peha: better define variables only in context where they are used*/
    FUEL_STATION_OBJECT hFuelStation;
    /*Extract the fuel station handle*/
    hFuelStation = DSRL_ENTRY.hFuelStation(hEntry);

    if (hFuelStation == FUEL_STATION_INVALID_OBJECT)
    {
        /*We don't want invalid stations!*/
        ETG_TRACE_ERR(("Invalid Station in DSRL filter"));
        return FALSE;
    }

    /*Find out how many fuel types we have*/
    tU8 u8NumTypes = FUEL_STATION.un8NumAvailableFuelTypes(hFuelStation);

    /*Check if the user has specified the fuel type*/
    tBool bTypeFound = TRUE;
    if (rFilterCfg.enFuelType != FUEL_TYPE_UNKNOWN)
    {
        bTypeFound = FALSE;
        STRING_OBJECT sFuelName;
        FUEL_TYPE_ENUM enFuelType;
        tBool bAvailable = FALSE;

        /* peha: allways use tUInt or tU32 as counter. */
        for (tU32 u32Index = 0; u32Index < u8NumTypes; u32Index++)
        {
            /*Get the fuel info for this entry*/
            SMSAPI_RETURN_CODE_ENUM eReturnCode = FUEL_STATION.eFuelInfo(hFuelStation,
                                                                         u32Index, &enFuelType, &sFuelName, &bAvailable, NULL, NULL);

            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                ETG_TRACE_ERR(("FUEL_STATION.eFuelInfo failed, status = %d", eReturnCode));
                /* peha: why do we return here (next;???)*/
                return FALSE;
            }

            /*Check if the fuel type matches*/
            if(enFuelType == rFilterCfg.enFuelType)
            {
                bTypeFound = TRUE;
                /* peha: break ?*/
                break;
            }
        }
    }

    /*Check if the brand is specified*/
    tBool bBrandFound = TRUE;
    if (rFilterCfg.sBrand.length() != 0)
    {
        bBrandFound = FALSE;
        /*Get the fuel station brand name*/
        STRING_OBJECT sFuelName = FUEL_STATION.hBrand(hFuelStation);

        /*Compare the brand name*/
        if (STRING.n16CompareCStr(rFilterCfg.sBrand.c_str(), 0, sFuelName) == 0)
        {
            bBrandFound = TRUE;
        }
    }

    return (bTypeFound && bBrandFound);
    #endif
    return TRUE;
}

/*********************************************************************
 *
 *FUNCTION:     cb_n16DSRLSort
 *
 *DESCRIPTION:  Fuel service DSRL sort callback
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tS16 fc_sxm_tclCanadianFuelDSRL::s16HandleSortCallback(DSRL_OBJECT hDSRL,
                                               DSRL_ENTRY_OBJECT hEntry1,
                                               DSRL_ENTRY_OBJECT hEntry2)
{  
    (tVoid) hDSRL; 
    tS16 n16Return = 0;
	
    //ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::cb_n16DSRLSort"));

    /* Default sorting is only required to get nearest fuel station
       for nearby and near destination DSRL.*/
    if(fc_sxm_enDSRLType_Favorites != enGetDSRLType()) {

       fc_sxm_tenCanadianFuelDSRLSortMethod enSortMethod = rGetCurrentConfig().enSortMethod;

       /*Check the sort method selected*/
       if(enSortMethod == fc_sxm_enCanadianFuelDSRLSortMethod_DISTANCE) {
           FUEL_STATION_OBJECT hStation1, hStation2;
           LOCATION_OBJECT hLocation1, hLocation2, hClosestLoc;
		   
		   hLocation1 = LOCATION_INVALID_OBJECT;
		   hLocation2 = LOCATION_INVALID_OBJECT;

           /* Extract stations & locations*/
           hStation1 = DSRL_ENTRY.hFuelStation(hEntry1);
           if (FUEL_STATION_INVALID_OBJECT != hStation1)
        	   hLocation1 = FUEL_STATION.hLocation(hStation1);

       	   hStation2 = DSRL_ENTRY.hFuelStation(hEntry2);
           if (FUEL_STATION_INVALID_OBJECT != hStation2)
        	   hLocation2 = FUEL_STATION.hLocation(hStation2);

           if (LOCATION_INVALID_OBJECT != hLocation1 &&
        	   LOCATION_INVALID_OBJECT != hLocation2) {

        	   LOCATION_OBJECT hSMSLocation = hGetSMSLocation();

        	   if (LOCATION_INVALID_OBJECT != hSMSLocation) {
				   /* Determine the closest location*/
				   hClosestLoc = LOCATION.hClosest(hSMSLocation,
												   hLocation1, hLocation2);

				   if (hClosestLoc != LOCATION_INVALID_OBJECT) {
					   if (hClosestLoc == hLocation1) {
						   n16Return = -1;
					   }
					   else {
						   n16Return = 1;
					   }
				   }
				   else {
					   ETG_TRACE_ERR(("LOCATION.hClosest failed"));
				   }
			   }
			   else {
				   ETG_TRACE_ERR(("hSMSLocation invalid"));
			   }
           }
           else {
        	   ETG_TRACE_ERR(("hLocation1 or hLocation2 invalid"));
           }
       }
       else if (enSortMethod == fc_sxm_enCanadianFuelDSRLSortMethod_PRICE) {
         /*Default Sort for price is not required .*/
       }
    }

    return n16Return;
}

/*********************************************************************
 *
 *FUNCTION:     vHandleDsrlEntryInvalid
 *
 *DESCRIPTION:  Removes invalid entry
 *
 *PARAMETER:    [IN]tEntryID - Entry id

 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclCanadianFuelDSRL::vHandleDsrlEntryInvalid(DSRL_ENTRY_ID tEntryID) {
    ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vHandleDsrlEntryInvalid(tEntryID=%d) (ListId=%d)",
                    tEntryID,u32GetId()));

    fc_sxm_trCanadianFuelList::vRemoveEntry(tEntryID);/* remove invalid entry*/
}
/*********************************************************************
 *
 *FUNCTION:     vApplyTestModification
 *
 *DESCRIPTION:  
 *
 *PARAMETER:    [IN]tEntryID -
                rListEntry -
 *              [Out]eStatus -
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclCanadianFuelDSRL::vApplyTestModification(DSRL_ENTRY_ID tEntryID, fc_sxm_trCanadianFuelListEntry &rListEntry, DSRL_ENTRY_STATUS_ENUM &eStatus) {
    ETG_TRACE_USR4(("vApplyTestModification: tEntryID=%u received=%u (active=%d)",
                    _rTestModification.u32Id, tEntryID, _rTestModification.bActive));

    if (!_rTestModification.bActive || _rTestModification.u32Id!=tEntryID) {
        return;
    }
    if (_rTestModification.bOnce) {
        _rTestModification.bActive=FALSE;
    }
    ETG_TRACE_USR4(("vApplyTestModification: modify entry tEntryID=%u sxmId=0x%08x (type=%d)",
                    tEntryID,
                    rListEntry.u32SxmId,
                    ETG_CENUM(fc_sxm_trDsrlTestModification::tenType, _rTestModification.enType)));
    switch(_rTestModification.enType) {
        case fc_sxm_trDsrlTestModification::enType_EntryRemoved:
            eStatus=DSRL_ENTRY_STATUS_REMOVED;
            break;
        case fc_sxm_trDsrlTestModification::enType_Change1:
            ETG_TRACE_USR4(("vApplyTestModification: enType_Change1: modify price to %u",
                            _rTestModification.u32Val));
            // modify prices of all fuel-types
            SXM_FOREACH_MAP(FUEL_TYPE_ENUM, fc_sxm_trCanadianFuelTypeListEntry, iter, rListEntry.listFuelTypes) {
                iter->second.u32Price=_rTestModification.u32Val;
            }
            break;
        default:
            ETG_TRACE_USR4(("Unknown test-modification"));
            break;
    }
}
/*********************************************************************
 *
 *FUNCTION:     cb_bDSRLIterate
 *
 *DESCRIPTION:  Fuel service DSRL iterate callback
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclCanadianFuelDSRL::bHandleIterateCallback(DSRL_OBJECT hDSRL,
                                                 DSRL_ENTRY_ID tEntryID,
                                                 DSRL_ENTRY_STATUS_ENUM eStatus,
                                                 DSRL_ENTRY_OBJECT hEntryObject)
{
   /* This line is added to avoid LINT warnings. Please
   remove as soon as the variables are used.*/
   (tVoid) hDSRL;
   (tVoid) tEntryID; 
    ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::cb_bDSRLIterate : %d",enGetDSRLType()));
   fc_sxm_trCanadianFuelListEntry rListEntry;
   FUEL_STATION_OBJECT hStation;
   tU32 u32NumPrices;
   LOCATION_OBJECT hStationLocation;
   LOCID_OBJECT hLocID;
   LOC_ID hThisID;
   SMSAPI_RETURN_CODE_ENUM eReturnCode;
   tU32 u32PriceIndex;
   midw_ext_fi_tcl_SxmLocation trFuelLoc;

   /*Get this entry's current favorite status*/
   rListEntry.bFavorite = FAVORITES.bIsFavorite(hEntryObject);
   /*Extract the station object*/
   hStation = DSRL_ENTRY.hFuelStation(hEntryObject);
   if (hStation == FUEL_STATION_INVALID_OBJECT)
   {
      ETG_TRACE_ERR(("Station Object not created!"));
      return TRUE;
   }
   vGetAmenity(hStation,rListEntry.oAmenity);
   vGetAmenityEnum(hStation,rListEntry.ListAmenity);
   /*Extract relevant portions of station*/
   hStationLocation = FUEL_STATION.hLocation(hStation);
   fc_sxm_vCopySmsString2Stl(LOCATION.hDescription(hStationLocation),rListEntry.oStationName);
   fc_sxm_vCopySmsString2Stl(FUEL_STATION.hBrand(hStation),rListEntry.oBrandName);
   /*Collect Available Brand Name*/
   if(0==fc_sxm_tclCanadianFuelDSRL::_poBrandNameList.count(rListEntry.oBrandName))
   {
      /*Trace all new Brand Name Found*/
      ETG_TRACE_USR4(("New Brand Name Found %s",rListEntry.oBrandName.c_str()));
      /* Set the status flag if any new brand found*/
      _u8BrandUpdateStatus=TRUE;
   }
   fc_sxm_tclCanadianFuelDSRL::_poBrandNameList.insert(rListEntry.oBrandName);

   /*Get Fuel Info a Fuel station*/
   hLocID = LOCATION.hLocID(hStationLocation);
   hThisID = LOCID.tID(hLocID);
   /*Get Sxm ID*/
   rListEntry.u32SxmId=hThisID;
   rListEntry.u32DsrlEntryID=tEntryID;
   rListEntry.hLocation=hStationLocation;
   /*Get Lat Lon From Location Object*/
   trFuelLoc=fc_sxm_tclCanadianFuelApp::instance()->rGetCanadianFuelLocation(rListEntry.hLocation);
   rListEntry.oLat = trFuelLoc.Lat;
   rListEntry.oLon = trFuelLoc.Lon;
   /*Get Address From a fuel station*/
   fc_sxm_vCopySmsString2Stl(LOCATION.hStreetName(hStationLocation),rListEntry.oStreetName);
   fc_sxm_vCopySmsString2Stl(LOCATION.hCity(hStationLocation),rListEntry.oCity);
   fc_sxm_vCopySmsString2Stl(LOCATION.hState(hStationLocation),rListEntry.oState);
   fc_sxm_vCopySmsString2Stl(LOCATION.hZipCode(hStationLocation),rListEntry.oZipCode);
   fc_sxm_vCopySmsString2Stl(LOCATION.hPhone(hStationLocation),rListEntry.oPhone);

   /*Trace to get all fuel info*/
   ETG_TRACE_USR4(("Display FUEL Info List"));
   ETG_TRACE_USR4(("tEntryID :: %u",tEntryID));
   ETG_TRACE_USR4(("Brand Name :: %s",rListEntry.oBrandName.c_str()));
   ETG_TRACE_USR4(("Station Name:: %s",rListEntry.oStationName.c_str()));
   ETG_TRACE_USR4(("SXM Loc ID :: %u",rListEntry.u32SxmId));
   ETG_TRACE_USR4(("Street Name:: %s",rListEntry.oStreetName.c_str()));
   ETG_TRACE_USR4(("City :: %s",rListEntry.oCity.c_str()));
   ETG_TRACE_USR4(("Zip Code :: %s",rListEntry.oZipCode.c_str()));
   ETG_TRACE_USR4(("Phone No :: %s",rListEntry.oPhone.c_str()));
   ETG_TRACE_USR4(("Lat :: %d",rListEntry.oLat));
   ETG_TRACE_USR4(("Lon::%d",rListEntry.oLon));
   ETG_TRACE_USR4(("State :: %s",rListEntry.oState.c_str()));
   ETG_TRACE_USR4(("Distance In KM::%7f",rListEntry.DistanceKm));
   ETG_TRACE_USR4(("Distance In Miles::%7f",rListEntry.DistanceMiles));
   ETG_TRACE_USR4(("Favorite Info::%d",rListEntry.bFavorite));
   ETG_TRACE_USR4(("Amenity::%s",rListEntry.oAmenity.c_str()));
   
   SXM_FOREACH(vector<midw_ext_fi_tcl_e8_CanFuelStationAmenity>,iter,rListEntry.ListAmenity)
   {
      ETG_TRACE_USR4(("Amenity Enum::%d",ETG_CENUM(midw_ext_fi_tcl_e8_CanFuelStationAmenity::tenType,(*iter).enType)));
   }
   
   /*Extract the fuel types*/
   u32NumPrices = (tU32)FUEL_STATION.un8NumAvailableFuelTypes(hStation);
   rListEntry.u32NumPrices=u32NumPrices;
   ETG_TRACE_USR4(("Number of fuel type available :: %d", rListEntry.u32NumPrices));

   /*Grab this entry and extract its price*/
   for(u32PriceIndex = 0; u32PriceIndex < u32NumPrices; u32PriceIndex++)
   {
       FUEL_INFO_STRUCT sFuelInfo;
       eReturnCode = FUEL_STATION.eFuelInfo(hStation,
                                            (tU8)u32PriceIndex,
                                            &sFuelInfo);
       if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
       {
           ETG_TRACE_ERR(("Fuel Info not created!"));
           return FALSE;
       }

      /*Get Fuel type Info*/
      fc_sxm_trCanadianFuelTypeListEntry rEntry;
      if((sFuelInfo.tInfoMask & FUEL_INFO_PRICE) == FUEL_INFO_PRICE) 
      {
         rEntry.u32Price=sFuelInfo.un32PriceInTenthsOfLowestDenomination;
      }
      rEntry.enCanadianFuelType=sFuelInfo.eFuelType;
      rEntry.bAvailable=(sFuelInfo.eAvailability==FUEL_AVAILABLE);
      if((sFuelInfo.tInfoMask & FUEL_INFO_TIME) == FUEL_INFO_TIME) 
      {
         rEntry.u32AgeInDays=sFuelInfo.un32ReportTime;
      }
      // peha todo: long or short name?
      fc_sxm_vCopySmsString2Stl(sFuelInfo.hShortFuelName,rEntry.oFuelName);

      if (rEntry.enCanadianFuelType==FUEL_TYPE_REGULAR) {
                ETG_TRACE_USR4(("Regular Price: %u", rEntry.u32Price));
      }
      ETG_TRACE_USR4(("***** Fuel Info For Each Type *****"));

      ETG_TRACE_USR4(("Fuel Type :: %d",rEntry.enCanadianFuelType));
      ETG_TRACE_USR4(("Fuel Name:: %s",rEntry.oFuelName.c_str()));
      ETG_TRACE_USR4(("Age of Fuel :: %d",rEntry.u32AgeInDays));
      ETG_TRACE_USR4(("Fuel Price :: %u",rEntry.u32Price));
	  if(rEntry.bAvailable == TRUE)
	  {
	     ETG_TRACE_USR4(("Fuel Price Available"));
	  }
	  else
	  {
	     ETG_TRACE_USR4(("Fuel Price Sold Out"));
	  }
	  //We are interested in valid age only
	  if(rEntry.u32AgeInDays < FC_SXM_CANADIAN_FUEL_AGE_EXPIRY )
	  {
         rListEntry.listFuelTypes.insert(pair<FUEL_TYPE_ENUM, fc_sxm_trCanadianFuelTypeListEntry>(rEntry.enCanadianFuelType, rEntry));
	  }
	  else
	  {
	      ETG_TRACE_USR4(("Fuel Price Age is older than %d days",FC_SXM_CANADIAN_FUEL_AGE_EXPIRY));
	  }
   }
   vApplyTestModification(tEntryID, rListEntry, eStatus);
   
    /* framework has set correct list for us ...*/
    ETG_TRACE_USR4(("Send correct Fuel List to framework"));
    fc_sxm_trCanadianFuelList::vUpdateEntry(eStatus, rListEntry);
    
    return TRUE;
}

/*********************************************************************
 *
 *FUNCTION:     vGetAmenity
 *
 *DESCRIPTION:  Get Available Amenitys
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclCanadianFuelDSRL::vGetAmenity(FUEL_STATION_OBJECT hStation, string &sAmenity) const
{
   ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vGetAmenity START"));
   AMENITY_STRUCT asAmenities[FUEL_STATION_NUM_AMENITIES];
   SMSAPI_RETURN_CODE_ENUM eReturnCode;
   UN8 unAvailAmenity=1;

   /*Initialize Status*/
   for(UN8 tIndex = 0; tIndex < FUEL_STATION_NUM_AMENITIES; tIndex++)
   {
        /*Invalidate all the amenity entries*/
        asAmenities[tIndex].eStatus = AMENITY_STATUS_INVALID;
   }
   eReturnCode = FUEL_STATION.eAmenities(
                                hStation,
                                FUEL_STATION_NUM_AMENITIES,
                                &asAmenities[0]);
   if(eReturnCode==SMSAPI_RETURN_CODE_SUCCESS)
   {
      for (UN8 tIndex = 0; tIndex < FUEL_STATION_NUM_AMENITIES; tIndex++)
      {
         if(asAmenities[tIndex].eStatus == AMENITY_STATUS_UNAVAILABLE)
         {
            unAvailAmenity=(tU8)(unAvailAmenity+1u);
         }
         else if(asAmenities[tIndex].eStatus==AMENITY_STATUS_AVAILABLE)
         {
//            /*Get all available Amenitys*/
//            switch (asAmenities[tIndex].uAmenity.eFuelStationAmenity)
//            {
//               case FUEL_STATION_AMENITY_OPEN_24HR:
//                {
//                  sAmenity+="Open 24 Hours ";
//                }
//                break;
//               case FUEL_STATION_AMENITY_EMERGENCY_ROAD_SERVICE:
//                {
//                    sAmenity+="Emergency Road Service ";
//                }
//                break;
//               case FUEL_STATION_AMENITY_FULL_SERVICE:
//                {
//                    sAmenity+="Full Service ";
//                }
//                break;
//               case FUEL_STATION_AMENITY_OIL_CHANGE:
//                {
//                    sAmenity+="Oil Change Service ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_INTERSTATE_ACCESS_UNDER_HALF_MILE:
//                {
//                    sAmenity+="Interstate Access (less than .5 miles) ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_INTERSTATE_ACCESS_OVER_HALF_MILE:
//                {
//                    sAmenity+="Interstate Access (over .5 miles) ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_CASH_DISCOUNT:
//                {
//                    sAmenity+="Cash Discount ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_COVENIENCE_STORE:
//                {
//                    sAmenity+="Convenience Store ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_SUPERMARKET:
//                {
//                    sAmenity+="Supermarket ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_SNACKS_FAST_FOOD:
//                {
//                    sAmenity+="Snacks/Fast Food ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_RESTAURANT:
//                {
//                    sAmenity+="Restaurant ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_TRUCK_STOP:
//                {
//                    sAmenity+="Truck Stop ";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL:
//                {
//                    sAmenity+="Truck Stop with Hotel ";
//                }
//                break;
//                case FUEL_STATION_AMENITY_AFFILIATED_NETWORK_CARDS_ACCEPTED:
//                {
//                    sAmenity += "Affiliated Service Cards Accepted";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_ALL_NETWORK_CARDS_ACCEPTED:
//                {
//                    sAmenity += "All Service Cards Accepted";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_CREDIT_DEBIT:
//                {
//                    sAmenity += "Credit/Debit Accepted";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_PUBLIC_ACCESS:
//                {
//                    sAmenity += "Public Access";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_PATRONS_ONLY:
//                {
//                    sAmenity += "Patrons Only";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_CASH_ACCEPTED:
//                {
//                    sAmenity += "Cash Accepted";
//                }
//                break;
//
//                case FUEL_STATION_AMENITY_RESERVATIONS:
//                {
//                    sAmenity += "Reservations";
//                }
//                break;
//                default:
//                break;
//            }

        	 vGetAllAmenity(asAmenities[tIndex].uAmenity.eFuelStationAmenity, sAmenity);
         }
         else
         {
           // ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vGetAmenity INVALID"));
         }
      }
      if(unAvailAmenity==FUEL_STATION_NUM_AMENITIES)
      {
        sAmenity="Amenity is Unavailable";
      }
   }
   ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vGetAmenity %s End",sAmenity.c_str()));
}


tVoid fc_sxm_tclCanadianFuelDSRL::vGetAllAmenity(FUEL_STATION_AMENITY_ENUM eAmenity, string &sAmenity) const
{
	/*Get all available Amenitys*/
		switch (eAmenity)
		{
		   case FUEL_STATION_AMENITY_OPEN_24HR:
			{
			  sAmenity+="Open 24 Hours ";
			}
			break;
		   case FUEL_STATION_AMENITY_EMERGENCY_ROAD_SERVICE:
			{
				sAmenity+="Emergency Road Service ";
			}
			break;
		   case FUEL_STATION_AMENITY_FULL_SERVICE:
			{
				sAmenity+="Full Service ";
			}
			break;
		   case FUEL_STATION_AMENITY_OIL_CHANGE:
			{
				sAmenity+="Oil Change Service ";
			}
			break;

			case FUEL_STATION_AMENITY_INTERSTATE_ACCESS_UNDER_HALF_MILE:
			{
				sAmenity+="Interstate Access (less than .5 miles) ";
			}
			break;

			case FUEL_STATION_AMENITY_INTERSTATE_ACCESS_OVER_HALF_MILE:
			{
				sAmenity+="Interstate Access (over .5 miles) ";
			}
			break;

			case FUEL_STATION_AMENITY_CASH_DISCOUNT:
			{
				sAmenity+="Cash Discount ";
			}
			break;

			case FUEL_STATION_AMENITY_COVENIENCE_STORE:
			{
				sAmenity+="Convenience Store ";
			}
			break;

			case FUEL_STATION_AMENITY_SUPERMARKET:
			{
				sAmenity+="Supermarket ";
			}
			break;

			case FUEL_STATION_AMENITY_SNACKS_FAST_FOOD:
			{
				sAmenity+="Snacks/Fast Food ";
			}
			break;

			case FUEL_STATION_AMENITY_RESTAURANT:
			{
				sAmenity+="Restaurant ";
			}
			break;

			case FUEL_STATION_AMENITY_TRUCK_STOP:
			{
				sAmenity+="Truck Stop ";
			}
			break;

			case FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL:
			{
				sAmenity+="Truck Stop with Hotel ";
			}
			break;
			case FUEL_STATION_AMENITY_AFFILIATED_NETWORK_CARDS_ACCEPTED:
			{
				sAmenity += "Affiliated Service Cards Accepted";
			}
			break;

			case FUEL_STATION_AMENITY_ALL_NETWORK_CARDS_ACCEPTED:
			{
				sAmenity += "All Service Cards Accepted";
			}
			break;

			case FUEL_STATION_AMENITY_CREDIT_DEBIT:
			{
				sAmenity += "Credit/Debit Accepted";
			}
			break;

			case FUEL_STATION_AMENITY_PUBLIC_ACCESS:
			{
				sAmenity += "Public Access";
			}
			break;

			case FUEL_STATION_AMENITY_PATRONS_ONLY:
			{
				sAmenity += "Patrons Only";
			}
			break;

			case FUEL_STATION_AMENITY_CASH_ACCEPTED:
			{
				sAmenity += "Cash Accepted";
			}
			break;

			case FUEL_STATION_AMENITY_RESERVATIONS:
			{
				sAmenity += "Reservations";
			}
			break;
			default:
			break;
		}
}
/*********************************************************************
 *
 *FUNCTION:     vGetAmenity
 *
 *DESCRIPTION:  Get fuel Amenitys as enum type
 *
 *PARAMETER:    Fuel Station
 *
 *RETURNVALUE:  list of Amenity Enum types
 *
 ********************************************************************/
tVoid fc_sxm_tclCanadianFuelDSRL::vGetAmenityEnum(FUEL_STATION_OBJECT hStation, vector<midw_ext_fi_tcl_e8_CanFuelStationAmenity> &hAmenity) const
{
   ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vGetAmenity(Enum) START"));
   AMENITY_STRUCT asAmenities[FUEL_STATION_NUM_AMENITIES];
   SMSAPI_RETURN_CODE_ENUM eReturnCode;
   UN8 unAvailAmenity=1;

   /*Initialize Status*/
   for(UN8 tIndex = 0; tIndex < FUEL_STATION_NUM_AMENITIES; tIndex++)
   {
        /*Invalidate all the amenity entries*/
        asAmenities[tIndex].eStatus = AMENITY_STATUS_INVALID;
   }
   /*SMS Api to get available fuel amenities.*/
   eReturnCode = FUEL_STATION.eAmenities(
                                hStation,
                                FUEL_STATION_NUM_AMENITIES,
                                &asAmenities[0]);
   if(eReturnCode==SMSAPI_RETURN_CODE_SUCCESS)
   {
      for (UN8 tIndex = 0; tIndex < FUEL_STATION_NUM_AMENITIES; tIndex++)
      {
         if(asAmenities[tIndex].eStatus == AMENITY_STATUS_UNAVAILABLE)
         {
            /*Increment the status to check amenity is available for a fuel station*/
            unAvailAmenity= (tU8)(unAvailAmenity+1u);
         }
         else if(asAmenities[tIndex].eStatus==AMENITY_STATUS_AVAILABLE)
         {
            midw_ext_fi_tcl_e8_CanFuelStationAmenity rAmenity;
            /*Get all available Amenitys*/ 
            rAmenity.enType=(midw_ext_fi_tcl_e8_CanFuelStationAmenity::tenType)asAmenities[tIndex].uAmenity.eFuelStationAmenity;
            hAmenity.push_back(rAmenity);                       
         }
         else
         {
            //ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vGetAmenity INVALID"));
         }
      }
      if(unAvailAmenity==FUEL_STATION_NUM_AMENITIES)
      {
         ETG_TRACE_USR4(("fc_sxm_tclCanadianFuelDSRL::vGetAmenity Unavailable"));
        //hAmenity.push_back((midw_ext_fi_tcl_e8_CanFuelTypeEnum::tenType)FUEL_STATION_NUM_AMENITIES);
      }
   }
}
