/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_agw_tiles.cpp
* @brief       Implementation of the AGW_TILE product
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_service_sxm_agw.h"
#include "fc_sxm_tcl_agw_app.h"
#include "fc_sxm_tcl_agw_tiles.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AGW_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_agw_tiles.cpp.trc.h"
#endif

/*
  ####### Base-Class for all tile-Products #######
*/
fc_sxm_tclTileProductBase::fc_sxm_tclTileProductBase(AGW_PRODUCT_TYPE_ENUM enProductType_, 
                                                     midw_ext_fi_tcl_e8_SXMTileType::tenType enTileType_, 
                                                     midw_ext_fi_tcl_e8_SXMTileDataType::tenType enTileDataType_,
                                                     string oSharedMemName,
                                                     tU32 u32SharedMemSize,
                                                     tS32 s32SharedMemTimeOutMs_
                                                     ):
    fc_sxm_tclAgwProductBase(enProductType_),
    enTileType(enTileType_),
    enTileDataType(enTileDataType_),
    oSharedMem(oSharedMemName, u32SharedMemSize),
    u32SharedMemTimeOutMs((tU32)s32SharedMemTimeOutMs_)
{
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::CTOR %d",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    oFiResult.TileType.enType=enTileType_;
    _oSharedMemTimer.vSetMsg(fc_sxm_trMsgAgwSharedMemTimeOut(enProductType));
    fc_sxm_tclAgwTiles::instance()->vAddProduct(this);  
}

fc_sxm_tclTileProductBase::~fc_sxm_tclTileProductBase() {
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::DTOR %d",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    fc_sxm_tclAgwTiles::instance()->vRemoveProduct(this);  
};


tBool fc_sxm_tclTileProductBase::bIsLocked() {
    tBool bRes = _oSharedMemTimer.bIsRunning();
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::bIsLocked(%d) --> bRes=%d",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType),
                    bRes));
    return bRes;
}

midw_ext_fi_tcl_SXMSharedMemoryDescription fc_sxm_tclTileProductBase::oGetDescription() {
    midw_ext_fi_tcl_SXMSharedMemoryDescription oDescription;
    oDescription.TileType.enType=enTileType;
	oDescription.TileDataType.enType=enTileDataType;
    oDescription.SharedMemoryName=oSharedMem.oGetName().c_str();
    oDescription.SharedMemorySize=oSharedMem.u32GetSize();
    oDescription.Timeout=u32SharedMemTimeOutMs;
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::oGetDescription(%d) size=%u enTileType=%d u32SharedMemTimeOutMs=%u SharedMemName=%s",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType),
                    oSharedMem.u32GetSize(),
                    enTileType,
                    u32SharedMemTimeOutMs,
                    oSharedMem.oGetName().c_str() ));
    return oDescription;
}


tBool fc_sxm_tclTileProductBase::bLock() {

    if (bIsLocked()) {
        ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::bLock(%d) Already Locked",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
        return FALSE;
    }
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::bLock(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    // start timer
    _oSharedMemTimer.vStart(u32SharedMemTimeOutMs);

    return TRUE;
}

tVoid fc_sxm_tclTileProductBase::vUnLock() {
    // stop timer
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vUnLock(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));

    _oSharedMemTimer.vStop();
}

tBool fc_sxm_tclTileProductBase::bAquire(fc_sxm_trAdressing rAdressing) {
    if (!bLock()) {
        ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::bAquire(%d) LOCKED",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
		oFiResult.AccessGranted = FALSE;

       
    }
    //make AccessGranted = TRUE, as per modified FI 2.0
	else {

			oFiResult.AccessGranted = TRUE;
	}

    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::bAquire(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    // Send the already prepared fi-message desribing the shared-memory.
  
	//added as UpdateCounter modified FI 2.0
	oFiResult.UpdateCounter=u32UpdateCounter;
	ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vAddTile u32UpdateCounter(%d) ",
                    oFiResult.UpdateCounter));
	if(AIL_EN_N_NO_ERROR ==  fc_sxm_tclAgwService::instance()->enSendFiMessage(rAdressing, oFiResult))
    {
        ETG_TRACE_USR4(("AGW Tile Data result sent successfully"));
    }
    else
    {
        ETG_TRACE_USR4(("AGW Tile Data result send failed"));
    }


    return TRUE;
}

tVoid fc_sxm_tclTileProductBase::vRelease(fc_sxm_trAdressing rAdressing) {
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vRelease(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    vUnLock();

    // Reset the share memory
    oSharedMem.vReset();
	
    if (bPendingChange) {

        ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vRelease(%d): bPendingChange",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
        // dsrl has changed while shared mem was locked
        if (fc_sxm_tclAgwApp::instance()->bIsDsrlReady()) {
            ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vRelease(%d): DsrlReady",
                            ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
            bPendingChange=FALSE;
            bChanged=TRUE;
            fc_sxm_tclAgwApp::instance()->vReadDsrl();
            vNotify();
        }
    }

    if (rAdressing.bIsValid()) {
        midw_ext_sxm_agwfi_tclMsgReleaseAccessFromSharedMemoryMethodResult oMRes;
        fc_sxm_tclAgwService::instance()->enSendFiMessage(rAdressing, oMRes);
    }
}


tVoid fc_sxm_tclTileProductBase::vClearOutput() {
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vClearOutput(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    oFiResult.TileList.clear();
    if (!bIsLocked()) {
        oSharedMem.vReset();
    }
}


tVoid fc_sxm_tclTileProductBase::vAddTile(AGW_TILE_OBJECT hAgwTile) {
    // 1.) Write tile to shared mem
    tU32 u32Offset;
    tU32 u32TileSize=oSharedMem.u32Write(hAgwTile, &u32Offset);
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::vAddTile(%d) u32Offset=%u u32TileSize=%u",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType),
                    u32Offset,
                    u32TileSize));
    if (0==u32TileSize) {
        ETG_TRACE_USR1(("fc_sxm_tclTileProductBase::vAddTile(%d) TILE EMPTY",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
        return;
    }
        
    // 2.) Add info to fi-message
    midw_ext_fi_tcl_SXMSharedMemoryTile oFiTile;
    //oFiTile.TileDataType.enType=enTileDataType;
    oFiTile.TileSize=u32TileSize;
    oFiTile.AddressOffset=u32Offset;
    // todo: fill generic part of oFiTile according to hAgwTile here

    // if there is tile-specific information to be filled, virtual method vAddSpecificTileInfo() has to be implemented
    this->vAddSpecificTileInfo(hAgwTile, oFiTile);

    oFiResult.TileList.push_back(oFiTile);
}

/*
  ####### Container-class for all shape-products #######
*/
tVoid fc_sxm_tclAgwTiles::vProcessTimer(fc_sxm_trMsgAgwSharedMemTimeOut const *prMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclTileProductBase::fc_sxm_trMsgAgwSharedMemTimeOut(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, prMsg->enProductId)));
    fc_sxm_tclTileProductBase *poProduct=poGetProduct(prMsg->enProductId);
    if (OSAL_NULL != poProduct) {
        poProduct->vRelease();
    }
}

tVoid fc_sxm_tclAgwTiles::vProcess(fc_sxm_trMsgAgwMStartAquireSharedMemory const *prMsg) {
    midw_ext_fi_tcl_e8_SXMTileType::tenType enTileType=prMsg->oFiMsg.TileType.enType;
    // get Tiles Class
    fc_sxm_tclTileProductBase *poProduct=poGetProductByFiTileType(enTileType);
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartAquireSharedMemory(TileType=%d)",
                    enTileType));

    if (OSAL_NULL==poProduct) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartAquireSharedMemory(TileType=%d) NO PRODUCT FOUND",
                        enTileType));
        // send Error, todo: check error-code
        fc_sxm_tclAgwService::instance()->vSendError(prMsg->rAdressing, 0x05); // parameter out of range
        return;
    }
    if (!poProduct->bAquire(prMsg->rAdressing)) {
		//from FI 2.0 this condition never happens, due to inroduction of new parameter AccessGranted in FI result
        ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartAquireSharedMemory(%d) aquire failed",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
        // send Error, todo: check error-code
        fc_sxm_tclAgwService::instance()->vSendError(prMsg->rAdressing, 0x0A); // busy
        return;
    }
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartAquireSharedMemory(%d) OK",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));

}

tVoid fc_sxm_tclAgwTiles::vProcess(fc_sxm_trMsgAgwMStartReleaseSharedMemory const *prMsg) {
    midw_ext_fi_tcl_e8_SXMTileType::tenType enTileType=prMsg->oFiMsg.TileType.enType;
    fc_sxm_tclTileProductBase *poProduct=poGetProductByFiTileType(enTileType);
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartReleaseSharedMemory(TileType=%d)",
                    enTileType));
    if (OSAL_NULL==poProduct) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartReleaseSharedMemory(TileType=%d) NO PRODUCT FOUND",
                        enTileType));
        // send Error, todo: check error-code
        fc_sxm_tclAgwService::instance()->vSendError(prMsg->rAdressing, 0x05); // parameter out of range
        return;
    }
    poProduct->vRelease(prMsg->rAdressing);
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartReleaseSharedMemory(%d) OK",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
}


fc_sxm_tclAgwTiles::fc_sxm_tclAgwTiles() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::CTOR"));
}

fc_sxm_tclAgwTiles::~fc_sxm_tclAgwTiles() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::DTOR"));
    _mapProducts.clear();
}

fc_sxm_tclTileProductBase *fc_sxm_tclAgwTiles::poGetProduct(AGW_PRODUCT_TYPE_ENUM enProductType) {
    SXM_IF_FIND_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclTileProductBase*, iter, _mapProducts, enProductType) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::poGetProduct(%d) FOUND",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
        return iter->second;
    }
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::poGetProduct(%d) NOT FOUND",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    return OSAL_NULL;
}


fc_sxm_tclTileProductBase *fc_sxm_tclAgwTiles::poGetProductByFiTileType(midw_ext_fi_tcl_e8_SXMTileType::tenType enTileType) {
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::poGetProductByFiTileType(TileType%d)",
                    enTileType));

    SXM_FOREACH_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclTileProductBase*, iter, _mapProducts) {
        fc_sxm_tclTileProductBase *poProduct=iter->second;
        if (poProduct->enGetTileType()==enTileType) {
            ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::poGetProduct(%d) FOUND",
                            ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
            return poProduct;
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::poGetProductByFiTileType(TileType%d) NOT FOUND",
                    enTileType));
    return OSAL_NULL;
}


tVoid fc_sxm_tclAgwTiles::vProcess(fc_sxm_trMsgAgwMStartExchangeCapabilities const *prMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::fc_sxm_trMsgAgwMStartExchangeCapabilities"));

    midw_ext_sxm_agwfi_tclMsgExchangeCapabilitiesMethodResult oMRes1;
    SXM_FOREACH_CONST(vector<midw_ext_fi_tcl_SXMSharedMemoryDescription>, iter, prMsg->oFiMsg.SharedMemories) {
        fc_sxm_tclTileProductBase *poProduct=poGetProductByFiTileType(iter->TileType.enType);
        if (OSAL_NULL != poProduct) {
            oMRes1.SharedMemories.push_back(poProduct->oGetDescription());
            // todo: add functionality to activate product (shared memory+filter)
        }
    }
    fc_sxm_tclAgwService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes1);
}



tVoid fc_sxm_tclAgwTiles::vAddProduct(fc_sxm_tclTileProductBase *poProduct) {
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::vAddProduct(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
        _mapProducts[poProduct->enProductType]=poProduct;
}

tVoid fc_sxm_tclAgwTiles::vRemoveProduct(fc_sxm_tclTileProductBase *poProduct) {
    ETG_TRACE_USR4(("fc_sxm_tclAgwTiles::vRemoveProduct(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
    SXM_IF_FIND_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclTileProductBase*, iter, _mapProducts, poProduct->enProductType) {
        _mapProducts.erase(iter);
    } else {
        ETG_TRACE_USR1(("fc_sxm_tclAgwTiles::vRemoveProduct(%d) NOT FOUND",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
    }
}

#if 0

tVoid fc_sxm_tclTileProductNowRadData::vNotify() {
    ETG_TRACE_USR4(("fc_sxm_tclTileProductNowRadData::vNotify(%d): u32UpdateCounter=%u",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType),
                    u32UpdateCounter));
    fc_sxm_tcl_trAgwPropertyNowRadDataStatus oNowRadDataStatus;
    oNowRadDataStatus.oFiMsg.UpdateCounter=u32UpdateCounter;
    fc_sxm_tclAgwProperties::instance()->oNowRadDataStatus.vSetAndNotify(oNowRadDataStatus);
};

#endif 





