/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_agw_surface_data.cpp
* @brief       Implementation of Surface Features & Pressures – information about surface ,
*              features like atmospheric fronts areas with similar temperature and fields of high or low pressure.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_tcl_agw_app.h"
#include "fc_sxm_tcl_agw_surface_data.h"
#include "fc_sxm_service_sxm_agw.h"
#include "fc_sxm_tcl_sxmapp_manager.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AGW_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_agw_surface_data.cpp.trc.h"
#endif


tVoid fc_sxm_tclShapeProductSurfaceData::vNotify() {
    fc_sxm_tcl_trAgwPropertySurfaceDataStatus oSurfaceDataStatus;
    oSurfaceDataStatus.oFiMsg.UpdateCounter=u32UpdateCounter;
    ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::vNotify() UpdateCounter = %d", u32UpdateCounter));
    fc_sxm_tclAgwProperties::instance()->oSurfaceDataStatus.vSetAndNotify(oSurfaceDataStatus);
};

tVoid fc_sxm_tclShapeProductSurfaceData::vEmit(fc_sxm_trAdressing rAdressing) {
	 oMResSurfaceData.UpdateCounter=u32UpdateCounter;
    ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::vEmit() u32UpdateCounter = %d", u32UpdateCounter));
    fc_sxm_tclAgwService::instance()->enSendFiMessage(rAdressing, oMResSurfaceData);
}; 


tBool fc_sxm_tclShapeProductSurfaceData::bAddShape(AGW_SHAPE_OBJECT hAgwShape, midw_ext_fi_tcl_SXMShape &roShapeObj)
{
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::bAddShape() -- Enter"));
    tBool bRes=TRUE;
	//data-structure to send via fi-message --end

	//populating the data-structure and posting it in fi-message

	AGW_SHAPE_TYPE_ENUM eShapeType = AGW_SHAPE.eShapeType(hAgwShape);

	switch(eShapeType)
    {
        case AGW_SHAPE_TYPE_FRONT:
			ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::bAddShape() -- AGW_SHAPE_TYPE_FRONT"));
            vGetFront(hAgwShape, roShapeObj);
            break;
		case AGW_SHAPE_TYPE_PRESSURE_CENTER:
			ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::bAddShape() -- AGW_SHAPE_TYPE_PRESSURE_CENTER"));
            vGetPressureCenter(hAgwShape, roShapeObj);
            break;	
		case AGW_SHAPE_TYPE_ISOBAR:
			ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::bAddShape() -- AGW_SHAPE_TYPE_ISOBAR"));
            vGetIsoBar(hAgwShape, roShapeObj);
            break;
		default:
            // no specific shape has been added, so remove generic shape.
            bRes=FALSE;
            break;
	}

	ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bAddShape() -- Exit"));
    return bRes;
}


void fc_sxm_tclShapeProductSurfaceData::vGetPressureCenter(AGW_SHAPE_OBJECT hAgwShape, midw_ext_fi_tcl_SXMShape &roShapeObj) const {
    
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::vGetPressureCenter -- Enter"));

    midw_ext_fi_tcl_SXMShapeTypePressureCenter *poPressureCenter = OSAL_NEW midw_ext_fi_tcl_SXMShapeTypePressureCenter;
	if (OSAL_NULL == poPressureCenter)
	{
		return;
	}
	roShapeObj.ShapeTypeType.enType =  midw_ext_fi_tcl_e8_SXMShapeType::FI_EN_T_SXMSHAPETYPEPRESSURECENTER;
	roShapeObj.poShapeTypeData = poPressureCenter;
	
    if((AGW_SHAPE.eShapeType (hAgwShape))== AGW_SHAPE_TYPE_PRESSURE_CENTER) {
		ETG_TRACE_USR4(("#AGW Shape Type is %s", "AGW_SHAPE_TYPE_PRESSURE_CENTER"));
    }

    poPressureCenter->IssueTime.Time = (tS32)(AGW_SHAPE.tTimeStamp(hAgwShape));

	ETG_TRACE_USR4(( "#AGW Pressure Center issue time is %d", poPressureCenter->IssueTime.Time));

	//AGW_ISOBAR.ePressure



	OSAL_FIXED_OBJECT_DATA pressureData[OSAL_FIXED_OBJECT_SIZE];
	OSAL_FIXED_OBJECT hPressure = OSAL_FIXED_INVALID_OBJECT;
	AGW_PRESSURE_CENTER_OBJECT hPressureCenter = AGW_SHAPE.hPressureCenter(hAgwShape);
	hPressure = OSAL_FIXED.hCreateInMemory(0, 0, pressureData);
	AGW_PRESSURE_CENTER.ePressure(hPressureCenter, hPressure);
	poPressureCenter->Pressure.s32Value = (tS32)fc_sxm_fFixed2Float(hPressure);
	// Send Pressure value in Pascal (1 millibar = 100 * pascal)
	poPressureCenter->Pressure.s32Value = (poPressureCenter->Pressure.s32Value) * AGW_PRESSURE_VALUE_IN_PASCAL;
	ETG_TRACE_USR4(("#AGW_PRESSURECENTER.ePressure is %d", poPressureCenter->Pressure.s32Value));

	switch(AGW_PRESSURE_CENTER.eType(AGW_SHAPE.hPressureCenter(hAgwShape)))
	{
		case AGW_PRESSURE_TYPE_HIGH:
			ETG_TRACE_USR4(("#AGW_PRESSURE_CENTER.eType is AGW_PRESSURE_TYPE_HIGH"));
			poPressureCenter->PressureType.enType = midw_ext_fi_tcl_e8_PressureType::FI_EN_HIGHPRESSURE;
			break;
		case AGW_PRESSURE_TYPE_LOW:
			ETG_TRACE_USR4(("#AGW_PRESSURE_CENTER.eType is AGW_PRESSURE_TYPE_LOW"));
			poPressureCenter->PressureType.enType = midw_ext_fi_tcl_e8_PressureType::FI_EN_LOWPRESSURE;
			break;
		case AGW_PRESSURE_TYPE_UNKNOWN:
		default:
			ETG_TRACE_USR4(("#AGW_PRESSURE_CENTER.eType is AGW_PRESSURE_TYPE_UNKNOWN"));
			poPressureCenter->PressureType.enType = midw_ext_fi_tcl_e8_PressureType::FI_EN_UNKNOWN;
			break;
	}

    //AGW_SHAPE.un16Points
	ETG_TRACE_USR4(( "#AGW_SHAPE.un16Points is %d", AGW_SHAPE.un16Points(hAgwShape)));

    //AGW_SHAPE.eIteratePoints
    ETG_TRACE_USR4(( "Iterate for %d (lat,lon) -- START", 1));
    vector<fc_sxm_trPoint> vecPoints;
    tU32 u32NumPoints = u32GetPoints(hAgwShape, vecPoints);
    for (tU32 i=0; i<u32NumPoints; i++) {
        fc_sxm_trPoint rPoint=vecPoints[i];
               
        ETG_TRACE_USR4(("Pressure Center SMS co-ordinates lat=%f lon=%f", 
           rPoint.fLat, rPoint.fLon));

        poPressureCenter->GeoPoint.Latitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLat);
        poPressureCenter->GeoPoint.Longitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLon);
        
        ETG_TRACE_USR4(("Pressure Center co-ordinates lat=%d, lon=%d", 
           poPressureCenter->GeoPoint.Latitude.s32Value , poPressureCenter->GeoPoint.Longitude.s32Value));
    }
}


void fc_sxm_tclShapeProductSurfaceData::vGetFront(AGW_SHAPE_OBJECT hAgwShape, midw_ext_fi_tcl_SXMShape &roShapeObj) const {
    
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::vGetFront -- Enter"));

	if ((AGW_SHAPE_INVALID_OBJECT == hAgwShape))
	{


		ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::vGetFront return"));
		return;
	}
    midw_ext_fi_tcl_SXMShapeTypeFront *poFront = OSAL_NEW midw_ext_fi_tcl_SXMShapeTypeFront;
	if (OSAL_NULL == poFront)
	{
		return;
	}
	roShapeObj.ShapeTypeType.enType =  midw_ext_fi_tcl_e8_SXMShapeType::FI_EN_T_SXMSHAPETYPEFRONT;
	roShapeObj.poShapeTypeData = poFront;
	
    if((AGW_SHAPE.eShapeType (hAgwShape))== AGW_SHAPE_TYPE_FRONT) {
		ETG_TRACE_USR4(("#AGW Shape Type is %s", "AGW_SHAPE_TYPE_FRONT"));
    }

    poFront->IssueTime.Time = (tS32)(AGW_SHAPE.tTimeStamp(hAgwShape));

	ETG_TRACE_USR4(( "#AGW Front issue time is %d", poFront->IssueTime.Time));

	AGW_FRONT_TYPE_ENUM eFrontType = AGW_FRONT_TYPE_UNKNOWN;
	eFrontType = AGW_FRONT.eType(AGW_SHAPE.hFront(hAgwShape));
	
	switch(eFrontType)
	{
		case AGW_FRONT_TYPE_COLD:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_COLD"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_COLDFRONT;
			break;
		case AGW_FRONT_TYPE_WARM:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_WARM"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_WARMFRONT;
			break;
		case AGW_FRONT_TYPE_STATIONARY:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_STATIONARY"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_STATIONARYFRONT;
			break;
		case AGW_FRONT_TYPE_OCCLUDED:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_OCCLUDED"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_OCCLUDEDFRONT;
			break;
		case AGW_FRONT_TYPE_TROUGH:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_TROUGH"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_TROUGH;
			break;
		case AGW_FRONT_TYPE_SQUALL_LINE:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_SQUALL_LINE"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_SQUALLLINE;
			break;
		case AGW_FRONT_TYPE_DRY_LINE:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_DRY_LINE"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_DRYLINE;
			break;
		case AGW_FRONT_TYPE_DISSIPATING_COLD:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_DISSIPATING_COLD"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_DISSIPATINGCOLDFRONT;
			break;
		case AGW_FRONT_TYPE_DISSIPATING_WARM:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_DISSIPATING_WARM"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_DISSIPATINGWARMFRONT;
			break;
		case AGW_FRONT_TYPE_DISSIPATING_STATIONARY:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_DISSIPATING_STATIONARY"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_DISSIPATINGSTATIONARYFRONT;
			break;
		case AGW_FRONT_TYPE_DISSIPATING_OCCLUDED:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_DISSIPATING_OCCLUDED"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_DISSIPATINGOCCLUDEDFRONT;
			break;
		case AGW_FRONT_TYPE_FORMING_COLD:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_FORMING_COLD"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_FORMINGCOLDFRONT;
			break;
		case AGW_FRONT_TYPE_FORMING_WARM:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_FORMING_WARM"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_FORMINGWARMFRONT;
			break;
		case AGW_FRONT_TYPE_FORMING_STATIONARY:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_FORMING_STATIONARY"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_FORMINGSTATIONARYFRONT;
			break;
		case AGW_FRONT_TYPE_FORMING_OCCLUDED:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_FORMING_OCCLUDED"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_FORMINGOCCLUDEDFRONT;
			break;
		case AGW_FRONT_TYPE_UNKNOWN:
		default:
			ETG_TRACE_USR4(("#AGW_FRONT.eFrontType is AGW_FRONT_TYPE_UNKNOWN"));
			poFront->FrontType.enType = midw_ext_fi_tcl_e8_FrontType::FI_EN_UNKNOWN;
			break;
	}

	//AGW_SHAPE.un16Points
	ETG_TRACE_USR4(( "#AGW_SHAPE.un16Points is %d", AGW_SHAPE.un16Points(hAgwShape)));

    //AGW_SHAPE.eIteratePoints
    ETG_TRACE_USR4(( "Iterate for %d (lat,lon) -- START", 1));
    vector<fc_sxm_trPoint> vecPoints;
    tU32 u32NumPoints = u32GetPoints(hAgwShape, vecPoints);
    for (tU32 i=0; i<u32NumPoints; i++) {
        fc_sxm_trPoint rPoint=vecPoints[i];
        midw_ext_fi_tcl_PositionWGS84 var_geoPosisition;
        
        ETG_TRACE_USR4(("Front SMS co-ordinates lat=%f, lon=%f", 
           rPoint.fLat, rPoint.fLon));

        var_geoPosisition.Latitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLat);
        var_geoPosisition.Longitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLon);
        
        ETG_TRACE_USR4(("Front co-ordinates lat=%d, lon=%d", 
           var_geoPosisition.Latitude.s32Value, var_geoPosisition.Longitude.s32Value));
        
        poFront->GeoPoints.push_back(var_geoPosisition);
    }

}


void fc_sxm_tclShapeProductSurfaceData::vGetIsoBar(AGW_SHAPE_OBJECT hAgwShape, midw_ext_fi_tcl_SXMShape &roShapeObj) const {
    
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductSurfaceData::vGetIsoBar -- Enter"));

    midw_ext_fi_tcl_SXMShapeTypeIsobar *poIsoBar = OSAL_NEW midw_ext_fi_tcl_SXMShapeTypeIsobar;
	if (OSAL_NULL == poIsoBar)
	{
		return;
	}
	roShapeObj.ShapeTypeType.enType =  midw_ext_fi_tcl_e8_SXMShapeType::FI_EN_T_SXMSHAPETYPEISOBAR;
	roShapeObj.poShapeTypeData = poIsoBar;
	
    if((AGW_SHAPE.eShapeType (hAgwShape))== AGW_SHAPE_TYPE_ISOBAR) {
		ETG_TRACE_USR4(("#AGW Shape Type is %s", "AGW_SHAPE_TYPE_ISOBAR"));
    }

    poIsoBar->IssueTime.Time = (tS32)(AGW_SHAPE.tTimeStamp(hAgwShape));

	ETG_TRACE_USR4(( "#AGW Isobar issue time is %d", poIsoBar->IssueTime.Time));

	//AGW_ISOBAR.ePressure

	OSAL_FIXED_OBJECT_DATA pressureData[OSAL_FIXED_OBJECT_SIZE];
	OSAL_FIXED_OBJECT hPressure = OSAL_FIXED_INVALID_OBJECT;
	AGW_ISOBAR_OBJECT hIsoBar = AGW_SHAPE.hIsobar(hAgwShape);
	hPressure = OSAL_FIXED.hCreateInMemory(0, 0, pressureData);
	AGW_ISOBAR.ePressure(hIsoBar, hPressure);
	poIsoBar->Pressure.s32Value = (tS32) fc_sxm_fFixed2Float(hPressure);
	// Send Pressure value in Pascal (1 millibar = 100 * pascal)
	poIsoBar->Pressure.s32Value = (poIsoBar->Pressure.s32Value) * AGW_PRESSURE_VALUE_IN_PASCAL;

	ETG_TRACE_USR4(("#AGW_ISOBAR.ePressure is %d", poIsoBar->Pressure.s32Value));

    //AGW_SHAPE.un16Points
	ETG_TRACE_USR4(( "#AGW_SHAPE.un16Points is %d", AGW_SHAPE.un16Points(hAgwShape)));

    //AGW_SHAPE.eIteratePoints
	ETG_TRACE_USR4(( "Iterate for %d (lat,lon) -- START", 1));
    vector<fc_sxm_trPoint> vecPoints;
    tU32 u32NumPoints = u32GetPoints(hAgwShape, vecPoints);
    for (tU32 i=0; i<u32NumPoints; i++) {
        fc_sxm_trPoint rPoint=vecPoints[i];
        midw_ext_fi_tcl_PositionWGS84 var_geoPosisition;
        
        ETG_TRACE_USR4(("Isobar SMS co-ordinates lat=%f, lon=%f", 
           rPoint.fLat, rPoint.fLon));

        var_geoPosisition.Latitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLat);
        var_geoPosisition.Longitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLon);

        ETG_TRACE_USR4(("Isobar co-ordinates lat=%d, lon=%d", 
           var_geoPosisition.Latitude.s32Value, var_geoPosisition.Longitude.s32Value));
        
        poIsoBar->GeoPoints.push_back(var_geoPosisition);
    }
}
