/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_agw_storm_track_data.cpp
* @brief       Implementation of the AGW track data.
*              The Storm Track product describes tropical storms in terms of their
*              historical and forecast positions including their wind radii fields.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_tcl_agw_app.h"
#include "fc_sxm_tcl_agw_storm_track_data.h"
#include "fc_sxm_service_sxm_agw.h"
#include "fc_sxm_tcl_sxmapp_manager.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AGW_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_agw_storm_track_data.cpp.trc.h"
#endif

set<string> _setStormIds;

tVoid fc_sxm_tclShapeProductStormTrackData::vNotify() {
    fc_sxm_tcl_trAgwPropertyStormTrackDataStatus oStormTrackDataStatus;
    oStormTrackDataStatus.oFiMsg.UpdateCounter=u32UpdateCounter;
    ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::vNotify() UpdateCounter = %d", u32UpdateCounter));
    fc_sxm_tclAgwProperties::instance()->oStormTrackDataStatus.vSetAndNotify(oStormTrackDataStatus);
};

tVoid fc_sxm_tclShapeProductStormTrackData::vEmit(fc_sxm_trAdressing rAdressing) {
	 oMResStormTrackData.UpdateCounter=u32UpdateCounter;
    ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::vEmit() u32UpdateCounter = %d", u32UpdateCounter));
    fc_sxm_tclAgwService::instance()->enSendFiMessage(rAdressing, oMResStormTrackData);
}; 


tBool fc_sxm_tclShapeProductStormTrackData::bAddShape(AGW_SHAPE_OBJECT hAgwShape, midw_ext_fi_tcl_SXMShape &roShapeObj)
{
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::bAddShape() -- Enter"));
    tBool bRes=TRUE;
	AGW_STORM_POSITION_OBJECT hStormPos;
	STRING_OBJECT hId;
	string oStormId;
	
	//data-structure to send via fi-message --end

	//populating the data-structure and posting it in fi-message

	AGW_SHAPE_TYPE_ENUM eShapeType = AGW_SHAPE.eShapeType(hAgwShape);

	switch(eShapeType)
    {

		case AGW_SHAPE_TYPE_STORM_POSITION:
			{
				ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::bAddShape() -- AGW_SHAPE_TYPE_STORM_POSITION"));
				hStormPos = AGW_SHAPE.hStormPosition(hAgwShape);
				hId = AGW_STORM_POSITION.hStormID(hStormPos);

				fc_sxm_vCopySmsString2Stl(hId, oStormId);

				SXM_IF_FIND_SET(string, stormIter, _setStormIds, oStormId) {
					// with FALSE you indicate, that you didn't use the entry.
					return FALSE;
				}
				// store that we have collected data for this storm:
				_setStormIds.insert(oStormId);
				return vGetStormTrack(hAgwShape, roShapeObj);
			}
			break;
		default:
            // no specific shape has been added, so remove generic shape.
            bRes=FALSE;
            break;
	}
	ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bAddShape() -- Exit"));
    return bRes;
}

tVoid fc_sxm_tclShapeProductStormTrackData::vClearOutput() {
    _setStormIds.clear();
	ETG_TRACE_USR4(("*************************CLEAR***************************************"));
    // call base-class
    fc_sxm_tclShapeProductBase::vClearOutput();
}

tBool fc_sxm_tclShapeProductStormTrackData::vGetStormTrack(AGW_SHAPE_OBJECT hStormPosShape, midw_ext_fi_tcl_SXMShape &roShapeObj) const {
    
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::vGetStormTrack -- Enter"));
	SMSAPI_RETURN_CODE_ENUM eReturnCode;
	AGW_STORM_POSITION_OBJECT hStormPos =  AGW_SHAPE.hStormPosition(hStormPosShape);

	midw_ext_fi_tcl_SXMShapeTypeStormTrack *poStormTrack=OSAL_NEW midw_ext_fi_tcl_SXMShapeTypeStormTrack;
	if (OSAL_NULL == poStormTrack)
	{
		return FALSE;
	}
	
	roShapeObj.ShapeTypeType.enType =  midw_ext_fi_tcl_e8_SXMShapeType::FI_EN_T_SXMSHAPETYPESTORMTRACK;
	roShapeObj.poShapeTypeData = poStormTrack;

	//AGW_STORM_TRACK.hStormID
	STRING_OBJECT hStormId = AGW_STORM_POSITION.hStormID(hStormPos);
    fc_sxm_vCopySmsString2Fi(hStormId, poStormTrack->StormUniqueId);    
    ETG_TRACE_USR4(("#AGW_STORM_TRACK.hStormID() is %s", poStormTrack->StormUniqueId.szValue));
	
	//AGW_STORM_TRACK.hStormName
	STRING_OBJECT hStormName = AGW_STORM_POSITION.hStormName(hStormPos);    
    fc_sxm_vCopySmsString2Fi(hStormName, poStormTrack->StormName);
    ETG_TRACE_USR4(("#AGW_STORM_TRACK.hStormName() is %s", poStormTrack->StormName.szValue));

	eReturnCode = AGW_STORM_POSITION.eIterateStormPosition(
                                hStormPos,
                                fc_sxm_tclShapeProductStormTrackData::bFtchAgwStmTrkPosObjIterator,
                                (void*)(poStormTrack)
                                    );
	return TRUE;
}

static BOOLEAN bIteratePointsCallback(
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    void* pvArg
        )
{
    // peha: from where do you know that the last point of the shape is the one you searched for?
    if (OSAL_NULL != pvArg) {
        fc_sxm_trPoint rPoint;
        rPoint.fLat = fc_sxm_fFixed2Float(hLat);
        rPoint.fLon = fc_sxm_fFixed2Float(hLon);

        ((vector<fc_sxm_trPoint> *)pvArg)->push_back(rPoint);
    }
	return TRUE;		
}


BOOLEAN fc_sxm_tclShapeProductStormTrackData::bFtchAgwStmTrkPosObjIterator(AGW_SHAPE_OBJECT hAgwStormPos,void* pvArg)
{
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::bFtchAgwStmTrkPosObjIterator -- Enter"));

	AGW_STORM_POSITION_OBJECT hPosition = AGW_SHAPE.hStormPosition(hAgwStormPos);
	
	if (hPosition == AGW_STORM_POSITION_INVALID_OBJECT)
	{
		return FALSE;
	}

	midw_ext_fi_tcl_SXMShapeTypeStormTrack *pStormTrack = (midw_ext_fi_tcl_SXMShapeTypeStormTrack *)(pvArg);

	midw_ext_fi_tcl_SXMShapeTypeStormPosition stormPosition;
	
    
    stormPosition.IssueTime.Time = (tS32)(AGW_SHAPE.tTimeStamp(hAgwStormPos));
    ETG_TRACE_USR4(( "Storm position Radii issue time is %d", stormPosition.IssueTime.Time));
    
    //AGW_STORM_POSITION.ePositionType
    switch(AGW_STORM_POSITION.ePositionType(hPosition))
    {
		case AGW_STORM_POSITION_TYPE_HISTORIC:
            stormPosition.PositionType.enType = midw_ext_fi_tcl_e8_StormPositionType::FI_EN_HISTORIC ;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.ePositionType() is AGW_STORM_POSITION_TYPE_HISTORIC"));
            break;

		case AGW_STORM_POSITION_TYPE_CURRENT:
            stormPosition.PositionType.enType = midw_ext_fi_tcl_e8_StormPositionType::FI_EN_CURRENT ;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.ePositionType() is AGW_STORM_POSITION_TYPE_CURRENT"));
            break;
			
		case AGW_STORM_POSITION_TYPE_FORECAST:
            stormPosition.PositionType.enType = midw_ext_fi_tcl_e8_StormPositionType::FI_EN_FORECAST ;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.ePositionType() is AGW_STORM_POSITION_TYPE_FORECAST"));
            break;
		default: 
            stormPosition.PositionType.enType = midw_ext_fi_tcl_e8_StormPositionType::FI_EN_UNKNOWN ;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.ePositionType() is AGW_STORM_POSITION_TYPE_UNKNOWN"));
            break;
    }

	//AGW_STORM_POSITION.eDirection	
    OSAL_FIXED_OBJECT hDirection = OSAL_FIXED_INVALID_OBJECT;
	OSAL_FIXED_OBJECT_DATA atDirectionData[OSAL_FIXED_OBJECT_SIZE];
    hDirection = OSAL_FIXED.hCreateInMemory(0, 0, atDirectionData);
	if (SMSAPI_RETURN_CODE_NOT_FOUND == AGW_STORM_POSITION.eDirection(hPosition, hDirection)) {
      stormPosition.Direction.s16Value = -1;
	} 
	else 
	{
	   stormPosition.Direction.s16Value = (tS16) fc_sxm_fFixed2Float(hDirection);
	}
	ETG_TRACE_USR4(("#AGW_STORM_POSITION.eDirection() is  %d", stormPosition.Direction.s16Value));
    
	//AGW_STORM_POSITION.eSpeed 
	OSAL_FIXED_OBJECT_DATA atSpeedData[OSAL_FIXED_OBJECT_SIZE];
	OSAL_FIXED_OBJECT hSpeed = OSAL_FIXED.hCreateInMemory(0, 0, atSpeedData);
	if (SMSAPI_RETURN_CODE_NOT_FOUND == AGW_STORM_POSITION.eSpeed(hPosition, hSpeed)) 
	{
		stormPosition.Speed.s16Value = -1;
	} 
	else 
	{
	   stormPosition.Speed.s16Value = (tS16)fc_sxm_fFixed2Float(hSpeed);
	}

	ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eSpeed() is  %d", stormPosition.Speed.s16Value));

	//AGW_STORM_POSITION.eMaxSustainedWinds
    OSAL_FIXED_OBJECT hMaxSustainedWinds=OSAL_FIXED_INVALID_OBJECT;
	OSAL_FIXED_OBJECT_DATA athSusWindsData[OSAL_FIXED_OBJECT_SIZE];
    hMaxSustainedWinds = OSAL_FIXED.hCreateInMemory(0, 0, athSusWindsData);
	AGW_STORM_POSITION.eMaxSustainedWinds(hPosition, hMaxSustainedWinds);	
    stormPosition.MaxSustainedWinds.s16Value = (tS16)fc_sxm_fFixed2Float(hMaxSustainedWinds);
	ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eMaxSustainedWinds() is  %d", stormPosition.MaxSustainedWinds.s16Value));

	//AGW_STORM_POSITION.eGusts
    OSAL_FIXED_OBJECT hGusts=OSAL_FIXED_INVALID_OBJECT;
	OSAL_FIXED_OBJECT_DATA athGustsData[OSAL_FIXED_OBJECT_SIZE];
    hGusts = OSAL_FIXED.hCreateInMemory(0, 0, athGustsData);
	AGW_STORM_POSITION.eGusts(hPosition, hGusts);
    stormPosition.Gusts.s16Value = (tS16)fc_sxm_fFixed2Float(hGusts);
	ETG_TRACE_USR4(("#AGW_STORM_POSITION.eGusts() is  %d", stormPosition.Gusts.s16Value));

	//AGW_STORM_POSITION.ePressure
    OSAL_FIXED_OBJECT hPressure=OSAL_FIXED_INVALID_OBJECT;
	OSAL_FIXED_OBJECT_DATA athPressureData[OSAL_FIXED_OBJECT_SIZE];
    hPressure = OSAL_FIXED.hCreateInMemory(0, 0, athPressureData);
	AGW_STORM_POSITION.ePressure(hPosition, hPressure);
	stormPosition.Pressure.s32Value = (tS32)fc_sxm_fFixed2Float(hPressure);
	// Send Pressure value in Pascal (1 millibar = 100 * pascal)
	stormPosition.Pressure.s32Value = (stormPosition.Pressure.s32Value) * AGW_PRESSURE_VALUE_IN_PASCAL;
	ETG_TRACE_USR4(("#AGW_STORM_POSITION.ePressure() is  %d", stormPosition.Pressure.s32Value));

	
	//stormPosition.Timestamp.Time = (tS32) AGW_STORM_POSITION.un32Time(hPosition);
	//conversion from TIME_T to tS32
	stormPosition.Timestamp.Time = (tS32) AGW_STORM_POSITION.tObservationTime(hPosition);
	ETG_TRACE_USR4(("#AGW_STORM_POSITION.un32Time() is  %u", stormPosition.Timestamp.Time));

	//AGW_STORM_POSITION.eStormType
	switch(AGW_STORM_POSITION.eStormType(hPosition))
	{
        case AGW_STORM_TYPE_TROPICAL_DISTURBANCE:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_TROPICALDISTURBANCE;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_TROPICAL_DISTURBANCE"));		
            break;
            
        case AGW_STORM_TYPE_HURRICANE_CAT_1:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_HURRICANECAT1;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_HURRICANE_CAT_1"));		
            break;
        case AGW_STORM_TYPE_HURRICANE_CAT_2:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_HURRICANECAT2;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_HURRICANE_CAT_2"));		
            break;
        case AGW_STORM_TYPE_HURRICANE_CAT_3:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_HURRICANECAT3;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_HURRICANE_CAT_3"));		
            break;
        case AGW_STORM_TYPE_HURRICANE_CAT_4:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_HURRICANECAT4;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_HURRICANE_CAT_4"));		
            break;
        case AGW_STORM_TYPE_HURRICANE_CAT_5:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_HURRICANECAT5;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_HURRICANE_CAT_5"));		
            break;
        case AGW_STORM_TYPE_TROPICAL_STORM:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_TROPICALSTORM;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_TROPICAL_STORM"));		
            break;
        case AGW_STORM_TYPE_TROPICAL_DEPRESSION:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_TROPICALDEPRESSION;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_TROPICAL_DEPRESSION"));		
            break;
        case AGW_STORM_TYPE_TYPHOON:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_TYPHOON;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_TYPHOON"));		
            break;
        case AGW_STORM_TYPE_SUPER_TYPHOON:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_SUPERTYPHOON;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_SUPER_TYPHOON"));		
            break;
        case AGW_STORM_TYPE_TROPICAL_CYCLONE:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_TROPICALCYCLONE;
            ETG_TRACE_USR4(("#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_TROPICAL_CYCLONE"));		
            break;
        default:
            stormPosition.StormType.enType = midw_ext_fi_tcl_e8_StormType::FI_EN_UNKNOWN;
            ETG_TRACE_USR4(( "#AGW_STORM_POSITION.eStormType() is AGW_STORM_TYPE_UNKNOWN"));		
            break;
	}

	//AGW_STORM_POSITION.un16SequenceID
	stormPosition.SequenceId =  AGW_STORM_POSITION.un16SequenceID(hPosition);
	ETG_TRACE_USR4(( "#AGW_STORM_POSITION.un16SequenceID() is  %u",stormPosition.SequenceId));

	//AGW_SHAPE.un16Points
	ETG_TRACE_USR4(("#AGW_SHAPE.un16Points is %d", AGW_SHAPE.un16Points(hAgwStormPos)));
    
    //AGW_SHAPE.eIteratePoints
	ETG_TRACE_USR4(( "Iterate for %d (lat,lon) -- START", 1));
    
    vector<fc_sxm_trPoint> vecPoints;
    //tU32 u32NumPoints = u32GetPoints(hAgwStormPos, vecPoints);
	static vector<fc_sxm_trPoint> *pVecPoints=&vecPoints;
	SMSAPI_RETURN_CODE_ENUM eReturnCode=AGW_SHAPE.eIteratePoints (hAgwStormPos, bIteratePointsCallback,pVecPoints);
	(tVoid)eReturnCode;
	tU32 u32NumPoints = (tU32)(vecPoints.size());

    if (u32NumPoints) {
        fc_sxm_trPoint rPoint=vecPoints[u32NumPoints-1];

        ETG_TRACE_USR4(("Storm position SMS co-ordinates lat=%f, lon=%f",
           rPoint.fLat, rPoint.fLon));

        stormPosition.GeoPoint.Latitude.s32Value =  fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLat);
        stormPosition.GeoPoint.Longitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLon);    

        ETG_TRACE_USR4(("Storm position WGS co-ordinates lat=%d, lon=%d", 
        stormPosition.GeoPoint.Latitude.s32Value, stormPosition.GeoPoint.Longitude.s32Value));
	}

	eReturnCode = AGW_STORM_POSITION.eIterateWindRadii(
                                    hPosition,
									fc_sxm_tclShapeProductStormTrackData::bFtchAgwWindRadiiObjIterator,
                                    (void*)(&stormPosition));

	pStormTrack->StormPositions.push_back(stormPosition);
 	return TRUE;	

}

BOOLEAN fc_sxm_tclShapeProductStormTrackData::bFtchAgwWindRadiiObjIterator(AGW_SHAPE_OBJECT hAgwStormPos,void* pvArg)
{
	ETG_TRACE_USR4(("fc_sxm_tclShapeProductStormTrackData::bFtchAgwWindRadiiObjIterator -- Enter"));

	AGW_WIND_RADII_AREA_OBJECT hWindRadii = AGW_SHAPE.hWindRadiiArea(hAgwStormPos);
	
	if (hWindRadii == AGW_WIND_RADII_AREA_INVALID_OBJECT)
    {
       return FALSE;
    }

	midw_ext_fi_tcl_SXMShapeTypeStormPosition *stormPosObj = (midw_ext_fi_tcl_SXMShapeTypeStormPosition *)pvArg;

	midw_ext_fi_tcl_SXMShapeTypeWindRadius windRadii;
	
	//AGW_WIND_RADII_AREA.eWindSpeed
    OSAL_FIXED_OBJECT hWindSpeed = OSAL_FIXED_INVALID_OBJECT;
	OSAL_FIXED_OBJECT_DATA atWindSpeedData[OSAL_FIXED_OBJECT_SIZE];
    hWindSpeed = OSAL_FIXED.hCreateInMemory(0, 0, atWindSpeedData);
	AGW_WIND_RADII_AREA.eWindSpeed(hWindRadii, hWindSpeed);
    windRadii.WindSpeed.s16Value = (tS16)fc_sxm_fFixed2Float(hWindSpeed);
	ETG_TRACE_USR4(("#AGW_WIND_RADII_AREA.eWindSpeed() is  %d", windRadii.WindSpeed.s16Value));
	
	//AGW_SHAPE.un16Points
	ETG_TRACE_USR4(( "#AGW_SHAPE.un16Points is %d", AGW_SHAPE.un16Points(hAgwStormPos)));

    //AGW_SHAPE.eIteratePoints
	ETG_TRACE_USR4(( "Iterate for %d (lat,lon) -- START", AGW_SHAPE.un16Points(hAgwStormPos)));
    vector<fc_sxm_trPoint> vecPoints;
    //tU32 u32NumPoints = u32GetPoints(hAgwStormPos, vecPoints);
	
	static vector<fc_sxm_trPoint> *pVecPoints=&vecPoints;

	SMSAPI_RETURN_CODE_ENUM eReturnCode=AGW_SHAPE.eIteratePoints (hAgwStormPos, bIteratePointsCallback,pVecPoints);
	(tVoid)eReturnCode;
	tU32 u32NumPoints = (tU32)(vecPoints.size());

    for (tU32 i=0; i<u32NumPoints; i++) {
        fc_sxm_trPoint rPoint=vecPoints[i];
        midw_ext_fi_tcl_PositionWGS84 var_geoPosisition;
        
        ETG_TRACE_USR4(("WindRadii SMS co-ordinates lat=%f, lon=%f",
           rPoint.fLat, rPoint.fLon));

        var_geoPosisition.Latitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLat);
        var_geoPosisition.Longitude.s32Value = fc_sxm_tclHelpers::fc_sxm_s32SmsDegreeFloatToNav(rPoint.fLon);
        
        ETG_TRACE_USR4(("WindRadii WGS co-ordinates lat=%d, lon=%d", 
           var_geoPosisition.Latitude.s32Value, var_geoPosisition.Longitude.s32Value));
        
        windRadii.GeoPoints.push_back(var_geoPosisition);
    }
	stormPosObj->WindRadii.push_back(windRadii);
	return TRUE;	
}
  
