/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_agw_shared_mem.cpp
* @brief       Implementation of the AGW shared memory for sharing tile-info
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_agw_shared_mem.h"
#include "fc_sxm_sms_util.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AGW_SHARED_MEMORY
#include "trcGenProj/Header/fc_sxm_tcl_agw_shared_mem.cpp.trc.h"
#endif


/* 
 * CONSTRUCTOR
 */
fc_sxm_tclSharedMem::fc_sxm_tclSharedMem(string oName, tU32 u32Size):
    _oName(oName),
    _u32Size(u32Size),
    _hSharedMem(OSAL_ERROR),
    _pu8SharedMem(OSAL_NULL),
    _u32CurOffset(0)
{
    ETG_TRACE_USR4(("fc_sxm_tclSharedMem constructor"));
    vInit();
}


/* Class fc_sxm_tclSharedMem destructor */
fc_sxm_tclSharedMem::~fc_sxm_tclSharedMem()
{
    ETG_TRACE_USR4(("fc_sxm_tclSharedMem destructor"));
    vDeInit();
}


tU32 fc_sxm_tclSharedMem::u32Write(AGW_TILE_OBJECT hAgwTile, tU32 *pu32StartOffset) {
    // no wrap-around by now
    if (OSAL_NULL != pu32StartOffset) {
        *pu32StartOffset=_u32CurOffset;
    }
    if (OSAL_NULL == _pu8SharedMem) {
        ETG_TRACE_USR4(("fc_sxm_tclSharedMem u32Write: _pu8SharedMem=NULL"));
        return 0;
    }

    STRING_OBJECT hFilePath;
    size_t tImageFileSize;
    IMAGE_OBJECT hImage;
    BOOLEAN bOk = FALSE;
    string sFilePath = "";

    tU32 u32TileSize;

	// Check input
    if (hAgwTile == AGW_TILE_INVALID_OBJECT)
    {
        ETG_TRACE_USR4(("Invalid tile object"));
		return 0;
    }

    // Get IMAGE object
    hImage = AGW_TILE.hImage(hAgwTile);
    if (hImage == IMAGE_INVALID_OBJECT)
    {
        ETG_TRACE_USR4(("Invalid image object"));
		return 0;
    }

    // Get path
    hFilePath = IMAGE.hFileName(hImage);
    if (hFilePath == STRING_INVALID_OBJECT)
    {
        ETG_TRACE_USR4(("Invalid file path"));
		return 0;
    }
	
    fc_sxm_vCopySmsString2Stl(hFilePath, sFilePath);

	// Open file
    OSAL_FILE_STRUCT *psImageFile = OSAL.psFopen(sFilePath.c_str(), "rb");
    if (psImageFile == NULL)
    {
    	ETG_TRACE_USR4(("Failed to open the File"));
    	return 0;
    }

    // Check out file size in bytes
    bOk = OSAL.bFileSystemGetFileSize(psImageFile, &tImageFileSize);
    if (bOk == FALSE)
    {
 	    ETG_TRACE_USR4(("Invalid file size"));
 	    OSAL.iFclose(psImageFile);
        return 0;
    }
	
	u32TileSize = (tU32)tImageFileSize;
    ETG_TRACE_USR4(("fc_sxm_tclSharedMem u32Write: u32TileSize=%d", u32TileSize));
    if (u32TileSize > (_u32Size -_u32CurOffset)) {
        ETG_TRACE_USR4(("fc_sxm_tclSharedMem u32Write: u32TileSize does not fit"));
        OSAL.iFclose(psImageFile);
        return 0;
    }

    size_t tBytesRead =OSAL.tFread(_pu8SharedMem +_u32CurOffset, sizeof(tU8), u32TileSize, psImageFile);
    if (tBytesRead != u32TileSize )
    {
    	ETG_TRACE_USR4(("failed to read data correctly"));
    	OSAL.iFclose(psImageFile);
    	return 0;
    }

    OSAL.iFclose(psImageFile);
    _u32CurOffset+=u32TileSize;
    return u32TileSize;
        
}

tVoid fc_sxm_tclSharedMem::vInit() {
    ETG_TRACE_USR4(("fc_sxm_tclSharedMem vInit(%s)", _oName.c_str()));
    
	//Create the Shared Memory
    _hSharedMem = OSAL_SharedMemoryCreate(_oName.c_str(), 
                                          OSAL_EN_READWRITE,
										  _u32Size);
    //Map the shared memory
	if(OSAL_ERROR != _hSharedMem )
	{
	  //Map the shared memory
	  _pu8SharedMem = (tPU8)OSAL_pvSharedMemoryMap(_hSharedMem,OSAL_EN_READWRITE,
												 _u32Size,0);
	  SXM_ASSERT_RETURN(OSAL_NULL != _pu8SharedMem);
	  OSAL_pvMemorySet(_pu8SharedMem,0,_u32Size);
	}
	else
	{
	  ETG_TRACE_ERR(("fc_sxm_tclSharedMem::vInit()-Error in Creating Shared Memory"));
	}

	vReset();
}

tVoid fc_sxm_tclSharedMem::vDeInit() {
    ETG_TRACE_USR4(("fc_sxm_tclSharedMem vDeInit(%s)", _oName.c_str()));

    tS32 s32RetVal;
    vReset();

    //Release the pointer to Shared Memory
    if(OSAL_NULL != _hSharedMem)
    {
        s32RetVal = OSAL_s32SharedMemoryUnmap(_pu8SharedMem, _u32Size);

        if(OSAL_ERROR != s32RetVal)
        {
            ETG_TRACE_ERR(("Error in unmapping pointer from Shared Memory"));
        }
        _pu8SharedMem = OSAL_NULL;

    }

    //Close and Delete the Shared Memory
    if(OSAL_ERROR != _hSharedMem)
    {
        s32RetVal = OSAL_s32SharedMemoryClose(_hSharedMem);
        if(OSAL_ERROR != s32RetVal)
        {
            ETG_TRACE_ERR(("Error in Closing Shared Memory"));
        }
        s32RetVal = OSAL_s32SharedMemoryDelete(_oName.c_str());
        if(OSAL_ERROR != s32RetVal)
        {
            ETG_TRACE_ERR(("Error in Deleting Shared Memory"));
        }
    }  
}

tVoid fc_sxm_tclSharedMem::vReset() {
    ETG_TRACE_USR4(("fc_sxm_tclSharedMem vReset"));
    _u32CurOffset=0;
}
