/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_agw_dsrl.cpp
* @brief       Implementation of the AGW DSRL functionalities and invokes SMS libraries.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_agw_types.h"
#include "fc_sxm_tcl_agw_app.h"
#include "fc_sxm_tcl_agw_dsrl.h"
#include "fc_sxm_tcl_agw_products.h"
#include "fc_sxm_tcl_agw_shapes.h"
#include "fc_sxm_tcl_agw_tiles.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AGW_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_agw_dsrl.cpp.trc.h"
#endif



/*********************************************************
 *
 *   // Shape points iterator
		bCallback
        *
        **********************************************************/
/*
  callback used for second iteration where data are actually read
*/
static tBool cb_bDSRLIterateRead(DSRL_OBJECT hSmsDSRL, 
                                 DSRL_ENTRY_ID tEntryID,
                                 DSRL_ENTRY_STATUS_ENUM eStatus, 
                                 DSRL_ENTRY_OBJECT hEntry, 
                                 tVoid *pvIteratorArg) {
    fc_sxm_tclAgwDSRL *poThis=(fc_sxm_tclAgwDSRL *)pvIteratorArg;
    return poThis->bHandleIterateCallbackRead(hSmsDSRL, tEntryID, eStatus, hEntry);
}





fc_sxm_tclAgwDSRL::fc_sxm_tclAgwDSRL(fc_sxm_tenDSRLType enDsrlType):
    fc_sxm_tclConfigDSRL<fc_sxm_trAgwDSRLCfg>(enDsrlType,
                                              fc_sxm_tclAgwApp::instance(),
                                              TR_CLASS_FC_SXM_AGW_DSRL)
{
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL CTOR"));
}
    



/*********************************************************************
 *
 *FUNCTION:     ~fc_sxm_tclAgwDSRL
 *
 *DESCRIPTION:  Destructor
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclAgwDSRL::~fc_sxm_tclAgwDSRL()
{
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL DTOR"));
}

/*********************************************************************
 *
 *FUNCTION:     vOnDSRLUpdate
 *
 *DESCRIPTION:  Called when DSRL is updated
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclAgwDSRL::vOnDSRLUpdate() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::vOnDSRLUpdate"));
    if (!fc_sxm_tclAgwProducts::instance()->bIsChanged()) {
        return;
    }
    vReadDsrl();
    fc_sxm_tclAgwProducts::instance()->vNotify();
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::vOnDSRLUpdate"));
}


tVoid fc_sxm_tclAgwDSRL::vReadDsrl() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::vReadDsrl START"));
    DSRL.eIterate(hGetSMSDSRL(), cb_bDSRLIterateRead, (tVoid *)this);
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::vReadDsrl END"));
}

/*********************************************************************
 *
 *FUNCTION:     cb_bDSRLFilter
 *
 *DESCRIPTION:  Agw service DSRL filter callback
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclAgwDSRL::bHandleFilterCallback(DSRL_OBJECT hDSRL, 
                                               DSRL_ENTRY_OBJECT hEntry)
{
	(tVoid)hDSRL;
	(tVoid)hEntry;

    return TRUE;
#if 0
	fc_sxm_trAgwDSRLFilterCfg  const &rFilterCfg= rGetCurrentConfig().rFilterCfg;

    //	ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::cb_bDSRLFilter"));

	DSRL_ENTRY_TYPE_ENUM hEntryType = DSRL_ENTRY_TYPE_UNKNOWN;
	AGW_PRODUCT_TYPE_ENUM enProductType = AGW_PRODUCT_TYPE_UNKNOWN;

	hEntryType = DSRL_ENTRY.eType(hEntry);
	if (hEntryType == DSRL_ENTRY_TYPE_AGW_TILE)
	{
		AGW_TILE_OBJECT hAgwTile = DSRL_ENTRY.hAgwTile(hEntry);
		enProductType = AGW_TILE.eProductType(hAgwTile);
	}
	else if (hEntryType == DSRL_ENTRY_TYPE_AGW_SHAPE)
	{
		AGW_SHAPE_OBJECT hAgwShape = DSRL_ENTRY.hAgwShape(hEntry);
		enProductType = AGW_SHAPE.eProductType(hAgwShape);
	}

	if (enProductType == rFilterCfg.enProductType)
	{
        //		ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::cb_bDSRLFilter Return TRUE"));
		return TRUE;
	}
	else
	{
        //		ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::cb_bDSRLFilter Return FALSE"));
		return FALSE;
	} 
#endif
	
}

/*********************************************************************
 *
 *FUNCTION:     cb_tS16DSRLSort
 *
 *DESCRIPTION:  Agw service DSRL sort callback
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tS16 fc_sxm_tclAgwDSRL::s16HandleSortCallback(DSRL_OBJECT hDSRL,
                                              DSRL_ENTRY_OBJECT hEntry1,
                                              DSRL_ENTRY_OBJECT hEntry2)
{
	(tVoid)hDSRL;
	//ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::s16HandleSortCallback"));
    DSRL_ENTRY_TYPE_ENUM eEntryType1 = DSRL_ENTRY.eType(hEntry1);
    DSRL_ENTRY_TYPE_ENUM eEntryType2 = DSRL_ENTRY.eType(hEntry2);

    tS16 tS16Return = 1;

    // peha: why do you use non-default sorting?

    // Separating TILES from SHAPES first
    if (eEntryType1 != eEntryType2)
    {
        tS16Return = (tS16)(eEntryType1 - eEntryType2);
    }
    else
    {
        switch (eEntryType1)
        {
            case DSRL_ENTRY_TYPE_AGW_TILE:
            {
                AGW_TILE_OBJECT hTile1 = DSRL_ENTRY.hAgwTile(hEntry1);
                AGW_TILE_OBJECT hTile2 = DSRL_ENTRY.hAgwTile(hEntry2);
                AGW_PRODUCT_TYPE_ENUM eProdType1 = AGW_TILE.eProductType(hTile1);
                AGW_PRODUCT_TYPE_ENUM eProdType2 = AGW_TILE.eProductType(hTile2);

                if (eProdType1 != eProdType2)
                {
                    // Divide tiles by types into different groups
                    tS16Return = (tS16)(eProdType1 - eProdType2);
                }
                else
                {
                    // For TILES of the same type, sort by NW location
                    // of area rectangle
                    LOCATION_OBJECT hLoc1 = AGW_TILE.hNorthWest(hTile1);
                    LOCATION_OBJECT hLoc2 = AGW_TILE.hNorthWest(hTile2);
                    OSAL_FIXED_OBJECT hValue1 = LOCATION.hLat(hLoc1);
                    OSAL_FIXED_OBJECT hValue2 = LOCATION.hLat(hLoc2);

                    // Compare lat for NW location
                    tS16Return = OSAL_FIXED.n16Compare(hValue1, hValue2);
                    if (tS16Return == 0)
                    {
                        // Compare lon for NW location
                        hValue1 = LOCATION.hLon(hLoc1);
                        hValue2 = LOCATION.hLon(hLoc2);
                        tS16Return = OSAL_FIXED.n16Compare(hValue1, hValue2);
                    }
                }
            }
            break;
            case DSRL_ENTRY_TYPE_AGW_SHAPE:
            {
                AGW_SHAPE_OBJECT hShape1 = DSRL_ENTRY.hAgwShape(hEntry1);
                AGW_SHAPE_OBJECT hShape2 = DSRL_ENTRY.hAgwShape(hEntry2);
                AGW_PRODUCT_TYPE_ENUM eProdType1 = AGW_SHAPE.eProductType(hShape1);
                AGW_PRODUCT_TYPE_ENUM eProdType2 = AGW_SHAPE.eProductType(hShape2);

                if (eProdType1 != eProdType2)
                {
                    // Divide shapes by types into different groups
                    tS16Return = (tS16)(eProdType1 - eProdType2);
                }
                else
                {
                    // SHAPES of the same product are sorted by own type
                    AGW_SHAPE_TYPE_ENUM eShapeType1 = AGW_SHAPE.eShapeType(hShape1);
                    AGW_SHAPE_TYPE_ENUM eShapeType2 = AGW_SHAPE.eShapeType(hShape2);

                    tS16Return = (tS16)(eShapeType1 - eShapeType2);
                }
            }
            break;
            default:
            {
                
            }
            break;
        }
    }

	//ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::s16HandleSortCallback END tS16Return=%d", tS16Return));
    return tS16Return;
}




/*********************************************************************
 *
 *FUNCTION:     cb_bDSRLIterate
 *
 *DESCRIPTION:  Agw service DSRL iterate callback
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/




/* peha: we should iterate twice:
   1.) check which products have changed

   2.) copy data of changed products
   Second iteration us done in vOnDSRLUpdate() if changes have been detected in first iteration



*/

static tBool bUseEntry(DSRL_ENTRY_STATUS_ENUM eStatus, DSRL_ENTRY_OBJECT hEntryObject)
{
    if (eStatus != DSRL_ENTRY_STATUS_NEW && eStatus != DSRL_ENTRY_STATUS_CHANGED && eStatus != DSRL_ENTRY_STATUS_UNCHANGED)
    {
        ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bUseEntry: ignore entry with status %d",
                        ETG_CENUM(DSRL_ENTRY_STATUS_ENUM, eStatus)));
        return FALSE;
    }

    if (hEntryObject==DSRL_ENTRY_INVALID_OBJECT) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bUseEntry: ignore DSRL_ENTRY_INVALID_OBJECT"));
        return FALSE;
    }
    return TRUE;
}


/* this is the automatic iterate callback
   that is called each time the dsrl goes to ready
   Only check if there are changes in our products
*/
tBool fc_sxm_tclAgwDSRL::bHandleIterateCallback(DSRL_OBJECT hDSRL, 
                                                DSRL_ENTRY_ID tEntryID,
                                                DSRL_ENTRY_STATUS_ENUM eStatus,
                                                DSRL_ENTRY_OBJECT hEntryObject)
{
	(tVoid) hDSRL;
	(tVoid)tEntryID;
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::cb_bDSRLIterate"));
    if (!bUseEntry(eStatus, hEntryObject)) {
        return TRUE;
    }



    DSRL_ENTRY_TYPE_ENUM eEntryType = DSRL_ENTRY.eType(hEntryObject);
    AGW_PRODUCT_TYPE_ENUM enProductType = AGW_PRODUCT_TYPE_UNKNOWN;


	//NIKAI-2616 fix --start
	if(eStatus==DSRL_ENTRY_STATUS_UNCHANGED)
	{

		if(DSRL_ENTRY_TYPE_AGW_TILE != eEntryType)
			return TRUE; 

		//-- allow all the tiles to be processed in unchanged state	
	#if 0
		else 
		{
			AGW_TILE_OBJECT hAgwTile = DSRL_ENTRY.hAgwTile(hEntryObject);
			enProductType = AGW_TILE.eProductType(hAgwTile);
			if(enProductType != AGW_PRODUCT_TYPE_NOWRAD)
				return TRUE;
		}
		
	#endif 
	}


    //NIKAI-2616 fix --end



    switch (eEntryType)
    {
        case DSRL_ENTRY_TYPE_AGW_TILE:
        {
            AGW_TILE_OBJECT hAgwTile = DSRL_ENTRY.hAgwTile(hEntryObject);
            enProductType = AGW_TILE.eProductType(hAgwTile);
        }
            break;
        case DSRL_ENTRY_TYPE_AGW_SHAPE:
        {
            AGW_SHAPE_OBJECT hAgwShape = DSRL_ENTRY.hAgwShape(hEntryObject);
            enProductType = AGW_SHAPE.eProductType(hAgwShape);
        }
        break;
        default:
            break;
    }

    if (AGW_PRODUCT_TYPE_UNKNOWN != enProductType) {
		
		ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bHandleIterateCallback: DSRL_ENTRY_STATUS_ENUM eStatus %d",
                        ETG_CENUM(DSRL_ENTRY_STATUS_ENUM, eStatus)));
        ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bHandleIterateCallback: Processing enProductType %d",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));

        fc_sxm_tclAgwProducts::instance()->vSetChanged(enProductType);
    }
            
    return TRUE;
        
}
 

/*
  here you actually fetch the data of changed products
*/
tBool fc_sxm_tclAgwDSRL::bHandleIterateCallbackRead(DSRL_OBJECT hDSRL, 
                                                    DSRL_ENTRY_ID tEntryID,
                                                    DSRL_ENTRY_STATUS_ENUM eStatus,
                                                    DSRL_ENTRY_OBJECT hEntryObject) const
{
	(tVoid) hDSRL;
	(tVoid) tEntryID;
    ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bHandleIterateCallbackRead"));

    if (!bUseEntry(eStatus, hEntryObject)) {
        return TRUE;
    }

	ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bHandleIterateCallbackRead: DSRL_ENTRY_STATUS_ENUM eStatus %d",
                     ETG_CENUM(DSRL_ENTRY_STATUS_ENUM, eStatus)));
        
    DSRL_ENTRY_TYPE_ENUM eEntryType = DSRL_ENTRY.eType(hEntryObject);
	AGW_PRODUCT_TYPE_ENUM enProductType = AGW_PRODUCT_TYPE_UNKNOWN;

    // todo: only act if product is marked as changed
    switch (eEntryType)
    {
        case DSRL_ENTRY_TYPE_AGW_TILE:
        {
            AGW_TILE_OBJECT hAgwTile = DSRL_ENTRY.hAgwTile(hEntryObject);
			if (hAgwTile == AGW_TILE_INVALID_OBJECT) {
                        break;
            }
            enProductType = AGW_TILE.eProductType(hAgwTile);

            switch (enProductType)
            {
                case AGW_PRODUCT_TYPE_NOWRAD:
                case AGW_PRODUCT_TYPE_WIND_DIRECTION:
                case AGW_PRODUCT_TYPE_WIND_MAGNITUDE:
                {                   
				
 					ETG_TRACE_USR4(("fc_sxm_tclAgwDSRL::bHandleIterateCallbackRead: Processing enProductType %d",
									 ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
					
					enProductType = AGW_TILE.eProductType(hAgwTile);
                    
                    fc_sxm_tclTileProductBase *poProduct=fc_sxm_tclAgwTiles::instance()->poGetProduct(enProductType);
                    
                    if (OSAL_NULL != poProduct && (poProduct->bChanged==TRUE)) {
                        poProduct->vAddTile(hAgwTile);
                    }
                }
                break;
                case AGW_PRODUCT_TYPE_UNKNOWN:
					break;
                default:
                    break;
            }
        }
		break;
        case DSRL_ENTRY_TYPE_AGW_SHAPE:
        {
            AGW_SHAPE_OBJECT hAgwShape = DSRL_ENTRY.hAgwShape(hEntryObject);
            if (hAgwShape == AGW_SHAPE_INVALID_OBJECT) {
                break;
            }
			


			enProductType = AGW_SHAPE.eProductType(hAgwShape);
            fc_sxm_tclShapeProductBase *poProduct=fc_sxm_tclAgwShapes::instance()->poGetProduct(enProductType);
                
            if (OSAL_NULL != poProduct && (poProduct->bChanged==TRUE)) {
                poProduct->vAddShape(hAgwShape);
            }
        }
        break; 
        default:
            break;
    }

	return TRUE;
}











