/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_time_profile.cpp
* @brief       Implementation of the time profile
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_time_profile.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_TIME_PROFILE
#include "trcGenProj/Header/fc_sxm_time_profile.cpp.trc.h"
#endif


/*********************************************************************
*FUNCTION:     vStart
*DESCRIPTION:  Method to store the current time. 
*PARAMETER:    NONE
*RETURNVALUE:  NONE
 ********************************************************************/

tVoid fc_sxmTimeProfile::vStart()
{
   ETG_TRACE_USR4(("Inside fc_sxmTimeProfile::vStart() UseCase = %s ", _strUseCase.c_str()));
	_msecStartTime = OSAL_ClockGetElapsedTime() ;
   ETG_TRACE_USR4((" ----> TIME PROFILE TRACE STARTED: Time since OSAL_BOOTUP = %u  <---- ", _msecStartTime));
}

/*********************************************************************
*FUNCTION:     vStop
*DESCRIPTION:  Method to note the current time and calculate the elapsed time. 
*PARAMETER:    NONE
*RETURNVALUE:  NONE
 ********************************************************************/
tVoid fc_sxmTimeProfile::vStop()
{
   ETG_TRACE_USR4(("Inside fc_sxmTimeProfile::vStop() Addr = %p ",(void *)this));
   if (_msecStartTime == 0)
   {
      ETG_TRACE_ERR(("Initial Time Profile Not Available -- Cannot Calculate Elapsed Time For use case %s", _strUseCase.c_str()));
      return ;
   }

	/*The diff between the start time and the current time is taken to get Elapsed time*/
   /* First take elapsed time value before doing other processing to avoid 
   delays in ElapsedTime Calculation */
	OSAL_tMSecond msecElapsedTime = ( OSAL_ClockGetElapsedTime() - _msecStartTime );
	ETG_TRACE_USR4((" (((( TIME ELAPSED IN msec = %u  )))) ", msecElapsedTime));
   vPrintFormattedOutput(msecElapsedTime);
}

/*********************************************************************
*FUNCTION:     vPrintFormattedOutput
*DESCRIPTION:  Method to print the elapsed time in 
               dd hrs::mm min::ss sec::mm msec format
*PARAMETER:    Elapsed time in milliSec
*RETURNVALUE:  NONE
 ********************************************************************/
tVoid fc_sxmTimeProfile::vPrintFormattedOutput(const OSAL_tMSecond& msecElapsedTime) const
{
   /** Convert mSec to hrs**/
   tU32 u32Hrs = msecElapsedTime / (1000*60*60);

   /** Convert mSec to min**/
   tU32 u32Min = ( msecElapsedTime % ( 1000*60*60) ) / (1000*60) ;

   /** Convert mSec to sec**/
   tU32 u32Sec =  ( msecElapsedTime % ( 1000*60*60) % ( 1000*60) )/ 1000 ;

   /** Calculate the milliSec expired from the given time**/
   tU32 u32MilliSec = msecElapsedTime % 1000;

   ETG_TRACE_USR4(("================= Elapsed Time for UseCase %s is =======================",_strUseCase.c_str()));
   ETG_TRACE_USR4(("== %2u hrs  %2u mins %2u secs %3u milliSec ", u32Hrs, u32Min, u32Sec, u32MilliSec));
   ETG_TRACE_USR4(("========================================================================================="));
}

tVoid fc_sxmProfiler::vPrint(fc_sxm_tenProfilerOp enOp, string oComment) {
    
    ETG_TRACE_COMP(("fc_sxmProfiler(%15s, %d): startTime=%u usec duration=%u usec onCpu=%u usec (%s)", 
                    _oName.c_str(), 
                    ETG_CENUM(fc_sxm_tenProfilerOp, enOp),
                    u32GetMSec(_rStartTime),
                    u32GetMSec(_rEndTime)-u32GetMSec(_rStartTime),
                    u32GetMSec(_rEndTimeCpu)-u32GetMSec(_rStartTimeCpu),
                    oComment.c_str()
                    ));
}
