/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_prio_queue.h
* @brief       Provides message priority queue implementation
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef FC_SXM_TCL_PRIO_QUEUE_H
#define FC_SXM_TCL_PRIO_QUEUE_H

#include <list>


#include "fc_sxm_unique_name.h"
#include "fc_sxm_tcl_sem.h"


typedef enum {
    //    SXM_eMsgPrioHighest,
    fc_sxm_eMsgPrioHigh,
    fc_sxm_eMsgPrioNormal,
    //    SXM_eMsgPrioLowest,
    fc_sxm_eMsgNumPrios
} fc_sxm_tenMsgPrio;

template <class M>
class fc_sxm_tclPrioQueue {
    typedef std::list<M> fc_sxm_tMsgQueue;

 public:

    fc_sxm_tclPrioQueue(tBool bProtected=TRUE):
        _enMaxPrio(fc_sxm_eMsgNumPrios),
        _bProtected(bProtected)
    {
        if (_bProtected) {
            _oSem.vOpen();
        }
    };

    ~fc_sxm_tclPrioQueue() {
        clear();
    };

    tVoid push(M const &oMsg, fc_sxm_tenMsgPrio ePrio=fc_sxm_eMsgPrioNormal) {
        vLock();

        if (((tU32)ePrio >=  (tU32)fc_sxm_eMsgNumPrios)) {
            ePrio = fc_sxm_eMsgNumPrios;
        }

        _aMsgQueues[ePrio].push_back(oMsg);

             if (ePrio<_enMaxPrio) {
                 _enMaxPrio=ePrio;
             }


        vUnLock();
    };


    tVoid pop_front() {
        vLock();
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)fc_sxm_eMsgNumPrios;u32ListNr++) {
            fc_sxm_tMsgQueue *pQ=&_aMsgQueues[u32ListNr];
            if(!pQ->empty()) {
                pQ->pop_front();
                break;
            } else {
                _enMaxPrio=(fc_sxm_tenMsgPrio)(u32ListNr+1);
            }
        }

        vUnLock();

    }

    M const &front() {
        static M oDummyRes;
        vLock();
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)fc_sxm_eMsgNumPrios;u32ListNr++) {
            fc_sxm_tMsgQueue *pQ=&_aMsgQueues[u32ListNr];
            if(!pQ->empty()) {
                vUnLock();
                return pQ->front();
            }
        }
        vUnLock();
        return oDummyRes;
        
    }
    tBool bPop(M *pMsg) {
        tBool bRes=FALSE;
        if (OSAL_NULL==pMsg) {
            return FALSE;
        }
        vLock();
        if (_enMaxPrio>=fc_sxm_eMsgNumPrios) {
            vUnLock();
            return FALSE;

        }
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)fc_sxm_eMsgNumPrios;u32ListNr++) {
            fc_sxm_tMsgQueue *pQ=&_aMsgQueues[u32ListNr];
            if(!pQ->empty()) {
                *pMsg=pQ->front();
                pQ->pop_front();
                bRes=TRUE;
                break;
            } else {
                _enMaxPrio=(fc_sxm_tenMsgPrio)(u32ListNr+1);
            }
        }
        vUnLock();
        return bRes;
    };

    tVoid clear() {
        vLock();
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)fc_sxm_eMsgNumPrios;u32ListNr++) {
            _aMsgQueues[u32ListNr].clear();
        }
        _enMaxPrio=fc_sxm_eMsgNumPrios;
        vUnLock();
    };

    tU32 size(tBool bLock=TRUE) const {
        if (bLock) {
            vLock();
        }
        tU32 u32Size=0;
        for (tU32 u32ListNr=(tU32)_enMaxPrio;u32ListNr<(tU32)fc_sxm_eMsgNumPrios;u32ListNr++) {
            u32Size+=_aMsgQueues[u32ListNr].size();
        }
        if (bLock) {
            vUnLock();
        }
        return u32Size;
    };

    bool empty() {
        return size()==0;
    };


 private:
    fc_sxm_tMsgQueue _aMsgQueues[fc_sxm_eMsgNumPrios+1];
    fc_sxm_tenMsgPrio _enMaxPrio; // contains prio of non-empty queue with highest priority
    mutable fc_sxm_tclSem _oSem;
    tBool _bProtected;
    tVoid vLock() const {
        _oSem.vGet();
    }
    tVoid vUnLock() const {
        _oSem.vPost();
    }

};

#endif
