/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_data_list.hpp
* @brief       Declaration of data list handling
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
#include "fc_sxm_tcl_data_list_new.hpp"
#else
#ifndef FC_SXM_TCK_DATA_LIST_H
#define FC_SXM_TCK_DATA_LIST_H


#include "fc_sxm_tcl_base_dsrl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_BASE_APP
#include "trcGenProj/Header/fc_sxm_tcl_data_list.hpp.trc.h"
#endif

class midw_ext_fi_tcl_SxmListInfos;

// base-class for list-configuration
struct fc_sxm_trBaseListConfig {
public:
    // has to be overwritten if location is used for sorting or filtering
	virtual ~fc_sxm_trBaseListConfig() {
	}

    virtual tBool bUsesLocation() const {
        return FALSE;
    }
    tVoid vPrint() const;
    virtual tVoid vPrintSpecific() const {};
};


typedef enum {
    fc_sxm_enListEntryState_Normal,
    fc_sxm_enListEntryState_New,
    fc_sxm_enListEntryState_Updated,
    fc_sxm_enListEntryState_Deleted
} fc_sxm_tenListEntryState;
// base-class for all list-entries that a contained in the list.
/* app-specific list has to derive from this class and add members
   that shall be send to hmi 
*/
struct fc_sxm_trBaseListEntry {
public:
    LOCATION_OBJECT hLocation;  
    tU32 u32SxmId;
    DSRL_ENTRY_ID u32DsrlEntryID;
    fc_sxm_tenListEntryState enState;
    fc_sxm_trBaseListEntry():
        u32SxmId(0),
        u32DsrlEntryID(0),
        enState(fc_sxm_enListEntryState_Normal),
        hLocation(LOCATION_INVALID_OBJECT)
    {}

    virtual ~fc_sxm_trBaseListEntry()  {hLocation = LOCATION_INVALID_OBJECT;}
};

/*
  list-types
  currently special handling of location for NearBy and NearDest is implemented.
  For these types the framework will apply the location to be used.
*/
typedef enum {
    fc_sxm_tenListType_Invalid,
    fc_sxm_tenListType_NearBy,
    fc_sxm_tenListType_NearDest,
    fc_sxm_tenListType_Favorites,
    fc_sxm_tenListType_AnotherLocation,
    fc_sxm_tenListType_Custom
} fc_sxm_tenListType;

typedef enum {
    fc_sxm_enListState_Initial,
    fc_sxm_enListState_Ready,
    fc_sxm_enListState_Changed,
    fc_sxm_enListState_Updating,
    fc_sxm_enListState_Activated,
    fc_sxm_enListState_Frozen,
    fc_sxm_enListState_Error
} fc_sxm_tenListState;

typedef enum {
    fc_sxm_enListMode_Invalid,
    fc_sxm_enListMode_Release,
    fc_sxm_enListMode_Activate,
    fc_sxm_enListMode_Pending_Activate,
    fc_sxm_enListMode_Freeze,
    fc_sxm_enListMode_Pending_Freeze
} fc_sxm_tenListMode;

typedef enum {
    fc_sxm_enListFreezeState_Idle,
    fc_sxm_enListFreezeState_Enter,
    fc_sxm_enListFreezeState_Freezing,
    fc_sxm_enListFreezeState_Frozen
} fc_sxm_tenListFreezeState;

typedef enum {
    fc_sxm_tenListActivateState_Idle,
    fc_sxm_tenListActivateState_Enter,
    fc_sxm_tenListActivateState_Activated
} fc_sxm_tenListActivateState;

class midw_ext_fi_tcl_SxmListInfo;

/*
  struct that contains generic information for a list.
*/
struct fc_sxm_listStatusInfo {
    fc_sxm_listStatusInfo():
        u32Id(0),
        enListType(fc_sxm_tenListType_Invalid),
        enListState(fc_sxm_enListState_Initial),
        bActivated(FALSE),
        bFrozen(FALSE),
        u32Size(0)
    {}

    bool operator!=(fc_sxm_listStatusInfo const &b) const;
    // operator to convert listStatus to fi-type
    operator midw_ext_fi_tcl_SxmListInfo() const;

    tU32 u32Id;
    fc_sxm_tenListType enListType;
    fc_sxm_trDsrlLocation rLocation;
    fc_sxm_tenListState enListState;
    tBool bActivated; // list shall indicate with "fc_sxm_enListState_Changed" to hmi that data have changed
    tBool bFrozen; // list-items and sorting are frozen, only content of items will be updated
    tU32 u32Size;
} ;

class fc_sxm_tclDataListBase {
 public:
    fc_sxm_tclDataListBase(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp, tU16 u16TraceClass);

    virtual ~fc_sxm_tclDataListBase();

    /* request if a request for the list is pending.
       List goes to pending if hmi requests a list that is not in state ready or changed.
       Once the list goes into ready or changed the pending request will be answered.
    */
    tBool bRequestPending() const;

    /*
      we get informed that our dsrl gets iterated now
    */
    tVoid vOnDsrlUpdateStart() {
        vActivate();
    }

    char const *szGetAppName() const;

    virtual fc_sxm_trBaseListConfig const *poGetBaseCfg() const =0;

    tVoid vPrintReport() const;
    virtual tVoid vPrintStatistics() const=0;


private:
    /*
      take action to enter new state and set the new state
    */
    tVoid vUpdateListState(fc_sxm_tenListState enNewState);

private:
    /*
      method to initiate sending of list to hmi now
    */
    tVoid vEmit() {
        ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vEmit() START"));

        vEmit(_rPendingRequest);

        /* check if we are ready for freezing the list */
        if (_enListFreezeState==fc_sxm_enListFreezeState_Freezing) {

            /* OK , then change list to frozen */
            _enListFreezeState=fc_sxm_enListFreezeState_Frozen;
        }
        _rPendingRequest=fc_sxm_trAdressing();
        ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vEmit() END"));
    }

protected:
    /*
      method to be overwritten by base-class to send list to hmi according to rAdressing
    */
    virtual tVoid vEmit(fc_sxm_trAdressing const &rAdressing) {
    	(tVoid)rAdressing;
        ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vEmit: please overwrite"));
    }
    /*
      method to be overwritten by base-class to start the timer
    */
    virtual tVoid vStartTimer() {
        ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vStartTimer: please overwrite"));
    }
    virtual tBool bIsListModeTimerRunning() {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase::vIsListModeTimerRunning overwrite"));
        return FALSE;
    }
    /*
      method to be overwritten by base-class to stop the timer
    */
    virtual tVoid vStopTimer() {
        ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vStopTimer: please overwrite"));
    }
    /*
      an DSRL-entry became invalid, remove it
    */
    virtual tVoid vRemoveEntry_(DSRL_ENTRY_ID tEntryID)=0;
    virtual tU32 u32GetListSize()=0;


public:

    /*
      we get informed about a new state of our dsrl
    */
    tVoid vOnSmsDsrlState(fc_sxm_tenDsrlState enSmsDsrlState,tBool bDataAvailable=TRUE);
    
    /* 
      we get a request to update the ListMode
    */
    tVoid vSetListMode(fc_sxm_trAdressing const &rAdressing, fc_sxm_tenListMode enListMode); 

    tVoid vOnListTimerExpire(tVoid);

    /*
      we get request to send the list according to rAdressing
    */
    tVoid vRequest(fc_sxm_trAdressing const &rAdressing);


    /*
      return pointer to our dsrl
    */
    fc_sxm_tclBaseDSRL *poGetDsrl() {
        return _poBaseDsrl;
    }
    fc_sxm_tclBaseDSRL const *poGetDsrl() const {
        return _poBaseDsrl;
    }

    /*
      set our dsrl and take actions accordingly
    */
    tVoid vSetDsrl(fc_sxm_tclBaseDSRL *poBaseDsrl);

    /*
      set new location and take actions accordingly
    */
    tVoid vSetLocation(fc_sxm_trDsrlLocation const &rLocation);
    /*
      Notify HMI that List is Updating
    */    
    tVoid vNotifyUpdatingState();

    /*
      return our id
    */
    tU32 u32GetId() {
        return _u32ListId;
    }
    

    tBool bIsFrozen() const {
        return _enListFreezeState==fc_sxm_enListFreezeState_Frozen;
    }

    tBool bIsFreezing() const {
        return _enListFreezeState==fc_sxm_enListFreezeState_Freezing;
    }

    fc_sxm_tenListFreezeState enGetListFreezeState() const {
        return _enListFreezeState;
    }
     tBool bIsActivated() const {
        return _enListActiveState==fc_sxm_tenListActivateState_Activated;
    }

    fc_sxm_tenListActivateState enGetListActivateState() const {
        return _enListActiveState;
    }
    fc_sxm_tenListMode enGetListMode() const {
        return _enListMode;
    }
    /* 
      method to clear list specific data
    */
    tVoid vClearListData();

    /* method to be overriden by each service
      who wants to validate data from sms */
    virtual tBool bIsListDataValid() {
      return TRUE;
    }

protected:
    /*
      initiate filtering and sorting of current data
    */
    virtual tVoid vRefresh()=0;

    /*
      set all entry-state from Changed to Normal
    */
    virtual tVoid vResetChanged()=0;

    /*
      clear the set of sorted data.
      Sorted data are cleared and created again on request if bIsActivated() is not set.
      If bIsActivated is set, sorted data a kept up to data.
      Sorted data are always cleared if the configuration has changed.
    */
    virtual tVoid vClearSort()=0;
    /*
      Clear data receive from DSRL.
      e.g. a new dsrl has been set or the old DSRL has gone to state init or error.
    */
    tVoid vClearData() {
        vClearData(TRUE);
    }
    virtual tVoid vClearData(tBool bAll)=0;

    /*
      Set active list-instance. 
      The active instance is needed by the sort- and filter-functionality
    */
    virtual tVoid vActivate()=0;

public:
    /*
      get status of the list
    */
    fc_sxm_listStatusInfo rGetStatusInfo() const;

protected:
    /*
      static method to get the dataList according to u32Id.
      The data-app is different for each template deriving from us.
    */
    static fc_sxm_tclDataListBase *poGetDataListBase(tU32 u32Id, fc_sxm_tclDataApp *poDataApp);

protected:
    // base-sort to store all entries received from DSRL-iterate
    static bool bCompareListEntryId(fc_sxm_trBaseListEntry const *prLeft, fc_sxm_trBaseListEntry const *prRight);

    bool bCompareSmsDistance(fc_sxm_trBaseListEntry const *prLeft, fc_sxm_trBaseListEntry const *prRight) const;
    bool bCompareFavoriteIndex(fc_sxm_trBaseListEntry const *prLeft, fc_sxm_trBaseListEntry const *prRight) const;

    tVoid vSetSmsLocation(fc_sxm_trDsrlLocation const &rLocation);

    virtual tBool bUsesLocation()=0;
 private:
    tU32 _u32ListId;
    fc_sxm_tenListType _enListType;
    fc_sxm_tenListState _enListState;
    fc_sxm_tclBaseDSRL *_poBaseDsrl;
    fc_sxm_trAdressing _rPendingRequest;
    fc_sxm_tenListFreezeState _enListFreezeState;
    fc_sxm_tenListActivateState _enListActiveState;
    fc_sxm_tenListMode _enListMode;
    tU32 _u32ListSize;

protected:
    LOCATION_OBJECT _hLocation;
    fc_sxm_trDsrlLocation _rLocation;

private:
    fc_sxm_tclDataApp *_poDataApp;

protected:
    tU16 _u16TraceClass;

public:
    tU16 u16GetTraceClass() {
        return _u16TraceClass;
    }

};

template<class LISTTYPE, class LISTCFG>
class fc_sxm_tclDataListTyped:public fc_sxm_tclDataListBase {
 public:
    fc_sxm_tclDataListTyped(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp, tU16 u16TraceClass):
        fc_sxm_tclDataListBase(u32ListId, enListType, poDataApp, u16TraceClass)
    {
        _poDataAppStatic=poDataApp;
    };

    virtual ~fc_sxm_tclDataListTyped() {};
protected:
      LISTCFG _rListCfg;
      typedef typename LISTCFG::trListEntry _trListEntry;

      virtual tVoid vAddEntry(const _trListEntry& /*rEntry*/){
         /*To be implemented by Derived classes */
      }

      virtual tVoid vDeleteEntry(const _trListEntry& /*rEntry*/){
         /*To be implemented by Derived classes */
      }

 public:
    virtual fc_sxm_trBaseListConfig const *poGetBaseCfg() const {
        return &_rListCfg;
    }

    virtual tVoid vPrintStatistics() const {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tnEntriesRaw=%d", _mapEntries.size()));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tnEntriesSorted=%d", _setSortedEntries.size()));
        if (bIsFrozen()) {
            tU32 u32NumDeleted=0;
            typename set<_trListEntry*, trCompareListEntry>::const_iterator iter =_setSortedEntries.begin();
            while (iter != _setSortedEntries.end()) {
                if ((*iter)->enState == fc_sxm_enListEntryState_Deleted) {
                    u32NumDeleted++;
                }
                iter++;
            }
            ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tnEntriesFrozenDeleted=%d", u32NumDeleted));
        }
    }
    tVoid vSetCfg(LISTCFG const &rListCfg) {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vSetCfg START"));
        /* operator != is implemented, that is returns FALSE if actual handling is not affected.
           So we alway copy the new config, but we only apply it, if it changes the current handling
           e.g. the brand in the filter-configuration is changed but the current filter-mode does not use the brand.
        */
        tBool bChanged=rListCfg != _rListCfg;
        _rListCfg=rListCfg;
        if (bChanged) {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vSetCfg CHANGED"));
            vClearSort();
            if (bIsActivated()) {
                vRefresh();
            }
        }
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vSetCfg END"));

    }

    static tVoid vSetCfg(tU32 u32Id, LISTCFG const &rListCfg) {
        fc_sxm_tclDataListBase *poListBase=poGetDataListBase(u32Id, _poDataAppStatic);
        fc_sxm_tclDataListTyped<LISTTYPE, LISTCFG> *poTypedList=static_cast<fc_sxm_tclDataListTyped<LISTTYPE, LISTCFG> *>(poListBase);
        if (poTypedList) {
            poTypedList->vSetCfg(rListCfg);
        } else 
        {
            ETG_TRACE_USR4(("fc_sxm_tclDataListTyped:vSetCfg u32Id=%d NOT FOUND", u32Id));
        }
        
    }
    
    static tVoid vRemoveEntry(DSRL_ENTRY_ID tEntryID) {
       LISTTYPE *poList=prGetActiveList();
       if (OSAL_NULL !=poList) {
           ETG_TRACE_USR4_DCL((poList->_u16TraceClass, "fc_sxm_tclDataListTyped:vRemoveEntry(0x%08x) FOUND LIST ", tEntryID));
           poList->vRemoveEntry_(tEntryID);
       }
       else {
           ETG_TRACE_USR1(("fc_sxm_tclDataListTyped:vRemoveEntry(0x%08x) NOT FOUND", tEntryID));
       }
    }

   static tVoid vUpdateEntry(DSRL_ENTRY_STATUS_ENUM eStatus, _trListEntry &rEntry) {
       LISTTYPE *poList=prGetActiveList();
       if (OSAL_NULL !=poList) {
           ETG_TRACE_USR4_DCL((poList->_u16TraceClass, "fc_sxm_tclDataListTyped:vUpdateEntry FOUND LIST "));
           poList->vUpdateEntry_(eStatus, rEntry);
       }
       else {
           ETG_TRACE_USR1(("fc_sxm_tclDataListTyped:vUpdateEntry LIST NOT FOUND"));
       }
   }

    tBool bGetListEntry(tU32 u32Id, _trListEntry &rEntry) {
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:bGetListEntry u32Id=%d",u32Id));
        // check if val exist:
        typename map< tU32, _trListEntry >::iterator iter = _mapEntries.find(u32Id);
        if (iter!=_mapEntries.end()) {
            rEntry=iter->second;
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:bGetListEntry TRUE"));
            return TRUE;
        }
        // fill at least the id
        rEntry=_trListEntry();
        rEntry.u32SxmId=u32Id;
        return FALSE;
    }
    
 protected:

    virtual tBool bUsesLocation() {
        return _rListCfg.bUsesLocation();
    }

   // sort-method for STL-containers,
   struct trCompareListEntryId {
       bool operator()(_trListEntry const &rLeft, _trListEntry const &rRight) const {
           return rLeft.u32SxmId<rRight.u32SxmId;
       }
   };


   struct trCompareListEntry {
       bool operator()(_trListEntry const *prLeft, _trListEntry const *prRight) const {
           if(prLeft == OSAL_NULL || prRight == OSAL_NULL){
              ETG_TRACE_USR4(("fc_sxm_tclDataListTyped:trCompareListEntry NULL"));            
              return false;
           }

           LISTTYPE *poList=prGetActiveList();
           if (OSAL_NULL !=poList) {
               ETG_TRACE_USR4_DCL((poList->_u16TraceClass, "fc_sxm_tclDataListTyped:trCompareListEntry prLeft=%08x prRight=%08x ",
                                   prLeft, prRight));
               return poList->bCompareItem(prLeft, prRight);
           }
           else {
               ETG_TRACE_USR4(("fc_sxm_tclDataListTyped:trCompareListEntry prLeft=%08x prRight=%08x ",
                               prLeft,prRight));               
               return bCompareListEntryId(prLeft, prRight);
           }
       }
   };

   virtual bool bCompareItem(_trListEntry const *prLeft, _trListEntry const *prRight) const =0;
   virtual bool bFilterPassed(_trListEntry const &rItem) const =0;


    #define trSortedSetType set<_trListEntry*, trCompareListEntry>

    const trSortedSetType &rGetSortedEntries() {
        return _setSortedEntries;
    }

private:
   virtual tU32 u32GetListSize()
      {
         tU32 u32Size = 0;
         u32Size = bIsFrozen() ?(tU32)_setSortedEntries.size() :(tU32)_mapEntries.size();
         return u32Size;
      }

    /*
      an DSRL-entry became invalid, remove it
    */
    virtual tVoid vRemoveEntry_(DSRL_ENTRY_ID tEntryID) {
        typename map< tU32, _trListEntry >::iterator iter = _mapEntries.begin();
        while (iter != _mapEntries.end()) {
            if (iter->second.u32DsrlEntryID == tEntryID) {
                _trListEntry rEntryCopy=iter->second;
                vUpdateEntry_(DSRL_ENTRY_STATUS_REMOVED, rEntryCopy);
                return;
            }
        iter++;
       }
    }


    tVoid vUpdateEntry_(DSRL_ENTRY_STATUS_ENUM eStatus, _trListEntry &rEntry) {
        tU32 u32SxmId=rEntry.u32SxmId;
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vUpdateEntry_ eStatus=%d u32SxmId=0x%08x size=%d START",
                            ETG_CENUM(DSRL_ENTRY_STATUS_ENUM, eStatus),
                            u32SxmId,
                            _mapEntries.size()));
        
        if (eStatus==DSRL_ENTRY_STATUS_UNCHANGED) {
            // do nothing
            return;
        }

        fc_sxm_tclBaseDSRL *poActiveDsrl = poGetDsrl();
        if (OSAL_NULL == poActiveDsrl) {
            
            ETG_TRACE_USR4_DCL((_u16TraceClass, "vUpdateEntry DSRL returning NULL pointer"));
            return;
        }

        // check if val exist:
        typename map< tU32, _trListEntry >::iterator iter = _mapEntries.find(u32SxmId);
        tBool bExists=FALSE;
        if (iter!=_mapEntries.end()) {
            bExists=TRUE;
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:bExists!"));
        }

        /* Check for automatic updates - Not Frozen case */
        if(bIsActivated() && !bIsFrozen()) {
            /*we should immediately update HMI for NEW,CHANGED,REMOVED cases */
            poActiveDsrl->vSetChangedFlag(TRUE);
           }
        
        if ((eStatus==DSRL_ENTRY_STATUS_NEW) || (eStatus==DSRL_ENTRY_STATUS_CHANGED)) {
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:NEW OR CHANGED"));

            if (bExists) {
                ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:updated"));
                // keep old location
                rEntry.hLocation=iter->second.hLocation;
                // write new value
                iter->second=rEntry;
                if (bIsFrozen() && (FALSE == bIsListModeTimerRunning())) {
                    poActiveDsrl->vSetChangedFlag(TRUE);
                }
            }
            else
               {
                  ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:new"));
                  // allocate new location
                  // insert new value
                  iter=_mapEntries.insert(make_pair(rEntry.u32SxmId,rEntry)).first;
                  vAddEntry(rEntry);
                  if (LOCATION_INVALID_OBJECT != rEntry.hLocation)
                  {
                     iter->second.hLocation=LOCATION.hDuplicate(rEntry.hLocation);
                     SXM_ASSERT_RETURN(LOCATION_INVALID_OBJECT !=iter->second.hLocation);
                  }
               }
        } 
        else if (eStatus==DSRL_ENTRY_STATUS_REMOVED) {
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:remove?"));
            if (bExists) {

                /* Property Update sent to HMI that List is frozen
                   So on freezing & Frozen State don't remove any entries from list */
                if (bIsFreezing() || bIsFrozen()) {
                    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:dont remove because frozen"));
                    iter->second.enState=fc_sxm_enListEntryState_Deleted;
                                        
                } else {
                    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:remove!"));
                    LOCATION.vDestroy(iter->second.hLocation);
                    _mapEntries.erase(iter);
                    vDeleteEntry(rEntry);
                }
            }
        } 
        else {
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:IGNORE"));
        }
        
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vUpdateEntry_ size=%d END",
                        _mapEntries.size()));
    }

protected:



   virtual tVoid vRefresh() {
       if (bIsFrozen()) {
           return;
       }
       _prActiveList=static_cast<LISTTYPE *>(this);
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vRefresh (numEntries=%d) START",
                           _mapEntries.size()));
       vClearSort();
       for (typename map< tU32, _trListEntry >::iterator iter = _mapEntries.begin();
            iter != _mapEntries.end();
            ++iter) {
           _trListEntry &rEntry=iter->second;
           if (!bFilterPassed(rEntry)) {
               ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:FILTERED OUT"));

               continue;
           }
           ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:FILTER PASSED"));
           _setSortedEntries.insert(&rEntry);
       }
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vRefresh END"));

   };

    virtual tVoid vResetChanged() {
       for (typename map< tU32, _trListEntry >::iterator iter = _mapEntries.begin();
            iter != _mapEntries.end();
            ++iter) {
           if (iter->second.enState==fc_sxm_enListEntryState_Updated) {
               iter->second.enState=fc_sxm_enListEntryState_Normal;
           }
       }
    }

   virtual tVoid vClearData(tBool bAll) {
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vClearData(bAll=%d) START",
                           bAll));
       vClearSort();
       if (bAll) {
    	   _mapEntries.clear();
       }
       else {
    	   typename map< tU32, _trListEntry >::iterator iter = _mapEntries.begin();
    	   while (iter != _mapEntries.end()) {
           typename map< tU32, _trListEntry >::iterator iterDelete=iter;
           iter++;
           if (iterDelete->second.enState==fc_sxm_enListEntryState_Deleted) {
               LOCATION.vDestroy(iterDelete->second.hLocation);
               _mapEntries.erase(iterDelete);
           	   }
    	   }
       }
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vClearData END"));
   };




   virtual tVoid vClearSort() {
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListTyped:vClearSort START"));
       _setSortedEntries.clear();
       ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vClearSort END"));
   }
       


    map<tU32, _trListEntry > _mapEntries;
    set<_trListEntry*, trCompareListEntry> _setSortedEntries;
    static LISTTYPE *prGetActiveList() {
        return _prActiveList;
    }
    tVoid vActivate() {
        _prActiveList=static_cast<LISTTYPE *>(this);;
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vActivate=%08x", _prActiveList));
    }

public:
    static LISTTYPE *poGetTypedList(fc_sxm_tclDataListBase *poDataListBase) {
        LISTTYPE *poRes=OSAL_NULL;
        if (OSAL_NULL != poDataListBase) {
            poRes = static_cast<LISTTYPE *>(poDataListBase);
        }
        return poRes;
    }
    static LISTTYPE *poGetTypedListById(tU32 u32Id) {
        return poGetTypedList(poGetDataListBase(u32Id, _poDataAppStatic));
    }

    // removed const type from rGetMapEntries()
    map<tU32, _trListEntry > &rGetMapEntries() { return _mapEntries;}


 private:
    static LISTTYPE *_prActiveList;
    static fc_sxm_tclDataApp *_poDataAppStatic;
};


template <typename LISTTYPE, typename LISTCFG> LISTTYPE* fc_sxm_tclDataListTyped <LISTTYPE, LISTCFG>::_prActiveList = 0;
template <typename LISTTYPE, typename LISTCFG> fc_sxm_tclDataApp* fc_sxm_tclDataListTyped <LISTTYPE, LISTCFG>::_poDataAppStatic = 0;




#endif
#endif
