/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_generic_utils.cpp
* @brief       Providing generic functionalities that is used across the project and those
*              which are not dependent on any SMS Library calls
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

// just take any data-fi for processing FI-based functionalities
#include "fc_sxm_fuel_fi.h"
#include "fc_sxm_generic_utils.h"

#define FC_SXM_FACTOR_NAV_POSITION (360.0f/4294967295.0f)
#define FC_SXM_FACTOR_SMS_TO_NAV ((360.0f/4294967295.0f) * 0xFFFF)

// Method converts char* pointer to FI string
tVoid fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(tCString sSrcString,  midw_ext_fi_tclString& rffiObj)
{
    if(OSAL_NULL != sSrcString)
    {
        // Convert the string to ISO-8859-1 format Call the function to convert the string to ISO-8859-1 format
        (tVoid) rffiObj.bSet(sSrcString, midw_ext_fi_tclString::FI_EN_ISO8859_1M);
    }
}

/*********************************************************************
 *FUNCTION:     vReplaceChars
 *DESCRIPTION:  Utility method that does a search and replace of a character
                in the given string
 *PARAMETER:    1) Handle to to std::string object which has to be searched.
                2) Character to be searched
				3) Character to be replaced with
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclGenericUtils::vReplaceChars(string& sInput, tChar chFind, tChar chReplace)
{
	size_t found = sInput.find_first_of(chFind);
	while(found != string::npos)
	{
		sInput[found] = chReplace;
		found = sInput.find_first_of(chFind, found+1);
	}
}

/*********************************************************************
 *FUNCTION:     vConvertTTFisString2FIString
 *DESCRIPTION:  Utility method reads TTFIs string input and converts it into
 *DESCRIPTION:   FI string object.
 *PARAMETER:    Pointer to the TTFIs string object, Handle to FI string object
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclGenericUtils::vConvertTTFisString2FIString( tCString chInString, midw_ext_fi_tclString& rffiObj)
{
	//Method reads TTFIs string object and converts it into FI string
	//This method cannot be invoked directly from HMI and gets executed only when user enters
	// string inputs from TTFI
	string sTTFisString(chInString);
	fc_sxm_tclGenericUtils::vReplaceChars(sTTFisString, '+', ' ');
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(sTTFisString.c_str() ,rffiObj);
}

/*
 * Method converts string to unique hash code
 */
tU32 fc_sxm_tclGenericUtils::u32ComputeHashCode(const string& strInput)
{
	const tU32 u32HashMultiplier = 3U;
	tU32 u32Result = 0U;
	for (size_t tIndex = 0; tIndex < strInput.size(); ++tIndex)
	{
		u32Result = u32HashMultiplier * u32Result + strInput[tIndex];
	}
	return u32Result;
}

/*
 * Method clears a bit from number u32Value
 */
tVoid fc_sxm_tclGenericUtils::vClearBit(tU32& u32Value, const tU32 u32BitPosition)
{
	u32Value &= ~(1U << u32BitPosition);
}

/*
 * Method sets u32BitPosition to 1 from number u32Value
 */
tVoid fc_sxm_tclGenericUtils::vSetBit(tU32& u32Value, const tU32 u32BitPosition)
{
	u32Value |= (1U << u32BitPosition);
}

/*
 * Method toggles u32BitPosition's ON/OFF status from u32Value
 */
tVoid fc_sxm_tclGenericUtils::vToggleBit(tU32& u32Value, const tU32 u32BitPosition)
{
	u32Value ^= (1U << u32BitPosition);
}

/*
 * Method checks if a bit is set or not
 */
tBool fc_sxm_tclGenericUtils::bCheckBit(const tU32 u32Value, const tU32 u32BitPosition)
{
	return (static_cast <tBool> ((u32Value >> u32BitPosition) & 1U));
}

/*
 * Method converts SMS float value to Nav value
 */
tS32 fc_sxm_tclGenericUtils::s32SmsDegreeFloatToNav(tFloat fFloatSmsCoordinate)
{
    return (tS32)(fFloatSmsCoordinate / FC_SXM_FACTOR_NAV_POSITION);
}
/*
 * Method converts Nav value to SDK float
 */
tF32 fc_sxm_tclGenericUtils::f32DegreeNavToFloat(tS32 s32NavCoordinate)
{
    return ((tF32)(s32NavCoordinate) * FC_SXM_FACTOR_NAV_POSITION);
}
/*
 * Method converts SMS value to Nav type
 */
tS32 fc_sxm_tclGenericUtils::s32DegreeSmsToNav(tS32 s32SmsCoordinate)
{
    return (tS32)((tFloat)s32SmsCoordinate / FC_SXM_FACTOR_SMS_TO_NAV);
}

/*
  helper function to convert string to unicode
*/
tU32 fc_sxm_tclGenericUtils::vGetUnicode(tU8* copU8Char, tU32& rfu32Unicode)
{
	tU32 u32Size = 0;
   if (copU8Char == NULL)
   {
      rfu32Unicode = 0;
   }
   else if ( 0x80 > *copU8Char ) // is one byte
   {
      rfu32Unicode = *copU8Char;
      u32Size = 1;

   }
   else if ( 0xE0 > *copU8Char ) // is 2 bytes
   {
      rfu32Unicode = 0x1F & *copU8Char;
      rfu32Unicode <<= 6; ++copU8Char;
      rfu32Unicode |= 0x3F & *copU8Char;
      u32Size = 2;
   }
   else if ( 0xF0 > *copU8Char ) // is 3 bytes
   {
      rfu32Unicode = 0x0F & *copU8Char;
      rfu32Unicode <<= 6; ++copU8Char;
      rfu32Unicode |= 0x3F & *copU8Char;
      rfu32Unicode <<= 6; ++copU8Char;
      rfu32Unicode |= 0x3F & *copU8Char;
      u32Size = 3;
   }
   else if ( 0xF8 > *copU8Char ) // is 4 bytes
   {
      rfu32Unicode = 0x07 & *copU8Char;
      rfu32Unicode <<= 6; ++copU8Char;
      rfu32Unicode |= 0x3F & *copU8Char;
      rfu32Unicode <<= 6; ++copU8Char;
      rfu32Unicode |= 0x3F & *copU8Char;
      rfu32Unicode <<= 6; ++copU8Char;
      rfu32Unicode |= 0x3F & *copU8Char;
      u32Size = 4;
   }
   else // longer as 4 bytes
   {
      rfu32Unicode = LIMIT_C_MAX_U32;
      u32Size = LIMIT_C_MAX_U32;
   }
   return u32Size;
}
/*
  helper function to convert unicode to utf8 string
*/
tCString fc_sxm_tclGenericUtils::strUnicode2UTF8(tU32 u32Unicode, tChar* utf8Char )
{
   tU8 u8Idx = 4;
   if (u32Unicode < 0x80  )
   {
      --u8Idx;
      utf8Char[u8Idx] = (tU8) u32Unicode;
   }
   else
   {
      tU8 u8ValContent = 0;
      for (; u8Idx>0 && u32Unicode != 0 ; --u8Idx, u32Unicode >>= 6)
      {
         u8ValContent = tU8( 0x3F & u32Unicode );
         utf8Char[u8Idx-1] = 0x80 | u8ValContent;
      }
      // if the data bytes of the current byte are sufficient
      // The high bytes re the interesting ones:
      // 2 bytes 110xxxxx -> 5 bits content
      // 3 bytes 1110xxxx -> 4 bits content
      // 4 bytes 11110xxx -> 3 bits content
      if( (0 != u8Idx && u8ValContent) > (0xFF >> (5-u8Idx)) )
         --u8Idx;

      utf8Char[u8Idx] =  utf8Char[u8Idx] | (tU8)(0xF0 << u8Idx);
   }
   return &utf8Char[u8Idx];
}

/*
  helper function to convert unicode to string
*/
tVoid fc_sxm_tclGenericUtils::vUnicodeToString(tU32 u32Unicode, string& outputString)
{
   if (u32Unicode == 0 || u32Unicode > 0x001FFFFF)
      return;
   char utf8Char[5] = {0,0,0,0,0};

   (void)outputString.append( strUnicode2UTF8(u32Unicode, &utf8Char[0]) );
}


/*
   helper to convert lower case to Upper string
*/
tVoid fc_sxm_tclGenericUtils::toUpper(tU32& rfu32Val)
{
   //Basic Latin (0x0000 - 0x007F)
   if( rfu32Val <= 0x7f )
   {
      if( (rfu32Val >= 0x61) && (rfu32Val <= 0x7a) )
      {
         rfu32Val -= 32;
      }
   }
   //Latin-1 Supplement (0x0080 - 0x00FF)
   else if( rfu32Val >= 0x80 && rfu32Val <= 0xff )
   {
      if( ((rfu32Val >= 0xe0) && (rfu32Val <= 0xf6)) ||
         ((rfu32Val >= 0xf8) && (rfu32Val <= 0xfe)) )
      {
         rfu32Val -= 32;
      }
      else if( rfu32Val == 0xff )
      {
         rfu32Val += 121;
      }
      else if( rfu32Val == 0xb5 )
      {
         rfu32Val += 743;
      }
   }
   //Latin Extended-A (0x0100 - 0x017F)
   else if( rfu32Val >= 0x100 && rfu32Val <= 0x17f )
   {
      if( rfu32Val == 0x17f )
      {
         rfu32Val -= 300;
      }
      else if( rfu32Val == 0x131 )
      {
         rfu32Val -= 232;
      }
      else if( ((rfu32Val >= 0x101) && (rfu32Val <= 0x12f) && ((rfu32Val & 0x1) == 1)) ||
         ((rfu32Val >= 0x133) && (rfu32Val <= 0x137) && ((rfu32Val & 0x1) == 1)) ||
         ((rfu32Val >= 0x13a) && (rfu32Val <= 0x148) && ((rfu32Val & 0x1) == 0)) ||
         ((rfu32Val >= 0x14b) && (rfu32Val <= 0x177) && ((rfu32Val & 0x1) == 1)) ||
         ((rfu32Val >= 0x17a) && (rfu32Val <= 0x17e) && ((rfu32Val & 0x1) == 0)) )
      {
         rfu32Val -= 1;
      }
      else if( rfu32Val == 0x149 )
      {
         rfu32Val += 371;
      }
   }
   //Latin Extended-B (0x0180 - 0x024F)
   else if( rfu32Val >= 0x180 && rfu32Val <= 0x24f )
   {
      if( rfu32Val == 0x1f0 )
      {
         rfu32Val -= 422;
      }
      else if( rfu32Val == 0x1dd )
      {
         rfu32Val -= 79;
      }
      else if( (rfu32Val == 0x1c6) ||
         (rfu32Val == 0x1c9) ||
         (rfu32Val == 0x1cc) )
      {
         rfu32Val -= 2;
      }
      else if( ((rfu32Val >= 0x183) && (rfu32Val <= 0x185) && ((rfu32Val & 0x1) == 1)) ||
         (rfu32Val == 0x188) ||
         (rfu32Val == 0x18c) ||
         (rfu32Val == 0x192) ||
         (rfu32Val == 0x199) ||
         ((rfu32Val >= 0x1a1) && (rfu32Val <= 0x1a5) && ((rfu32Val & 0x1) == 1)) ||
         (rfu32Val == 0x1a8) ||
         (rfu32Val == 0x1ad) ||
         (rfu32Val == 0x1b0) ||
         ((rfu32Val >= 0x1b4) && (rfu32Val <= 0x1b6) && ((rfu32Val & 0x1) == 0)) ||
         (rfu32Val == 0x1b9) ||
         (rfu32Val == 0x1bd) ||
         (rfu32Val == 0x1c5) ||
         (rfu32Val == 0x1c8) ||
         (rfu32Val == 0x1cb) ||
         ((rfu32Val >= 0x1ce) && (rfu32Val <= 0x1dc) && ((rfu32Val & 0x1) == 0)) ||
         ((rfu32Val >= 0x1df) && (rfu32Val <= 0x1ef) && ((rfu32Val & 0x1) == 1)) ||
         (rfu32Val == 0x1f2) ||
         (rfu32Val == 0x1f5) ||
         ((rfu32Val >= 0x1f9) && (rfu32Val <= 0x21f) && ((rfu32Val & 0x1) == 1)) ||
         ((rfu32Val >= 0x223) && (rfu32Val <= 0x233) && ((rfu32Val & 0x1) == 1)) ||
         (rfu32Val == 0x23c) ||
         (rfu32Val == 0x242) ||
         ((rfu32Val >= 0x247) && (rfu32Val <= 0x24f) && ((rfu32Val & 0x1) == 1)) )
      {
         rfu32Val -= 1;
      }
      else if( rfu32Val == 0x1bf )
      {
         rfu32Val += 56;
      }
      else if( rfu32Val == 0x195 )
      {
         rfu32Val += 97;
      }
      else if( rfu32Val == 0x19e )
      {
         rfu32Val += 130;
      }
      else if( rfu32Val == 0x19a )
      {
         rfu32Val += 163;
      }
      else if( rfu32Val == 0x180 )
      {
         rfu32Val += 195;
      }
      else if( (rfu32Val >= 0x23f) && (rfu32Val <= 0x240) )
      {
         rfu32Val += 10815;
      }
      else
      {
    	  //Retain same Unicode value for Punctuation
      }
    }
   else
   {
	   //Retain same Unicode value for General Punctuation
   }

}


#undef FC_SXM_FACTOR_NAV_POSITION
#undef FC_SXM_FACTOR_SMS_TO_NAV
