/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_base_thread.cpp
* @brief       It provides the implementation of the worker-thread that is part of each
*              sxm-app class is only used by framework
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_main.h"
#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_base_thread.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_UTIL
#include "trcGenProj/Header/fc_sxm_base_thread.cpp.trc.h"
#endif

//#define FC_SXM_TRIGGER_EVT_MASK 0x1
#define FC_SXM_TIMER_EVT_MASK 0x2
#define FC_SXM_MSG_EVT_MASK 0x4
#define FC_SXM_KILL_EVT_MASK 0x8
#define FC_SXM_USER_EVT_MASK 0xFFFFFF00
/*
  constructor:
  poApp: application that shall dispatch the received messages
  szThreadName: Name that shall be given to the thread
*/
fc_sxm_tclBaseThread::fc_sxm_tclBaseThread(fc_sxm_tclBaseApp *poApp, tCString szThreadName):
    ahl_tclEventThread(fc_sxm_u16GetAppId(), szThreadName),
    _poApp(poApp)
{
    _oAccessSem.vOpen();
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%s)::CTOR", szGetName()));
}



/*
  overwrite method that is called
*/
tBool fc_sxm_tclBaseThread::bThreadFunction(OSAL_tEventMask u32EventMask) {
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::bThreadFunction: u32EventMask=0x%08x name=%s START", u32EventMask, szGetName()));
    _oAccessSem.vGet();
    if (u32EventMask & FC_SXM_KILL_EVT_MASK) {
        ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%s)::bThreadFunction: FC_SXM_KILL_EVT_MASK", szGetName()));
        vClearQ();
        return FALSE;
    }


    if (u32EventMask & FC_SXM_TIMER_EVT_MASK) {
        ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%s)::bThreadFunction: FC_SXM_TIMER_EVT_MASK START", szGetName()));
        // handle time-out-events
        if (OSAL_NULL != _poApp) {
            _poApp->vHandleTimerEvent();
        }
        ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%s)::bThreadFunction: FC_SXM_TIMER_EVT_MASK DONE", szGetName()));
    }


    if (u32EventMask & FC_SXM_MSG_EVT_MASK) {
        ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%s)::bThreadFunction: FC_SXM_MSG_EVT_MASK", szGetName()));
      fc_sxm_tclMessage *poThreadMsg;
      while (_oQ.bPop(&poThreadMsg)) {
          ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%10s)::bThreadFunction: SID=%x action=%d START",
                          szGetName(), 
                          poThreadMsg->enGetServiceId(), poThreadMsg->u16GetActionOnly()));
          if (OSAL_NULL != _poApp) {
              _poApp->vDispatchMsgFromQInternal(poThreadMsg);
          }
          ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%10s)::bThreadFunction: SID=%x action=%d DONE",
                          szGetName(), 
                          poThreadMsg->enGetServiceId(), poThreadMsg->u16GetActionOnly()));

    /* delete allocated memory */
          OSAL_DELETE poThreadMsg;
      }
    }
    if (u32EventMask & FC_SXM_USER_EVT_MASK) {
        tU32 u32UsrEventMask= u32EventMask & FC_SXM_USER_EVT_MASK;
        vHandleUserEvent(u32UsrEventMask);
    }


    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::bThreadFunction: u32EventMask=0x%08x name=%s END", u32EventMask, szGetName()));
    _oAccessSem.vPost();

   return TRUE;
}


tVoid fc_sxm_tclBaseThread::vStart() {
   //Start the worker thread
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread(0x%04x %s)::vStart ",fc_sxm_u16GetAppId(), szGetName()));
    tBool bRes=bStart(0); // 0 means spawning of thread is not awaited. AHL_THREAD_START_WAIT_INFINITE could be used to wait
    SXM_ASSERT_RETURN(TRUE==bRes);
    
    //Success
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread(%s)::started", szGetName()));
}

tVoid fc_sxm_tclBaseThread::vClearQ() {
    fc_sxm_tclMessage *poThreadMsg;
    while (_oQ.bPop(&poThreadMsg)) {
        OSAL_DELETE poThreadMsg;
    }
}

tVoid fc_sxm_tclBaseThread::vStop() {

      // stop the thread. method should return after the thread has really stopped
      tBool bRes=bStop();
      SXM_ASSERT_RETURN(TRUE==bRes);
  
      // Clear the message queue
      vClearQ();
}

/*
  Method to post the message "prMsg"  to the owner of this thread (_poApp)
  note: memory "prMsg" is pointing to has to be allocated by caller
*/
tVoid fc_sxm_tclBaseThread::vPostMsg(fc_sxm_tclMessage *prMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::vPostMsg: SID=%x action=%d name=%s START", 
                    ETG_CENUM(fc_sxm_tenServiceID, prMsg->enGetServiceId()), 
                    prMsg->u16GetActionOnly(), 
                    szGetName()));
    _oQ.push(prMsg, prMsg->enGetMsgPrio());
    vPostEvent(FC_SXM_MSG_EVT_MASK); 
}

tVoid fc_sxm_tclBaseThread::vPostTimerEvent() {
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::vPostTimerEvent START"));
    vPostEvent(FC_SXM_TIMER_EVT_MASK);
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::vPostTimerEvent END"));
}

tVoid fc_sxm_tclBaseThread::vPostUserEvent(tU32 u32Mask) {
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::vPostUserEvent (0x%08x) START", u32Mask));
    u32Mask &=FC_SXM_USER_EVT_MASK;
    vPostEvent(u32Mask);
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::vPostUserEvent END"));
}

tBool fc_sxm_tclBaseThread::bAccess(tU32 u32TimeoutMs) {
    tBool bRes=_oAccessSem.bGet(u32TimeoutMs);
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::bAccess: bRes=%d", bRes));    
    return bRes;
}

tVoid fc_sxm_tclBaseThread::vRelease() {
    ETG_TRACE_USR4(("fc_sxm_tclBaseThread::vRelease"));    
    _oAccessSem.vPost();
}
