/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_diaglib_iocontrollistener.cpp
* @brief       Definition of diagnosis IO control  listener
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/


// +++ Specify what part of diaglib you need in this file +++
#define DIAGLIB_INCLUDE_IO_CONTROL
#include "diaglib_if.h"

#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_diaglibhandler.h"
#include "fc_sxm_diaglib_iocontrollistener.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_DIAGHANDLER 
#include "trcGenProj/Header/fc_sxm_diaglib_iocontrollistener.cpp.trc.h"
#endif


/*************************************************************************
*
* FUNCTION: Constructor
* 
* DESCRIPTION: Init Members and setup connection with diaglib
*
* PARAMETER: NONE
*
* RETURNVALUE: NONE
*
*************************************************************************/
fc_sxm_diagLibIoControlListener::fc_sxm_diagLibIoControlListener() :
m_IoControlIF(OSAL_NULL),
//m_poGWMain(OSAL_NULL),
m_MsgContext(0),
m_bIoControlRunning(FALSE)
{
    ETG_TRACE_USR4(("fc_sxm_diagLibIoControlListener:fc_sxm_diagLibIoControlListener"));
}


/*************************************************************************
*
* FUNCTION:tVoid Destructor
* 
* DESCRIPTION: make sure we are unregistered
*
* PARAMETER:  void
*
* RETURNVALUE: void
*
*************************************************************************/
fc_sxm_diagLibIoControlListener::~fc_sxm_diagLibIoControlListener
()
{
    ETG_TRACE_USR4(("fc_sxm_diagLibIoControlListener:~fc_sxm_diagLibIoControlListener"));
    m_IoControlIF = OSAL_NULL;
    //m_poGWMain = OSAL_NULL;
    m_MsgContext = 0;
    m_bIoControlRunning = FALSE;
}

/*************************************************************************
*
* FUNCTION:tVoid fc_sxm_diagLibIoControlListener::vInit
*(
* diaglib::tclIoControlIF* pIoControlIF,
* )
* 
* DESCRIPTION: //method to initialise the IO control listener
*
* PARAMETER:  diaglib::tclIoControlIF* pIoControlIF,
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid fc_sxm_diagLibIoControlListener::vInit
(
 diaglib::tclIoControlIF* pIoControlIF
 )
{
    ETG_TRACE_USR4(("fc_sxm_diagLibIoControlListener:vInit"));

    m_IoControlIF = pIoControlIF;

    //m_poGWMain = poMain;
}


/*************************************************************************
*
* FUNCTION: tU32 fc_sxm_diagLibIoControlListener::vOnIoControlFreeze
*(
*    tU32 u32IoControlId,
*    diaglib::tContext MsgContext
*    
*)
* 
* DESCRIPTION: Trigger for IO control freeze
*
* PARAMETER: tU32 u32IoControlId //the diag lib identifier for this IO control
*            diaglib::tContext MsgContext
*            
*
* RETURNVALUE: tU32
*
*************************************************************************/
tU32 fc_sxm_diagLibIoControlListener::vOnIoControlFreeze
(
     tU32 u32IoControlId,
     diaglib::tContext MsgContext
 )
{
    (tVoid)u32IoControlId;
    (tVoid)MsgContext;
    tU32 u32Ret=0;

    ETG_TRACE_USR1(("fc_sxm_diagLibIoControlListener:vOnIoControlFreeze : Nothing to be done"));
   
    return u32Ret;
}

/*************************************************************************
*
* FUNCTION: tU32 fc_sxm_diagLibIoControlListener::vOnIoControlFreeze
*(
*    tU32 u32IoControlId,
*    diaglib::tContext MsgContext
*    
*)
* 
* DESCRIPTION: Trigger for IO control unfreeze
*
* PARAMETER: tU32 u32IoControlId //the diag lib identifier for this IO control
*            diaglib::tContext MsgContext
*            
*
* RETURNVALUE: tU32
*
*************************************************************************/
tU32 fc_sxm_diagLibIoControlListener::vOnIoControlUnfreeze
(
    tU32 u32IoControlId,
    diaglib::tContext MsgContext
)
{
    (tVoid)u32IoControlId;
    (tVoid)MsgContext;
    tU32 u32Ret=0;
    ETG_TRACE_USR1(("fc_sxm_diagLibIoControlListener:vOnIoControlUnfreeze : Nothing to be done"));
    return u32Ret;
}


/*************************************************************************
*
* FUNCTION: tU32 fc_sxm_diagLibIoControlListener::vOnIoControl
*(
*    tU32 u32IoControlId,   
*    diaglib::tenIoControlAction enActionId, 
*   const diaglib::tclParameterVector& oParameterVector,
*    diaglib::tContext MsgContext
*)
*
* DESCRIPTION: Trigger for IO control activity
*
* PARAMETER: tU32 u32IoControlId,   //the diag lib identifier for this IO control
*            diaglib::tenIoControlAction enActionId, 
*            const diaglib::tclParameterVector& oParameterVector,
*            diaglib::tContext MsgContext
*
*
* RETURNVALUE: tU32
*
*************************************************************************/
tU32 fc_sxm_diagLibIoControlListener::vOnIoControl
(
 tU32 u32IoControlId, 
 diaglib::tenIoControlAction enActionId, 
 const diaglib::tclParameterVector& oParameterVector,
 diaglib::tContext MsgContext
)
{
    ETG_TRACE_ERR(("fc_sxm_diagLibIoControlListener:vOnIoControl: entered"));
    (tVoid)oParameterVector;
    m_MsgContext = MsgContext;
    tU32 u32RetValue = diaglib::U32_DIAGLIB_RETURN_OK;
    
    switch (u32IoControlId)
    {
        case IOCONTROLID_SXM_AUDIOTEST:
        {
            ETG_TRACE_ERR(("fc_sxm_diagLibIoControlListener:vOnIoControl: Audio test entered"));
            fc_sxm_trMsgAudioDiagAudioTestStart rMsg;
            rMsg.u16ChannelId = oParameterVector[0].u8Value;
            fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);  
            m_bIoControlRunning = TRUE;
        }
        break;

        default:
        {
            NORMAL_M_ASSERT_ALWAYS();
            ETG_TRACE_ERR(("fc_sxm_diagLibIoControlListener:vOnIoControl: u32IoControlID=%u enActionId=%u", u32IoControlId, enActionId));
            vSendResult(m_MsgContext, diaglib::EN_ACTION_SET, diaglib::EN_ERROR_ID_NOT_SUPPORTED);
            u32RetValue = diaglib::U32_DIAGLIB_RETURN_NOT_OK;
        }
    } // switch (u32IoControlId)

   /* if ( u32RetValue == diaglib::U32_DIAGLIB_RETURN_OK )
        vSendResult(m_MsgContext, diaglib::EN_ACTION_SET, diaglib::EN_ERROR_NO_ERROR);*/

    return u32RetValue;
}

/*************************************************************************
*
* FUNCTION: tVoid fc_sxm_diagLibIoControlListener::vProcess
*(
* fc_sxm_trMsgAudioSmsEvtTuneState const *prMsg
* )
*
* DESCRIPTION: Function to handle the tune event state
*
* PARAMETER: fc_sxm_trMsgDiagSmsEvtTuneState const *prMsg //Tune state
*
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid fc_sxm_diagLibIoControlListener::vProcess
(
 fc_sxm_trMsgDiagSmsEvtTuneState const *prMsg
 )
{
    if(FALSE != m_bIoControlRunning) //Process the tune state only when Io control is running
    {
        if (OSAL_NULL != prMsg) 
        {
            if(TUNE_STATE_TUNED == prMsg->eTuneState)
            {
                 ETG_TRACE_USR1(("fc_sxm_diagLibIoControlListener:vProcess(fc_sxm_trMsgAudioSmsEvtTuneState) : success"));
                //send result ::success
                vSendResult(m_MsgContext, diaglib::EN_ACTION_SET, diaglib::EN_ERROR_NO_ERROR);
            }
            else if(TUNE_STATE_TUNING_IN_PROGRESS ==  prMsg->eTuneState)
            {
                ETG_TRACE_USR1(("fc_sxm_diagLibIoControlListener:vProcess(fc_sxm_trMsgAudioSmsEvtTuneState) : Nothing to be done"));
            }
            else
            {
                //send result ::error
                 ETG_TRACE_USR1(("fc_sxm_diagLibIoControlListener:vProcess(fc_sxm_trMsgAudioSmsEvtTuneState) : fail"));
                vSendResult(m_MsgContext, diaglib::EN_ACTION_SET, diaglib::EN_ERROR_FATAL_INTERNAL);
            }
        }        
    }
    else
    {
        ETG_TRACE_USR1(("fc_sxm_diagLibIoControlListene:trMsgAudioSmsEvtTuneState): Audio test not running"));
    }
}


tVoid fc_sxm_diagLibIoControlListener::vProcess(fc_sxm_trMsgDiagDownloadResult const *prMsg ) const {
    (tVoid)prMsg; //not used now resolving prio 2 lint
    // todo: send result to diag-lib
}


/*************************************************************************
*
* FUNCTION: tU32 fc_sxm_diagLibIoControlListener::vSendResult
*(
*   diaglib::tContext MsgContext,
*   diaglib::tenIoControlAction enAction,
*   diaglib::tenInternalError enError
*   
*)
*
* DESCRIPTION: Trigger for IO control activity
*
* PARAMETER: diaglib::tContext MsgContext,
*            diaglib::tenIoControlAction enAction,
*            diaglib::tenInternalError enError
*            
*
*
* RETURNVALUE: tU32
*
*************************************************************************/
tVoid fc_sxm_diagLibIoControlListener::vSendResult
(
 diaglib::tContext MsgContext,
 diaglib::tenIoControlAction enAction,
 diaglib::tenInternalError enError
 ) 
{
    diaglib::tenIoCtrlResult enIoControlResult = diaglib::EN_IO_CONTROL_NOT_OK;
    diaglib::tclParameterVector oResultList;
    if (enError == diaglib::EN_ERROR_NO_ERROR) {
        enIoControlResult = diaglib::EN_IO_CONTROL_OK;
    } 
    else {
        diaglib::trParameter rParameter;
        rParameter.enType=diaglib::EN_PARAMETER_TYPE_U8ERROR;
        rParameter.u8ErrorValue=enError;
        oResultList.push_back(rParameter);
    }
    ETG_TRACE_USR4(("fc_sxm_diagLibIoControlListener:vSendResult: OK=%d, enError=0x%04x",
        (enIoControlResult == diaglib::EN_IO_CONTROL_OK) ? 1 : 0,
        enError));
    // send result to diaglib
    //#if 0
    if(OSAL_NULL != m_IoControlIF)
    {
        if ( !m_IoControlIF->bSendIoCtrlMethodResult( enIoControlResult,
            enAction,
            oResultList,
            MsgContext) )
        {
            ETG_TRACE_USR2(("fc_sxm_diagLibIoControlListener:vSendResult: could not bSendIoControlResult"));
        }
    }
    else
    {
        ETG_TRACE_USR2(("fc_sxm_diagLibIoControlListener:vSendResult: m_IoControlIF in null"));
    }

    m_bIoControlRunning = FALSE;
    /*#else
    (tVoid)enIoControlResult;
    (tVoid)oResultList;
    (tVoid)MsgContext;
    #endif*/
}


