/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_twnowproxy.cpp
* @brief       Implementation of interface between fc_sxm app and SMS library for 
*              TWNow service specific calls.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/
#include "fc_sxm_tcl_twnow_proxy.h"

// This API is used to find out if the list of available REPORT content is complete.
// The REPORTs which are available can change following a Decoder Update.
// Following the update, the REPORT content list will be cleared and the REPORT Content will be incomplete.
// As content is received over the air, REPORTS will be added to the REPORT content list.
tBool fc_sxm_tclTWNowProxy::bIsTWNowReportContentComplete() const
{
	return REPORT.bContentComplete();
}

// Method requests SMS to iterate through the
// currently available list of REPORT content using a provided call-back function.
tBool fc_sxm_tclTWNowProxy::bGetTWNowMarkets (Ifc_sxm_tclTWNow* poTWNow) const
{
	(void) REPORT.bIterateContent(fc_sxm_tclTWNowProxy::cb_bTWNowReportIterator, (void *)poTWNow);
	return (REPORT.bContentComplete());
}

// fc_sxm_tclTWNowProxy::cb_bTWNowReportIterator() is called by SMS for each report market (sorted in alphabetical order of Full Market Names)
// until the available reports are exhausted or the return value from the provided call-back is FALSE.
// This call-back provides the report id, name and CID as well as the caller provided argument.
// The call-back is run each time with exclusive access to the CID being provided so the CID may be manipulated using the CID APIs.
// Returning FALSE from this method would stop iteration.
tBool fc_sxm_tclTWNowProxy::cb_bTWNowReportIterator (const char* pacId,
		const char* pacName,
		CID_OBJECT hId,
		void* pvContentIteratorCbArg)
{
	tBool bContinueIteration = FALSE;
	Ifc_sxm_tclTWNow* poTWNow = static_cast <Ifc_sxm_tclTWNow *> (pvContentIteratorCbArg);
	if (OSAL_NULL != poTWNow)
		bContinueIteration = poTWNow->bAddTWNowMarket(pacId, pacName, hId);
	return bContinueIteration;
}

// Callback reported from SMS for every TWNow Event
tVoid fc_sxm_tclTWNowProxy::cb_TWNowEvent (
		DECODER_OBJECT hDecoder,
		TW_NOW_BULLETIN_ID u32BulletinID,
		TW_NOW_BULLETIN_STATUS_ENUM enBulletinStatus,
		CID_OBJECT marketID,
		tVoid* pvEventCallbackArg )
{
	// pvEventCallbackArg is passed with fc_sxm_tclTWNow object's "this" pointer
	Ifc_sxm_tclTWNow* poTWNow = static_cast< Ifc_sxm_tclTWNow * > (pvEventCallbackArg);
	if (OSAL_NULL != poTWNow)
		poTWNow->vHandleTWNowEvent(hDecoder, u32BulletinID, enBulletinStatus, marketID);
}


// Over-ride TWNow object wrappers

// Start TWNow service
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNowProxy::enStartTWNow(DECODER_OBJECT hDecoder, Ifc_sxm_tclTWNow* poTWNow) const
{
	return (TW_NOW.eStart(hDecoder, fc_sxm_tclTWNowProxy::cb_TWNowEvent,(tVoid*) poTWNow));
}

// Stop T&W Now service and release the object handle.
tVoid fc_sxm_tclTWNowProxy::vStopTWNow(DECODER_OBJECT hDecoder) const
{
	return (TW_NOW.vStop(hDecoder));
}

// Set TWNow Market
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNowProxy::enSetTWNowMarket(DECODER_OBJECT hDecoder, CID_OBJECT hMarketID) const
{
	return (TW_NOW.eSetMarketID(hDecoder, hMarketID));
}

// Get currently set TWNow market's CID object
CID_OBJECT fc_sxm_tclTWNowProxy::hGetCurrentTWNowMarket(DECODER_OBJECT hDecoder) const
{
	return (TW_NOW.hGetMarketID(hDecoder));
}

// Play the user provided TWNow BulletinID
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNowProxy::enPlayTWNowBulletin(DECODER_OBJECT hDecoder, tU32 u32BulletinID) const
{
	return (TW_NOW.ePlayBulletinAudio(hDecoder, u32BulletinID));
}

// Abort Playback of TWNow BulletinID
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclTWNowProxy::enAbortTWNowBulletin(DECODER_OBJECT hDecoder) const
{
	return (TW_NOW.eAbortBulletinPlayback(hDecoder));
}

// Retrieve current available TWNowBulletinID
tU32 fc_sxm_tclTWNowProxy::u32GetCurrentTWNowBulletinID(DECODER_OBJECT hDecoder) const
{
	return (TW_NOW.tCurrentBulletinID(hDecoder));
}

// Over-ride CID object wrappers
// Performs CID.Destroy operation
tVoid fc_sxm_tclTWNowProxy::vDestroyCIDObject(CID_OBJECT hCID) const
{
	return (CID.vDestroy(hCID));
}

// Performs CID.n16Equal operation
tS16 fc_sxm_tclTWNowProxy::s16CompareCIDObjects(CID_OBJECT hCID1, CID_OBJECT hCID2) const
{
	return (CID.n16Equal(hCID1, hCID2));
}

// Method to start timer for polling the availability of TWNow Markets
tVoid fc_sxm_tclTWNowProxy::vStartPollingAvailableMarkets(tU32 u32DelayTime, Ifc_sxm_tclTWNowMarketReport* poAvlMarkets)
{
	_poAvlMarkets = poAvlMarkets;
	_mTWNowMarketsTimer.vStart(u32DelayTime);
}

// Method to stop timer which polls for the availability of TWNow Markets
tVoid fc_sxm_tclTWNowProxy::vStopPollingAvailableMarkets()
{
	if(_mTWNowMarketsTimer.bIsRunning())
		_mTWNowMarketsTimer.vStop();
}

// Handle timer expiry event
tVoid fc_sxm_tclTWNowProxy::vProcessTimer(fc_sxm_trMsgTWNowMarketsTimer const * /* prMsg */ )
{
	vStopPollingAvailableMarkets();
	if (_poAvlMarkets)
		_poAvlMarkets->vGetAvailableMarkets();
}

