/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_states.cpp
* @brief       State machine implementation for SMS, SRM, Module, Decoder and Audio states.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sxmapp_manager.h"

#include "fc_sxm_diaglibhandler.h"
#include "fc_sxm_tcl_advisories.h"

#include "fc_sxm_diag_if.h"
#include "fc_sxm_tcl_states.h"
#include "fc_sxm_tcl_channel_list.h"
#include "fc_sxm_tcl_category_list.h"
#include "fc_sxm_audiorouting_lib.h"
#include "fc_sxm_tcl_sms_init.h"

#include "fc_sxm_tcl_playback.h"
#include "fc_sxm_tcl_presets.h"
#include "fc_sxm_tcl_content_alerts.h"
#include "fc_sxm_service_sxm_audio.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_STATES
#include "trcGenProj/Header/fc_sxm_tcl_states.cpp.trc.h"
#endif

#define BUFF_SIZE 256

int SMSLIB_bCvOn=0;
int fc_sxm_bInStartupPhase=FALSE;

tVoid fc_sxm_vTraceRetry(fc_sxm_tenSmType enSmType, char const *szText, SMSAPI_RETURN_CODE_ENUM eReturnCode, int iRetry) {
    ETG_TRACE_USR1(("%d::%20s res=%d iRetry=%d",
                    ETG_CENUM(fc_sxm_tenSmType, enSmType),
                    szText,
                    ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode),
                    iRetry));    
}

const tBool fc_sxm_tclSystemStates::_ActivateMatrix[en_fc_sxm_tclSmTypeMax][fc_sxm_enSystemState_INVALID]= {
    /*             Error Defset OFF  INIT DL FREEZE NORM */
    /* SMS */    { 0,    0,      0,   1,   1, 1,     1},
    /* SRM */    { 0,    0,      0,   1,   1, 1,     1},
    /* MODULE */ { 0,    0,      0,   1,   1, 1,     1},
    /* DECODER */{ 0,    0,      0,   1,   0, 1,     1},
    /* DATA */   { 0,    0,      0,   0,   0, 1,     1},
    /* AUDIO */  { 0,    0,      0,   0,   0, 1,     1}
};

// utilities
tBool fc_sxm_bChangeMask(fc_sxm_tenMaskMode enMode, tU32 u32Mask, tU32 &ru32InOutMask) {
    tU32 u32OldMask = ru32InOutMask;
    switch (enMode) {
        case fc_sxm_enMaskMode_Set:
            ru32InOutMask = u32Mask;
            break;
        case fc_sxm_enMaskMode_Add:
            ru32InOutMask |= u32Mask;
            break;
        case fc_sxm_enMaskMode_Remove:
            ru32InOutMask &= ~u32Mask;
            break;
        default:
            break;
    }
    return u32OldMask != ru32InOutMask;
}

//to get systemstate
fc_sxm_tenSystemState fc_sxm_enGetSystemState() {
    return fc_sxm_tclSystemStates::instance()->enGetSystemState();
}

tBool fc_sxm_bIsCvOn() {
   if(fc_sxm_enSystemState_OFF == fc_sxm_tclSystemStates::instance()->enGetSystemState()) {
      return(FALSE);
   } else {
      return fc_sxm_tclSystemStates::instance()->bIsCvOn();
   }
}
extern "C"
{
unsigned char fc_sxm_bIsSMSLIB_CvOn() {
   #if((FC_SXM_CVON_ENABLE) || (FC_SXM_SMSLIB_CVON_ENABLE))
		return (tU8)SMSLIB_bCvOn;
#else
		return FALSE;
#endif
}
}

unsigned char fc_sxm_tclSystemStates::bIsSMSLIBCvOn() {
   #if((FC_SXM_CVON_ENABLE) || (FC_SXM_SMSLIB_CVON_ENABLE))
		return (tU8)SMSLIB_bCvOn;
#else
		return FALSE;
#endif
}
// base-class for states
fc_sxm_tclSmsSm::fc_sxm_tclSmsSm():
    _enState(fc_sxm_enSmsObjectState_STOPPED),
    _poParent(OSAL_NULL),
    _bParentUp(FALSE),
    _bChildrenActive(FALSE),
    _bWantsActivate(FALSE)
{}

tVoid fc_sxm_tclSmsSm::vTraceStart(fc_sxm_tenSmEvent enEvent) const {
    ETG_TRACE_USR2(("fc_sxm_tclSmsSm(%d)::vTraceStart() enEvent=%d oldState=%d bWantsActivate=%d _bChildrenActive=%d _bParentUp=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmEvent, enEvent),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState),
                    _bWantsActivate,
                    _bChildrenActive,
                    _bParentUp));
}
    
tVoid fc_sxm_tclSmsSm::vTraceEnd(fc_sxm_tenSmEvent enEvent) const {
    ETG_TRACE_USR2(("fc_sxm_tclSmsSm(%d)::vTraceEnd() enEvent=%d newState=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmEvent, enEvent),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vUpdateState(fc_sxm_tenSmsObjectState enState) {
    ETG_TRACE_USR2(("fc_sxm_tclSmsSm(%d)::vUpdateState() %d->%d",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, enState)));
    //update state
    if (_enState != enState) {
        vUpdateState_(enState);
        //invase of error state handle the error
        if (enState==fc_sxm_enSmsObjectState_ERROR) {
            ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vUpdateState: call vHandleError",
                            ETG_CENUM(fc_sxm_tenSmType, enGetType())));
            fc_sxm_tclSystemStates::instance()->vHandleError(this);
        }
    }
}

tVoid fc_sxm_tclSmsSm::vPostEvent(fc_sxm_tenSmEvent enEvent) {
    ETG_TRACE_USR2(("fc_sxm_tclSmsSm(%d)::vPostEvent(%d)",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmEvent, enEvent)));

    fc_sxm_trSmEvent rEvent;
    rEvent.poToSm=this;
    rEvent.poArgSm=OSAL_NULL;
    rEvent.enEvent=enEvent;
    fc_sxm_trMsgAudioSmEvent rMsg(rEvent);
    fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
}

tVoid fc_sxm_tclSmsSm::vHandleEvent(fc_sxm_tenSmEvent enEvent) {
    vTraceStart(enEvent);
    //based on event handle it
    switch (enEvent) {
        case fc_sxm_enSmEvent_SmsInitial:
            vHandleSmsInitial();
            break;
        case fc_sxm_enSmEvent_SmsError:
            vHandleSmsError();
            break;
        case fc_sxm_enSmEvent_SmsStopped:
            vHandleSmsStopped();
            break;
        case fc_sxm_enSmEvent_SmsReady:
            vHandleSmsReady();
            break;
        case fc_sxm_enSmEvent_SmsUpdating:
            vHandleSmsUpdating();
            break;
        case fc_sxm_enSmEvent_ChildrenUp:
            if (!_bChildrenActive) {
                _bChildrenActive = TRUE;
                vHandleChildrenUp();
            }
            break;
        case fc_sxm_enSmEvent_ChildrenDown:
            if (_bChildrenActive) {
                _bChildrenActive = FALSE;
                vHandleChildrenDown();
            }
            break;
        case fc_sxm_enSmEvent_ParentUp:
            if (!_bParentUp) {
                _bParentUp=TRUE;
                vHandleParentUp();
            }
            break;
        case fc_sxm_enSmEvent_ParentDown:
            if (_bParentUp) {
                _bParentUp=FALSE;
                vHandleParentDown();
            }
            break;
        case fc_sxm_enSmEvent_Activate:
            if (!_bWantsActivate) {
                _bWantsActivate=TRUE;
                vHandleActivate();
            }
            break;
        case fc_sxm_enSmEvent_DeActivate:
            if (_bWantsActivate || (fc_sxm_tclSystemStates::instance()->bIsInErrorRecovery() &&  (_enState==fc_sxm_enSmsObjectState_ERROR || _enState==fc_sxm_enSmsObjectState_READY))) {
                _bWantsActivate=FALSE;
                vHandleDeActivate();
            }
            break;
        default:
            break;
    }
    vTraceEnd(enEvent);
}

tVoid fc_sxm_tclSmsSm::vEnterError() {
    if (fc_sxm_tclSystemStates::instance()->bIsInErrorRecovery() && !_bChildrenActive) {
        if (bZombie()) {
            ETG_TRACE_ERR(("fc_sxm_tclSmsSm(%d)::vEnterError: Zombie",
                             ETG_CENUM(fc_sxm_tenSmType, enGetType())));
            vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
        }
        else {
            ETG_TRACE_ERR(("fc_sxm_tclSmsSm(%d)::vEnterError: call vStopSmsObject",
                             ETG_CENUM(fc_sxm_tenSmType, enGetType())));
            vHandleDeActivate();
        }
    }
}


tBool fc_sxm_tclSmsSm::bStable() {
    tBool bRes=(_bWantsActivate ? _enState==fc_sxm_enSmsObjectState_READY : _enState==fc_sxm_enSmsObjectState_STOPPED);
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::bStable(bWantsActivate=%d, _enState=%d): %d",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    _bWantsActivate,
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState),
                    bRes));
    return bRes;
}

tVoid fc_sxm_tclSmsSm::vRegisterChild(fc_sxm_tclSmsSm *poChild) {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vRegisterChild(%d)",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmType, poChild->enGetType())));
    _lChildren.push_back(poChild);
}

tVoid fc_sxm_tclSmsSm::vSetParent(fc_sxm_tclSmsSm *poParent) {
    _poParent=poParent;
    if (OSAL_NULL != _poParent) {
        ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vSetParent(%d)",
                        ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                        ETG_CENUM(fc_sxm_tenSmType, poParent->enGetType())));
        _poParent->vRegisterChild(this);
    }
}

tVoid fc_sxm_tclSmsSm::vSetChildActive(fc_sxm_tclSmsSm *poChild, tBool bActive) {
    tBool bChildrenActive=FALSE;
    SXM_FOREACH(deque<fc_sxm_tclSmsSm *>, iter, _lChildren) {
        if ((*iter)->_enState != fc_sxm_enSmsObjectState_STOPPED) {
            bChildrenActive=TRUE;
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vSetChildActive(%d):bActive=%d --> bChildrenActive=%d",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmType, poChild->enGetType()),
                    bActive,
                    bChildrenActive));
    vPostEvent(bChildrenActive ? fc_sxm_enSmEvent_ChildrenUp : fc_sxm_enSmEvent_ChildrenDown);
}

tVoid fc_sxm_tclSmsSm::vSetParentUp(tBool bUp){
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vSetParentUp() bUp: %d-->%d",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    _bParentUp, bUp));
    vPostEvent(bUp ? fc_sxm_enSmEvent_ParentUp : fc_sxm_enSmEvent_ParentDown);
};

tVoid fc_sxm_tclSmsSm::vNotifyChildren(tBool bUp) {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vNotifyChildren() bUp: %d",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    bUp));
    SXM_FOREACH(deque<fc_sxm_tclSmsSm *>, iter,_lChildren) {
        (*iter)->vSetParentUp(bUp);
    }
}

tVoid fc_sxm_tclSmsSm::vNotifyParent(tBool bActive) {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d)::vNotifyParent() bActive: %d",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    bActive));
    if (OSAL_NULL != _poParent) {
        _poParent->vSetChildActive(this, bActive);
    }
}

// SRM
static tVoid cb_vSRMEventCallback(SRM_OBJECT hSRM, 
                                  SRM_EVENT_MASK tEventMask, 
                                  tVoid * /* pvEventCallbackArg */)
{
    ETG_TRACE_USR3(("fc_sxm_tclAudioApp::cb_vSRMEventCallback tEventMask=%x", tEventMask));
    if(tEventMask & SRM_OBJECT_EVENT_STATE)
    {
        fc_sxm_trMsgAudioSmsEvtSrmState rMsg;
        rMsg.eState = SRM.eState(hSRM);
        ETG_TRACE_USR1(("cb_vSRMEventCallback: State: %d", 
                        ETG_CENUM(SRM_STATE_ENUM, rMsg.eState)));
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }
}

// SRM state-machine
tVoid fc_sxm_tclSmsSrm::vProcess(fc_sxm_trMsgAudioSmsEvtSrmState const *prMsg) {

    ETG_TRACE_USR2(("fc_sxm_tclSmsSrm::vProcess(fc_sxm_trMsgAudioSmsEvtSrmState) state = %d", 
                    ETG_CENUM(SRM_STATE_ENUM, prMsg->eState)));
    if (prMsg->eState == SRM_STATE_ERROR && _hSRM != SRM_INVALID_OBJECT) {
        SRM_ERROR_CODE_ENUM enError = SRM.eErrorCode (_hSRM);
        ETG_TRACE_USR2(("fc_sxm_tclSmsSrm, error = %d", 
                        ETG_CENUM(SRM_ERROR_CODE_ENUM, enError)));
    }
    if(_enSmsState==prMsg->eState) {
        return;
    }
    _enSmsState=prMsg->eState;
    fc_sxm_tenSmEvent enEvent = fc_sxm_enSmEvent_SmsError;
    switch (_enSmsState) {
        case SRM_STATE_STOPPED:
            enEvent=fc_sxm_enSmEvent_SmsStopped;
            break;
        case SRM_STATE_INITIAL:
            enEvent=fc_sxm_enSmEvent_SmsInitial;
            break;
        case SRM_STATE_READY:
            enEvent=fc_sxm_enSmEvent_SmsReady;
            break;
        default:
            enEvent=fc_sxm_enSmEvent_SmsError;
            break;
    }
    vPostEvent(enEvent);
}

tVoid fc_sxm_tclSmsSm::vHandleSmsInitial() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsInitial state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_STOPPING:
            break;
        case fc_sxm_enSmsObjectState_UPDATING:
            // firmware-download is over, module should go automatically to initial
            vUpdateState(fc_sxm_enSmsObjectState_STARTING);
            break;
        default:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsInitial state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSrm::vHandleSmsInitial() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSrm:: vHandleSmsInitial state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_STOPPING:
            break;

        case fc_sxm_enSmsObjectState_READY:
            fc_sxm_tclSystemStates::instance()->vStartDownloadRecovery();
            if (fc_sxm_tclSystemStates::instance()->bIsDownloadRecovery()) {
                // firmware-download is over, srm should go automatically to initial,
                // we just ignore this state
                /*Have to consider this state for Firmware download as the SMS now
                  cannot restart the module*/
                 fc_sxm_tclSMSInit::instance()->vStopX65Module();
                 fc_sxm_tclSMSInit::instance()->vStartX65Module();
            }
            break;
        default:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSrm:: vHandleSmsInitial state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsModule::vHandleSmsInitial() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsModule:: vHandleSmsInitial state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));

    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_STOPPING:
            break;

        case fc_sxm_enSmsObjectState_UPDATING:
            fc_sxm_tclSystemStates::instance()->vStartDownloadRecovery();
            vUpdateState(fc_sxm_enSmsObjectState_STARTING);
            break;
        default:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsModule:: vHandleSmsInitial state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleSmsError() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsError state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));

    if( (_enState != fc_sxm_enSmsObjectState_STOPPED) && (_enState != fc_sxm_enSmsObjectState_STOPPING) ) {
        vUpdateState(fc_sxm_enSmsObjectState_ERROR);
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsError state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleSmsStopped() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsStopped state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_READY:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
        case fc_sxm_enSmsObjectState_UPDATING:
            // firmware-download is over, module should go automatically to initial
            vUpdateState(fc_sxm_enSmsObjectState_STARTING);
            break;
        case fc_sxm_enSmsObjectState_STOPPING:
            vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsStopped state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleSmsUpdating() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsUpdating state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));

    vUpdateState(fc_sxm_enSmsObjectState_ERROR);

    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsUpdating state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleSmsReady() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsReady state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
            if (fc_sxm_bIsCvOn()) {
                vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            }
            else {
                vUpdateState(fc_sxm_enSmsObjectState_READY);
            }
            break;
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_STOPPING:
            break;
        default:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleSmsReady state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleChildrenUp() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleChildrenUp state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    return;
}

tVoid fc_sxm_tclSmsSm::vHandleChildrenDown() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleChildrenDown state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    switch (_enState) {
        case fc_sxm_enSmsObjectState_READY:
            if (!_bWantsActivate && !fc_sxm_bIsCvOn()) {
                vUpdateState(fc_sxm_enSmsObjectState_STOPPING);
            }
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            if (!_bWantsActivate && fc_sxm_tclSystemStates::instance()->bIsInErrorRecovery()) {
                if (bZombie()) {
                    vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
                } else {
                    vUpdateState(fc_sxm_enSmsObjectState_STOPPING);
                }
            }
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleChildrenDown state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleParentUp() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleParentUp state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    vLinkSmsParent();
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STOPPED:
            if (_bWantsActivate && !fc_sxm_bIsCvOn()) {
                vUpdateState(fc_sxm_enSmsObjectState_STARTING);
                if (FALSE == bStartSmsObject()) {
                    vUpdateState(fc_sxm_enSmsObjectState_ERROR);
                }
            }
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleParentUp state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleParentDown() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleParentDown state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));

    vLinkSmsParent();

    switch (_enState) {
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_STARTING:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleParentDown state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSms::vHandleActivate() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSms::vHandleActivate state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));

    _bParentUp=TRUE;
    fc_sxm_tclSmsSm::vHandleActivate();
    ETG_TRACE_USR4(("fc_sxm_tclSmsSms::vHandleActivate state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleActivate() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleActivate state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STOPPED:
            if (_bParentUp) {
                vUpdateState(fc_sxm_enSmsObjectState_STARTING);
                if (FALSE == bStartSmsObject()) {
                    vUpdateState(fc_sxm_enSmsObjectState_ERROR);
                }
            }
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleActivate state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

tVoid fc_sxm_tclSmsSm::vHandleDeActivate() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleDeActivate state=%d START",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));

    switch (_enState) {
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_ERROR:
            if (!_bChildrenActive) {
                if (bZombie()) {
                    vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
                } else {
                    vUpdateState(fc_sxm_enSmsObjectState_STOPPING);
                }
            }
            break;

        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsSm(%d):: vHandleDeActivate state=%d END",
                    ETG_CENUM(fc_sxm_tenSmType, enGetType()),
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}




tBool fc_sxm_tclSmsSms::bStartSmsObject() {
    // Start SMS
    tBool bOk=TRUE;
    BOOLEAN lreplace;
    fc_sxm_arl_tclISource::instance()->vSetMute(TRUE);
    _enSmsState=fc_sxm_enSmsState_READY;

    fc_sxm_tclSMSInit::instance()->bWaitSmsDown(5000);


    if (fc_sxm_tclAudioApp::instance()->bIsSMSCfgError()) {
        ETG_TRACE_ERR(("fc_sxm_tclSmsSms::bStartSmsObject : Deleting sms.cfg"));
        fc_sxm_tclSystemStates::instance()->vDeleteSmsCfg();
        fc_sxm_tclAudioApp::instance()->vSetSMSCfgError(FALSE);
    }

    ETG_TRACE_USR1(("fc_sxm_tclSmsSms::bStartSmsObject"));
    fc_sxm_tclSMSInit::instance()->vStartX65Module();

    const tU8 fileNameLength =64;
    char fileName[fileNameLength];
    if(fc_sxm_tclConfig::instance()->bGetPresetType() == 1)
    {
        strncpy(fileName,"sms_mixed_preset.cfg", strlen("sms_mixed_preset.cfg") + 1);
    }
    else
    {
        strncpy(fileName,"sms.cfg", strlen("sms.cfg") + 1);
    }

    FC_SXM_STATES_SM_RETRY("bStartSmsObject!", FC_SXM_STATES_MAX_SMS_START_RETRY, SMS.eInitialize(FC_SXM_DEFAULT_SMS_CFG_PATH, FC_SXM_STATIC_SMS_CFG_PATH, fileName, &lreplace));

    ETG_TRACE_USR4(("File Restored = %d", lreplace));

    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) {
        ETG_TRACE_ERR(("Error! Cannot initialize SMS"));
        fc_sxm_tclDiagDefset::instance()->vHandleSmsInitError(eReturnCode);
        bOk = FALSE;
        _enSmsState=fc_sxm_enSmsState_ERROR;
    }

    else if(SMSAPI_RETURN_CODE_SUCCESS != SMS.eBehavior(SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE, TRUE)) {
        ETG_TRACE_ERR(("Failed to override SMS behaviour to tune mature channel at start up"));
    }


    fc_sxm_tenSmEvent enEvent = bOk ? fc_sxm_enSmEvent_SmsReady : fc_sxm_enSmEvent_SmsError;
    vPostEvent(enEvent); //post the event
    return  bOk;
}


// Method resets the sxm module to factory default state
// and restarts all the services
tVoid fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioMStartResetSxmModule const *prMsg )
 {
   ETG_TRACE_USR1(("fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioMStartResetSxmModule)Start"));
   // Set the SXM Initializing advisory
   fc_sxm_tclAdvisories::instance()->vSetSXMInitializingAdvisory();

	if (SMSAPI_RETURN_CODE_SUCCESS
			!= SMS.eBehavior(SMS_BEHAVIOR_RESTORE_FACTORY_DEFAULTS, TRUE)) {
		ETG_TRACE_ERR(("SMS.eBehavior(FactoryDefault) Failed"));
	} else {
		ETG_TRACE_USR1(("SMS.eBehavior(FactoryDefault) Success"));
	}

   //Enter into error mode and restart SMS objects
   _enErrorRecoveryMode = enErrorRecoveryMode_Active;
   vUpdate(TRUE);

   ETG_TRACE_USR1(("fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioMStartResetSxmModule)::vUpdate() complete"));
   midw_ext_sxm_audiofi_tclMsgResetSXMModuleMethodResult oMRes;
   oMRes.Status = TRUE;
   fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing,oMRes);
   ETG_TRACE_USR1(("fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioMStartResetSxmModule)::END "));
}

tBool fc_sxm_tclSmsSrm::bStartSmsObject() {
    ETG_TRACE_USR1(("fc_sxm_tclSmsSrm::bStartSmsObject"));
    _hSRM = SRM.hGet("srh:", "srm:", SRM_OBJECT_EVENT_ALL,
                     cb_vSRMEventCallback, NULL);
    tBool bOk=_hSRM != SRM_INVALID_OBJECT;
    ETG_TRACE_USR1(("fc_sxm_tclSmsSrm::bStartSmsObject bOk=%d _hSRM=0x%08x",
                    bOk, _hSRM));
    return  bOk;
}

//constructor
fc_sxm_tclSmsSms::fc_sxm_tclSmsSms():
    _enSmsState(fc_sxm_enSmsState_STOPPED)
{ }

//constructor
fc_sxm_tclSmsSrm::fc_sxm_tclSmsSrm():
    _hSRM(SRM_INVALID_OBJECT),
    _enSmsState(SRM_STATE_STOPPED)
{
    vSetParent(fc_sxm_tclSmsSms::instance());
}

tVoid fc_sxm_tclSmsSms::vStopSmsObject() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSms::vStopSmsObject"));
    if (_enSmsState != fc_sxm_enSmsState_STOPPED) {
        FC_SXM_STATES_SM_RETRY("vStopSmsObject!", 10, SMS.eUninitialize());
        _enSmsState = fc_sxm_enSmsState_STOPPED;
        if (!fc_sxm_tclSystemStates::instance()->bIsInErrorRecovery()) {
            // in error-recovery the module has already been restarted to speed up.
            fc_sxm_tclSMSInit::instance()->vStopX65Module();
        }
        fc_sxm_tenSmEvent enEvent = fc_sxm_enSmEvent_SmsStopped;
        vPostEvent(enEvent);
   }
}

tVoid fc_sxm_tclSmsSrm::vStopSmsObject() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsSrm::vStopSmsObject?"));
    //SRM.vRelease
    if (_hSRM != SRM_INVALID_OBJECT) {
        ETG_TRACE_USR1(("fc_sxm_tclSmsSrm::vStopSmsObject!"));
        SRM.vRelease(_hSRM);
        _hSRM=SRM_INVALID_OBJECT;
    }
}


tVoid fc_sxm_tclSmsSms::vUpdateState_(fc_sxm_tenSmsObjectState enState) {
    _enState = enState;
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STOPPED:
        case fc_sxm_enSmsObjectState_STOPPING:
            vNotifyChildren(FALSE);
            vStopSmsObject();
            break;
        case fc_sxm_enSmsObjectState_STARTING:
            vNotifyChildren(FALSE);
            break;
        case fc_sxm_enSmsObjectState_READY:
              vNotifyChildren(TRUE);
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            vEnterError();
            break;
        default:
            break;
    }
}

tVoid fc_sxm_tclSmsSrm::vUpdateState_(fc_sxm_tenSmsObjectState enState) {
    _enState = enState;
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STOPPED:
            vNotifyChildren(FALSE);
            vStopSmsObject();
            // give srm time to be really destroyed
            OSAL_s32ThreadWait(100);
            vNotifyParent(TRUE);
            break;
        case fc_sxm_enSmsObjectState_STOPPING:
            vNotifyChildren(FALSE);
            vStopSmsObject();
            break;
        case fc_sxm_enSmsObjectState_STARTING:
            vNotifyChildren(FALSE);
            vNotifyParent(TRUE);
            break;
        case fc_sxm_enSmsObjectState_READY:
            if (!fc_sxm_tclSystemStates::instance()->bIsDownloadRecovery()) {
                /* after download srm automatically restarts and goes to ready.
                   We hide this from tclSmsModule
                */
                vNotifyChildren(TRUE);
            }
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            vEnterError();
            break;
        default:
            break;
    }
}


// MODULE
tVoid cb_vModuleEventCallback(MODULE_OBJECT hModule, 
                              MODULE_EVENT_MASK tEventMask, 
                              tVoid * /* pvEventCallbackArg */)
{
    if(tEventMask & MODULE_OBJECT_EVENT_STATE)
    {
        fc_sxm_trMsgAudioSmsEvtModuleState rMsg;
        rMsg.eState=MODULE.eState(hModule);
        rMsg.hModule=hModule;
        fc_sxm_tclSmsModule::instance()->_hLastCallBackModule= (rMsg.eState== MODULE_STATE_STOPPED) ? MODULE_INVALID_OBJECT :hModule;
        ETG_TRACE_USR1(("cb_vModuleEventCallback State: %d", ETG_CENUM(MODULE_STATE_ENUM, rMsg.eState)));
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }

    if ((tEventMask & MODULE_OBJECT_EVENT_SUBSTATUS) && !fc_sxm_bIsCvOn()) {
        ETG_TRACE_USR1(("cb_vModuleEventCallback  MODULE_OBJECT_EVENT_SUBSTATUS"));
        // todo: let diag only hanle this if the module is not stopped
        fc_sxm_trMsgEvtAudioSmsModuleSubStatusChanged rMsg;
        rMsg.hModule=hModule;
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }

    if (tEventMask & MODULE_OBJECT_EVENT_UPDATE_PROGRESS) {
        fc_sxm_trMsgDiagModuleUpdateEvent rMsg;
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }

}

// MODULE State machine
tVoid fc_sxm_tclSmsModule::vProcess(fc_sxm_trMsgAudioSmsEvtModuleState const *prMsg) {
    ETG_TRACE_USR1(("fc_sxm_tclSmsModule::fc_sxm_trMsgAudioSmsEvtModuleState state = %d", 
                    ETG_CENUM(MODULE_STATE_ENUM, prMsg->eState)));
    if (prMsg->eState == MODULE_STATE_ERROR && _hModule != MODULE_INVALID_OBJECT) {
        MODULE_ERROR_CODE_ENUM enError = MODULE.eErrorCode (_hModule);
        ETG_TRACE_USR1(("fc_sxm_tclSmsModule, error = %d", 
                        ETG_CENUM(MODULE_ERROR_CODE_ENUM, enError)));
    }
    vSendHardwareError();
    if(_enSmsState==prMsg->eState) {
        return;
    }
    _enSmsState=prMsg->eState;
    fc_sxm_tenSmEvent enEvent = fc_sxm_enSmEvent_SmsError;
    switch (_enSmsState) {
        case MODULE_STATE_STOPPED:
            enEvent=fc_sxm_enSmEvent_SmsStopped;
            break;
        case MODULE_STATE_INITIAL:
            enEvent=fc_sxm_enSmEvent_SmsInitial;
            break;
        case MODULE_STATE_READY:
            enEvent=fc_sxm_enSmEvent_SmsReady;
            break;
        case MODULE_STATE_UPDATING:
            enEvent=fc_sxm_enSmEvent_SmsUpdating;
            break;
        case MODULE_STATE_ERROR:
        {
        	fc_sxm_tclAdvisories::instance()->vSetSXMInitializingAdvisory();

            if (((_enState==fc_sxm_enSmsObjectState_STOPPED) || (_enState==fc_sxm_enSmsObjectState_STOPPING)) && 
               _hLastCallBackModule != MODULE_INVALID_OBJECT) {
                // strange behavior of sms-module, sometime the stop-request gets lost.
                ETG_TRACE_ERR(("Module state error in state stopped"));
                _hModule=_hLastCallBackModule;
                vStopSmsObject();
            }
            break;
        }
        default:
            enEvent=fc_sxm_enSmEvent_SmsError;
            break;
    }
    vPostEvent(enEvent);
}


tVoid fc_sxm_tclSmsModule::vLinkSmsParent() {
    _hSRM=_bParentUp ? fc_sxm_tclSmsSrm::instance()->hGetSmsSrm() : SRM_INVALID_OBJECT;
    ETG_TRACE_USR4(("fc_sxm_tclSmsModule::vLinkSmsParent _bParentUp=%d _hSRM=0x%08x",
                    _bParentUp, _hSRM));
}

tVoid fc_sxm_tclSmsModule::vChangeMask(fc_sxm_tenMaskMode enMode, tU32 u32Mask) {
    if (fc_sxm_bChangeMask(enMode, u32Mask, _u32Mask)) {
        if (_enState==fc_sxm_enSmsObjectState_READY) {
            MODULE.eModifyRegisteredEventMask(_hModule, (MODULE_EVENT_MASK)_u32Mask, SMSAPI_MODIFY_EVENT_MASK_REPLACE);
        }
    }
}

tBool fc_sxm_tclSmsModule::bStartSmsObject() {
    ETG_TRACE_USR1(("fc_sxm_tclSmsModule::bStartSmsObject"));

    _hModule = MODULE.hGet(_hSRM, "Audio", (MODULE_EVENT_MASK)_u32Mask,
                           cb_vModuleEventCallback, NULL);
    tBool bOk=_hModule != MODULE_INVALID_OBJECT;
    ETG_TRACE_USR1(("fc_sxm_tclSmsModule::bStartSmsObject bOk=%d _hModule=0x%08x _hSRM=0x%08x",
                    bOk, _hModule, _hSRM));
    return  bOk;
}

//Constructor
fc_sxm_tclSmsModule::fc_sxm_tclSmsModule():
    _hModule(MODULE_INVALID_OBJECT),
    _hLastCallBackModule(MODULE_INVALID_OBJECT),
    _enSmsState(MODULE_STATE_STOPPED),
    _u32Mask(MODULE_OBJECT_EVENT_ALL  ^ MODULE_OBJECT_EVENT_UPDATE_PROGRESS),
    _hSRM(SRM_INVALID_OBJECT)
{
    vSetParent(fc_sxm_tclSmsSrm::instance());
}

tVoid fc_sxm_tclSmsModule::vStopSmsObject() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsModule::vStopSmsObject?"));
    //MODULE.vRelease
    if (_hModule != MODULE_INVALID_OBJECT) {
        ETG_TRACE_USR1(("fc_sxm_tclSmsModule::vStopSmsObject!"));
        MODULE.vRelease(_hModule);
       _hModule=MODULE_INVALID_OBJECT;
       _hLastCallBackModule=MODULE_INVALID_OBJECT;
    }
}

tVoid fc_sxm_tclSmsModule::vSendHardwareError() {
    // todo: put this to advisories class
    if (_enSmsState == MODULE_STATE_READY) {
        fc_sxm_tclAdvisories::instance()->vUpdateHardwareStatus(MODULE_STATE_READY);
        
    } 
    else if (_enSmsState == MODULE_STATE_ERROR) {
        /* In case of error state, get the error code */
        MODULE_ERROR_CODE_ENUM hErrorCode = MODULE.eErrorCode(_hModule);
        /* If related to hardware, then notify registered clients */
        if( (MODULE_ERROR_CODE_NO_CAPABLE_DECODER == hErrorCode) ||
            (MODULE_ERROR_CODE_DECODER_LIST_ERROR == hErrorCode) ||
            (MODULE_ERROR_CODE_RADIO_COMM_FAILURE == hErrorCode) ||
            (MODULE_ERROR_CODE_RADIO_DEVICE_ERROR == hErrorCode) ||
            (MODULE_ERROR_CODE_SRM_ERROR == hErrorCode) )
        {
            fc_sxm_tclAdvisories::instance()->vUpdateHardwareStatus(MODULE_STATE_ERROR);
        }                
    }
}

tVoid fc_sxm_tclSmsModule::vUpdateState_(fc_sxm_tenSmsObjectState enState) {
    _enState = enState;
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STOPPING:
            vNotifyChildren(FALSE);
            vStopSmsObject();
            break;
        case fc_sxm_enSmsObjectState_STOPPED:
            vStopSmsObject();
            // give module time to be really destroyed
            OSAL_s32ThreadWait(200);
            vNotifyChildren(FALSE);
            vNotifyParent(FALSE);
            break;
        case fc_sxm_enSmsObjectState_STARTING:
            vNotifyParent(TRUE);
            break;

        case fc_sxm_enSmsObjectState_UPDATING:
            vNotifyChildren(FALSE);
            break;

        case fc_sxm_enSmsObjectState_READY:
            if (fc_sxm_tclSystemStates::instance()->bIsDownloadRecovery()) {
                fc_sxm_tclSystemStates::instance()->vEndDownloadRecovery();
            }
            vNotifyChildren(TRUE);
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            vEnterError();
            break;
        default:
            break;
    }
}

tVoid fc_sxm_tclSmsModule::vHandleSmsUpdating() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsModule:: vHandleSmsUpdating state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    switch (_enState) {
        case fc_sxm_enSmsObjectState_READY:
            vUpdateState(fc_sxm_enSmsObjectState_UPDATING);
            break;
        default:
            vUpdateState(fc_sxm_enSmsObjectState_ERROR);
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclSmsModule:: vHandleSmsUpdating state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}


// DECODER
static tVoid cb_vDecoderEventCallback(DECODER_OBJECT hDecoder, 
                                      DECODER_EVENT_MASK tEventMask, 
                                      CHANNEL_OBJECT hChannel,
                                      tVoid * /* pvEventCallbackArg */ )
{
    ETG_TRACE_USR4(("fc_sxm_tclAudioApp::cb_vDecoderEventCallback tEventMask=%x", tEventMask));

    if(tEventMask & DECODER_OBJECT_EVENT_STATE)
    {
        DECODER_STATE_ENUM eState = DECODER.eState(hDecoder);
        ETG_TRACE_USR1(("cb_vDecoderEventCallback State: %d", ETG_CENUM(DECODER_STATE_ENUM, eState)));

        // Set Decoder State to Error
        if (eState == DECODER_STATE_ERROR) {
        	fc_sxm_tclSystemStates::instance()->vSetSMSDecoderError(TRUE);
        }
        else if (eState == DECODER_STATE_READY) {
        	fc_sxm_tclSystemStates::instance()->vSetSMSDecoderError(FALSE);
        }

        /* Post to audio thread */
        fc_sxm_trMsgAudioSmsEvtDecoderState rMsg;
        rMsg.eState = eState;
        rMsg.hDecoder = hDecoder;
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }

    /* Check for antenna state updates */
    if(DECODER_OBJECT_EVENT_ANTENNA & tEventMask)
    {
        ANTENNA_STATE_ENUM eAntennaState=DECODER.eAntennaState(hDecoder, 1);
        ETG_TRACE_USR3(("DECODER Antenna State: Number of Antennas=%d Antenna State=%d", 
					    DECODER.n8NumAntennas(hDecoder),
		                eAntennaState
                        ));		

        /* Post antenna state to all data services */
        fc_sxm_trMsgLoopBack_AntennaState rMsgAntennaState;
        rMsgAntennaState.eAntennaState = eAntennaState;
        fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsgAntennaState);
        /* Get current antenna state */
        fc_sxm_trMsgAudioSmsEvtAntennaQuality rMsg;
        rMsg.eAntennaState = eAntennaState;
        /* Post to audio thread */
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }

    /* Check for signal quality updates */
    if(DECODER_OBJECT_EVENT_SIGNAL & tEventMask)
    {
        fc_sxm_trMsgAudioSmsEvtSignalState rMsg;
        rMsg.eSignalQuality.eState = SIGNAL_STATE_UNKNOWN;
        /* Get current signal quality */
        SMSAPI_RETURN_CODE_ENUM eRetCode = DECODER.eSignalQuality(hDecoder, &rMsg.eSignalQuality);
        if(SMSAPI_RETURN_CODE_SUCCESS == eRetCode)
        {
            ETG_TRACE_USR3(("DECODER Signal Quality: res=%d eState=%d eSatellite=%d",
                            ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eRetCode),
                            ETG_CENUM(SIGNAL_STATE_ENUM, rMsg.eSignalQuality.eState),
                            rMsg.eSignalQuality.eSatellite ));
                /* Post signal state to all data services */
            fc_sxm_trMsgLoopBack_SignalState rMsgSignalState;
            rMsgSignalState.eSignalState = ((rMsg).eSignalQuality).eState;
            fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsgSignalState);
            /* Post to audio thread */
            fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
        }
        else
        {
            ETG_TRACE_ERR(("Failed to get signal quality from decoder, Error code - %d", 
                           ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eRetCode) ));
        }
    }

    if (DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED & tEventMask)
    {
    	// a decoder subscription update event occurred
    	ETG_TRACE_USR2(("DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED "));
    	fc_sxm_trMsgAudioSmsSubscriptionUpdatedEvt rSubscriptionUpdateMsg;
    	// Post Subscription Update Event Msg to Audio Thread
    	fc_sxm_tclAudioApp::instance()->vPostMsgNew(rSubscriptionUpdateMsg);
    }

    /* Check for tune status */
    if(DECODER_OBJECT_EVENT_TUNE & tEventMask)
    {
        ETG_TRACE_USR4(("DECODER_OBJECT_EVENT_TUNE"));
        /* Post to audio thread */
        fc_sxm_trMsgAudioSmsEvtTuneState rMsg;
        rMsg.eTuneState = DECODER.eTuneState(hDecoder);
        rMsg.eTunedChannelID = DECODER.tCurrentChannelId(hDecoder);
        rMsg.eTunedCategoryID = DECODER.tCurrentCategoryId(hDecoder);
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
        //Todo:change later
        fc_sxm_trMsgDiagSmsEvtTuneState rDMsg;
        rDMsg.eTuneState = rMsg.eTuneState;
        fc_sxm_vPostMsgToCcaObj(fc_sxm_diagLibHandler::instance(), rDMsg);
    }
    /*Get the Engineering data information from the SMS*/
    if(tEventMask & DECODER_OBJECT_EVENT_ENGINEERING_DATA)
    {
        /*Function to update Diag data.Introduced the function 
        to prevent logiscope from throwing error (Max statements in a function)
        If function heavy for this call back .Do the activity here itself :check later-todo*/
        fc_sxm_tclDiagTmHandler::instance()->vHandleEngineeringDataCallBack(hDecoder);
    }

    if(DECODER_OBJECT_EVENT_CHANNEL & tEventMask)
    {
       ETG_TRACE_USR4(("DECODER_OBJECT_EVENT_CHANNEL"));
       ETG_TRACE_USR4(("Current Channel ID - %d", CHANNEL.tChannelId(hChannel) ));

       CHANNEL_EVENT_MASK tChannelEventMask = CHANNEL.tEventMask(hChannel);
       /* If channel id and service id both are changed then the tuned channel has changed in SMS */
       if( (tChannelEventMask & CHANNEL_OBJECT_EVENT_CHANNEL_ID) && (tChannelEventMask & CHANNEL_OBJECT_EVENT_SERVICE_ID) )
       {
          /* Post to audio thread */
          fc_sxm_trMsgAudioSmsEvtChannelEvt rMsg;
          fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
       }
    }

    if(DECODER_OBJECT_EVENT_PLAYBACK & tEventMask)
    {
       ETG_TRACE_USR1(("DECODER_OBJECT_EVENT_PLAYBACK event received"));
       fc_sxm_trMsgAudioSmsEvtPlaybackEvt rMsg;
   
       /* Extract required playback info from playback object in sms thread context itself */
       PLAYBACK_OBJECT pPlaybackHandle = DECODER.hPlayback(hDecoder); /* Get playback object for the decoder */
       if(PLAYBACK_INVALID_OBJECT != pPlaybackHandle) {
          rMsg.ePlaybackEventMask = PLAYBACK.tEventMask(pPlaybackHandle); /* Get current playback event mask */
          if(PLAYBACK_OBJECT_EVENT_PLAY & rMsg.ePlaybackEventMask) {
             rMsg.IsPlayActive = TRUE; 
          }
          if(PLAYBACK_OBJECT_EVENT_PAUSE & rMsg.ePlaybackEventMask) {
             rMsg.IsPlayActive = FALSE; 
          }
          if(PLAYBACK_OBJECT_EVENT_TIME_FROM_TRACK_START & rMsg.ePlaybackEventMask) {
             PLAYBACK.eTimeFromTrackStart(pPlaybackHandle, &(rMsg.u32TimeFromTrackStart));
          }
          if(PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO & rMsg.ePlaybackEventMask) {
             PLAYBACK.eRecordedContentTimeInfo(pPlaybackHandle, &(rMsg.u32RecordDuration), &(rMsg.u32RecordElapsedTime), &(rMsg.u32RecordRemainingTime));
          }
          if(PLAYBACK_OBJECT_EVENT_TRACKS_BEFORE & rMsg.ePlaybackEventMask) {
             PLAYBACK.eTracksBefore(pPlaybackHandle, &(rMsg.u16TracksBefore));
          }
          if(PLAYBACK_OBJECT_EVENT_TRACKS_REMAINING & rMsg.ePlaybackEventMask) {
             PLAYBACK.eTracksRemaining(pPlaybackHandle, &(rMsg.u16TracksRemaining));
          }
		  if(PLAYBACK_OBJECT_EVENT_PLAY_PERCENTAGE & rMsg.ePlaybackEventMask) {
        	  PLAYBACK.ePlayPercentage(pPlaybackHandle, &(rMsg.u8PlayPercentage));
          }
          if(PLAYBACK_OBJECT_EVENT_FILL_PERCENTAGE & rMsg.ePlaybackEventMask) {
			  PLAYBACK.eFillPercentage(pPlaybackHandle, &(rMsg.u8FillPercentage));
		  }
          if (PLAYBACK_OBJECT_EVENT_TIME_OFFSET & rMsg.ePlaybackEventMask) {
        	  PLAYBACK.eTimeOffset(pPlaybackHandle, &(rMsg.s32TimeOffset));
          }
          if (PLAYBACK_OBJECT_EVENT_WARNING & rMsg.ePlaybackEventMask) {
        	  PLAYBACK.eWarningOffsetGet(pPlaybackHandle, &(rMsg.u32WarningOffset));
          }
       }
       fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }
    if (DECODER_OBJECT_EVENT_TUNE_SCAN & tEventMask)
    {
    	DECODER_TUNE_SCAN_STATUS_MASK ptTuneScanStatus;
    	fc_sxm_trMsgAudioSmsEvtTuneScanEvt rMsg;
    	ETG_TRACE_USR1(("DECODER_OBJECT_EVENT_TUNE_SCAN event received"));

    	/*Get the current tune scan status*/
    	SMSAPI_RETURN_CODE_ENUM eReturnCode = DECODER.eIsTuneScanContentAvailable(hDecoder, &ptTuneScanStatus);

       	if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode){
    		/*Copy the current tune scan status*/
    		rMsg.u8TuneScanState = ptTuneScanStatus;
    	}
    	/* Post to audio thread */
    	fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }
    if(DECODER_OBJECT_EVENT_SEEK_ALERT & tEventMask)
    {
       fc_sxm_trMsgAudioSeekAlertEvent rMsg;
       fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }
}

// DECODER state machine
tVoid fc_sxm_tclSmsDecoder::vProcess(fc_sxm_trMsgAudioSmsEvtDecoderState const *prMsg) {
    DECODER_ERROR_CODE_ENUM enError=DECODER_ERROR_CODE_NONE;
    if (prMsg->eState == DECODER_STATE_ERROR && _hDecoder != DECODER_INVALID_OBJECT) {
        enError = DECODER.eErrorCode (_hDecoder);
    } 
    ETG_TRACE_USR1(("fc_sxm_tclSmsDecoder, state:%d->%d error = %d", 
                    ETG_CENUM(DECODER_STATE_ENUM, _enSmsState),
                    ETG_CENUM(DECODER_STATE_ENUM, prMsg->eState),
                    ETG_CENUM(DECODER_ERROR_CODE_ENUM, enError)));

    if(_enSmsState==prMsg->eState) {
        return;
    }
    _enSmsState=prMsg->eState;
    fc_sxm_tenSmEvent enEvent = fc_sxm_enSmEvent_SmsError;
    switch (_enSmsState) {
        case DECODER_STATE_RELEASED:
            enEvent=fc_sxm_enSmEvent_SmsStopped;
            break;

        case DECODER_STATE_INITIAL:
            enEvent=fc_sxm_enSmEvent_SmsInitial;
			fc_sxm_arl_tclISource::instance()->vSetMute(FALSE);
            break;

        case DECODER_STATE_UPDATING:
            // todo: notify audio
            return;
        case DECODER_STATE_READY:
            fc_sxm_tclSMSInit::instance()->vRestoreThreadPrios();
            fc_sxm_tclAudioProxy::instance()->vRegisterDecoderCallbacks(TRUE);
            enEvent=fc_sxm_enSmEvent_SmsReady;
            break;
        default:
            fc_sxm_tclSMSInit::instance()->vRestoreThreadPrios();
            enEvent=fc_sxm_enSmEvent_SmsError;
            break;
     
    }
    vPostEvent(enEvent);
}

tVoid fc_sxm_tclSmsDecoder::vLinkSmsParent() {
    _hSRM=_bParentUp ? fc_sxm_tclSmsSrm::instance()->hGetSmsSrm() : SRM_INVALID_OBJECT;
    ETG_TRACE_USR4(("fc_sxm_tclSmsDecoder::vLinkSmsParent _bParentUp=%d _hSRM=0x%08x",
                    _bParentUp, _hSRM));
}

tVoid fc_sxm_tclSmsDecoder::vChangeMask(fc_sxm_tenMaskMode enMode, tU32 u32Mask) {
    ETG_TRACE_USR4(("fc_sxm_tclSmsDecoder::vChangeMask enMode=%d u32Mask:0x%08x old:0x%08x",
                    ETG_CENUM(fc_sxm_tenMaskMode, enMode),
                    u32Mask, _u32Mask));
    if (fc_sxm_bChangeMask(enMode, u32Mask, _u32Mask)) {
        ETG_TRACE_USR2(("fc_sxm_tclSmsDecoder::vChangeMask set new:0x%08x", _u32Mask));
        if (_enState==fc_sxm_enSmsObjectState_READY) {
            ETG_TRACE_USR4(("fc_sxm_tclSmsDecoder::vChangeMask READY"));
            DECODER.eModifyRegisteredEventMask(_hDecoder, (DECODER_EVENT_MASK)_u32Mask, SMSAPI_MODIFY_EVENT_MASK_REPLACE);
        }
    }
}

static tVoid cb_vChannelEventCallback( 
    CHANNEL_OBJECT hChannel,
    CHANNEL_EVENT_MASK tEventMask,
    tVoid * /* pvEventCallbackArg */ )
{
    tBool bProcess=
        (CHANNEL_OBJECT_EVENT_REMOVED & tEventMask)    ||
        (CHANNEL_OBJECT_EVENT_SERVICE_ID & tEventMask) ||
        (CHANNEL_OBJECT_EVENT_CHANNEL_ID & tEventMask) ||
        (CHANNEL_OBJECT_EVENT_ATTRIBUTES & tEventMask) ||
        (CHANNEL_OBJECT_EVENT_NAME & tEventMask)       ||
        (CHANNEL_OBJECT_EVENT_CATEGORY & tEventMask)   ||
        (CHANNEL_OBJECT_EVENT_TITLE & tEventMask)      ||
        (CHANNEL_OBJECT_EVENT_ARTIST & tEventMask)     ||
        (CHANNEL_OBJECT_EVENT_ART & tEventMask);

    ETG_TRACE_USR3_CLS((TR_CLASS_FC_SXM_CHANNEL_LIST, "CHANNEL OBJECT event received - %d for "
                        "service id - %d and channel id - %d bProcess=%d", 
                        tEventMask,
                        CHANNEL.tServiceId(hChannel), 
                        CHANNEL.tChannelId(hChannel),
                        bProcess));

    /* Pass the data only if data modified is needed by channel list */
    if( bProcess )
    {
        fc_sxm_trMsgChannelEventCallback prMsgChannelEventUpdate = fc_sxm_trMsgChannelEventCallback(hChannel, tEventMask);
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(prMsgChannelEventUpdate);
    }
}

static tVoid cb_vCategoryEventCallback(
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    tVoid * /* pvEventCallbackArg */ )
{
    tBool bProcess = 
        (CATEGORY_OBJECT_EVENT_REMOVED & tEventMask)   ||
        (CATEGORY_OBJECT_EVENT_CONTENTS & tEventMask)  ||
        (CATEGORY_OBJECT_EVENT_NAME & tEventMask);

        ETG_TRACE_USR3_CLS((TR_CLASS_FC_SXM_CATEGORY_LIST, "CATEGORY OBJECT received event - %d for "
                            "Category ID - %d bProcess=%d", 
                            tEventMask,
                            CATEGORY.tGetCategoryId(hCategory),
                            bProcess));

    /* Process the data only if modified data is needed by category list */
    if( bProcess )
    {
        /* Pass event data to process further */
       fc_sxm_trMsgCategoryEventCallback prMsgCategoryEventUpdate = fc_sxm_trMsgCategoryEventCallback(hCategory,tEventMask);
       fc_sxm_tclAudioApp::instance()->vPostMsgNew(prMsgCategoryEventUpdate);
    }
}

tBool fc_sxm_tclSmsDecoder::bStartSmsObject() {
    _u32Mask=_u32MaskStart;
    ETG_TRACE_USR1(("fc_sxm_tclSmsDecoder::bStartSmsObject"));
    _hDecoder = DECODER.hGet(_hSRM, "Audio", (DECODER_EVENT_MASK)_u32Mask,
                             cb_vDecoderEventCallback, NULL);
    tBool bOk=_hDecoder != DECODER_INVALID_OBJECT;

#if 0
    /* Register for channel object and category object event callbacks */
    fc_sxm_tclAudioProxy::instance()->vRegisterDecoderCallbacks(TRUE);
#endif
    ETG_TRACE_USR1(("fc_sxm_tclSmsDecoder::bStartSmsObject bOk=%d _hDecoder=0x%08x _hSRM=0x%08x",
                    bOk, _hDecoder, _hSRM));
    return  bOk;
}

fc_sxm_tclSmsDecoder::fc_sxm_tclSmsDecoder():
    _hDecoder(DECODER_INVALID_OBJECT),
    _enSmsState(DECODER_STATE_RELEASED),
    _u32MaskStart(DECODER_OBJECT_EVENT_STATE | 
                  DECODER_OBJECT_EVENT_TUNE),
    _u32MaskReady(DECODER_OBJECT_EVENT_STATE |
             DECODER_OBJECT_EVENT_SIGNAL |
             DECODER_OBJECT_EVENT_ANTENNA |
             DECODER_OBJECT_EVENT_TUNE |
             DECODER_OBJECT_EVENT_CHANNEL |
             DECODER_OBJECT_EVENT_TUNE_SCAN |
             DECODER_OBJECT_EVENT_PLAYBACK|
			 DECODER_OBJECT_EVENT_SEEK_ALERT |
			 DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED),
    _u32Mask(DECODER_OBJECT_EVENT_STATE),
    _hSRM(SRM_INVALID_OBJECT),
    _u32NumStopRequests(0)
{
    vSetParent(fc_sxm_tclSmsModule::instance());
}

tVoid fc_sxm_tclSmsDecoder::vStopSmsObject() {
    ETG_TRACE_USR4(("fc_sxm_tclSmsDecoder::vStopSmsObject?"));
    if (_hDecoder != DECODER_INVALID_OBJECT) {
		ETG_TRACE_USR1(("fc_sxm_tclSmsDecoder::vStopSmsObject!"));
		DECODER.vRelease(_hDecoder);
		_hDecoder=DECODER_INVALID_OBJECT;
    }
}


tVoid fc_sxm_tclSmsDecoder::vUpdateState_(fc_sxm_tenSmsObjectState enState) {
    _enState = enState;
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STOPPING:
            vStopSmsObject();
            vNotifyChildren(FALSE);
            break;
        case fc_sxm_enSmsObjectState_STOPPED:
            vStopSmsObject();
            fc_sxm_tclSMSInit::instance()->bWaitDecoderDown(5000);
            vNotifyChildren(FALSE);
            vNotifyParent(FALSE);
            break;
        case fc_sxm_enSmsObjectState_STARTING:
            vNotifyParent(TRUE);
            break;
        case fc_sxm_enSmsObjectState_READY:
            vChangeMask(fc_sxm_enMaskMode_Add, _u32MaskReady);
            vNotifyChildren(TRUE);
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            vEnterError();
            break;
        default:
            break;
            
    }
}

// state-machine data-proxy (proxy using app-manager)
tVoid fc_sxm_tclDataProxy::vProcess(fc_sxm_trMsgAudioDataStopped const * /* prMsg */ )
{
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy::fc_sxm_trMsgAudioDataStopped?"));
    if (_bChildrenActive) {
        ETG_TRACE_USR1(("fc_sxm_tclDataProxy::fc_sxm_trMsgAudioDataStopped!"));
        vPostEvent(fc_sxm_enSmEvent_ChildrenDown);
    }
}

/* fc_sxm_tclDataProxy constructor */
fc_sxm_tclDataProxy::fc_sxm_tclDataProxy()
{
    vSetParent(fc_sxm_tclSmsDecoder::instance());
}

tVoid fc_sxm_tclDataProxy::vNotifyChildren(tBool bUp) {
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy::vNotifyChildren bUp=%d",
                    bUp));
    if (!bUp) {
        ETG_TRACE_USR4(("fc_sxm_tclDataProxy::vNotifyChildren _oDataDelayTimer.vStop()"));
        _oDataDelayTimer.vStop();
    }

    fc_sxm_trMsgLoopBack_DecoderState rMsgDecoderStateReady(bUp);
    fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsgDecoderStateReady);
}

// contains message delay timeout for audio start
tVoid fc_sxm_tclDataProxy::vProcessTimer(fc_sxm_trMsgAudioDataStartDelayTimeout * /* prMsg */ ) {
    ETG_TRACE_USR3(("fc_sxm_tclDataProxy::fc_sxm_trMsgAudioDataStartDelayTimeout"));
    vPostEvent(fc_sxm_enSmEvent_SmsReady);                
}

// method to change Sms Object state during deactivate
tVoid fc_sxm_tclDataProxy::vHandleDeActivate() {
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy:: vHandleDeActivate state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    // based on the state do the action
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
            vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
            break;
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_ERROR:
                vUpdateState(fc_sxm_enSmsObjectState_STOPPING);
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy:: vHandleDeActivate state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

// updates the Sms object state
tVoid fc_sxm_tclDataProxy::vUpdateState_(fc_sxm_tenSmsObjectState enState) {
    fc_sxm_tenSmsObjectState enOldState=_enState;
    _enState = enState;
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy::vUpdateState_ _oDataDelayTimer.vStop()"));
    _oDataDelayTimer.vStop();
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
        {
            vNotifyParent(TRUE);
            tU32 u32DelayMs=fc_sxm_tclConfig::instance()->u32GetDelayDataStartMs();
            ETG_TRACE_USR4(("fc_sxm_tclDataProxy::vUpdateState_ _oDataDelayTimer.vStart(%d)", u32DelayMs));
            if (u32DelayMs) {
                _oDataDelayTimer.vStart(fc_sxm_tclConfig::instance()->u32GetDelayDataStartMs());
            }
            else {
                vPostEvent(fc_sxm_enSmEvent_SmsReady);
            }
        }
        break;
        case fc_sxm_enSmsObjectState_STOPPED:
            if (enOldState != fc_sxm_enSmsObjectState_STOPPING) {
                vNotifyChildren(FALSE);
            }
            _bChildrenActive=FALSE;
            vNotifyParent(FALSE);
            
            break;
        case fc_sxm_enSmsObjectState_READY:
            vNotifyChildren(TRUE);
            vPostEvent(fc_sxm_enSmEvent_ChildrenUp);
            break;
        case fc_sxm_enSmsObjectState_STOPPING:
            if (_bChildrenActive) {
                vNotifyChildren(FALSE);
            } else {
                vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
            }
            break;
        case fc_sxm_enSmsObjectState_ERROR:
            vEnterError();
            break;
            
        default:
            break;
    }
}

tVoid fc_sxm_tclDataProxy::vHandleChildrenDown() {
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy:: vHandleChildrenDown state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    switch (_enState) {
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_STOPPING:
            if (!_bWantsActivate) {
                vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
            }
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy:: vHandleChildrenDown state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

// update the decoder object based on parent status
tVoid fc_sxm_tclAudioProxy::vLinkSmsParent() {
    _hDecoder=_bParentUp ? fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder() : DECODER_INVALID_OBJECT;
    ETG_TRACE_USR4(("fc_sxm_tclAudioProxy::vLinkSmsParent _bParentUp=%d _hDecoder=0x%08x",
                    _bParentUp, _hDecoder));
}

/*
AudioStopped case, handle the children threads
*/
tVoid fc_sxm_tclAudioProxy::vSetAudioStopped() {
   ETG_TRACE_USR2(("fc_sxm_tclAudioProxy::vSetAudioStopped?"));

   /*Skip this process if external diag mode is on as we require the Audio 
     service to be available.This is required so that we receive the CCA 
     message from HMI to stop external Diag mode*/
   if(TRUE != (fc_sxm_tclDiagExtDiagnosisMode::instance()-> bGetExtDiagModeStatus())) 
   {
      fc_sxm_tclAudioApp::instance()->vAllowService(FALSE, CCA_C_U16_SRV_SXM_AUDIO);
   }
   else
   {
      ETG_TRACE_USR4(("fc_sxm_tclAudioProxy::vSetAudioStopped Ext Diag mode ON"));
   }

   if (_bChildrenActive) {
      ETG_TRACE_USR4(("fc_sxm_tclAudioProxy::vSetAudioStopped!"));
      vPostEvent(fc_sxm_enSmEvent_ChildrenDown);
   }
}

/* fc_sxm_tclAudioProxy- Constructor */
fc_sxm_tclAudioProxy::fc_sxm_tclAudioProxy():
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
	_enAudioState(fc_sxm_enAudioInvalid),
#endif
    _hDecoder(DECODER_INVALID_OBJECT),
    _bChnCatCbRegistered(FALSE)
{
    vSetParent(fc_sxm_tclSmsDecoder::instance());
}

// overwrite, because our child is no fc_sxm_tclSmsSm
tVoid fc_sxm_tclAudioProxy::vNotifyChildren(tBool bUp) {
    ETG_TRACE_USR4(("fc_sxm_tclAudioProxy::vNotifyChildren bUp=%d START",
                    bUp));
    if (bUp) {
        fc_sxm_tclAudioApp::instance()->vSetDecoder(_hDecoder);
        fc_sxm_tclAudioApp::instance()->vAllowService(TRUE, CCA_C_U16_SRV_SXM_AUDIO);
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
                _enAudioState = fc_sxm_enAudioReady;
                fc_sxm_tclApp::instance()->vResetWDTCount();
#endif
    }
    ETG_TRACE_USR4(("fc_sxm_tclAudioProxy::vNotifyChildren bUp=%d END",
                    bUp));
}

// Message containing Audio Start delay time
tVoid fc_sxm_tclAudioProxy::vProcessTimer(fc_sxm_trMsgAudioAudioStartDelayTimeout * /* prMsg */ ) {
    vPostEvent(fc_sxm_enSmEvent_SmsReady);
}

tVoid fc_sxm_tclAudioProxy::vHandleDeActivate() {
    ETG_TRACE_USR4(("fc_sxm_tclDataProxy:: vHandleDeActivate state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    // based on sms object state, do the action
    switch (_enState) {
        case fc_sxm_enSmsObjectState_STARTING:
            vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
            break;
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_ERROR:
                vUpdateState(fc_sxm_enSmsObjectState_STOPPING);
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclAudioProxy:: vHandleDeActivate state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

// Method to register decoder callabck
tVoid fc_sxm_tclAudioProxy::vRegisterDecoderCallbacks(tBool bRegister) {
    ETG_TRACE_COMP(("fc_sxm_tclAudioProxy::vRegisterDecoderCallbacks bRegister=%d", bRegister));
    _bChnCatCbRegistered=bRegister;
    if(SMSAPI_RETURN_CODE_SUCCESS != CHANNEL.eNotifyOnChange(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), bRegister ? cb_vChannelEventCallback : OSAL_NULL, OSAL_NULL))
    {
        ETG_TRACE_ERR(("Failed to register for channel notification bRegister=%d", bRegister));
    }
    
    if(SMSAPI_RETURN_CODE_SUCCESS != CATEGORY.eNotifyOnChange(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), bRegister ? cb_vCategoryEventCallback : OSAL_NULL, OSAL_NULL))
    {
        ETG_TRACE_ERR(("Failed to register for category notification bRegister=%d", bRegister));
    }
}

tVoid fc_sxm_tclAudioProxy::vUpdateState_(fc_sxm_tenSmsObjectState enState) {
    if (_enState != enState) {
        _enState = enState;
        _oAudioDelayTimer.vStop();
        switch (_enState) {
            case fc_sxm_enSmsObjectState_STARTING:
            {
                vNotifyParent(TRUE);
                tU32 u32DelayMs=fc_sxm_tclConfig::instance()->u32GetDelayAudioStartMs();
                ETG_TRACE_USR4(("fc_sxm_tclAudioProxy::vUpdateState_ _oAudioDelayTimer.vStart(%d)", u32DelayMs));
                _oAudioDelayTimer.vStart(u32DelayMs ? u32DelayMs : 250);

            }
            break;

            case fc_sxm_enSmsObjectState_STOPPED:
                _bChildrenActive=FALSE;
                vNotifyChildren(FALSE);
                vNotifyParent(FALSE);
                break;
            case fc_sxm_enSmsObjectState_READY:
               ETG_TRACE_USR1(("fc_sxm_tclAudioProxy::fc_sxm_enSmsObjectState_READY: call vNotifyChildren"));
                vNotifyChildren(TRUE);
               ETG_TRACE_USR1(("fc_sxm_tclAudioProxy::fc_sxm_enSmsObjectState_READY: post fc_sxm_enSmEvent_ChildrenUp"));
                vPostEvent(fc_sxm_enSmEvent_ChildrenUp);
                break;
            case fc_sxm_enSmsObjectState_STOPPING:
#if((FC_SXM_ENABLE_WATCHDOG_TIMER) || (FC_SXM_ENABLE_LCM_WATCHDOG_TIMER))
            	// Change the state, as we start monitoring through watchdog
            	if((fc_sxm_tclSystemStates::instance()->enGetSystemState() != fc_sxm_enSystemState_OFF) &&
            	  (fc_sxm_tclSystemStates::instance()->enGetActivityType() != fc_sxm_enSystemState_DownLoad) &&
            	  (fc_sxm_tclDiagExtDiagnosisMode::instance()->bGetExtDiagModeStatus() != TRUE) &&
            	  (fc_sxm_tclSystemStates::instance()->bIsDiagDefsetTEF() != TRUE))
            	{
            		_enAudioState = fc_sxm_enAudioStopped;
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
            		fc_sxm_tclAppManager::instance()->vStartSXMWatchdog();
#endif
            	}
#endif
                // special case for proxy, we have to tell child to stopp
                vRegisterDecoderCallbacks(FALSE);
                if (_bChildrenActive) {
                	fc_sxm_tclSmsDecoder::instance()->vChangeMask(fc_sxm_enMaskMode_Set, DECODER_OBJECT_EVENT_STATE);
                    fc_sxm_tclAudioApp::instance()->vStop();
                }
                else {
                    vUpdateState(fc_sxm_enSmsObjectState_STOPPED);
                }
                break;
            case fc_sxm_enSmsObjectState_ERROR:
                vEnterError();
                break;

            default:
                break;
        }
    }
}

/*
   Based on the state update the state of children threads
*/
tVoid fc_sxm_tclAudioProxy::vHandleChildrenDown() {
    ETG_TRACE_USR4(("fc_sxm_tclAudioProxy:: vHandleChildrenDown state=%d START",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
    // based on state do the action
    switch (_enState) {
        case fc_sxm_enSmsObjectState_READY:
        case fc_sxm_enSmsObjectState_STARTING:
        case fc_sxm_enSmsObjectState_STOPPING:
            if (!_bWantsActivate) {
                vUpdateState(fc_sxm_enSmsObjectState_STOPPED); // update the state to stopped
            }
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("fc_sxm_tclAudioProxy:: vHandleChildrenDown state=%d END",
                    ETG_CENUM(fc_sxm_tenSmsObjectState, _enState)));
}

// class ActivityBase
fc_sxm_tclActivityBase::fc_sxm_tclActivityBase(fc_sxm_tenSystemState enType, fc_sxm_tenActivityPrio enPrio, tBool bInterruptible):
    _enType(enType),
    _enPrio(enPrio),
    _enState(fc_sxm_enActivityState_Idle),
    _bInterruptible(bInterruptible)
{};

tBool fc_sxm_tclActivityBase::bSetType(fc_sxm_tenSystemState enType) {
   // based on the enstate,
   // set the enType
    if (_enState != fc_sxm_enActivityState_Idle) {
        return FALSE;
    }
    _enType=enType;
    return TRUE;
}
// Request the activity
tBool fc_sxm_tclActivityBase::bRequest() {
    return fc_sxm_tclSystemStates::instance()->bRequestActivity(this);
}

// On Completion Update the activity
tVoid fc_sxm_tclActivityBase::vOnDone() {
    fc_sxm_tclSystemStates::instance()->vActivityDone(this);
};

// update system state
tVoid fc_sxm_tclSystemStates::vUpdateSystemState() {
    vStartSequence();
    _listSMs.vUpdateSystemState(_enSystemState);
    vEndSequence();
}

// inner class fc_sxm_tclSystemStates::tclSmList
tVoid fc_sxm_tclSystemStates::tclSmList::vUpdateSystemState(fc_sxm_tenSystemState enSystemState) {
    fc_sxm_tclSystemStates *poSys=fc_sxm_tclSystemStates::instance();
    SXM_FOREACH(deque<fc_sxm_tclSmsSm *>, iter, _oQ) {
        fc_sxm_tclSmsSm *poSM=*iter;
        fc_sxm_trSmEvent rEvent;
        rEvent.poToSm=poSM;
        rEvent.poArgSm=OSAL_NULL;
        rEvent.enEvent=fc_sxm_tclSystemStates::bWantsActivate(poSM->enGetType(), enSystemState) ? 
            fc_sxm_enSmEvent_Activate : fc_sxm_enSmEvent_DeActivate;   
        poSys->vPostEvent(rEvent);
    }
}

tBool fc_sxm_tclSystemStates::tclSmList::bStable() {
    tBool bRes=TRUE;
    SXM_FOREACH(deque<fc_sxm_tclSmsSm *>, iter, _oQ) {
        fc_sxm_tclSmsSm *poSM=*iter;
        if (OSAL_NULL != poSM) {
            if (!poSM->bStable()) {
                bRes= FALSE;
                //                break;
                // todo: use bIsTraceActive()
            }
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::tclSmList::bStable()=%d",
                    bRes));
    return bRes;
}

// class fc_sxm_tclSystemStates
fc_sxm_tclSystemStates::~fc_sxm_tclSystemStates() {
    _poCurActivity = OSAL_NULL;
}

tVoid fc_sxm_tclSystemStates::vInit() 
{
    _bFirstInit=TRUE;
    fc_sxm_tclSMSInit::instance()->vWaitSmsInitialized();
    vUpdate();
}

// Contains the message for Audio SMS start delay time out
tVoid fc_sxm_tclSystemStates::vProcessTimer(fc_sxm_trMsgAudioSmsStartDelayTimeout const * /* prMsg */ )
{
    ETG_TRACE_USR3(("fc_sxm_tclSystemStates::fc_sxm_trMsgAudioSmsStartDelayTimeout()"));
    vUpdate();
}

// Message for new app state
tVoid fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioCmdNewAppState const *prMsg) {
    tBool bCcaOn = prMsg->bCcaOn;
    ETG_TRACE_USR1(("fc_sxm_tclSystemStates::fc_sxm_trMsgAudioCmdNewAppState() %d->%d",
    		ETG_CENUM(fc_sxm_tenCcaState, _enCcaState), bCcaOn));


    if (bCcaOn) {
    	_enCcaState = fc_sxm_enCcaState_On;

        tU32 u32DelayMs=fc_sxm_tclConfig::instance()->u32GetDelaySmsStartMs();
        if (u32DelayMs) {
            _oSmsDelayTimer.vStart(u32DelayMs);
        }
        else {

            // Check if we have to stop any objects, before we start
            if (_enSystemState == fc_sxm_enSystemState_OFF) {
            	_enErrorRecoveryMode=enErrorRecoveryMode_Active;
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
            	fc_sxm_tclAppManager::instance()->vStartSXMWatchdog();
#endif
            }
            vUpdate();
        }
    }
    else {
        _oSmsDelayTimer.vStop();
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
        fc_sxm_tclAppManager::instance()->vStopSXMWatchdog();
#endif
        _enCcaState = fc_sxm_enCcaState_Off;
        vUpdate();

    }
}

// Message for CriticalVoltage
tVoid fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioSetCriticalVoltage const *prMsg) {
	if (_bCvOn != prMsg->bCvOn) {
        ETG_TRACE_USR1(("fc_sxm_tclSystemStates::fc_sxm_trMsgAudioSetCriticalVoltage() %d->%d _enErrorRecoveryMode=%d",
                        _bCvOn, prMsg->bCvOn,
                        ETG_CENUM(fc_sxm_tenErrorRecoveryMode, _enErrorRecoveryMode)));
        _bCvOn=prMsg->bCvOn;
        if (_bCvOn) {

        	// Mark the start of critical voltage handling
        	SetCriticalVoltageStart();

			// Set the SXM Initializing advisory
			fc_sxm_tclAdvisories::instance()->vSetSXMInitializingAdvisory();

            fc_sxm_tclSMSInit::instance()->vStopX65Module();
            if (_enErrorRecoveryMode==enErrorRecoveryMode_Active){
                _enErrorRecoveryMode=enErrorRecoveryMode_Pending;
            }

#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
			ETG_TRACE_USR4(("Reset watchdogtimer counter"));
			fc_sxm_tclAppManager::instance()->vSXMWatchdogReset();
#endif
        }
        else {
            // while starting of x65 has to be done by us:
            // todo: should be done by spm
          //  fc_sxm_tclSMSInit::instance()->vStartX65Module();
            // when spm does its job replace by:
            // fc_sxm_tclSMSInit::instance()->vSetX65AutoOn();

            _enErrorRecoveryMode=enErrorRecoveryMode_Active;
            vUpdate(TRUE);
        }
    }
}


// Message for handling SMSCfgError
tVoid fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioSMSCfgError const * /*prMsg*/) {

	 ETG_TRACE_ERR(("fc_sxm_tclSystemStates::fc_sxm_trMsgAudioSMSCfgError()"));

	_enErrorRecoveryMode=enErrorRecoveryMode_Active;
	vUpdate(TRUE);
}

tVoid fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioCvmEvent const *prMsg) {
#ifdef VARIANT_S_FTR_ENABLE_SPM_CORE_FI	
    spm_fi_tcl_SPM_e32_CVM_EVENT::tenType enSpmCvmEvt=prMsg->oFiMsg.CvmEvent.enType;
#else
    cfc_fi_tcl_SPM_e32_CVM_EVENT::tenType enSpmCvmEvt=prMsg->oFiMsg.CvmEvent.enType;
#endif
    tBool bCvOn=FALSE;

    switch (enSpmCvmEvt) {
#ifdef VARIANT_S_FTR_ENABLE_SPM_CORE_FI			
        case spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_START:
        case spm_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_END:
#else
        case cfc_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_START:
        case cfc_fi_tcl_SPM_e32_CVM_EVENT::FI_EN_SPM_U32_CVM_CRITICAL_LOW_VOLTAGE_END:
#endif
            bCvOn=TRUE;
            break;
        default:
            break;
    }
    ETG_TRACE_USR1(("fc_sxm_tclSystemStates::fc_sxm_trMsgAudioCvmEvent(enSpmCvmEvt=%d) bCvOn=%d",
                    enSpmCvmEvt, bCvOn));
    fc_sxm_trMsgAudioSetCriticalVoltage rMsg = fc_sxm_trMsgAudioSetCriticalVoltage(bCvOn);
    vProcess(&rMsg);
}

// fc_sxm_tclSystemStates Constructor
fc_sxm_tclSystemStates::fc_sxm_tclSystemStates():
    _enSystemState(fc_sxm_enSystemState_INVALID),
    _enCcaState(fc_sxm_enCcaState_Invalid),
    _bCvOn(FALSE),
    _bFirstInit(FALSE),
    _bSMSDecoderError(FALSE),
    _poCurActivity(OSAL_NULL),
    _bEnteringActity(FALSE),
    _bTerminateActivity(FALSE),
    _bProcessingQ(FALSE),
    _u32SequenceDepth(0),
    _bDownloadRecovery(FALSE),
	_enErrorRecoveryMode(enErrorRecoveryMode_Idle)
{
    // add default instances to the list for system state
    _listSMs.poAdd(fc_sxm_tclSmsSms::instance());
    _listSMs.poAdd(fc_sxm_tclSmsSrm::instance());
    _listSMs.poAdd(fc_sxm_tclSmsModule::instance());
    _listSMs.poAdd(fc_sxm_tclSmsDecoder::instance());
    _listSMs.poAdd(fc_sxm_tclDataProxy::instance());
    _listSMs.poAdd(fc_sxm_tclAudioProxy::instance());
}

tVoid fc_sxm_tclSystemStates::vCheckAndRunActivities() {
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vCheckAndRunActivities()"));
    if (OSAL_NULL != _poCurActivity) {
        // we have an activity
        if (_poCurActivity->_enState==fc_sxm_enActivityState_Pending && _bEnteringActity && _listSMs.bStable()) {
            // start activity
            ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vCheckAndRunActivities() START activity"));
            _poCurActivity->_enState=fc_sxm_enActivityState_Running;
            _poCurActivity->vRun();
        } else if (_poCurActivity->_enState==fc_sxm_enActivityState_Running && _bTerminateActivity) {
            // stop activity
            ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vCheckAndRunActivities() TERMINATE activity"));
            _poCurActivity->_enState=fc_sxm_enActivityState_Terminating;
            _poCurActivity->vStop();
        }
    }
}

tVoid fc_sxm_tclSystemStates::vUpdate(tBool bForced) {
    // check for activities:
    fc_sxm_tenSystemState enActityType = enGetActivityType();
    fc_sxm_tenSystemState enSystemState=_enSystemState;
    _bEnteringActity=FALSE;
    _bTerminateActivity=FALSE;

	if (_bFirstInit) {
		enSystemState=fc_sxm_enSystemState_INIT;
		_bFirstInit = FALSE;
	}
	else if (_enCcaState == fc_sxm_enCcaState_Off) {
   		enSystemState=fc_sxm_enSystemState_OFF;
   		if((_bCvOn) && (_enErrorRecoveryMode == enErrorRecoveryMode_Pending)) {
   		    _enErrorRecoveryMode = enErrorRecoveryMode_Active;
   		}
    }
    else if (bIsInErrorRecovery()) {
        enSystemState=fc_sxm_enSystemState_Error;
    }
    else {
        enSystemState=fc_sxm_enSystemState_Normal;
        if (enActityType <= enSystemState) {
            _bEnteringActity=TRUE;
            enSystemState=enActityType;             
        } else {
            _bTerminateActivity=TRUE;
        }
    }
    
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vUpdate() _enSystemState:%d-->%d activity=%d (enter=%d, terminate=%d)",
                    ETG_CENUM(fc_sxm_tenSystemState, _enSystemState),
                    ETG_CENUM(fc_sxm_tenSystemState, enSystemState),
                    ETG_CENUM(fc_sxm_tenSystemState, enActityType),
                    _bEnteringActity,
                    _bTerminateActivity));
    vPropagateSystemState(enSystemState, bForced);
}

tVoid fc_sxm_tclSystemStates::vPropagateSystemState(fc_sxm_tenSystemState enSystemState, tBool bForced) {
    if (_enSystemState != enSystemState || bForced) {
        ETG_TRACE_USR2(("fc_sxm_tclSystemStates::vPropagateSystemState() %d-->%d bForced=%d START",
                        ETG_CENUM(fc_sxm_tenSystemState, _enSystemState),
                        ETG_CENUM(fc_sxm_tenSystemState, enSystemState),
                        bForced));
        _enSystemState = enSystemState;
        vUpdateSystemState();
		ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vPropagateSystemState() END"));
    }
}

fc_sxm_tenSystemState fc_sxm_tclSystemStates::enGetActivityType() const
{
    if (OSAL_NULL == _poCurActivity) {
        return fc_sxm_enSystemState_INVALID;
    }
    return _poCurActivity->enGetType();
}


tBool  fc_sxm_tclSystemStates::bRequestActivity(fc_sxm_tclActivityBase *poActivity) {
    tBool bRes=FALSE;
    if ((tU32)poActivity->enGetType()>=(tU32)fc_sxm_enSystemState_INVALID) {
        return FALSE;
    }
    
    if (OSAL_NULL == _poCurActivity) {
        _poCurActivity = poActivity;
        _poCurActivity->_enState=fc_sxm_enActivityState_Pending;
        bRes=TRUE;
        ETG_TRACE_USR2(("fc_sxm_tclSystemStates::bRequestActivity(%d) START",
                        ETG_CENUM(fc_sxm_tenSystemState, _poCurActivity->enGetType())));
    }
    vUpdate();
    return bRes;
}



tVoid  fc_sxm_tclSystemStates::vActivityDone(fc_sxm_tclActivityBase *poActivity) {
    ETG_TRACE_USR1(("fc_sxm_tclSystemStates::vActivityDone:_enErrorRecoveryMode=%d _poCurActivity=0x%08x poActivity=0x%08x ",
                    ETG_CENUM(fc_sxm_tenErrorRecoveryMode, _enErrorRecoveryMode), _poCurActivity, poActivity))
    if (_enErrorRecoveryMode == enErrorRecoveryMode_Failed) {
        // dont stop activity;
    }
    if(OSAL_NULL != _poCurActivity)
    {
        ETG_TRACE_USR1(("enGetActivityType()=%d", enGetActivityType()));
        if (poActivity == _poCurActivity)
        {
            _poCurActivity->_enState=fc_sxm_enActivityState_Idle;
            if (enGetActivityType()==fc_sxm_enSystemState_DownLoad) {
                _bDownloadRecovery=TRUE;
            }
            _poCurActivity = OSAL_NULL;
            vUpdate();
        }
    }
}

tBool fc_sxm_tclSystemStates::bWantsActivate(fc_sxm_tenSmType enSmType, fc_sxm_tenSystemState enSytemState) {
    tBool bRes=FALSE;
    if (!fc_sxm_tclSystemStates::instance()->bIsInErrorRecovery() && (enSmType < en_fc_sxm_tclSmTypeMax) && (enSytemState < fc_sxm_enSystemState_INVALID) ) {
        bRes = _ActivateMatrix[enSmType][enSytemState];
    }
    
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::bWantsActivate(%d, %d): %d",
                    ETG_CENUM(fc_sxm_tenSmType, enSmType),
                    ETG_CENUM(fc_sxm_tenSystemState, enSytemState),
                    bRes));
    return bRes;
}

tVoid fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioSmEvent const *prMsg) 
{
   ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vProcess(fc_sxm_trMsgAudioSmEvent)"));

   if(OSAL_NULL != prMsg) {
      fc_sxm_trSmEvent rEvent = fc_sxm_trSmEvent(prMsg->_tSmEvent);
      if(rEvent.enEvent == fc_sxm_enSmEvent_EndSequence) {
	      _u32SequenceDepth--;
      } else {
         if (OSAL_NULL != rEvent.poToSm) {
            rEvent.poToSm->vHandleEvent(rEvent.enEvent);
         }
      }
      
      if (bIsSequenceComplete()) {
         if (_listSMs.bStable()) {
            if (_enSystemState==fc_sxm_enSystemState_OFF) {
               fc_sxm_trMsgOffReached rMsg;
               fc_sxm_tclApp::instance()->vPostMsgNew(rMsg);
               if (_enErrorRecoveryMode==enErrorRecoveryMode_Active) {
                  _enErrorRecoveryMode=enErrorRecoveryMode_Idle;

                  // Critical voltage handling is now complete
                  SetCriticalVoltageEnd();
               }
            }
            else if (_enErrorRecoveryMode==enErrorRecoveryMode_Active) {
               ETG_TRACE_USR2(("trErrorRecovery: Error Recovery: stop all done"));
               _enErrorRecoveryMode=enErrorRecoveryMode_Idle;

               // Critical voltage handling is now complete
               SetCriticalVoltageEnd();
               vUpdate();
            }
            else {
               vCheckAndRunActivities();
            } 
         } 
      }
   }
}

tVoid fc_sxm_tclSystemStates::SetCriticalVoltageStart()
{
	SMSLIB_bCvOn = TRUE;
}

tVoid fc_sxm_tclSystemStates::SetCriticalVoltageEnd()
{
	SMSLIB_bCvOn = FALSE;
}

tVoid fc_sxm_tclSystemStates::vPostEvent(fc_sxm_trSmEvent const &rEvent) const {
    ETG_TRACE_USR2(("fc_sxm_tclSystemStates::vPostEvent"));
    fc_sxm_trMsgAudioSmEvent rMsg(rEvent);
    fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
}

/*
   Method to handle Sms State Machine error
*/
tVoid fc_sxm_tclSystemStates::vHandleError(fc_sxm_tclSmsSm * /* poSM */ ) {
    ETG_TRACE_USR2(("fc_sxm_tclSystemStates::vHandleError START: currenMode=%u",
                    ETG_CENUM(fc_sxm_tenErrorRecoveryMode, _enErrorRecoveryMode)));
    // based on the recoverymode do the action
    if (_bCvOn) {
        _enErrorRecoveryMode=enErrorRecoveryMode_Pending;
        /* no error-recovery done here,
           we will restart all later on anyhow
        */
        return;
    }

    /*
      tear down whole sms for leaf to root.
      problem: in most cases other sms-objects will also go to error-state.
    */
    if (!bIsInErrorRecovery()) {
        // stop sms
        _enErrorRecoveryMode=enErrorRecoveryMode_Active;
        vUpdate();
    }
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vHandleError END: newMode=%u",
                    ETG_CENUM(fc_sxm_tenErrorRecoveryMode, _enErrorRecoveryMode)));
}
    
tBool fc_sxm_tclSystemStates::bIsDownloadRecovery() const {
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::bIsDownloadRecovery: %d", _bDownloadRecovery));
    return _bDownloadRecovery;
}

tVoid fc_sxm_tclSystemStates::vEndDownloadRecovery() {
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vEndDownloadRecovery: bDownloadRecovery_old=%d", _bDownloadRecovery));
    _bDownloadRecovery=FALSE;
}

tVoid fc_sxm_tclSystemStates::vStartDownloadRecovery() {
    ETG_TRACE_USR4(("fc_sxm_tclSystemStates::vStartDownloadRecovery: enGetActivityType()=%d bDownloadRecovery_old=%d", 
                    ETG_CENUM(fc_sxm_tenSystemState, enGetActivityType()),
                    _bDownloadRecovery));
    
    if (enGetActivityType()==fc_sxm_enSystemState_DownLoad) {
        _bDownloadRecovery=TRUE;
    }
}

tBool fc_sxm_tclSystemStates::bIsDiagDefsetTEF() const {
	return ((enGetSystemState() == fc_sxm_enSystemState_DefsetFactory) &&
			(fc_sxm_tclDiagDefset::instance()->enDiagDefsetType() == fc_sxm_enDiagDefsetType_Factory));
}
tVoid fc_sxm_tclSystemStates::vRemove(string &cmd) const
{
	tChar arCmd[BUFF_SIZE];
	tS32 retVal;

	size_t u32Length = cmd.copy(arCmd,cmd.size(),0);
	arCmd[u32Length] = '\0';

	retVal = remove(arCmd);
	if (retVal == 0)
	{
		ETG_TRACE_USR1(("file removed successfully"));
	}
	else
	{
		ETG_TRACE_ERR(("Error: unable to remove the file"));
	}
	cmd.clear();
}

tVoid fc_sxm_tclSystemStates::vDeleteSmsCfg()
{
    ETG_TRACE_USR3(("vDeleteSmsCfg"));
    string cmd;
    cmd = FC_SXM_DEFAULT_SMS_CFG_PATH;
    cmd += "/sms.cfg";
    vRemove(cmd);
}
