/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_flash_proxy_if.h
* @brief       Interface for sports flash proxy class.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef FC_SXM_TCL_SPORTS_FLASH_PROXY_IF_H_
#define FC_SXM_TCL_SPORTS_FLASH_PROXY_IF_H_


#include "fc_sxm_types.h"
#include "fc_sxm_audio_types.h"

class Ifc_sxm_tclDecoder;

struct fc_sxm_trMsgSFGameAlert : public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_FLASHEVENTS + 25);

   fc_sxm_trMsgSFGameAlert() : enProgramStatus(SPORTS_FLASH_PROGRAM_ENDED), trGameInfo()
   { /* Default CTOR*/ }

   tVoid operator() (SPORTS_FLASH_PROGRAM_STATUS_ENUM enStatus, const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT* poGameInfo) {
	   // Using functor to copy values
	   enProgramStatus = enStatus;
	   trGameInfo.tChannelID = poGameInfo->tChannelID;
	   trGameInfo.hHomeLeagueId = poGameInfo->hHomeLeagueId;
	   trGameInfo.hHomeTeamId = poGameInfo->hHomeTeamId;
	   trGameInfo.hAwayLeagueId = poGameInfo->hAwayLeagueId;
	   trGameInfo.hAwayTeamId = poGameInfo->hAwayTeamId;
	   trGameInfo.un16Bias = poGameInfo->un16Bias;
   }
   SPORTS_FLASH_PROGRAM_STATUS_ENUM enProgramStatus;
   SPORTS_FLASH_GAME_EVENT_INFO_STRUCT trGameInfo;
};

struct fc_sxm_trMsgSFFlashEventAlert : public fc_sxm_tclMessage
{
   SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_FLASHEVENTS + 26);

   fc_sxm_trMsgSFFlashEventAlert(CHANNEL_ID chnID = 0,
		   SPORTS_FLASH_EVENT_ID eventID = SPORTS_FLASH_INVALID_EVENT_ID,
		   SPORTS_FLASH_EVENT_STATUS_ENUM enStatus = SPORTS_FLASH_EVENT_EXPIRED) :
			   tChannelID(chnID), tFlashEventID(eventID), eFlashEventStatus(enStatus)
   { /* Default CTOR*/ }

   CHANNEL_ID tChannelID;
   SPORTS_FLASH_EVENT_ID tFlashEventID;
   SPORTS_FLASH_EVENT_STATUS_ENUM eFlashEventStatus;
};

struct fc_sxm_trSFChannel {
	CHANNEL_ID tChannelID;
	SERVICE_ID tServiceID;
	tBool bIsLocked;
	tBool bIsSkipped;
	tBool bIsSubscribed;
	tBool bIsMature;
	tBool bIsFreeToAir;
	tBool bIsChnArtAvl;

	tVoid vReset() {
		// Reset all data
		tChannelID = CHANNEL_INVALID_ID;
		tServiceID = SERVICE_INVALID_ID;
		bIsLocked = bIsSkipped = bIsSubscribed = bIsMature = bIsFreeToAir = bIsChnArtAvl =FALSE;
	}

	bool bChannelStatus() const {
		return ((!bIsLocked) && (!bIsSkipped) && (bIsSubscribed | bIsFreeToAir));
	}
};

struct fc_sxm_trSFTeam {
	string sTeamName;
	string sTeamNickName;
	string sTeamAbbrName;
	TEAM_OBJECT hTeamObj;
	CID_OBJECT hTeamCID;
	tS32 s32TableVersion;
	tBool bIsSFEligible;
	tBool bIsFavoriteTeam;
	tBool bIsMonitored;
	tU16 u16TeamID;

	tVoid vReset() {
		// Reset all data
		sTeamName = sTeamNickName = sTeamAbbrName = "";
		hTeamObj = TEAM_INVALID_OBJECT;
		hTeamCID = CID_INVALID_OBJECT;
		s32TableVersion = 0;
		bIsSFEligible = FALSE;
		bIsFavoriteTeam = FALSE;
		bIsMonitored = TRUE;
		u16TeamID = 0U;
	}
};

struct fc_sxm_trSFLeague {
	// Sports Details to which the League is affiliated to
	string sAbbrSportName;
	string sSportName;
	CID_OBJECT hSportCID;
	SPORTS_ENUM enSports;
	midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType enType;

	// League's Details
	string sAbbrLeagueName;
	string sLeagueName;
	CID_OBJECT hLeagueCID;
	LEAGUE_ENUM enLeague;
	LEAGUE_OBJECT hLeagueObj;
	tS32 s32TableVersion;
	tBool bIsSFEnabled;
	tU8 u8LeagueID;

	// List of Teams
	map<tU16, fc_sxm_trSFTeam> mapTeams;
	set<tU16> setFavTeams;
	tBool bAddTeams(fc_sxm_trSFTeam& );
	tVoid vReset() {
		// Clear all member data
		sAbbrSportName = sSportName = sAbbrLeagueName = sLeagueName = "";
		hSportCID = hLeagueCID = CID_INVALID_OBJECT;
		enLeague = LEAGUE_UNKNOWN;
		hLeagueObj = LEAGUE_INVALID_OBJECT;
		enSports = SPORTS_UNKNOWN;
		s32TableVersion = 0;
		bIsSFEnabled = FALSE;
		u8LeagueID = 0U;
		mapTeams.clear();
		setFavTeams.clear();
		enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_UNKNOWN;
	}
};

struct fc_sxm_trSFSport {
	string sAbbrSportName;
	string sSportName;
	SPORTS_ENUM enSports;
	midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType enType;
	CID_OBJECT hSportCID;

	tVoid vReset() {
		// Clear all member data
		sAbbrSportName = sSportName = "";
		enSports = SPORTS_UNKNOWN;
		enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_UNKNOWN;
		hSportCID = CID_INVALID_OBJECT;
	}
};

class Ifc_sxm_tclSportsFlashProxy {
public:
	virtual ~Ifc_sxm_tclSportsFlashProxy() { /*Virtual DTOR for the interface*/ }
	// Set audio app's instance
	virtual tVoid vSetDecoder(Ifc_sxm_tclDecoder* )  = 0;
	// Use all SMS calls specific for sports flash
	virtual SMSAPI_RETURN_CODE_ENUM enStartSportsFlash( DECODER_OBJECT,
			tUInt* ,
			SPORTS_FLASH_GAME_EVENT_CALLBACK,
			tVoid*) const = 0;
	virtual tVoid vStopSportsFlash() const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enAddFavoriteTeam(CID_OBJECT, CID_OBJECT) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enRemoveFavoriteTeam(CID_OBJECT, CID_OBJECT) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enRemoveAllFavoriteTeams() const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enIterateFavoriteTeams(SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK,
			tVoid*) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enIterateCurrentGames(SPORTS_FLASH_GAMES_ITERATOR_CALLBACK,
			tVoid*) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enSetGamesMonitor(CHANNEL_ID*,
			tU16,
			SPORTS_FLASH_FLASH_EVENT_CALLBACK,
			tVoid*) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enIterateCurrentFlashEvents(SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK,
			tVoid*) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enPlayFlashEvent(SPORTS_FLASH_EVENT_ID) const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enAbortFlashEvent() const = 0;
	virtual SPORTS_FLASH_EVENT_ID u32CurrentFlashEventID() const = 0;
	virtual SMSAPI_RETURN_CODE_ENUM enEnableNotification(tBool) const = 0;
	virtual tBool bIsNotificationEnabled() const = 0;
	// Fill Channel Attributes
	virtual tVoid vExtractChannelAttributes(CHANNEL_ID, fc_sxm_trSFChannel&) const = 0;
	// Extract Team Attributes
	virtual tBool bExtractTeamInfo(LEAGUE_OBJECT,TEAM_OBJECT, fc_sxm_trSFTeam&) const = 0;
	//Extract League Attributes
	virtual tBool bExtractLeagueInfo(LEAGUE_OBJECT, fc_sxm_trSFLeague&) const = 0;
	// Sports, League and Team Iterate content Callbacks
	virtual tBool bSportsIterateContent(SPORTS_CONTENT_ITERATOR_CALLBACK,
			tVoid*) const = 0;
	virtual tBool bLeagueIterateContent(LEAGUE_CONTENT_ITERATOR_CALLBACK,
			tVoid*) const = 0;
	virtual tBool bTeamIterateContent(LEAGUE_OBJECT,
			TEAM_CONTENT_ITERATOR_CALLBACK,
			tVoid*) const = 0;
	virtual tS16 s16CompareCIDObjects(CID_OBJECT, CID_OBJECT) const = 0;
};


#endif /* FC_SXM_TCL_SPORTS_FLASH_PROXY_IF_H_ */
