/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_flash.cpp
* @brief       Implementation for sports flash service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_sports_flash.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_generic_utils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AUDIO_SPORTS_FLASH
#include "trcGenProj/Header/fc_sxm_tcl_sports_flash.cpp.trc.h"
#endif

// One of the two asynchronous callbacks from SMS.
// THIS CB is invoked on detection of a new game event.
// enProgramStatus contains details whether a game is started or ended
tVoid fc_sxm_tclSportsFlash::cb_GameEventCallback(DECODER_OBJECT /* hDecoder */, SPORTS_FLASH_PROGRAM_STATUS_ENUM enProgramStatus,
			const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT* poGameInfo, tVoid* poEventCallBackArg)
{
	Ifc_sxm_tclAudioApp* poAudioApp = static_cast <Ifc_sxm_tclAudioApp*> (poEventCallBackArg);
	if (poAudioApp)
	{
		// Post game alert message to audio thread.
		fc_sxm_trMsgSFGameAlert oMsgGameAlert;
		oMsgGameAlert(enProgramStatus, poGameInfo);
		poAudioApp->vSFGameAlertMsg(oMsgGameAlert);
	}
}
// One of the two asynchronous callbacks from SMS.
// THIS CB is invoked on detection of a flash event.
tVoid fc_sxm_tclSportsFlash::cb_FlashEventCallback(DECODER_OBJECT /* hDecoder */, CHANNEL_ID tChannelID, SPORTS_FLASH_EVENT_ID tFlashEventID,
			SPORTS_FLASH_EVENT_STATUS_ENUM eFlashEventStatus, tVoid *poEventCallBackArg)
{
	Ifc_sxm_tclAudioApp* poAudioApp = static_cast <Ifc_sxm_tclAudioApp*> (poEventCallBackArg);
	if (poAudioApp)
	{
		// Post flash event alert message to audio thread.
		fc_sxm_trMsgSFFlashEventAlert oMsgFlashAlert(tChannelID, tFlashEventID, eFlashEventStatus);
		poAudioApp->vSFFlashEventAlertMsg(oMsgFlashAlert);
	}
}

// Sports Iterator Callback. Returning TRUE would continue iteration
tBool fc_sxm_tclSportsFlash::cb_SportsIteratorCallback(const tChar* chPacID, const tChar* chPacNam, SPORTS_ENUM enSport, CID_OBJECT hID, tVoid* poContentIteratorCBArg)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_tclSportsFlash* poSportsFlash = static_cast <fc_sxm_tclSportsFlash *> (poContentIteratorCBArg);
	if (poSportsFlash)
		bContinueIteration = poSportsFlash->bHandleSportsIteratorCB(chPacID, chPacNam, enSport, hID);
	return bContinueIteration;
}

// League Iterator Callback. Returning TRUE would continue iteration
tBool fc_sxm_tclSportsFlash::cb_LeagueIteratorCallback(LEAGUE_OBJECT hLeague, tVoid* poContentIteratorCBArg)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_tclSportsFlash* poSportsFlash = static_cast <fc_sxm_tclSportsFlash *> (poContentIteratorCBArg);
	if (poSportsFlash)
		bContinueIteration = poSportsFlash->bHandleLeagueIteratorCB(hLeague);
	return bContinueIteration;
}

// TeamIterator Callback
tBool fc_sxm_tclSportsFlash::cb_TeamIteratorCallback(TEAM_OBJECT hTeam, tVoid* poContentIteratorCBArg)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_trTeamIteratorCBArg* poTeamIteratorCBArg = static_cast <fc_sxm_trTeamIteratorCBArg *> (poContentIteratorCBArg);
	// poTeamIteratorCBArg->poLeague is the League for which the team needs to be made part of
	if ((poTeamIteratorCBArg) &&
			(poTeamIteratorCBArg->poSportsFlash) &&
			(poTeamIteratorCBArg->poLeague))
	{
		bContinueIteration = poTeamIteratorCBArg->poSportsFlash->bHandleTeamIteratorCB(poTeamIteratorCBArg->poLeague, hTeam);
	}
	return bContinueIteration;
}

// Favorites TeamIterator Callback
tBool fc_sxm_tclSportsFlash::cb_FavoritesIteratorCallback(CID_OBJECT hLeagueId, CID_OBJECT hTeamId, tVoid* poContentIteratorCBArg)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_tclSportsFlash* poSportsFlash = static_cast <fc_sxm_tclSportsFlash *> (poContentIteratorCBArg);
	if (poSportsFlash)
		bContinueIteration = poSportsFlash->bHandleFavoriteTeamIteratorCB(hLeagueId, hTeamId);
	return bContinueIteration;
}

// Current Active Games Iterator CallBack
tBool fc_sxm_tclSportsFlash::cb_ActiveGamesIteratorCallback(const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *poGameInfo, tVoid* poContentIteratorCBArg)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_tclSportsFlash* poSportsFlash = static_cast <fc_sxm_tclSportsFlash *>(poContentIteratorCBArg);
	if (poSportsFlash)
		bContinueIteration = poSportsFlash->bHandleGamesIteratorCB(poGameInfo);
	return bContinueIteration;
}

// Current Active FlashEvents Iterator Callback
tBool fc_sxm_tclSportsFlash::cb_ActiveFlashEventsIteratorCallback( CHANNEL_ID tChannelID, SPORTS_FLASH_EVENT_ID tFlashEventID, tVoid* poContentIteratorCBArg)
{
	tBool bContinueIteration = FALSE;
	fc_sxm_tclSportsFlash* poSportsFlash = static_cast <fc_sxm_tclSportsFlash *>(poContentIteratorCBArg);
	if (poSportsFlash)
		bContinueIteration = poSportsFlash->bHandleFlashEventsIteratorCB(tChannelID, tFlashEventID);
	return bContinueIteration;
}

// Default CTOR
fc_sxm_tclSportsFlash::fc_sxm_tclSportsFlash() :
_poAudioApp(OSAL_NULL), _poSFProxy(OSAL_NULL), _bIsSFInitialized(FALSE), _bEnableSF(TRUE), _bIsMetaDataChanged(FALSE), _bIsSFPlayback(FALSE), _uMaxSFChannels(0U), _channelID(CHANNEL_INVALID_ID)
{
	// CTOR
}

// Default DTOR
fc_sxm_tclSportsFlash::~fc_sxm_tclSportsFlash()
{
	// DTOR
	_poAudioApp = OSAL_NULL;
	_poSFProxy = OSAL_NULL;
}

// Method invoked for any update to the state of the decode
tVoid fc_sxm_tclSportsFlash::vUpdateDecoderState(
				DECODER_OBJECT hDecoderObject,
				DECODER_STATE_ENUM enDecoderState)
{
	// Act only if the SF is not yet initialized
	if (( FALSE == _bIsSFInitialized) &&
			(DECODER_STATE_READY == enDecoderState) &&
			(DECODER_INVALID_OBJECT != hDecoderObject) )
	{
		_bIsSFInitialized = TRUE;
		_bIsMetaDataChanged = FALSE;
		_vectSports.clear();
		_vectActiveGames.clear();
		_vectPendingAlerts.clear();
		//Start SportsFlash Service
		vNotifySFServiceState((enStartSF(hDecoderObject)), TRUE);
	}
}

// Overrides fc_sxm_tclAudioMember::vInit() method to do initilization
tVoid fc_sxm_tclSportsFlash::vInit()
{
	// Reset all member variables
	_bIsSFInitialized = FALSE;
	_bIsMetaDataChanged = FALSE;
	_bEnableSF = TRUE;
	_bIsSFPlayback = FALSE;
	_uMaxSFChannels = 0U;
	_channelID = CHANNEL_INVALID_ID;
	_vectSports.clear();
	_vectLeague.clear();
	_vectActiveGames.clear();
	_vectPendingAlerts.clear();
}

// Overrides fc_sxm_tclAudioMember::vDeInit() method to do de-init
tVoid fc_sxm_tclSportsFlash::vDeInit()
{
	// Reset all member variables
	_bIsSFInitialized = FALSE;
	_bEnableSF = TRUE;
	_uMaxSFChannels = 0U;
	_channelID = CHANNEL_INVALID_ID;
	_vectSports.clear();
	_vectLeague.clear();
	_vectActiveGames.clear();
	_vectPendingAlerts.clear();
}

//  Clear class member variables and stop sports flash service
tVoid fc_sxm_tclSportsFlash::vClearMemberData(tVoid)
{
	_bIsSFInitialized = FALSE;
	_bEnableSF = TRUE;
	_bIsSFPlayback = FALSE;
	_uMaxSFChannels = 0U;
	_channelID = CHANNEL_INVALID_ID;
	_vectSports.clear();
	_vectLeague.clear();
	_vectActiveGames.clear();
	_vectPendingAlerts.clear();
	// Stop Sports Flash
	vStopSF(FALSE);
	vResetAlerts();
}

// Set Audio App && SFProxy References
tVoid fc_sxm_tclSportsFlash::vSetApp(Ifc_sxm_tclAudioApp* poAudioApp, Ifc_sxm_tclSportsFlashProxy* poSFProxy)
{
	_poAudioApp = poAudioApp;
	_poSFProxy = poSFProxy;
	if (_poSFProxy && _poAudioApp)
		_poSFProxy->vSetDecoder(_poAudioApp);
}

// Handle MethodStart for method "SportsFlashEnableNotification"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashEnableNotification const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashEnableNotificationMethodStart const &oMStart = prMsg->oFiMsg;
	_bEnableSF = oMStart.Enable;
	SMSAPI_RETURN_CODE_ENUM enRetCode = enEnableSFNotifications(oMStart.Enable);
	// If oMStart.Enable = TRUE; then user wants to enable SF notifications, otherwise
	// don't send any notification updates
	midw_ext_sxm_audiofi_tclMsgSportsFlashEnableNotificationMethodResult oMRes;
	oMRes.Status = (tBool)((SMSAPI_RETURN_CODE_SUCCESS ==  enRetCode) ? TRUE : FALSE);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashEnableNotification enRetCode = %u && oMRes.Status = %u && _bEnableSF = %u", enRetCode, oMRes.Status, _bEnableSF));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle MethodStart for method "SportsFlashAddFavoriteTeam"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashAddFavoriteTeam const *prMsg)
{
	// Add a favorite
	midw_ext_sxm_audiofi_tclMsgSportsFlashAddFavoriteTeamMethodStart const &oMStart = prMsg->oFiMsg;
	midw_ext_sxm_audiofi_tclMsgSportsFlashAddFavoriteTeamMethodResult oMRes;
	oMRes.Status = bAddFavoriteTeam(oMStart.SportType.enType, oMStart.LeagueID, oMStart.TeamID);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashAddFavoriteTeam oMRes.Status = %u", oMRes.Status));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle MethodStart for method "SportsFlashRemoveFavoriteTeam"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashRemoveFavoriteTeam const *prMsg)
{
	// Remove a favorite team
	midw_ext_sxm_audiofi_tclMsgSportsFlashRemoveFavoriteTeamMethodStart const &oMStart = prMsg->oFiMsg;
	midw_ext_sxm_audiofi_tclMsgSportsFlashRemoveFavoriteTeamMethodResult oMRes;
	oMRes.Status = bRemoveFavoriteTeam(oMStart.SportType.enType, oMStart.LeagueID, oMStart.TeamID);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashRemoveFavoriteTeam oMRes.Status = %u", oMRes.Status));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashRemoveAllFavoriteTeams"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashRemoveAllFavoriteTeams const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashRemoveAllFavoriteTeamsMethodResult oMRes;
	// Invoke unset all Favorites
	oMRes.Status = bUnsetAllFavorites();
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashRemoveAllFavoriteTeams oMRes.Status = %u", oMRes.Status));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashGetAllFavorites"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashGetAllFavorites const *prMsg)
{
	vSFSetup();
	midw_ext_sxm_audiofi_tclMsgSportsFlashGetAllFavoritesMethodResult oMRes;
	// Method assumes that the List of favorite teams is already filled up in _vectLeague
	vFillFavoriteTeamsList(oMRes);
	//vFillFavoriteTeamsListBySort(oMRes);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashGetAllFavorites FavoritesList Size = %u", oMRes.FavouritesList.size()));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashGetCurrentGames"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashGetCurrentGames const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentGamesMethodResult oMRes;
	// Provide only the active games involving favorite teams
	vFillGamesList(oMRes);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashGetAllFavorites Current GamesList Size = %u", oMRes.Games.size()));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashSetGamesMonitor"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashSetGamesMonitor const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashSetGamesMonitorMethodStart const &oMStart = prMsg->oFiMsg;
	midw_ext_sxm_audiofi_tclMsgSportsFlashSetGamesMonitorMethodResult oMRes;
	oMRes.Status = static_cast <tBool> (SMSAPI_RETURN_CODE_SUCCESS == enSetGamesMonitor(oMStart.ChannelIDs));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashGetCurrentFlashEvents"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashGetCurrentFlashEvents const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentFlashEventsMethodResult oMRes;
	vFillFlashEventsList(oMRes);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashGetCurrentFlashEvents Current FlashEvents Size = %u", oMRes.FlashEvents.size()));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashPlayFlashEvent"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashPlayFlashEvent const *prMsg)
{
	// Extract the flash event ID
	midw_ext_sxm_audiofi_tclMsgSportsFlashPlayFlashEventMethodStart const &oMStart = prMsg->oFiMsg;
	midw_ext_sxm_audiofi_tclMsgSportsFlashPlayFlashEventMethodResult oMRes;
	oMRes.Status = static_cast <tBool> (SMSAPI_RETURN_CODE_SUCCESS == enPlayFlashEvent(oMStart.FlashEventID));
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashPlayFlashEvent Status = %u", oMRes.Status));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashAbortFlashEvent"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashAbortFlashEvent const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashAbortFlashEventMethodResult oMRes;
	oMRes.Status = static_cast <tBool> (SMSAPI_RETURN_CODE_SUCCESS == enAbortFlashEvent());
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashAbortFlashEvent Status = %u", oMRes.Status));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashGetCurrentFlashEventID"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashGetCurrentFlashEventID const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentFlashEventIDMethodResult oMRes;
	oMRes.FlashEventID = u32CurrentFlashEventID();
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashGetCurrentFlashEventID EventID = %u", oMRes.FlashEventID));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashEnabledLeagues"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashEnabledLeagues const *prMsg)
{
	vSFSetup();
	// Send only those leagues which support sport flash feature
	midw_ext_sxm_audiofi_tclMsgSportsFlashEnabledLeaguesMethodResult oMRes;
	vFillSFEnabledLeagueList(oMRes);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashEnabledLeagues LeaguesSize = %u", oMRes.Leagues.size()));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashGetTeams"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashGetTeams const *prMsg)
{
	midw_ext_sxm_audiofi_tclMsgSportsFlashGetTeamsMethodStart const &oMStart = prMsg->oFiMsg;
	// Send only teams that provide sports flash support
	midw_ext_sxm_audiofi_tclMsgSportsFlashGetTeamsMethodResult oMRes;
	//vFillTeamList((poFindLeaguebyLID(oMStart.LeagueID)), oMRes);
	vFillTeamListBySort((poFindLeaguebyLID(oMStart.LeagueID)), oMRes);
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashEnabledLeagues TeamsSize = %u", oMRes.Teams.size()));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle method start for method "SportsFlashMonitorTeam"
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioMStartSportsFlashMonitorTeam const *prMsg)
{
	vSFSetup();
	midw_ext_sxm_audiofi_tclMsgSportsFlashMonitorTeamsMethodStart const &oMStart = prMsg->oFiMsg;
	midw_ext_sxm_audiofi_tclMsgSportsFlashMonitorTeamsMethodResult oMRes;
	vector<midw_ext_fi_tcl_SportsFlash_MonitorTeam, std::allocator<midw_ext_fi_tcl_SportsFlash_MonitorTeam> >::const_iterator cIt =
			oMStart.FavoriteTeams.begin();
	size_t tCount = 0;
	while (cIt != oMStart.FavoriteTeams.end())
	{
		if (bUpdateMonitorStatus(cIt->LeagueID, cIt->TeamID, cIt->Status))
		{
			// inCrease u8Count by one for each succesfull updation
			++tCount;
		}
		// Continue iteration
		++cIt;
	}
	enSetGamesMonitor();
	oMRes.Status = static_cast <tBool> ( tCount == oMStart.FavoriteTeams.size());
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioMStartSportsFlashMonitorTeam count = %u  oMRes.Status = %u", tCount, oMRes.Status));
	vSendSFFiMsg(prMsg->rAdressing, oMRes);
}

// Handle notification message for game alert posted by SMS and routed through AudioApp
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgSFGameAlert const *prMsg)
{
	vSFSetup();
	if (prMsg)
	{
		ETG_TRACE_USR4(("fc_sxm_trMsgSFGameAlert ProgramStatus = %u", prMsg->enProgramStatus));
		// Handle game event callback... The method could be invoked if a game is started or is finished
		vHandleGameEventCB(prMsg->enProgramStatus, &prMsg->trGameInfo);
	}
}

// Handle notification message for flash event alert posted by SMS and routed through AudioApp
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgSFFlashEventAlert const *prMsg)
{
	if (prMsg)
	{
		ETG_TRACE_USR4(("fc_sxm_trMsgSFFlashEventAlert ChannelID = %u \t EventID = %u \t EventStatus = %u",
				prMsg->tChannelID, prMsg->tFlashEventID, prMsg->eFlashEventStatus));
		vHandleFlashEventCB(prMsg->tChannelID, prMsg->tFlashEventID, prMsg->eFlashEventStatus);
	}
}

// Notify Client about the sports flash service state updates
// bStartService = TRUE, then a request to Start SPOrts flash service was done
tVoid fc_sxm_tclSportsFlash::vNotifySFServiceState(SMSAPI_RETURN_CODE_ENUM enRetCode,tBool bStartService)
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertySportsFlash_ServiceStatus oProperty;
		switch (enRetCode)
		{
			case SMSAPI_RETURN_CODE_SUCCESS:
			{
				// Service successfully started
				oProperty.oFiMsg.ServiceStatus.enType = (bStartService)?
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_STARTED :
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_STOPPED ;
				break;
			}
			case SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED:
			{
				// TW_Now service is already started for the Decoder object associated with service
				oProperty.oFiMsg.ServiceStatus.enType =
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_ALREADY_STARTED;
				break;
			}
			case SMSAPI_RETURN_CODE_ERROR:
			default:
			{
				oProperty.oFiMsg.ServiceStatus.enType =
						midw_ext_fi_tcl_e8_FlashServiceStatus::FI_EN_FLASH_SERVICE_ERROR;
				break;
			}
		}
		ETG_TRACE_USR1(("vNotifySFServiceStartState \t bStartService(%u) \t enType(%u) \t enReturnCode(%u)",
				bStartService, oProperty.oFiMsg.ServiceStatus.enType,
				ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode) ));
		_poAudioApp->vSFlashServiceStatus(oProperty);
	}
}

// Wrapper method to start SF servic
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enStartSF(DECODER_OBJECT hDecoder)
{
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if(_poSFProxy)
	{
		enReturnCode = _poSFProxy->enStartSportsFlash(hDecoder,
						&_uMaxSFChannels,
						fc_sxm_tclSportsFlash::cb_GameEventCallback,
						static_cast <tVoid *>(_poAudioApp));
		_bIsSFInitialized = TRUE;
	}
	ETG_TRACE_USR3(("fc_sxm_tclSportsFlash::enStartSF enReturnCode(%u)", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
	return enReturnCode;
}

// Stop SportsFlash service and release the object handle.
tVoid fc_sxm_tclSportsFlash::vStopSF(tBool bIsBroadscopeChange)
{
	if(_poSFProxy)
	{
		_poSFProxy->vStopSportsFlash();
		if (bIsBroadscopeChange)
		{
			ETG_TRACE_USR2(("Broadscope change is in effect... Setting service state to ERROR"));
			vNotifySFServiceState(SMSAPI_RETURN_CODE_ERROR, FALSE);
		}
		else
		{
			ETG_TRACE_USR2(("Stop SportsFlash Service"));
			vNotifySFServiceState(SMSAPI_RETURN_CODE_SUCCESS, FALSE);
		}
	}
}

// Setup Method for Sports Flash service;
// This method needs to be invoked on first game event notification received by the application
tVoid fc_sxm_tclSportsFlash::vSFSetup()
{
	if (0 == _vectLeague.size())
	{
		ETG_TRACE_USR3(("fc_sxm_tclSportsFlash::vSFSetup _vectLeague.size = 0"));
		// Collect sports, leagues, teams and favorite teams information
		(tVoid) bCollectSports();
		(tVoid) bCollectLeagues();
		(tVoid) bCollectTeams();
		(tVoid) bCollectFavorites();
		_vectActiveGames.clear();
	}
}

// Handle a new game event reported by SMS;
tVoid fc_sxm_tclSportsFlash::vHandleGameEventCB(SPORTS_FLASH_PROGRAM_STATUS_ENUM enProgramStatus, const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT* poGameInfo)
{
	if (poGameInfo)
	{
		fc_sxm_trFlashGameEvent trGameEvent;
		// Extract game details from the reported poGameInfo value
		vExtractGameEventDetails(poGameInfo, trGameEvent);
		// Based on the addition or deletion of program, add or remove the game from ActiveGamesLIst
		tBool bModifyStatus = (SPORTS_FLASH_PROGRAM_AVAILABLE == enProgramStatus) ?
				bAddActiveGame(trGameEvent) :
				bRemoveActiveGame(trGameEvent);
		ETG_TRACE_USR3(("fc_sxm_tclSportsFlash::vHandleGameEventCB bModifyStatus = %u \t ProgramStatus = %u", bModifyStatus, enProgramStatus));
		if (TRUE == bModifyStatus)
		{
			// Send notification to HMI
			vNotifyGameEventUpdate(enProgramStatus, trGameEvent);
			enSetGamesMonitor();
		}
	}
}

// Handle a new flash event reported by SMS
tVoid fc_sxm_tclSportsFlash::vHandleFlashEventCB(CHANNEL_ID tChannelID, SPORTS_FLASH_EVENT_ID tFlashEventID, SPORTS_FLASH_EVENT_STATUS_ENUM eFlashEventStatus)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::vHandleFlashEventCB ChannelID(%u) \t eventID(%u) \t status(%u)", tChannelID, tFlashEventID, eFlashEventStatus));
	// A flash event can happen only involving favorite teams that are currently playing a live game
	SXM_FOREACH(vector<fc_sxm_trFlashGameEvent>, gameIter, _vectActiveGames)
	{
		if (tChannelID == gameIter->trChannel.tChannelID)
		{
			ETG_TRACE_USR3(("GAME for the reported flash event Identified"));
			if ((gameIter->trAwayTeam.bIsMonitored) && (gameIter->trHomeTeam.bIsMonitored))
			{
				ETG_TRACE_USR3(("Teams involved in reported flash event are monitored"));
				vRetrieveChannelInfo(tChannelID, gameIter->trChannel);
				gameIter->vUpdateEventQueue(tFlashEventID, eFlashEventStatus);
				vNotifyFlashEventUpdate(*gameIter, tFlashEventID, eFlashEventStatus);
			}
			else
			{
				ETG_TRACE_USR3(("Teams involved in reported flash event is/are NOT monitored (%u) (%u)",
						gameIter->trAwayTeam.bIsMonitored, gameIter->trHomeTeam.bIsMonitored));
			}
			break;
		}
	}
}

// Method is invoked on receipt of Sports Iteration CB being invoked
tBool fc_sxm_tclSportsFlash::bHandleSportsIteratorCB(const tChar* chPacID, const tChar* chPacNam, SPORTS_ENUM enSport, CID_OBJECT hID)
{
	fc_sxm_trSFSport trSport;
	trSport.sAbbrSportName = chPacID;
	trSport.sSportName = chPacNam;
	trSport.enSports = enSport;
	trSport.hSportCID = hID;
	trSport.enType = fc_sxm_tclSportsFlash::enGetSportType(enSport);
	_vectSports.push_back(trSport);
	return TRUE;
}

// Send property update for new game event
tVoid fc_sxm_tclSportsFlash::vNotifyGameEventUpdate(SPORTS_FLASH_PROGRAM_STATUS_ENUM enProgramStatus, const fc_sxm_trFlashGameEvent& trGame)
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification oProperty;
		oProperty.oFiMsg.ProgramStatus.enType = static_cast <midw_ext_fi_tcl_e8_SportsFlash_ProgramStatus::tenType> (enProgramStatus);
		ETG_TRACE_USR3(("Send property update New Game status (%u) at Channel (%u)", enProgramStatus, trGame.trChannel.tChannelID));
		vFillGameDetails(trGame, oProperty.oFiMsg.GameInfo);
		_poAudioApp->vSFlashGameEventNotify(oProperty);
	}
}

// Send property update for new flash event
tVoid fc_sxm_tclSportsFlash::vNotifyFlashEventUpdate(const fc_sxm_trFlashGameEvent& trGame,
		SPORTS_FLASH_EVENT_ID tFlashEventID,
		SPORTS_FLASH_EVENT_STATUS_ENUM eFlashEventStatus)
{
	if ((_poAudioApp) && (_bEnableSF) && (trGame.trChannel.bChannelStatus()))
	{
		fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification oProperty;
		oProperty.oFiMsg.FlashEventID = tFlashEventID;
		ETG_TRACE_USR3(("fc_sxm_tclSportsFlash::vNotifyFlashEventUpdate eventID = %u \t eventStatus = %u",
				tFlashEventID, eFlashEventStatus));
		vFillGameDetails(trGame, oProperty.oFiMsg.GameInfo);
		oProperty.oFiMsg.FlashEventStatus.enType = static_cast <midw_ext_fi_tcl_e8_SportsFlash_FlashEventStatus::tenType> (eFlashEventStatus);
		if (TRUE == bNotifyFlashEvent(trGame.trChannel.tChannelID))
		{
			ETG_TRACE_USR3(("I. Send property update that a new status for the flash event has happened at Ch %u",
					trGame.trChannel.tChannelID));
			_poAudioApp->vSFlashFlashEventNotify(oProperty);
		}
		else if (FALSE == bUpdatePendingAlertsList(oProperty))
		{
			ETG_TRACE_USR3(("II. Send property update that a new status for the flash event has happened at Ch %u",
					trGame.trChannel.tChannelID));
			_poAudioApp->vSFlashFlashEventNotify(oProperty);
		}
	}
}

// Method to convert SPORTS_ENUM reported by SMS to corresponding FI enum type
midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType fc_sxm_tclSportsFlash::enGetSportType(SPORTS_ENUM enSport)
{
	midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType enType;
	switch(enSport)
	{
		case SPORTS_FOOTBALL:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_FOOTBALL;
			break;
		}
		case SPORTS_HOCKEY:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_HOCKEY;
			break;
		}
		case SPORTS_BASKETBALL:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_BASKETBALL;
			break;
		}
		case SPORTS_BASEBALL:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_BASEBALL;
			break;
		}
		case SPORTS_SOCCER:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_SOCCER;
			break;
		}
		case SPORTS_AUTO_RACING:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_AUTO_RACING;
			break;
		}
		case SPORTS_OTHER:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_OTHER;
			break;
		}
		case SPORTS_UNKNOWN:
		default:
		{
			enType = midw_ext_fi_tcl_e8_SportsFlash_Sport::FI_EN_SPORTS_UNKNOWN;
			break;
		}
	}
	return enType;
}

// Add a new league to the internal list
tBool fc_sxm_tclSportsFlash::bHandleLeagueIteratorCB(LEAGUE_OBJECT hLeague)
{
	tBool bIsSFEnabled = FALSE;
	fc_sxm_trSFLeague trSFLeague;
	// Information from League is extracted by calling method bExtractLeagueInfo
	if (_poSFProxy)
		bIsSFEnabled = _poSFProxy->bExtractLeagueInfo(hLeague, trSFLeague);
	// Add only SF supported leagues
	if(bIsSFEnabled)
	{
		ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bHandleLeagueIteratorCB--> LeagueName = %s",
					trSFLeague.sAbbrLeagueName.c_str()));
		// Information for sport can only be retrieved by calling vFillSportsInfo
		vFillSportsInfo(trSFLeague);
		_vectLeague.push_back(trSFLeague);
	}
	return TRUE;
}

// Add a new team to the given league
tBool fc_sxm_tclSportsFlash::bHandleTeamIteratorCB(fc_sxm_trSFLeague* poSFLeague, TEAM_OBJECT hTeam) const
{
	if (poSFLeague && _poSFProxy)
	{
		fc_sxm_trSFTeam trTeam;
		_poSFProxy->bExtractTeamInfo(poSFLeague->hLeagueObj, hTeam, trTeam);
		// Add team to the league irrespective of whether the team supports Sports Flash or not
		// There are leagues where some teams support Sports Flash and some other teams within league which do not support sports flash.
		// If only sports flash supported teams are added, we may not have the necessary data if a sports flash event is available for
		// game involving SF supported team (vs) SF not supported team
		poSFLeague->mapTeams[trTeam.u16TeamID] = trTeam;
	}
	return TRUE;
}

// Method to set a team reported by SMS as a favorite
tBool fc_sxm_tclSportsFlash::bHandleFavoriteTeamIteratorCB(CID_OBJECT hLeagueId, CID_OBJECT hTeamId)
{
	fc_sxm_trSFLeague* trLeague(poFindLeaguebyCID(hLeagueId));
	(tVoid) bSetFavoritebyCID(trLeague, hTeamId);
	return TRUE;
}

// Method to add a game to the list of active internal games
tBool fc_sxm_tclSportsFlash::bHandleGamesIteratorCB(const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *poGameInfo)
{
	tBool bSuccess = FALSE;
	if (poGameInfo)
	{
		// Add an active game to the list
		bSuccess = TRUE;
		fc_sxm_trFlashGameEvent trGame;
		vExtractGameEventDetails(poGameInfo, trGame);
		(tVoid) bAddActiveGame(trGame);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bHandleGamesIteratorCB bSuccess = %u", bSuccess));
	return bSuccess;
}

// Method to set flash event ID and Flash event status to the list of active games
tBool fc_sxm_tclSportsFlash::bHandleFlashEventsIteratorCB( CHANNEL_ID tChannelID, SPORTS_FLASH_EVENT_ID tFlashEventID )
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bHandleFlashEventsIteratorCB tChannelID = %u \t tFlashEventID = %u", tChannelID, tFlashEventID));
	SXM_FOREACH(vector<fc_sxm_trFlashGameEvent>, gameIter, _vectActiveGames)
	{
		if (tChannelID == gameIter->trChannel.tChannelID)
		{
			//gameIter->tFlashEventID = tFlashEventID;
			//gameIter->enFlashEventStatus = SPORTS_FLASH_EVENT_OCCURRED;
			break;
		}
	}
	return TRUE;
}

/*
 * Method decides which team needs to be monitored for the condition
 * When the product detects more than one live game broadcast channel involving the same selected team, as per UIRR "SFLASH_012",
 * only one of those channels needs to be monitored.
 *
 * For example: IN League NFL, if there is live game between teams, Arizona Cardinals(ARI) and Atlanta Falcons(ATL),
 * and the same information is broadcasted in multiple channels, this method helps deciding which of those channels needs to be monitored
 * and which of those needs to discarded from monitoring.
 */

/*        ----------------------------------------------------------------------------------------------
 *        |    Participating Teams         |      Sports Broadcast Bias/ Game Selection Priority**     |
 *        ----------------------------------------------------------------------------------------------
 *        |  Is Home Team |  Is Away Team  |      0      |       1     |         2     |       3       |
 *        |  a Favorite?  |   a Favorite?  | (National)  | (Home Bias) |   (Away Bias) |  (Alternate)  |
 *        ----------------------------------------------------------------------------------------------
 *        |     Yes       |       No       |      B      |       A     |       C       |      D        |
 *        ----------------------------------------------------------------------------------------------
 *        |     No        |      Yes       |      B      |       C     |       A       |      D        |
 *        ----------------------------------------------------------------------------------------------
 *        |     Yes       |      Yes       |      C      |       A     |       B       |      D        |
 *        ----------------------------------------------------------------------------------------------
 *
 *        ** - These values correspond to the Bias values reported from SMS thru GameEvent CB
  */

fc_sxm_tclSportsFlash::enActiveGameBiasState fc_sxm_tclSportsFlash::enCalculateBias(
		const fc_sxm_trFlashGameEvent& trExistingGameEntry,
		const fc_sxm_trFlashGameEvent& trNewGameEntry) const
{
	// All details within struct "fc_sxm_trFlashGameEvent" are expected to be the same for
	// "trExistingGameEntry" and "trNewGameEntry" except the Bias value
	tBool bIsHomeTeamFavorite = trExistingGameEntry.trHomeTeam.bIsFavorite;
	tBool bIsAwayTeamFavorite = trExistingGameEntry.trAwayTeam.bIsFavorite;

	const tU16 u16PriorityLookUp[3][4] = { { 1, 0, 2, 3}, { 2, 0, 1, 3}, { 1, 2, 0, 3} } ;
	tU8 u8Index;
	if ((bIsHomeTeamFavorite) && (!bIsAwayTeamFavorite))
		u8Index = 0U;
	else if ((!bIsHomeTeamFavorite) && (bIsAwayTeamFavorite))
		u8Index = 1U;
	else if ((bIsHomeTeamFavorite) && (bIsAwayTeamFavorite))
		u8Index = 2U;
	else
		return enActiveGameBiasState_Discard_NewGameEntry;

	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enCalculateBias.... bIsHomeTeamFavorite(%u) \t bIsAwayTeamFavorite(%u) \t u8Index(%u)",
			bIsHomeTeamFavorite,bIsAwayTeamFavorite, u8Index));
	ETG_TRACE_USR4(("trExistingGameEntry ---> ChannelID(%u) \t Bias(%u) \t HashCode(%u)",
			trExistingGameEntry.trChannel.tChannelID, trExistingGameEntry.u16Bias, trExistingGameEntry.u32HashCode));
	ETG_TRACE_USR4(("trNewGameEntry ---> ChannelID(%u) \t Bias(%u) \t HashCode(%u)",
			trNewGameEntry.trChannel.tChannelID, trNewGameEntry.u16Bias, trNewGameEntry.u32HashCode));

	enActiveGameBiasState enBias = enActiveGameBiasState_Unknown_Action;
	tU8 u8Order = 0U;

	do {
		if (trExistingGameEntry.u16Bias == u16PriorityLookUp[u8Index][u8Order])
			enBias = enActiveGameBiasState_Discard_NewGameEntry;
		else if (trNewGameEntry.u16Bias == u16PriorityLookUp[u8Index][u8Order])
			enBias = enActiveGameBiasState_Replace_NewGameEntry;
		++u8Order;
	} while ((enBias == enActiveGameBiasState_Unknown_Action) && (u8Order < 4U));

	if (enBias == enActiveGameBiasState_Unknown_Action)
		enBias = enActiveGameBiasState_Discard_NewGameEntry;

	ETG_TRACE_USR4(("enCalculateBias Result = %u", ETG_CENUM(fc_sxm_tclSportsFlash::enActiveGameBiasState, enBias)));
	return enBias;
}

// Add a game to the list of active games
tBool fc_sxm_tclSportsFlash::bAddActiveGame(const fc_sxm_trFlashGameEvent& trGame)
{
	SXM_IF_FIND(vector<fc_sxm_trFlashGameEvent>, gameIter, _vectActiveGames, trGame)
	{
		// Check for Bias
		if (enActiveGameBiasState_Replace_NewGameEntry == enCalculateBias(*gameIter, trGame))
		{
			_vectActiveGames.erase(gameIter);
			ETG_TRACE_USR4(("Replacing existing game entry trGame.tChannelID = %u", trGame.trChannel.tChannelID));
			_vectActiveGames.push_back(trGame);
		}
	}
	else
	{
		ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bAddActiveGame trGame.tChannelID = %u", trGame.trChannel.tChannelID));
		_vectActiveGames.push_back(trGame);
	}
	return TRUE;
}

// Remove game from active games list
tBool fc_sxm_tclSportsFlash::bRemoveActiveGame(const fc_sxm_trFlashGameEvent& trGame)
{
	// Remove game as and when it is reported from SMS that the game has ended
	tBool bGameRemoveStatus = FALSE;
	SXM_FOREACH(vector<fc_sxm_trFlashGameEvent>, gameIter, _vectActiveGames)
	{
		if (trGame.trChannel.tChannelID == gameIter->trChannel.tChannelID)
		{
			ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bRemoveActiveGame trGame.tChannelID = %u", trGame.trChannel.tChannelID));
			bGameRemoveStatus = TRUE;
			_vectActiveGames.erase(gameIter);
			break;
		}
	}
	return bGameRemoveStatus;
}

// Fill only the sports details to the trSFLeague structure passed
tVoid fc_sxm_tclSportsFlash::vFillSportsInfo(fc_sxm_trSFLeague& trSFLeague) const
{
	vector<fc_sxm_trSFSport>::const_iterator cSportsIter = _vectSports.begin();
	while (cSportsIter != _vectSports.end())
	{
		if (cSportsIter->enSports == trSFLeague.enSports)
		{
			trSFLeague.sAbbrSportName = cSportsIter->sAbbrSportName;
			trSFLeague.sSportName = cSportsIter->sSportName;
			trSFLeague.hSportCID = cSportsIter->hSportCID;
			trSFLeague.enType = cSportsIter->enType;
			break;
		}
		// Iterate until all sports are iterared
		++cSportsIter;
	}
}

// Wrapper method to invoke sports iterator callback from SMS
tBool fc_sxm_tclSportsFlash::bCollectSports()
{
	_vectSports.clear();
	return((_poSFProxy)?
			_poSFProxy->bSportsIterateContent(fc_sxm_tclSportsFlash::cb_SportsIteratorCallback, static_cast <tVoid *>(this)) :
			FALSE);
}
// Wrapper method to invoke League iterator callback from SMS
tBool fc_sxm_tclSportsFlash::bCollectLeagues()
{
	_vectLeague.clear();
	return((_poSFProxy)?
				_poSFProxy->bLeagueIterateContent(fc_sxm_tclSportsFlash::cb_LeagueIteratorCallback, static_cast <tVoid *>(this)) :
				FALSE);
}
// Wrapper method to invoke Team iterator callback from SMS
tBool fc_sxm_tclSportsFlash::bCollectTeams()
{
	if (_poSFProxy)
	{
		// Collect teams from every league
		vector<fc_sxm_trSFLeague>::iterator leagueIter = _vectLeague.begin();
		while(leagueIter != _vectLeague.end())
		{
			fc_sxm_trTeamIteratorCBArg trTeamIteratorCBArg;
			trTeamIteratorCBArg.poSportsFlash = this;
			trTeamIteratorCBArg.poLeague = &(*leagueIter);
			_poSFProxy->bTeamIterateContent(leagueIter->hLeagueObj, fc_sxm_tclSportsFlash::cb_TeamIteratorCallback, static_cast <tVoid *>(&trTeamIteratorCBArg));
			ETG_TRACE_USR4(("Completed Collected teams belonging to league %s", leagueIter->sLeagueName.c_str()));
			++leagueIter;
		}
	}
	return TRUE;
}

// Wrapper method to collect all Favorite teams
tBool fc_sxm_tclSportsFlash::bCollectFavorites()
{
	tBool bRetValue = FALSE;
	if ((_poSFProxy) &&
			(SMSAPI_RETURN_CODE_SUCCESS ==
			_poSFProxy->enIterateFavoriteTeams(fc_sxm_tclSportsFlash::cb_FavoritesIteratorCallback,static_cast <tVoid *>(this))))
	{
		bRetValue = TRUE;
	}
	ETG_TRACE_USR4(("Completed Iterating Favorite Teams, status = %u", bRetValue));
	return bRetValue;
}

// Utility method to find a League from internal list through CID_OBJECT value of the league
fc_sxm_trSFLeague* fc_sxm_tclSportsFlash::poFindLeaguebyCID(CID_OBJECT hLeagueId)
{
	vector<fc_sxm_trSFLeague>::iterator leagueIter = _vectLeague.begin();
	while(leagueIter != _vectLeague.end())
	{
		// s16CompareCIDObjects returns 0 if the CID objects match
		if (0 == s16CompareCIDObjects(hLeagueId, leagueIter->hLeagueCID))
		{
			ETG_TRACE_USR3(("LeagueFound --> LeagueName = %s", leagueIter->sLeagueName.c_str()));
			return (&(*leagueIter));
		}
		++leagueIter;
	}
	return (OSAL_NULL);
}

// Utility method to find a Team within League from internal list through CID_OBJECT value of the team
fc_sxm_trSFTeam* fc_sxm_tclSportsFlash::poFindTeambyCID(fc_sxm_trSFLeague* poLeague, CID_OBJECT hTeamId) const
{
	if (poLeague)
	{
		map<tU16, fc_sxm_trSFTeam>::iterator leagueIter = poLeague->mapTeams.begin();
		while (leagueIter != poLeague->mapTeams.end())
		{
			if (0 == s16CompareCIDObjects(hTeamId, leagueIter->second.hTeamCID))
			{
				ETG_TRACE_USR3(("TeamFound --> TeamAbbrName = %s", leagueIter->second.sTeamAbbrName.c_str()));
				// Team found; Return immediately
				return (&(leagueIter->second));
			}
			++leagueIter;
		}
	}
	return (OSAL_NULL);
}

// Utility method to find a League from internal list through LeagueID value of the league
fc_sxm_trSFLeague* fc_sxm_tclSportsFlash::poFindLeaguebyLID(const tU8 u8LeagueID)
{
	vector<fc_sxm_trSFLeague>::iterator leagueIter = _vectLeague.begin();
	while(leagueIter != _vectLeague.end())
	{
		if (leagueIter->u8LeagueID == u8LeagueID)
		{
			ETG_TRACE_USR3(("LeagueFound --> LeagueName = %s", leagueIter->sLeagueName.c_str()));
			// League Found;
			return (&(*leagueIter));
		}
		++leagueIter;
	}
	return (OSAL_NULL);
}

// Utility method to find a Team within League from internal list through TeamID value of the team
fc_sxm_trSFTeam* fc_sxm_tclSportsFlash::poFindTeambyTID(fc_sxm_trSFLeague* poLeague, const tU16 u16TeamId) const
{
	if (poLeague)
	{
		SXM_IF_FIND_MAP(tU16, fc_sxm_trSFTeam, teamIt, poLeague->mapTeams, u16TeamId)
		{
			ETG_TRACE_USR3(("TeamFound --> TeamAbbrName = %s", teamIt->second.sTeamAbbrName.c_str()));
			// Team with given teamID identified
			return (&(teamIt->second));
		}
	}
	return (OSAL_NULL);
}

// Method sets a team as favorite through the CID object passed
tBool fc_sxm_tclSportsFlash::bSetFavoritebyCID(fc_sxm_trSFLeague* poLeague, CID_OBJECT hTeamId) const
{
	// Method to be invoked when SMS enquiry for iterating through the list of favorite teams is made
	tBool bStatus = FALSE;
	if (poLeague)
	{
		map<tU16, fc_sxm_trSFTeam>::iterator teamIt = poLeague->mapTeams.begin();
		while (teamIt != poLeague->mapTeams.end())
		{
			if (0 == s16CompareCIDObjects(hTeamId, teamIt->second.hTeamCID))
			{
				ETG_TRACE_USR3(("SET SF FAVORITE BY CID -- Success"));
				teamIt->second.bIsFavoriteTeam = bStatus = TRUE;
				teamIt->second.bIsMonitored = TRUE;
				poLeague->setFavTeams.insert(teamIt->second.u16TeamID);
				// The favorites flag for the team is set to true and the entry is added to be part of
				// set of favorite teams
				break;
			}
			++teamIt;
		}
	}
	return bStatus;
}

// Method unsets a team from favorite-list through the CID object passed
tBool fc_sxm_tclSportsFlash::bUnSetFavoritebyCID(fc_sxm_trSFLeague* poLeague, CID_OBJECT hTeamId) const
{
	tBool bStatus = FALSE;
	if (poLeague)
	{
		map<tU16, fc_sxm_trSFTeam>::iterator teamIt = poLeague->mapTeams.begin();
		while (teamIt != poLeague->mapTeams.end())
		{
			if (0 == s16CompareCIDObjects(hTeamId, teamIt->second.hTeamCID))
			{
				ETG_TRACE_USR3(("UNSET SF FAVORITE BY CID -- Success"));
				teamIt->second.bIsFavoriteTeam = FALSE;
				teamIt->second.bIsMonitored = TRUE;
				poLeague->setFavTeams.erase(teamIt->second.u16TeamID);
				bStatus = TRUE;
				// The favorites flag for the team is set to true and the entry is deleted from the
				// set of favorite teams
				break;
			}
			++teamIt;
		}
	}
	return bStatus;
}

// Method sets a team as favorite through the teamID passed
tBool fc_sxm_tclSportsFlash::bSetFavoritebyTID(fc_sxm_trSFLeague* poLeague, const tU16 u16TeamId) const
{
	tBool bStatus = FALSE;
	// Method to be invoked when HMI makes a choice of setting a team as favorite
	if (poLeague && _poSFProxy)
	{
		SXM_IF_FIND_MAP(tU16, fc_sxm_trSFTeam, teamIt, poLeague->mapTeams, u16TeamId)
		{
			ETG_TRACE_USR3(("SET SF FAVORITE BY TID -- Success"));
			if (SMSAPI_RETURN_CODE_SUCCESS == _poSFProxy->enAddFavoriteTeam(poLeague->hLeagueCID, teamIt->second.hTeamCID))
			{
				teamIt->second.bIsFavoriteTeam = TRUE;
				teamIt->second.bIsMonitored = TRUE;
				poLeague->setFavTeams.insert(teamIt->second.u16TeamID);
				bStatus = TRUE;
			}
		}
	}
	return bStatus;
}

// Method unsets a team from favorite-list through the TeamID passed
tBool fc_sxm_tclSportsFlash::bUnSetFavoritebyTID(fc_sxm_trSFLeague* poLeague, const tU16 u16TeamId) const
{
	tBool bStatus = FALSE;
	// Method to be invoked when HMI makes a choice of un-setting a team as favorite
	if (poLeague && _poSFProxy)
	{
		SXM_IF_FIND_MAP(tU16, fc_sxm_trSFTeam, teamIt, poLeague->mapTeams, u16TeamId)
		{
			if (SMSAPI_RETURN_CODE_SUCCESS == _poSFProxy->enRemoveFavoriteTeam(poLeague->hLeagueCID, teamIt->second.hTeamCID))
			{
				ETG_TRACE_USR3(("UNSET SF FAVORITE BY TID -- Success"));
				teamIt->second.bIsFavoriteTeam = FALSE;
				teamIt->second.bIsMonitored = TRUE;
				poLeague->setFavTeams.erase(teamIt->second.u16TeamID);
				bStatus = TRUE;
			}
		}
	}
	return bStatus;
}

// Method unsets all favorites
tBool fc_sxm_tclSportsFlash::bUnsetAllFavorites()
{
	// Step 1: Instruct SMS to remove all favorites
	tBool bStatus = bRemoveAllFavorites();
	if (bStatus)
	{
		// Step 2: Clear internal list of favorites if SMS return code is successful
		vClearFavoritesList();
		// Step 3: Re-poll for active games
		(tVoid) enIterateCurrentGames();
		// Step 4: Re-set games monitor to NULL
		(tVoid) enSetGamesMonitor();
	}
	return bStatus;
}

// Method invokes SMS API call to remove all favorite teams stored in sms.cfg file
tBool fc_sxm_tclSportsFlash::bRemoveAllFavorites() const
{
	tBool bSuccess = FALSE;
	// Request SMS to clear its internal cache of favorite teams
	if (_poSFProxy)
		bSuccess = static_cast <tBool>
				(SMSAPI_RETURN_CODE_SUCCESS == _poSFProxy->enRemoveAllFavoriteTeams());
	return bSuccess;
}

// Utility method to clear the internal list of favorites
tVoid fc_sxm_tclSportsFlash::vClearFavoritesList()
{
	// Iterate thru the list of leagues;
	SXM_FOREACH(vector<fc_sxm_trSFLeague>, leagueIt, _vectLeague)
	{
		// Get set of favorite teams within the league;
		SXM_FOREACH(set<tU16>, favTeamIt, leagueIt->setFavTeams)
		{
			// unset Favorites bit of those teams
			SXM_IF_FIND_MAP(tU16, fc_sxm_trSFTeam, teamIt, leagueIt->mapTeams, *favTeamIt)
			{
				teamIt->second.bIsFavoriteTeam = FALSE;
				// Reset the monitor flag to true before marking a team as non-favorite
				teamIt->second.bIsMonitored = TRUE;
			}
		}
		// Clear the set of favorite teams belonging to the league
		leagueIt->setFavTeams.clear();
	}
}

// Method for changing the Monitor status
tBool fc_sxm_tclSportsFlash::bSetTeamMonitorStatusbyTID(fc_sxm_trSFLeague* poLeague, const tU16 u16TeamId, tBool bMonitorStatus) const
{
	tBool bStatus = FALSE;
	// Method to be invoked when HMI makes a choice of changing a favorite team's monitor status
	if (poLeague)
	{
		SXM_IF_FIND_MAP(tU16, fc_sxm_trSFTeam, teamIt, poLeague->mapTeams, u16TeamId)
		{
			ETG_TRACE_USR3(("TeamID (%u)'s Monitor status successfully set to (%u)", u16TeamId, bMonitorStatus));
			teamIt->second.bIsMonitored = bMonitorStatus;
			bStatus = TRUE;
		}
	}
	return bStatus;
}

// Method fills in all flash enabled leagues to appropriate FI structures
tVoid fc_sxm_tclSportsFlash::vFillSFEnabledLeagueList(midw_ext_sxm_audiofi_tclMsgSportsFlashEnabledLeaguesMethodResult& oMRes) const
{
	oMRes.Leagues.clear();
	SXM_FOREACH_CONST(vector<fc_sxm_trSFLeague>, cLeagueIt, _vectLeague)
	{
		midw_ext_fi_tcl_SportsFlash_League fiLeague;
		vFillLeagueInfo(*cLeagueIt, fiLeague);
		oMRes.Leagues.push_back(fiLeague);
	}
}

// Method fills in one flash enabled league' details to appropriate FI structure fiLeague
tVoid fc_sxm_tclSportsFlash::vFillLeagueInfo(const fc_sxm_trSFLeague& trLeague, midw_ext_fi_tcl_SportsFlash_League& fiLeague) const
{
	fiLeague.SportType.enType = trLeague.enType;
	fiLeague.LeagueID = trLeague.u8LeagueID;
	fiLeague.LeagueType.enType = static_cast <midw_ext_fi_tcl_e8_SportsFlash_League::tenType> (trLeague.enLeague);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trLeague.sAbbrLeagueName.c_str(), fiLeague.LeagueAbbrName);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trLeague.sLeagueName.c_str(), fiLeague.LeagueFullName);
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::vFillLeagueInfo LeagueID(%u) \t LeagueType.enType(%u)", fiLeague.LeagueID, fiLeague.LeagueType.enType));
	ETG_TRACE_USR4(("trLeague.sAbbrLeagueName = %s", trLeague.sAbbrLeagueName.c_str()));
}

// Method fills the list of all teams to be passed as MethodResult
tVoid fc_sxm_tclSportsFlash::vFillTeamList(const fc_sxm_trSFLeague* poLeague, midw_ext_sxm_audiofi_tclMsgSportsFlashGetTeamsMethodResult& oMRes) const
{
	oMRes.Teams.clear();
	if (poLeague)
	{
		// Iterate through the internal list of teams belonging to poLeague
		SXM_FOREACH_MAP_CONST(tU16, fc_sxm_trSFTeam, cTeamIt, poLeague->mapTeams)
		{
			if (TRUE == cTeamIt->second.bIsSFEligible)
			{
				// Append teams list one by one
				midw_ext_fi_tcl_SportsFlash_Team fiTeam;
				vFillTeamInfo(cTeamIt->second, fiTeam);
				oMRes.Teams.push_back(fiTeam);
			}
		}
	}
}

// Method fills the list of all teams sorted alphabetically and passes MethodResult
tVoid fc_sxm_tclSportsFlash::vFillTeamListBySort(const fc_sxm_trSFLeague* poLeague, midw_ext_sxm_audiofi_tclMsgSportsFlashGetTeamsMethodResult& oMRes) const
{
	oMRes.Teams.clear();
	if (poLeague)
	{
		// Iterate through the internal list of teams belonging to poLeague
		SXM_FOREACH_MAP_CONST(tU16, fc_sxm_trSFTeam, cTeamIt, poLeague->mapTeams)
		{
			if (TRUE == cTeamIt->second.bIsSFEligible)
			{
				// Append teams list one by one
				midw_ext_fi_tcl_SportsFlash_Team fiTeam;
				vFillTeamInfo(cTeamIt->second, fiTeam);
				oMRes.Teams.push_back(fiTeam);
			}
		}
		vSortTeams(oMRes.Teams);
	}
}

// Method fills the details of one Team's info to fi structure "fiTeam"
tVoid fc_sxm_tclSportsFlash::vFillTeamInfo(const fc_sxm_trSFTeam& trTeam, midw_ext_fi_tcl_SportsFlash_Team& fiTeam) const
{
	// Copy team details
	fiTeam.TeamID = trTeam.u16TeamID;
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trTeam.sTeamName.c_str(), fiTeam.TeamName);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trTeam.sTeamNickName.c_str(), fiTeam.TeamNickName);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trTeam.sTeamAbbrName.c_str(), fiTeam.TeamAbbrName);
	string strFullName = trTeam.sTeamName + " " + trTeam.sTeamNickName;
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(strFullName.c_str(), fiTeam.TeamFullName);
	fiTeam.IsFavorite = trTeam.bIsFavoriteTeam;
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::vFillLeagueInfo TeamID(%u) \t IsFavorite(%u)", fiTeam.TeamID, fiTeam.IsFavorite));
	ETG_TRACE_USR4(("Team Full Name = %s", strFullName.c_str()));
}

// Method fills the list of all favorites teams to be passed as MethodResult
// If the Favorite teams needs to be in sorted order, use method fc_sxm_tclSportsFlash::vFillFavoriteTeamsListBySort
tVoid fc_sxm_tclSportsFlash::vFillFavoriteTeamsList(midw_ext_sxm_audiofi_tclMsgSportsFlashGetAllFavoritesMethodResult& oMRes) const
{
	oMRes.FavouritesList.clear();
	SXM_FOREACH_CONST(vector<fc_sxm_trSFLeague>, cLeagueIt, _vectLeague)
	{
		SXM_FOREACH_CONST(set<tU16>, cFavoritesIt, cLeagueIt->setFavTeams)
		{
			SXM_IF_FIND_MAP_CONST(tU16, fc_sxm_trSFTeam, cTeamIt, cLeagueIt->mapTeams, *cFavoritesIt)
			{
				midw_ext_fi_tcl_SportsFlash_FavoriteTeam fiTeam;
				vFillFavoriteTeamInfo(*cLeagueIt, cTeamIt->second, fiTeam);
				oMRes.FavouritesList.push_back(fiTeam);
			}
		}
	}
}
// Method fills the list of all favorite teams to be passed as MethodResult in sorted order
// If sorted order is not required as per project requirements, use method fc_sxm_tclSportsFlash::vFillFavoriteTeamsList
tVoid fc_sxm_tclSportsFlash::vFillFavoriteTeamsListBySort(midw_ext_sxm_audiofi_tclMsgSportsFlashGetAllFavoritesMethodResult& oMRes) const
{
	oMRes.FavouritesList.clear();
	SXM_FOREACH_CONST(vector<fc_sxm_trSFLeague>, cLeagueIt, _vectLeague)
	{
		SXM_FOREACH_CONST(set<tU16>, cFavoritesIt, cLeagueIt->setFavTeams)
		{
			SXM_IF_FIND_MAP_CONST(tU16, fc_sxm_trSFTeam, cTeamIt, cLeagueIt->mapTeams, *cFavoritesIt)
			{
				midw_ext_fi_tcl_SportsFlash_FavoriteTeam fiTeam;
				vFillFavoriteTeamInfo(*cLeagueIt, cTeamIt->second, fiTeam);
				oMRes.FavouritesList.push_back(fiTeam);
			}
		}
	}
	vSortFavorites(oMRes.FavouritesList);
}

// Sort key for re-ordering sports flash favorite teams alphabetically
struct trCompareSFTeamsByFullName
{
	// Overloaded () operator
	bool operator()(const midw_ext_fi_tcl_SportsFlash_FavoriteTeam &trLeftTeam, const midw_ext_fi_tcl_SportsFlash_FavoriteTeam &trRightTeam) const
	{
		string strLTeam(trLeftTeam.TeamFullName.szValue);
		string strRTeam(trRightTeam.TeamFullName.szValue);
		return (strLTeam.compare(strRTeam) < 0);
	}

	bool operator()(const midw_ext_fi_tcl_SportsFlash_Team &trLeftTeam, const midw_ext_fi_tcl_SportsFlash_Team &trRightTeam) const
	{
		string strLTeam(trLeftTeam.TeamFullName.szValue);
		string strRTeam(trRightTeam.TeamFullName.szValue);
		return (strLTeam.compare(strRTeam) < 0);
	}
};

// Utility method which re-orders the sports flash teams based on Alphabetical order of Full Team names which are Favorites
tVoid fc_sxm_tclSportsFlash::vSortFavorites(vector<midw_ext_fi_tcl_SportsFlash_FavoriteTeam, allocator<midw_ext_fi_tcl_SportsFlash_FavoriteTeam> > &teamList) const
{
	// Run std::sort
	sort(teamList.begin(), teamList.end(), trCompareSFTeamsByFullName());
}

// Utility method which re-orders the sports flash teams based on Alphabetical order of Full Team names
tVoid fc_sxm_tclSportsFlash::vSortTeams(vector<midw_ext_fi_tcl_SportsFlash_Team, allocator<midw_ext_fi_tcl_SportsFlash_Team> > &teamList) const
{
	// Run std::sort
	sort(teamList.begin(), teamList.end(), trCompareSFTeamsByFullName());
}

// Method fills the details of one Team's info to fi structure "fiTeam"
tVoid fc_sxm_tclSportsFlash::vFillFavoriteTeamInfo(const fc_sxm_trSFLeague& trLeague, const fc_sxm_trSFTeam& trTeam, midw_ext_fi_tcl_SportsFlash_FavoriteTeam& fiTeam) const
{
	fiTeam.SportType.enType = trLeague.enType;
	fiTeam.LeagueID = trLeague.u8LeagueID;
	fiTeam.LeagueType.enType = static_cast <midw_ext_fi_tcl_e8_SportsFlash_League::tenType> (trLeague.enLeague);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trLeague.sAbbrLeagueName.c_str(), fiTeam.LeagueAbbrName);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trLeague.sLeagueName.c_str(), fiTeam.LeagueFullName);
	fiTeam.TeamID = trTeam.u16TeamID;
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trTeam.sTeamName.c_str(), fiTeam.TeamName);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trTeam.sTeamNickName.c_str(), fiTeam.TeamNickName);
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(trTeam.sTeamAbbrName.c_str(), fiTeam.TeamAbbrName);
	string strFullTeamName = trTeam.sTeamName + " " + trTeam.sTeamNickName;
	fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(strFullTeamName.c_str(), fiTeam.TeamFullName);
	fiTeam.IsFavorite = trTeam.bIsFavoriteTeam;
	fiTeam.IsMonitored = trTeam.bIsMonitored;
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::vFillLeagueInfo TeamID(%u) \t LeagueID(%u)", fiTeam.TeamID, fiTeam.LeagueID));
	ETG_TRACE_USR4(("Team Full Name = %s", strFullTeamName.c_str()));
}

// Add a team as a favorite... Method is invoked in response to HMI reaction of setting a team as favorite
tBool fc_sxm_tclSportsFlash::bAddFavoriteTeam(const midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType enType, const tU8 u8LeagueID, const tU16 u16TeamId)
{
	tBool bSuccess = FALSE;
	fc_sxm_trSFLeague* poLeague = poFindLeaguebyLID(u8LeagueID);
	if ((poLeague) && (poLeague->enType == enType))
	{
		bSuccess = bSetFavoritebyTID(poLeague, u16TeamId);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bAddFavoriteTeam bSuccess = %u", bSuccess));
	return bSuccess;
}
// Remove a team from favorite... Method is invoked in response to HMI reaction of removing a team from being monitored
tBool fc_sxm_tclSportsFlash::bRemoveFavoriteTeam(const midw_ext_fi_tcl_e8_SportsFlash_Sport::tenType enType, const tU8 u8LeagueID, const tU16 u16TeamId)
{
	tBool bSuccess = FALSE;
	fc_sxm_trSFLeague* poLeague = poFindLeaguebyLID(u8LeagueID);
	if ((poLeague) && (poLeague->enType == enType))
	{
		bSuccess = bUnSetFavoritebyTID(poLeague, u16TeamId);
		if (bSuccess)
		{
			vRemoveTeamFromActiveGamesList(poLeague->u8LeagueID, u16TeamId);
			// Re-set games to be monitored
			(tVoid) enSetGamesMonitor();
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bRemoveFavoriteTeam bSuccess = %u", bSuccess));
	return bSuccess;
}

/* Method "vRemoveTeamFromActiveGamesList"
 * Removes a team from active games list based on following conditions
 *
 * Ground Rule: A game will have entry in vector "_vectActiveGames" only if
 *                       "At-least one of the two teams involved is a Favorite team and its playing a live game".
 *
 *
 * Condition 1: If both home & away teams are favorites AND a live game is happening between them,
 *             Action: Find if the reported team is away team or home team.
 *             Condition 1 a: If reported team is away team,
 *                             Action: Unset the favorites flag for away team
 *                             Precaution: a) Don't disturb favorites flag for home team,
 *                                         b) Don't remove the game from "_vectActiveGames"
 *             Condition 1 b: If reported team is home team,
 *                             Action: Unset the favorites flag for home team
 *                             Precaution: a) Don't disturb favorites flag for away team
 *                                         b) Don't remove the game from "_vectActiveGames"
 *
 * Condition 2: If ONLY HOME team is a favorite and it is the reported team;
 *            Action: a) Unset the favorites flag for home team
 *                    b) REMOVE game from "_vectActiveGames"
 *
 *
 * Condition 3: If ONLY AWAY team is a favorite and it is the reported team;
 *            Action: a) Unset the favorites flag for away team
 *                    b) REMOVE game from "_vectActiveGames"
 *
 * Condition 4: If NEITHER Home team NOR Away Team is a Favorite but it is reported
 *            Action: TAKE no action and hope for the best.
 *            Note: a) This is an Invalid condition and can never happen in an ideal scenario.
 *                  b) This condition deviates from "Ground Rule".
 *            FAQ: If this condition occurs even once... What does it mean? What's the cause? What's the impact?
 *                 Answer: It means a programming fault.
 *                         Possible causes are the prior handling of conditions 1, 2 and 3 were wrong when those events were reported.
 *                         The impact is providing wrong "ActiveGamesList" and "ActiveAlertsList" to the clients.
 */

tVoid fc_sxm_tclSportsFlash::vRemoveTeamFromActiveGamesList(const tU8 u8LeagueID, const tU16 u16TeamID)
{
	SXM_FOREACH(vector<fc_sxm_trFlashGameEvent>, gamesIter, _vectActiveGames)
	{
		tBool bIsHomeTeamFavorite = gamesIter->trHomeTeam.bIsFavorite;
		tBool bIsAwayTeamFavorite = gamesIter->trAwayTeam.bIsFavorite;
		tBool bIsBothTeamsFavorite = (bIsHomeTeamFavorite && bIsAwayTeamFavorite);

		// Check Condition 1
		if (bIsBothTeamsFavorite)
		{
			// Check Condition 1 a
			if (gamesIter->bIsAwayTeam(u8LeagueID, u16TeamID))
			{
				gamesIter->trAwayTeam.bIsFavorite = FALSE;
				break;
			}
			// Check Condition 1 b
			else if (gamesIter->bIsHomeTeam(u8LeagueID, u16TeamID))
			{
				gamesIter->trHomeTeam.bIsFavorite = FALSE;
				break;
			}
		}

		// Check Condition 2
		else if ((bIsHomeTeamFavorite) && (gamesIter->bIsHomeTeam(u8LeagueID, u16TeamID)))
		{
			gamesIter->trHomeTeam.bIsFavorite = FALSE;
			vClearActiveGameEvents(*gamesIter);
			_vectActiveGames.erase(gamesIter);
			break;
		}
		// Check Condition 3
		else if ((bIsAwayTeamFavorite) && (gamesIter->bIsAwayTeam(u8LeagueID, u16TeamID)))
		{
			gamesIter->trAwayTeam.bIsFavorite = FALSE;
			vClearActiveGameEvents(*gamesIter);
			_vectActiveGames.erase(gamesIter);
			break;
		}
	}
}

// Wrapper method to play the flash event
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enPlayFlashEvent(SPORTS_FLASH_EVENT_ID tEventID) const
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	if (_poSFProxy)
	{
		enRetCode = _poSFProxy->enPlayFlashEvent(tEventID);
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enPlayFlashEvent enRetCode = %u", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode)));
	return enRetCode;
}

// Wrapper method to abort the flash event
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enAbortFlashEvent() const
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	if (_poSFProxy)
	{
		enRetCode = _poSFProxy->enAbortFlashEvent();
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enAbortFlashEvent enRetCode = %u", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode)));
	return enRetCode;
}

// Wrapper method to enquire the current playing sports flash event's FlashEventID
SPORTS_FLASH_EVENT_ID fc_sxm_tclSportsFlash::u32CurrentFlashEventID() const
{
	SPORTS_FLASH_EVENT_ID tEventID = SPORTS_FLASH_INVALID_EVENT_ID;
	// If currently sports flash event is not playing, this method returns SPORTS_FLASH_INVALID_EVENT_ID(2^32.)
	if (_poSFProxy)
	{
		tEventID = _poSFProxy->u32CurrentFlashEventID();
	}
	return tEventID;
}

// Wrapper method to iterate current games
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enIterateCurrentGames()
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	_vectActiveGames.clear();
	if (_poSFProxy)
	{
		enRetCode = _poSFProxy->enIterateCurrentGames(fc_sxm_tclSportsFlash::cb_ActiveGamesIteratorCallback,
				static_cast <tVoid *>(this));
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enIterateCurrentGames enRetCode = %u _vectActiveGames.size = %u",
			ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode), _vectActiveGames.size()));
	return enRetCode;
}

// Wrapper method to iterate current flash events
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enIterateFlashEvents()
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	if (_poSFProxy)
	{
		enRetCode = _poSFProxy->enIterateCurrentFlashEvents(fc_sxm_tclSportsFlash::cb_ActiveFlashEventsIteratorCallback,
				static_cast <tVoid *>(this));
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enIterateCurrentFlashEvents enRetCode = %u", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode)));
	return enRetCode;
}


// Utility method to extract game event details and pack it into corresponding FI structure
tVoid fc_sxm_tclSportsFlash::vExtractGameEventDetails(const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *poGameInfo, fc_sxm_trFlashGameEvent& trGameEvent)
{
	if (poGameInfo && _poSFProxy)
	{
		vRetrieveChannelInfo(poGameInfo->tChannelID, trGameEvent.trChannel);
		string sHomeTeamIdentifier = sExtractGameEventTeam(poGameInfo->hHomeLeagueId, poGameInfo->hHomeTeamId, trGameEvent.trHomeTeam);
		string sAwayTeamIdentifier = sExtractGameEventTeam(poGameInfo->hAwayLeagueId, poGameInfo->hAwayTeamId, trGameEvent.trAwayTeam);
		string sHashInput= sHomeTeamIdentifier + sAwayTeamIdentifier;
		// Compute HashCode using Abbreviated League+TeamNames of Home and Away Teams
		trGameEvent.u32HashCode = fc_sxm_tclGenericUtils::u32ComputeHashCode(sHashInput);
		trGameEvent.u16Bias = poGameInfo->un16Bias;
		ETG_TRACE_USR4(("ChanneID = %u \t HASHCode = %u \t BIAS = %u",
				trGameEvent.trChannel.tChannelID, trGameEvent.u32HashCode, trGameEvent.u16Bias));
	}
}

// Method does two functionalities; 1) Fills up trTeamInfo structure; 2) Returns LeagueAbbrName+TeamAbbrName string
string fc_sxm_tclSportsFlash::sExtractGameEventTeam(CID_OBJECT hLeagueID, CID_OBJECT hTeamID, fc_sxm_trSFTeamInfo& trTeamInfo)
{
	fc_sxm_trSFLeague* poLeague = poFindLeaguebyCID(hLeagueID);
	string strIdentifier = "";
	// Find league and then the team and fill up  strIdentifier with LeagueAbbrName+TeamAbbrName
	if (poLeague)
	{
		trTeamInfo.u8LeagueID = poLeague->u8LeagueID;
		strIdentifier += poLeague->sAbbrLeagueName;
		fc_sxm_trSFTeam* poTeam = poFindTeambyCID(poLeague, hTeamID);
		if (poTeam)
		{
			trTeamInfo.u16TeamID = poTeam->u16TeamID;
			trTeamInfo.bIsFavorite = poTeam->bIsFavoriteTeam;
			trTeamInfo.bIsMonitored = poTeam->bIsMonitored;
			strIdentifier += poTeam->sTeamAbbrName;
		}
	}
	ETG_TRACE_USR4(("sExtractGameEventTeam  strIdentifier = %s", strIdentifier.c_str()));
	return strIdentifier;
}

// Fill the list of active games into method result structure for method  "SportsFlashGetCurrentGames"
tVoid fc_sxm_tclSportsFlash::vFillGamesList(midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentGamesMethodResult& oMRes)
{
	oMRes.Games.clear();
	SXM_FOREACH_CONST(vector<fc_sxm_trFlashGameEvent>, gamesIter, _vectActiveGames)
	{
		midw_ext_fi_tcl_SportsFlash_Game fiGame;
		vFillGameDetails(*gamesIter, fiGame);
		oMRes.Games.push_back(fiGame);
	}
}

// Fill the list of flash events that have occured
tVoid fc_sxm_tclSportsFlash::vFillFlashEventsList(midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentFlashEventsMethodResult& oMRes)
{
	oMRes.FlashEvents.clear();
	map<tU32, midw_ext_fi_tcl_SportsFlash_FlashEvent> mapFlashEvents;
	SXM_FOREACH(vector<fc_sxm_trFlashGameEvent>, gamesIter, _vectActiveGames)
	{
		vRetrieveChannelInfo(gamesIter->trChannel.tChannelID, gamesIter->trChannel);
		if (gamesIter->trChannel.bChannelStatus())
		{
			SXM_FOREACH_CONST(deque <fc_sxm_trSFFlashEventDetails>, flashEvents, gamesIter->eventQueue)
			{
				// Add only the flash events that have not yet expired and played
				if (SPORTS_FLASH_EVENT_OCCURRED == flashEvents->enFlashEventStatus)
				{
					midw_ext_fi_tcl_SportsFlash_FlashEvent fiFlashEvt;
					fiFlashEvt.FlashEventID = flashEvents->tFlashEventID;
					vFillGameDetails(*gamesIter, fiFlashEvt.GameInfo);
					//oMRes.FlashEvents.push_back(fiFlashEvt);
					mapFlashEvents[flashEvents->u32TimeStamp] = fiFlashEvt;
				}
			}
		}
	}

	map<tU32, midw_ext_fi_tcl_SportsFlash_FlashEvent>::reverse_iterator rIt = mapFlashEvents.rbegin();
	for (; rIt != mapFlashEvents.rend(); ++rIt)
	{
		oMRes.FlashEvents.push_back(rIt->second);
	}
	vPrintFlashEventsList(oMRes);
}

// Method prints the list of flash events to be sent to client
tVoid fc_sxm_tclSportsFlash::vPrintFlashEventsList(const midw_ext_sxm_audiofi_tclMsgSportsFlashGetCurrentFlashEventsMethodResult& oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::vPrintFlashEventsList Count of Active flash events = %u", oMRes.FlashEvents.size()));
	vector<midw_ext_fi_tcl_SportsFlash_FlashEvent, allocator<midw_ext_fi_tcl_SportsFlash_FlashEvent> >::const_iterator cIt =
			oMRes.FlashEvents.begin();
	tU8 u8Index = 0;
	while ( cIt != oMRes.FlashEvents.end())
	{
		ETG_TRACE_USR4(("oMRes.FlashEvents[%u] \t FlashEventID = %u \t ChannelID(%u) ServID(%u)",
				u8Index, cIt->FlashEventID, cIt->GameInfo.ChannelID, cIt->GameInfo.ServiceID));
		ETG_TRACE_USR4(("IsLocked(%u) \t IsSkipped(%u) \t IsMature(%u) \t Bias(%u)",
				cIt->GameInfo.IsLocked, cIt->GameInfo.IsSkipped, cIt->GameInfo.IsMature, cIt->GameInfo.Bias));
		ETG_TRACE_USR4(("%s(Away Team)", cIt->GameInfo.AwayTeam.TeamFullName.szValue));
		ETG_TRACE_USR4(("%s(Home Team)", cIt->GameInfo.HomeTeam.TeamFullName.szValue));
		ETG_TRACE_USR4(("Away Team Details: TeamID = %u \t IsFavorite = %u \t IsMonitored = %u",
						cIt->GameInfo.AwayTeam.TeamID, cIt->GameInfo.AwayTeam.IsFavorite, cIt->GameInfo.AwayTeam.IsMonitored));
		ETG_TRACE_USR4(("Home Team Details: TeamID = %u \t IsFavorite = %u \t IsMonitored = %u",
				cIt->GameInfo.HomeTeam.TeamID, cIt->GameInfo.HomeTeam.IsFavorite, cIt->GameInfo.HomeTeam.IsMonitored));
		++cIt;
		++u8Index;
	}
}

// Identify the teams involved in the game and pack their details into FI structure midw_ext_fi_tcl_SportsFlash_Game
tVoid fc_sxm_tclSportsFlash::vFillGameDetails(const fc_sxm_trFlashGameEvent& trGameEvent, midw_ext_fi_tcl_SportsFlash_Game& fiGame)
{
	// Fill the attributes of the channel broadcasting the game
	fiGame.ChannelID = trGameEvent.trChannel.tChannelID;
	fiGame.ServiceID = trGameEvent.trChannel.tServiceID;
	fiGame.IsLocked = trGameEvent.trChannel.bIsLocked;
	fiGame.IsMature = trGameEvent.trChannel.bIsMature;
	fiGame.IsSkipped = trGameEvent.trChannel.bIsSkipped;
	fiGame.Bias = trGameEvent.u16Bias;
	fiGame.ChnGraphicsAvailable.enType = (trGameEvent.trChannel.bIsChnArtAvl) ?
			midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH  :
			midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
	// Find home team
	fc_sxm_trSFLeague* poHomeTeamLeague = poFindLeaguebyLID(trGameEvent.trHomeTeam.u8LeagueID);
	if (poHomeTeamLeague)
	{
		fc_sxm_trSFTeam* poHomeTeam = poFindTeambyTID(poHomeTeamLeague, trGameEvent.trHomeTeam.u16TeamID);
		if (poHomeTeam)
			vFillFavoriteTeamInfo(*poHomeTeamLeague, *poHomeTeam, fiGame.HomeTeam);
	}
	// Find away team
	fc_sxm_trSFLeague* poAwayTeamLeague = poFindLeaguebyLID(trGameEvent.trAwayTeam.u8LeagueID);
	if (poAwayTeamLeague)
	{
		fc_sxm_trSFTeam* poAwayTeam = poFindTeambyTID(poAwayTeamLeague, trGameEvent.trAwayTeam.u16TeamID);
		if (poAwayTeam)
			vFillFavoriteTeamInfo(*poAwayTeamLeague, *poAwayTeam, fiGame.AwayTeam);
	}
}

// Sets a list of games to be monitored
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enSetGamesMonitor()
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	// Currently SMS can monitor 5 games in parallel. So we instruct SMS to monitor only a max of 5 channels.
	if (_poSFProxy)
	{
		CHANNEL_ID chID[5] = { 0,0,0,0,0};
		vector<fc_sxm_trFlashGameEvent>::const_iterator gamesIter = _vectActiveGames.begin();
		size_t tIndex = 0;
		for(; ((gamesIter != _vectActiveGames.end()) && (tIndex < 5)); ++gamesIter)
		{
			// Setting games to be monitored to be on only when both the away team and home team's bIsMonitored flag is ON
			if ((gamesIter->trAwayTeam.bIsMonitored) || (gamesIter->trHomeTeam.bIsMonitored))
			{
				chID[tIndex] = gamesIter->trChannel.tChannelID;
				++tIndex;
			}
		}
		enRetCode = _poSFProxy->enSetGamesMonitor(chID, (tU16)tIndex, fc_sxm_tclSportsFlash::cb_FlashEventCallback, static_cast <tVoid *>(_poAudioApp));
		ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enSetGamesMonitor tIndex(%u) \t _vectActiveGames.size(%u) \t enRetCode(%u)",
				tIndex, _vectActiveGames.size(), ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode) ));
		ETG_TRACE_USR4(("chID[0]=%u \t chID[1]=%u \t chID[2]=%u \t chID[3]=%u",
				chID[0], chID[1], chID[2], chID[3]));
	}
	return enRetCode;
}

// Sets a list of games to be monitored
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enSetGamesMonitor(const vector<tU16, allocator<tU16> > & vectChannelIDs)
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	// Currently SMS can monitor 5 games in parallel. So we instruct SMS to monitor only a max of 5 channels.
	if (_poSFProxy)
	{
		CHANNEL_ID chID[5] = { 0,0,0,0,0};
		vector<tU16, allocator<tU16> >::const_iterator channelIter = vectChannelIDs.begin();
		size_t tIndex = 0;
		for(; ((channelIter != vectChannelIDs.end()) && (tIndex < 5)); ++tIndex, ++channelIter)
		{
			chID[tIndex] = *channelIter;
		}
		enRetCode = _poSFProxy->enSetGamesMonitor(chID, (tU16)tIndex, fc_sxm_tclSportsFlash::cb_FlashEventCallback, static_cast <tVoid *>(_poAudioApp));
		ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::enSetGamesMonitorOverloaded tIndex(%u) \t enRetCode(%u)",
				tIndex, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enRetCode) ));
	}
	return enRetCode;
}

// Enable or Disable Sports Flash notifications
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclSportsFlash::enEnableSFNotifications(tBool bEnable) const
{
	SMSAPI_RETURN_CODE_ENUM enRetCode = SMSAPI_RETURN_CODE_ERROR;
	if (_poSFProxy)
	{
		enRetCode = _poSFProxy->enEnableNotification(bEnable);
	}
	return enRetCode;
}

// Is SF notifications enabled or disabled
tBool fc_sxm_tclSportsFlash::bIsSFNotificationEnabled() const
{
	tBool bEnabled = (_poSFProxy) ? _poSFProxy->bIsNotificationEnabled() : FALSE;
	return bEnabled;
}

// Utility method to compare 2 CID objects
tS16 fc_sxm_tclSportsFlash::s16CompareCIDObjects(CID_OBJECT hCID1, CID_OBJECT hCID2) const
{
	return (( (_poSFProxy != OSAL_NULL) ?
			_poSFProxy->s16CompareCIDObjects(hCID1, hCID2)  :  -1 ));
}


// Utility method that redirects FI Messages to be sent to Audio App
tVoid fc_sxm_tclSportsFlash::vSendSFFiMsg(fc_sxm_trAdressing const &rAddr,
		const midw_ext_sxm_audiofi_tclMsgBaseMessage& oMessage)
{
	// This method is executed for all posts of MethodResult
	if (OSAL_NULL != _poAudioApp)
	{
		_poAudioApp->vSendAudioFiMsg(rAddr, oMessage);
	}
}

// Utility method that redirects FI Error Messages to be sent to Audio App
tVoid fc_sxm_tclSportsFlash::vSendSFFiError(fc_sxm_trAdressing const &rAddr, tInt s32ErrorMsg)
{
	if (OSAL_NULL != _poAudioApp)
	{
		_poAudioApp->vSendAudioErrorMsg(rAddr, s32ErrorMsg);
	}
}

// Method to update a team's monitor status
tBool fc_sxm_tclSportsFlash::bUpdateMonitorStatus(tU8 u8LeagueID, tU16 u16TeamID, tBool bMonitorStatus)
{
	tBool bStatus = FALSE;
	// Identify the League
	fc_sxm_trSFLeague* poLeague = poFindLeaguebyLID(u8LeagueID);
	if (poLeague)
	{
		// Change team's monitor status
		bStatus = bSetTeamMonitorStatusbyTID(poLeague, u16TeamID, bMonitorStatus);
	}

	// Find if the team is involved in a live game and modify the bIsMonitored bit
	SXM_FOREACH(vector<fc_sxm_trFlashGameEvent>, activeGamesIt, _vectActiveGames)
	{
		if (activeGamesIt->trAwayTeam.u16TeamID == u16TeamID)
		{
			activeGamesIt->trAwayTeam.bIsMonitored = bMonitorStatus;
		}
		else if (activeGamesIt->trHomeTeam.u16TeamID == u16TeamID)
		{
			activeGamesIt->trHomeTeam.bIsMonitored = bMonitorStatus;
		}
	}
	return bStatus;
}

// Method retrieves channel information from ChannelList class
tVoid fc_sxm_tclSportsFlash::vRetrieveChannelInfo(CHANNEL_ID channelID, fc_sxm_trSFChannel& trChannel)
{
	if (_poAudioApp)
	{
		const fc_sxm_trChannel* chnInfo = _poAudioApp->trGetChannel(channelID);
		if (chnInfo)
		{
			// Once the channel is identified from channel list,
			// fill up all relevant details for making up a SF alert
			trChannel.tChannelID = chnInfo->u16ChannelId;
			trChannel.tServiceID = chnInfo->u16ServiceId;
			trChannel.bIsLocked = chnInfo->bIsLocked();
			trChannel.bIsSkipped = chnInfo->bIsSkipped();
			trChannel.bIsSubscribed = chnInfo->bIsSubscribed();
			trChannel.bIsMature = chnInfo->bIsMature();
			trChannel.bIsFreeToAir = chnInfo->bIsFreeToAir();
			trChannel.bIsChnArtAvl = chnInfo->bIsChannelArtAvailable;
		}
	}
}

// Method handles DECODER_OBJECT_EVENT_TUNE notifications from SMS.
// Once this method is invoked by framework, a check is made if any sports flash event is currently played or not
// and appropriate property updates are sent to clients
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioSmsEvtTuneState const * )
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus oProperty;
		oProperty.oFiMsg.FlashEventPlayState.enType = midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_STOPPED;
		_bIsSFPlayback = FALSE;
		_channelID = CHANNEL_INVALID_ID;
		if (SPORTS_FLASH_INVALID_EVENT_ID != u32CurrentFlashEventID())
		{
			// A valid SportsFlash Event id is returned by SMS if SF event is currently being played
			oProperty.oFiMsg.FlashEventPlayState.enType = midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_PLAYING;
			_bIsSFPlayback = TRUE;
			_channelID = _poAudioApp->u16GetTunedChnID();
		}
		ETG_TRACE_USR4(("Notify SF  fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus %u \t _bIsSFPlayback = %u",
				oProperty.oFiMsg.FlashEventPlayState.enType, _bIsSFPlayback));
		_poAudioApp->vSFlashPlaybackNotify(oProperty);
		vNotifyPendingAlerts();
	}
}

// Method is introduced for condition when SF playback is in progress and Tune Scan is started
// For every successive tune scan channel change,this method would be invoked
tVoid fc_sxm_tclSportsFlash::vProcess(fc_sxm_trMsgAudioSmsEvtChannelEvt const *)
{
	if (_poAudioApp && _bIsSFPlayback)
	{
		CHANNEL_ID tunedChannelID = _poAudioApp->u16GetTunedChnID();
		ETG_TRACE_USR4(("Event fc_sxm_trMsgAudioSmsEvtChannelEvt:  \t tunedChannelID(%u) _channelID(%u)", tunedChannelID, _channelID));
		fc_sxm_tcl_trAudioPropertySportsFlash_PlaybackStatus oProperty;
		oProperty.oFiMsg.FlashEventPlayState.enType = (tunedChannelID != _channelID) ?
				midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_STOPPED :
				midw_ext_fi_tcl_e8_FlashEventPlaybackStatus::FI_EN_FLASHEVENT_PLAYING ;
		_poAudioApp->vSFlashPlaybackNotify(oProperty);
	}
}

// Method prepares sports flash service to handle MetaDataChanges(Broadscope Changes)
tVoid fc_sxm_tclSportsFlash::vMetaDataUpdateStart()
{
	_bIsMetaDataChanged = TRUE;
	ETG_TRACE_USR4(("Handle META DATA UPDATION/ BROADSCOPE CHANGE in SPORTSFLASH"));

	// Clear all internal caches and stop sports flash service
	_bIsSFInitialized = FALSE;
	_bEnableSF = TRUE;
	_bIsSFPlayback = FALSE;
	_uMaxSFChannels = 0U;
	_channelID = CHANNEL_INVALID_ID;
	_vectSports.clear();
	_vectLeague.clear();
	_vectActiveGames.clear();
	_vectPendingAlerts.clear();
	// Stop Sports Flash and set system state to ERROR
	vStopSF(TRUE);
	vResetAlerts();
}

// Handle Meta Data Updation Completion
tVoid fc_sxm_tclSportsFlash::vMetaDataUpdateComplete()
{
	if (_bIsMetaDataChanged && _poAudioApp)
	{
		ETG_TRACE_USR4(("META DATA UPDATION/ BROADSCOPE CHANGE has completed"));
		_bIsMetaDataChanged = FALSE;
		//Re-start SportsFlash Service
		DECODER_OBJECT hDecoderObject = _poAudioApp->oGetDecoderObject();
		vNotifySFServiceState((enStartSF(hDecoderObject)), TRUE);
	}
	else
	{
		ETG_TRACE_USR4(("Not a META DATA UPDATION/ BROADSCOPE CHANGE .... Don't process"));
	}
}


// Method returns TRUE if the flash event can be notified to clients, FALSE if not
// Consideration is made for the following conditions,
// 1) If currentTunedChannel = chnID
// 2) If sports flash event is currently played
// 3) Is any advisory notification running
tBool fc_sxm_tclSportsFlash::bNotifyFlashEvent(CHANNEL_ID chnID) const
{
	tBool bNotify = TRUE;
	if ((_bIsSFPlayback) ||
			((_poAudioApp) && (chnID == _poAudioApp->u16GetTunedChnID())))
	{
		bNotify = FALSE;
	}
	return bNotify;
}

// Method adds property updates to be sent to HMI to the pending alerts list
tBool fc_sxm_tclSportsFlash::bUpdatePendingAlertsList(const fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification& oProperty)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bUpdatePendingAlertsList size = %u", _vectPendingAlerts.size()));
	if (midw_ext_fi_tcl_e8_SportsFlash_FlashEventStatus::FI_EN_SPORTS_FLASH_EVENT_OCCURED == oProperty.oFiMsg.FlashEventStatus.enType)
	{
		_vectPendingAlerts.push_back(oProperty);
		ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bUpdatePendingAlertsList incremented by one... current size = %u", _vectPendingAlerts.size()));
		return TRUE;
	}
	else
	{
		ETG_TRACE_USR4(("bUpdatePendingAlertsList FI_EN_SPORTS_FLASH_EVENT_EXPIRED or PLAYED"));
		tBool bFound = TRUE;
		tBool bErased = FALSE;
		// Find and delete the pending alert .... No need to notify to HMI
		vector <fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification>::iterator it = _vectPendingAlerts.begin();
		while ((it != _vectPendingAlerts.end()) && (TRUE == bFound))
		{
			bFound = FALSE;
			for (; it != _vectPendingAlerts.end(); ++it)
			{
				if (oProperty.bIsSavedEvent(*it))
				{
					_vectPendingAlerts.erase(it);
					bFound = TRUE;
					bErased = TRUE;
					break;
				}
			}
			it = _vectPendingAlerts.begin();
		}
		ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::bUpdatePendingAlertsList size = %u \t bFound = %u \t bErased = %u", _vectPendingAlerts.size(), bFound, bErased));
		return bErased;
	}
}

// Run thru the pending alerts list and notify pending alerts to HMI sequentially
tVoid fc_sxm_tclSportsFlash::vNotifyPendingAlerts()
{
	if ((_vectPendingAlerts.size()) && (_poAudioApp) && (!_bIsSFPlayback))
	{
		CHANNEL_ID tunedChnID = _poAudioApp->u16GetTunedChnID();
		ETG_TRACE_USR4(("vNotifyPendingAlerts() Begin: Pending Alerts queue size = %u \t tunedChanID = %u", _vectPendingAlerts.size(), tunedChnID));
		vector <fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification>::const_iterator cIt =
				_vectPendingAlerts.begin();
		deque <tU8> indices;
		tU8 u8Index = 0u;
		do {
			if (tunedChnID != cIt->oFiMsg.GameInfo.ChannelID)
			{
				ETG_TRACE_USR4(("FlashEventNotification for alert at ChanID = %u, FlashID = %u, FlashEventStatus = %d",
						cIt->oFiMsg.GameInfo.ChannelID, cIt->oFiMsg.FlashEventID, cIt->oFiMsg.FlashEventStatus.enType));
				_poAudioApp->vSFlashFlashEventNotify(*cIt);
				indices.push_front(u8Index);
			}
			++cIt;
			++u8Index;
		} while (cIt != _vectPendingAlerts.end());

		deque <tU8>::const_iterator queueIter =  indices.begin();
		for (; queueIter != indices.end(); ++queueIter)
		{
			if (*queueIter < _vectPendingAlerts.size())
			{
				ETG_TRACE_USR4(("Remove pending alert at index(%u)", *queueIter));
				_vectPendingAlerts.erase(_vectPendingAlerts.begin() + *queueIter);
			}
		}
		ETG_TRACE_USR4(("vNotifyPendingAlerts() After: Pending Alerts queue size = %u", _vectPendingAlerts.size()));
	}
}

// Constructor for structure fc_sxm_trSFFlashEventDetails
fc_sxm_trSFFlashEventDetails::fc_sxm_trSFFlashEventDetails(SPORTS_FLASH_EVENT_ID eventID, SPORTS_FLASH_EVENT_STATUS_ENUM enStatus) :
		tFlashEventID(eventID), enFlashEventStatus(enStatus)
{
	u32TimeStamp = OSAL_ClockGetElapsedTime();
}

// Method checks if new Flash event reported is already available.
//  -- If available, then delete exisiting event if new status is expired; else update status value
//  -- If not available, then the new flash event is added to vector
tVoid fc_sxm_trFlashGameEvent::vUpdateEventQueue(SPORTS_FLASH_EVENT_ID eventID, SPORTS_FLASH_EVENT_STATUS_ENUM enStatus)
{
	ETG_TRACE_USR4(("fc_sxm_trFlashGameEvent vUpdateEventQueue  eventID(%u) \t enStatus(%u)", eventID, enStatus));
	// Iterate thru the list of Flash Events for one game
	SXM_FOREACH(deque<fc_sxm_trSFFlashEventDetails>, eventIT, eventQueue)
	{
		// Erase the flash event ID
		if (eventID == eventIT->tFlashEventID)
		{
			eventQueue.erase(eventIT, eventQueue.end());
			break;
		}
	}

	if (SPORTS_FLASH_EVENT_EXPIRED != enStatus)
	{
		fc_sxm_trSFFlashEventDetails newEvent(eventID, enStatus);
		eventQueue.push_front(newEvent);
		ETG_TRACE_USR4(("Adding newFlashEvent to the Game  eventQueue.size() = %u", eventQueue.size()));
	}
}

// Remove flash events from a game
tVoid fc_sxm_tclSportsFlash::vClearActiveGameEvents(fc_sxm_trFlashGameEvent& trGameEvent)
{
	const SPORTS_FLASH_EVENT_ID currentEventID = u32CurrentFlashEventID();
	ETG_TRACE_USR4(("fc_sxm_tclSportsFlash::vClearActiveGameEvents currentEventID = %u", currentEventID));
	// Iterate thru the list of Flash Events for the given game
	SXM_FOREACH(deque<fc_sxm_trSFFlashEventDetails>, eventIT, trGameEvent.eventQueue)
	{
		// If currently playing a sports flash alert, then abort playing SF
		if ((_bIsSFPlayback) &&
				( currentEventID == eventIT->tFlashEventID ) &&
				( _channelID == trGameEvent.trChannel.tChannelID))
		{
			ETG_TRACE_USR3(("Removing a team from FAV list whose sports flash event(%u) is being played at (%u)",
					currentEventID, _channelID));
			(tVoid) enAbortFlashEvent();
			_bIsSFPlayback = FALSE;
			_channelID = CHANNEL_INVALID_ID;
		}
		// Notify HMI Client that the flash event is expired
		vNotifyFlashEventUpdate(trGameEvent, eventIT->tFlashEventID, SPORTS_FLASH_EVENT_EXPIRED);
	}
	vNotifyGameEventUpdate(SPORTS_FLASH_PROGRAM_ENDED, trGameEvent);
	ETG_TRACE_USR3(("Game is now Ended -- Home Team(%u) vs Away Team(%u)",
			trGameEvent.trHomeTeam.u16TeamID, trGameEvent.trAwayTeam.u16TeamID));
}

// Reset GameEvent and FlashEvent Alerts
tVoid fc_sxm_tclSportsFlash::vResetAlerts() const
{
	vResetGameEventAlerts();
	vResetFlashEventAlerts();
}

// Reset Game Event Alerts
tVoid fc_sxm_tclSportsFlash::vResetGameEventAlerts() const
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertySportsFlash_GameEventNotification oProperty;
		ETG_TRACE_USR3(("Set Empty Game Event Alert"));
		_poAudioApp->vSFlashGameEventNotify(oProperty);
	}
}

// Reset Flash Event Alerts
tVoid fc_sxm_tclSportsFlash::vResetFlashEventAlerts() const
{
	if (_poAudioApp)
	{
		fc_sxm_tcl_trAudioPropertySportsFlash_FlashEventNotification oProperty;
		ETG_TRACE_USR3(("Set Empty Flash Event Alert"));
		_poAudioApp->vSFlashFlashEventNotify(oProperty);
	}
}
