/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_rapid_channel_browse.h
* @brief       Class declaration for Rapid Channel Browsing Functionality.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef FC_SXM_TCL_RAPID_CHANNEL_BROWSE_H_
#define FC_SXM_TCL_RAPID_CHANNEL_BROWSE_H_

#include "fc_sxm_types.h"
#include "fc_sxm_audio_types.h"

/**
 * BrowseRateConfiguration stores the a) Duration, b) Rate of Channel Change,
 *  c) No of steps that rate of channel change has to be maintained, d) Completed Steps
 */
struct fc_sxm_trBrowseRateConfiguration {
	tU16 u16Duration;  //!< (Units in seconds) Entire duration that this configurations has to be considered.
	tU16 u16Rate;   //!< (Units in millSeconds) Rate at which the channels would be changed.
	tU16 u16MaxSteps;  //!< Maximum number of steps applicable for this configuration
	tU16 u16CompletedSteps; //!< Number of steps completed for the configuration.
	/**
	 * @constructor.
	 * Apart from setting values for member variables, an error correction is performed if u16Rate
	 * is lesser than 100 mSecs. In such cases the default channel change rate is set to 500 mSec.
	 */
	fc_sxm_trBrowseRateConfiguration(tU16 duration = 0u, tU16 rate = 0u, tU16 steps = 0u) :
		u16Duration(duration), u16Rate(rate), u16MaxSteps(steps), u16CompletedSteps(0u) {
		if (100u > u16Rate)
		{
			u16Rate = 500u;
		}
	}
};


/******************************************************************************
 * CLASS        : fc_sxm_tclRapidChannelBrowse                                *
 * ---------------------------------------------------------------------------*
 * DESCRIPTION  : Class fc_sxm_tclRapidChannelBrowse to enable channel browsing
 *                    at an accelerated speed
 *                Following functionalities are realized
 *                - Initialize Rapid Channel Browsing with new configurations at run time
 *                - Reset Old Configurations                                  *
 *                - When Rapid Channel Browse Mode is ON, provide GetNextChannelID      *
 *                - When Rapid Channel Browse Mode is ON, provide GetNextTimerExpiry    *
 *****************************************************************************/
class fc_sxm_tclRapidChannelBrowse {
public:
	tVoid vReset();
	tVoid vInit(midw_ext_fi_tcl_e8_BrowseDirection enDirection,
			const midw_ext_fi_tcl_BrowseChannelConfig& configuration,
			CHANNEL_ID tunedChnID,
			const vector<CHANNEL_ID>& vectChannels);

	tU16 u16GetNextDuration();
	CHANNEL_ID u16GetNextChannelID();

	fc_sxm_tclRapidChannelBrowse();
	~fc_sxm_tclRapidChannelBrowse();

private:
#ifdef UNITTEST_FC_SXM
    // For unit test product, to perform effective unit-testing of all private members,
    // the private methods/variables are made public.
    // In production-code, the methods and variables still remain private
public:
#endif //UNITTEST_FC_SXM

	tVoid vRemoveConfig();
	tVoid vFillConfig(const vector<midw_ext_fi_tcl_BrowseRateConfiguration, allocator<midw_ext_fi_tcl_BrowseRateConfiguration> >& cfg);
	tVoid vDiscernSteps(fc_sxm_trBrowseRateConfiguration& config) const;
	bool bFindPivot(CHANNEL_ID tunedChannel);
	tVoid vCalibratePivot(CHANNEL_ID tunedChannel);
	tVoid vCalibrateDown(CHANNEL_ID tunedChannel);
	tVoid vCalibrateUp(CHANNEL_ID tunedChannel);

	CHANNEL_ID u16ChannelUp();
	CHANNEL_ID u16ChannelDown();

	tU16 _u16Pivot; //!< Pivot Index. Default value is set to (1u << 15)
	midw_ext_fi_tcl_e8_BrowseDirection _enDirection; //!< Browse Direction.
	midw_ext_fi_tcl_e8_BrowseChannelLoop _enLoop; //!< Loop Type.
	vector <fc_sxm_trBrowseRateConfiguration> _vectConfig; //!< List of BrowseRate Configurations.
	vector<CHANNEL_ID> _vectChannels; //!< List of channels considered for Rapid Channel Browsing.
};

#endif /* FC_SXM_TCL_RAPID_CHANNEL_BROWSE_H_ */
