/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_presets.h
* @brief       Class declaration for Presets and Smart Favorites Functionalities.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

#ifndef FC_SXM_TCL_PRESETS_H
#define FC_SXM_TCL_PRESETS_H

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_tcl_presets_if.h"


/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/

/* Defining preset and preset band names */
#define FC_SXM_AUDIO_PRESET_NAME               "PRESET_OBJECT_ONE"
#define FC_SXM_AUDIO_PRESET_SHORT_NAME         "PRESET_ONE"
#define FC_SXM_AUDIO_PRESET_BAND_ONE           "PRESET_BAND_ONE"
#define FC_SXM_AUDIO_PRESET_BAND_TWO           "PRESET_BAND_TWO"
#define FC_SXM_AUDIO_PRESET_BAND_THREE         "PRESET_BAND_THREE"

/* Defining featured favorite service configuration. 
   Need to update the same at Category list*/
#define FC_SXM_AUDIO_FEATURED_FAV_CAPACITY     32
#define FC_SXM_AUDIO_FEATURED_FAV_NUM_BANDS    32
#define FC_SXM_AUDIO_FEATURED_FAV_PURPOSE      FAVORITES_BAND_PURPOSE_CONVENIENCE 

/* Defining max presets in each band */
#define FC_SXM_AUDIO_MAX_PRESETS_IN_EACH_BAND  06
#define FC_SXM_AUDIO_MAX_PRESET_BANDS          03

/* Defining preset valid range */
#define FC_SXM_AUDIO_MIN_PRESET_COUNT          01
#define FC_SXM_AUDIO_MAX_PRESET_COUNT          06

/* Defining presets */
#define FC_SXM_PRESET_BAND_ONE                 01
#define FC_SXM_PRESET_BAND_TWO                 02
#define FC_SXM_PRESET_BAND_THREE               03
#define FC_SXM_PRESET_BAND_DEFAULT             255

/* Relative offsets of preset bands */
#define FC_SXM_BAND_ONE_OFFSET                 00
#define FC_SXM_BAND_TWO_OFFSET                 06
#define FC_SXM_BAND_THREE_OFFSET               12

/* Defining preset range for each bands */
#define FC_SXM_BAND_ONE_MIN_INDEX              01
#define FC_SXM_BAND_ONE_MAX_INDEX              06
#define FC_SXM_BAND_TWO_MIN_INDEX              07
#define FC_SXM_BAND_TWO_MAX_INDEX              12
#define FC_SXM_BAND_THREE_MIN_INDEX            13
#define FC_SXM_BAND_THREE_MAX_INDEX            18

/* Defining smart favorite channel count */
#define FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT      12

/* Defining Tune Scan Controls*/

// Scan defa
#define FC_SXM_TUNE_SCAN_TIME              06

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

struct fc_sxm_trMsgAudioSmsEvtTuneState;

/* MethodStart for recall preset from CCA client */
struct fc_sxm_trMsgAudioMStartRecallPreset
    : public fc_sxm_tclMessage 
{
    /* Service and service ID */
    SXM_MAKE_ACTION(
        fc_sxm_enServiceID_Audio,
        FC_SXM_AUDIO_ACTION_BASE_PRESETS + 1);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgRecallPresetMethodStart oFiMsg;
};

/* MethodStart for store preset from CCA client */
struct fc_sxm_trMsgAudioMStartStorePreset
    : public fc_sxm_tclMessage 
{
    /* Service and service ID */
    SXM_MAKE_ACTION(
        fc_sxm_enServiceID_Audio, 
        FC_SXM_AUDIO_ACTION_BASE_PRESETS + 2);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgStorePresetMethodStart oFiMsg;
};

/* MethodStart for set auto compare from CCA client */
struct fc_sxm_trMsgAudioMStartSetAutoCompare
    : public fc_sxm_tclMessage 
{
    /* Service and service ID */
    SXM_MAKE_ACTION(
        fc_sxm_enServiceID_Audio, 
        FC_SXM_AUDIO_ACTION_BASE_PRESETS + 3);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSetAutoCompareParametersMethodStart oFiMsg;
};

/* MethodStart for tune start control from CCA client */
struct fc_sxm_trMsgAudioMStartTuneStart : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 4);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgEnableTuneStartMethodStart oFiMsg;
};

/* MethodStart for smart favorite list control from CCA client */
struct fc_sxm_trMsgAudioMStartSFListControl : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 5);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSmartFavoriteListControlMethodStart oFiMsg;
};

/* MethodStart for GetPresetList from CCA client */
struct fc_sxm_trMsgAudioMStartGetPresetList : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 6);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgGetPresetListMethodStart oFiMsg;
};


/* MethodStart for smart favorite list control from CCA client */
struct fc_sxm_trMsgAudioMStartTuneScanControl : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 7);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodStart oFiMsg;
};

/* Message sent from SMS context containing current tune scan info */
struct fc_sxm_trMsgAudioSmsEvtTuneScanEvt : public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 8);
    fc_sxm_trMsgAudioSmsEvtTuneScanEvt():
        u8TuneScanState(0)
    {}

    UN8 u8TuneScanState;
};

/* Message for preset objects events */
struct fc_sxm_trMsgPresetBandUpdateCallback: public fc_sxm_tclMessage
{
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 9);
    fc_sxm_trMsgPresetBandUpdateCallback(PRESET_BAND_OBJECT hBand_, PRESETS_EVENT_MASK tEventMask_):
        hBand(hBand_),
        tEventMask(tEventMask_)
    {
        tCapacity = BAND.tCapacity(hBand);
        STRING_OBJECT hBandLongName = BAND.hLongName(hBand);
        fc_sxm_vCopySmsString2Stl(hBandLongName, sBandLongName);
    }

    PRESET_BAND_OBJECT hBand;
    PRESETS_EVENT_MASK tEventMask;
    size_t tCapacity;
    string sBandLongName;
};

// Message for User profile actions
struct fc_sxm_trMsgAudioMStartUserProfile : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 10);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgUserProfileMethodStart oFiMsg;
};

// Message for User profile actions
struct fc_sxm_trMsgAudioMStartGetPresetsConfig : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 11);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgGetPresetsConfigMethodStart oFiMsg;
};

// Message for User profile actions
struct fc_sxm_trMsgAudioMStartSetPresetList : public fc_sxm_tclMessage
{
    /* Service and service ID */
    SXM_MAKE_ACTION(fc_sxm_enServiceID_Audio, FC_SXM_AUDIO_ACTION_BASE_PRESETS + 12);
    /* Re-addressing info */
    fc_sxm_trAdressing rAdressing;
    /* Method start request */
    midw_ext_sxm_audiofi_tclMsgSetPresetListMethodStart oFiMsg;
};


struct fc_sxm_trFeaturedFavInfo{
    N16 n16CategoryId;
    size_t tBandId;
};

class fc_sxm_tclPresets;

struct fc_sxm_trFeaturedBandIteratorCBArg {
    fc_sxm_tclPresets* poFeaturedPresets;
    tBool bFeaturedFavModified;
};
struct fc_sxm_trMsgAudioSmsEvtChannelEvt;
/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*=============================================================================
=======                              CLASSES                            =======
=============================================================================*/

/*****************************************************************************
 * CLASS        : fc_sxm_tclPresets                                          *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION  : Class to handle all preset related services in             *
 *                fc sxm audio component                                     *
 *****************************************************************************/
class Ifc_sxm_tclAudioApp;

class fc_sxm_tclPresets
    : public fc_sxm_tclAudioMember
    , public Ifc_sxm_tclPresets
    , public fc_sxm_tclSingleton<fc_sxm_tclPresets>
{
    /*=========================================================================
    =======                     public section                          =======
    =========================================================================*/
    public:

        /*********************************************************************
         * DESCRIPTION : Class destructor                                    *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        ~fc_sxm_tclPresets();

        /*********************************************************************
         * DESCRIPTION : Method that will start preset service once decoder  *
         *               is up and is in ready state                         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hDecoderObject : Valid decoder object from SXM      *
         * PARAMETER   : enDecoderState : Current decoder state              *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vUpdateDecoderState(DECODER_OBJECT hDecoderObject, 
                                  DECODER_STATE_ENUM enDecoderState);

        /*********************************************************************
         * DESCRIPTION : Method to fetch persisted preset and band info      *
         *               and store them in internal preset list              *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hBand        - Valid preset band object             *
         * PARAMETER   : hBandIndex   - Preset band index being iterated     *
         * PARAMETER   : hPresetIndex - Preset index in preset band          *
         * PARAMETER   : hChannel     - Valid channel object                 *
         * RETURNVALUE : N16          - Steps to iterate further             *
         *********************************************************************/
        N16 s16UpdatePresetInfo(PRESETS_OBJECT hPresets,PRESET_BAND_OBJECT hBand,
                                size_t hBandIndex,
                                size_t hPresetIndex,
                                CHANNEL_OBJECT hChannel);

        /*********************************************************************
         * DESCRIPTION : Get the preset index for given service ID           *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hServiceID   - Valid service ID                     *
         * RETURNVALUE : tU8          - Preset index value                   *
         *********************************************************************/
        tU8 u8GetPresetIndex(SERVICE_ID hServiceID) const;

        /*********************************************************************
         * DESCRIPTION : Send auto compare property update for current tuned *
         *               channel                                             *
         * ----------------------------------------------------------------- *
         * PARAMETER   : hServiceID - Current tuned channel service ID       *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vNotifyTunedChannelPresetIndex(SERVICE_ID hServiceID);

        /* ================== CCA Request Handlers ========================= */

        tVoid vProcess(fc_sxm_trMsgPresetBandUpdateCallback const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to RecallPreset                         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Recall preset method start request          *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartRecallPreset const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to store preset                         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Store preset method start request           *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartStorePreset const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to set auto compare parameter           *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Set auto compare parameters method start    *
         *               request                                             *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartSetAutoCompare const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to enable or disable tune start         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Tune start method start request             *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartTuneStart const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request for Smart favorite list control         *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - SF List control method start request        *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartSFListControl const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to Get Preset List                      *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - GetPresetList method start request          *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartGetPresetList const *prMsg);

           /*********************************************************************
         * DESCRIPTION : CCA request for Tune Scan control                   *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Tune scan control method start request       *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartTuneScanControl const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to User profile action                  *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Method for various User profile actions     *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartUserProfile const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to Get Preset Configuration             *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Method to get preset Configuration          *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartGetPresetsConfig const *prMsg);

        /*********************************************************************
         * DESCRIPTION : CCA request to Set Preset list                      *
         * ----------------------------------------------------------------- *
         * PARAMETER   : prMsg - Set presets list method start request       *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vProcess(fc_sxm_trMsgAudioMStartSetPresetList const *prMsg);

        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtTuneScanEvt const *prMsg);
        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtChannelEvt const *);
    #ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
        tVoid vProcess(fc_sxm_trMsgPresetsProfileChange const * /* prMsg */ );
    #endif

        /* Clear class member variables */
        tVoid vClearMemberData(tVoid);

        /* Update smart favorite list channels */
        N16 s16updateSFChannelInfo(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX tCurrentIndex, CHANNEL_OBJECT hChannel, tVoid* pItrData=OSAL_NULL);

        /* Update instant replay mode */
        tVoid vUpdateIRMode(tBool bValidate, tBool bSetMode);
        tVoid vNotifyIRMode(tBool bValidate, tBool bSetMode);

        /* Set Audio App Pointer Reference*/
        tVoid vSetApp(Ifc_sxm_tclAudioApp* audioApp);

        N16 n16AddPreset(PRESETS_OBJECT hPresets, PRESET_BAND_OBJECT hBand, size_t tBandIndex,
                CHANNEL_OBJECT hChannel, STRING_OBJECT hPresetName, size_t tPresetindex);

        N16 n16AddFeaturedPreset(PRESETS_OBJECT hPresets, PRESET_BAND_OBJECT hBand, size_t tBandIndex,
                CHANNEL_OBJECT hChannel, STRING_OBJECT hPresetName, size_t tPresetindex, tBool bFeaturedFavModified);

        tVoid vPresetBandUpdate(fc_sxm_trChannel const &, CHANNEL_EVENT_MASK enEventMask) const;

        tVoid vUpdateFeaturedBandInfo(PRESET_BAND_OBJECT hBand, PRESETS_EVENT_MASK hEventMask, string sBandLongName, size_t tCapacity);

        tVoid vGetPresetIndices(map<SERVICE_ID, tU8>& ) const;

        tVoid vSetTuneScanStatus(tBool bSetTuneScanState);
        tVoid vCheckChannelStatus(tVoid);
        tVoid vSetSavedChannelId();
        CHANNEL_ID u16GetSavedChannelId() const;
        N16 s16UserCategoryRemoveChannel(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX tCurrentIndex, CHANNEL_OBJECT hChannel, tVoid* pIterData);
        tVoid vProcess(fc_sxm_trMsgAudioSmsEvtTuneState const * /* prMsg */ );

        tBool bHandleChangeProfilePresetList();


    private:
        string sPacStringName(STRING_OBJECT hName) const;

        /* Method to clear all presets of a given band*/
        SMSAPI_RETURN_CODE_ENUM enClearAllPresets(PRESET_BAND_OBJECT hBand, tBool bClearNames = TRUE) const;

        /* Method to set preset*/
        SMSAPI_RETURN_CODE_ENUM enSetPreset(PRESET_BAND_OBJECT hBand, tU8 u8PresetIndex, CHANNEL_ID tChannelId) const;

        /* Method to tune to the selected preset channel.*/
        SMSAPI_RETURN_CODE_ENUM enBandTune(PRESET_BAND_OBJECT hBand, tU8 u8PresetIndex, tBool bLockOverride = TRUE) const;

        /* Method to iterate contents of a preset band.*/
        SMSAPI_RETURN_CODE_ENUM enIteratePresetBand(PRESET_BAND_OBJECT hBand);

        /* Method to retrieve the appropriate Preset Band Object from preset band list index*/
        PRESET_BAND_OBJECT hGetPresetBandObject(const tU32 u32PresetBand) const;

        /* Method retrieves the number of preset band objects associted with the Presets object */
        size_t tGetPresetBandsSize() const;

        /* Method prints the PresetChannel's Info*/
        tVoid vPrintPresetChannelInfo(const midw_ext_fi_tcl_PresetListEntry& trPreset) const;
        
        /*Method to get current tune state*/
        tBool bGetTuneScanStatus() const
        {
            return _bIsScanInProgress;
        }

        tVoid vGetBandLongName(PRESET_BAND_OBJECT hBand, string& strBandLongName) const;
        tVoid vNotifyTuneScanContentAvailable(tBool bTuneScanState);
        tVoid vNotifyTuneScanAborted();
        tVoid vCheckSMSCfgError();
        tBool vCheckInvalidPreset();
        tBool bValidatePresetIndices(tU8 u8FirstElement, tU8 u8LastElement);
    /*=========================================================================
    =======                     private section                         =======
    =========================================================================*/
    private:

        /* ===================== Friend Classes ============================ */

        /* Defining singleton class as a friend class in category list class */
        friend class fc_sxm_tclSingleton<fc_sxm_tclPresets>;

        /* ==================== Member Functions =========================== */

        /*********************************************************************
         * DESCRIPTION : Class constructor                                   *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        fc_sxm_tclPresets();

        /*********************************************************************
         * DESCRIPTION : Method to initialize preset object and create       *
         *               preset bands                                        *
         * ----------------------------------------------------------------- *
         * PARAMETER   : None                                                *
         * RETURNVALUE : None                                                *
         *********************************************************************/
        tVoid vInitPresetsAndBands(tVoid);

        /* Get smart favorite preset list */
        tVoid GetSFChannelList(vector<midw_ext_fi_tcl_SFListElement> &SFPresetList, tBool &EnableAddPreset) const;
        /* Get non smart favorite preset list */
        tVoid GetNonSFChannelList(vector<midw_ext_fi_tcl_SFListElement> &SFPresetList) const;
        /* Update smart favorite list */
        tVoid vUpdateSFList(tVoid);
        /* Update preset smart favorite list */
        tVoid vUpdatePresetSFList(tVoid) const;
        /* Check and remove preset from sf list */
        tVoid vCheckAndRemoveSFPreset(tU8 u8PresetIndex);
        /* Evaluate Preset channel change impact on Smart Favorites List*/
        tVoid vUpdateSmartFavorites(tU8 u8PresetIndex);
        /* Method Fills FI element midw_ext_fi_tcl_PresetListEntry from the supplied channel*/
        tVoid vFillPresetChannelInfo(fc_sxm_trChannel const& rChannel,
                midw_ext_fi_tcl_PresetListEntry& trPreset,
                tU32 tPresetBandIndex, bool bRefreshArt = false) const;
         /*method to start tune scan*/
        tVoid vTuneScanStart(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj);
        /*method to stop tune scan*/
        tVoid vTuneScanStop(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj);
        /*method to skip forward tune scan*/
        tVoid vTuneScanSkipForward(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj);
        /*method to skip backward tune scan*/
        tVoid vTuneScanSkipBackward(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj);
        /*method to abort tune scan*/
        tVoid vTuneScanAbort(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj);
        /*method to configure tune scan*/
        tVoid vTuneScanConfigure();
        /*method to start featured favorite*/
        tVoid vStartFeaturedFavorites();
        /* Method to create user category */
        CATEGORY_ID tCreateUserCategory(size_t tCapacity, const string& sBandLongName);
        /* Method to add channels into user category */
        tVoid vAddChannelIntoCategory(const string& sBandLongName, SERVICE_ID hServiceId);

        /* Method to add featured preset user category */
        tVoid vAddFeaturedBands(PRESET_BAND_OBJECT hBand, const string& sBandLongName, size_t tCapacity);
        /* Method to remove featured user category */
        tVoid vRemoveFeaturedBands(const string& sBandLongName);
        /* Method to modify featured user category */
        tVoid vModifyFeaturedBands(PRESET_BAND_OBJECT hBand, const string& sBandLongName, size_t tCapacity);
        /* Utility method which can be used for debugging purpose to know the list of Preset and Smat Favorites*/
        tVoid vPrintSFAndPresetsList() const;
        DECODER_OBJECT hGetDecoder() const;
        tBool bGetAllPresetIndices(tU32, midw_ext_sxm_audiofi_tclMsgGetPresetListMethodResult &, midw_ext_fi_tcl_PresetListEntry &);
        N16 n16GetCategoryId(size_t tBandIndex) const;
        N16 n16GetCategoryId(const string& strBandName) const;
        tVoid vRenameUserCategory(N16 n16CategoryId, const string& sBandLongName);
        tVoid vUpdateUserCategory(N16 n16CategoryId, const string& sBandLongName);
        tVoid vRecoverSFList();
        tVoid vStartSmartFavorite();

        PRESET_BAND_OBJECT hGetPresetBand(tU32 u32BandNumber) const;
        tU8 u8GetRelativePresetIndex(tU8 lPresetIndex);
        tVoid vResetPresetList();
        tVoid vResetSFList();
        tBool bSetProfilePresetList(SERVICE_ID *pPresetList);
        // Send property update for Preset List Change
        tVoid vPresetListChangeNotification();

        tVoid vPrintPresetList() const;
        tVoid vPrintSFList() const;
         tU8 u8GetUserProfileID() const{
            return _u8UserProfileID;
        }

        tVoid vSetUserProfileID(tU8 u8ProfileID){
            _u8UserProfileID = u8ProfileID;
        }

        tBool bIsValidSID (tU16 u16SID) const {
        return (u16SID != SERVICE_INVALID_ID);

        }
        tBool bIsValidIndex(tU16 u16Index) const {
            return (u16Index < FC_SXM_BAND_THREE_MAX_INDEX);
        }

        tBool bPresetListUpdate(SERVICE_ID *aPresetList, const vector<midw_ext_fi_tcl_PresetEntry>& VecPresetList)const;

        tVoid vCreateMixedPresetBand();

        /* ==================== Member Variables =========================== */

        tBool _bPresetsInitialized; /* Flag to handle preset and preset band creation */
        tBool _bSmartFavoriteEnabled; /* Flag to handle smart favorite feature*/
        /* To hold current preset band info */
        tU32 _u32CurrentBandNumber;
        /* To handle current auto compare preset element */
        tU8 _u8PreviousTunerElemNr;
        /* Tune start enable type */
        SMART_FAVORITES_PLAY_POINT_CTRL_ENUM _ePlayPoint;
        /* Preset object */
        PRESETS_OBJECT _hPresetObject;
        /* Preset band objects */
        PRESET_BAND_OBJECT _hBandObjectOne;
        PRESET_BAND_OBJECT _hBandObjectTwo;
        PRESET_BAND_OBJECT _hBandObjectThree;

        tBool _bIsContentAvailable;
        tBool _bIsScanInProgress;

        /* Array to hold preset service ID's */
        SERVICE_ID *_u16PresetChannels;

        /* Array to hold profile preset service ID's until decoder is ready */
        SERVICE_ID *_u16ProfilePresetChannels;


        /* Array to hold smart favorite channel service ID's */
        SERVICE_ID _u16SFChannels[FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT];

        Ifc_sxm_tclAudioApp* _poAudioApp;

        map < tU32, fc_sxm_trChannel > _mPresetChannels;
        size_t _tLastBandIndex;
        size_t _tLastPresetIndex;
        map <string, fc_sxm_trFeaturedFavInfo> _mFeaturedFavInfo;
        CHANNEL_ID _u16ChannelIdBeforeScan;
        vector<SERVICE_ID> _vectServiceId;
        SERVICE_ID _u16StoredServiceid;

        /* variables for user profile actions */
        tU8 _u8UserProfileID;
        tBool _bIsProfileChange;

        tBool _bIsMixedPreset;
        tU8 _u8NumberOfPresets;

        tU8 _u8MaxNumberOfPresets;
        tU8 _u8MaxNumberOfBands;

        tU8 _u8MaxCountOfPresetsInOneBand;

};

#endif

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
