/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_presets.cpp
* @brief       Implementation for functionalities related to Presets and Smart Favorites.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/

#include "fc_sxm_common.h"
#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_tcl_presets.h"
#include "fc_sxm_tcl_advisories.h"
#include "fc_sxm_tcl_audio_properties.h"
#include "fc_sxm_tcl_states.h"
#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_channel_list_if.h"
#include "fc_sxm_tcl_playback_if.h"
#include "fc_sxm_tcl_audio_utils.h"
#include "fc_sxm_tcl_pers_data.h"


/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PRESETS
#include "trcGenProj/Header/fc_sxm_tcl_presets.cpp.trc.h"
#endif

#define FC_SXM_AUDIO_CREATE_INITIAL_SF_LIST 0
#define FC_SXM_AUDIO_TO_DEBUG               1
#define FC_SXM_PREVIEW_CHANNEL_ID           1

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/

/*=============================================================================
=======                   PROTOTYPES & FUNCTIONS                        =======
=============================================================================*/

/*****************************************************************************
 * FUNCTION    : cb_vPresetIterateHandler                                    *
 * CLASS       : None                                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Preset iterator event callback function                     *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hPresets     - Valid preset object                          *
 * PARAMETER   : hBand        - Valid band object                            *
 * PARAMETER   : tBandIndex   - Band index that is being iterated            *
 * PARAMETER   : hChannel     - Valid channel object in current preset       *
 * PARAMETER   : hPresetName  - Preset name                                  *
 * PARAMETER   : tPresetindex - Preset index in band that is iterated        *
 * PARAMETER   : pvIterateArg - Data from calling function                   *
 * RETURNVALUE : N16          - Steps to iterate further                     *
 *****************************************************************************/
static N16 cb_vPresetIterateHandler(PRESETS_OBJECT hPresets, PRESET_BAND_OBJECT hBand, size_t tBandIndex, 
                                    CHANNEL_OBJECT hChannel, STRING_OBJECT /* hPresetName */, size_t tPresetindex, void* /* pvIterateArg */ )
{
   ETG_TRACE_USR4(("cb_vPresetIterateHandler()"));
   /* Process preset update and request for further data if required */
   N16 hReturnData = fc_sxm_tclPresets::instance()->s16UpdatePresetInfo(hPresets, hBand, tBandIndex, tPresetindex, hChannel);
   return (hReturnData);
}


/*****************************************************************************
 * FUNCTION    : cb_vBandIterateHandler                                    *
 * CLASS       : None                                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : This is a specialized iterator function used to iterate the
 * contents of the presets service.  The direction of iteration is controlled
 * via the return code this function provides to the PRESETS/BAND iterator
 * API.  A return value of -1 indicates that iteration is to continue in the
 * reverse direction, a return value of 1 indicates that iteration is to
 * continue in the forward direction, and a return value of 0 indicates that
 * iteration is to stop.                     *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hPresets     - Valid preset object                          *
 * PARAMETER   : hBand        - Valid band object                            *
 * PARAMETER   : tBandIndex   - Band index that is being iterated            *
 * PARAMETER   : hChannel     - Valid channel object in current preset       *
 * PARAMETER   : hPresetName  - Preset name                                  *
 * PARAMETER   : tPresetindex - Preset index in band that is iterated        *
 * PARAMETER   : pvIterateArg - Data from calling function                   *
 * RETURNVALUE : N16          - Steps to iterate further                     *
 *****************************************************************************/
static N16 cb_vBandIterateHandler(PRESETS_OBJECT hPresets, PRESET_BAND_OBJECT hBand, size_t tBandIndex,
                                    CHANNEL_OBJECT hChannel, STRING_OBJECT hPresetName, size_t tPresetindex, void* pvIterateArg)
{
   ETG_TRACE_USR1(("cb_vBandIterateHandler()"));
   N16 n16Ret = 0;
   fc_sxm_tclPresets* poPresets = static_cast < fc_sxm_tclPresets* > (pvIterateArg);
   if ( OSAL_NULL != poPresets )
   {
       n16Ret = poPresets->n16AddPreset (hPresets, hBand, tBandIndex, hChannel, hPresetName, tPresetindex);
   }
   return n16Ret;
}

static N16 cb_vFeaturedBandIterateHandler(PRESETS_OBJECT hPresets, PRESET_BAND_OBJECT hBand, size_t tBandIndex,
        CHANNEL_OBJECT hChannel, STRING_OBJECT hPresetName, size_t tPresetindex, void* pvIterateArg)
{
    ETG_TRACE_USR1(("cb_vFeaturedBandIterateHandler()"));
    fc_sxm_trFeaturedBandIteratorCBArg* poPresetIteratorCBArg = static_cast <fc_sxm_trFeaturedBandIteratorCBArg *> (pvIterateArg);
    N16 n16Ret = 0;
    if ((poPresetIteratorCBArg) &&
            (poPresetIteratorCBArg->poFeaturedPresets))
    {
       n16Ret = poPresetIteratorCBArg->poFeaturedPresets->n16AddFeaturedPreset(hPresets, hBand, tBandIndex, hChannel, hPresetName, tPresetindex, poPresetIteratorCBArg->bFeaturedFavModified);
    }
    return n16Ret;
}


/*****************************************************************************
 * FUNCTION    : cb_s16SFListIterateHandler                                  *
 * CLASS       : None                                                        *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Smart Favorite channel iterator event callback              *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategory       - Valid category object                     *
 * PARAMETER   : s16CurrentIndex - Channel index in category iterated        *
 * PARAMETER   : hChannel        - Valid channel object                      *
 * PARAMETER   : pvIterateArg    - Data from calling function                *
 * RETURNVALUE : N16             - Steps to iterate further                  *
 *****************************************************************************/
static N16 cb_s16SFListIterateHandler(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX s16CurrentIndex, CHANNEL_OBJECT hChannel, tVoid* pvIterateArg)
{
   ETG_TRACE_USR4(("cb_s16SFListIterateHandler(), CategoryId - %d  s16CurrentIndex - %d", CATEGORY.tGetCategoryId(hCategory), s16CurrentIndex));
   /* Process channel object */
   N16 hReturnData = fc_sxm_tclPresets::instance()->s16updateSFChannelInfo(hCategory, s16CurrentIndex, hChannel, pvIterateArg);
   return (hReturnData);
}
// Callback method to remove all channels in user category. This callback method may also be used to remove channels from Smart Favorites category
static N16 cb_s16RemoveAllCategoryChannels(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX s16CurrentIndex, CHANNEL_OBJECT /*hChannel*/, tVoid* /*pvIterateArg*/)
{
    ETG_TRACE_USR4(("cb_s16UserCategoryRemoveChannelIterateHandler(), CategoryId - %d  s16CurrentIndex - %d", CATEGORY.tGetCategoryId(hCategory), s16CurrentIndex));
    CATEGORY.eRemoveChannel(hCategory);
    return 1;
}

/* Event Callback for Featured Favorites Bands */
void cb_vPresetNotifyHandler(PRESETS_OBJECT /* _hPresetObject */, PRESET_BAND_OBJECT hBand, PRESETS_EVENT_MASK tEventMask, void * /* pvCallbackArg */ )
{
    ETG_TRACE_USR4(("cb_vPresetNotifyHandler()"));

    /* Updating only the featured Bands */
    if(PRESET_BAND_TYPE_FEATURED == BAND.eType(hBand))
    {
      fc_sxm_trMsgPresetBandUpdateCallback prMsgPresetBandUpdate(hBand, tEventMask);
      fc_sxm_tclAudioApp::instance()->vPostMsgNew(prMsgPresetBandUpdate);
    }
}

static N16 cb_s16UserCategoryRemoveChannelIterateHandler(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX s16CurrentIndex, CHANNEL_OBJECT hChannel, tVoid* pvIterateArg)
{
    ETG_TRACE_USR4(("cb_s16UserCategoryRemoveChannelIterateHandler(), CategoryId - %d  s16CurrentIndex - %d", CATEGORY.tGetCategoryId(hCategory), s16CurrentIndex));
    fc_sxm_tclPresets* poPresets = static_cast < fc_sxm_tclPresets* > (pvIterateArg);
    N16 hReturnData = 0;
    if ( OSAL_NULL != poPresets )
    {
        hReturnData = poPresets->s16UserCategoryRemoveChannel(hCategory, s16CurrentIndex, hChannel, pvIterateArg);
    }
    return (hReturnData);
}

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/*****************************************************************************
 * FUNCTION    : fc_sxm_tclPresets                                           *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class constructor                                           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclPresets::fc_sxm_tclPresets()
    : _bPresetsInitialized(FALSE)
    , _bSmartFavoriteEnabled(FALSE)
    , _bIsContentAvailable(TRUE)
    , _bIsScanInProgress(FALSE)
    , _u32CurrentBandNumber(1)
    , _u8PreviousTunerElemNr(0)
    , _ePlayPoint(SMART_FAVORITES_PLAY_POINT_CTRL_START)
    , _hPresetObject(PRESETS_INVALID_OBJECT)
    , _hBandObjectOne(PRESET_BAND_INVALID_OBJECT)
    , _hBandObjectTwo(PRESET_BAND_INVALID_OBJECT)
    , _hBandObjectThree(PRESET_BAND_INVALID_OBJECT)
    , _poAudioApp(NULL)
    , _tLastBandIndex(0U)
    , _tLastPresetIndex(0U)
    , _u16ChannelIdBeforeScan(0)
    , _u16StoredServiceid(SERVICE_INVALID_ID)
    , _bIsProfileChange(FALSE)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets constructor start"));

   _bIsMixedPreset = fc_sxm_tclConfig::instance()->bGetPresetType();
   _u8NumberOfPresets = fc_sxm_tclConfig::instance()->u8GetNumberOfPresets();

   ETG_TRACE_USR4(("fc_sxm_tclPresets constructor u8PresetType: %d, u8NumberOfPresets: %u", _bIsMixedPreset, _u8NumberOfPresets));



   _u8MaxNumberOfPresets = FC_SXM_AUDIO_MAX_PRESETS_IN_EACH_BAND;
   _u8MaxNumberOfBands = FC_SXM_AUDIO_MAX_PRESET_BANDS;
   _u8MaxCountOfPresetsInOneBand = FC_SXM_AUDIO_MAX_PRESET_COUNT;

   if(_bIsMixedPreset)
   {
       _u8MaxNumberOfPresets = _u8NumberOfPresets;
       _u8MaxNumberOfBands = 1;
       _u8MaxCountOfPresetsInOneBand = _u8NumberOfPresets;
   }
   _u16PresetChannels = new SERVICE_ID[_u8MaxNumberOfPresets * _u8MaxNumberOfBands];
   for(U32 uLoopCount = 0; uLoopCount < (_u8MaxNumberOfPresets * _u8MaxNumberOfBands); uLoopCount++) {
         _u16PresetChannels[uLoopCount] = FC_SXM_AUDIO_DEFAULT_CHANNEL;
   }

   _u16ProfilePresetChannels = new SERVICE_ID[_u8MaxNumberOfPresets * _u8MaxNumberOfBands];

   /* Initialize preset array */
   memset(_u16ProfilePresetChannels, 0x01, (_u8MaxNumberOfPresets * _u8MaxNumberOfBands)*sizeof(SERVICE_ID));

   /* Initialize smart favorite channel array */
   for(U32 uLoopCount = 0; uLoopCount < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; uLoopCount++) {
      _u16SFChannels[uLoopCount] = SERVICE_INVALID_ID;
   }
   /* Initialization of user profile ID
    * if user profile ID is 255 then it is invalid case (personalization not supported)  */
    fc_sxm_tclPersData::instance()->vGetDPProfileId(_u8UserProfileID);

   ETG_TRACE_USR4(("fc_sxm_tclPresets constructor end"));
}

/*****************************************************************************
 * FUNCTION    : ~fc_sxm_tclPresets                                          *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Class destructor                                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
fc_sxm_tclPresets::~fc_sxm_tclPresets()
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets destructor"));
   /* Set all the class member variables to default values */
   _bPresetsInitialized = FALSE;
   _bSmartFavoriteEnabled = FALSE;
   _ePlayPoint          = SMART_FAVORITES_PLAY_POINT_CTRL_START;
   _hPresetObject       = PRESETS_INVALID_OBJECT;
   _hBandObjectOne      = PRESET_BAND_INVALID_OBJECT;
   _hBandObjectTwo      = PRESET_BAND_INVALID_OBJECT;
   _hBandObjectThree    = PRESET_BAND_INVALID_OBJECT;
   _poAudioApp          = NULL;
   _u16ChannelIdBeforeScan = CHANNEL_INVALID_ID;

   delete[] _u16PresetChannels;
   delete[] _u16ProfilePresetChannels;
}

/*****************************************************************************
 * FUNCTION    : vUpdateDecoderState                                         *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method that will start preset service once decoder is up    *
 *               and is in ready state                                       *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hDecoderObject - Valid decoder object from SXM              *
 * PARAMETER   : enDecoderState - Current decoder state                      *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vUpdateDecoderState(DECODER_OBJECT hDecoderObject, DECODER_STATE_ENUM enDecoderState)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vUpdateDecoderState()"));
   /* Initialize preset object and create preset bands for the first time */
   if((DECODER_STATE_READY == enDecoderState) && (FALSE == _bPresetsInitialized) && (DECODER_INVALID_OBJECT != hDecoderObject)) {
      ETG_TRACE_USR4(("Creating presets objects and preset bands"));
      /* Initialize presets object and create presets bands */
      vInitPresetsAndBands();

      Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
              _poAudioApp->poGetChannelListInstance(): NULL;
      SXM_ASSERT_RETURN(NULL != poChannelList)

      /* Notify auto compare properties */
      vNotifyTunedChannelPresetIndex(poChannelList->u16GetSIDOfTunedChannel());

      //start smart favorite service
      vStartSmartFavorite();

      //Set Preset list for profile changes request once decoder is in ready state
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
      if((_poAudioApp != NULL) && (TRUE == _poAudioApp->bIsProfileChange())) {
          bSetProfilePresetList(_u16ProfilePresetChannels);
      }
#endif
      if(_bIsProfileChange)
      {
          bSetProfilePresetList(_u16ProfilePresetChannels);
          _bIsProfileChange = false;
      }


      vRecoverSFList();

      /*Tune Scan Configuration*/
      vTuneScanConfigure();
   }
}

/* Start the Smart Favorite Service
 *
 */
tVoid fc_sxm_tclPresets::vStartSmartFavorite()
{
     SMSAPI_RETURN_CODE_ENUM eReturnCode =
          SMART_FAVORITES.eStart(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), FALSE);
     if((SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) || (SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED == eReturnCode)) {
        if(SMSAPI_RETURN_CODE_SUCCESS != SMART_FAVORITES.eEnable(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), SMART_FAVORITES_PLAY_POINT_CTRL_AUTOMATIC)) {
           ETG_TRACE_ERR(("Failed to enable smart favorite service -%d",ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode)));
        } else {
           ETG_TRACE_USR4(("SMART_FAVORITES.eEnable() successful"));
           _bSmartFavoriteEnabled = TRUE;  /* Smart Favorite feature enabled */
           vUpdateSFList(); /* Then update smart favorite list */
        }
     } else {
        ETG_TRACE_ERR(("Failed to start smart favorite service - %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode) ));
     }
}


tVoid fc_sxm_tclPresets::vUpdateSFList(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vUpdateSFList()"));
   if(_bSmartFavoriteEnabled) {
      CATEGORY_ID eSmartFavCatId = SMART_FAVORITES.tCategoryId(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder());

#if(FC_SXM_AUDIO_CREATE_INITIAL_SF_LIST)
      /* TODO: This needs proper update */
      ETG_TRACE_USR2(("Category ID of SMART_FAVORITES.tCategoryId - %d", eSmartFavCatId));
      ETG_TRACE_USR2(("Smart Fav cat size - %d", CATEGORY.tSize(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId)));
      SMSAPI_RETURN_CODE_ENUM eRetEnum = SMSAPI_RETURN_CODE_ERROR;
      eRetEnum = CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, 3);
      if(SMSAPI_RETURN_CODE_ERROR == eRetEnum) { ETG_TRACE_ERR(("Error Error Error 1")); }
      eRetEnum = CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, 8);
      if(SMSAPI_RETURN_CODE_ERROR == eRetEnum) { ETG_TRACE_ERR(("Error Error Error 2")); }
      eRetEnum = CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, 122);
      if(SMSAPI_RETURN_CODE_ERROR == eRetEnum) { ETG_TRACE_ERR(("Error Error Error 3")); }
      eRetEnum = CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, 84);
      if(SMSAPI_RETURN_CODE_ERROR == eRetEnum) { ETG_TRACE_ERR(("Error Error Error 4")); }
      eRetEnum = CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, 148);
      if(SMSAPI_RETURN_CODE_ERROR == eRetEnum) { ETG_TRACE_ERR(("Error Error Error 5")); }
      eRetEnum = CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, 147);
      if(SMSAPI_RETURN_CODE_ERROR == eRetEnum) { ETG_TRACE_ERR(("Error Error Error 6")); }
      ETG_TRACE_USR2(("Smart Fav cat size - %d", CATEGORY.tSize(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId) ));
#endif

      /* Initialize smart favorite channel array */
      U32 uLoopCount = 0;
      for(uLoopCount = 0; uLoopCount < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; uLoopCount++) {
         _u16SFChannels[uLoopCount] = SERVICE_INVALID_ID;
      }

      /* Get smart favorite channel service ID's */
      SMSAPI_RETURN_CODE_ENUM hReturnCode = CATEGORY.eIterate(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), 
                                                              eSmartFavCatId, cb_s16SFListIterateHandler, NULL);
      if(SMSAPI_RETURN_CODE_SUCCESS != hReturnCode) {
         ETG_TRACE_ERR(("Failed to register for channel callback for SF category %d, ERROR_CODE - %d", 
                        eSmartFavCatId, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, hReturnCode)));
      }
   }
}

/* Update preset smart favorite list */
tVoid fc_sxm_tclPresets::vUpdatePresetSFList(tVoid) const
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vUpdatePresetSFList()"));

   /* Auto compare property */
   fc_sxm_tcl_trAudioPropertyAutoCompare AutoCompareStatus;
   /* Retain earlier highlighted preset number */
   ((AutoCompareStatus).oFiRes).TunerElemNr = _u8PreviousTunerElemNr;
   /* Get index and range based on current preset band */
   tU8 u8LoopCount = (tU8)(((_u32CurrentBandNumber - 1) * _u8MaxCountOfPresetsInOneBand));
   tU8 u8LoopRange = (tU8)((u8LoopCount + _u8MaxCountOfPresetsInOneBand));
   /* Within given band search for smart favorite presets */
   for(; u8LoopCount <u8LoopRange; u8LoopCount++) {
      /* Create SF List element */
      midw_ext_fi_tcl_SmartFavoritePreset tSFListElement;
      tSFListElement.PresetIndex = (tU8)(u8LoopCount+1);
      tSFListElement.IsSmartFavorite = FALSE;
      /* Search if preset is smart favorite preset */
      for(tU8 u8SFLoopCount=0; u8SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u8SFLoopCount++) {
         if(_u16PresetChannels[u8LoopCount] == _u16SFChannels[u8SFLoopCount]) {
            tSFListElement.IsSmartFavorite = TRUE;
            ETG_TRACE_USR4(("AUTO SF - %d", tSFListElement.PresetIndex));
            break;
         }
      }
      /* Insert list element in auto compare property */
      (((AutoCompareStatus).oFiRes).SmartFavoriteList).push_back(tSFListElement);
   }

   /* Set the status info and notify all clients */
   fc_sxm_tclAudioProperties::instance()->oAutoCompare.vSet(AutoCompareStatus);
   fc_sxm_tclAudioProperties::instance()->oAutoCompare.vNotify();
}

/* Clear class member variables */
tVoid fc_sxm_tclPresets::vClearMemberData(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vClearMemberData()"));
   /* Clear member variables */
   _bPresetsInitialized = FALSE;
   _bSmartFavoriteEnabled = FALSE;
   _hPresetObject       = PRESETS_INVALID_OBJECT;
   _hBandObjectOne      = PRESET_BAND_INVALID_OBJECT;
   _hBandObjectTwo      = PRESET_BAND_INVALID_OBJECT;
   _hBandObjectThree    = PRESET_BAND_INVALID_OBJECT;

   /* Set the status info and notify all clients */
   fc_sxm_tcl_trAudioPropertyAutoCompare AutoCompareStatus;
   ((AutoCompareStatus).oFiRes).TunerElemNr = 0;
   _u8PreviousTunerElemNr = 0;
   fc_sxm_tclAudioProperties::instance()->oAutoCompare.vSet(AutoCompareStatus);
   fc_sxm_tclAudioProperties::instance()->oAutoCompare.vNotify();

   /* Instant Replay mode property */
   fc_sxm_tcl_trAudioPropertyIRMode IRMode;
   (IRMode.oFiRes).IsIRModeActive = FALSE;  /* Set to default value */
   fc_sxm_tclAudioProperties::instance()->oInstantReplayMode.vSet(IRMode);
   fc_sxm_tclAudioProperties::instance()->oInstantReplayMode.vNotify();

   /* Disable smart favorite */
   if(SMSAPI_RETURN_CODE_SUCCESS != SMART_FAVORITES.eDisable(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder())) {
      ETG_TRACE_ERR(("Failed to disable smart favorite service"));
   }
   /* Stop smart favorite service */
   SMART_FAVORITES.vStop(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder());

   /* Initialize smart favorite channel array */
   for(U32 uLoopCount = 0; uLoopCount < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; uLoopCount++) {
      _u16SFChannels[uLoopCount] = SERVICE_INVALID_ID;
   }
   _mFeaturedFavInfo.clear();
}

/*****************************************************************************
 * FUNCTION    : vInitPresetsAndBands                                        *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to initialize preset object and create preset bands  *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vInitPresetsAndBands(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vInitPresetsAndBands()"));
    /* Start preset object */
    SMSAPI_RETURN_CODE_ENUM enReturnCode = PRESETS.eStart(&_hPresetObject, fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
            FC_SXM_AUDIO_PRESET_NAME, FC_SXM_AUDIO_PRESET_SHORT_NAME);
    if(SMSAPI_RETURN_CODE_LOAD_SERVICE == enReturnCode) {
        ETG_TRACE_USR3(("Previous preset config file loaded successfully"));
        _bPresetsInitialized = TRUE;
    } else if(SMSAPI_RETURN_CODE_NEW_SERVICE == enReturnCode) {
        ETG_TRACE_USR3(("New preset config file loaded successfully"));
        _bPresetsInitialized = TRUE;
    } else {
        ETG_TRACE_ERR(("Failed to start preset object"));
    }

    if(TRUE == _bPresetsInitialized) {
        size_t uNumberOfBands = PRESETS.tNumBands(_hPresetObject);

        if(0 == uNumberOfBands) {
            ETG_TRACE_USR3(("No Preset bands available, creating new bands"));

            if(_bIsMixedPreset)
            {
                vCreateMixedPresetBand();
            }
            else
            {
                /* Create preset band one */
                _hBandObjectOne = BAND.hCreate(_hPresetObject, _u8MaxNumberOfPresets, FC_SXM_AUDIO_PRESET_BAND_ONE);
                if(PRESET_BAND_INVALID_OBJECT == _hBandObjectOne) {
                    ETG_TRACE_ERR(("Failed to create preset band one object"));
                }
                /* Create preset band two */
                _hBandObjectTwo = BAND.hCreate(_hPresetObject, _u8MaxNumberOfPresets, FC_SXM_AUDIO_PRESET_BAND_TWO);
                if(PRESET_BAND_INVALID_OBJECT == _hBandObjectTwo) {
                    ETG_TRACE_ERR(("Failed to create preset band two object"));
                }
                /* Create preset band three */
                _hBandObjectThree = BAND.hCreate(_hPresetObject, _u8MaxNumberOfPresets, FC_SXM_AUDIO_PRESET_BAND_THREE);
                if(PRESET_BAND_INVALID_OBJECT == _hBandObjectThree) {
                    ETG_TRACE_ERR(("Failed to create preset band three object"));
                }

                }
            }

        else {
            ETG_TRACE_USR3(("Number of preset bands available is %d", uNumberOfBands));
        }
        /* Start Featured Favorite Service */
        vStartFeaturedFavorites();

        if(SMSAPI_RETURN_CODE_SUCCESS != PRESETS.eIterate(_hPresetObject, cb_vPresetIterateHandler, NULL)) {
            ETG_TRACE_ERR(("Failed to iterate presets in preset objects"));
        }
        /* register event callback for preset bands */
        PRESETS.eNotifyOnChange(_hPresetObject, cb_vPresetNotifyHandler, NULL);

        //Check if there is any ERROR in sms.cfg
        vCheckSMSCfgError();
    }
    vPrintPresetList();
}


tVoid fc_sxm_tclPresets::vCreateMixedPresetBand()
{
    /* Create preset band one */
    _hBandObjectOne = BAND.hCreate(_hPresetObject, _u8MaxNumberOfPresets, FC_SXM_AUDIO_PRESET_BAND_ONE);
    if(PRESET_BAND_INVALID_OBJECT == _hBandObjectOne) {
       ETG_TRACE_ERR(("Failed to create preset band one object"));
    }
    else {
        ETG_TRACE_USR3(("vBandCreation, Band creation Success! capacity is %d", BAND.tCapacity(_hBandObjectOne)));
    }
}




tBool fc_sxm_tclPresets::vCheckInvalidPreset()
{
    for(unsigned int index = 0; index < FC_SXM_BAND_THREE_MAX_INDEX; index++)
    {
        if(SERVICE_INVALID_ID == _u16PresetChannels[index])
        {
            ETG_TRACE_ERR(("fc_sxm_tclPresets::vCheckInvalidPreset(): One of the preset is having Invalid SID!"));
            return true;
        }
    }
    return false;
}

tVoid fc_sxm_tclPresets::vCheckSMSCfgError()
{
    tBool bCheckError = false;
    if(_bIsMixedPreset)
    {
        bCheckError = ((PRESET_BAND_INVALID_OBJECT == _hBandObjectOne) || vCheckInvalidPreset());
    }
    else
    {
        bCheckError = ((PRESET_BAND_INVALID_OBJECT == _hBandObjectOne)
                    || (PRESET_BAND_INVALID_OBJECT == _hBandObjectTwo)
                    || (PRESET_BAND_INVALID_OBJECT == _hBandObjectThree)
                    || vCheckInvalidPreset());
    }

    if(bCheckError == true)
    {
        ETG_TRACE_ERR(("fc_sxm_tclPresets::vCheckSMSCfgError(): One of the objects are not created! "));

        fc_sxm_tclAudioApp::instance()->vSetSMSCfgError(TRUE);
        fc_sxm_trMsgAudioSMSCfgError rMsg = fc_sxm_trMsgAudioSMSCfgError();
        fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
    }
}


tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgPresetBandUpdateCallback const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgPresetBandUpdateCallback) Mask=(%u)",
           prMsg->tEventMask));
       vUpdateFeaturedBandInfo(prMsg->hBand, prMsg->tEventMask, prMsg->sBandLongName, prMsg->tCapacity);
}


/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartRecallPreset)               *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to RecallPreset                                 *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Recall preset method start request                  *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartRecallPreset const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartRecallPreset)"));
   /* Method result data */
   midw_ext_sxm_audiofi_tclMsgRecallPresetMethodResult ofiTxObj;
   ofiTxObj.PresetElement = (prMsg->oFiMsg).PresetElement;
   ofiTxObj.Status        = FALSE;

   /* To map to relative preset index in appropriate preset band */
   tU8 lPresetElement = 0;

   /* local copy of preset index */
   tU8 lPresetIndex = (prMsg->oFiMsg).PresetElement;

   if(_bIsMixedPreset)
   {
       _u32CurrentBandNumber = FC_SXM_PRESET_BAND_ONE;
       lPresetElement = lPresetIndex;
   }
   else
   {
       /* Get relative preset index from actual preset index */
       if((FC_SXM_BAND_ONE_MIN_INDEX <= lPresetIndex) && (FC_SXM_BAND_ONE_MAX_INDEX >= lPresetIndex)) {
          _u32CurrentBandNumber = FC_SXM_PRESET_BAND_ONE;
          lPresetElement = lPresetIndex;
       } else if((FC_SXM_BAND_TWO_MIN_INDEX <= lPresetIndex) && (FC_SXM_BAND_TWO_MAX_INDEX >= lPresetIndex)) {
          _u32CurrentBandNumber = FC_SXM_PRESET_BAND_TWO;
          lPresetElement = (tU8)(lPresetIndex - _u8MaxCountOfPresetsInOneBand);
       } else if((FC_SXM_BAND_THREE_MIN_INDEX <= lPresetIndex) && (FC_SXM_BAND_THREE_MAX_INDEX >= lPresetIndex)) {
          _u32CurrentBandNumber = FC_SXM_PRESET_BAND_THREE;
          lPresetElement = (tU8) (lPresetIndex - ( 2 * _u8MaxCountOfPresetsInOneBand));
       } else {
          /* Do nothing */
       }
   }

   ETG_TRACE_USR2(("Current band number is %d", _u32CurrentBandNumber));
   ETG_TRACE_USR2(("Preset element is %d", lPresetElement));
   /* Tune to selected preset if index is in proper range */
   if((FC_SXM_AUDIO_MIN_PRESET_COUNT <= lPresetElement) && (_u8MaxCountOfPresetsInOneBand >= lPresetElement)) {
      SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;
      /* To notify current tuned channel */
      tS8 lRelativeOffset = -1;
      /* To hold current band object */
      PRESET_BAND_OBJECT hBandObject = PRESET_BAND_INVALID_OBJECT;

      /* Get current preset band */
      switch(_u32CurrentBandNumber) {
         case FC_SXM_PRESET_BAND_ONE: { /* To update preset array and band object */
            lRelativeOffset = FC_SXM_BAND_ONE_OFFSET; 
            hBandObject = _hBandObjectOne;
         } break;
         case FC_SXM_PRESET_BAND_TWO: { /* To update preset array and band object */
            lRelativeOffset = FC_SXM_BAND_TWO_OFFSET; 
            hBandObject = _hBandObjectTwo;
         } break;
         case FC_SXM_PRESET_BAND_THREE: { /* To update preset array and band object */
            lRelativeOffset = FC_SXM_BAND_THREE_OFFSET;
            hBandObject = _hBandObjectThree;
         } break;
         default:
           break;
      }

      Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                  _poAudioApp->poGetChannelListInstance(): NULL;
      SXM_ASSERT_RETURN(NULL != poChannelList)

      /* Check if channel is valid, available and subscribed */
      if(PRESET_BAND_INVALID_OBJECT != hBandObject) {
         tTunable tDisplayAdvisory = e_UNAVAILABLE;
         ofiTxObj.PresetListElement = 
                 poChannelList->enGetPresetChannelInfo(_u16PresetChannels[lRelativeOffset +lPresetElement - 1], FALSE, tDisplayAdvisory);
         //if(FALSE == ((ofiTxObj).PresetListElement).ChannelActive)
         if (tDisplayAdvisory != e_TUNABLE) {
            /* Stop advisory timer if any */
            fc_sxm_tclAdvisories::instance()->vStopAndClearChannelAdvisory();
            /* Stop channel select dwell timer if any */
            poChannelList->vStopDwellTimer();

            /* If channel is unavailable send system message */
            switch(tDisplayAdvisory) {
               case e_UNAVAILABLE: {
                   poChannelList->_u16RequestedUnavailableChannel = ofiTxObj.PresetListElement.ChannelID;
                  /* Channel unavailable send system message */
                  (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, FALSE);
                  poChannelList->vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
               } break;
               case e_UNAUTHORIZED: {
                  /* Channel un authorized send system message */
                   poChannelList->_u16RequestedUnavailableChannel = ofiTxObj.PresetListElement.ChannelID;
                  /* Channel unavailable send system message */
                  (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, FALSE);
                  poChannelList->vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
               } break;
               case e_LOCKED: {
                    poChannelList->_u16RequestedUnavailableChannel = ofiTxObj.PresetListElement.ChannelID;             
                   // Channel is locked.... send system msg
                   (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_MATURE, FALSE);
                   poChannelList->vNotifyChannelInformationUpdate(e_MATURE_LOCKED_CHANNEL);
               }  break;
               case e_SKIPPED: {
                    poChannelList->_u16RequestedUnavailableChannel = ofiTxObj.PresetListElement.ChannelID;
                   // Channel is skipped.... send system msg
                   (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_LOCKED, FALSE);
                   poChannelList->vNotifyChannelInformationUpdate(e_SKIPPED_CHANNEL);
               } break;

               default:
                 break;
            }
         } else {
//            /* Get current tuned channel from channel list class */
//            SERVICE_ID uCurrentTunedSID = poChannelList->u16GetSIDOfTunedChannel();
//            if(uCurrentTunedSID == _u16PresetChannels[lRelativeOffset + lPresetElement - 1]) {
//               eReturnCode = SMSAPI_RETURN_CODE_SUCCESS;
//            } else {
                /* Setting the bIsScanInProgress flag to FALSE state. Stopping Tune Scan is
                handled internally by SMS.*/
                vSetTuneScanStatus(FALSE);
               eReturnCode = enBandTune(hBandObject, (tU8)(lPresetElement - 1), TRUE);
//            }
            /* Tune to selected preset from current band */
            if(SMSAPI_RETURN_CODE_SUCCESS != eReturnCode) {
               ETG_TRACE_ERR(("Failed to tune selected preset, error code - %d", eReturnCode));
               /* If not subscribed then send system message */
               (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, TRUE);
            } else {
               ETG_TRACE_USR3(("Successfully tuned to selected preset"));
               /* Update method result accordingly */
               ofiTxObj.Status = TRUE;
               _u8PreviousTunerElemNr = ofiTxObj.PresetElement;

               if(0 <= lRelativeOffset) {
                  tDisplayAdvisory = e_UNAVAILABLE;
                  ofiTxObj.PresetListElement = 
                          poChannelList->enGetPresetChannelInfo(_u16PresetChannels[lRelativeOffset + lPresetElement - 1], TRUE, tDisplayAdvisory);
               }
               /* Stop advisory timer if any */
               fc_sxm_tclAdvisories::instance()->vStopAndClearChannelAdvisory();
               /* Stop channel select dwell timer if any */
               poChannelList->vStopDwellTimer();
               /* Notify channel info */
               poChannelList->vNotifyChannelInformationUpdate(e_RECALL_PRESET);
               vUpdateIRMode(TRUE, FALSE); /* Set IR Mode when SF channel is tuned */
               fc_sxm_tcl_trAudioPropertyAutoCompare AutoCompareStatus;
               ((AutoCompareStatus).oFiRes).TunerElemNr = (prMsg->oFiMsg).PresetElement;
               fc_sxm_tclAudioProperties::instance()->oAutoCompare.vSet(AutoCompareStatus);
               fc_sxm_tclAudioProperties::instance()->oAutoCompare.vNotify();
               _u16StoredServiceid = poChannelList->u16GetSIDOfTunedChannel();
            }
         }
      }
   } else {
      ETG_TRACE_ERR(("Invalid preset value received"));
   }

   /* Post the Method Result */
   if(AIL_EN_N_NO_ERROR ==  fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing,  ofiTxObj)) {
      ETG_TRACE_USR3(("Recall preset method result sent successfully"));
   } else {
      ETG_TRACE_ERR(("Failed to send method result for recall preset"));
   }
}

// Evaluate Preset channel change impact on Smart Favorites List
tVoid fc_sxm_tclPresets::vUpdateSmartFavorites(tU8 u8PresetIndex)
{
    DECODER_OBJECT hDecoder = (_poAudioApp) ? _poAudioApp->oGetDecoderObject(): DECODER_INVALID_OBJECT;
    Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                      _poAudioApp->poGetChannelListInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poChannelList)

    // 1. Validate preset index

    tU8 u8NumberOfPresets = FC_SXM_BAND_THREE_MAX_INDEX;
    if(_bIsMixedPreset)
    {
        u8NumberOfPresets = _u8NumberOfPresets;
    }

    if((FC_SXM_BAND_ONE_MIN_INDEX <= u8PresetIndex+1) &&  (u8NumberOfPresets >= u8PresetIndex+1)) {
        // 2. Get SID of selected preset channel.... This SID has to be removed from SF list if the preset channels are not duped
        SERVICE_ID hRemoveSID = _u16PresetChannels[u8PresetIndex];
        // 3. Get SID of tuned channel. This channel would be added to SF list if it's not part of SF list already
        SERVICE_ID hAddSID = poChannelList->u16GetSIDOfTunedChannel();
        ETG_TRACE_USR4(("fc_sxm_tclPresets::vUpdateSmartFavorites.. u8PresetIndex = %u  hRemoveSID = %u  hAddSID = %u",
                u8PresetIndex, hRemoveSID, hAddSID));

        // Significance of the variables,
        // 1. u8PresetsRemoveCount - The number(count) of entries of channel to be removed in presets list
        // 2. u8SFAddCount - The number(count) of entries of channel to be added in SF list
        // 3. u8SFRemoveCount - The number(count) of entries of channel to be removed from SF list
        tU8 u8PresetsRemoveCount = 0U;
        tU8 u8SFAddCount = 0U;
        tU8 u8SFRemoveCount = 0U;

        for(tU8 u8Index = 0U; u8Index < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; ++u8Index) {
            if (hRemoveSID == _u16SFChannels[u8Index]) {
                ++u8SFRemoveCount;
            }
            if (hRemoveSID == _u16PresetChannels[u8Index]) {
                ++u8PresetsRemoveCount;
            }
            if (hAddSID == _u16SFChannels[u8Index]) {
                ++u8SFAddCount;
            }
        } // end of for loop

        ETG_TRACE_USR4(("u8PresetsRemoveCount = %u  u8SFAddCount = %u  u8SFRemoveCount = %u",
                u8PresetsRemoveCount, u8SFAddCount, u8SFRemoveCount));
        // Get impact of storepreset channel on SF list
        enPresetChange_Action enAction = fc_sxm_tclAudioUtils::enGetPresetChangeAction(u8PresetsRemoveCount, u8SFAddCount, u8SFRemoveCount);
        ETG_TRACE_USR2(("Impact = %u", ETG_CENUM(enPresetChange_Action, enAction) ));
        tBool bAddSFChannel = ((enAction_AddChn == enAction) || (enAction_ReplaceChn == enAction)) ? TRUE : FALSE;

        SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
        CATEGORY_ID eSmartFavCatId = SMART_FAVORITES.tCategoryId(hDecoder);

        if ((enAction_ReplaceChn == enAction) || ( enAction_DeleteChn == enAction)) {
            enReturnCode = CATEGORY.eIterate(hDecoder, eSmartFavCatId, cb_s16SFListIterateHandler, (void*)&hRemoveSID);
            if(SMSAPI_RETURN_CODE_SUCCESS != enReturnCode) {
                ETG_TRACE_ERR(("Failed to register for channel callback for SF category %d, ERROR_CODE - %d",
                                           eSmartFavCatId, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
            }
        }
        if (TRUE == bAddSFChannel) {
            // From SID get CID
            CHANNEL_ID hChannelId = DECODER.tGetChannelId(hDecoder, hAddSID);
            // Insert selected index preset channel into smart favorite category
            if(SMSAPI_RETURN_CODE_ERROR ==  CATEGORY.eInsertNewChannel(hDecoder, eSmartFavCatId, hChannelId)) {
                ETG_TRACE_ERR(("CATEGORY.eInsertNewChannel() failed ChannelID = %u ", hChannelId));
            }
        }
        // Update smart favorite list
        vUpdateSFList();
    }
}

tVoid fc_sxm_tclPresets::vCheckAndRemoveSFPreset(tU8 u8PresetIndex)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vCheckAndRemoveSFPreset(%d)", u8PresetIndex));
   DECODER_OBJECT hDecoder = (_poAudioApp) ? _poAudioApp->oGetDecoderObject(): DECODER_INVALID_OBJECT;

   /* 1. Validate preset index */
   if((FC_SXM_BAND_ONE_MIN_INDEX <= u8PresetIndex+1) &&  (FC_SXM_BAND_THREE_MAX_INDEX >= u8PresetIndex+1)) {   
      /* 2. Get SID of selected preset channel */
      SERVICE_ID SIDAtPresetIndex = _u16PresetChannels[u8PresetIndex];
      tU32 u32LoopCount = 0;                                                                             
      tBool bIsRepeated = FALSE;
      /* 3. Check if selected preset is also a smart favorite channel */
      for(; u32LoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u32LoopCount++) {
         if(SIDAtPresetIndex == _u16SFChannels[u32LoopCount]) {
            bIsRepeated = TRUE;
            break;
         }
      }
      /* 4. Check if the same channel is stored in any other preset */
      if(bIsRepeated) {
         for(u32LoopCount=0; u32LoopCount<(_u8MaxNumberOfPresets * _u8MaxNumberOfBands); u32LoopCount++) {
            if((SIDAtPresetIndex == _u16PresetChannels[u32LoopCount]) && (u8PresetIndex != u32LoopCount)) {
               bIsRepeated = FALSE;
               break;
            }
         }
      }
      /* 5. If it is the only entry in preset set then Get Smart Favorite Category Id */
      if(bIsRepeated) {
         CATEGORY_ID eSmartFavCatId = SMART_FAVORITES.tCategoryId(hDecoder);
         /* 6. Iterate through smart favorite channel, pass service id to be removed to the callback function */
         SMSAPI_RETURN_CODE_ENUM hReturnCode = CATEGORY.eIterate(hDecoder,
                                                                 eSmartFavCatId, cb_s16SFListIterateHandler, (void*)&SIDAtPresetIndex);
         if(SMSAPI_RETURN_CODE_SUCCESS != hReturnCode) {
            ETG_TRACE_ERR(("Failed to register for channel callback for SF category %d, ERROR_CODE - %d", 
                           eSmartFavCatId, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, hReturnCode)));
         }  /* 7. Inside callback function delete selected channel */
         /* 8. Update smart favorite list */
         // vUpdateSFList(); 
         /* 9. Now set newly stored channel as a smart favorite channel only if it is not a preview channel */

         /* Get service ID of current tuned channel from  channel list class to store in preset list */
         Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                          _poAudioApp->poGetChannelListInstance(): NULL;
         SXM_ASSERT_RETURN(NULL != poChannelList)

         SERVICE_ID hServiceID = poChannelList->u16GetSIDOfTunedChannel();
         if(poChannelList->u16GetSIDOfPreviewChannel() != hServiceID) {
            /* From SID get CID */
            CHANNEL_ID hChannelId = DECODER.tGetChannelId(hDecoder, hServiceID);
            /* Insert selected index preset channel into smart favorite category */
            if(SMSAPI_RETURN_CODE_ERROR ==  CATEGORY.eInsertNewChannel(hDecoder, eSmartFavCatId, hChannelId)) {
               ETG_TRACE_ERR(("CATEGORY.eInsertNewChannel() failed"));
            }
         }
         /* Update smart favorite list */
         vUpdateSFList();
         /* Update preset smart favorites */
         // vUpdatePresetSFList();
      }
   }
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartStorePreset)                *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to store preset                                 *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Store preset method start request                   *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartStorePreset const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartStorePreset)"));

   /* Method result data */
   midw_ext_sxm_audiofi_tclMsgStorePresetMethodResult ofiTxObj;
   ofiTxObj.PresetElement = (prMsg->oFiMsg).PresetElement;
   ofiTxObj.Status = FALSE;

   /* To map to relative preset index in appropriate preset band */
   tU8 lPresetElement = 0;
   /* local copy of preset index */
   tU8 lPresetIndex = (prMsg->oFiMsg).PresetElement;

   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                  _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN(NULL != poChannelList)
   /* Get current tuned channel from channel list class */
     CHANNEL_ID uCurrentTunedChannel = poChannelList->u16GetCurrentTunedChannel();
   if(_bIsMixedPreset)
   {
       _u32CurrentBandNumber = FC_SXM_PRESET_BAND_ONE;
       lPresetElement = lPresetIndex;
   }
   else
   {
       /* Get relative preset index from actual preset index */
       if((FC_SXM_BAND_ONE_MIN_INDEX <= lPresetIndex) && (FC_SXM_BAND_ONE_MAX_INDEX >= lPresetIndex)) {
          _u32CurrentBandNumber = FC_SXM_PRESET_BAND_ONE;
          lPresetElement = lPresetIndex;
       } else if((FC_SXM_BAND_TWO_MIN_INDEX <= lPresetIndex) && (FC_SXM_BAND_TWO_MAX_INDEX >= lPresetIndex)) {
          _u32CurrentBandNumber = FC_SXM_PRESET_BAND_TWO;
          lPresetElement = (tU8)(lPresetIndex - _u8MaxCountOfPresetsInOneBand);
       } else if((FC_SXM_BAND_THREE_MIN_INDEX <= lPresetIndex) && (FC_SXM_BAND_THREE_MAX_INDEX >= lPresetIndex)) {
          _u32CurrentBandNumber = FC_SXM_PRESET_BAND_THREE;
          lPresetElement = (tU8)(lPresetIndex - (2 * _u8MaxCountOfPresetsInOneBand));
       }
       else {
          /* Do nothing */
       }
   }

   ETG_TRACE_USR2(("Current band number is %u ... Preset element is %u", _u32CurrentBandNumber, lPresetElement));
   SMSAPI_RETURN_CODE_ENUM eReturnCode = SMSAPI_RETURN_CODE_ERROR;

   /* Store current tuned channel to selected preset if index is in proper range */
   if((FC_SXM_AUDIO_MIN_PRESET_COUNT <= lPresetElement) && (_u8MaxNumberOfPresets >= lPresetElement)
           && (0u != uCurrentTunedChannel))
   {
      /* Update the same in preset array */
      tS8 lRelativeOffset = -1;

      /* Get current preset band */
      switch(_u32CurrentBandNumber) {
         case FC_SXM_PRESET_BAND_ONE: {
            eReturnCode =  enSetPreset(_hBandObjectOne, (tU8)(lPresetElement - 1), uCurrentTunedChannel);
            /* To update preset array */
            lRelativeOffset = FC_SXM_BAND_ONE_OFFSET; 
         } break;
         case FC_SXM_PRESET_BAND_TWO: {
            eReturnCode = enSetPreset(_hBandObjectTwo, (tU8)(lPresetElement - 1), uCurrentTunedChannel);
            /* To update preset array */
            lRelativeOffset = FC_SXM_BAND_TWO_OFFSET; 
         } break;
         case FC_SXM_PRESET_BAND_THREE: {
            eReturnCode = enSetPreset(_hBandObjectThree, (tU8)(lPresetElement - 1), uCurrentTunedChannel);
            /* To update preset array */
            lRelativeOffset = FC_SXM_BAND_THREE_OFFSET; 
         } break;
         default:
           break;
      }

      /* Store current tuned channel as preset in current band */
      if(SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) {
         ETG_TRACE_USR3(("Successfully stored tuned channel as preset"));
         _u8PreviousTunerElemNr = ofiTxObj.PresetElement;

         /* Get service ID of current tuned channel from  channel list class to store in preset list */
         SERVICE_ID hServiceID = poChannelList->u16GetSIDOfTunedChannel();

         /* Before storing a new channel remove it from sf list if it was a sf channel */
         // vCheckAndRemoveSFPreset(((prMsg->oFiMsg).PresetElement) - 1);

         if (_u32CurrentBandNumber != FC_SXM_PRESET_BAND_THREE)
         {
             vUpdateSmartFavorites((tU8)(((prMsg->oFiMsg).PresetElement) - 1));
         }

         /* Update method result accordingly */
         ofiTxObj.Status = TRUE;
         tTunable tDisplayAdvisory = e_UNAVAILABLE;
         ofiTxObj.PresetListElement = 
                 poChannelList->enGetPresetChannelInfo(hServiceID, FALSE, tDisplayAdvisory);
        for(tU8 u8SFLoopCount=0; u8SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u8SFLoopCount++) {
            if (ofiTxObj.PresetListElement.ServiceID == _u16SFChannels[u8SFLoopCount]) {
                ofiTxObj.PresetListElement.IsSmartFavorite = TRUE;
                break;
            }
        }
         /* Save the tuned channel info in preset array */
         if((0 <= lRelativeOffset) && (hServiceID != SERVICE_INVALID_ID)) {
            _u16PresetChannels[lRelativeOffset + lPresetElement - 1] = hServiceID;
         }

         /* Update profile preset list in datapool for change profile request from profile manager
          * Ignoring only for Guest user(profile ID = 0)
          */

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
         if((_poAudioApp != NULL) && (_poAudioApp->u8GetActiveProfileId() != 0)) {
            fc_sxm_tclPersData::instance()->vSetPresetData(_u16PresetChannels);
         }

#endif
         if((u8GetUserProfileID() != 255) && (u8GetUserProfileID()))
         {

             fc_sxm_tclPersData::instance()->vSetPresetData(_u16PresetChannels);
         }

         /* Update preset smart favorites */
         vUpdatePresetSFList();

         /* Get index and range based on current preset band */
         tU8 u8LoopCount = (tU8)((_u32CurrentBandNumber - 1) * _u8MaxCountOfPresetsInOneBand);
         tU8 u8LoopRange = (tU8)(u8LoopCount + _u8MaxCountOfPresetsInOneBand);
         /* Within given band search for smart favorite presets */
         for(; u8LoopCount < u8LoopRange; u8LoopCount++) {
            /* Create SF List element */
            midw_ext_fi_tcl_SmartFavoritePreset tSFListElement;
            tSFListElement.PresetIndex = (tU8)(u8LoopCount+1);
            tSFListElement.IsSmartFavorite = FALSE;
            /* Search if preset is smart favorite preset */
            for(tU8 u8SFLoopCount=0; u8SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u8SFLoopCount++) {
               if(_u16PresetChannels[u8LoopCount] == _u16SFChannels[u8SFLoopCount]) {
                  tSFListElement.IsSmartFavorite = TRUE;
                  ETG_TRACE_USR4(("AUTO SF Store Preset- %d", tSFListElement.PresetIndex));
                  break;
               }
            }
            /* Insert list element in store preset method result */
            ofiTxObj.SmartFavoriteList.push_back(tSFListElement);
         }
      }
   }
   // Post method result
   if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) {
       fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj);
       ETG_TRACE_USR4(("IsSmartFavorite- %d", ofiTxObj.PresetListElement.IsSmartFavorite));
   }
   else {
       ETG_TRACE_ERR(("Invalid preset value received uCurrentTunedChannel = %u \t lPresetElement = %u \t eReturnCode = %u",
                  uCurrentTunedChannel, lPresetElement, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM,eReturnCode)));
       fc_sxm_tclAudioService::instance()->vSendError(prMsg->rAdressing,
               (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INVALID_PARAMETER);
   }
}


//Method to check if decoder state is ready then process the change profile request
//else restore the preset list and sent back the ACK as success(CHANGED_PROFILE) to Tuner Master
tBool fc_sxm_tclPresets::bHandleChangeProfilePresetList()
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::bHandleChangeProfilePresetList entered"));
    tBool bResult = FALSE;


    //initializing preset list
    SERVICE_ID aPresetList[_u8MaxNumberOfPresets * _u8MaxNumberOfBands] = {0,};

    //get preset list from dp
    fc_sxm_tclPersData::instance()->vGetPresetData(aPresetList);

    // Check for Decoder Ready state
    DECODER_OBJECT hDecoder = hGetDecoder();
    if(hDecoder == DECODER_INVALID_OBJECT) {
        ETG_TRACE_USR4(("fc_sxm_tclPresets::bHandleChangeProfilePresetList Decoder Object is InValid!"));

        //decoder is not ready, keep back-up of preset list in a buffer, buffer is reset before use
        memset(_u16ProfilePresetChannels, 0x01, (_u8MaxNumberOfPresets * _u8MaxNumberOfBands)*sizeof(SERVICE_ID));
        memcpy(_u16ProfilePresetChannels, aPresetList, (_u8MaxNumberOfPresets * _u8MaxNumberOfBands)*sizeof(SERVICE_ID));

        return TRUE;
    }
    else {
        bResult = bSetProfilePresetList(aPresetList);
        return bResult;
    }
}

//Method to set preset list and Smart Favorite once decoder is ready
//during Change Profile request from profile manager and sent back the result
tBool fc_sxm_tclPresets::bSetProfilePresetList(SERVICE_ID *pPresetList)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::bSetProfilePresetList entered"));


    //1. RESET presets ARRAY to default preview channel
    vResetPresetList();

    //2. RESET SF ARRAY to SERVICE_INVALID_ID
    vResetSFList();

    //3. Set Preset & SF list for profile change
    //Get SF category ID
    DECODER_OBJECT hDecoder = hGetDecoder();
    CATEGORY_ID smartFavID = SMART_FAVORITES.tCategoryId(hDecoder);
    set<SERVICE_ID> setSFChannels;

    SMSAPI_RETURN_CODE_ENUM eReturnCode = CATEGORY.eIterate(hDecoder, smartFavID, cb_s16RemoveAllCategoryChannels, NULL);

    if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode )
        ETG_TRACE_ERR(("Error removing channels from Smart favorites errorcode = %u",eReturnCode));

    tU8 lPresetElement = 0;
    tBool bResult = FALSE;

    for(tU8 lPresetIndex = 0, u8SFIndex = 0; lPresetIndex < (_u8MaxNumberOfPresets * _u8MaxNumberOfBands); lPresetIndex ++)
    {
        CHANNEL_ID uChannelID = DECODER.tGetChannelId(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), pPresetList[lPresetIndex]);

                ETG_TRACE_USR2(("PresetSID: %d ChannelID: %d", pPresetList[lPresetIndex], uChannelID));

        /* If Channel is INVALID(32767) then set to preview default channel */
        if(uChannelID == CHANNEL_INVALID_ID)
        {
            uChannelID = FC_SXM_PREVIEW_CHANNEL_ID;
            pPresetList[lPresetIndex] = FC_SXM_PREVIEW_CHANNEL_ID;
            fc_sxm_tclPersData::instance()->vSetPresetData(pPresetList);
        }
		
        tBool bCheckRadioIDChannel = false;
        if(_bIsMixedPreset)
        {
            lPresetElement = lPresetIndex;
            bCheckRadioIDChannel = true;
        }
        else
        {
            /* Get relative preset index from actual preset index */
            lPresetElement = u8GetRelativePresetIndex(lPresetIndex);
            bCheckRadioIDChannel = (0u != uChannelID);
        }

        ETG_TRACE_USR2(("Current band number is %u ... Preset element is %u SID: %d ChannelID: %d", _u32CurrentBandNumber, lPresetElement, pPresetList[lPresetIndex], uChannelID));

        /* Store current tuned channel to selected preset if index is in proper range */
        if((_u8MaxNumberOfPresets - 1 >= lPresetElement)
                && (bCheckRadioIDChannel == true))
        {
            /* Get current preset band */
            PRESET_BAND_OBJECT hBandObject = PRESET_BAND_INVALID_OBJECT;
            hBandObject = hGetPresetBand(_u32CurrentBandNumber);
            eReturnCode =  enSetPreset(hBandObject, lPresetElement, uChannelID);


            /* Store current tuned channel as preset in current band */
            if(SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) {
                ETG_TRACE_USR3(("Successfully stored tuned channel as preset"));

                /* Get service ID of current tuned channel from  channel list class to store in preset list */
                SERVICE_ID hServiceID = pPresetList[lPresetIndex];

                tS8 lRelativeOffset = 0;
                tBool bCheckSmartFavorite = false;

                if(_bIsMixedPreset)
                {
                    bCheckSmartFavorite = (setSFChannels.size() < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT);
                }
                else
                {
                    lRelativeOffset = (tS8)((_u32CurrentBandNumber - 1)*_u8MaxCountOfPresetsInOneBand);
                    bCheckSmartFavorite = (_u32CurrentBandNumber != FC_SXM_PRESET_BAND_THREE);
                }


                //4. Set SF through SMS API & Updates ARRAY for SF
                // Only 12 channels are allowed to be Smart Favorites and the smart favorite channels need not be repeatedly inserted into the array.
                if ((bCheckSmartFavorite == true) && (setSFChannels.count(hServiceID) == 0) && (hServiceID != 0))
                {
                    setSFChannels.insert(hServiceID);
                    eReturnCode = CATEGORY.eInsertNewChannel(hDecoder, smartFavID, uChannelID);
                    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) {
                        ETG_TRACE_ERR(("Failed to insert new channel into SF smartFavID: %d eReturnCode: %d",smartFavID, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode)));
                        break; //sms API failed
                    }
                    _u16SFChannels[u8SFIndex] = hServiceID;
                    ETG_TRACE_USR4(("fc_sxm_tclPresets::bSetProfilePresetList, _u16SFChannels[%d]:%d",u8SFIndex,_u16SFChannels[u8SFIndex]));
                    ++u8SFIndex;
                }
                //5. Updates ARRAY for Presets
                /* Save the tuned channel info in preset array */
                if((0 <= lRelativeOffset) && (hServiceID != SERVICE_INVALID_ID)) {
                    _u16PresetChannels[lRelativeOffset + lPresetElement] = hServiceID;
                    ETG_TRACE_USR4(("fc_sxm_tclPresets::_u16PresetChannels[%d]:%d ",lRelativeOffset + lPresetElement,
                            _u16PresetChannels[lRelativeOffset + lPresetElement]));
                }

                //everything is fine, notify profile manager status as success
                bResult = TRUE;
            } else {
                ETG_TRACE_ERR(("Failed to store tune channel as preset: %d",eReturnCode));
                break;
            }
        }
    }

    if(bResult == TRUE)
    {
        //6. Notify client
        vPresetListChangeNotification();
        Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                _poAudioApp->poGetChannelListInstance(): NULL;
        if(poChannelList != NULL)
        {
            SERVICE_ID hServiceID = poChannelList->u16GetSIDOfTunedChannel();
            vNotifyTunedChannelPresetIndex(hServiceID);
        }
    }

    return bResult;
}


tBool fc_sxm_tclPresets::bValidatePresetIndices(tU8 u8FirstElement, tU8 u8LastElement)
{
    tBool bResult = false;
    if(_bIsMixedPreset)
    {
        if( (FC_SXM_AUDIO_MIN_PRESET_COUNT <= (u8FirstElement)) && (_u8MaxNumberOfPresets >= (u8LastElement)) ){
           _u32CurrentBandNumber = FC_SXM_PRESET_BAND_ONE;
            bResult = true;
        }
    }
    else
    {
        if( (FC_SXM_AUDIO_MIN_PRESET_COUNT <= u8FirstElement) &&
                ((_u8MaxNumberOfPresets * _u8MaxNumberOfBands) >= u8LastElement) &&
                (u8FirstElement < u8LastElement) &&
                (_u8MaxCountOfPresetsInOneBand - 1) == (u8LastElement - u8FirstElement) &&
                (1 == (u8FirstElement % _u8MaxCountOfPresetsInOneBand) ) )
        {
            _u32CurrentBandNumber = ((u8FirstElement / _u8MaxCountOfPresetsInOneBand) + 1);
            bResult = true;
        }
    }
    return bResult;
}


/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartSetAutoCompare)             *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to set auto compare parameter                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Set auto compare parameters method start request    *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartSetAutoCompare const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartSetAutoCompare)"));
   /* Method result data */
   midw_ext_sxm_audiofi_tclMsgSetAutoCompareParametersMethodResult ofiTxObj;
   ofiTxObj.Status = FALSE;

       fc_sxm_tcl_trAudioPropertyAutoCompare AutoCompareStatus;
       ((AutoCompareStatus).oFiRes).CurrentTunerElemNr = 0;

   /* Set current band info based on range provided */
     if(bValidatePresetIndices((prMsg->oFiMsg).u8FirstElement, (prMsg->oFiMsg).u8LastElement)) {

      /* Get current tuned channel from channel list class */
      Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                          _poAudioApp->poGetChannelListInstance(): NULL;
      SXM_ASSERT_RETURN(NULL != poChannelList)
      SERVICE_ID uCurrentTunedSID = poChannelList->u16GetSIDOfTunedChannel();

      /* Within the given index, find tuned channel ID */
      for(tU8 u8LoopCount = (prMsg->oFiMsg).u8FirstElement; u8LoopCount <= (prMsg->oFiMsg).u8LastElement; u8LoopCount++) {
         /* If current tuned channel is one of the preset element, get its array index */
         if(uCurrentTunedSID == _u16PresetChannels[u8LoopCount - 1]) {
            ((AutoCompareStatus).oFiRes).CurrentTunerElemNr = u8LoopCount;
            break;
         }
      }

      /* Within given band search for smart favorite presets */
      for(tU8 u8LoopCount = (prMsg->oFiMsg).u8FirstElement; u8LoopCount <= (prMsg->oFiMsg).u8LastElement; u8LoopCount++) {
         /* Create SF List element */
         midw_ext_fi_tcl_SmartFavoritePreset tSFListElement;
         tSFListElement.PresetIndex = u8LoopCount;
         tSFListElement.IsSmartFavorite = FALSE;
         /* Search if preset is smart favorite preset */
         for(tU8 u8SFLoopCount=0; u8SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u8SFLoopCount++) {
            if(_u16PresetChannels[u8LoopCount-1] == _u16SFChannels[u8SFLoopCount]) {
               tSFListElement.IsSmartFavorite = TRUE;
               ETG_TRACE_USR4(("Auto smart favorite preset is  %d", u8LoopCount));
               break;
            }
         }
         /* Insert list element in auto compare property */
         (((AutoCompareStatus).oFiRes).SmartFavoriteList).push_back(tSFListElement);
      }

      /* Set the status info and notify all clients */
      _u8PreviousTunerElemNr = ((AutoCompareStatus).oFiRes).CurrentTunerElemNr;
      fc_sxm_tclAudioProperties::instance()->oAutoCompare.vSet(AutoCompareStatus);
      fc_sxm_tclAudioProperties::instance()->oAutoCompare.vNotify();

      /* Update method result accordingly */
      ofiTxObj.Status = TRUE;
   } else {
      ETG_TRACE_ERR(("Invalid input parameters received"));
   }

   /* Post the Method Result */
   if(AIL_EN_N_NO_ERROR == fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj)) {
      ETG_TRACE_USR3(("Set auto compare parameters method result sent successfully"));
   } else {
      ETG_TRACE_ERR(("Failed to send method result for set auto compare parameters"));
   }
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartTuneStart)                  *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to enable or disable tune start                 *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Tune start method start request                     *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartTuneStart const *prMsg)
{
   midw_ext_sxm_audiofi_tclMsgEnableTuneStartMethodResult ofiTxObj;
   ofiTxObj.bEnableTuneStart = (prMsg->oFiMsg).bEnableTuneStart;

   _ePlayPoint = (prMsg->oFiMsg).bEnableTuneStart ?
            SMART_FAVORITES_PLAY_POINT_CTRL_AUTOMATIC :  //Fix for the NCG3D-42854
            SMART_FAVORITES_PLAY_POINT_CTRL_LIVE;

   SMSAPI_RETURN_CODE_ENUM enReturnCode = SMART_FAVORITES.eEnable(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), _ePlayPoint);

   ofiTxObj.Status.enType = (SMSAPI_RETURN_CODE_SUCCESS == enReturnCode) ?
            midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS :
            midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;

   ETG_TRACE_USR4(("vProcess(fc_sxm_trMsgAudioMStartTuneStart) SMART_FAVORITES.eEnable, ePlayPoint=%d enReturnCode=%d",
            ETG_CENUM(SMART_FAVORITES_PLAY_POINT_CTRL_ENUM, _ePlayPoint),
            ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));

   if(AIL_EN_N_NO_ERROR == fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj)) {
      ETG_TRACE_USR3(("Tune start method result sent successfully"));
   } else {
      ETG_TRACE_ERR(("Failed to send method result for tune start method"));
   }
}

/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartSFListControl)              *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to enable or disable tune start                 *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Tune start method start request                     *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartSFListControl const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartSFListControl)"));


   /* Method result data */
   midw_ext_sxm_audiofi_tclMsgSmartFavoriteListControlMethodResult ofiTxObj;
   ((ofiTxObj).Operation).enType = ((prMsg->oFiMsg).Operation).enType;
   ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;

   switch(((prMsg->oFiMsg).Operation).enType) {
      case midw_ext_fi_tcl_e8_SFOperation::FI_EN_SXM_AUDIO_GET_SF_LIST: {
         ETG_TRACE_USR4(("Get SF List"));
         /* Get smart favorite channel list */
         GetSFChannelList((ofiTxObj).SmartFavoriteList, (ofiTxObj).EnableAddPreset);
         ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
#if(FC_SXM_AUDIO_TO_DEBUG)
         vector<midw_ext_fi_tcl_SFListElement>::iterator mIter;
         for(mIter=((ofiTxObj).SmartFavoriteList).begin(); mIter!=((ofiTxObj).SmartFavoriteList).end(); ++mIter) {
            ETG_TRACE_USR3(("vector element - %d", (*mIter).PresetIndex));
            ETG_TRACE_USR3(("Is Repeated ---- %d", (*mIter).IsRepeated));
         }
         ETG_TRACE_USR3(("EnableAddPreset - %d", (ofiTxObj).EnableAddPreset));
#endif
      } break;
      case midw_ext_fi_tcl_e8_SFOperation::FI_EN_SXM_AUDIO_GET_NON_SF_LIST: {
         ETG_TRACE_USR4(("Get Non SF List"));
         /* Get non smart favorite channel list */
         GetNonSFChannelList((ofiTxObj).SmartFavoriteList);
         (ofiTxObj).EnableAddPreset = FALSE;
         ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
#if(FC_SXM_AUDIO_TO_DEBUG)
         vector<midw_ext_fi_tcl_SFListElement>::iterator mIter;
         for(mIter=((ofiTxObj).SmartFavoriteList).begin(); mIter!=((ofiTxObj).SmartFavoriteList).end(); ++mIter) {
            ETG_TRACE_USR3(("vector element - %d", (*mIter).PresetIndex));
            ETG_TRACE_USR3(("Is Repeated ---- %d", (*mIter).IsRepeated));
         }
         ETG_TRACE_USR3(("EnableAddPreset - %d", (ofiTxObj).EnableAddPreset));
#endif
      } break;
      case midw_ext_fi_tcl_e8_SFOperation::FI_EN_SXM_AUDIO_ADD_TO_SF_LIST: {
         ETG_TRACE_USR4(("Add element into SF List from index %d", (prMsg->oFiMsg).ListIndex));

         if( (FC_SXM_AUDIO_MIN_PRESET_COUNT <= (prMsg->oFiMsg).ListIndex) &&
             ((_u8MaxNumberOfPresets * _u8MaxNumberOfBands) >= (prMsg->oFiMsg).ListIndex)) {

            /* 1. Get non smart favorite list that is being displayed to the user */
            GetNonSFChannelList((ofiTxObj).SmartFavoriteList);
            /* 2. From selected index get preset number */
            tU8 u8SelectedListIndex = (ofiTxObj).SmartFavoriteList.at((prMsg->oFiMsg).ListIndex - 1).PresetIndex;
            /* 3. From preset index get stored service Id */
            SERVICE_ID hServiceId = _u16PresetChannels[u8SelectedListIndex - 1];
            /* 4. From SID get CID */
            CHANNEL_ID hChannelId = DECODER.tGetChannelId(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), hServiceId);
            /* 5. Get Smart Favorite category Id */
            CATEGORY_ID eSmartFavCatId = SMART_FAVORITES.tCategoryId(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder());
            /* 6. Insert selected index preset channel into smart favorite category */
            if(SMSAPI_RETURN_CODE_ERROR !=  CATEGORY.eInsertNewChannel(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), eSmartFavCatId, hChannelId)) 
            {
               /* 7. Update method result status */
               ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
               /* 8. Update smart favorite list and pass it as method result data */
               (ofiTxObj).SmartFavoriteList.clear();
               vUpdateSFList();
               GetSFChannelList((ofiTxObj).SmartFavoriteList, (ofiTxObj).EnableAddPreset);
               /* Update preset smart favorites */
               vUpdatePresetSFList();
            }
         }
      } break;
      case midw_ext_fi_tcl_e8_SFOperation::FI_EN_SXM_AUDIO_REMOVE_FROM_SF_LIST: {
         ETG_TRACE_USR4(("Remove element from SF List from index %d", (prMsg->oFiMsg).ListIndex));
         /* 1. Get smart favorite channel list that is being displayed */
         GetSFChannelList((ofiTxObj).SmartFavoriteList, (ofiTxObj).EnableAddPreset);
         /* 2. Check if selected index is valid */
         if((ofiTxObj).SmartFavoriteList.size() >= (prMsg->oFiMsg).ListIndex) {
            /* 3. From selected index get preset number */
            tU8 u8SelectedListIndex = (ofiTxObj).SmartFavoriteList.at((prMsg->oFiMsg).ListIndex - 1).PresetIndex;
            /* 3. From preset index get stored service Id */
            SERVICE_ID hServiceId = _u16PresetChannels[u8SelectedListIndex - 1];
            /* 4. Get Smart Favorite Category Id */
            CATEGORY_ID eSmartFavCatId = SMART_FAVORITES.tCategoryId(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder());
            /* 5. Iterate through smart favorite channel, pass service id to be removed to the callback function */
            SMSAPI_RETURN_CODE_ENUM hReturnCode = CATEGORY.eIterate(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), 
                                                                    eSmartFavCatId, cb_s16SFListIterateHandler, (void*)&hServiceId);
            if(SMSAPI_RETURN_CODE_SUCCESS != hReturnCode) {
               ETG_TRACE_ERR(("Failed to register for channel callback for SF category %d, ERROR_CODE - %d", 
                              eSmartFavCatId, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, hReturnCode)));
            } /* 6. Inside callback function delete selected channel */
         }
         /* 7. Update method result status */
         ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
         /* 8. Updated smart favorite channel list and pass it as method result data */
        (ofiTxObj).SmartFavoriteList.clear();
         vUpdateSFList();
         GetSFChannelList((ofiTxObj).SmartFavoriteList, (ofiTxObj).EnableAddPreset);
         /* Update preset smart favorites */
         vUpdatePresetSFList();
      } break;
      default:
        break;
   }

   /* Post the Method Result */
   if(AIL_EN_N_NO_ERROR == fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj)) {
      ETG_TRACE_USR3(("SF List control method result sent successfully"));
   } else {
      ETG_TRACE_ERR(("Failed to send method result for SF List control method"));
   }
}


/*****************************************************************************
 * FUNCTION    : vProcess(fc_sxm_trMsgAudioMStartTuneScanControl)            *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : CCA request to handle Tune Scan controls                    *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : prMsg - Tune Scan method start request                      *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartTuneScanControl const *prMsg)
{
    midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult ofiTxObj;
    ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartTuneScan) Type = %d", (prMsg->oFiMsg).TuneScanMode.enType));
    tVoid (fc_sxm_tclPresets::*funcPtrArray[]) (midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj) =  {
            &fc_sxm_tclPresets::vTuneScanStart,
            &fc_sxm_tclPresets::vTuneScanStop,
            &fc_sxm_tclPresets::vTuneScanSkipForward,
            &fc_sxm_tclPresets::vTuneScanSkipBackward,
            &fc_sxm_tclPresets::vTuneScanAbort
    };
    (this->*funcPtrArray[(prMsg->oFiMsg).TuneScanMode.enType])(ofiTxObj);
    /* Post the Method Result */
    if(AIL_EN_N_NO_ERROR == fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj)) {
        ETG_TRACE_USR3(("Tune Scan control method result sent successfully"));
    } else {
        ETG_TRACE_ERR(("Failed to send method result for Tune Scan control method"));
    }
}

tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioSmsEvtTuneScanEvt const *prMsg)
{
    vNotifyTuneScanContentAvailable(DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE & prMsg->u8TuneScanState);
    if(DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED & prMsg->u8TuneScanState)
    {
        vNotifyTuneScanAborted();
    }
    ETG_TRACE_USR4(("u8TuneScanState=%d", prMsg->u8TuneScanState));
}

tVoid fc_sxm_tclPresets::vNotifyTuneScanContentAvailable(tBool bTuneScanState)
{
    if(bTuneScanState != _bIsContentAvailable)
    {
        fc_sxm_tcl_trAudioPropertyTuneScanContentStatus oProperty;
        if(TRUE == bTuneScanState)
        {
            oProperty.oFiMsg.TuneScanStatus.enType = midw_ext_fi_tcl_e8_TuneScanStatus::FI_EN_TUNE_SCAN_AVAILABLE;
        }
        else
        {
            oProperty.oFiMsg.TuneScanStatus.enType = midw_ext_fi_tcl_e8_TuneScanStatus::FI_EN_TUNE_SCAN_NA;
        }
        _bIsContentAvailable = bTuneScanState;
        /* Set the status info and notify all clients */
        fc_sxm_tclAudioProperties::instance()->oTuneScanContentStatus.vSet(oProperty);
        fc_sxm_tclAudioProperties::instance()->oTuneScanContentStatus.vNotify();
        ETG_TRACE_USR4(("TuneScanStatus_available=%d", oProperty.oFiMsg.TuneScanStatus.enType));
    }
}

tVoid fc_sxm_tclPresets::vNotifyTuneScanAborted()
{
    fc_sxm_tcl_trAudioPropertyTuneScanContentStatus oProperty;
    if (_bIsScanInProgress) {
    _bIsScanInProgress = FALSE;
    }
    vCheckChannelStatus();
    oProperty.oFiMsg.TuneScanStatus.enType = midw_ext_fi_tcl_e8_TuneScanStatus::FI_EN_TUNE_SCAN_ABORTED;
    /* Set the status info and notify all clients */
    fc_sxm_tclAudioProperties::instance()->oTuneScanContentStatus.vSet(oProperty);
    fc_sxm_tclAudioProperties::instance()->oTuneScanContentStatus.vNotify();
    ETG_TRACE_USR4(("TuneScanStatus_aborted=%d", oProperty.oFiMsg.TuneScanStatus.enType));
}

/****************************************************************************************
 * Method checks the channel availability status and sends the appropriate system message
 ****************************************************************************************/

tVoid fc_sxm_tclPresets::vCheckChannelStatus(tVoid)
{
    Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                         _poAudioApp->poGetChannelListInstance(): NULL;
    if(poChannelList != NULL)
    {
        DECODER_OBJECT hDecoder = hGetDecoder();
        CHANNEL_ID hChannelId = DECODER.tCurrentChannelId(hDecoder);
        tTunable tDisplayAdvisory = e_UNAVAILABLE;
        /* Retreive the Display advisory */
         midw_ext_fi_tcl_PresetListEntry presetEntry = poChannelList->enGetPresetChannelInfo(hChannelId, FALSE, tDisplayAdvisory);

        /* If channel is unavailable send system message */
        switch(tDisplayAdvisory) {

           case e_UNAVAILABLE: {
              /* Channel unavailable send system message */
               poChannelList->_u16RequestedUnavailableChannel = presetEntry.ChannelID;
              (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAVAILABLE, TRUE);
              poChannelList->vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
           } break;

           case e_UNAUTHORIZED: {
               poChannelList->_u16RequestedUnavailableChannel = presetEntry.ChannelID;
              /* Channel unavailable send system message */
              (fc_sxm_tclAdvisories::instance())->vUpdateChannelStatus(midw_ext_fi_tcl_e8_SystemMessage::FI_EN_XMTUN_SYSTEM_MSG_CHANNEL_UNAUTHORIZED, TRUE);
              poChannelList->vNotifyChannelInformationUpdate(e_UNAVAILABLE_CHANNEL);
           } break;

           default:
               break;
        }
    }
}
/* Method Processes FI request to GetPresetList*/
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartGetPresetList const *prMsg)
{
    ETG_TRACE_USR1(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartGetPresetList) ENTERED"));
    midw_ext_sxm_audiofi_tclMsgGetPresetListMethodResult oMRes;
    midw_ext_fi_tcl_PresetListEntry presetEntry;
    tU32 tPresetBandIndex(prMsg->oFiMsg.PresetBand);
    ETG_TRACE_USR1(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartGetPresetList) tPresetBandIndex: %d", tPresetBandIndex));
    tBool bStatus = TRUE;
    if (tPresetBandIndex == FC_SXM_PRESET_BAND_DEFAULT)
    {

       if(_bIsMixedPreset)
       {
           tU32 tPresetBand = FC_SXM_PRESET_BAND_ONE;
           bStatus = (bStatus) && (bGetAllPresetIndices(tPresetBand, oMRes, presetEntry));
       }
       else
       {
            for (tU32 tPresetBand = FC_SXM_PRESET_BAND_ONE; tPresetBand <= FC_SXM_PRESET_BAND_THREE; ++tPresetBand)
            {
                bStatus = (bStatus) && (bGetAllPresetIndices(tPresetBand, oMRes, presetEntry));
            }
       }
    }
    else
    {
        bStatus = bGetAllPresetIndices(tPresetBandIndex, oMRes, presetEntry);
    }
    if(bStatus)
    {
        oMRes.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
        fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
        ETG_TRACE_USR1(("oMRes.PresetChannels.ChannelList.size = %u\t", oMRes.PresetChannels.ChannelList.size()));
    }
    else
    {
        // Error occurred. Either an invalid index is passed or iterating presets resulted
        // in an error
        ETG_TRACE_ERR(("EITHER INVALID PRESET BAND or FAILURE TO ITERATE PRESETS LIST"));
        fc_sxm_tclAudioService::vSendError(prMsg->rAdressing,
                (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR);
    }
    ETG_TRACE_USR1(("fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartGetPresetList) EXIT"));
}

/* Method Process FI request for User Profile */

tVoid fc_sxm_tclPresets::vProcess(
        fc_sxm_trMsgAudioMStartUserProfile const *prMsg) {
    ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartUserProfile : start"));

    midw_ext_sxm_audiofi_tclMsgUserProfileMethodResult oMRes;
    oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_FAILED;
    oMRes.ProfileResponse.ProfileStatus.enType = prMsg->oFiMsg.ProfileStatus.enType;
    oMRes.ProfileResponse.ProfileAction.enType = prMsg->oFiMsg.ProfileAction.enType;

    tU8 DestID(prMsg->oFiMsg.DestinationID);
    tU8 SrcID(prMsg->oFiMsg.SourceID);

    ETG_TRACE_USR2(
            ("fc_sxm_trMsgAudioMStartUserProfile : DestinationID %u,SourceID = %u initial userActiveProfileID = %u", DestID, SrcID, u8GetUserProfileID()));
    ETG_TRACE_USR2(
            ("fc_sxm_trMsgAudioMStartUserProfile : ProfileAction %d\tProfileStatus %d ",
                    ETG_CENUM(midw_ext_fi_tcl_ProfileAction::tenType,prMsg->oFiMsg.ProfileAction.enType),
                    ETG_CENUM(midw_ext_fi_tcl_ProfileStatus::tenType,prMsg->oFiMsg.ProfileStatus.enType)));

    switch (prMsg->oFiMsg.ProfileAction.enType) {
    case midw_ext_fi_tcl_ProfileAction::FI_EN_ACTIVEPROFILE: {
        vSetUserProfileID(SrcID);
        tU8 u8DPProfileId = 0;
        fc_sxm_tclPersData::instance()->vGetDPProfileId(u8DPProfileId);
                     if (((u8DPProfileId != SrcID)|| (u8DPProfileId == 0 && (u8DPProfileId == SrcID)))&& (prMsg->oFiMsg.ProfileStatus.enType== midw_ext_fi_tcl_ProfileStatus::FI_EN_CHANGED)) {
                    tBool bRes = bHandleChangeProfilePresetList();

                    if (bRes) {
                        _bIsProfileChange = true;
                        fc_sxm_tclPersData::instance()->vSetDPProfileId(SrcID);
                        tU8 DPID = 0;
                        fc_sxm_tclPersData::instance()->vGetDPProfileId(DPID);
                        ETG_TRACE_USR2(("dp profile id = %d",DPID));
                        oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_SUCCESS;
                    } else {
                        oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_FAILED;
                    }

                }
        oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_SUCCESS;
    }
        break;
    case midw_ext_fi_tcl_ProfileAction::FI_EN_COPYPROFILE: {
        oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_SUCCESS;
    }
        break;
    case midw_ext_fi_tcl_ProfileAction::FI_EN_DATACHANGE: {
        vSetUserProfileID(SrcID);

        ETG_TRACE_USR2(("fc_sxm_tcl_profilemanager() fc_sxm_trMsgAppMgrDataChangeUpdate User ID: %u DP profileID:", u8GetUserProfileID()));
        _bIsProfileChange = false;
        if (prMsg->oFiMsg.ProfileStatus.enType== midw_ext_fi_tcl_ProfileStatus::FI_EN_PREPARE) {
            oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_SUCCESS;
        }
    }
        break;
    case midw_ext_fi_tcl_ProfileAction::FI_EN_DELETEPROFILE: {
        oMRes.ProfileResponse.Status.enType = midw_ext_fi_tcl_Status::FI_EN_SUCCESS;
    }
        break;
    default: {
        ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartUserProfile : default "));
    }
        break;

    }

    ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartUserProfile : exit"));
    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing,
            oMRes);
}

tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartGetPresetsConfig const *prMsg) {
    ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartGetPresetsConfig : start"));
    midw_ext_sxm_audiofi_tclMsgGetPresetsConfigMethodResult oMRes;

    oMRes.PresetsConfiguration.BandCapacity = _u8MaxNumberOfPresets;
    oMRes.PresetsConfiguration.BandCount = _u8MaxNumberOfBands;

    fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing,oMRes);
}

tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioMStartSetPresetList const *prMsg) {
    ETG_TRACE_USR2(("fc_sxm_trMsgAudioMStartSetPresetList : start"));
    DECODER_OBJECT hDecoder = hGetDecoder();
    if (prMsg != OSAL_NULL) {
        if (prMsg->oFiMsg.PresetEntry.size())
        {
        midw_ext_sxm_audiofi_tclMsgSetPresetListMethodResult oMRes;

        //initializing preset list
        SERVICE_ID aPresetList[_u8MaxNumberOfPresets * _u8MaxNumberOfBands] = { 0, };

        //check for decoder ready state
        if (hDecoder == DECODER_INVALID_OBJECT) {
            ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartSetPresetList : Decoder state error"));

            //get preset list from dp
            fc_sxm_tclPersData::instance()->vGetPresetData(aPresetList);

            if (!(bPresetListUpdate(aPresetList,prMsg->oFiMsg.PresetEntry))) {
                ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartSetPresetList : Invalid SID or Index received from Tuner Master "));
                fc_sxm_tclAudioService::vSendError(prMsg->rAdressing,
                        (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INVALID_PARAMETER);
            }

            //decoder is not ready, keep back-up of preset list in a buffer, buffer is reset before use
            memset(_u16ProfilePresetChannels, 0x01,(_u8MaxNumberOfPresets * _u8MaxNumberOfBands) * sizeof(SERVICE_ID));
            memcpy(_u16ProfilePresetChannels, aPresetList,(_u8MaxNumberOfPresets * _u8MaxNumberOfBands) * sizeof(SERVICE_ID));

            //update the DP preset list
            fc_sxm_tclPersData::instance()->vSetPresetData(aPresetList);

            //Set Profile Change bit, so that preset list will be updated in config file once the decoder state is ready
            _bIsProfileChange = true;

            oMRes.Status = true;
            fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
        }

        else {

            //Copy the preset list locally
            memset(aPresetList, 0x01,(_u8MaxNumberOfPresets * _u8MaxNumberOfBands) * sizeof(SERVICE_ID));
            memcpy(aPresetList,_u16PresetChannels,(_u8MaxNumberOfPresets * _u8MaxNumberOfBands) * sizeof(SERVICE_ID));

            //if decoder is ready just read the preset list from internal preset array _u16PresetChannels
            if (!(bPresetListUpdate(aPresetList,prMsg->oFiMsg.PresetEntry))) {
                ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartSetPresetList : Invalid SID or Index received from Tuner Master "));
                fc_sxm_tclAudioService::vSendError(prMsg->rAdressing,
                        (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INVALID_PARAMETER);
            }

            //update the preset list
            tBool bRes = bSetProfilePresetList(aPresetList);

            if (bRes) {
                //update DP preset list
                fc_sxm_tclPersData::instance()->vSetPresetData(aPresetList);
                oMRes.Status = true;
                fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
            } else {
                ETG_TRACE_ERR(("Failed to update preset list"));
                oMRes.Status = false;
                fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

            }
        }

    }
        else
        {
            ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartSetPresetList : preset list received from tuner master is empty"));
            fc_sxm_tclAudioService::vSendError(prMsg->rAdressing,(tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INVALID_PARAMETER);
        }
    }
    else {
        ETG_TRACE_ERR(("fc_sxm_trMsgAudioMStartSetPresetList : fc_sxm_trMsgAudioMStartGetPresetsConfig *prMsg is null"));
    }

}

tBool fc_sxm_tclPresets::bPresetListUpdate(SERVICE_ID *aPresetList,const vector<midw_ext_fi_tcl_PresetEntry>& VecPresetList) const
{
    tBool bRes = false;
    for(tU16 u16LoopCount = 0 ; u16LoopCount < VecPresetList.size(); u16LoopCount++)
    {
        if(bIsValidSID(VecPresetList[u16LoopCount].SID) && bIsValidIndex((VecPresetList[u16LoopCount].Index)-1))
        {
            aPresetList[(VecPresetList[u16LoopCount].Index)-1] = VecPresetList[u16LoopCount].SID;
            bRes = true;
        }
        else
        {
            ETG_TRACE_ERR(("bPresetListUpdate : invalid SID or Index"));
            bRes = false;
            break;
        }
    }
    return bRes;
}
tBool fc_sxm_tclPresets::bGetAllPresetIndices(tU32 tPresetBandIndex, midw_ext_sxm_audiofi_tclMsgGetPresetListMethodResult &oMRes, midw_ext_fi_tcl_PresetListEntry &presetEntry)
{
    tBool bIsValidBand = FALSE;
    PRESET_BAND_OBJECT hBand = hGetPresetBandObject(tPresetBandIndex);
    SMSAPI_RETURN_CODE_ENUM eRetCode = enIteratePresetBand(hBand);
    ETG_TRACE_USR1(("eRetCode = %u tPresetBandIndex = %d hBand = %p", eRetCode, tPresetBandIndex, hBand));
    if ((PRESET_BAND_INVALID_OBJECT != hBand)
            && (SMSAPI_RETURN_CODE_SUCCESS == eRetCode)) {
        // The selected preset band is completely iterated
        //midw_ext_sxm_audiofi_tclMsgGetPresetListMethodResult oMRes;
        map<tU32, fc_sxm_trChannel>::const_iterator cPresetIter =
                _mPresetChannels.begin();
        for (; cPresetIter != _mPresetChannels.end(); ++cPresetIter) {
            vFillPresetChannelInfo(cPresetIter->second, presetEntry,
                    tPresetBandIndex, true);
            oMRes.PresetChannels.ChannelList.push_back(presetEntry);
            vPrintPresetChannelInfo(presetEntry);
        }
        bIsValidBand = TRUE;
    }
    ETG_TRACE_USR1(("bIsValidBand = %d", bIsValidBand));
    return bIsValidBand;
}

/*****************************************************************************
 * FUNCTION    : vTuneScanConfigure(tVoid)                                   *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to Configure tune scan process                       *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vTuneScanConfigure()
{
    Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                     _poAudioApp->poGetChannelListInstance(): NULL;
    tBool bScanLockedMature = FALSE;
    if(poChannelList != NULL)
    {
        bScanLockedMature = poChannelList->bGetMatureChannelLockStatus() ? FALSE : TRUE;
    }

    /*Tune Scan Configuration*/   
    if (SMSAPI_RETURN_CODE_SUCCESS == DECODER.eTuneScanConfigure(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), FC_SXM_TUNE_SCAN_TIME
                                                                                , bScanLockedMature, FALSE, DECODER_TUNE_SCAN_STYLE_SMART_FAVORITES )){
    ETG_TRACE_USR4(("Tune scan configured for 6 seconds scan, bScanLockedMature=%d", bScanLockedMature));
    } else {
        ETG_TRACE_USR4(("Tune scan is not configured"));
    }     
}

/*****************************************************************************
 * FUNCTION    : vSetSavedChannelId(tVoid)                                   *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method saves channel id before starting tune scan           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vSetSavedChannelId()
{
    Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                     _poAudioApp->poGetChannelListInstance(): NULL;
    if(poChannelList != NULL)
    {
        _u16ChannelIdBeforeScan = poChannelList->u16GetCurrentTunedChannel();
        ETG_TRACE_USR1(("_u16ChannelIdBeforeScan = %d", _u16ChannelIdBeforeScan));
    }

}

/*****************************************************************************
 * FUNCTION    : u16GetSavedChannelId(tVoid)                                 *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method returns saved channel id before starting tune scan   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : CHANNEL_ID                                                  *
 *****************************************************************************/
CHANNEL_ID fc_sxm_tclPresets::u16GetSavedChannelId() const
{
    return _u16ChannelIdBeforeScan;
}

/*****************************************************************************
 * FUNCTION    : vTuneScanStart(tVoid)                                       *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to Start tune scan process                           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vTuneScanStart(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj)
{
    if ((_bIsContentAvailable) && (!_bIsScanInProgress))
    {
        vSetSavedChannelId();
        fc_sxm_tclAdvisories::instance()->vStopAndClearChannelAdvisory();
        SMSAPI_RETURN_CODE_ENUM eReturnCode = DECODER.eTuneScan(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), DECODER_TUNE_SCAN_CMD_START);
        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode){
        ETG_TRACE_USR4(("Tune scan started"));
        _bIsScanInProgress = TRUE;
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
        }
        else{
            ETG_TRACE_ERR(("Failed to start Tune Scan service - %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode) ));
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        }
    }
    else{
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
            ETG_TRACE_USR4(("Data not available for Scanning"));
    }
}

/*****************************************************************************
 * FUNCTION    : vTuneScanSkipBackward(tVoid)                                *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to Stop tune scan process                            *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vTuneScanStop(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj)
{
    if (_bIsScanInProgress)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode = DECODER.eTuneScan(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), DECODER_TUNE_SCAN_CMD_STOP);
        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode) {
            ETG_TRACE_USR4(("Tune scan stopped"));
            _bIsScanInProgress = FALSE;
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
            Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                                     _poAudioApp->poGetChannelListInstance(): NULL;
            if(poChannelList != NULL)
            {
               SERVICE_ID hServiceID = poChannelList->u16GetSIDOfTunedChannel();
               vNotifyTunedChannelPresetIndex(hServiceID);
            }
            else
            {
                ETG_TRACE_ERR(("Invalid Channel List"));
            }
        }
        else{
            ETG_TRACE_ERR(("Failed to stop Tune Scan service - %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode) ));
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        }
    }
    else {
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
            ETG_TRACE_USR4(("Scanning is not in progress"));
        }
}

/*****************************************************************************
 * FUNCTION    : vTuneScanSkipBackward(tVoid)                                *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to Skip Forward tune scan process                    *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vTuneScanSkipForward(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj)
{
    if (_bIsScanInProgress)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode = DECODER.eTuneScan(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), DECODER_TUNE_SCAN_CMD_SKIP_FORWARD);
        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode){
        ETG_TRACE_USR4(("Tune scan Skip Forward"));
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
        }
        else{
            ETG_TRACE_ERR(("Failed to skip forward Tune Scan - %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode) ));
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        }
    }
    else
    {
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        ETG_TRACE_USR4(("Scanning is not in progress.. can not skip forward"));
    }
}

/*****************************************************************************
 * FUNCTION    : vTuneScanSkipBackward(tVoid)                                *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to Skip Backward tune scan process                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vTuneScanSkipBackward(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj)
{
    if (_bIsScanInProgress)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode = DECODER.eTuneScan(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), DECODER_TUNE_SCAN_CMD_SKIP_BACKWARD);
        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode){
        ETG_TRACE_USR4(("Tune scan Skip Backward"));
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
        }
        else{
            ETG_TRACE_ERR(("Failed to skip forward Tune Scan - %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode) ));
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        }
    }
    else
    {
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        ETG_TRACE_USR4(("Scanning is not in progress.. can not skip backward"));
    }
}

/*****************************************************************************
 * FUNCTION    : vTuneScanAbort(tVoid)                                       *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to Abort tune scan process                           *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : None                                                        *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vTuneScanAbort(midw_ext_sxm_audiofi_tclMsgTuneScanControlMethodResult &ofiTxObj)
{
    if (_bIsScanInProgress)
    {
        SMSAPI_RETURN_CODE_ENUM eReturnCode = DECODER.eTuneScan(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), DECODER_TUNE_SCAN_CMD_ABORT);
        if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode){
        ETG_TRACE_USR4(("Tune scan Aborted"));
        _bIsScanInProgress = FALSE;
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
        }
        else{
            ETG_TRACE_ERR(("Failed to abort Tune Scan - %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode) ));
            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        }
    }
    else
    {
        ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
        ETG_TRACE_USR4(("Scanning is not in progress.. can not abort"));
    }
}

tVoid fc_sxm_tclPresets::vSetTuneScanStatus(tBool bSetTuneScanState)
{
    _bIsScanInProgress = bSetTuneScanState;
}

/*****************************************************************************
 * FUNCTION    : s16UpdatePresetInfo                                         *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Method to fetch persisted preset and band info and store    *
 *               them in internal preset list                                *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hBand        - Valid preset band object                     *
 * PARAMETER   : hBandIndex   - Preset band index that is being iterated     *
 * PARAMETER   : hPresetIndex - Preset index in preset band being iterated   *
 * PARAMETER   : hChannel     - Valid channel object in current preset       *
 * RETURNVALUE : N16          - Steps to iterate further                     *
 *****************************************************************************/
N16 fc_sxm_tclPresets::s16UpdatePresetInfo(PRESETS_OBJECT hPresets,PRESET_BAND_OBJECT hBand, size_t hBandIndex, size_t hPresetIndex, CHANNEL_OBJECT hChannel)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::s16UpdatePresetInfo() hBandIndex: %d", hBandIndex));
   tS8 lRelativeOffset = -1;
   if(PRESET_BAND_TYPE_USER == BAND.eType(hBand))
   {
   switch(hBandIndex) {
      case 0: {
         _hBandObjectOne = hBand;
         lRelativeOffset = FC_SXM_BAND_ONE_OFFSET;
      } break;
      case 1: {
         _hBandObjectTwo = hBand;
         lRelativeOffset = FC_SXM_BAND_TWO_OFFSET;
      } break;
      case 2: {
         _hBandObjectThree = hBand;
         lRelativeOffset = FC_SXM_BAND_THREE_OFFSET;
      } break;
      default: {
      } break;
    }
   }
   else if(PRESET_BAND_TYPE_FEATURED == BAND.eType(hBand))
   {
      size_t tCapacity = BAND.tCapacity(hBand); 
      STRING_OBJECT hBandName = BAND.hLongName(hBand);
      string sBandLongName;
      fc_sxm_vCopySmsString2Stl(hBandName, sBandLongName);
      vUpdateFeaturedBandInfo(hBand, PRESETS_OBJECT_EVENT_BAND_ADDED, sBandLongName, tCapacity);
   }
   N16 hReturnData = 0;
   if(0 <= lRelativeOffset) {
      size_t uNumberOfBands = PRESETS.tNumBands(hPresets);
      if(BAND.eType (hBand) == PRESET_BAND_TYPE_USER) {
      /* Get service ID and save in preset array */
      SERVICE_ID hServiceID = CHANNEL.tServiceId(hChannel);
      if(hServiceID != SERVICE_INVALID_ID)
      {
          _u16PresetChannels[lRelativeOffset + hPresetIndex] = hServiceID;
      }
      ETG_TRACE_USR1(("SERVICE_ID of A PRESET = %d \t uNumberOfBands = %u \t (lRelativeOffset + hPresetIndex) = %u",
              hServiceID, uNumberOfBands, (lRelativeOffset + hPresetIndex)));
      }
      /* Check if further data is required */
      if((hBandIndex == (uNumberOfBands - 1)) && (hPresetIndex ==  (BAND.tCapacity(hBand) - 1))) {
         hReturnData = 0;
      } else {
         hReturnData++;
      }
   }
   return hReturnData;
}

/*****************************************************************************
 * FUNCTION    : u8GetPresetIndex                                            *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Get the preset index for given service ID                   *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hServiceID   - Valid service ID                             *
 * RETURNVALUE : tU8          - Preset index value                           *
 *****************************************************************************/
tU8 fc_sxm_tclPresets::u8GetPresetIndex(SERVICE_ID hServiceID) const
{
   tU8 lPresetIndex = 0;
   // Find the first occurrence of service ID in preset list
   for(tU8 u8LoopCount = FC_SXM_AUDIO_MIN_PRESET_COUNT; u8LoopCount <= (_u8MaxCountOfPresetsInOneBand * _u8MaxNumberOfBands); u8LoopCount++) {
      /* Copy the preset index if service ID's matches */
      if(hServiceID == _u16PresetChannels[u8LoopCount - 1]) {
         lPresetIndex = u8LoopCount;
         ETG_TRACE_USR2(("lPresetIndex - %d", lPresetIndex));
         break;
      }
   }
   return lPresetIndex;
}

// Method fills "mPresetIndices" with the list of presets
// where SERVICE_ID of the channels are the primary key and their corresponding preset index as the Secondary key
tVoid fc_sxm_tclPresets::vGetPresetIndices(map<SERVICE_ID, tU8>& mPresetIndices) const
{
	mPresetIndices.clear();
	for(tU8 u8LoopCount = FC_SXM_AUDIO_MIN_PRESET_COUNT; u8LoopCount <= (_u8MaxCountOfPresetsInOneBand * _u8MaxNumberOfBands); u8LoopCount++) {
		SERVICE_ID hServiceID = _u16PresetChannels[u8LoopCount - 1];
		// Filling only the first occurrence of service ID in preset list
		if (mPresetIndices.count(hServiceID) == 0) {
			mPresetIndices[hServiceID] = u8LoopCount;
		}
	}
	ETG_TRACE_USR2(("fc_sxm_tclPresets::vGetPresetIndices mPresetIndices size = %u", mPresetIndices.size()));
}

/*****************************************************************************
 * FUNCTION    : vNotifyTunedChannelPresetIndex                              *
 * CLASS       : fc_sxm_tclPresets                                           *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Send auto compare property update for current tuned channel *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hServiceID - Current tuned channel service ID               *
 * RETURNVALUE : None                                                        *
 *****************************************************************************/
tVoid fc_sxm_tclPresets::vNotifyTunedChannelPresetIndex(SERVICE_ID hServiceID)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vNotifyTunedChannelPresetIndex()"));

   if(SERVICE_INVALID_ID != hServiceID) {
      /* Set channel list property to true */
      fc_sxm_tcl_trAudioPropertyAutoCompare AutoCompareStatus;
      ((AutoCompareStatus).oFiRes).TunerElemNr = 0;
      ((AutoCompareStatus).oFiRes).CurrentTunerElemNr = 0;
      /* Get index and range based on current preset band */
      tU8 u8LoopCount = (tU8)((_u32CurrentBandNumber - 1) * _u8MaxCountOfPresetsInOneBand);
      tU8 u8LoopRange = (tU8)(u8LoopCount + _u8MaxCountOfPresetsInOneBand);
      tU8 u8LoopCountMax = (_u8MaxNumberOfPresets * _u8MaxNumberOfBands);

      /* Handling auto compare status for the current band */
      for( ; u8LoopCount<u8LoopRange; u8LoopCount++) {
         ETG_TRACE_USR4(("u8LoopCount - %u \t u8LoopRange - %u", u8LoopCount, u8LoopRange));
         if(hServiceID == _u16PresetChannels[(tU16)u8LoopCount]) {
            ((AutoCompareStatus).oFiRes).CurrentTunerElemNr = (tU8)(u8LoopCount + 1);
            break;
         }
      }
      ETG_TRACE_USR3(("CurrentTunerElemNr - %d", ((AutoCompareStatus).oFiRes).CurrentTunerElemNr));

      /* Handling auto compare status for all bands */
      for (tU8 u8LoopCountMin = 0; u8LoopCountMin<u8LoopCountMax; u8LoopCountMin++) {
          if (hServiceID == _u16PresetChannels[u8LoopCountMin]) {
              ((AutoCompareStatus).oFiRes).TunerElemNr = (tU8)(u8LoopCountMin + 1);
              break;
         }
      }
      ETG_TRACE_USR3(("TunerElemNr - %d", ((AutoCompareStatus).oFiRes).TunerElemNr));

      /* Get index and range based on current preset band */
      u8LoopCount = (tU8)((_u32CurrentBandNumber - 1) * _u8MaxCountOfPresetsInOneBand);
      u8LoopRange = (tU8)(u8LoopCount + _u8MaxCountOfPresetsInOneBand);
      /* Within given band search for smart favorite presets */
      for(; u8LoopCount <u8LoopRange; u8LoopCount++) {
         /* Create SF List element */
         midw_ext_fi_tcl_SmartFavoritePreset tSFListElement;
         tSFListElement.PresetIndex = (tU8)(u8LoopCount+1);
         tSFListElement.IsSmartFavorite = FALSE;
         /* Search if preset is smart favorite preset */
         for(tU8 u8SFLoopCount=0; u8SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u8SFLoopCount++) {
            if(_u16PresetChannels[u8LoopCount] == _u16SFChannels[u8SFLoopCount]) {
               tSFListElement.IsSmartFavorite = TRUE;
               ETG_TRACE_USR4(("AUTO SF - %d", tSFListElement.PresetIndex));
               break;
            }
         }
         /* Insert list element in auto compare property */
         (((AutoCompareStatus).oFiRes).SmartFavoriteList).push_back(tSFListElement);
      }

      /* Set the status info and notify all clients */
      _u8PreviousTunerElemNr = ((AutoCompareStatus).oFiRes).CurrentTunerElemNr;
      fc_sxm_tclAudioProperties::instance()->oAutoCompare.vSet(AutoCompareStatus);
      fc_sxm_tclAudioProperties::instance()->oAutoCompare.vNotify();
   } else {
      ETG_TRACE_ERR(("Invalid service ID received for auto compare update"));
   }
}

/*****************************************************************************
 * FUNCTION    : s16updateChannelInfo                                        *
 * CLASS       : fc_sxm_tclCategoryList                                      *
 * ------------------------------------------------------------------------- *
 * DESCRIPTION : Update channel info in category list                        *
 * ------------------------------------------------------------------------- *
 * PARAMETER   : hCategory     - Valid category object                       *
 * PARAMETER   : tCurrentIndex - Channel index being iterated                *
 * PARAMETER   : hChannel      - Valid channel object                        *
 * RETURNVALUE : N16           - Steps to iterate further                    *
 *****************************************************************************/
N16 fc_sxm_tclPresets::s16updateSFChannelInfo(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX tCurrentIndex, CHANNEL_OBJECT hChannel, tVoid* pIterData)
{
   /* Get Category ID */
   CATEGORY_ID hCategoryID = CATEGORY.tGetCategoryId(hCategory);
   /* Get total number of channels available in the given category */
   size_t u32ChannelCount = CATEGORY.tSize(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(), hCategoryID);;
   ETG_TRACE_USR4(("fc_sxm_tclPresets::s16updateSFChannelInfo() tCurrentIndex = %d  hCategoryID=%d  ChnCount = %d",
           tCurrentIndex, hCategoryID, u32ChannelCount));
   if(0 != u32ChannelCount) {
      if(OSAL_NULL == pIterData) { /* If no data is passed then add elements to the list */
         /* Store smart favorite channel service ID in internal array */
         if((0 <= tCurrentIndex) &&(FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT > tCurrentIndex)) {
            SERVICE_ID hServiceID = CHANNEL.tServiceId(hChannel);
            _u16SFChannels[tCurrentIndex] = hServiceID;
            ETG_TRACE_USR1(("Index - %d, Service ID - %d", tCurrentIndex, hServiceID));
         }
      } else { /* Search for selected SF channel and remove it from the list */
         SERVICE_ID hRemoveServiceId = *((SERVICE_ID*)pIterData);
         SERVICE_ID hServiceId = CHANNEL.tServiceId(hChannel);
         ETG_TRACE_USR4(("hRemoveServiceId = %d  hServiceId=%d", hRemoveServiceId, hServiceId));
         if(hServiceId == hRemoveServiceId) {
            /* Remove selected category from smart favorite list */
            if(SMSAPI_RETURN_CODE_SUCCESS != CATEGORY.eRemoveChannel(hCategory)) {
               ETG_TRACE_ERR(("Failed to remove selected channel from smart favorite list"));
            } else {
               u32ChannelCount--; /* Decrement list count */
            }
         }
      }
   }
   /* If data recived is not the last element, then request for next element data */
   N16 n16HandlerReturnValue = 0;
   if(u32ChannelCount != (tCurrentIndex+1)) {
      n16HandlerReturnValue = 1;
   }
   return (n16HandlerReturnValue);
}

/* Update instant replay mode */
tVoid fc_sxm_tclPresets::vUpdateIRMode(tBool bValidate, tBool bSetMode)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vUpdateIRMode()"));

   /* Instant Replay mode property */
   fc_sxm_tcl_trAudioPropertyIRMode IRMode;
   (IRMode.oFiRes).IsIRModeActive = bSetMode;  /* Set to default value */

   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                      _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN(NULL != poChannelList)

   if((bValidate) && (SMART_FAVORITES_PLAY_POINT_CTRL_START == _ePlayPoint)) {
      /* Get SID of current tuned channel */
      SERVICE_ID uTunedChannelSID = poChannelList->u16GetSIDOfTunedChannel();
      /* Check if current tuned channel is a smart favorite channel */
      tU32 u32SFLoopCount = 0;
      for(u32SFLoopCount=0; u32SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u32SFLoopCount++) {
         if(uTunedChannelSID == _u16SFChannels[u32SFLoopCount]) {
            (IRMode.oFiRes).IsIRModeActive = TRUE;
         }
      }
   } else {
      /* Do nothing */
   }
   if(!(IRMode.oFiRes).IsIRModeActive) {
      /* Set the status info and notify all clients */
      fc_sxm_tclAudioProperties::instance()->oInstantReplayMode.vSet(IRMode);
      fc_sxm_tclAudioProperties::instance()->oInstantReplayMode.vNotify();
   } else {
      /* HACK: Block the first instant replay property update soon after new channel is tuned */
      Ifc_sxm_tclPlayback* poPlayback = (_poAudioApp) ?
                                  _poAudioApp->poGetPlaybackInstance(): NULL;
      SXM_ASSERT_RETURN(NULL != poPlayback)
   }
}

/* Update instant replay mode */
tVoid fc_sxm_tclPresets::vNotifyIRMode(tBool bValidate, tBool bSetMode)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vNotifyIRMode()"));

   /* Instant Replay mode property */
   fc_sxm_tcl_trAudioPropertyIRMode IRMode;
   (IRMode.oFiRes).IsIRModeActive = bSetMode;  /* Set to default value */

   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                          _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN(NULL != poChannelList)

   if((bValidate) && (SMART_FAVORITES_PLAY_POINT_CTRL_START == _ePlayPoint)) {
      /* Get SID of current tuned channel */
      SERVICE_ID uTunedChannelSID = poChannelList->u16GetSIDOfTunedChannel();
      /* Check if current tuned channel is a smart favorite channel */
      tU32 u32SFLoopCount = 0;
      for(u32SFLoopCount=0; u32SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u32SFLoopCount++) {
         if(uTunedChannelSID == _u16SFChannels[u32SFLoopCount]) {
            (IRMode.oFiRes).IsIRModeActive = TRUE;
         }
      }
   } else {
      /* Do nothing */
   }

   //if((IRMode.oFiRes).IsIRModeActive) {
   //   ETG_TRACE_USR2(("***************************************** IR MODE TRUE "));
   //} else {
   //   ETG_TRACE_USR2(("***************************************** IR MODE FALSE "));
   //}

   /* Set the status info and notify all clients */
   fc_sxm_tclAudioProperties::instance()->oInstantReplayMode.vSet(IRMode);
   fc_sxm_tclAudioProperties::instance()->oInstantReplayMode.vNotify();
}

tVoid fc_sxm_tclPresets::GetSFChannelList(vector<midw_ext_fi_tcl_SFListElement> &SFPresetList, tBool &EnableAddPreset) const
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::GetSFChannelList()"));

   tU32 u32PresetLoopCount = 0;
   tU32 u32SFLoopCount = 0;
   /* Make a local copy to sort preset elements */
   map<tU8, tBool>SortedSFList;
   for(; u32SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u32SFLoopCount++) {
      tBool bIsRepeated = FALSE;

      for(u32PresetLoopCount=0; u32PresetLoopCount< _u8MaxNumberOfPresets * _u8MaxNumberOfBands; u32PresetLoopCount++) {
         if(_u16SFChannels[u32SFLoopCount] == _u16PresetChannels[u32PresetLoopCount]) {
            tU32 u32FlagLoop = u32PresetLoopCount+1; /* Check if the same channel is stored in other presets as well */
            for(; u32FlagLoop<(_u8MaxNumberOfPresets * _u8MaxNumberOfBands);u32FlagLoop++) {
               if(_u16PresetChannels[u32PresetLoopCount] == _u16PresetChannels[u32FlagLoop]) {
                  bIsRepeated = TRUE;
                  break;
               }
            }
            break;
         }
      }
         if((_u8MaxNumberOfPresets * _u8MaxNumberOfBands) != u32PresetLoopCount) {
#if(FC_SXM_AUDIO_TO_DEBUG)
         ETG_TRACE_USR2(("Preset %d contains smart favorite channel %d", u32PresetLoopCount+1, _u16SFChannels[u32SFLoopCount]));
#endif
         SortedSFList.insert(pair<tU8, tBool>((tU8)(u32PresetLoopCount+1), bIsRepeated));
      }
   }
   /* From local copy pass it to vector list */
   map<tU8,tBool>::iterator mIter;
   midw_ext_fi_tcl_SFListElement pSFListElement;
   for(mIter=SortedSFList.begin(); mIter!=SortedSFList.end(); ++mIter) {
      pSFListElement.PresetIndex = (*mIter).first;
      pSFListElement.IsRepeated = (*mIter).second;
      (SFPresetList).push_back(pSFListElement);
   }
   /* Clear data and exit */
   SortedSFList.clear();

   EnableAddPreset = FALSE;
   for(u32PresetLoopCount=0; u32PresetLoopCount<(_u8MaxNumberOfPresets * _u8MaxNumberOfBands);u32PresetLoopCount++) {
      for(u32SFLoopCount=0; u32SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u32SFLoopCount++) {
         if(_u16PresetChannels[u32PresetLoopCount] == _u16SFChannels[u32SFLoopCount]) {
            break; /* If service id match then move to next preset channel */
         }
      }
      if(u32SFLoopCount == FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT) {
         EnableAddPreset = TRUE;
         break; /* Preset element found that has no sf channel */
      }
   }
}

tVoid fc_sxm_tclPresets::GetNonSFChannelList(vector<midw_ext_fi_tcl_SFListElement> &SFPresetList) const
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::GetNonSFChannelList()"));
   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
                          _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN(NULL != poChannelList)
   tU32 u32PresetLoopCount = 0;
   tU32 u32SFLoopCount = 0;

   for(; u32PresetLoopCount<(_u8MaxNumberOfPresets * _u8MaxNumberOfBands); u32PresetLoopCount++) {
      for(u32SFLoopCount=0; u32SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u32SFLoopCount++) {
         if(_u16PresetChannels[u32PresetLoopCount] == _u16SFChannels[u32SFLoopCount]) {
            break;
         }
      }
      if((FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT == u32SFLoopCount) && 
         (_u16PresetChannels[u32PresetLoopCount] != poChannelList->u16GetSIDOfPreviewChannel())) {
#if(FC_SXM_AUDIO_TO_DEBUG)
         ETG_TRACE_USR2(("Preset %d do not contain smart favorite channel", u32PresetLoopCount+1));
#endif
         midw_ext_fi_tcl_SFListElement pSFListElement;
         pSFListElement.PresetIndex = (tU8)(u32PresetLoopCount+1);
         pSFListElement.IsRepeated = FALSE;
         (SFPresetList).push_back(pSFListElement);
      }
   }
}

/* Set Audio App Pointer Reference*/
tVoid fc_sxm_tclPresets::vSetApp(Ifc_sxm_tclAudioApp* audioApp)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vSetApp()"));
    _poAudioApp = audioApp;
}

/**
 * Get Decoder Object
 * @return decoder object
 */
DECODER_OBJECT fc_sxm_tclPresets::hGetDecoder() const
{
   return (((_poAudioApp) ? _poAudioApp->oGetDecoderObject() : DECODER_INVALID_OBJECT));
}

/* Method to set preset*/
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclPresets::enSetPreset(PRESET_BAND_OBJECT hBand, tU8 u8PresetIndex, CHANNEL_ID tChannelId) const
{
    return (BAND.eSetPreset(hBand, u8PresetIndex, tChannelId));
}

/* Method to tune to the selected preset channel.
 * bLockOverride: This argument is used to signify if the caller wishes to override both the mature AND
 *                      the lock attribute of the channel associated with this preset.
 * */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclPresets::enBandTune(PRESET_BAND_OBJECT hBand, tU8 u8PresetIndex, tBool bLockOverride) const
{
    return (BAND.eTune(hBand, u8PresetIndex, bLockOverride));
}

/* Method to iterate contents of a preset band.*/
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclPresets::enIteratePresetBand(PRESET_BAND_OBJECT hBand)
{
    // Prepare for iteration of preset channels within the BAND
    _mPresetChannels.clear();
    _tLastBandIndex = 0U;
    _tLastPresetIndex = 0U;
    return (BAND.eIterate(hBand, cb_vBandIterateHandler, (tVoid*)this));
}

/* Method to retrieve the appropriate Preset Band Object from preset band list index*/
PRESET_BAND_OBJECT fc_sxm_tclPresets::hGetPresetBandObject (const tU32 u32PresetBand) const
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::hGetPresetBandObject() Entered\tPresetBandIndex = %u", u32PresetBand));
    PRESET_BAND_OBJECT hBand = PRESET_BAND_INVALID_OBJECT;
    switch(u32PresetBand)
    {
        case FC_SXM_PRESET_BAND_ONE:
        {
            hBand = _hBandObjectOne;
            break;
        }
        case FC_SXM_PRESET_BAND_TWO:
        {
            hBand = _hBandObjectTwo;
            break;
        }
        case FC_SXM_PRESET_BAND_THREE:
        {
            hBand = _hBandObjectThree;
            break;
        }
        default:
        {
            // Unknown Band... Return invalid preset band object
            ETG_TRACE_ERR(("FC_SXM_PRESET_BAND...... UNKNOWN"));
            break;
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclPresets::hGetPresetBandObject() Exit"));
    return hBand;
}

N16 fc_sxm_tclPresets::s16UserCategoryRemoveChannel(CATEGORY_OBJECT hCategory, CATEGORY_CHANNEL_INDEX /*tCurrentIndex*/, CHANNEL_OBJECT /*hChannel*/, tVoid* /*pIterData*/)
{
    if(SMSAPI_RETURN_CODE_SUCCESS != CATEGORY.eRemoveChannel(hCategory)) {
        ETG_TRACE_ERR(("Failed to remove selected channel from featured favorite list"));
    }

    //Iteration stop condition implicitly taken care by SMS by checking category size.
    return 1;
}


/* Method to add a Preset belonging to a specific band*/
N16 fc_sxm_tclPresets::n16AddPreset(PRESETS_OBJECT /*hPresets*/, PRESET_BAND_OBJECT hBand, size_t tBandIndex,
                CHANNEL_OBJECT hChannel, STRING_OBJECT /*hPresetName*/, size_t tPresetindex)
{
    (tVoid)hBand; //To remove lint warning
    ETG_TRACE_USR1(("fc_sxm_tclPresets::n16AddPreset() Entered\t tBandIndex = %u\t tPresetindex = %u", tBandIndex, tPresetindex));
    // If the preset index wrapped back to zero and the band id didn't change
    // or if the preset index wrapped back to zero and so did the band id
    if ( (tPresetindex < _tLastPresetIndex) && (tBandIndex <= _tLastBandIndex))
    {
        // We have gone through all the presets belonging to the band
        ETG_TRACE_USR1(("Iterated all Presets from the Band"));
        return 0;
    }
    // Store the indices
    _tLastBandIndex = tBandIndex;
    _tLastPresetIndex = tPresetindex;

    // Does this channel have any info for us??
    if ((CHANNEL_INVALID_OBJECT != hChannel) || (CD_INVALID_OBJECT != CHANNEL.hCDO( hChannel) ))
    {
        ETG_TRACE_USR1(("Adding VALID Channel to the Preset Band"));

         fc_sxm_trChannel rValidChannel(hChannel);
         rValidChannel.bIsChannelArtAvailable = static_cast <tBool> (CHANNEL_ART_INVALID_OBJECT != CHANNEL.hArt(hChannel));
         _mPresetChannels[(tU32)tPresetindex] = rValidChannel;
    }
    else
    {
        ETG_TRACE_ERR(("CHANNEL has NO INFO"));
    }
    // Continue iteration of presets within the BAND
    return 1;

}
/* Method to add a Preset belonging to a specific featured band*/
N16 fc_sxm_tclPresets::n16AddFeaturedPreset(PRESETS_OBJECT /*hPresets*/, PRESET_BAND_OBJECT hBand, size_t tBandIndex,
        CHANNEL_OBJECT hChannel, STRING_OBJECT /*hPresetName*/, size_t tPresetindex, tBool bFeaturedFavModified)
{
    ETG_TRACE_USR1(("fc_sxm_tclPresets::n16AddFeaturedPreset() Entered\t tBandIndex = %u\t tPresetindex = %u\t bFeaturedFavModified = %d",
                         tBandIndex, tPresetindex, bFeaturedFavModified));
    // If the preset index wrapped back to zero and the band id didn't change
    // or if the preset index wrapped back to zero and so did the band id
    if ( (tPresetindex < _tLastPresetIndex) && (tBandIndex <= _tLastBandIndex))
    {
        // We have gone through all the presets belonging to the band
        ETG_TRACE_USR1(("Iterated all Presets from the Band"));
        return 0;
    }
    // Store the indices
    _tLastBandIndex = tBandIndex;
    _tLastPresetIndex = tPresetindex;

    // Does this channel have any info for us??
    if ((CHANNEL_INVALID_OBJECT != hChannel) || (CD_INVALID_OBJECT != CHANNEL.hCDO( hChannel) ))
    {
        ETG_TRACE_USR1(("Adding VALID Channel to the Preset Band"));
        SERVICE_ID hServiceID = CHANNEL.tServiceId(hChannel);
        if(bFeaturedFavModified == PRESETS_OBJECT_EVENT_BAND_MODIFIED) {
            _vectServiceId.push_back(hServiceID);
        } else {
            STRING_OBJECT hBandName = BAND.hLongName(hBand);
            string sBandLongName;
            fc_sxm_vCopySmsString2Stl(hBandName, sBandLongName);
            vAddChannelIntoCategory(sBandLongName, hServiceID);
        }
    }
    else
    {
        ETG_TRACE_ERR(("Channel has no info"));
    }
    // Continue iteration of presets within the BAND
    return 1;
}

/* Method retrieves the number of preset band objects associted with the Presets object */
size_t fc_sxm_tclPresets::tGetPresetBandsSize() const
{
    return ( PRESETS.tNumBands(_hPresetObject) );
}

/* Method sends property "PresetChannelUpdate" to HMI */
tVoid fc_sxm_tclPresets::vPresetBandUpdate (fc_sxm_trChannel const& rChannel, CHANNEL_EVENT_MASK enEventMask) const
{
    const SERVICE_ID hServiceID = rChannel.u16ServiceId;
    const size_t tBandCount = tGetPresetBandsSize();
	const tU32 u32Range = (tU32)(_u8MaxNumberOfPresets *  tBandCount);
    for(tU32 u32Index = 0U; u32Index < u32Range; u32Index++)
    {
        if (_u16PresetChannels[u32Index] == hServiceID)
        {
            // Fill up the FI element and update HMI
            fc_sxm_tcl_trAudioPropertyPresetChannelUpdate oProperty;
            oProperty.oFiMsg.PresetIndex = 1 + (u32Index % _u8MaxNumberOfPresets);
            oProperty.oFiMsg.PresetBand = 1 + (u32Index / _u8MaxNumberOfPresets);
            oProperty.oFiMsg.LinearPresetIndex = 1 + u32Index;
            vFillPresetChannelInfo (rChannel, oProperty.oFiMsg.ChannelInfo, oProperty.oFiMsg.PresetBand, (CHANNEL_OBJECT_EVENT_ART & enEventMask));
            // Separate property updates would be sent to HMI if the same preset is stored
            // multiple times in same or in different bands irrespective if the band is the
            // current band.
            fc_sxm_tclAudioProperties::instance()->oPresetInfo.vSetAndNotify(oProperty);
            ETG_TRACE_USR1(("fc_sxm_tclPresets::vPresetBandUpdate()\t PresetIndex = %u\t PresetBand = %u \t ChannelID = %u ServiceID = %u",
                    oProperty.oFiMsg.PresetIndex, oProperty.oFiMsg.PresetBand, rChannel.u16ChannelId, rChannel.u16ServiceId));

            ETG_TRACE_USR1(("oProperty.oFiMsg.LinearPresetIndex = %u",oProperty.oFiMsg.LinearPresetIndex));

            vPrintPresetChannelInfo(oProperty.oFiMsg.ChannelInfo);
        }
    }
}

/* Method Fills FI element midw_ext_fi_tcl_PresetListEntry from the supplied channel*/
tVoid fc_sxm_tclPresets::vFillPresetChannelInfo(
        fc_sxm_trChannel const& rChannel,
        midw_ext_fi_tcl_PresetListEntry& trPreset,
        tU32 tPresetBandIndex, bool bRefreshArt) const
{
    // Fill all FI elements from the channel object
    trPreset.ServiceID = static_cast <tU16> (rChannel.u16ServiceId);
    trPreset.ChannelID = static_cast <tU16> (rChannel.u16ChannelId);
    trPreset.ChannelActive = rChannel.bIsAvailable();
    if (rChannel.oAudioChannelName.length())
    {
        fc_sxm_vString2Fi(rChannel.oAudioChannelName.c_str(), trPreset.ChannelName);
    }
    else if ((TRUE == trPreset.ChannelActive) && (NULL != _poAudioApp))
    {
        // The channel attributes notification from SMSLib did not return any channel name.
        // So collect the channel name from the internal list
        const fc_sxm_trChannel* poChannel = _poAudioApp->trGetChannel(rChannel.u16ChannelId);
        if (NULL != poChannel)
        {
            fc_sxm_vString2Fi(poChannel->oAudioChannelName.c_str(), trPreset.ChannelName);
        }
    }


    if ( rChannel.bIsChannelArtAvailable )
    {
        trPreset.ChnGraphicAvail.enType = (bRefreshArt) ?
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH :
                midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_NOREFRESH;
    }
    else
    {
        trPreset.ChnGraphicAvail.enType = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
    }
    /* Get index and range based on current preset band */
	tU8 u8LoopCount = (tU8)((tPresetBandIndex - 1) * _u8MaxCountOfPresetsInOneBand);
	tU8 u8LoopRange = (tU8)(u8LoopCount + _u8MaxCountOfPresetsInOneBand);
    /* Create SF List element */
    trPreset.IsSmartFavorite = FALSE;
    /* Search if preset is smart favorite preset */
    for(tU8 u8SFLoopCount=0; u8SFLoopCount<FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; u8SFLoopCount++) {
        if(trPreset.ServiceID == _u16SFChannels[u8SFLoopCount])
        {
         trPreset.IsSmartFavorite = TRUE;
         break;
        }
    }
    ETG_TRACE_USR4(("vFillPresetChannelInfo IsSmartFavorite- %d", trPreset.IsSmartFavorite));
    trPreset.IsLocked = rChannel.bIsLocked();
    trPreset.IsAvailable = rChannel.bIsAvailable();
    trPreset.IsMature = rChannel.bIsMature();
    trPreset.IsSkipped = rChannel.bIsSkipped();
}

/* Method prints FI structure PresetChannelInfo*/
tVoid fc_sxm_tclPresets::vPrintPresetChannelInfo(const midw_ext_fi_tcl_PresetListEntry& trPreset) const
{
    ETG_TRACE_USR4(("---> fc_sxm_tclPresets::vPrintPresetChannelInfo <--- IN"));
    ETG_TRACE_USR4(("ServiceID = %u \t ChannelID = %u", trPreset.ServiceID, trPreset.ChannelID));
    ETG_TRACE_USR4(("ChannelName = %s", trPreset.ChannelName.szValue));
    ETG_TRACE_USR4(("ChnGraphicAvail = %u \t ChannelActive = %u", trPreset.ChnGraphicAvail.enType, trPreset.ChannelActive));
    ETG_TRACE_USR4(("IsSmartFavorite = %u \t IsLocked = %u", trPreset.IsSmartFavorite, trPreset.IsLocked));
    ETG_TRACE_USR4(("IsAvailable = %u \t IsMature = %u", trPreset.IsAvailable, trPreset.IsMature));
    ETG_TRACE_USR4(("IsLocked = %u \t IsSkipped = %u", trPreset.IsLocked, trPreset.IsSkipped));
    ETG_TRACE_USR4(("<--- fc_sxm_tclPresets::vPrintPresetChannelInfo ---> OUT"));
}

/* Method to start featured favorite service */
tVoid fc_sxm_tclPresets::vStartFeaturedFavorites()
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vStartFeaturedFavorites()"));
   SMSAPI_RETURN_CODE_ENUM enReturnCode = PRESETS.eFeaturedFavoritesStart(_hPresetObject, FC_SXM_AUDIO_FEATURED_FAV_CAPACITY,
                                                                                 FC_SXM_AUDIO_FEATURED_FAV_NUM_BANDS,
                                                                                    FC_SXM_AUDIO_FEATURED_FAV_PURPOSE);
      if(enReturnCode == SMSAPI_RETURN_CODE_SUCCESS) {
         ETG_TRACE_USR4(("Featured Favorite successfully started"));
        } else {
         ETG_TRACE_ERR(("Failed to start Featured Favorite due to err code - %d",enReturnCode));
        }
}


/* Method to create user category */
CATEGORY_ID fc_sxm_tclPresets::tCreateUserCategory(size_t tCapacity, const string& sBandLongName)
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::vCreateUserCategory()"));
    DECODER_OBJECT hDecoder = (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
                                               DECODER_INVALID_OBJECT;
    if(DECODER_INVALID_OBJECT != hDecoder) {
        return (CATEGORY.tCreate(hDecoder, sBandLongName.c_str(), NULL, (N16)(tCapacity)));
    } else {
       ETG_TRACE_ERR(("Invalid Decoder Object"));
       return CATEGORY_INVALID_ID;
    }

}

/* Method to insert channel into user category */
tVoid fc_sxm_tclPresets::vAddChannelIntoCategory(const string& sBandLongName, SERVICE_ID hServicelId)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vAddChannelIntoCategory()"));
    DECODER_OBJECT hDecoder = (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
            DECODER_INVALID_OBJECT;
    if(DECODER_INVALID_OBJECT != hDecoder) {
        CHANNEL_ID hChannelId = DECODER.tGetChannelId (hDecoder, hServicelId);
        if(hChannelId == CHANNEL_INVALID_ID) {
            ETG_TRACE_ERR(("Invalid Channel id"));
        } else {
            ETG_TRACE_USR4(("Band to compare (%s)", sBandLongName.c_str()));
            CATEGORY_ID n16CatId = n16GetCategoryId(sBandLongName);
            if(n16CatId != CATEGORY_INVALID_ID) {
               SMSAPI_RETURN_CODE_ENUM enRetCode = CATEGORY.eInsertNewChannel(hDecoder, n16CatId, hChannelId);
               if(enRetCode == SMSAPI_RETURN_CODE_ERROR) {
                  ETG_TRACE_ERR(("Failed to add the channel (%u) into Category(%s)", hChannelId, n16CatId));
               }
           } else {
               ETG_TRACE_ERR(("Failed to find valid category"));
           }
       }
    } else {
         ETG_TRACE_ERR(("Invalid Decoder Object"));
    }
}

/* Method to update Featured bands by SMS */
tVoid fc_sxm_tclPresets::vUpdateFeaturedBandInfo(PRESET_BAND_OBJECT hBand, PRESETS_EVENT_MASK hEventMask, string sBandLongName, size_t tCapacity)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vUpdateFeaturedBandInfo() ... EventMask = %x tCapacity = %u Band = %s", hEventMask, tCapacity, sBandLongName.c_str()));
    /* Band added */
    if(PRESETS_OBJECT_EVENT_BAND_ADDED & hEventMask){
        /* Discarding the duplicate bands for addition */
        if(_mFeaturedFavInfo.count(sBandLongName) == 0){
            vAddFeaturedBands(hBand, sBandLongName, tCapacity);
        } else{
            ETG_TRACE_USR4(("Discarding duplicate featured bands for addition"));
        }
    }
    /* Band removed */
    if(PRESETS_OBJECT_EVENT_BAND_REMOVED & hEventMask)
    {
        vRemoveFeaturedBands(sBandLongName);
    }
    /* Band modified */
    if(PRESETS_OBJECT_EVENT_BAND_MODIFIED & hEventMask)
    {
        vModifyFeaturedBands(hBand, sBandLongName, tCapacity);
    }
}

/* Method to add Featured band by SMS */
tVoid fc_sxm_tclPresets::vAddFeaturedBands(PRESET_BAND_OBJECT hBand, const string& sBandLongName, size_t tCapacity)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vAddFeaturedBands() BandName = %s", sBandLongName.c_str()));
    if(PRESET_BAND_INVALID_OBJECT == hBand)
    {
        ETG_TRACE_ERR(("Failed to get valid Preset band object"));
    } else {
        /* Create user category for featured favorites band */
        CATEGORY_ID nCatId = tCreateUserCategory(tCapacity, sBandLongName);
        if(nCatId == CATEGORY_INVALID_ID) {
            ETG_TRACE_ERR(("Invalid Category Id"));
        } else {
            _tLastBandIndex = 0U;
            _tLastPresetIndex = 0U;

            /* Update the list with band name and category id */
            fc_sxm_trFeaturedFavInfo sCatBandId;
            sCatBandId.n16CategoryId = nCatId;
            sCatBandId.tBandId = _tLastBandIndex;
            _mFeaturedFavInfo.insert(pair<string, fc_sxm_trFeaturedFavInfo>(sBandLongName, sCatBandId));

            fc_sxm_trFeaturedBandIteratorCBArg trFeaturedBandIteratorCBArg;
            trFeaturedBandIteratorCBArg.poFeaturedPresets = this;
            trFeaturedBandIteratorCBArg.bFeaturedFavModified = PRESETS_OBJECT_EVENT_BAND_ADDED;
            /* Iterate the new added band */
            BAND.eIterate(hBand, cb_vFeaturedBandIterateHandler, static_cast <tVoid *>(&trFeaturedBandIteratorCBArg));

            map<string, fc_sxm_trFeaturedFavInfo>::iterator mIter;
            mIter = _mFeaturedFavInfo.find(sBandLongName);
            if (mIter != _mFeaturedFavInfo.end())
            {
                ((*mIter).second).tBandId = _tLastBandIndex;
            }
        }
    }

}

/* Method to remove Featured band by SMS */
tVoid fc_sxm_tclPresets::vRemoveFeaturedBands(const string& sBandLongName)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vRemoveFeaturedBands() %s", sBandLongName.c_str()));
    /* Band removed */

        CATEGORY_ID n16CatId = n16GetCategoryId(sBandLongName);
        if(n16CatId != CATEGORY_INVALID_ID) {
            DECODER_OBJECT hDecoder = (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
                                                           DECODER_INVALID_OBJECT;
            if(DECODER_INVALID_OBJECT != hDecoder) {
              /* delete the user category for featured favorites band */
              CATEGORY.vDestroy(hDecoder, n16CatId);

              /* If found delete the entry from the list */
              map<string, fc_sxm_trFeaturedFavInfo>::iterator mIter;
              mIter = _mFeaturedFavInfo.find(sBandLongName);
              if(mIter != _mFeaturedFavInfo.end()) {
                _mFeaturedFavInfo.erase(mIter);
              }
            } else {
               ETG_TRACE_ERR(("Band could not be found"));
            }
        } else {
             ETG_TRACE_ERR(("Failed to find the band for deletion"));
        }
}

/* Method to modify Featured band by SMS */
tVoid fc_sxm_tclPresets::vModifyFeaturedBands(PRESET_BAND_OBJECT hBand, const string& sBandLongName, size_t tCapacity)
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vModifyFeaturedBands()"));

    if(PRESET_BAND_INVALID_OBJECT == hBand) {
        ETG_TRACE_ERR(("Failed to get valid Preset band object"));
    } else {
        _tLastBandIndex = 0U;
        _tLastPresetIndex = 0U;
        _vectServiceId.clear();

        fc_sxm_trFeaturedBandIteratorCBArg trFeaturedBandIteratorCBArg;
        trFeaturedBandIteratorCBArg.poFeaturedPresets = this;
        trFeaturedBandIteratorCBArg.bFeaturedFavModified = PRESETS_OBJECT_EVENT_BAND_MODIFIED;
        //Iterate the existing band
        BAND.eIterate(hBand, cb_vFeaturedBandIterateHandler, static_cast <tVoid *>(&trFeaturedBandIteratorCBArg));

        ETG_TRACE_USR4(("Restoring all Channel Objects & get the Band Index: %d from Band Iterator....",_tLastBandIndex));
        CATEGORY_ID nCatId = n16GetCategoryId(_tLastBandIndex);

        ETG_TRACE_USR4(("Get the Category Id: %d for the Band Index: %d from Band Iterator....",nCatId,_tLastBandIndex));
        map<string, fc_sxm_trFeaturedFavInfo>::const_iterator cIter;
        cIter = _mFeaturedFavInfo.find(sBandLongName);
        if(cIter == _mFeaturedFavInfo.end()) {
            ETG_TRACE_USR4(("Band name has been changed... %s", sBandLongName.c_str()));
           vRenameUserCategory(nCatId, sBandLongName);
        }
        vUpdateUserCategory(nCatId, sBandLongName);
    }
}

tVoid fc_sxm_tclPresets::vRenameUserCategory(N16 n16CategoryId, const string& sBandLongName)
{
    DECODER_OBJECT hDecoder = (_poAudioApp)? _poAudioApp->oGetDecoderObject() :
            DECODER_INVALID_OBJECT;
    if((n16CategoryId != CATEGORY_INVALID_ID) && (hDecoder != DECODER_INVALID_OBJECT)) {
        CATEGORY.eRename(hDecoder, n16CategoryId, sBandLongName.c_str(),sBandLongName.c_str());
        ETG_TRACE_USR4(("Rename done for the Category Id: %d with Band name: %s ....",n16CategoryId,sBandLongName.c_str()));

        //Updating the Band name.
        map<string, fc_sxm_trFeaturedFavInfo>::iterator mIterCatList;
        for(mIterCatList = _mFeaturedFavInfo.begin();
                mIterCatList != _mFeaturedFavInfo.end(); ++mIterCatList)
        {
            if(mIterCatList->second.n16CategoryId == n16CategoryId) {
                ETG_TRACE_USR4(("Deleted category Id: %d & Band name: %s", mIterCatList->second.n16CategoryId, (mIterCatList->first).c_str()));
                _mFeaturedFavInfo.erase(mIterCatList);
                break;
            }
        }
        ETG_TRACE_USR4(("Updating CategoryId: %d & the Band name: %s", n16CategoryId, sBandLongName.c_str()));

        fc_sxm_trFeaturedFavInfo sCatBandId;
        sCatBandId.n16CategoryId = n16CategoryId;
        sCatBandId.tBandId = _tLastBandIndex;
        _mFeaturedFavInfo.insert(pair<string, fc_sxm_trFeaturedFavInfo>(sBandLongName, sCatBandId));
    }
}

tVoid fc_sxm_tclPresets::vUpdateUserCategory(N16 n16CategoryId, const string& sBandLongName)
{
    //Clearing the channels from the user category list.
    SMSAPI_RETURN_CODE_ENUM hReturnCode = CATEGORY.eIterate(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder(),
            n16CategoryId, cb_s16UserCategoryRemoveChannelIterateHandler, this);

    if(SMSAPI_RETURN_CODE_ERROR == hReturnCode) { return; }
    //Adding Channels into the user category.
    ETG_TRACE_USR4(("Adding channels into the user category: %d....", n16CategoryId));

    for(U32 uLoopCount = 0; uLoopCount < _vectServiceId.size(); uLoopCount++)
    {
        vAddChannelIntoCategory(sBandLongName, _vectServiceId[uLoopCount]);
    }
}

/* Utility method to print the list of all smart favorites and presets at one shot
 * Assumptions: Count of Smart favorites and Presets always remains the same
 */
tVoid fc_sxm_tclPresets::vPrintSFAndPresetsList() const
{
    // Print format would be
    // FC_SXM_PRESETS_US4:_u16PresetChannels[0] = 20     _u16SFChannels[0] = 20
    for(U32 uLoopCount = 0; uLoopCount < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; uLoopCount++) {
        ETG_TRACE_USR4(("_u16PresetChannels[%u] = %u \t _u16SFChannels[%u] = %u",
                uLoopCount, _u16PresetChannels[uLoopCount], uLoopCount, _u16SFChannels[uLoopCount] ));
    }
}

N16 fc_sxm_tclPresets::n16GetCategoryId(size_t tBandIndex) const
{
    map<string, fc_sxm_trFeaturedFavInfo>::const_iterator cIter;
    for(cIter = _mFeaturedFavInfo.begin();
            cIter != _mFeaturedFavInfo.end(); ++cIter)
    {
        if (((*cIter).second).tBandId == tBandIndex)
        {
            return ((*cIter).second).n16CategoryId;
        }
    }
    return CATEGORY_INVALID_ID;
}

N16 fc_sxm_tclPresets::n16GetCategoryId(const string& strBandName) const
{
    map<string, fc_sxm_trFeaturedFavInfo>::const_iterator cIter;
    cIter = _mFeaturedFavInfo.find(strBandName);
    if (cIter != _mFeaturedFavInfo.end())
    {
        return cIter->second.n16CategoryId;
    }
    return CATEGORY_INVALID_ID;
}


// Method returns Band Long name of the Preset band object
tVoid fc_sxm_tclPresets::vGetBandLongName(PRESET_BAND_OBJECT hBand, string& strBandLongName) const
{
    STRING_OBJECT hBandLongName = BAND.hLongName(hBand);
    fc_sxm_vCopySmsString2Stl(hBandLongName, strBandLongName);
}

// Method handles the Decoder object event Tune to update Auto compare Property
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioSmsEvtTuneState const * /* prMsg */ )
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::fc_sxm_trMsgAudioSmsEvtTuneState"));
    Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
            _poAudioApp->poGetChannelListInstance(): NULL;
    SXM_ASSERT_RETURN(NULL != poChannelList)

    SERVICE_ID hServiceID = poChannelList->u16GetSIDOfTunedChannel();
    if (hServiceID != _u16StoredServiceid)
    {
        vNotifyTunedChannelPresetIndex(hServiceID);
        _u16StoredServiceid = SERVICE_INVALID_ID;
    }
}

// Method handles the Decoder object event Tune to update Auto compare Property
tVoid fc_sxm_tclPresets::vProcess(fc_sxm_trMsgAudioSmsEvtChannelEvt const * /* prMsg */ )
{
   ETG_TRACE_USR4(("fc_sxm_tclPresets::fc_sxm_trMsgAudioSmsEvtChannelEvt"));
   Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
         _poAudioApp->poGetChannelListInstance(): NULL;
   SXM_ASSERT_RETURN(NULL != poChannelList)

   SERVICE_ID hServiceID = poChannelList->u16GetSIDOfTunedChannel();
   if (hServiceID != _u16StoredServiceid)
   {
      vNotifyTunedChannelPresetIndex(hServiceID);
      _u16StoredServiceid = SERVICE_INVALID_ID;
   }
}

// Method adds a recovery mechanism if there happen to be any synchronisation issues between List of Smart favorites and the first 12 presets
tVoid fc_sxm_tclPresets::vRecoverSFList()
{
    set <SERVICE_ID> presetsSIDs;
    set <SERVICE_ID> smartFavoritesSIDs;
    DECODER_OBJECT hDecoder = (_poAudioApp) ? _poAudioApp->oGetDecoderObject(): DECODER_INVALID_OBJECT;
    CATEGORY_ID categoryID = SMART_FAVORITES.tCategoryId(hDecoder);
    SMSAPI_RETURN_CODE_ENUM enReturnCode;
    for (tU8 u8Index = 0; u8Index < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; ++u8Index)
    {
        presetsSIDs.insert(_u16PresetChannels[u8Index]);
        if (SERVICE_INVALID_ID != _u16SFChannels[u8Index])
        {
            smartFavoritesSIDs.insert(_u16SFChannels[u8Index]);
        }
    }

    // First check if all presets from first two bands are part of SF List as well .... if not add them now
    set<SERVICE_ID>::iterator it = presetsSIDs.begin();
    tBool bChanged = FALSE;
    for(; it != presetsSIDs.end(); ++it)
    {
        CHANNEL_ID hChannelId = DECODER.tGetChannelId(hDecoder, *it);
        if ((CHANNEL_INVALID_ID != hChannelId) && (smartFavoritesSIDs.count(*it) == 0))
        {
            // Channel part of first 12 presets  but not part of smart favorites list ... so add them to the SF list
            enReturnCode = CATEGORY.eInsertNewChannel(hDecoder, categoryID, hChannelId);
            if (SMSAPI_RETURN_CODE_SUCCESS  == enReturnCode )
            {
                ETG_TRACE_USR4(("VRecoverSFList Channel newly added to Smart Favprites List chnID = %u SID = %u", hChannelId, *it));
                bChanged = TRUE;
            }
            else
            {
                ETG_TRACE_ERR(("VRecoverSFList Channel ADD ERROR List chnID = %u SID = %u", hChannelId, *it));
            }
        }
    }

    //Next check if there are redundant Smart Favorites which are not part of presets list and remove them
    it = smartFavoritesSIDs.begin();
    for(; it != smartFavoritesSIDs.end(); ++it)
    {
        if (presetsSIDs.count(*it) == 0)
        {
            CHANNEL_ID hChannelId = DECODER.tGetChannelId(hDecoder, *it);
            SERVICE_ID hRemoveSID = *it;

            enReturnCode = CATEGORY.eIterate(hDecoder, categoryID, cb_s16SFListIterateHandler, (void*)&hRemoveSID);
            if(SMSAPI_RETURN_CODE_SUCCESS != enReturnCode)
            {
                ETG_TRACE_ERR(("VRecoverSFList Channel REMOVE ERROR from Smart Favprites ---- FAILchnID = %u SID = %u , ERROR_CODE - %d",
                        hChannelId, hRemoveSID, ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
            }
            else
            {
                ETG_TRACE_USR4(("VRecoverSFList Channel REMOVED from Smart Favprites List --- SUCCESS chnID = %u SID = %u", hChannelId, hRemoveSID));
                bChanged = TRUE;
            }
        }
    }

    // If any changes were seen in the first 2 stages, then reset internal list of SF array and re-populate the list
    if (TRUE == bChanged)
    {
        vResetSFList();
        enReturnCode = CATEGORY.eIterate(hDecoder, categoryID, cb_s16SFListIterateHandler, NULL);
        if(SMSAPI_RETURN_CODE_SUCCESS == enReturnCode)
        {
            ETG_TRACE_USR4(("New Smart Favorites List Iteration Successfully completed"));
        }
        else
        {
            ETG_TRACE_ERR(("Smart Favorites List Iteration FAILED : %u", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
        }
    }
}

//Method to reset smart favorite list
tVoid fc_sxm_tclPresets::vResetSFList()
{
    for(U32 uLoopCount = 0; uLoopCount < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; uLoopCount++) {
        _u16SFChannels[uLoopCount] = SERVICE_INVALID_ID;
    }

}

//Method to reset the preset list
tVoid fc_sxm_tclPresets::vResetPresetList()
{
    for(U32 uLoopCount = 0; uLoopCount < (_u8MaxNumberOfPresets * _u8MaxNumberOfBands); uLoopCount++) {
       _u16PresetChannels[uLoopCount] = FC_SXM_AUDIO_DEFAULT_CHANNEL;
    }

}



//Method to get relative preset index(0 to 5) from preset index
tU8 fc_sxm_tclPresets::u8GetRelativePresetIndex(tU8 lPresetIndex)
{
    tU8 lPresetElement = 0;
    if(FC_SXM_BAND_ONE_MAX_INDEX - 1 >= lPresetIndex) {
        _u32CurrentBandNumber = FC_SXM_PRESET_BAND_ONE;
        lPresetElement = lPresetIndex;
    } else if((FC_SXM_BAND_TWO_MIN_INDEX - 1 <= lPresetIndex) && (FC_SXM_BAND_TWO_MAX_INDEX - 1 >= lPresetIndex)) {
        _u32CurrentBandNumber = FC_SXM_PRESET_BAND_TWO;
        lPresetElement = (tU8)(lPresetIndex - _u8MaxCountOfPresetsInOneBand);
    } else if((FC_SXM_BAND_THREE_MIN_INDEX - 1 <= lPresetIndex) && (FC_SXM_BAND_THREE_MAX_INDEX - 1 >= lPresetIndex)) {
        _u32CurrentBandNumber = FC_SXM_PRESET_BAND_THREE;
        lPresetElement = (tU8)(lPresetIndex - (2 * _u8MaxCountOfPresetsInOneBand));
    } else {
        /* Do nothing */
    }
    return lPresetElement;
}


//Method to get preset band object from band number
PRESET_BAND_OBJECT fc_sxm_tclPresets::hGetPresetBand(tU32 u32BandNumber) const
{
    PRESET_BAND_OBJECT hBandObject = PRESET_BAND_INVALID_OBJECT;
    switch(u32BandNumber) {
    case FC_SXM_PRESET_BAND_ONE: {
        hBandObject = _hBandObjectOne;
    } break;
    case FC_SXM_PRESET_BAND_TWO: {
        hBandObject = _hBandObjectTwo;
    } break;
    case FC_SXM_PRESET_BAND_THREE: {
        hBandObject = _hBandObjectThree;
    } break;
    default:
        break;
    }
    return hBandObject;
}

tVoid fc_sxm_tclPresets::vPrintPresetList() const
{

    for(tU8 index=0; index<(_u8MaxNumberOfPresets * _u8MaxNumberOfBands); index++)
    {
       ETG_TRACE_USR4(("fc_sxm_tclPresets::vPrintPresetList _u16PresetChannels[%u]: %u",index,_u16PresetChannels[index]));
    }
}


tVoid fc_sxm_tclPresets::vPrintSFList() const
{
    for(tU8 index=0; index < FC_SXM_AUDIO_MAX_SF_CHANNEL_COUNT; index++)
    {
       ETG_TRACE_USR4(("fc_sxm_tclPresets::vPrintSFList _u16SFChannels[%u]: %u",index,_u16SFChannels[index]));
    }
}


// Method notifies the clients that Preset List has been updated. Upon receiving the status message parameter
// as TRUE, the clients need to request for the complete Preset List
tVoid fc_sxm_tclPresets::vPresetListChangeNotification()
{
    ETG_TRACE_USR4(("fc_sxm_tclPresets::vPresetListChangeNotification"));
    fc_sxm_tcl_trAudioPropertyPresetListUpdate oProperty;
    oProperty.oFiMsg.ListUpdate = TRUE; // blindly setting the property to be true
    fc_sxm_tclAudioProperties::instance()->oPresetListUpdate.vSet(oProperty);
    fc_sxm_tclAudioProperties::instance()->oPresetListUpdate.vNotify();
}

/*****************************************************************************
 * ----------------------------- END OF FILE ------------------------------- *
 *****************************************************************************/
