/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_playback.cpp
* @brief       Implementation for handling instant replay control functionalities, 
*              updating buffered data info such as progress bar, status bar, and play point indicators.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_playback.h"
#include "fc_sxm_tcl_advisories.h"
#include "fc_sxm_tcl_states.h"
#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_channel_list_if.h"
#include "fc_sxm_tcl_presets_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PLAYBACK
#include "trcGenProj/Header/fc_sxm_tcl_playback.cpp.trc.h"
#endif

#define FC_SXM_FORGROUND_CHANNEL_BUFFER_MAX 3600

fc_sxm_tclPlayback::fc_sxm_tclPlayback()
: _u32BufferTime(0)
, _bUpdateIRStatus(FALSE)
, _s16IRSkipCount(0)
, _poAudioApp(NULL)
, _u32ThresholdValue(1)
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback constructor"));
}

fc_sxm_tclPlayback::~fc_sxm_tclPlayback()
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback destructor"));
   /* De-initialization class member variables */
   _u32BufferTime = 0;
   _bUpdateIRStatus = FALSE;
   _s16IRSkipCount = 0;

   _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.ElapsedTime     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = 
      midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_INVALID_STATE;
   _poAudioApp = NULL;
}

tVoid fc_sxm_tclPlayback::vUpdateDecoderState(DECODER_OBJECT hDecoderObject, DECODER_STATE_ENUM enDecoderState)
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback::vUpdateDecoderState()"));
   if((DECODER_STATE_READY == enDecoderState) && (DECODER_INVALID_OBJECT != hDecoderObject)) {
      /* Get total buffer duration */
      _u32BufferTime = PLAYBACK.un32IRBufferDuration(DECODER.hPlayback(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder()));
      if(0 == _u32BufferTime) {
         ETG_TRACE_ERR(("No buffer available for playback in this decoder"));
      } else {
         ETG_TRACE_USR4(("Total time buffer available = %d", _u32BufferTime));
      }
      _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = 
         midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE;
   } else {
      _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = 
         midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_INVALID_STATE;
   }
}

tVoid fc_sxm_tclPlayback::vClearMemberData(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback::vClearMemberData"));
   /* Clear record content info */
   _u32BufferTime = 0;
   _bUpdateIRStatus = FALSE;
   _s16IRSkipCount = 0;
   _tRecordInfo.vClear();
   /* De-initialization class member variables */
   _tIRStatus.oFiRes.InstantReplayStatus.RecordDuration     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.ElapsedTime     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.FillPercentage     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.PlayPercentage     = 0;
   _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = 
      midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_INVALID_STATE;
}

tVoid fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioMStartIRPlaybackControl const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioMStartIRPlaybackControl)"));

   /* Define method result data to send */
   midw_ext_sxm_audiofi_tclMsgInstantReplayControlMethodResult ofiTxObj;

   DECODER_OBJECT hDecoder = fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder();
   if(hDecoder != DECODER_INVALID_OBJECT)
   {
	   SERVICE_ID tServiceID =  DECODER.tCurrentServiceId(hDecoder);
	      ETG_TRACE_USR4(("Playback mode is %d for service id %u", (prMsg->oFiMsg).u8IRPlaybackMode.enType,tServiceID));

           ETG_TRACE_USR4(("PlaybackStatus=%d",
              ETG_CENUM(midw_ext_fi_tcl_e8_IRPlaybackStatus::tenType,_tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType)));

	      if(tServiceID == FC_SXM_AUDIO_RADIO_ID_CHANNEL)
	      {
	   	   ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE;
	   	   ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
	   	   ofiTxObj.u8IRPlaybackMode.enType = midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_PLAY_MODE;
	      }
	      else
	      {
	   	   ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_INVALID_STATE;
	   	   ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_FAILURE;
	   	   ofiTxObj.u8IRPlaybackMode.enType = (prMsg->oFiMsg).u8IRPlaybackMode.enType;
	   	   switch(((prMsg->oFiMsg).u8IRPlaybackMode).enType)
	   	      {
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_PLAY_MODE:
	   	         {
	   	            if(_mIRSeekTimer.bIsRunning()) {
	   	               _mIRSeekTimer.vStop();
	   	               _tRecordInfo.vClear();
	   	            }
	   	            if(midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE != _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType) {
	   	               /* If seek timer is running then stop and clear data */

	   	               if(SMSAPI_RETURN_CODE_SUCCESS == PLAYBACK.ePlay(DECODER.hPlayback(hDecoder))) {
	   	                  ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE;
	   	                  ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
                                 _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE;
	   	               } else {
	   	                  ETG_TRACE_ERR(("PLAYBACK.ePlay() failed"));
	   	               }
	   	            }
	   	         }
	   	         break;
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_PAUSE_MODE:
	   	         {
	   	            if(midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PAUSE_STATE != _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType) {
	   	               if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.ePause(DECODER.hPlayback(hDecoder))) {
	   	                  ETG_TRACE_ERR(("PLAYBACK.ePause() failed"));
	   	               } else {
	   	                  ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PAUSE_STATE;
	   	                  ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
                                 _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PAUSE_STATE;
	   	               }
	   	               /* If seek timer is running then stop and clear data */
	   	               if(_mIRSeekTimer.bIsRunning()) {
	   	                  _mIRSeekTimer.vStop();
	   	                  _tRecordInfo.vClear();
	   	               }
	   	            }
	   	         }
	   	         break;
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_SEEK_FORWARD_MODE:
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_SEEK_BACKWARD_MODE:
	   	         {
	   	            _tRecordInfo.s32SeekDirection = +1;
	   	            _tRecordInfo.u32SeekTime      = 5;
	   	            tBool bPauseAfterSeek         = FALSE;
	   	            if(midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_SEEK_BACKWARD_MODE == ((prMsg->oFiMsg).u8IRPlaybackMode).enType) {
	   	               _tRecordInfo.s32SeekDirection = -1;
	   	               _eIRStatusControl.bIgnoreIRUpdate = TRUE;  /* To Fix Replay -> Live -> Replay issue on HMI */
	   	            }
	   	            /* If seek is not possible then inform the same to HMI */
	   	            if( ((0 < _tRecordInfo.s32SeekDirection) && (0 == _tRecordInfo.u32RecordRemainingTime)) ||
	   	                ((0 > _tRecordInfo.s32SeekDirection) && (FC_SXM_IR_MIN_SEEK_TIME >= _tRecordInfo.u32RecordElapsedTime)) )
	   	            {
	   	               _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SEEK_NA_STATE;
	   	            } else {
	   	               /* Seek for 5 sec */
	   	               if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekTime(DECODER.hPlayback(hDecoder),
	   	                                                                 (_tRecordInfo.u32SeekTime * _tRecordInfo.s32SeekDirection),
	   	                                                                 bPauseAfterSeek)) {
	   	                  ETG_TRACE_ERR(("PLAYBACK.eSeekTime() failed"));
	   	               } else {
	   	                  ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SEEK_STATE;
	   	               }
	   	            }
	   	            /* In any case start the timer and send the method result as success */
	   	            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
	   	            /* Start Timer */
	   	            _mIRSeekTimer.vStart(FC_SXM_IR_SEEK_TIME_DELAY);
	   	            _tRecordInfo.bIsSeekActive = TRUE;
	   	            _tRecordInfo.u32TimeOutCount = 0;
	   	         }
	   	         break;
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_SKIP_FORWARD_MODE:
	   	         {
	   	            if(!_mIRSkipTimer.bIsRunning()) {
	   	               if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekNext(DECODER.hPlayback(hDecoder), FALSE)) {
	   	                  ETG_TRACE_ERR(("PLAYBACK.eSeekNext() failed"));
	   	               } else {
	   	                  ETG_TRACE_USR2(("PLAYBACK.eSeekNext() successful !!!"));
	   	                  ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SKIP_STATE;
	   	                  _mIRSkipTimer.vStart(FC_SXM_IR_SKIP_TIME_DELAY);
	   	                  _s16IRSkipCount = 0;
	   	               }
	   	            } else {
	   	               _s16IRSkipCount++;
	   	            }
	   	            /* In any case send method result as success */
	   	            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
	   	         } break;
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_SKIP_BACKWARD_MODE:
	   	         {
	   	            if(!_mIRSkipTimer.bIsRunning()) {
	   	               if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekPrevious(DECODER.hPlayback(hDecoder), FALSE)) {
	   	                  ETG_TRACE_ERR(("PLAYBACK.eSeekPrevious() failed"));
	   	               } else {
	   	                  ETG_TRACE_USR2(("PLAYBACK.eSeekPrevious() successful !!!"));
	   	                  ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SKIP_STATE;
	   	                  _mIRSkipTimer.vStart(FC_SXM_IR_SKIP_TIME_DELAY);
	   	               }
	   	            } else {
	   	               _s16IRSkipCount--;
	   	            }
	   	            /* In any case send method result as success */
	   	            ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
	   	         } break;
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_JUMP_TO_LIVE_MODE:
	   	         {
	   	            if(0 < _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive) {
	   	               tBool bPauseAfterSeek = FALSE;
	   	               if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekTime(DECODER.hPlayback(hDecoder),
	   	                                                                 (N32)PLAYBACK_TIME_OFFSET_LIVE,
	   	                                                                 bPauseAfterSeek)) {
	   	                  ETG_TRACE_ERR(("PLAYBACK.eSeekTime() failed"));
	   	               } else {
	   	                  ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SEEK_STATE;
	   	                  ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
	   	               }
	   	            } else if(midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE != _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType) {
	   	               if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.ePlay(DECODER.hPlayback(hDecoder))) {
	   	                  ETG_TRACE_ERR(("PLAYBACK.ePlay() failed"));
	   	               }
	   	            }
	   	         }
	   	         break;
	   	         case midw_ext_fi_tcl_e8_IRPlaybackMode::FI_EN_SXM_AUDIO_IR_JUMP_TO_BUFFER_START:
	   	         {
	   	       	  if(0 < _tIRStatus.oFiRes.InstantReplayStatus.ElapsedTime) {
	   	       		  tBool bPauseAfterSeek = FALSE;
	   	       		  if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekTime(DECODER.hPlayback(hDecoder),
	   	       				  	  	  	  	  	  	  	  	  	  	  	(N32)PLAYBACK_TIME_OFFSET_START,
	   	   																bPauseAfterSeek)) {
	   	   				 ETG_TRACE_ERR(("PLAYBACK.eSeekTime() to PLAYBACK_TIME_OFFSET_START failed"));
	   	   			  } else {
	   	   				 ofiTxObj.u8IRPlaybackState.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SEEK_STATE;
	   	   				 ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;
	   	   			  }
	   	       	  }
	   	       	  else if(midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE != _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType) {
	   	   			  if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.ePlay(DECODER.hPlayback(hDecoder))) {
	   	   				 ETG_TRACE_ERR(("PLAYBACK.ePlay() failed"));
	   	   			  }
	   	       	  }
	   	         }
	   	         break;
	   	         default:
	   	         {
	   	         }
	   	         break;
	   	      }

	      }


	      /* Post the Method Result */
	      if(AIL_EN_N_NO_ERROR == fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj)) {
	         ETG_TRACE_USR3(("IR Playback method result sent successfully"));
	      } else {
	         ETG_TRACE_ERR(("Failed to send method result for IR Playback control"));
	      }
   }

   else
   {
	   ETG_TRACE_ERR(("vProcess(fc_sxm_trMsgAudioMStartIRPlaybackControl) Invalid Decoder Object"));
   }


}

tVoid fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioMStartCurrentVolume const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioMStartCurrentVolume)"));

   /* Define method result data to send */
   midw_ext_sxm_audiofi_tclMsgSetCurrentVolumeMethodResult ofiTxObj;
   ofiTxObj.IsMuteActive = (prMsg->oFiMsg).IsMuteActive;
   ofiTxObj.Status.enType = midw_ext_fi_tcl_e8_MethodStatus::FI_EN_XMTUN_MS_SUCCESS;

   /* Post the Method Result */
   if(AIL_EN_N_NO_ERROR == fc_sxm_tclAudioService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj)) {
      ETG_TRACE_USR3(("Current volume method result sent successfully"));
   } else {
      ETG_TRACE_ERR(("Failed to send method result for current volume"));
   }
}

tVoid fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioSmsEvtPlaybackEvt const *prMsg)
{
   // Use for testing
   //prMsg->vPrint();
   tU32 u32PreviousRecordRemainingTime = 0;
   tU32 u32PreviousRecordDuration = 0;

   if( (PLAYBACK_OBJECT_EVENT_PLAY & prMsg->ePlaybackEventMask) || (PLAYBACK_OBJECT_EVENT_PAUSE & prMsg->ePlaybackEventMask) ) {

      _tIROldStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType;

      if(prMsg->IsPlayActive) { /* If play state is PLAY */
         if(_tRecordInfo.bIsSeekActive) { /* Then check if seek is active */
            _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_SEEK_STATE;
         } else { /* Else set PLAY state */
            _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE;
         }
      } else { /* If not PLAY then set PAUSE state */
         _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PAUSE_STATE;
      }

      if(_tIROldStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType != _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType) {
    	  ETG_TRACE_USR4(("_bUpdateIRStatus ACTIVATED @ PLAYBACK_OBJECT_EVENT_PLAY/PLAYBACK_OBJECT_EVENT_PAUSE"));
         _bUpdateIRStatus = TRUE;
      }
   }

   ETG_TRACE_USR4(("fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioSmsEvtPlaybackEvt) _bUpdateIRStatus = %u, PlaybackStatus=%d",
            _bUpdateIRStatus,
            ETG_CENUM(midw_ext_fi_tcl_e8_IRPlaybackStatus::tenType,_tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType)));

   if(PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO & prMsg->ePlaybackEventMask) {
      u32PreviousRecordRemainingTime = (_tRecordInfo).u32RecordRemainingTime;
      u32PreviousRecordDuration = (_tRecordInfo).u32RecordDuration;

      ETG_TRACE_USR3(("(prMsg->u32RecordDuration) -------- %d", (prMsg->u32RecordDuration)));
      ETG_TRACE_USR4(("(prMsg->u32RecordElapsedTime) ----- %d", (prMsg->u32RecordElapsedTime)));
      ETG_TRACE_USR3(("(prMsg->u32RecordRemainingTime) --- %d", (prMsg->u32RecordRemainingTime)));

      /* Copy record info, this is required when seek operation is initiated */
      (_tRecordInfo).u32RecordDuration      = (prMsg->u32RecordDuration);                          /* Total record duration */
      (_tRecordInfo).u32RecordElapsedTime   = (prMsg->u32RecordElapsedTime);                       /* Elapsed time in recorded buffer */
      (_tRecordInfo).u32RecordRemainingTime = (prMsg->u32RecordRemainingTime);                     /* Remaining time in recorded buffer */
      /* Update the same in instant replay property data as well */
      _tIROldStatus.oFiRes.InstantReplayStatus.RecordDuration = (_tRecordInfo).u32RecordDuration;     /* Copy buffer recorded time info */
      _tIROldStatus.oFiRes.InstantReplayStatus.TimeToLive = (_tRecordInfo).u32RecordRemainingTime;    /* Copy time to LIVE info */
      _tIROldStatus.oFiRes.InstantReplayStatus.ElapsedTime = (_tRecordInfo).u32RecordElapsedTime;	  /* Copy time from start of the buffer info*/

      Ifc_sxm_tclChannelList* poChannelList = (_poAudioApp) ?
    		  _poAudioApp->poGetChannelListInstance(): NULL;
      SXM_ASSERT_RETURN(NULL != poChannelList)

      /* Handling no signal general advisory here */
      if(((_tRecordInfo).u32RecordRemainingTime == 0) && (u32PreviousRecordRemainingTime != 0)) { 
         /* Cond 1: The moment audio from buffer is played completely & buffering is halted due to no signal available. */
    	  poChannelList->vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);

         /* In case of no signal condition we have to forcefully update time = 0 as the event is not triggered once again by sms */
         if(TRUE == (fc_sxm_tclAdvisories::instance())->bIsHighPriorityAdvisory()) {
            _eIRStatusControl.bIgnoreIRUpdate = FALSE;
            ETG_TRACE_USR4(("_bUpdateIRStatus ACTIVATED @ PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO/bIsHighPriorityAdvisory"));
            _bUpdateIRStatus = TRUE;
         }
      } else if( (u32PreviousRecordRemainingTime == 0) && ((_tRecordInfo).u32RecordRemainingTime != 0)) {
         /* Cond 2: In no signal if user plays audio from available buffer */
    	  poChannelList->vNotifyChannelInformationUpdate(e_ADVISORY_UPDATE);
      } 

      if(((_tIROldStatus.oFiRes.InstantReplayStatus.RecordDuration != _tIRStatus.oFiRes.InstantReplayStatus.RecordDuration) && (!_mIRSkipTimer.bIsRunning()))|| 
         (_tIROldStatus.oFiRes.InstantReplayStatus.TimeToLive != _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive) ||
         (_tIROldStatus.oFiRes.InstantReplayStatus.ElapsedTime != _tIRStatus.oFiRes.InstantReplayStatus.ElapsedTime))
      {
         _tIRStatus.oFiRes.InstantReplayStatus.RecordDuration = _tIROldStatus.oFiRes.InstantReplayStatus.RecordDuration;
         _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive = _tIROldStatus.oFiRes.InstantReplayStatus.TimeToLive;
         _tIRStatus.oFiRes.InstantReplayStatus.ElapsedTime = _tIROldStatus.oFiRes.InstantReplayStatus.ElapsedTime;
         ETG_TRACE_USR4(("_bUpdateIRStatus ACTIVATED @ PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO/RecordDur, Time to Live, Elapsed Time"));
         _bUpdateIRStatus = TRUE;
      }
   }

   if(PLAYBACK_OBJECT_EVENT_PLAY_PERCENTAGE & prMsg->ePlaybackEventMask) {
	   ETG_TRACE_USR3(("(prMsg->u8PlayPercentage) --- %d", (prMsg->u8PlayPercentage)));

	   (_tRecordInfo).u8PlayPercentage = (prMsg->u8PlayPercentage);								   	  /* Percentage of buffer at which the current play point is present */
	   _tIROldStatus.oFiRes.InstantReplayStatus.PlayPercentage = (_tRecordInfo).u8PlayPercentage; 	  /* Copy Play percentage*/

	   if (_tIROldStatus.oFiRes.InstantReplayStatus.PlayPercentage != _tIRStatus.oFiRes.InstantReplayStatus.PlayPercentage){
		   _tIRStatus.oFiRes.InstantReplayStatus.PlayPercentage = _tIROldStatus.oFiRes.InstantReplayStatus.PlayPercentage;
		   ETG_TRACE_USR4(("_bUpdateIRStatus ACTIVATED @ PLAYBACK_OBJECT_EVENT_PLAY_PERCENTAGE/PlayPercentage"));
		   _bUpdateIRStatus = TRUE;
	   }
   }

   if(PLAYBACK_OBJECT_EVENT_FILL_PERCENTAGE & prMsg->ePlaybackEventMask) {
	   ETG_TRACE_USR3(("(prMsg->u8FillPercentage) --- %d", (prMsg->u8FillPercentage)));

	   (_tRecordInfo).u8FillPercentage = (prMsg->u8FillPercentage);								   	  /* Percentage of buffer that currently contains stored data */
	   _tIROldStatus.oFiRes.InstantReplayStatus.FillPercentage = (_tRecordInfo).u8FillPercentage;	  /* Copy Fill Percenytage*/

	   if (_tIROldStatus.oFiRes.InstantReplayStatus.FillPercentage != _tIRStatus.oFiRes.InstantReplayStatus.FillPercentage){
		   _tIRStatus.oFiRes.InstantReplayStatus.FillPercentage = _tIROldStatus.oFiRes.InstantReplayStatus.FillPercentage;
		   ETG_TRACE_USR4(("_bUpdateIRStatus ACTIVATED @ PLAYBACK_OBJECT_EVENT_FILL_PERCENTAGE/FillPercentage"));
		   _bUpdateIRStatus = TRUE;
	   }
   }

   ETG_TRACE_USR2(("_eIRStatusControl.bIgnoreIRUpdate - %u _bUpdateIRStatus = %u", _eIRStatusControl.bIgnoreIRUpdate, _bUpdateIRStatus));
   if(_eIRStatusControl.bIgnoreIRUpdate) {
      _eIRStatusControl.bIgnoreIRUpdate = FALSE;
   } else {
      if(_bUpdateIRStatus) {
    	  if (bPlaybackStatus(prMsg->u32RecordDuration, u32PreviousRecordDuration, prMsg->u32RecordElapsedTime, _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus, prMsg->s32TimeOffset)) {

			 if ( _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive != FC_SXM_FORGROUND_CHANNEL_BUFFER_MAX)
			 {
				 vHandleInstantReplayStatus();
			 }
    		  /* Notify all clients */
    		  fc_sxm_tclAudioProperties::instance()->oInstantReplayStatus.vSet(_tIRStatus);
    		  fc_sxm_tclAudioProperties::instance()->oInstantReplayStatus.vNotify();
    		  _bUpdateIRStatus = FALSE;
    		  ETG_TRACE_USR2(("FI RecordDuration -------- %d", _tIRStatus.oFiRes.InstantReplayStatus.RecordDuration));
    		  ETG_TRACE_USR2(("FI TimeToLive ------------ %d", _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive));
    		  ETG_TRACE_USR2(("FI PlayStatus ------------ %d", _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType));
    		  ETG_TRACE_USR2(("FI ElapsedTime ------------ %d", _tIRStatus.oFiRes.InstantReplayStatus.ElapsedTime));
    	  }
      }
      if(_eIRStatusControl.bHandleIRMode) {
    	  Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
    	               		  _poAudioApp->poGetPresetsInstance(): NULL;
    	  SXM_ASSERT_RETURN(NULL != poPresets)
    	  poPresets->vNotifyIRMode(_eIRStatusControl.bValidate, _eIRStatusControl.bSetMode);
         _eIRStatusControl.bHandleIRMode = FALSE;
      }
   }

   if(PLAYBACK_OBJECT_EVENT_TIME_FROM_TRACK_START & prMsg->ePlaybackEventMask) {
      (_tRecordInfo).u32TimeFromTrackStart = (prMsg->u32TimeFromTrackStart);                       /* Time from beginning of current track */
   }
   if(PLAYBACK_OBJECT_EVENT_TRACKS_BEFORE & prMsg->ePlaybackEventMask) { 
      (_tRecordInfo).u16TracksBefore = (prMsg->u16TracksBefore);                                   /* Total tracks played before current track */
   }
   if(PLAYBACK_OBJECT_EVENT_TRACKS_REMAINING & prMsg->ePlaybackEventMask) {
      (_tRecordInfo).u16TracksRemaining = (prMsg->u16TracksRemaining);                             /* Total tracks remaining after current track */
   }

   /* Disable this once development is complete */
   // _tRecordInfo.vPrintRecordInfo();
}
/*
 * This method checks the instant replay property events received from sms
 */
tBool fc_sxm_tclPlayback::bPlaybackStatus(const UN32 &u32RecordDuration, const tU32 &u32PreviousRecordDuration,
		const UN32 &u32ElapsedTime, const midw_ext_fi_tcl_e8_IRPlaybackStatus& status, const N32& s32TimeOffset) const
{
	tBool bRetVal = FALSE;

	// check if the current record duration is non-zero
	if (u32RecordDuration != 0u)
	{
		// No updates to be sent if PAUSE state and recordduration and previous record duration varies
		if (( midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PAUSE_STATE == status.enType) &&
				(0 == s32TimeOffset)) {
			bRetVal = FALSE;
		}
		// check if the previous record duration is non-zero
		else if ( 0 != u32PreviousRecordDuration) {
			bRetVal = TRUE;
		}
		//check if the play point is not at end of buffer
		else if ( 0 != u32ElapsedTime) {
			bRetVal = TRUE;
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclPlayback::bPlaybackStatus() bRetVal(%u) \t u32RecordDuration(%u) \t u32PreviousRecordDuration(%u) \t u32ElapsedTime(%u)",
			bRetVal, u32RecordDuration, u32PreviousRecordDuration, u32ElapsedTime));
	return bRetVal;
}

/***********************************************************************************************************************************
 * This method is implemented to handle TimeToLive values toggling and compares the TimeToLive values received from SMS to a
 * threshhold value. The threshhold is kept at a +1 or -1 difference from TimeToLive values.
 * if the Subtraction (ThresholdValue - TimeToLive) is found more than 1, then threshhold is readjusted to +1 or -1 difference
 * from TimeToLive Values Otherwise we consider ThresholdValue-1 as TimeToLive.
 * With this implementation, we are always sending RecordDurationand and TimeToLive value with a lag of 2 sec. until foreground buffer
 * is full (60 min) and user go to end of buffer.
 * *********************************************************************************************************************************/
tVoid fc_sxm_tclPlayback::vHandleInstantReplayStatus()
{
	if(_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive != _u32ThresholdValue) {
		if (1 != std::abs((int)_u32ThresholdValue - (int)_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive)) {
			   if(_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive > _u32ThresholdValue) {
				   _u32ThresholdValue = (_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive - 1);
			   	   } else {
			   		   _u32ThresholdValue = (_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive + 1);
			   	   }
		}
		_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive = _u32ThresholdValue;
	}
	_tIRStatus.oFiRes.InstantReplayStatus.RecordDuration = (_tIRStatus.oFiRes.InstantReplayStatus.RecordDuration - 2);
	_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive = (_tIRStatus.oFiRes.InstantReplayStatus.TimeToLive - 1);
}

tVoid fc_sxm_tclPlayback::vChangeSeekSpeed()
{
   /* Based on timeout count increase seek speed */
   if((FC_SXM_IR_SEEK_TIMEOUT_COUNT_1 < _tRecordInfo.u32TimeOutCount) && ((FC_SXM_IR_SEEK_TIMEOUT_COUNT_2 >= _tRecordInfo.u32TimeOutCount)))
   {
      _tRecordInfo.u32SeekTime = FC_SXM_IR_SEEK_SPEED_1;
   } else if ((FC_SXM_IR_SEEK_TIMEOUT_COUNT_2 < _tRecordInfo.u32TimeOutCount) && ((FC_SXM_IR_SEEK_TIMEOUT_COUNT_3 >= _tRecordInfo.u32TimeOutCount)))
   {
      _tRecordInfo.u32SeekTime = FC_SXM_IR_SEEK_SPEED_2;
   } else if(FC_SXM_IR_SEEK_TIMEOUT_COUNT_3 < _tRecordInfo.u32TimeOutCount)
   {
      _tRecordInfo.u32SeekTime = FC_SXM_IR_SEEK_SPEED_3;
      _tRecordInfo.u32TimeOutCount = FC_SXM_IR_SEEK_TIMEOUT_COUNT_3;
   }

   ETG_TRACE_USR4(("vCheckAndChangeSeekSpeed , _tRecordInfo.u32SeekTime=%d _tRecordInfo.u32TimeOutCount=%d",
            _tRecordInfo.u32SeekTime,
            _tRecordInfo.u32TimeOutCount));
}


tVoid fc_sxm_tclPlayback::vProcessTimer(fc_sxm_trMsgIRSeekTimer const * /* prMsg */ )
{
   _tRecordInfo.u32TimeOutCount++;
   _mIRSeekTimer.vStart(FC_SXM_IR_SEEK_TIME_DELAY);

   vChangeSeekSpeed();

   if( ((FC_SXM_IR_SEEK_TIMEOUT_COUNT_1 < _tRecordInfo.s32SeekDirection) && (FC_SXM_IR_SEEK_TIMEOUT_COUNT_1 == _tRecordInfo.u32RecordRemainingTime)) ||
       ((FC_SXM_IR_SEEK_TIMEOUT_COUNT_1 > _tRecordInfo.s32SeekDirection) && (_tRecordInfo.u32RecordDuration == _tRecordInfo.u32RecordRemainingTime)) )
   {
      /* HACK */
      _mIRSeekTimer.vStop();
      _tRecordInfo.vClear();
      _tIRStatus.oFiRes.InstantReplayStatus.PlaybackStatus.enType = midw_ext_fi_tcl_e8_IRPlaybackStatus::FI_EN_SXM_AUDIO_IR_PLAY_STATE; /* HACK */

      ETG_TRACE_USR4(("fc_sxm_trMsgIRSeekTimer , _tRecordInfo(s32SeekDirection,u32RecordRemainingTime, u32RecordDuration)= (%d,%d,%d)",
               _tRecordInfo.s32SeekDirection,
               _tRecordInfo.u32RecordRemainingTime,
               _tRecordInfo.u32RecordDuration));
   }
   else
   {
      tBool bPauseAfterSeek = FALSE;
      if(SMSAPI_RETURN_CODE_SUCCESS == PLAYBACK.eSeekTime(DECODER.hPlayback(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder()),
                                                        (_tRecordInfo.u32SeekTime * _tRecordInfo.s32SeekDirection),
                                                        bPauseAfterSeek))
      {
         ETG_TRACE_USR4(("PLAYBACK.eSeekTime() callback is not received from SMS, _tRecordInfo.u32TimeOutCount=%d",
                  _tRecordInfo.u32TimeOutCount));
      }
      else
      {
         ETG_TRACE_ERR(("PLAYBACK.eSeekTime() callback failed"));
      }
   }
}

tVoid fc_sxm_tclPlayback::vProcessTimer(fc_sxm_trMsgIRSkipTimer const * /* prMsg */ )
{
   ETG_TRACE_USR2(("fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgIRSkipTimer) with _s16IRSkipCount - %d", _s16IRSkipCount));

   if(0 < _s16IRSkipCount) {
      if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekNext(DECODER.hPlayback(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder()), FALSE)) {
         ETG_TRACE_ERR(("PLAYBACK.eSeekNext() failed"));
      } else {
         ETG_TRACE_USR2(("PLAYBACK.eSeekNext() successful !!!"));
         _mIRSkipTimer.vStart(FC_SXM_IR_SKIP_TIME_DELAY);
      }
   } else if(0 > _s16IRSkipCount) {
      if((_tRecordInfo.u32RecordDuration - 3) >= _tRecordInfo.u32RecordRemainingTime) {
         if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekPrevious(DECODER.hPlayback(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder()), FALSE)) {
            ETG_TRACE_ERR(("PLAYBACK.eSeekPrevious() failed"));
         } else {
            ETG_TRACE_USR2(("PLAYBACK.eSeekPrevious() successful !!!"));
            _mIRSkipTimer.vStart(FC_SXM_IR_SKIP_TIME_DELAY);
         }
      }
   } else {
      /* Do nothing */
   }
   _s16IRSkipCount = 0;
}

tBool fc_sxm_tclPlayback::bIsReplayModeActive(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclPlayback::bIsReplayModeActive()"));
   return((tBool)(_tRecordInfo.u32RecordRemainingTime));
}

tVoid fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioMuteState const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclPlayback::vProcess(fc_sxm_trMsgAudioMuteState)"));
    // The smart favorite buffer size for each channel is 60 mins i.e 3600 seconds
    const tU16 u16MAX_SMART_FAVORITE_BUFFER_SIZE_IN_SECONDS = 3600;  
    switch(prMsg->eSourceActivity)
    {
       case e_SXM_EN_OFF: 
       {
    	   Ifc_sxm_tclPresets* poPresets = (_poAudioApp) ?
    	                		  _poAudioApp->poGetPresetsInstance(): NULL;
    	   SXM_ASSERT_RETURN(NULL != poPresets)

          if(0 < _tIRStatus.oFiRes.InstantReplayStatus.TimeToLive) {
             tBool bPauseAfterSeek = FALSE;
             // When user changes to any other source, SXM should be in live mode irrespective of replay or live mode.
             if(SMSAPI_RETURN_CODE_ERROR == PLAYBACK.eSeekTime(DECODER.hPlayback(fc_sxm_tclSmsDecoder::instance()->hGetSmsDecoder()),
                                                               u16MAX_SMART_FAVORITE_BUFFER_SIZE_IN_SECONDS,
                                                               bPauseAfterSeek)) {
                ETG_TRACE_ERR(("PLAYBACK.eSeekTime() failed"));
             } 
          } else {
             ETG_TRACE_USR4(("LIVE data is being played"));
          }
    	   poPresets->vUpdateIRMode(FALSE, FALSE);
       }break;
       case e_SXM_EN_PAUSE: {
       } break;
       case e_SXM_EN_ON: {
       } break;
       default:
         break;
    }
}

tVoid fc_sxm_tclPlayback::vIgnoreIRUpdate(tBool bValidate, tBool bSetMode)
{
   ETG_TRACE_USR1(("fc_sxm_tclPlayback::vIgnoreIRUpdate()"));
   _eIRStatusControl.bIgnoreIRUpdate = TRUE;
   _eIRStatusControl.bHandleIRMode = TRUE;
   _eIRStatusControl.bValidate = bValidate;
   _eIRStatusControl.bSetMode = bSetMode;
   /* When new channel is selected, update IR status atleast once */
   _bUpdateIRStatus = TRUE;
}

/* Set Audio App Pointer Reference*/
tVoid fc_sxm_tclPlayback::vSetApp(Ifc_sxm_tclAudioApp* audioApp)
{
	ETG_TRACE_USR4(("fc_sxm_tclChannelList::vSetApp()"));
	_poAudioApp = audioApp;
}

tVoid fc_sxm_trRecordContentInfo::vPrintRecordInfo(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_trRecordContentInfo::vPrintData()"));
   ETG_TRACE_USR4(("u32TimeFromTrackStart ===================== %d", u32TimeFromTrackStart));
   ETG_TRACE_USR4(("u32RecordDuration ========================= %d", u32RecordDuration));
   ETG_TRACE_USR4(("u32RecordElapsedTime ====================== %d", u32RecordElapsedTime));
   ETG_TRACE_USR4(("u32RecordRemainingTime ==================== %d", u32RecordRemainingTime));
   ETG_TRACE_USR4(("bIsSeekActive ----------------------------- %d", bIsSeekActive));
   ETG_TRACE_USR4(("s32SeekDirection -------------------------- %d", s32SeekDirection));
   ETG_TRACE_USR4(("u32SeekTime ------------------------------- %d", u32SeekTime));
   ETG_TRACE_USR4(("u32TimeOutCount --------------------------- %d", u32TimeOutCount));
   ETG_TRACE_USR4(("u16TracksBefore *************************** %d", u16TracksBefore));
   ETG_TRACE_USR4(("u16TracksRemaining ************************ %d", u16TracksRemaining));
}

tVoid fc_sxm_trMsgAudioSmsEvtPlaybackEvt::vPrint() const
{
	ETG_TRACE_USR4(("fc_sxm_trMsgAudioSmsEvtPlaybackEvt::vPrint()"));
	ETG_TRACE_USR4(("^^^^^^^ IsPlayActive(%u)           ^^^^^^^^", IsPlayActive));
	ETG_TRACE_USR4(("^^^^^^^ u16TracksBefore(%u)        ^^^^^^^^", u16TracksBefore));
	ETG_TRACE_USR4(("^^^^^^^ u16TracksRemaining(%u)     ^^^^^^^^", u16TracksRemaining));
	ETG_TRACE_USR4(("^^^^^^^ u32TimeFromTrackStart(%u)  ^^^^^^^^", u32TimeFromTrackStart));
	ETG_TRACE_USR4(("^^^^^^^ u32RecordDuration(%u)      ^^^^^^^^", u32RecordDuration));
	ETG_TRACE_USR4(("^^^^^^^ u32RecordElapsedTime(%u)   ^^^^^^^^", u32RecordElapsedTime));
	ETG_TRACE_USR4(("^^^^^^^ u32RecordRemainingTime(%u) ^^^^^^^^", u32RecordRemainingTime));
	ETG_TRACE_USR4(("^^^^^^^ u8PlayPercentage(%u)       ^^^^^^^^", u8PlayPercentage));
	ETG_TRACE_USR4(("^^^^^^^ u8FillPercentage(%u)       ^^^^^^^^", u8FillPercentage));
	ETG_TRACE_USR4(("^^^^^^^ s32TimeOffset(%d)          ^^^^^^^^", s32TimeOffset));
	ETG_TRACE_USR4(("^^^^^^^ u32WarningOffset(%u)       ^^^^^^^^", u32WarningOffset));
	ETG_TRACE_USR4(("^^^^^^^ ePlaybackEventMask(%x)       ^^^^^^^^", ePlaybackEventMask));
}
