/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_content_alerts.cpp
* @brief       Artist and song content alerts functionality implementation.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_audio_app_if.h"
#include "fc_sxm_tcl_content_alerts.h"
#include "fc_sxm_tcl_audio_properties.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AUDIO_CONTENT_ALERT
#include "trcGenProj/Header/fc_sxm_tcl_content_alerts.cpp.trc.h"
#endif

/**
 * Default CTOR
 */
fc_sxm_tclContentAlert::fc_sxm_tclContentAlert()
         : _poAudioApp(OSAL_NULL),
           _poContentAlertProxy(OSAL_NULL),
           _bIsSeekServiceInitialized(FALSE),
           _bIsArtistAlertBlocked(FALSE),
           _bIsSongAlertBlocked(FALSE),
           _u8RegisteredArtistCount(0),
           _u8RegisteredSongCount(0),
           _hSeekCategoryID(CATEGORY_INVALID_ID)
{
   _vecArtistList.clear();
   _vecSongList.clear();
}

/**
 * Default DTOR
 */
fc_sxm_tclContentAlert::~fc_sxm_tclContentAlert()
{
   _poAudioApp = OSAL_NULL;
   _poContentAlertProxy = OSAL_NULL;
   _bIsSeekServiceInitialized = FALSE;
   _bIsArtistAlertBlocked = FALSE;
   _bIsSongAlertBlocked = FALSE;
   _u8RegisteredArtistCount = 0;
   _u8RegisteredSongCount = 0;
   _hSeekCategoryID = CATEGORY_INVALID_ID;
   _vecArtistList.clear();
   _vecSongList.clear();
}

/**
 * Clear all member data variables
 */
tVoid fc_sxm_tclContentAlert::vClearMemberData(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclContentAlert::vClearMemberData"));
   DECODER_OBJECT hDecoder = hGetDecoder();
   if((_bIsSeekServiceInitialized) && (_poContentAlertProxy) && (hDecoder))
      _poContentAlertProxy->vStop(hDecoder);

   _bIsSeekServiceInitialized = FALSE;
   _bIsArtistAlertBlocked = FALSE;
   _bIsSongAlertBlocked = FALSE;
   _u8RegisteredArtistCount = 0;
   _u8RegisteredSongCount = 0;
   _hSeekCategoryID = CATEGORY_INVALID_ID;
   _vecArtistList.clear();
   _vecSongList.clear();
}

/**
 * Set external application to content alert class
 * @param[in] IAudioApp  provide the audio app interfaces
 * @param[in] poContentAlertProxy provide the content alert proxy interfaces
 */
tVoid fc_sxm_tclContentAlert::vSetApp(Ifc_sxm_tclAudioApp* audioApp, Ifc_sxm_tclContentAlertProxy* poContentAlertProxy)
{
   _poAudioApp = audioApp;
   _poContentAlertProxy = poContentAlertProxy;
}

/**
 * Get Decoder Object
 * @return decoder object
 */
DECODER_OBJECT fc_sxm_tclContentAlert::hGetDecoder() const
{
   return (((_poAudioApp) ? _poAudioApp->oGetDecoderObject() : DECODER_INVALID_OBJECT));
}

/**
 * Send the content alert fi message i.e method result
 * @param[in] rAddr
 * @param[in] oMessage
 */
tVoid fc_sxm_tclContentAlert::vSendContentAlertFiMsg(fc_sxm_trAdressing const &rAddr,
   const midw_ext_sxm_audiofi_tclMsgBaseMessage& oMessage)
{
   if(OSAL_NULL != _poAudioApp)
      _poAudioApp->vSendAudioFiMsg(rAddr, oMessage);
}

/**
 * Send the content alert fi error message i.e method error
 * @param[in] rAddr
 * @param[in] s32ErrorMsg
 */
tVoid fc_sxm_tclContentAlert::vSendContentAlertFiError(fc_sxm_trAdressing const &rAddr, tInt s32ErrorMsg)
{
   if(OSAL_NULL != _poAudioApp)
      _poAudioApp->vSendAudioErrorMsg(rAddr, s32ErrorMsg);
}

/**
 * Update decoder state
 * @brief When decoder state ready is received then start the SEEK service and update the existing registered favorite
 *        count.
 * @param[in] hDecoder decoder object
 * @param[in] enDecoderState decoder state
 */
tVoid fc_sxm_tclContentAlert::vUpdateDecoderState(DECODER_OBJECT hDecoder, DECODER_STATE_ENUM enDecoderState)
{
   if((FALSE == _bIsSeekServiceInitialized) && (DECODER_INVALID_OBJECT != hDecoder)
      && (DECODER_STATE_READY == enDecoderState) && (_poContentAlertProxy))
   {
      _bIsSeekServiceInitialized = TRUE;
      SMSAPI_RETURN_CODE_ENUM enReturnCode = _poContentAlertProxy->enStart(hDecoder);
      ETG_TRACE_USR4(("Start Seek for Artist & Title enReturnCode = %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));

      _hSeekCategoryID = _poContentAlertProxy->hGetSeekCategoryId(hDecoder);

      _poContentAlertProxy->enEnableAlertsTuned(hDecoder, TRUE);
      _poContentAlertProxy->enEnableAlertsFinished(hDecoder, TRUE);

      _poContentAlertProxy->vIterateFavorites(hDecoder, fc_sxm_tclContentAlert::cb_bGetFavoriteCount, OSAL_NULL);

      tBool bAlertsFinishedStatus = _poContentAlertProxy->bGetAlertsFinishedStatus(hDecoder);
      ETG_TRACE_USR4(("Favorite Count (Artist, %d) and (Song, %d), AlertFinishedStatus=%d",
               _u8RegisteredArtistCount,
               _u8RegisteredSongCount,
               bAlertsFinishedStatus));
   }
}

/**
 * Increment favorite count if it is successfully register
 * @param[in] enFavoriteType type of the favorite either Artist, Title or Unknown
 */
tVoid fc_sxm_tclContentAlert::vIncrementFavoriteCount(AT_SEEK_ENUM enFavoriteType)
{
   if(AT_SEEK_ARTIST == enFavoriteType)
   {
      ++_u8RegisteredArtistCount;
   }
   else if(AT_SEEK_TITLE == enFavoriteType)
   {
      ++_u8RegisteredSongCount;
   }
}

/**
 * Decrement favorite count if it is deleted from favorite list
 * @param[in] enFavoriteType type of the favorite either Artist, Title or Unknown
 */
tVoid fc_sxm_tclContentAlert::vDecrementFavoriteCount(AT_SEEK_ENUM enFavoriteType)
{
   if(AT_SEEK_ARTIST == enFavoriteType)
   {
      --_u8RegisteredArtistCount;
   }
   else if(AT_SEEK_TITLE == enFavoriteType)
   {
      --_u8RegisteredSongCount;
   }
}

/**
 * Check for favorite list is full or not
 * @param[in] enFavoriteType type of the favorite either Artist, Title or Unknown
 * @param[in] u8MaxFavoriteCount maximum number of favorite
 * @return TRUE when favorite list is full else FALSE
 */
tBool fc_sxm_tclContentAlert::bIsFavoriteListFull(AT_SEEK_ENUM enFavoriteType, tU8 u8MaxFavoriteCount) const
{
   tU8 u8RegisteredCount = (AT_SEEK_ARTIST == enFavoriteType) ? _u8RegisteredArtistCount : _u8RegisteredSongCount;
   return (static_cast <tBool>(u8RegisteredCount >= u8MaxFavoriteCount));
}

/**
 * Check for favorite list is empty not
 * @param[in] enFavoriteType type of the favorite either Artist, Title or Unknown
 * @return TRUE when favorite list is empty else FALSE
 */
tBool fc_sxm_tclContentAlert::bIsFavoriteListEmpty(AT_SEEK_ENUM enFavoriteType) const
{
   tU8 u8RegisteredCount = (AT_SEEK_ARTIST == enFavoriteType) ? _u8RegisteredArtistCount : _u8RegisteredSongCount;
   return (!(static_cast <tBool>(u8RegisteredCount)));
}

/**
 * Check for the valid favorite type (ARTIST or TITLE)
 * @param[in] enFavoriteType type of the favorite
 * @return TRUE when favorite type is valid i.e ARTIST or TITLE else FALSE
 */
tBool fc_sxm_tclContentAlert::bIsFavoriteTypeValid(AT_SEEK_ENUM enFavoriteType) const
{
   tBool bResult = static_cast <tBool>((AT_SEEK_ARTIST == enFavoriteType) || (AT_SEEK_TITLE == enFavoriteType));
   return bResult;
}

/**
 * Check for modify favorite request
 * @param oModifyFavoriteRequest contain modify favorite request
 * @return TRUE when modify request is valid else FALSE
 */
tBool fc_sxm_tclContentAlert::bIsModifyFavoriteRequestValid(const fc_sxm_trMsgAudioModifyFavorite &oModifyFavoriteRequest) const
{
   tBool bResult = FALSE;
   if(bIsFavoriteTypeValid(oModifyFavoriteRequest.enFavoriteType))
   {
      if(!oModifyFavoriteRequest.sContentName.empty())
      {
         if((midw_ext_fi_tcl_e8_Action::FI_EN_ENABLE == oModifyFavoriteRequest.enAction)||
            (midw_ext_fi_tcl_e8_Action::FI_EN_DISABLE == oModifyFavoriteRequest.enAction)||
            (midw_ext_fi_tcl_e8_Action::FI_EN_DELETE == oModifyFavoriteRequest.enAction))
         {
            bResult = TRUE;
         }
      }
      else
      {
         if((midw_ext_fi_tcl_e8_Action::FI_EN_BLOCK == oModifyFavoriteRequest.enAction)||
            (midw_ext_fi_tcl_e8_Action::FI_EN_UNBLOCK == oModifyFavoriteRequest.enAction)||
            (midw_ext_fi_tcl_e8_Action::FI_EN_ENABLE_ALL == oModifyFavoriteRequest.enAction)||
            (midw_ext_fi_tcl_e8_Action::FI_EN_DISABLE_ALL == oModifyFavoriteRequest.enAction)||
            (midw_ext_fi_tcl_e8_Action::FI_EN_DELETE_ALL == oModifyFavoriteRequest.enAction))
         {
            bResult = TRUE;
         }
      }
   }
   return bResult;
}

/**
 * Check the alert type is blocked or not
 * @param enFavoriteType favorite type
 * @return TRUE when alert type is blocked else FALSE
 */
tBool fc_sxm_tclContentAlert::bIsFavoriteAlertBlocked(midw_ext_fi_tcl_e8_FavoriteType::tenType enFavoriteType) const
{
   tBool bIsArtistAlertBlocked = (_bIsArtistAlertBlocked) &&
            (midw_ext_fi_tcl_e8_FavoriteType::FI_EN_FAVORITE_ARTIST == enFavoriteType);
   tBool bIsSongAlertBlocked = (_bIsSongAlertBlocked) &&
            (midw_ext_fi_tcl_e8_FavoriteType::FI_EN_FAVORITE_SONG == enFavoriteType);
   return  (bIsArtistAlertBlocked || bIsSongAlertBlocked);
}

/**
 * Checkin the alert channel info for match
 * @param[in] oNotifiedAlert
 * @param[in] oNewAlert
 * @return TRUE if channel id ans service get matched else FALSE
 */
bool fc_sxm_tclContentAlert::bIsChannelInfoMatched(const midw_ext_fi_tcl_AlertInfo& oNotifiedAlert,
   const midw_ext_fi_tcl_AlertInfo& oNewAlert) const
{
   return ((oNotifiedAlert.ServiceID  == oNewAlert.ServiceID) &&
            (oNotifiedAlert.ChannelID  == oNewAlert.ChannelID));
}

/**
 * Checkin the alert song info for match with already notified alert
 * @param[in] oNotifiedAlert
 * @param[in] oNewAlert
 * @return TRUE when artist or song name get matched with new alert else FALSE
 */
bool fc_sxm_tclContentAlert::bIsSongInfoMatched(const midw_ext_fi_tcl_AlertInfo& oNotifiedAlert,
   const midw_ext_fi_tcl_AlertInfo& oNewAlert) const
{
   bool bResult = false;
   AT_SEEK_ENUM enFavoriteType = (AT_SEEK_ENUM)oNewAlert.FavoriteType.enType;
   switch(enFavoriteType)
   {
      case AT_SEEK_ARTIST:
      {
         tString chNotifiedArtistName= oNotifiedAlert.ArtistName.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
         string sNotifiedArtistName(chNotifiedArtistName);
         tString chNewArtistName = oNewAlert.ArtistName.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
         string sNewArtistName(chNewArtistName);
         bResult = (sNotifiedArtistName == sNewArtistName);

         OSAL_DELETE []chNotifiedArtistName;
         OSAL_DELETE []chNewArtistName;
         break;
      }
      case AT_SEEK_TITLE:
      {
         tString chNotifiedSongName = oNotifiedAlert.SongName.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
         string sNotifiedSongName(chNotifiedSongName);
         tString chNewSongName = oNewAlert.SongName.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
         string sNewSongName(chNewSongName);
         bResult = (sNotifiedSongName == sNewSongName);

         OSAL_DELETE []chNotifiedSongName;
         OSAL_DELETE []chNewSongName;
         break;
      }
      default:
      {
         ETG_TRACE_ERR(("Alert type is invalid"));
         break;
      }
   }
   return bResult;
}

/**
 * Check the alert is already notified or not as SMS is giving complete list again in new Callback for New alerts
 * @param[in] oAlert contain all alert information
 * @return TRUE when it is already notified else FALSE
 */
tBool fc_sxm_tclContentAlert::bIsAlertAlreadyNotified(const midw_ext_fi_tcl_AlertInfo& oAlert) const
{
   tBool bResult = FALSE;

   vector<midw_ext_fi_tcl_AlertInfo> vecActiveAlertList = ((AT_SEEK_ARTIST == ((AT_SEEK_ENUM)oAlert.FavoriteType.enType)) ?
            _vecArtistList :
            _vecSongList);

   for(vector<midw_ext_fi_tcl_AlertInfo>::iterator iter = vecActiveAlertList.begin();
            iter != vecActiveAlertList.end();
            ++iter)
   {
      bool bChannelInfoMatched = bIsChannelInfoMatched((*iter), oAlert);
      bool bSongInfoMatched = bIsSongInfoMatched((*iter), oAlert);
      if(bChannelInfoMatched && bSongInfoMatched)
      {
         ETG_TRACE_USR4(("bIsAlertAleradyNotified, ChannelID = %d, ServiceID = %d,  is already notified", oAlert.ChannelID, oAlert.ServiceID));
         bResult = TRUE;
         break;
      }
      else
      {
         ETG_TRACE_USR4(("bIsAlertAleradyNotified, ChannelID = %d, ServiceID = %d,  is new", oAlert.ChannelID, oAlert.ServiceID));
      }
   }
   return bResult;
}

/**
 * Check alert is on tuned channel or not
 * @param u16AlertChannelID channel if of the alert
 * @return true when Alert is on Tuned channel else false
 */
bool fc_sxm_tclContentAlert::bIsAlertOnTuned(tU16 u16TunedChannelID,
   tU16 u16AlertChannelID) const
{
   return ((u16AlertChannelID == u16TunedChannelID) ? true : false);
}

/**
 * Check alerts when Radio ID(0) channel is tuned
 * @return true when Alert is on Tuned channel else false
 */
bool fc_sxm_tclContentAlert::bIsRadioIDChannelTuned(tU16 u16TunedChannelID) const
{
   return ((u16TunedChannelID == 0) ?
            true :
            false);
}

/**
 * Check whether alert from active alert list is expired or not using "My Seek" category
 * @param hAlertChannelID[in] contains active alert channel id
 * @param vChannelList[in] contains list of  channel id's from my seek category
 * @return True if alert is expired i.e not found in the my seek category else false
 */
bool fc_sxm_tclContentAlert::bIsActiveAlertExpired(CHANNEL_ID hAlertChannelID,
   const vector<CHANNEL_ID>& vChannelList) const
{
   tBool bResult = TRUE;
   vector<CHANNEL_ID>::const_iterator iterChannelList = vChannelList.begin();
   for(; iterChannelList!= vChannelList.end(); ++iterChannelList)
   {
      CHANNEL_ID hChannelID = (*iterChannelList);
      // Compare alerts channel id with latest MySeek category channel list
      if(hChannelID == hAlertChannelID)
      {
         bResult = FALSE;
         break;
      }
   }
   return bResult;
}

/**
 * Set block alert flag
 * @param[in] enFavoriteType type of the favorite list to block the Alerts
 * @param[in] enAction Action to be taken favorite list either BLOCK or UNBLOCK
 */
tVoid fc_sxm_tclContentAlert::vSetBlockAlertsFlag(AT_SEEK_ENUM enFavoriteType,
   midw_ext_fi_tcl_e8_Action::tenType enAction)
{
   tBool bAction = (midw_ext_fi_tcl_e8_Action::FI_EN_BLOCK == enAction) ? TRUE : FALSE;
   if(AT_SEEK_ARTIST == enFavoriteType)
   {
      _bIsArtistAlertBlocked = bAction;
   }
   else if(AT_SEEK_TITLE == enFavoriteType)
   {
      _bIsSongAlertBlocked = bAction;
   }
   ETG_TRACE_USR4(("vSetBlockAlertsFlag, enFavoriteType = %d, isAlertBlocked = %d", ETG_CENUM(AT_SEEK_ENUM,
      enFavoriteType), bAction));
}

/**
 * Get favorite count by favorite type
 * @param[in] enFavoriteType type of the favorite
 * @return registered count
 */
tU8 fc_sxm_tclContentAlert::u8GetFavoriteCount(AT_SEEK_ENUM enFavoriteType) const
{
   tU8 u8RegisteredCount = 0xFF;
   if(AT_SEEK_ARTIST == enFavoriteType)
   {
      u8RegisteredCount = _u8RegisteredArtistCount;
   }
   else if(AT_SEEK_TITLE == enFavoriteType)
   {
      u8RegisteredCount = _u8RegisteredSongCount;
   }
   return u8RegisteredCount;
}

/**
 * Get String object for registered favorite
 * @param[in] hSeekContent seek content object
 * @param[in] enRegisteredContentType registered content type
 * @return STRING_OBJECT for Artist name or Song name
 */
STRING_OBJECT fc_sxm_tclContentAlert::hGetRegisteredContent(SEEK_CONTENT_OBJECT hSeekContent,
   AT_SEEK_ENUM enFavoriteType) const
{
   STRING_OBJECT hRegisteredContent = STRING_INVALID_OBJECT;
   if((_poContentAlertProxy) && (hSeekContent != SEEK_CONTENT_INVALID_OBJECT))
   {
      if(AT_SEEK_ARTIST == enFavoriteType)
      {
         hRegisteredContent = _poContentAlertProxy->hGetArtistName(hSeekContent);
      }
      else if(AT_SEEK_TITLE == enFavoriteType)
      {
         hRegisteredContent = _poContentAlertProxy->hGetSongName(hSeekContent);
      }
   }
   return hRegisteredContent;
}

/**
 * Get registered content text as string
 * @param[in] hSeekContent seek content object
 * @param[in] enRegisteredContentType registered content type
 * @return string for Artist name or Song name
 */
string fc_sxm_tclContentAlert::hGetRegisteredContentTextAsString(SEEK_CONTENT_OBJECT hSeekContent,
   AT_SEEK_ENUM enFavoriteType) const
{
   STRING_OBJECT hRegisteredContent = (SEEK_CONTENT_INVALID_OBJECT != hSeekContent) ?
            hGetRegisteredContent(hSeekContent, enFavoriteType) :
            STRING_INVALID_OBJECT;
   string sContentText =
            ((_poContentAlertProxy) && (STRING_INVALID_OBJECT != hRegisteredContent)) ?
                     _poContentAlertProxy->sGetSmsStringAsStl(hRegisteredContent) : "";
   return (sContentText);
}

/**
 * Get registered content text as fi string
 * @param[in] hSeekContent seek content object
 * @param[in] enRegisteredContentType registered content type
 * @return fistring for Artist name or Song name
 */
midw_ext_fi_tclString fc_sxm_tclContentAlert::hGetRegisteredContentTextAsFiString(SEEK_CONTENT_OBJECT hSeekContent,
   AT_SEEK_ENUM enRegisteredContentType) const
{
   STRING_OBJECT hRegisteredContent = (SEEK_CONTENT_INVALID_OBJECT != hSeekContent) ?
            hGetRegisteredContent(hSeekContent, enRegisteredContentType) :
            STRING_INVALID_OBJECT;
   midw_ext_fi_tclString hRegisteredText;
   if((_poContentAlertProxy) && (STRING_INVALID_OBJECT != hRegisteredContent))
      hRegisteredText = _poContentAlertProxy->hGetSmsStringAsFi(hRegisteredContent);
   return hRegisteredText;
}

/**
 * Get the alert information and fill it into output parameter
 * @param[in] hSeekEvent seek event object
 * @return filled alert details message
 */
midw_ext_fi_tcl_AlertInfo fc_sxm_tclContentAlert::oGetAlertInfo(SEEK_EVENT_OBJECT hSeekEvent) const
{
   midw_ext_fi_tcl_AlertInfo oAlertDetails;
   if((_poContentAlertProxy) && (SEEK_EVENT_INVALID_OBJECT != hSeekEvent))
   {
      oAlertDetails = _poContentAlertProxy->oFillAlertInfo(hSeekEvent);
   }
   return oAlertDetails;
}

/**
 * Get save song availability status
 * @param[in] enTitleAvailability title availability
 * @param[in] enArtistAvailability artist availability
 * @return favorite save status
 */
midw_ext_fi_tcl_e8_FavoriteStatus::tenType fc_sxm_tclContentAlert::enGetSongAvailabilityStatus(SEEK_AVAILABILTY_ENUM enTitleAvailability,
   SEEK_AVAILABILTY_ENUM enArtistAvailability) const
{
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enFavoriteSaveStatus =
            midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_SONG_INFO;

   if((SEEK_AVAILIBILITY_UNAVAILABLE == enTitleAvailability) && (SEEK_AVAILIBILITY_UNAVAILABLE == enArtistAvailability))
   {
      enFavoriteSaveStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_ARTIST_SONG_INFO;
   }
   else if(SEEK_AVAILIBILITY_ALREADY_REGISTERED == enTitleAvailability)
   {
      enFavoriteSaveStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_ALREADY_REGISTERED;
   }
   else if(SEEK_AVAILABILTY_AVAILABLE == enTitleAvailability)
   {
      enFavoriteSaveStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_REGISTERED;
   }
   return enFavoriteSaveStatus;
}

/**
 * Get save artist availability status
 * @param[in] enTitleAvailability title availability
 * @param[in] enArtistAvailability artist availability
 * @return favorite save status
 */
midw_ext_fi_tcl_e8_FavoriteStatus::tenType fc_sxm_tclContentAlert::enGetArtistAvailabilityStatus(SEEK_AVAILABILTY_ENUM enTitleAvailability,
   SEEK_AVAILABILTY_ENUM enArtistAvailability) const
{
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enFavoriteSaveStatus =
            midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_ARTIST_INFO;

   if((SEEK_AVAILIBILITY_UNAVAILABLE == enTitleAvailability) && (SEEK_AVAILIBILITY_UNAVAILABLE == enArtistAvailability))
   {
      enFavoriteSaveStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_ARTIST_SONG_INFO;
   }
   else if(SEEK_AVAILIBILITY_ALREADY_REGISTERED == enArtistAvailability)
   {
      enFavoriteSaveStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_ALREADY_REGISTERED;
   }
   else if(SEEK_AVAILABILTY_AVAILABLE == enArtistAvailability)
   {
      enFavoriteSaveStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_REGISTERED;
   }
   return enFavoriteSaveStatus;
}

/**
 * Get save favorite status
 * @param[in] hDecoder decoder object
 * @param[in] u16ChannelID channel id
 * @param[in] enFavoriteType type of the favorite
 * @param[in] u8MaxFavoriteCount maximum number of favorites
 * @return favorite save status
 */
midw_ext_fi_tcl_e8_FavoriteStatus::tenType fc_sxm_tclContentAlert::enGetSaveFavoriteStatus(DECODER_OBJECT hDecoder,
   tU16 u16ChannelID,
   AT_SEEK_ENUM enFavoriteType,
   tU8 u8MaxFavoriteCount) const
{
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enStatus = enCheckArtistSongInfoAvailability(hDecoder, u16ChannelID, enFavoriteType);
   tBool bIsFavListFull = bIsFavoriteListFull(enFavoriteType, u8MaxFavoriteCount);

   if(bIsFavListFull && (midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_ALREADY_REGISTERED != enStatus))
   {
      enStatus = midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_MAX_FAVORITES_REACHED;
   }

   ETG_TRACE_USR4(("enGetSaveFavoriteStatus, Type=%d IsFavoritListFull=%d enStatus=%d",
                     ETG_CENUM(AT_SEEK_ENUM, enFavoriteType),
                     bIsFavListFull,
                     ETG_CENUM(midw_ext_fi_tcl_e8_FavoriteStatus::tenType, enStatus)));

   vPrintContentStatusMessage(enStatus);
   return enStatus;
}

/**
 * Get channel graphics update information
 * @param u16ChannelID valid channel id
 * @return FI_EN_GRAPHICS_NEEDS_REFRESH when graphics is updated else FI_EN_GRAPHICS_NEEDS_DEFAULT
 */
midw_ext_fi_tcl_e8_Graphics::tenType fc_sxm_tclContentAlert::enGetChannelGraphicsUpdate(tU16 u16ChannelID) const
{
   midw_ext_fi_tcl_e8_Graphics::tenType enChannelGraphic = midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
   if(_poAudioApp)
   {
      const fc_sxm_trChannel* chnInfo = _poAudioApp->trGetChannel((CHANNEL_ID)u16ChannelID);
      enChannelGraphic =((OSAL_NULL != chnInfo) && (chnInfo->bIsChannelArtAvailable)) ?
                   midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_REFRESH  :
                   midw_ext_fi_tcl_e8_Graphics::FI_EN_GRAPHICS_NEEDS_DEFAULT;
   }
   return enChannelGraphic;
}

/**
 * Check and Get channel status
 * @param bIsMature[in] channel is mature or not
 * @param bIsLocked[in] channel is locked or not
 * @param bIsSkipped[in] channel is skipped or not
 * @return oChannelStatus contains channel status info for mature, skip, lock, and tune
 */

midw_ext_fi_tcl_b8_ChannelStatus fc_sxm_tclContentAlert::oGetChannelStatus(const tBool bIsMature,
   const tBool bIsLocked,
   const tBool bIsSkipped) const
{
   midw_ext_fi_tcl_b8_ChannelStatus oChannelStatus;

   if(bIsMature)
   {
      oChannelStatus.vSetCHANNEL_STATUS_MATURED(TRUE);
   }
   if(bIsLocked)
   {
      oChannelStatus.vSetCHANNEL_STATUS_LOCKED(TRUE);
   }
   if(bIsSkipped)
   {
      oChannelStatus.vSetCHANNEL_STATUS_SKIPPED(TRUE);
   }

   if((!bIsLocked) && (!bIsSkipped))
   {
      oChannelStatus.vSetCHANNEL_STATUS_TUNABLE(TRUE);
   }

   ETG_TRACE_USR4(("oGetChannelStatus = %d", oChannelStatus.u8Value));
   return oChannelStatus;
}

/**
 * Update favorites count for artist or song
 * @param[in] hSeekContent seek content object
 */
tVoid fc_sxm_tclContentAlert::vUpdateFavoritesCount(SEEK_CONTENT_OBJECT hSeekContent)
{
   AT_SEEK_ENUM enRegisteredContentType = ((_poContentAlertProxy) && (SEEK_CONTENT_INVALID_OBJECT != hSeekContent)) ?
            _poContentAlertProxy->enGetFavoriteType(hSeekContent) :
            AT_SEEK_UNKNOWN;
   vIncrementFavoriteCount(enRegisteredContentType);
}

/**
 * Check Artist and song info availability
 * @param[in] hDecoder decoder object
 * @param[in] u16ChannelID channel id
 * @param[in] enFavoriteType type of the favorite
 * @return favorite save status
 */
midw_ext_fi_tcl_e8_FavoriteStatus::tenType fc_sxm_tclContentAlert::enCheckArtistSongInfoAvailability(DECODER_OBJECT hDecoder,
   tU16 u16ChannelID,
   AT_SEEK_ENUM enFavoriteType) const
{
   midw_ext_fi_tcl_e8_FavoriteStatus::tenType enFavoriteSaveStatus =
            midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_ARTIST_SONG_INFO;
   if(_poContentAlertProxy)
   {
      SEEK_AVAILABILTY_ENUM enTitleAvailablity = SEEK_AVAILIBILITY_UNKNOWN;
      SEEK_AVAILABILTY_ENUM enArtistAvailablity = SEEK_AVAILIBILITY_UNKNOWN;

      _poContentAlertProxy->vIsAvailableToRegister(hDecoder, (CHANNEL_ID) u16ChannelID, &enTitleAvailablity, &enArtistAvailablity);

      if(AT_SEEK_ARTIST == enFavoriteType)
      {
         enFavoriteSaveStatus = enGetArtistAvailabilityStatus(enTitleAvailablity, enArtistAvailablity);
      }
      else if(AT_SEEK_TITLE == enFavoriteType)
      {
         enFavoriteSaveStatus = enGetSongAvailabilityStatus(enTitleAvailablity, enArtistAvailablity);
      }
   }
   return enFavoriteSaveStatus;
}

/**
 * Check for current registered content then modify favorite for requested
 * @param[in] hSeekContent seek content object
 * @param[in] prMsg contain the parsed modify favorite request
 * @return TRUE for continue to iterate else FALSE to stop iterating
 */
tBool fc_sxm_tclContentAlert::bCheckAndModifyFavorite(SEEK_CONTENT_OBJECT hSeekContent,
   fc_sxm_trMsgAudioModifyFavorite const *prMsg)
{
   tBool bContinueIterating = TRUE;
   if((_poContentAlertProxy) && (prMsg) && (SEEK_CONTENT_INVALID_OBJECT != hSeekContent))
   {
      AT_SEEK_ENUM enRegisteredContentType = _poContentAlertProxy->enGetFavoriteType(hSeekContent);
      // Check for registered content type with modify favorite request
      if(enRegisteredContentType == prMsg->enFavoriteType)
      {
         string sRegisteredContentName = "";
         SMSAPI_RETURN_CODE_ENUM enStatusCode = SMSAPI_RETURN_CODE_ERROR;
         // Check for modify favorite request is for "Selected Favorite" or "All Favorites"
         if(!prMsg->sContentName.empty())
         {
            sRegisteredContentName = hGetRegisteredContentTextAsString(hSeekContent, enRegisteredContentType);
            // Check for registered content with selected favorite
            if(sRegisteredContentName == prMsg->sContentName)
            {
               enStatusCode = enUpdateFavorite(hSeekContent, prMsg->enAction, enRegisteredContentType);
               bContinueIterating = FALSE;
            }
         }
         else
         {
            enStatusCode = enUpdateFavorite(hSeekContent, prMsg->enAction, enRegisteredContentType);
         }
         vPrintModifyFavoriteStatus(prMsg->enAction, enRegisteredContentType, sRegisteredContentName, enStatusCode);
      }
   }
   else
   {
      bContinueIterating = FALSE;
   }
   return bContinueIterating;
}

/**
 * Check and get favorite
 * @param[in] hSeekContent seek content object
 * @param[in,out] poGetFavorite to be filled with requested favorite
 * @return TRUE for continue to iterate else FALSE to stop iterating
 */
tBool fc_sxm_tclContentAlert::bCheckAndGetFavorite(SEEK_CONTENT_OBJECT hSeekContent,
   fc_sxm_trMsgAudioGetFavorite *poGetFavorite) const
{
   tBool bContinueIterating = FALSE;
   if((SEEK_CONTENT_INVALID_OBJECT != hSeekContent) && (poGetFavorite) && (_poContentAlertProxy))
   {
      midw_ext_fi_tcl_RegisteredFavorite oRegisteredFavorite;
      AT_SEEK_ENUM enRegisteredContentType = _poContentAlertProxy->enGetFavoriteType(hSeekContent);
      if(enRegisteredContentType == poGetFavorite->enFavoriteType)
      {
         oRegisteredFavorite.ContentName = hGetRegisteredContentTextAsFiString(hSeekContent, enRegisteredContentType);
         oRegisteredFavorite.IsEnabled = _poContentAlertProxy->bGetFavoriteStatus(hSeekContent);

         ETG_TRACE_USR4(("Registered Favorite (Type, IsEnabled, Name) = (%d, %d, %s)",
                  ETG_CENUM(AT_SEEK_ENUM, enRegisteredContentType),
                  oRegisteredFavorite.IsEnabled,
                  oRegisteredFavorite.ContentName.szValue));

         poGetFavorite->ofiFavoriteList.RegisteredFavoriteList.push_back(oRegisteredFavorite);
      }
      bContinueIterating = TRUE;
   }
   else
   {
      ETG_TRACE_USR4(("Invalid Seek Content Object"));
   }
   return bContinueIterating;
}

/**
 * Check and remove alert from active content alert list reported through Seek Alert Event
 * @param oChannelId[in]
 * @param oArtist[in]
 * @param oSong[in]
 */
tVoid fc_sxm_tclContentAlert::vCheckAndRemoveAlerts(CHANNEL_ID oChannelId,
   STRING_OBJECT oArtist,
   STRING_OBJECT oSong)
{
   if(_poAudioApp && _poContentAlertProxy)
   {
      string sArtist = _poContentAlertProxy->sGetSmsStringAsStl(oArtist);
      string sSong = _poContentAlertProxy->sGetSmsStringAsStl(oSong);
      if(bRemoveExpiredAlert(_vecArtistList, oChannelId, sArtist, sSong))
      {
         vNotifyAlertExpirationEvent(oChannelId, AT_SEEK_ARTIST, oArtist, oSong);
         ETG_TRACE_USR4(("vCheckAndRemoveAlerts, oChannelId=%d, sArtist=%s", oChannelId,sArtist.c_str()));

      }
      if(bRemoveExpiredAlert(_vecSongList, oChannelId, sArtist, sSong))
      {
         vNotifyAlertExpirationEvent(oChannelId, AT_SEEK_TITLE, oArtist, oSong);
         ETG_TRACE_USR4(("vCheckAndRemoveAlerts, oChannelId=%d, sSong=%s", oChannelId,sSong.c_str()));
      }
   }
}

/**
 * Check and remove the expired alert from active alert list using My Seek category
 * @param vecAlertList[in] current alert list
 * @param vChannelList[in] list of active alert channels
 * @return bAlertExpiredCount number of expired alerts
 */
N16 fc_sxm_tclContentAlert::u16RemoveExpiredAlerts(vector<midw_ext_fi_tcl_AlertInfo>& vecAlertList,
   const vector<CHANNEL_ID>& vChannelList) const
{
   N16 n16AlertExpiredCount = 0;
   vector<midw_ext_fi_tcl_AlertInfo>::iterator iterAlertList = vecAlertList.begin();
   // Check for Alert list for latest active alert channels
   while(iterAlertList!= vecAlertList.end())
   {
      CHANNEL_ID hAlertChannelID = (CHANNEL_ID) (*iterAlertList).ChannelID;
      if(bIsActiveAlertExpired(hAlertChannelID ,vChannelList))
      {
         iterAlertList = vecAlertList.erase(iterAlertList);
         ++n16AlertExpiredCount;
         ETG_TRACE_USR4(("u16CheckAndRemoveExpiredAlert, Alert is expired for hAlertChannelID=%d",
                  hAlertChannelID));
      }
      else
      {
         ++iterAlertList;
         ETG_TRACE_USR4(("u16CheckAndRemoveExpiredAlert, Alert is not expired for hAlertChannelID=%d",
                           hAlertChannelID));
      }
   }
   return n16AlertExpiredCount;
}

/**
 * Add new alert in active content alert list
 * @param oActiveContentAlertList
 */
tVoid fc_sxm_tclContentAlert::vAddToActiveContentAlertList(const midw_ext_fi_tcl_AlertInfo& oAlert,
   fc_sxm_trMsgAudioATAlerts &oActiveContentAlertList) const
{
   if(midw_ext_fi_tcl_e8_FavoriteType::FI_EN_FAVORITE_ARTIST == oAlert.FavoriteType.enType)
   {
      oActiveContentAlertList.oArtistAlerts.vecAlertList.push_back(oAlert);
      ETG_TRACE_USR4(("Artist alert added into ACAL"));
   }
   else if(midw_ext_fi_tcl_e8_FavoriteType::FI_EN_FAVORITE_SONG == oAlert.FavoriteType.enType)
   {
      oActiveContentAlertList.oSongAlerts.vecAlertList.push_back(oAlert);
      ETG_TRACE_USR4(("Song alert added into ACAL"));
   }
}

/**
 * Notify the individual alert
 * @param oAlert which contain the alert info
 */
tVoid fc_sxm_tclContentAlert::vNotifyIndividualAlert(const midw_ext_fi_tcl_AlertInfo& oAlert) const
{
   DECODER_OBJECT hDecoder = hGetDecoder();
   if((_poAudioApp) && (_poContentAlertProxy) &&
      (!bIsFavoriteAlertBlocked(oAlert.FavoriteType.enType)) &&
      (DECODER_INVALID_OBJECT != hDecoder))
   {
      tU16 u16TunedChannelID = (tU16) _poContentAlertProxy->hGetTunedChannel(hDecoder);

      // Added check to suppress alert on current tuned channel or on tuned Radio ID 0 channel. Bug ID NCG3D-10686
      bool bIsAlertOnTunedChannel = bIsAlertOnTuned(u16TunedChannelID, oAlert.ChannelID);
      bool bIsTunedChannelRadioID = bIsRadioIDChannelTuned(u16TunedChannelID);
      if((!bIsAlertOnTunedChannel) && (!bIsTunedChannelRadioID))
      {
         ETG_TRACE_USR2(("vNotifyIndividualAlert, bIsAlertOnTunedChannel=%d, bIsTunedChannelRadioID=%d",
                  bIsAlertOnTunedChannel,
                  bIsTunedChannelRadioID));
         fc_sxm_tcl_trFavoriteContentAlertInfo oAlertInfo;
         oAlertInfo.oFiMsg.AlertInfo = oAlert;
         _poAudioApp->vFavoriteContentAlertInfo(oAlertInfo);
         vPrintAlertInfo(oAlertInfo.oFiMsg.AlertInfo);
      }
   }
}

/**
 * Notify alert expiration event
 * @param[in] oChannelId
 * @param[in] enAlertType
 * @param[in] oArtist
 * @param[in] oSong
 */
tVoid fc_sxm_tclContentAlert::vNotifyAlertExpirationEvent(CHANNEL_ID oChannelId,
   AT_SEEK_ENUM enAlertType,
   STRING_OBJECT oArtist,
   STRING_OBJECT oSong) const
{
   if((_poContentAlertProxy) && (_poAudioApp))
   {
      fc_sxm_tcl_trFavoriteContentAlertInfo oAlertInfo;
      oAlertInfo.oFiMsg.AlertInfo.ChannelID = static_cast<tU16>(oChannelId);
      oAlertInfo.oFiMsg.AlertInfo.FavoriteType.enType =
               static_cast<midw_ext_fi_tcl_e8_FavoriteType::tenType>(enAlertType);
      oAlertInfo.oFiMsg.AlertInfo.ArtistName = _poContentAlertProxy->hGetSmsStringAsFi(oArtist);
      oAlertInfo.oFiMsg.AlertInfo.SongName = _poContentAlertProxy->hGetSmsStringAsFi(oSong);
      oAlertInfo.oFiMsg.AlertInfo.IsExpired = TRUE;
      _poAudioApp->vFavoriteContentAlertInfo(oAlertInfo);
      ETG_TRACE_USR4(("vNotifyAlertExpirationEvent, oChannelId=%d", oChannelId));
   }
}

/**
 * Notify active content alert list to HMI
 * @param enAlertType type of the active alert list
 */
tVoid fc_sxm_tclContentAlert::vNotifyActiveContentAlertList(AT_SEEK_ENUM enAlertType)
{
   if(_poAudioApp)
   {
      fc_sxm_tcl_trAlertNoticeListUpdate oAlertNoticeListUpdate;
      oAlertNoticeListUpdate.oFiMsg.FavoriteType.enType = midw_ext_fi_tcl_e8_FavoriteType::tenType(enAlertType);
      oAlertNoticeListUpdate.oFiMsg.UpdateAvailable = TRUE;
      _poAudioApp->vAlertListUpdateStatus(oAlertNoticeListUpdate);
   }
}

/**
 * Handle favorite deletion
 * @param[in] hSeekContent
 * @param[in] enFavoriteType
 * @return SMSAPI_RETURN_CODE_SUCCESS on successful deletion of favorite
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlert::enHandleFavoriteDeletion(SEEK_CONTENT_OBJECT hSeekContent,
   AT_SEEK_ENUM enFavoriteType)
{
   SMSAPI_RETURN_CODE_ENUM enStatusCode = SMSAPI_RETURN_CODE_ERROR;
   if(_poContentAlertProxy && (SEEK_CONTENT_INVALID_OBJECT != hSeekContent))
   {
      if(AT_SEEK_ARTIST == enFavoriteType)
      {
         STRING_OBJECT hArtistName = _poContentAlertProxy->hGetArtistName(hSeekContent);
         string sArtistName = _poContentAlertProxy->sGetSmsStringAsStl(hArtistName);
         vRemoveAlertIfFavoriteDeleted(_vecArtistList, enFavoriteType, sArtistName);
      }
      else if(AT_SEEK_TITLE == enFavoriteType)
      {
         STRING_OBJECT hSongName = _poContentAlertProxy->hGetSongName(hSeekContent);
         string sSongName = _poContentAlertProxy->sGetSmsStringAsStl(hSongName);
         vRemoveAlertIfFavoriteDeleted(_vecSongList, enFavoriteType, sSongName);
      }

      enStatusCode = _poContentAlertProxy->enDeleteFavorite(hSeekContent);
      if(SMSAPI_RETURN_CODE_SUCCESS == enStatusCode) vDecrementFavoriteCount(enFavoriteType);
   }
   return enStatusCode;
}

/**
 * Remove the alert from active list if selecetd or all favorites get deleted from favorite list
 * @param[inout] vecAlertList
 * @param[in] enFavoriteType
 * @param[in] sDeleteContentName
 */
void fc_sxm_tclContentAlert::vRemoveAlertIfFavoriteDeleted(vector<midw_ext_fi_tcl_AlertInfo>& vecAlertList,
   AT_SEEK_ENUM enFavoriteType,
   string sDeleteContentName) const
{
   vector<midw_ext_fi_tcl_AlertInfo>::iterator iterAlert = vecAlertList.begin();
   for(; iterAlert != vecAlertList.end(); ++iterAlert)
   {

      tString chNotifiedContentName = (AT_SEEK_ARTIST ==enFavoriteType) ?
               ((*iterAlert).ArtistName.szGet(midw_ext_fi_tclString::FI_EN_UTF8)) :
               ((*iterAlert).SongName.szGet(midw_ext_fi_tclString::FI_EN_UTF8));
      string sNotifiedContentName(chNotifiedContentName);
      if(sNotifiedContentName == sDeleteContentName)
      {
         vecAlertList.erase(iterAlert);
         ETG_TRACE_USR4(("vRemoveAlertIfFavoriteDeleted, Alert (Type, Name)=(%d, %s)",
                  ETG_CENUM(AT_SEEK_ENUM, enFavoriteType),
                  sDeleteContentName.c_str()));
         break;
      }
      OSAL_DELETE []chNotifiedContentName;
   }
}

/**
 * Check and remove the expired alert from  active alert list
 * @param vecActiveAlertList[inout] current alert list
 * @param hExpiredChannelID
 * @param sExpiredArtistName
 * @param sExpiredSongName
 * @return TRUE if expired alert get successfully removed from active alert list else FALSE
 */
tBool fc_sxm_tclContentAlert::bRemoveExpiredAlert(vector<midw_ext_fi_tcl_AlertInfo>& vecActiveAlertList,
   CHANNEL_ID hExpiredChannelID,
   string sExpiredArtistName,
   string sExpiredSongName) const
{
   tBool bResult = FALSE;
   vector<midw_ext_fi_tcl_AlertInfo>::iterator iterAlert = vecActiveAlertList.begin();
   for(; iterAlert != vecActiveAlertList.end(); ++iterAlert)
   {
      CHANNEL_ID hNotifiedChannelID = (CHANNEL_ID) (*iterAlert).ChannelID;
      tString chNotifiedArtistName = ((*iterAlert).ArtistName.szGet(midw_ext_fi_tclString::FI_EN_UTF8));
      string sNotifiedArtistName(chNotifiedArtistName);

      tString chNotifiedSongName= ((*iterAlert).SongName.szGet(midw_ext_fi_tclString::FI_EN_UTF8));
      string sNotifiedSongName(chNotifiedSongName);
	     OSAL_DELETE []chNotifiedArtistName;
         OSAL_DELETE []chNotifiedSongName;

      if((hNotifiedChannelID == hExpiredChannelID) &&
         (sNotifiedArtistName == sExpiredArtistName) &&
         (sNotifiedSongName == sExpiredSongName))
      {
         vecActiveAlertList.erase(iterAlert);
         bResult = TRUE;
         break;
      }
   }
   return bResult;
}

/**
 * Update existing alert list
 * @param[in] bIsAlertBlocked flag to indicate alert is blocked or not
 * @param[in] oAlerts contains the alerts list detail
 */
tVoid fc_sxm_tclContentAlert::vUpdateAlertList(tBool bIsAlertBlocked,
   const fc_sxm_trMsgAudioATAlertList& oAlerts)
{
   if((!bIsAlertBlocked))
   {
      if(AT_SEEK_ARTIST == oAlerts.enFavoriteType)
      {
         std::copy(oAlerts.vecAlertList.begin(), oAlerts.vecAlertList.end(), back_inserter(_vecArtistList));
         vNotifyActiveContentAlertList(oAlerts.enFavoriteType);
         ETG_TRACE_USR4(("Active Artist Alert Notice List (%d):", _vecArtistList.size()));
         vPrintAlertList(_vecArtistList);
      }
      else if(AT_SEEK_TITLE == oAlerts.enFavoriteType)
      {
         std::copy(oAlerts.vecAlertList.begin(), oAlerts.vecAlertList.end(), back_inserter(_vecSongList));
         vNotifyActiveContentAlertList(oAlerts.enFavoriteType);
         ETG_TRACE_USR4(("Active Song Alert Notice List (%d):", _vecSongList.size()));
         vPrintAlertList(_vecSongList);
      }
   }
   else
   {
      ETG_TRACE_USR4(("%d alerts are blocked", ETG_CENUM(AT_SEEK_ENUM, oAlerts.enFavoriteType)));
   }
}


/**
 * Update registered favorite for enable, disable and delete
 * @param[in] hSeekContent seek content object
 * @param[in] enAction which is to be performed on favorite
 * @param[in] enFavoriteType type of the favorite
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlert::enUpdateFavorite(SEEK_CONTENT_OBJECT hSeekContent,
   midw_ext_fi_tcl_e8_Action::tenType enAction,
   AT_SEEK_ENUM enFavoriteType)
{
   SMSAPI_RETURN_CODE_ENUM enStatusCode = SMSAPI_RETURN_CODE_ERROR;  // Initialized to avoid prio1 warning
   if(_poContentAlertProxy && (SEEK_CONTENT_INVALID_OBJECT != hSeekContent))
   {
      switch(enAction)
      {
      case midw_ext_fi_tcl_e8_Action::FI_EN_ENABLE:
      case midw_ext_fi_tcl_e8_Action::FI_EN_ENABLE_ALL:
         enStatusCode = _poContentAlertProxy->enEnableFavorite(hSeekContent);
         break;
      case midw_ext_fi_tcl_e8_Action::FI_EN_DISABLE:
      case midw_ext_fi_tcl_e8_Action::FI_EN_DISABLE_ALL:
         enStatusCode = _poContentAlertProxy->enDisableFavorite(hSeekContent);
         break;
      case midw_ext_fi_tcl_e8_Action::FI_EN_DELETE:
      case midw_ext_fi_tcl_e8_Action::FI_EN_DELETE_ALL:
            enStatusCode = enHandleFavoriteDeletion(hSeekContent, enFavoriteType);
         break;
      default:
         enStatusCode = SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED;
         ETG_TRACE_ERR(("Action is incorrect, enAction=%d", ETG_CENUM(AT_SEEK_ENUM, enAction)));
         break;
      }
   }
   return enStatusCode;
}

/**
 * Register favorite
 * @param[in] hDecoder decoder object
 * @param[in] enFavoriteType favorite type of requested register favorite
 * @param[in] u16ChannelID channel id for register favorite is requested
 * @param[out] fiRegsiteredContentName return the regiseterd content name
 * @return TRUE if register favorite is successful else FALSE
 */
tBool fc_sxm_tclContentAlert::bRegisterFavorite(DECODER_OBJECT hDecoder,
   AT_SEEK_ENUM enFavoriteType,
   tU16 u16ChannelID)
{
   tBool bResult = FALSE;
   SEEK_CONTENT_OBJECT hSeekContent = SEEK_CONTENT_INVALID_OBJECT;
   SMSAPI_RETURN_CODE_ENUM enReturnCode = ((_poContentAlertProxy) && (DECODER_INVALID_OBJECT != hDecoder)) ?
            _poContentAlertProxy->enRegisterFavorite(hDecoder, enFavoriteType, (CHANNEL_ID) u16ChannelID, &hSeekContent) :
            SMSAPI_RETURN_CODE_ERROR;

   if(SMSAPI_RETURN_CODE_SUCCESS == enReturnCode)
   {
      bResult = TRUE;
      vIncrementFavoriteCount(enFavoriteType);
   }
   else
   {
      ETG_TRACE_ERR(("AT_SEEK.eRegister, type= %d enReturnCode = %d.", ETG_CENUM(AT_SEEK_ENUM, enFavoriteType),
               ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
   }
   return bResult;
}

/**
 * Save favorite
 * @param[in] hDecoder decoder object
 * @param[in] prMsg method start for save request
 * @param[out] poMResult method result for save request
 * @return TRUE if save request is processed successfully else FALSE
 */
tBool fc_sxm_tclContentAlert::bSaveFavorite(DECODER_OBJECT hDecoder,
   fc_sxm_trMsgAudioMStartSaveFavorite const *prMsg,
   midw_ext_sxm_audiofi_tclMsgSaveFavoriteMethodResult *poMResult)
{
   tBool bResult = TRUE;
   if((DECODER_INVALID_OBJECT != hDecoder) && (prMsg) && (_poAudioApp))
   {
      tU8 u8MaxFavoriteCount = (prMsg->oFiMsg).MaxFavoriteCount;
      AT_SEEK_ENUM enFavoriteType = (AT_SEEK_ENUM) prMsg->oFiMsg.FavoriteType.enType;
      tU16 u16ChannelID = (prMsg->oFiMsg).ChannelID;
      poMResult->Status.enType = enGetSaveFavoriteStatus(hDecoder,
         u16ChannelID,
         enFavoriteType,
         u8MaxFavoriteCount);

      const fc_sxm_trChannel* chnInfo = _poAudioApp->trGetChannel(static_cast<tS16>(u16ChannelID));
      string strcontentName = "";
      if(chnInfo)
      {
         if(AT_SEEK_ARTIST == enFavoriteType)
            strcontentName = chnInfo->oChannelArtistName;
         else if(AT_SEEK_TITLE == enFavoriteType)
            strcontentName = chnInfo->oChannelSongName;
      }

      if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_REGISTERED == poMResult->Status.enType)
      {
         bResult = bRegisterFavorite(hDecoder, enFavoriteType, u16ChannelID);
         ETG_TRACE_USR4(("Content registered, type=%d count = %d, content name=%s",
               ETG_CENUM(AT_SEEK_ENUM, enFavoriteType),
               poMResult->FavoriteCount,
               poMResult->ContentName.szValue));
      }

      poMResult->ContentName.bSet(strcontentName.c_str(), midw_ext_fi_tclString::FI_EN_ISO8859_1M);
      poMResult->FavoriteType.enType = prMsg->oFiMsg.FavoriteType.enType;
      poMResult->FavoriteCount = u8GetFavoriteCount(enFavoriteType);
   }
   else
   {
      bResult = FALSE;
      ETG_TRACE_ERR(("Decoder object is invalid."));
   }
   return bResult;
}

/**
 * Get favorite list
 * @param[in] hDecoder decoder object
 * @param[in] prMsg method start for get favorite list
 * @param[out] poGetFavorite method result for get favorite list
 * @return TRUE if get favorite list is processed successfully else FALSE
 */
tBool fc_sxm_tclContentAlert::bGetFavoriteList(DECODER_OBJECT hDecoder,
   fc_sxm_trMsgAudioMStartGetFavoriteList const *prMsg,
   fc_sxm_trMsgAudioGetFavorite* poGetFavorite) const
{
   tBool bResult = FALSE;
   if((DECODER_INVALID_OBJECT != hDecoder) && (_poContentAlertProxy) && (prMsg) && (poGetFavorite))
   {
      poGetFavorite->enFavoriteType = (AT_SEEK_ENUM) prMsg->oFiMsg.FavoriteType.enType;
      tBool bIsValidGetFavListReq = bIsFavoriteTypeValid(poGetFavorite->enFavoriteType);
      tBool bIsFavListEmpty = bIsFavoriteListEmpty(poGetFavorite->enFavoriteType);
      if((!bIsFavListEmpty) && bIsValidGetFavListReq)
      {
         _poContentAlertProxy->vIterateFavorites(hDecoder, fc_sxm_tclContentAlert::cb_bGetFavoriteList, (void*) poGetFavorite);
         tU8 u8RegisteredFavoriteCount = u8GetFavoriteCount(poGetFavorite->enFavoriteType);
         vPrintFavoriteCount(poGetFavorite->enFavoriteType, u8RegisteredFavoriteCount);
         bResult = TRUE;
      }
      else if(bIsFavListEmpty && bIsValidGetFavListReq)
      {
         bResult = TRUE;
         ETG_TRACE_USR4(("%d Favorite list is empty.", ETG_CENUM(AT_SEEK_ENUM, poGetFavorite->enFavoriteType)));
      }
      else
      {
         ETG_TRACE_ERR(("Invalid favorite list requested, enFavoriteType = %d", poGetFavorite->enFavoriteType));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Decoder object is invalid."));
   }
   return bResult;
}

/**
 * Get active alert channel list
 * @param hChannel[in] alert channel object
 * @param oActiveAlerts[out] filled with alert channel list
 */
N16 fc_sxm_tclContentAlert::u16GetActiveAlertChannel(CHANNEL_OBJECT hChannel,
   fc_sxm_trMsgAudioActiveAlertChannelList *oActiveAlerts) const
{
   N16 u16ReturnValue = 0;
   if(_poContentAlertProxy && (CHANNEL_INVALID_OBJECT != hChannel))
   {
      oActiveAlerts->vecChannelList.push_back(_poContentAlertProxy->hGetChannelId(hChannel));
      --oActiveAlerts->u32Size;
      u16ReturnValue = (oActiveAlerts->u32Size == 0) ? 0 : 1;
   }
   return u16ReturnValue;
}

/**
 * Modify favorite for operation likes enable, disable and delete favorite
 * @param[in] hDecoder decoder object
 * @param[in] prMsg method start for modify favorite
 * @param[out] poGetFavorite method result for modify favorite
 * @return TRUE if modify favorite is processed successfully else FALSE
 */
tBool fc_sxm_tclContentAlert::bModifyFavorite(DECODER_OBJECT hDecoder,
   fc_sxm_trMsgAudioMStartModifyFavorite const *prMsg,
   midw_ext_sxm_audiofi_tclMsgModifyFavoriteMethodResult* poMResult)
{
   tBool bResult = FALSE;
   if((DECODER_INVALID_OBJECT != hDecoder) && (_poContentAlertProxy) && (prMsg))
   {
      fc_sxm_trMsgAudioModifyFavorite oModifyFavoriteRequest;
      oModifyFavoriteRequest.enFavoriteType = (AT_SEEK_ENUM) prMsg->oFiMsg.FavoriteType.enType;
      oModifyFavoriteRequest.enAction = prMsg->oFiMsg.Action.enType;
      tString chContentName = prMsg->oFiMsg.ContentName.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
      oModifyFavoriteRequest.sContentName = (string) chContentName;
      OSAL_DELETE []chContentName;

      if(bIsModifyFavoriteRequestValid(oModifyFavoriteRequest))
      {
         if((midw_ext_fi_tcl_e8_Action::FI_EN_BLOCK == oModifyFavoriteRequest.enAction) ||
            (midw_ext_fi_tcl_e8_Action::FI_EN_UNBLOCK == oModifyFavoriteRequest.enAction))
         {
            vSetBlockAlertsFlag(oModifyFavoriteRequest.enFavoriteType, oModifyFavoriteRequest.enAction);
         }
         else
         {
            _poContentAlertProxy->vIterateFavorites(hDecoder, fc_sxm_tclContentAlert::cb_bModifyFavorite, (void*) &oModifyFavoriteRequest);
         }

         bResult = TRUE;
         poMResult->FavoriteType.enType = prMsg->oFiMsg.FavoriteType.enType;
         poMResult->Action.enType = prMsg->oFiMsg.Action.enType;
         poMResult->ContentName = prMsg->oFiMsg.ContentName;
      }
      else
      {
         ETG_TRACE_ERR(("Modify favorite request is not valid."));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Decoder object is invalid."));
   }
   return bResult;
}

/**
 * Print favorite count
 * @param[in] enFavoriteType type of the favorite
 * @param[in] u8RegisteredCount favorite count
 */
tVoid fc_sxm_tclContentAlert::vPrintFavoriteCount(AT_SEEK_ENUM enFavoriteType, tU8 u8RegisteredCount) const
{
   ETG_TRACE_USR4(("%d Favorite count = %d", ETG_CENUM(AT_SEEK_ENUM, enFavoriteType), u8RegisteredCount));
}

/**
 * Print the content status
 * @param[in] enFavoriteSaveStatus
 */
tVoid fc_sxm_tclContentAlert::vPrintContentStatusMessage(midw_ext_fi_tcl_e8_FavoriteStatus::tenType enFavoriteSaveStatus) const
{
   if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_ARTIST_SONG_INFO == enFavoriteSaveStatus)
   {
      ETG_TRACE_USR4(("Content has no artist & song info"));
   }
   else if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_ARTIST_INFO == enFavoriteSaveStatus)
   {
      ETG_TRACE_USR4(("Content has no artist info"));
   }
   else if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_NO_SONG_INFO == enFavoriteSaveStatus)
   {
      ETG_TRACE_USR4(("Content has no song info"));
   }
   else if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_REGISTERED == enFavoriteSaveStatus)
   {
      ETG_TRACE_USR4(("Content is available to register"));
   }
   else if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_ALREADY_REGISTERED == enFavoriteSaveStatus)
   {
      ETG_TRACE_USR4(("Content is already registered"));
   }
   else if(midw_ext_fi_tcl_e8_FavoriteStatus::FI_EN_CONTENT_MAX_FAVORITES_REACHED == enFavoriteSaveStatus)
   {
      ETG_TRACE_USR4(("Favorite List is full"));
   }
   else
   {
      ETG_TRACE_USR4(("Content is invalid"));
   }
}

/**
 * Print the modify favorite status
 * @param[in] enAction to be performed on current favorite
 * @param[in] enRegisteredFavoriteType type of the current favorite
 * @param[in] sRegisteredContentName registered content name
 * @param[in] enStatusCode status of the modify favorite
 */
tVoid fc_sxm_tclContentAlert::vPrintModifyFavoriteStatus(midw_ext_fi_tcl_e8_Action::tenType enAction,
   AT_SEEK_ENUM enRegisteredFavoriteType,
   const string& sRegisteredContentName,
   SMSAPI_RETURN_CODE_ENUM enStatusCode) const
{
   if(SMSAPI_RETURN_CODE_SUCCESS == enStatusCode)
   {
      if(!sRegisteredContentName.empty())
      {
         ETG_TRACE_USR4(("Favorite Modified, enAction=%d Type=%d Name=%s", enAction, ETG_CENUM(AT_SEEK_ENUM,
            enRegisteredFavoriteType), sRegisteredContentName.c_str()));
      }
      else
      {
         ETG_TRACE_USR4(("Favorite Modified, enAction=%d Type=%d", enAction, ETG_CENUM(AT_SEEK_ENUM,
            enRegisteredFavoriteType)));
      }
   }
   else
   {
      ETG_TRACE_ERR(("Favorite Modify Error, enStatusCode = %d", ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enStatusCode)));
   }
}

/**
 * Print alert information
 * @param[in] pAlerts contains artist or song alert info
 */
tVoid fc_sxm_tclContentAlert::vPrintAlertInfo(const midw_ext_fi_tcl_AlertInfo& pAlerts) const
{
   AT_SEEK_ENUM enAlertFavoriteType = (AT_SEEK_ENUM) pAlerts.FavoriteType.enType;
   midw_ext_fi_tclString strContentName =
            (AT_SEEK_ARTIST == enAlertFavoriteType) ? pAlerts.ArtistName : pAlerts.SongName;

   ETG_TRACE_USR4(("Alert(Type, ChannelID, ChannelArtAvaialability, ContentName) = (%d, %d, %d, %s)",
            ETG_CENUM(AT_SEEK_ENUM, enAlertFavoriteType),
            pAlerts.ChannelID,
            pAlerts.ChnGraphicsAvailable.enType,
            strContentName.szValue));
}

/**
 * Print alert list
 * @param[in] vecAlertList contains list artist or song alert
 */
tVoid fc_sxm_tclContentAlert::vPrintAlertList(const vector<midw_ext_fi_tcl_AlertInfo>& vecAlertList) const
{
   if(vecAlertList.size())
   {
      for(vector<midw_ext_fi_tcl_AlertInfo>::const_reverse_iterator iter = vecAlertList.rbegin();
               iter != vecAlertList.rend(); ++iter)
      {
         vPrintAlertInfo(*iter);
      }
   }
   else
   {
      ETG_TRACE_USR4(("Alert list is empty"));
   }
}

/**
 * Hnadling the expired alert for both alert types
 * @hDecoder valid decoder object
 */
tVoid fc_sxm_tclContentAlert::vHandleExpiredAlert(DECODER_OBJECT hDecoder)
{
   if((_poContentAlertProxy) && (DECODER_INVALID_OBJECT != hDecoder))
   {
      CHANNEL_ID oChannelId = CHANNEL_INVALID_ID;
      STRING_OBJECT oArtist = STRING_INVALID_OBJECT;
      STRING_OBJECT oSong = STRING_INVALID_OBJECT;
      SMSAPI_RETURN_CODE_ENUM enReturnCode;
      do
      {
         enReturnCode  = _poContentAlertProxy->enGetNextEndedEventAttributes(hDecoder,
            &oChannelId,
            &oArtist,
            &oSong);
         if(SMSAPI_RETURN_CODE_SUCCESS == enReturnCode)
         {
            vCheckAndRemoveAlerts(oChannelId, oArtist, oSong);
         }
         else
         {
            ETG_TRACE_USR4(("vHandleExpiredAlert, No more ended events"));
         }
      }while(SMSAPI_RETURN_CODE_SUCCESS == enReturnCode);
   }
}

/**
 * This method handling the notification of individual alert and addition of alert into active content alert list
 * @param[in] hDecoder valid decoder object
 * @param[inout] oActiveContentAlertList populate with new alerts
 */
tVoid fc_sxm_tclContentAlert::vHandleNewAlerts(DECODER_OBJECT hDecoder,
   fc_sxm_trMsgAudioATAlerts& oActiveContentAlertList) const
{
   SEEK_EVENT_OBJECT hSeekEvent;

   if((_poContentAlertProxy) && (_poAudioApp) && (DECODER_INVALID_OBJECT != hDecoder))
   {
      hSeekEvent = _poContentAlertProxy->hGetNextAlertEvent(hDecoder);
      while(SEEK_EVENT_INVALID_OBJECT != hSeekEvent)
      {
         midw_ext_fi_tcl_AlertInfo oAlert = oGetAlertInfo(hSeekEvent);

         // As SMS is giving complete alert list which has been notified already in previous call back
         // Hence added check to suppress already notified alert
         if(!bIsAlertAlreadyNotified(oAlert))
         {
        	 oAlert.ChnGraphicsAvailable.enType = enGetChannelGraphicsUpdate(oAlert.ChannelID);

        	 const fc_sxm_trChannel* chnInfo = _poAudioApp->trGetChannel(static_cast<tS16>(oAlert.ChannelID));
        	 if(OSAL_NULL != chnInfo)
        	 {
        		 tBool bIsMature = chnInfo->bIsMature();
        		 tBool bIsLocked = chnInfo->bIsLocked();
        		 tBool bIsSkipped = chnInfo->bIsSkipped();
        		 oAlert.ChannelStatus = oGetChannelStatus(bIsMature, bIsLocked, bIsSkipped);

        		 vNotifyIndividualAlert(oAlert);
        		 vAddToActiveContentAlertList(oAlert, oActiveContentAlertList);
        	 }
         }
         hSeekEvent = _poContentAlertProxy->hGetNextAlertEvent(hDecoder);
      }
   }
}

/**
 * Get favorite list content iterator call back
 * @param[in] hSeekContent seek content object
 * @param[in,out] pvSeekContentIteratorArg
 * @return TRUE for continue iteration else FALSE
 */
tBool fc_sxm_tclContentAlert::cb_bGetFavoriteList(DECODER_OBJECT /*hDecoder*/,
   SEEK_SERVICE_ENUM /*enSeekService*/,
   SEEK_CONTENT_OBJECT hSeekContent,
   void *pvSeekContentIteratorArg)
{
   fc_sxm_trMsgAudioGetFavorite *poGetFavorite = static_cast<fc_sxm_trMsgAudioGetFavorite *>(pvSeekContentIteratorArg);
   return fc_sxm_tclContentAlert::instance()->bCheckAndGetFavorite(hSeekContent, poGetFavorite);
}

/**
 * Get favorite count iterator call back
 * @param[in] hSeekContent seek content object
 * @return TRUE for continue iteration else FALSE
 */
tBool fc_sxm_tclContentAlert::cb_bGetFavoriteCount(DECODER_OBJECT /*hDecoder*/,
   SEEK_SERVICE_ENUM /*enSeekService*/,
   SEEK_CONTENT_OBJECT hSeekContent,
   void * /*pvSeekContentIteratorArg*/)
{
   if(SEEK_CONTENT_INVALID_OBJECT != hSeekContent)
   {
      fc_sxm_tclContentAlert::instance()->vUpdateFavoritesCount(hSeekContent);
      return TRUE;
   }
   return FALSE;
}

/**
 * Modify favorite iterator call back
 * @param[in] hSeekContent seek content object
 * @param[in] pvSeekContentIteratorArg
 * @return TRUE for continue iteration else FALSE
 */
tBool fc_sxm_tclContentAlert::cb_bModifyFavorite(DECODER_OBJECT /*hDecoder*/,
   SEEK_SERVICE_ENUM /*enSeekService*/,
   SEEK_CONTENT_OBJECT hSeekContent,
   void *pvSeekContentIteratorArg)
{
   fc_sxm_trMsgAudioModifyFavorite *poModifyFavorite =
            static_cast<fc_sxm_trMsgAudioModifyFavorite *>(pvSeekContentIteratorArg);
   return fc_sxm_tclContentAlert::instance()->bCheckAndModifyFavorite(hSeekContent, poModifyFavorite);
}

/**
 * Get alert channel list call back
 * @param[in] hChannel channel object
 * @param[in] pvIterateArg
 * @return TRUE for continue iteration else FALSE
 */
N16 fc_sxm_tclContentAlert::cb_bGetActiveAlertChannelList(CATEGORY_OBJECT /*hCategory*/,
   CATEGORY_CHANNEL_INDEX /*tCurrentIndex*/,
   CHANNEL_OBJECT hChannel,
   void * pvIterateArg)
{
   fc_sxm_trMsgAudioActiveAlertChannelList *oActiveAlerts =
            static_cast<fc_sxm_trMsgAudioActiveAlertChannelList *>(pvIterateArg);
   return fc_sxm_tclContentAlert::instance()->u16GetActiveAlertChannel(hChannel, oActiveAlerts);
}


/**
 * Handle Method start for FI Method "fc_sxm_trMsgAudioMStartSaveFavorite"
 * @param[in] prMsg save favorite method start request
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioMStartSaveFavorite const *prMsg)
{
   midw_ext_sxm_audiofi_tclMsgSaveFavoriteMethodResult ofiSaveFavorites;
   DECODER_OBJECT hDecoder = hGetDecoder();
   (bSaveFavorite(hDecoder, prMsg, &ofiSaveFavorites)) ?
            vSendContentAlertFiMsg(prMsg->rAdressing, ofiSaveFavorites) :
            vSendContentAlertFiError(prMsg->rAdressing,
               (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR);
}

/**
 * Handle Method start for FI Method "fc_sxm_trMsgAudioMStartGetFavoriteList"
 * @param[in] prMsg get favorite list method start request
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioMStartGetFavoriteList const *prMsg)
{
   fc_sxm_trMsgAudioGetFavorite oGetFavorite;
   DECODER_OBJECT hDecoder = hGetDecoder();
   bGetFavoriteList(hDecoder, prMsg, &oGetFavorite) ?
            vSendContentAlertFiMsg(prMsg->rAdressing, oGetFavorite.ofiFavoriteList) :
            vSendContentAlertFiError(prMsg->rAdressing,
               (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR);
}

/**
 * Handle Method start for FI Method "fc_sxm_trMsgAudioMStartModifyFavorite"
 * @param[in] prMsg modify favorite method start request
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioMStartModifyFavorite const *prMsg)
{
   midw_ext_sxm_audiofi_tclMsgModifyFavoriteMethodResult ofiModifyFavorite;
   DECODER_OBJECT hDecoder = hGetDecoder();
   bModifyFavorite(hDecoder, prMsg, &ofiModifyFavorite) ?
            vSendContentAlertFiMsg(prMsg->rAdressing, ofiModifyFavorite) :
            vSendContentAlertFiError(prMsg->rAdressing,
               (tInt) midw_ext_fi_tcl_e8_ErrorTypes::FI_EN_XMTUN_ERROR_INTERNAL_ERROR);
}

/**
 * Handle Method start for FI Method "fc_sxm_trMsgAudioMStartGetAlertNoticeList"
 * @param[in] prMsg Get alert notice list method start request
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioMStartGetAlertNoticeList const *prMsg)
{
   midw_ext_sxm_audiofi_tclMsgGetAlertNoticeListMethodResult oFiAlerts;
   AT_SEEK_ENUM enAlertType = AT_SEEK_ENUM(prMsg->oFiMsg.FavoriteType.enType);
   if(AT_SEEK_ARTIST == enAlertType)
   {
      oFiAlerts.AlertInfoList = _vecArtistList;
   }
   else if(AT_SEEK_TITLE == enAlertType)
   {
      oFiAlerts.AlertInfoList = _vecSongList;
   }

   reverse(oFiAlerts.AlertInfoList.begin(), oFiAlerts.AlertInfoList.end()); // Fix for NCG3D-5018
   vSendContentAlertFiMsg(prMsg->rAdressing, oFiAlerts);
}

/**
 * Handle Method start for FI Method "fc_sxm_trMsgAudioMStartGetAlertNoticeList"
 * @param[in] prMsg Get alert notice list method start request
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioATAlerts const *prMsg)
{
   vUpdateAlertList(_bIsArtistAlertBlocked, prMsg->oArtistAlerts);
   vUpdateAlertList(_bIsSongAlertBlocked, prMsg->oSongAlerts);
}

/**
 * Handle Seek Alert event for expired alert and new alerts
 * @param[in] prMsg
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioSeekAlertEvent const * /*prMsg*/)
{
   DECODER_OBJECT hDecoder = hGetDecoder();
   if(_poAudioApp && (DECODER_INVALID_OBJECT != hDecoder))
   {
      fc_sxm_trMsgAudioATAlerts oActiveContentAlertList;
      vHandleExpiredAlert(hDecoder);
      vHandleNewAlerts(hDecoder, oActiveContentAlertList);

      _poAudioApp->vPostATAlertMsg(oActiveContentAlertList);
   }
}

/**
 * Handle Seek Category update to remove an alerts from active alert list
 * @param[in] prMsg
 */
tVoid fc_sxm_tclContentAlert::vProcess(fc_sxm_trMsgAudioSeekCategoryUpdate const * /*prMsg*/)
{
   DECODER_OBJECT hDeocder = hGetDecoder();
   if((_poContentAlertProxy) && (DECODER_INVALID_OBJECT != hDeocder))
   {
      fc_sxm_trMsgAudioActiveAlertChannelList oAlertChannelList;
      oAlertChannelList.u32Size = _poContentAlertProxy->u32GetSeekChannelSize(hDeocder, _hSeekCategoryID);
      _poContentAlertProxy->vIterateAlertChannels(hDeocder, _hSeekCategoryID, cb_bGetActiveAlertChannelList, &oAlertChannelList);

      // Check and Remove for Artist alert if get expired
      if(u16RemoveExpiredAlerts(_vecArtistList, oAlertChannelList.vecChannelList))
      {
         vNotifyActiveContentAlertList(AT_SEEK_ARTIST);
         ETG_TRACE_USR4(("Active Artist Alert Notice List (%d)", _vecArtistList.size()));
      }

      // Check and remove for Song alert if get expired
      if(u16RemoveExpiredAlerts(_vecSongList, oAlertChannelList.vecChannelList))
      {
         vNotifyActiveContentAlertList(AT_SEEK_TITLE);
         ETG_TRACE_USR4(("Active Song Alert Notice List (%d)", _vecSongList.size()));
      }
   }
}

