/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_content_alert_proxy.cpp
* @brief       Proxy class implementation to interact with SMS library 
*              for enabling SMS seek services.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_content_alert_proxy.h"

/**
 * Start seek service
 * @param hDecoderObject[in] valid decoder object
 * @return SMSAPI_RETURN_CODE_SUCCESS if seek service is started successful else error value
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enStart(DECODER_OBJECT hDecoderObject) const
{
   return (SEEK.eStart(hDecoderObject, SEEK_SERVICE_ARTIST_TITLE));
}

/**
 * Stop seek service
 * @param hDecoderObject[in] valid decoder object
 */
tVoid fc_sxm_tclContentAlertProxy::vStop(DECODER_OBJECT hDecoderObject) const
{
   SEEK.vStop(hDecoderObject, SEEK_SERVICE_ARTIST_TITLE);
}

/**
 * Iterate through favorite list
 * @param hDecoderObject[in] valid decoder object
 * @param bSeekContentIterator[in] callback function
 * @param pvSeekContentIteratorArg
 */
tVoid fc_sxm_tclContentAlertProxy::vIterateFavorites(DECODER_OBJECT hDecoderObject,
   SEEK_CONTENT_ITERATOR_CALLBACK bSeekContentIterator,
   void *pvSeekContentIteratorArg) const
{
   SEEK.eIterate(hDecoderObject, SEEK_SERVICE_ARTIST_TITLE, bSeekContentIterator, pvSeekContentIteratorArg);
}

/**
 * Iterate Channel list
 * @param hDecoder[in] valid decoder object
 * @param hCategoryId[in] category id
 * @param n16IterateFxn
 * @param pvIterateArg
 */
tVoid fc_sxm_tclContentAlertProxy::vIterateAlertChannels(DECODER_OBJECT hDecoder,
   CATEGORY_ID hCategoryId,
   CATEGORY_ITERATION_HANDLER n16IterateFxn,
   void *pvIterateArg) const
{
   CATEGORY.eIterate(hDecoder, hCategoryId, n16IterateFxn, pvIterateArg);
}

/**
 * Get tuned channel id
 * @param hDecoder[in] valid decoder object
 * @return channel id
 */
CHANNEL_ID fc_sxm_tclContentAlertProxy::hGetTunedChannel(DECODER_OBJECT hDecoder) const
{
   return (DECODER.tCurrentChannelId(hDecoder));
}

/**
 * Get Channel
 * @param hSeekEvent[in] seek event object
 * @return channel object
 */
CHANNEL_OBJECT fc_sxm_tclContentAlertProxy::hGetChannel(SEEK_EVENT_OBJECT hSeekEvent) const
{
   return (SEEK_EVENT.hChannel(hSeekEvent));
}

/**
 * Get Seek Content object
 * @param hSeekEvent[in] seek event object
 * @return seek content object
 */
SEEK_CONTENT_OBJECT fc_sxm_tclContentAlertProxy::hGetSeekContent(SEEK_EVENT_OBJECT hSeekEvent) const
{
   return (SEEK_EVENT.hContent(hSeekEvent));
}

/**
 * Get Seek category id
 * @param hDecoder[in] valid decoder object
 * @param hServiceType[in] SEEK_SERVICE_ARTIST_TITLE
 * @return CATEGORY_ID of the "My Seeks" category
 */
CATEGORY_ID fc_sxm_tclContentAlertProxy::hGetSeekCategoryId(DECODER_OBJECT hDecoder) const
{
   return (SEEK.tCategoryId(hDecoder, SEEK_SERVICE_ARTIST_TITLE));
}

/**
 * Get Seek Category channel size
 * @param hDecoder[in] deocder object
 * @param hCategoryId[in] category id
 * @return number of channel in category
 */
tU32 fc_sxm_tclContentAlertProxy::u32GetSeekChannelSize(DECODER_OBJECT hDecoder, CATEGORY_ID hCategoryId) const
{
   return ((tU32)(CATEGORY.tSize(hDecoder, hCategoryId)));
}

/**
 * Get Service Id
 * @param hChannel[in] channel object
 * @return service id
 */
SERVICE_ID fc_sxm_tclContentAlertProxy::hGetServiceId(CHANNEL_OBJECT hChannel) const
{
   return (CHANNEL.tServiceId(hChannel));
}

/**
 * Get Channel Id
 * @param hChannel[in] channel object
 * @return channel id
 */
CHANNEL_ID fc_sxm_tclContentAlertProxy::hGetChannelId(CHANNEL_OBJECT hChannel) const
{
   return (CHANNEL.tChannelId(hChannel));
}

/**
 * Get SMS String object as stl string
 * @param hString[in] string object
 * @return string
 */
string fc_sxm_tclContentAlertProxy::sGetSmsStringAsStl(STRING_OBJECT hString) const
{
   string strString = "";
   fc_sxm_vCopySmsString2Stl(hString,strString);

   return strString;
}

/**
 * Get SMS String object as fi string
 * @param hString[in] string object
 * @return fi string
 */
midw_ext_fi_tclString fc_sxm_tclContentAlertProxy::hGetSmsStringAsFi(STRING_OBJECT hString) const
{
   midw_ext_fi_tclString hRegisteredText = NULL;
   fc_sxm_vCopySmsString2Fi(hString, hRegisteredText);
   return hRegisteredText;
}

/**
 * Get Channel name
 * @param hChannel[in] channel object
 * @return chanael name as string object
 */
STRING_OBJECT fc_sxm_tclContentAlertProxy::hGetChannelName(CHANNEL_OBJECT hChannel) const
{
   return (CHANNEL.hLongName(hChannel));
}

/**
 * Get artist name
 * @param hSeekContent[in] seek content object
 * @return artist name as string object
 */
STRING_OBJECT fc_sxm_tclContentAlertProxy::hGetArtistName(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return (AT_SEEK_CONTENT.hRegisteredArtistText(hSeekContent));
}

/**
 * Get song name
 * @param hSeekContent[in] seek content object
 * @return song name as string object
 */
STRING_OBJECT fc_sxm_tclContentAlertProxy::hGetSongName(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return (AT_SEEK_CONTENT.hRegisteredTitleText(hSeekContent));
}

/**
 * Get artist name from channel object
 * @param hChannel[in] channel object
 * @return artist name as string object
 */
STRING_OBJECT fc_sxm_tclContentAlertProxy::hGetArtistName(CHANNEL_OBJECT hChannel) const
{
   CD_OBJECT hCDO = CHANNEL.hCDO(hChannel);
   return ((CD_INVALID_OBJECT != hCDO) ?
            CDO.hArtist(hCDO) :
            STRING_INVALID_OBJECT);
}


/**
 * Get song name from channel object
 * @param hChannel[in] channel object
 * @return song name as string object
 */
STRING_OBJECT fc_sxm_tclContentAlertProxy::hGetSongName(CHANNEL_OBJECT hChannel) const
{
   CD_OBJECT hCDO = CHANNEL.hCDO(hChannel);
   return ((CD_INVALID_OBJECT != hCDO) ?
            CDO.hTitle(hCDO) :
            STRING_INVALID_OBJECT);
}

/**
 * Get favorite type
 * @param hSeekContent[in] seek content object
 * @return favorite type (at_seek_enum)
 */
AT_SEEK_ENUM fc_sxm_tclContentAlertProxy::enGetFavoriteType(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return (AT_SEEK_CONTENT.eType(hSeekContent));
}

/**
 * Get next alert event
 * @param hDecoder[in] decoder object
 * @return seek event object
 */
SEEK_EVENT_OBJECT fc_sxm_tclContentAlertProxy::hGetNextAlertEvent(DECODER_OBJECT hDecoder) const
{
   return (SEEK.hGetNextEvent(hDecoder));
}

SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enGetNextEndedEventAttributes(DECODER_OBJECT hDecoder,
      CHANNEL_ID *oChannelID,
      STRING_OBJECT *oArtist,
      STRING_OBJECT *oTitle) const
{
   return (SEEK.eGetNextEndedEventAttributes(hDecoder,
      oChannelID,
      oArtist,
      oTitle));
}

tBool fc_sxm_tclContentAlertProxy::bGetAlertsFinishedStatus(DECODER_OBJECT hDecoder) const
{
   return (SEEK.bGetAlertsFinishedStatus(hDecoder,
      SEEK_SERVICE_ARTIST_TITLE));
}

/**
 * Get Favorite status
 * @param hSeekContent[in] seek content object
 * @return TRUE when content is enabled else FALSE
 */
tBool fc_sxm_tclContentAlertProxy::bGetFavoriteStatus(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return ((tBool)SEEK_CONTENT.bEnabled(hSeekContent));
}

/**
 * Register content based on channel id
 * @param hDecoderObject[in] decoder object
 * @param enFavoriteType[in] favorite type
 * @param hChannelID[in] channel id
 * @param poSeekContentObject[out] seek content object for register favorite
 * @return SMSAPI_RETURN_CODE_SUCCESS if registration of favorite is successful else error return code
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enRegisterFavorite(DECODER_OBJECT hDecoderObject,
   AT_SEEK_ENUM enFavoriteType,
   CHANNEL_ID hChannelID,
   SEEK_CONTENT_OBJECT* poSeekContentObject) const
{
   return (AT_SEEK.eRegister(hDecoderObject,
         enFavoriteType,
         (CHANNEL_ID) hChannelID,
         poSeekContentObject));
}

/**
 * Check for artist and title availability
 * @param hDecoderObject[in] decoder object
 * @param hChannelID[in] channel id
 * @param enTitleAvailablility[out] title availability
 * @param enArtistAvailability[out] artist availability
 */
tVoid fc_sxm_tclContentAlertProxy::vIsAvailableToRegister(DECODER_OBJECT hDecoderObject,
   CHANNEL_ID hChannelID,
   SEEK_AVAILABILTY_ENUM* enTitleAvailability,
   SEEK_AVAILABILTY_ENUM* enArtistAvailability) const
{
   AT_SEEK.eAvailableToRegister(hDecoderObject,
         (CHANNEL_ID) hChannelID,
         enTitleAvailability,
         enArtistAvailability);
}

/**
 * Enable favorite
 * @param hSeekContent[in] seek content object
 * @return SMSAPI_RETURN_CODE_SUCCESS if enable favorite is successful else error return code
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enEnableFavorite(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return (SEEK_CONTENT.eEnable(hSeekContent));
}

/**
 * Disable favorite
 * @param hSeekContent[in] seek content object
 * @return SMSAPI_RETURN_CODE_SUCCESS if disable favorite is successful else error return code
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enDisableFavorite(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return (SEEK_CONTENT.eDisable(hSeekContent));
}

/**
 * Delete favorite
 * @param hSeekContent[in] seek content object
 * @return SMSAPI_RETURN_CODE_SUCCESS if delete favorite is successful else error return code
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enDeleteFavorite(SEEK_CONTENT_OBJECT hSeekContent) const
{
   return (SEEK_CONTENT.eRemove(hSeekContent));
}

/**
 * Enable alerts on tuned channel
 * @param hDecoder valid decoder object
 * @param bEnable TRUE want alert on tuned channel else FALSE
 * @return SMSAPI_RETURN_CODE_SUCCESS when enable alert on tuned channel successful
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enEnableAlertsTuned(DECODER_OBJECT hDecoder,
         BOOLEAN bEnable) const
{
   return (SEEK.eEnableAlertsTuned(hDecoder,
      SEEK_SERVICE_ARTIST_TITLE,
      bEnable));
}

/**
 * Enable Alert notification for Finished
 * @param hDecoder
 * @param bEnable
 * @return SMSAPI_RETURN_CODE_SUCCESS when enable or diable alert finished successful
 */
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclContentAlertProxy::enEnableAlertsFinished(DECODER_OBJECT hDecoder,
      BOOLEAN bEnable) const
{
   return (SEEK.eEnableAlertsFinished(hDecoder,
      SEEK_SERVICE_ARTIST_TITLE,
      bEnable));
}

/**
 * Fill Alert info
 * @param hSeekEvent seek event
 * @return alert info message
 */
midw_ext_fi_tcl_AlertInfo fc_sxm_tclContentAlertProxy::oFillAlertInfo(SEEK_EVENT_OBJECT hSeekEvent) const
{
   midw_ext_fi_tcl_AlertInfo oAlertDetails;
   if(SEEK_EVENT_INVALID_OBJECT != hSeekEvent)
   {
      CHANNEL_OBJECT hChannel = hGetChannel(hSeekEvent);
      SEEK_CONTENT_OBJECT hSeekContent = hGetSeekContent(hSeekEvent);

      STRING_OBJECT hChannelName = hGetChannelName(hChannel);
      STRING_OBJECT hArtistName = hGetArtistName(hChannel);
      STRING_OBJECT hSongName = hGetSongName(hChannel);
      AT_SEEK_ENUM enSeekType = enGetFavoriteType(hSeekContent);

      oAlertDetails.ServiceID = (tU16) hGetServiceId(hChannel);
      oAlertDetails.ChannelID = (tU16) hGetChannelId(hChannel);
      oAlertDetails.FavoriteType.enType = midw_ext_fi_tcl_e8_FavoriteType::tenType(enSeekType);
      oAlertDetails.ChannelName = hGetSmsStringAsFi(hChannelName);
      oAlertDetails.ArtistName = hGetSmsStringAsFi(hArtistName);
      oAlertDetails.SongName = hGetSmsStringAsFi(hSongName);
   }
   return oAlertDetails;
}
